from __future__ import annotations

import json
from typing import TYPE_CHECKING

import click

from ethbinductor import __version__

if TYPE_CHECKING:
	from collections.abc import Callable
	from typing import Any, Final


CONTEXT_SETTINGS: Final[dict[str, Any]] = {'help_option_names': ['-h', '--help']}


class OrderedArgsCommand(click.Command):

	def __init__(
		self: OrderedArgsCommand,
		sortf: Callable[[click.Argument | click.Option], str] = lambda p: p.name,
		*args: Any,
		**kwargs: Any,
	) -> None:
		super().__init__(*args, **kwargs)
		self.params = sorted(self.params, key=sortf)


@click.group(context_settings=CONTEXT_SETTINGS)
@click.version_option(version=__version__)
def cli() -> None:
	pass
	# Initial stuff..
	# TODO: Some links to project's GitHub & docs..


main = cli


def cli_command_oa(*args: Any, **kwargs: Any):
	return cli.command(*args, cls=OrderedArgsCommand, **kwargs)


class ProviderNotFoundError(ModuleNotFoundError):
	pass


def proc(
	*,
	abi_provider_arg: str,
	abi_provider: str,
	output: str,
	output_arg: str,
	# action is like a middleware
	action: Callable[[list[dict[str, Any]]], str],
) -> None:
	# TODO: Add abi & output provider `custom` to load user's providers..
	# TODO: Annotate..
	try:
		get_json = __import__(
			f'ethbinductor.providers.abi.{abi_provider}', fromlist=('ethbinductor'),
		).get_json
		put = __import__(f'ethbinductor.providers.output.{output}', fromlist=('ethbinductor')).put
	except ModuleNotFoundError as err:
		msg = f"""Provider not found: 'ethbinductor.{err.args[0].split("'")[1]}'"""
		raise ProviderNotFoundError(msg) from err
	result: list[dict[str, Any]] = get_json(abi_provider_arg)
	data = action(result)
	put(data, output_arg)


def proc_clicks(func: Callable[..., None]) -> Callable[..., None]:
	# FIXME: Mb I can use it in some way on the group itself?
	click.argument('abi_provider_arg', required=False)(func)
	# FIXME: Use commands objects instead of literally set it's name..
	click.option(
		'--abi-provider', default='file',
		help=(
			'Provider of abi json data.'
			' '
			'(use `info --list-abi-providers` to see available abi providers)'
		),
	)(func)
	click.argument('output_arg', required=False)(func)
	click.option(
		'--output', default='py_print',
		help=(
			'Output to write abi json data.'
			' '
			'(use `info --list-outputs` to see available output providers)'
		),
	)(func)

	return func


@cli_command_oa(help='Use to get abi from some data. (usually contract address)')
@proc_clicks
def get_abi(
	**kw: str,
) -> None:
	proc(
		**kw,
		action=lambda data: json.dumps(data, indent=4, ensure_ascii=False),
	)


@cli_command_oa(help='Translate ETH Contract json abi to native Python code binding.')
@proc_clicks
@click.option(
	'--cls-name',
	required=True,
	help='Class name of generated code.',
)
def abi2py(
	*,
	cls_name: str,
	**kw: str,
) -> None:
	from ethbinductor.conductor import abi2pys

	proc(
		**kw,
		action=lambda abi: abi2pys(abi, cls_name=cls_name) + (
			'\n# Generated by ETHBinductor: '
			'https://github.com/BlackCatDevel0per/ETHBinductor\n'
		),
	)


@cli.command()
@click.option('--list-abi-providers', is_flag=True)
@click.option('--list-outputs', is_flag=True)
def info(*, list_abi_providers: bool, list_outputs: bool) -> None:
	if list_abi_providers:
		flist_providers(ptype='abi')

	if list_outputs:
		flist_providers(ptype='output')


def flist_providers(*, ptype: str) -> None:
	# TODO: Modules lookup with desc read using ast..
	from pathlib import Path
	providers = list((Path(__file__).parent / 'providers' / ptype).glob('*'))

	print('Available providers:\n')

	for p in providers:
		provider: str = p.stem
		if provider.startswith('_') or provider.endswith('base'):
			continue

		print(provider)
