from google.oauth2.credentials import Credentials
from google_auth_oauthlib.flow import InstalledAppFlow
from google.auth.transport.requests import Request
from googleapiclient.discovery import build
from googleapiclient.errors import HttpError
import os
import sys

os.environ['OAUTHLIB_INSECURE_TRANSPORT'] = '1'

class GoogleAuthManager:
    def __init__(self, credentials_file, token_dir="credentials", scopes=None, token_filename="token.json", redirect_uri='http://localhost:8080/'):
        """
        Inicializa o gerenciador de autenticação.
        
        Args:
            credentials_file (str): Caminho para o client_secret.json
            token_dir (str): Diretório onde os tokens serão salvos.
            scopes (list): Lista de escopos necessários.
            token_filename (str): Nome do arquivo para salvar o token.
            redirect_uri (str): URI de redirecionamento autorizada no Google Cloud Console.
        """
        self.credentials_file = credentials_file
        self.token_dir = token_dir
        self.scopes = scopes or []
        self.redirect_uri = redirect_uri
        self.token_file = os.path.join(self.token_dir, token_filename)
        self.creds = None
        
        # Cria o diretório de tokens se não existir
        os.makedirs(token_dir, exist_ok=True)


    def authenticate(self):
        """
        Autentica o usuário. Carrega o token se existir, caso contrário,
        inicia o fluxo de autorização no console, ideal para Docker.
        """
        # 1. Tenta carregar o token salvo
        if os.path.exists(self.token_file):
            try:
                self.creds = Credentials.from_authorized_user_file(self.token_file, self.scopes)
            except Exception as e:
                print(f"Erro ao carregar token de '{self.token_file}': {e}. Um novo login será necessário.")
                self.creds = None

        # 2. Se não tiver token, ou se for inválido/expirado, faz o login/refresh
        if not self.creds or not self.creds.valid:
            if self.creds and self.creds.expired and self.creds.refresh_token:
                try:
                    print("Token expirado, tentando atualizar...")
                    self.creds.refresh(Request())
                except Exception as e:
                    print(f"Erro ao atualizar token: {e}. Será necessário um novo login.")
                    self.creds = None # Força novo login

            # Se ainda não tem credenciais, inicia o fluxo de autorização
            if not self.creds:
                if not os.path.exists(self.credentials_file):
                    print(f"Erro: Arquivo de credenciais '{self.credentials_file}' não encontrado.")
                    print("Certifique-se de que o arquivo JSON do Google Cloud Console está no lugar certo.")
                    sys.exit(1)
                
                # Este fluxo é ideal para rodar em um container Docker.
                # Ele mostra a URL no console, o usuário a abre no seu navegador local,
                # autoriza, e cola a URL de redirecionamento de volta no console.
                print("Iniciando fluxo de autenticação no console...")
                flow = InstalledAppFlow.from_client_secrets_file(self.credentials_file, self.scopes)
                flow.redirect_uri = self.redirect_uri

                # Gera a URL de autorização
                auth_url, _ = flow.authorization_url(prompt='consent')
                
                print("\n--- AÇÃO NECESSÁRIA PARA AUTENTICAÇÃO ---")
                print("Para autorizar o acesso, siga estes passos:")
                print(f"1. Abra a seguinte URL no seu navegador:\n")
                print(f"   {auth_url}\n")
                print("2. Faça login na sua Conta Google e autorize o acesso.")
                print(f"3. Você será redirecionado para uma página (provavelmente exibindo um erro 'Não é possível acessar esse site'), o que é esperado.")
                print("4. Copie a URL completa da barra de endereço do seu navegador.")
                
                redirect_response = input('\n5. Cole a URL de redirecionamento completa aqui e pressione Enter: ')

                # Troca o código de autorização por um token de acesso
                try:
                    flow.fetch_token(authorization_response=redirect_response)
                    self.creds = flow.credentials
                except Exception as e:
                    print(f"\nOcorreu um erro ao obter o token: {e}")
                    print("Verifique se a URL foi copiada corretamente. O aplicativo será encerrado.")
                    sys.exit(1)

            # Salva o token (novo ou atualizado) para as próximas execuções
            try:
                with open(self.token_file, "w") as token:
                    token.write(self.creds.to_json())
                print(f"Autenticação bem-sucedida. Token salvo em '{self.token_file}'")
            except Exception as e:
                print(f"Erro ao salvar o token em '{self.token_file}': {e}")
        else:
            print("Credenciais válidas carregadas do token.")
        

    def build_service(self, service_name, version):
        """
        Constrói um serviço da API do Google.
        
        Args:
            service_name (str): Nome do serviço (ex: 'youtube', 'calendar')
            version (str): Versão da API (ex: 'v3')
        
        Returns:
            Resource: Objeto do serviço construído ou None em caso de erro.
        """
        if not self.creds or not self.creds.valid:
            self.authenticate()
        
        if not self.creds:
            print("Não foi possível autenticar. Não é possível construir o serviço.")
            return None
            
        try:
            service = build(service_name, version, credentials=self.creds)
            print(f"Serviço '{service_name}' v'{version}' construído com sucesso.")
            return service
        except HttpError as error:
            print(f"Ocorreu um erro ao construir o serviço '{service_name}': {error}")
            return None





if __name__ == "__main__":
    auth_manager = GoogleAuthManager(
        credentials_file="client_secret.json",
        token_dir="credentials",
        token_filename="token_multiapi.json",
        redirect_uri='http://localhost:2000/',
        scopes=["https://www.googleapis.com/auth/youtube.readonly",
                "https://www.googleapis.com/auth/calendar.readonly"],
    )

    # Autenticação automática
    youtube = auth_manager.build_service('youtube', 'v3')
    calendar = auth_manager.build_service('calendar', 'v3')

    results = youtube.search().list(q="Python", part="snippet").execute()
    print("YouTube Search Results:")
    for item in results.get("items", []):
        print(f"- {item['snippet']['title']}")
