"""Fill in a module description here"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_database.ipynb.

# %% auto 0
__all__ = ['database_url', 'db', 'DatabaseService', 'utc_now', 'SQLModelDB', 'get_db_service', 'json_serializer', 'BaseTable']

# %% ../nbs/00_database.ipynb 3
from abc import ABC, abstractmethod
from typing import Generator, Any, Dict, List, Optional, Type
from datetime import datetime, timezone

import sqlalchemy as sa
from sqlalchemy import func, or_
from sqlmodel import Session, SQLModel, create_engine, select, Field
from uuid import UUID

import os

import json
from datetime import datetime, timezone
from functools import lru_cache
from typing import Any, ClassVar, Dict, List, Optional, Set, Type
from uuid import UUID, uuid4

from pydantic import ConfigDict
from pydantic.json import pydantic_encoder
from pydantic_core import PydanticUndefined
from pydantic_core.core_schema import SerializerFunctionWrapHandler

# %% ../nbs/00_database.ipynb 5
class DatabaseService(ABC):
    @abstractmethod
    def init_db(self) -> None:
        pass

    @abstractmethod
    def get_session(self) -> Generator[Any, None, None]:
        pass

    @abstractmethod
    def schema(self) -> str:
        pass

    @abstractmethod
    def all_records(self, model: Type[SQLModel]) -> List[SQLModel]:
        pass

    @abstractmethod
    def filter(
        self,
        model: Type[SQLModel],
        sorting_field: Optional[str] = None,
        sort_direction: str = "asc",
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        as_dict: bool = False,
        fields: Optional[List[str]] = None,
        exact_match: bool = True,  
        **kwargs
    ) -> List[Dict[str, Any]]:
        pass

    @abstractmethod
    def search(
        self,
        model: Type[SQLModel],
        search_value: Optional[str] = None,
        sorting_field: Optional[str] = None,
        sort_direction: str = "asc",
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        as_dict: bool = False,
        fields: Optional[List[str]] = None,
    ) -> List[Dict[str, Any]]:
        pass

    @abstractmethod
    def get_record(
        self, model: Type[SQLModel], id: Any, alt_key: str = None
    ) -> Optional[SQLModel]:
        pass

    @abstractmethod
    def update_record(
        self, model: Type[SQLModel], id: Any, data: Dict[str, Any]
    ) -> Dict[str, Any]:
        pass

    @abstractmethod
    def delete_record(self, model: Type[SQLModel], id: Any) -> None:
        pass

    @abstractmethod
    def upsert_record(self, model: Type[SQLModel], data: Dict[str, Any]) -> SQLModel:
        pass

    @abstractmethod
    def bulk_insert(
        self, model: Type[SQLModel], data: List[Dict[str, Any]]
    ) -> List[SQLModel]:
        pass

    @abstractmethod
    def bulk_update(
        self, model: Type[SQLModel], data: List[Dict[str, Any]]
    ) -> List[SQLModel]:
        pass

    @abstractmethod
    def count_records(self, model: Type[SQLModel]) -> int:
        pass


# %% ../nbs/00_database.ipynb 7
from typing import Optional, List, Any, Dict, Type, Union, get_args, get_origin
from datetime import datetime, date
from uuid import UUID
from decimal import Decimal

def utc_now() -> datetime:
    return datetime.now(timezone.utc)

class SQLModelDB(DatabaseService):
    def __init__(self, url: str, echo: bool = False):
        self.engine = create_engine(url, echo=echo)


    def init_db(self) -> None:
        SQLModel.metadata.create_all(self.engine)


    def get_session(self) -> Generator[Session, None, None]:
        with Session(self.engine) as session:
            yield session


    def schema(self) -> str:
        inspector = sa.inspect(self.engine)
        res = ""
        for table_name in inspector.get_table_names():
            res += f"Table: {table_name}\n"
            pk_cols = inspector.get_pk_constraint(table_name)["constrained_columns"]
            for column in inspector.get_columns(table_name):
                pk_marker = "*" if column["name"] in pk_cols else "-"
                res += f"  {pk_marker} {column['name']}: {column['type']}\n"
        return res


    def all_records(self, model: Type[SQLModel]) -> List[SQLModel]:
        with Session(self.engine) as session:
            statement = select(model)
            results = session.exec(statement).all()
            return results

    def filter(
        self,
        model: Type[SQLModel],
        sorting_field: Optional[str] = None,
        sort_direction: str = "asc",
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        as_dict: bool = False,
        fields: Optional[List[str]] = None,
        exact_match: bool = True,
        **kwargs
    ) -> List[Dict[str, Any]]:
        with Session(self.engine) as session:
            # Validate that all filter fields exist in the model
            invalid_fields = [field for field in kwargs.keys() if field not in model.__fields__]
            if invalid_fields:
                raise ValueError(f"Invalid fields for filtering: {', '.join(invalid_fields)}")

            # Build the base query
            if fields:
                query = select(*[getattr(model, field) for field in fields])
            else:
                query = select(model)

            # Add filters for each kwarg
            for field, value in kwargs.items():
                if value is None:
                    query = query.filter(getattr(model, field).is_(None))
                    continue

                field_type = model.__fields__[field].annotation
                # Get the underlying type if it's Optional
                if get_origin(field_type) is Union:
                    # Optional[T] is actually Union[T, None]
                    field_type = next((t for t in get_args(field_type) if t is not type(None)), str)

                if not exact_match and isinstance(value, str):
                    query = query.filter(getattr(model, field).ilike(f"%{value}%"))
                else:
                    # Handle different field types
                    if field_type in (str, Optional[str]):
                        if exact_match:
                            query = query.filter(getattr(model, field) == value)
                        else:
                            query = query.filter(getattr(model, field).ilike(f"%{value}%"))
                    
                    elif field_type in (int, float, Decimal, bool, Optional[int], Optional[float], Optional[Decimal], Optional[bool]):
                        query = query.filter(getattr(model, field) == value)
                    
                    elif field_type in (datetime, date, Optional[datetime], Optional[date]):
                        # Handle date/datetime range queries
                        if isinstance(value, (list, tuple)) and len(value) == 2:
                            start, end = value
                            query = query.filter(
                                getattr(model, field).between(start, end)
                            )
                        else:
                            query = query.filter(getattr(model, field) == value)
                    elif field_type is UUID:
                        # Handle UUID fields, converting string to UUID if needed
                        if isinstance(value, str):
                            try:
                                value = UUID(value)
                            except ValueError:
                                raise ValueError(f"Invalid UUID format for field {field}: {value}")
                        query = query.filter(getattr(model, field) == value)
                    
                    elif isinstance(value, (list, tuple)):
                        # Handle IN queries for lists
                        query = query.filter(getattr(model, field).in_(value))
                    
                    else:
                        # Default to exact match for unknown types
                        query = query.filter(getattr(model, field) == value)

            # Add sorting
            if sorting_field:
                if sorting_field in model.__fields__:
                    order_field = getattr(model, sorting_field)
                    query = query.order_by(
                        order_field.desc()
                        if sort_direction.lower() == "desc"
                        else order_field
                    )
                else:
                    raise ValueError(
                        f"Sorting field '{sorting_field}' does not exist in the model."
                    )
            else:
                query = query.order_by(model.id)

            # Add pagination
            if limit is not None:
                query = query.limit(limit)

            if offset is not None:
                query = query.offset(offset)

            results = session.exec(query).all()

            if as_dict:
                return [result.dict() for result in results]
            return results

    def search(
        self,
        model: Type[SQLModel],
        search_value: Optional[str] = None,
        sorting_field: Optional[str] = None,
        sort_direction: str = "asc",
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        as_dict: bool = False,
        fields: Optional[List[str]] = None,
    ) -> List[Dict[str, Any]]:
        with Session(self.engine) as session:
            if fields:
                query = select(*[getattr(model, field) for field in fields])
            else:
                query = select(model)

            if search_value:
                string_fields = [
                    k for k, v in model.__fields__.items() if v.annotation is str
                ]
                if string_fields:
                    conditions = [
                        getattr(model, field).ilike(f"%{search_value}%")
                        for field in string_fields
                    ]
                    query = query.filter(or_(*conditions))

            if sorting_field:
                if sorting_field in model.__fields__:
                    order_field = getattr(model, sorting_field)
                    query = query.order_by(
                        order_field.desc()
                        if sort_direction.lower() == "desc"
                        else order_field
                    )
                else:
                    raise ValueError(
                        f"Sorting field '{sorting_field}' does not exist in the model."
                    )
            else:
                query = query.order_by(model.id)

            if limit is not None:
                query = query.limit(limit)

            if offset is not None:
                query = query.offset(offset)

            results = session.exec(query).all()

            if as_dict:
                dict_results = [result._asdict() for result in results]
                return dict_results
            else:
                return results


    def get_record(self, model: Type[SQLModel], id: Any, alt_key: str = None) -> Optional[SQLModel]:
        with Session(self.engine) as session:
            if alt_key:
                stmt = select(model).where(getattr(model, alt_key) == id)
                result = session.exec(stmt).first()
            else:
                if isinstance(id, str):
                    id = UUID(id)
                result = session.get(model, id)
            return result


    def update_record(self, model: Type[SQLModel], id: Any, data: Dict[str, Any]) -> Dict[str, Any]:
        with Session(self.engine) as session:
            record = session.get(model, id)
            if not record:
                raise Exception(f"Record with id {id} not found")
            for key, value in data.items():
                setattr(record, key, value)
            session.add(record)
            session.commit()
            session.refresh(record)
            return record.dict()


    def delete_record(self, model: Type[SQLModel], id: Any) -> None:
        with Session(self.engine) as session:
            record = session.get(model, id)
            if record:
                session.delete(record)
                session.commit()


    def upsert_record(self, model: Type[SQLModel], data: Dict[str, Any]) -> SQLModel:
        with Session(self.engine) as session:
            if "id" in data:
                if isinstance(data["id"], str):
                    data["id"] = UUID(data["id"])
                data.pop("created_at",None)
                db_record = session.get(model, data["id"])
                if db_record:
                    for key, value in data.items():
                        setattr(db_record, key, value)
                    db_record.updated_at = utc_now()
                else:
                    db_record = model(**data)
                    db_record.created_at = utc_now()
                    db_record.updated_at = utc_now()
            else:
                db_record = model(**data)
                db_record.created_at = utc_now()
                db_record.updated_at = utc_now()

            session.add(db_record)
            session.commit()
            session.refresh(db_record)

            return db_record


    def bulk_insert(self, model: Type[SQLModel], data: List[Dict[str, Any]]) -> List[SQLModel]:
        with Session(self.engine) as session:
            records = [model(**item) for item in data]
            session.add_all(records)
            session.commit()
            for record in records:
                session.refresh(record)
            return records


    def bulk_update(self, model: Type[SQLModel], data: List[Dict[str, Any]]) -> List[SQLModel]:
        with Session(self.engine) as session:
            records = []
            for item in data:
                if "id" in item:
                    record = session.get(model, item["id"])
                    if record:
                        for key, value in item.items():
                            setattr(record, key, value)
                        records.append(record)
            session.add_all(records)
            session.commit()
            for record in records:
                session.refresh(record)
            return records


    def count_records(self, model: Type[SQLModel]) -> int:
        with Session(self.engine) as session:
            return session.exec(select(func.count()).select_from(model)).one()


# %% ../nbs/00_database.ipynb 8
database_url = os.getenv('DATABASE_URL','sqlite:///test.db')

def get_db_service():
    return SQLModelDB(url=database_url)
    
db = get_db_service()

# %% ../nbs/00_database.ipynb 10
def utc_now() -> datetime:
    return datetime.now(timezone.utc)


def json_serializer(value: Any, _: SerializerFunctionWrapHandler) -> str:
    return json.dumps(value, default=pydantic_encoder)


class BaseTable(SQLModel):
    model_config = ConfigDict(json_encoders={datetime: lambda dt: dt.isoformat()})
    id: UUID = Field(default_factory=uuid4, primary_key=True)
    created_at: datetime = Field(
        default_factory=utc_now,
        sa_type= sa.DateTime(timezone=True),
        sa_column_kwargs={"server_default": sa.func.now()},
        nullable=False,
        title="Created At",
        schema_extra={"icon": "clock", "input_type": "datetime"},
    )
    updated_at: datetime = Field(
        default_factory=utc_now,
        sa_type=sa.DateTime(timezone=True),
        sa_column_kwargs={
            "server_default": sa.func.now(),
            "server_onupdate": sa.func.now(),
        },
        # onupdate=utc_now,
        nullable=False,
        title="Updated At",
        schema_extra={"icon": "clock", "input_type": "datetime"},
    )

    db_xtra: ClassVar[dict] = {}
    sidebar_item: ClassVar[bool] = True
    # Class-level metadata for frontend rendering
    display_name: ClassVar[str] = "Untitled"
    sidebar_icon: ClassVar[str] = "table"

    default_sort_field: ClassVar[str] = "id"
    table_view_fields: ClassVar[List[str]] = []
    detail_page_fields: ClassVar[List[str]] = []
    detail_page_title: ClassVar[Optional[str]] = None
    field_groups: ClassVar[Dict[str, List[str]]] = {}

    create_priviledge: ClassVar[str] = "admin"
    read_priviledge: ClassVar[str] = "admin"
    update_priviledge: ClassVar[str] = "admin"
    delete_priviledge: ClassVar[str] = "admin"

    @classmethod
    def related_records(cls) -> dict[str, List]:
        pass

    @classmethod
    def all(cls) -> List["BaseTable"]:
        return db.all_records(cls)

    @classmethod
    def total_records(cls) -> List["BaseTable"]:
        return len(db.all_records(cls))

    @classmethod
    def search(
        cls,
        search_value: Optional[str] = None,
        sorting_field: Optional[str] = None,
        sort_direction: str = "asc",
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        as_dict: bool = False,
        fields: Optional[List[str]] = None,
    ) -> List[Dict[str, Any]]:
        return db.search(
            cls,
            search_value=search_value,
            sorting_field=sorting_field,
            sort_direction=sort_direction,
            limit=limit,
            offset=offset,
            as_dict=as_dict,
            fields=fields,
        )

    @classmethod
    def filter(cls,
        sorting_field: Optional[str] = None,
        sort_direction: str = "asc",
        limit: Optional[int] = None,
        offset: Optional[int] = None,
        as_dict: bool = False,
        fields: Optional[List[str]] = None,
        exact_match: bool = True,  # This parameter is correctly defined here
        **kwargs
    ) -> List[Dict[str, Any]]:
        return db.filter(
            model=cls,
            sorting_field=sorting_field,
            sort_direction=sort_direction,
            limit=limit,
            offset=offset,
            as_dict=as_dict,
            fields=fields,
            exact_match=exact_match,  # But it needs to be explicitly passed here
            **kwargs
        )

    @classmethod
    def table_view_data(cls, request) -> List[Dict[str, Any]]:
        search_value = None
        page = 1
        per_page = 10
        view_fields = cls.table_view_fields

        if "id" not in view_fields:
            view_fields.append("id")

        if hasattr(request, "query_params"):
            search_value = request.query_params.get("search_value")
            page = int(request.query_params.get("page", 1))
            per_page = int(request.query_params.get("per_page", 10))

        offset = (page - 1) * per_page

        records = cls.search(
            search_value=search_value,
            sorting_field=cls.default_sort_field,
            sort_direction="asc",
            limit=per_page,
            offset=offset,
            as_dict=True,
            fields=view_fields,
        )
        return records

    @classmethod
    def get(cls, id: Any, alt_key: str = None) -> Optional["BaseTable"]:
        return db.get_record(cls, id, alt_key)

    @classmethod
    def update_record(cls, id: Any, data: Dict[str, Any]) -> Dict[str, Any]:
        return db.update_record(cls, id, data)

    @classmethod
    def delete_record(cls, id: Any) -> None:
        db.delete_record(cls, id)

    @classmethod
    def upsert(cls, data: Dict[str, Any]) -> "BaseTable":
        return db.upsert_record(cls, data)

    @classmethod
    def _cast_data(cls, data: List[Dict[str, Any]]) -> List["BaseTable"]:
        return [cls(**item) for item in data]

    def inserted(self) -> Optional["BaseTable"]:
        if db.get_record(type(self), self.id):
            return True
        return False

    def save(self) -> "BaseTable":
        return db.upsert_record(self.__class__, self.dict())

    def delete(self) -> None:
        db.delete_record(self.__class__, self.id)

    def dict(self, *args, **kwargs):
        if database_url.startswith("sqlite"):
            return self.model_dump()
        else:
            return self._dict_with_custom_encoder(set(), *args, **kwargs)

    def _dict_with_custom_encoder(self, processed: Set[int], *args, **kwargs):
        if id(self) in processed:
            return {"id": getattr(self, "id", None)}

        processed.add(id(self))

        data = {}
        for field in self.model_fields:
            value = getattr(self, field)
            if isinstance(value, BaseTable):
                value = value._dict_with_custom_encoder(processed, *args, **kwargs)
            elif isinstance(value, list):
                value = [
                    item._dict_with_custom_encoder(processed, *args, **kwargs)
                    if isinstance(item, BaseTable)
                    else item
                    for item in value
                ]
            elif isinstance(value, dict):
                value = {
                    k: v._dict_with_custom_encoder(processed, *args, **kwargs)
                    if isinstance(v, BaseTable)
                    else v
                    for k, v in value.items()
                }
            elif isinstance(value, datetime):
                value = value.isoformat()

            data[field] = value

        return data

   

