# Copyright [2024] Expedia, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# pylint: disable=unused-argument
# pylint: disable=invalid-name
# pylint: disable=too-many-ancestors
# pylint: disable=no-member
from typing import List, Optional

import numpy as np
import pyspark.sql.functions as F
import tensorflow as tf
from pyspark import keyword_only
from pyspark.ml.param import Param, Params, TypeConverters
from pyspark.sql import DataFrame
from pyspark.sql.types import ArrayType, DataType, DoubleType, FloatType

from kamae.spark.params import MaskValueParams, SingleInputSingleOutputParams
from kamae.spark.utils import single_input_single_output_array_transform
from kamae.tensorflow.layers import MinMaxScaleLayer

from .base import BaseTransformer


class MinMaxScaleParams(MaskValueParams):
    """
    Mixin class containing minimum and maximum parameters needed
    for min/max scaler transformers.
    """

    min = Param(
        Params._dummy(),
        "min",
        "Minimum of the feature values.",
        typeConverter=TypeConverters.toListFloat,
    )

    max = Param(
        Params._dummy(),
        "max",
        "Maximum of the feature values.",
        typeConverter=TypeConverters.toListFloat,
    )

    def setMin(self, value: List[float]) -> "MinMaxScaleParams":
        """
        Sets the parameter min to the given list value.
        Saves the min as a list of floats.

        :param value: List of min values.
        :returns: Instance of class mixed in.
        """
        if None in set(value):
            ids = [i for i, x in enumerate(value) if x is None]
            raise ValueError("Got null Min values at positions: ", ids)
        return self._set(min=value)

    def getMin(self) -> List[float]:
        """
        Gets the min parameter.

        :returns: List of float min values.
        """
        return self.getOrDefault(self.min)

    def setMax(self, value: List[float]) -> "MinMaxScaleParams":
        """
        Sets the parameter max to the given list value.
        Saves the max as a list of floats.

        :param value: List of max values.
        :returns: Instance of class mixed in.
        """
        if None in set(value):
            ids = [i for i, x in enumerate(value) if x is None]
            raise ValueError("Got null Max values at positions: ", ids)
        return self._set(max=value)

    def getMax(self) -> List[float]:
        """
        Gets the max parameter.

        :returns: List of float standard deviation values.
        """
        return self.getOrDefault(self.max)


class MinMaxScaleTransformer(
    BaseTransformer,
    MinMaxScaleParams,
    SingleInputSingleOutputParams,
):
    """
    MinMax scale transformer for use in Spark pipelines.
    This is used to standardize/transform the input column
    to the range [0, 1] using the minimum and maximum values.

    Formula: (x - min)/(max - min)

    WARNING: If the input is an array, we assume that the array has a constant
    shape across all rows.
    """

    @keyword_only
    def __init__(
        self,
        inputCol: Optional[str] = None,
        outputCol: Optional[str] = None,
        inputDtype: Optional[str] = None,
        outputDtype: Optional[str] = None,
        layerName: Optional[str] = None,
        min: Optional[List[float]] = None,
        max: Optional[List[float]] = None,
        maskValue: Optional[float] = None,
    ) -> None:
        """
        Initializes a MinMaxScaleTransformer transformer.

        :param inputCol: Input column name to standardize.
        :param outputCol: Output column name.
        :param inputDtype: Input data type to cast input column to before
        transforming.
        :param outputDtype: Output data type to cast the output column to after
        transforming.
        :param layerName: Name of the layer. Used as the name of the tensorflow layer
        in the keras model.
        :param min: List of minimum values corresponding to the input column.
        :param max: List of maximum values corresponding to the
        input column.
        :param maskValue: Value which should be ignored in the min/max scaling process.
        :returns: None - class instantiated.
        """
        super().__init__()
        self._setDefault(maskValue=None)
        kwargs = self._input_kwargs
        self.setParams(**kwargs)

    @property
    def compatible_dtypes(self) -> Optional[List[DataType]]:
        """
        List of compatible data types for the layer.
        If the computation can be performed on any data type, return None.

        :returns: List of compatible data types for the layer.
        """
        return [FloatType(), DoubleType()]

    def _transform(self, dataset: DataFrame) -> DataFrame:
        """
        Transforms the input dataset using the minimum and maximum values
        to standardize the input column.

        :param dataset: Pyspark dataframe to transform.
        :returns: Pyspark dataframe with the input column standardized,
         named as the output column.
        """
        original_input_datatype = self.get_column_datatype(dataset, self.getInputCol())
        if not isinstance(original_input_datatype, ArrayType):
            input_col = F.array(F.col(self.getInputCol()))
            input_datatype = ArrayType(original_input_datatype)
        else:
            input_col = F.col(self.getInputCol())
            input_datatype = original_input_datatype

        shift = F.array([F.lit(m) for m in self.getMin()])
        scale = F.array(
            [
                F.lit(1.0 / (m1 - m0) if m1 != m0 else 0.0)
                for m0, m1 in zip(self.getMin(), self.getMax())
            ]
        )

        output_col = single_input_single_output_array_transform(
            input_col=input_col,
            input_col_datatype=input_datatype,
            func=lambda x: F.transform(
                x,
                lambda y, i: F.when(y == self.getMaskValue(), y).otherwise(
                    (y - F.lit(shift)[i]) * F.lit(scale)[i]
                ),
            ),
        )

        if not isinstance(original_input_datatype, ArrayType):
            output_col = output_col.getItem(0)

        return dataset.withColumn(self.getOutputCol(), output_col)

    def get_tf_layer(self) -> tf.keras.layers.Layer:
        """
        Gets the tensorflow layer for the min max transformation.

        :returns: Tensorflow keras layer with name equal to the layerName parameter
         that performs the standardization.
        """
        np_min = np.array(self.getMin())
        np_max = np.array(self.getMax())
        mask_value = self.getMaskValue()
        return MinMaxScaleLayer(
            name=self.getLayerName(),
            input_dtype=self.getInputTFDtype(),
            output_dtype=self.getOutputTFDtype(),
            min=np_min,
            max=np_max,
            mask_value=mask_value,
        )
