import os
import platform
import shutil
import subprocess
import sys
from datetime import datetime
from pathlib import Path
from shutil import rmtree
from typing import Optional

import rich

from codeenigma.core import obfuscate_file
from codeenigma.private import NONCE, SECRET_KEY


class Orchestrator:
    """Main orchestrator class for CodeEnigma's obfuscation process.

    This class handles the complete workflow of obfuscating Python modules,
    including file processing, code transformation, and runtime environment setup.

    Attributes:
        module_path: Path to the Python module to be obfuscated.
        output_dir: Directory where obfuscated files will be saved. Defaults to 'dist'.
        expiration_date: Optional datetime object specifying when the obfuscated
            code should expire. If provided, the code will stop working after this date.
    """

    def __init__(
        self,
        module_path: str,
        output_dir: str = "dist",
        expiration_date: Optional[datetime] = None,
    ):
        self.module_path = Path(module_path)
        self.output_dir = Path(output_dir)
        self.expiration_date = expiration_date

    @staticmethod
    def create_obfuscation_file(file_path: str, output_path: str) -> None:
        """Creates an obfuscated version of a single Python file.

        This method takes a Python file, obfuscates its contents, and wraps it
        in a secure execution environment.

        Args:
            file_path: Path to the source Python file to obfuscate.
            output_path: Path where the obfuscated file should be saved.

        Note:
            The generated file will import and use the codeenigma_runtime module
            to securely execute the obfuscated code.
        """
        secure_code = obfuscate_file(file_path)
        runtime_embedded_code = f"""
# This file is auto-generated by codeenigma. Do not edit !!!
# Origin: {Path(file_path).name}

from codeenigma_runtime import execute_secure_code
execute_secure_code({repr(secure_code)}, globals())
"""
        # Write the obfuscated module
        with open(output_path, "w", encoding="utf-8") as f:
            f.write(runtime_embedded_code)

    def _generate_setup(self) -> None:
        """Generates a setup.py file for compiling the codeenigma.pyx file"""
        setup_code = """
import os

from Cython.Build import cythonize
from setuptools import find_packages, setup
from setuptools.extension import Extension

# Get the current directory
current_dir = os.path.dirname(os.path.abspath(__file__))

# Define the extension module
codeenigma_extension = Extension(
    name="codeenigma_runtime",
    sources=[os.path.join(current_dir, "codeenigma_runtime.pyx")],
    extra_compile_args=["-O3", "-fPIC"],
    language="c",
)

# Setup configuration
setup(
    name="codeenigma_runtime",
    version="1.0.0",
    description="Python code obfuscation tool using AES and Base64 encryption",
    ext_modules=cythonize(
        [codeenigma_extension],
        compiler_directives={
            "language_level": 3,
            "boundscheck": False,
            "wraparound": False,
            "initializedcheck": False,
            "nonecheck": False,
            "cdivision": True,
            "c_string_type": "str",
            "c_string_encoding": "utf8",
            "legacy_implicit_noexcept": False,
        },
        nthreads=4,
    ),
    packages=find_packages(),
    zip_safe=False,
)
"""
        with open("codeenigma_setup.py", "w", encoding="utf-8") as f:
            f.write(setup_code)

    def generate_runtime(self) -> None:
        """Generates and compiles the secure runtime environment.

        This method performs the following steps:
        1. Generates the Cython runtime code with embedded encryption
        2. Compiles it to a shared library (.so/.pyd)
        3. Handles cleanup of temporary files
        4. Moves the compiled runtime to the output directory

        The generated runtime is responsible for securely decrypting and
        executing the obfuscated code at runtime.
        """
        if self.expiration_date:
            rich.print(
                f"\n[bold yellow]Note:[/bold yellow] The obfuscated code will expire on {self.expiration_date.strftime('%B %d, %Y %I:%M %p')}\n"
            )
            self.expiration_date = self.expiration_date.isoformat()

        runtime_code = f"""from datetime import datetime, UTC
import base64
import marshal
import zlib
import types
try:
    import rich
    from cryptography.hazmat.primitives.ciphers.aead import AESGCM
except ModuleNotFoundError:
    print("Error: rich and cryptography modules are required to run this code. Please install them using pip install rich cryptography or using poetry add rich cryptography.")
    exit(1)

NONCE = {NONCE}
SECRET_KEY = {SECRET_KEY}


def execute_secure_code(secure_code: bytes, globals_dict=None) -> bytes:

    if '{self.expiration_date}' != 'None':
        EXPIRATION_DATETIME = datetime.fromisoformat('{self.expiration_date}')
        now = datetime.now(tz=UTC)
        if now > EXPIRATION_DATETIME:
            rich.print("[bold red]Code expired on[/bold red]",EXPIRATION_DATETIME.strftime('%B %d, %Y %I:%M %p'),"[bold red] as per build time. This code can't be executed any further. Request the module owner to provide you with a new code. [/bold red]")
            exit(1)

    if globals_dict is None:
        globals_dict = globals()

    # Decrypt the obfuscated code
    aesgcm = AESGCM(SECRET_KEY)
    decrypted = aesgcm.decrypt(NONCE, secure_code, associated_data=None)

    # Decode and decompress
    compressed = base64.b64decode(decrypted)
    marshaled = zlib.decompress(compressed)

    # Unmarshal to get the code object
    code_obj = marshal.loads(marshaled)

    if isinstance(code_obj, types.CodeType):
        exec(code_obj, globals_dict)
    else:
        raise ValueError("Invalid code object in obfuscated module")
"""
        with open("codeenigma_runtime.pyx", "w", encoding="utf-8") as f:
            f.write(runtime_code)

        # Generate the setup.py file and build the runtime
        self._generate_setup()
        subprocess.run(
            [
                "poetry",
                "run",
                "python",
                "codeenigma_setup.py",
                "build_ext",
                "--inplace",
            ],
            check=True,
        )

        # Clean up temporary files
        for temp_file in [
            "codeenigma_setup.py",
            "codeenigma_runtime.pyx",
            "codeenigma_runtime.c",
        ]:
            try:
                os.remove(temp_file)
            except FileNotFoundError:
                pass
        rmtree("build", ignore_errors=True)

        py_version = f"{sys.version_info.major}{sys.version_info.minor}"
        platform_str = platform.system().lower()
        so_file = f"codeenigma_runtime.cpython-{py_version}-{platform_str}.so"

        try:
            shutil.move(so_file, self.output_dir / so_file)
        except FileNotFoundError as e:
            rich.print(f"[bold red]Error moving compiled runtime: {e}[/bold red]")
            raise

    def obfuscate_module(self) -> None:
        """Orchestrates the complete obfuscation process for the target module"""
        if not self.module_path.exists():
            raise FileNotFoundError(f"Module path not found: {self.module_path}")

        rich.print("\n[bold blue]Starting obfuscation process...[/bold blue]")

        # Process all Python files in the module
        py_files = list(self.module_path.glob("**/*.py"))
        if not py_files:
            rich.print("[yellow]No Python files found to obfuscate.[/yellow]")
            return

        for py_file in py_files:
            try:
                rich.print(f"[bold white]Obfuscating {py_file}[/bold white]")
                # Get relative path for module structure
                rel_path = py_file.relative_to(self.module_path.parent)
                output_path = self.output_dir / rel_path
                output_path.parent.mkdir(parents=True, exist_ok=True)

                self.create_obfuscation_file(str(py_file), str(output_path))
            except Exception as e:
                rich.print(f"[red]Error processing {py_file}: {e}[/red]")
                raise

        # Generate the secure runtime environment
        rich.print("\n[bold blue]Generating secure runtime environment...[/bold blue]")
        try:
            self.generate_runtime()
        except Exception as e:
            rich.print(f"[red]Error generating runtime: {e}[/red]")
            raise

        rich.print(
            f"\n[green]✓ Obfuscation complete! Files saved to: {self.output_dir}/[/green]"
        )
