"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToApiGatewayToLambda = void 0;
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const aws_cloudfront_apigateway_1 = require("@aws-solutions-constructs/aws-cloudfront-apigateway");
class CloudFrontToApiGatewayToLambda extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the CloudFrontToApiGatewayToLambda class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] =
            defaults.RegionalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps, props.logGroupProps);
        this.apiGateway.methods.forEach((apiMethod) => {
            // Override the API Gateway Authorization Type from AWS_IAM to NONE
            const child = apiMethod.node.findChild('Resource');
            if (child.authorizationType === 'AWS_IAM') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
                child.cfnOptions.metadata = {
                    cfn_nag: {
                        rules_to_suppress: [{
                                id: 'W59',
                                reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                            }]
                    }
                };
            }
        });
        const apiCloudfront = new aws_cloudfront_apigateway_1.CloudFrontToApiGateway(this, 'CloudFrontToApiGateway', {
            existingApiGatewayObj: this.apiGateway,
            cloudFrontDistributionProps: props.cloudFrontDistributionProps,
            insertHttpSecurityHeaders: props.insertHttpSecurityHeaders
        });
        this.cloudFrontWebDistribution = apiCloudfront.cloudFrontWebDistribution;
        this.edgeLambdaFunctionVersion = apiCloudfront.edgeLambdaFunctionVersion;
        this.cloudFrontLoggingBucket = apiCloudfront.cloudFrontLoggingBucket;
    }
}
exports.CloudFrontToApiGatewayToLambda = CloudFrontToApiGatewayToLambda;
//# sourceMappingURL=data:application/json;base64,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