"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
module.exports = {
    'fails when policy is empty'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synth(), /Policy is empty/);
        test.done();
    },
    'policy with statements'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy', { policyName: 'MyPolicyName' });
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicyName' } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' } } });
        test.done();
    },
    'policy name can be omitted, in which case the logical id will be used'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const user = new lib_1.User(stack, 'MyUser');
        user.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'policy can be attached users, groups and roles and added permissions via props'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            statements: [new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:PutItem'] })],
        });
        assert_1.expect(stack).toMatch({ Resources: { User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } },
                MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }] } } } });
        test.done();
    },
    'idempotent if a principal (user/group/role) is attached twice'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyPolicy');
        p.addStatements(new lib_1.PolicyStatement({ actions: ['*'], resources: ['*'] }));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'users, groups, roles and permissions can be added using methods'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:GetItem'] }));
        assert_1.expect(stack).toMatch({ Resources: { MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }] } },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'policy can be attached to users, groups or role via methods on the principal'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.attachInlinePolicy(policy);
        group.attachInlinePolicy(policy);
        role.attachInlinePolicy(policy);
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'fails if policy name is not unique within a user/group/role'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        // create two policies named Foo and attach them both to the same user/group/role
        const p1 = new lib_1.Policy(stack, 'P1', { policyName: 'Foo' });
        const p2 = new lib_1.Policy(stack, 'P2', { policyName: 'Foo' });
        const p3 = new lib_1.Policy(stack, 'P3'); // uses logicalID as name
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        p1.attachToUser(user);
        p1.attachToGroup(group);
        p1.attachToRole(role);
        // try to attach p2 to all of these and expect to fail
        test.throws(() => p2.attachToUser(user), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToGroup(group), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToRole(role), /A policy named "Foo" is already attached/);
        p3.attachToUser(user);
        p3.attachToGroup(group);
        p3.attachToRole(role);
        test.done();
    },
    'fails if policy is not attached to a principal'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synth(), /Policy must be attached to at least one principal: user, group or role/);
        test.done();
    },
    'generated policy name only uses the last 128 characters of the logical id'(test) {
        test.equal(util_1.generatePolicyName('Foo'), 'Foo');
        const logicalId50 = '[' + dup(50 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId50), logicalId50);
        const logicalId128 = '[' + dup(128 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId128), logicalId128);
        const withPrefix = 'PREFIX' + logicalId128;
        test.equal(util_1.generatePolicyName(withPrefix), logicalId128, 'ensure prefix is omitted');
        test.done();
        function dup(count) {
            let r = '';
            for (let i = 0; i < count; ++i) {
                r += 'x';
            }
            return r;
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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