from typing import Final, TypeAlias

from ._eo_collection import EOCollection
from ._eo_parameters import get_available_parameters

CollectionStr: TypeAlias = str
ProductTypeStr: TypeAlias = str

COLLECTION_PRODUCT_TYPE_DICT: Final[dict[CollectionStr, list[ProductTypeStr]]] = {
    "EarthCAREOrbitData": [],
    "EarthCAREXMETL1DProducts10": ["AUX_MET_1D"],
    "EarthCAREL2Products": [
        "ACM_CAP_2B",
        "ACM_COM_2B",
        "ACM_RT__2B",
        "AC__TC__2B",
        "ALL_3D__2B",
        "ALL_DF__2B",
        "AM__ACD_2B",
        "AM__CTH_2B",
        "ATL_AER_2A",
        "ATL_ALD_2A",
        "ATL_CTH_2A",
        "ATL_EBD_2A",
        "ATL_FM__2A",
        "ATL_ICE_2A",
        "ATL_TC__2A",
        "BMA_FLX_2B",
        "BM__RAD_2B",
        "CPR_APC_2A",
        "CPR_CD__2A",
        "CPR_CLD_2A",
        "CPR_FMR_2A",
        "CPR_TC__2A",
        "MSI_AOT_2A",
        "MSI_CM__2A",
        "MSI_COP_2A",
    ],
    "EarthCAREL2Validated": [
        "AC__TC__2B",
        "AM__ACD_2B",
        "AM__CTH_2B",
        "ATL_AER_2A",
        "ATL_ALD_2A",
        "ATL_CTH_2A",
        "ATL_EBD_2A",
        "ATL_FM__2A",
        "ATL_ICE_2A",
        "ATL_TC__2A",
        "BM__RAD_2B",
        "CPR_CD__2A",
        "CPR_CLD_2A",
        "CPR_FMR_2A",
        "CPR_TC__2A",
        "MSI_AOT_2A",
        "MSI_CM__2A",
        "MSI_COP_2A",
    ],
    "EarthCAREL2Validated_MAAP": [
        "AC__TC__2B",
        "AM__ACD_2B",
        "AM__CTH_2B",
        "ATL_AER_2A",
        "ATL_ALD_2A",
        "ATL_CTH_2A",
        "ATL_EBD_2A",
        "ATL_FM__2A",
        "ATL_ICE_2A",
        "ATL_TC__2A",
        "BM__RAD_2B",
        "CPR_CD__2A",
        "CPR_CLD_2A",
        "CPR_FMR_2A",
        "CPR_TC__2A",
        "MSI_AOT_2A",
        "MSI_CM__2A",
        "MSI_COP_2A",
    ],
    "EarthCAREL2Products_MAAP": [
        "AC__TC__2B",
        "AM__ACD_2B",
        "AM__CTH_2B",
        "ATL_AER_2A",
        "ATL_ALD_2A",
        "ATL_CTH_2A",
        "ATL_EBD_2A",
        "ATL_FM__2A",
        "ATL_ICE_2A",
        "ATL_TC__2A",
        "BM__RAD_2B",
        "CPR_CD__2A",
        "CPR_CLD_2A",
        "CPR_FMR_2A",
        "CPR_TC__2A",
        "MSI_AOT_2A",
        "MSI_CM__2A",
        "MSI_COP_2A",
    ],
    "JAXAL2InstChecked_MAAP": [],
    "JAXAL2Products_MAAP": [],
    "EarthCAREL0L1Products": [
        "ATL_CSC_1B",
        "ATL_DCC_1B",
        "ATL_FSC_1B",
        "ATL_NOM_0_",
        "ATL_NOM_1B",
        "AUX_JSG_1D",
        "AUX_MET_1D",
        "BBR_LIN_1B",
        "BBR_NOM_0_",
        "BBR_NOM_1B",
        "BBR_SNG_1B",
        "BBR_SOL_1B",
        "CPR_NOM_1B",
        "MSI_BBS_1B",
        "MSI_DRK_1B",
        "MSI_NOM_0_",
        "MSI_NOM_1B",
        "MSI_RGR_1C",
        "MSI_SD1_1B",
        "MSI_SD2_1B",
        "MSI_TRF_1B",
        "TLM_NOM_0_",
    ],
    "EarthCAREL2InstChecked_MAAP": [
        "AC__TC__2B",
        "AM__ACD_2B",
        "AM__CTH_2B",
        "ATL_AER_2A",
        "ATL_ALD_2A",
        "ATL_CTH_2A",
        "ATL_EBD_2A",
        "ATL_FM__2A",
        "ATL_ICE_2A",
        "ATL_TC__2A",
        "BM__RAD_2B",
        "CPR_CD__2A",
        "CPR_CLD_2A",
        "CPR_FMR_2A",
        "CPR_TC__2A",
        "MSI_AOT_2A",
        "MSI_CM__2A",
        "MSI_COP_2A",
    ],
    "EarthCAREL0L1Products_MAAP": [
        "ATL_NOM_1B",
        "BBR_LIN_1B",
        "BBR_NOM_1B",
        "BBR_SNG_1B",
        "BBR_SOL_1B",
        "CPR_NOM_1B",
        "MSI_BBS_1B",
        "MSI_DRK_1B",
        "MSI_NOM_1B",
        "MSI_RGR_1C",
        "MSI_SD1_1B",
        "MSI_SD2_1B",
        "MSI_TRF_1B",
    ],
    "EarthCAREL1Validated_MAAP": [
        "ATL_NOM_1B",
        "BBR_NOM_1B",
        "BBR_SNG_1B",
        "CPR_NOM_1B",
        "MSI_NOM_1B",
        "MSI_RGR_1C",
    ],
    "EarthCAREXMETL1DProducts10_MAAP": [],
    "EarthCAREL1InstChecked_MAAP": [
        "ATL_NOM_1B",
        "BBR_NOM_1B",
        "BBR_SNG_1B",
        "CPR_NOM_1B",
        "MSI_NOM_1B",
        "MSI_RGR_1C",
    ],
    "EarthCAREL2InstChecked": [
        "ACM_CAP_2B",
        "ACM_COM_2B",
        "ACM_RT__2B",
        "AC__TC__2B",
        "ALL_3D__2B",
        "ALL_DF__2B",
        "AM__ACD_2B",
        "AM__CTH_2B",
        "ATL_AER_2A",
        "ATL_ALD_2A",
        "ATL_CTH_2A",
        "ATL_EBD_2A",
        "ATL_FM__2A",
        "ATL_ICE_2A",
        "ATL_TC__2A",
        "BMA_FLX_2B",
        "BM__RAD_2B",
        "CPR_APC_2A",
        "CPR_CD__2A",
        "CPR_CLD_2A",
        "CPR_FMR_2A",
        "CPR_TC__2A",
        "MSI_AOT_2A",
        "MSI_CM__2A",
        "MSI_COP_2A",
    ],
    "EarthCAREL1InstChecked": [
        "ATL_CSC_1B",
        "ATL_DCC_1B",
        "ATL_FSC_1B",
        "ATL_NOM_1B",
        "AUX_JSG_1D",
        "AUX_MET_1D",
        "BBR_NOM_1B",
        "BBR_SNG_1B",
        "CPR_NOM_1B",
        "MSI_NOM_1B",
        "MSI_RGR_1C",
    ],
    "JAXAL2Validated_MAAP": [],
    "EarthCAREOrbitData_MAAP": [],
    "EarthCAREAuxiliary_MAAP": [],
    "EarthCAREL1Validated": [
        "ATL_NOM_1B",
        "AUX_JSG_1D",
        "BBR_NOM_1B",
        "BBR_SNG_1B",
        "CPR_NOM_1B",
        "MSI_NOM_1B",
        "MSI_RGR_1C",
    ],
    "JAXAL2Products": [
        "ACM_CLP_2B",
        "AC__CLP_2B",
        "ALL_RAD_2B",
        "ATL_CLA_2A",
        "CPR_CLP_2A",
        "CPR_ECO_2A",
        "MSI_CLP_2A",
    ],
    "JAXAL2Validated": [],
    "EarthCAREAuxiliary": [
        "AUX_CCAM_A",
        "AUX_CCAR_A",
        "AUX_CCAX_A",
        "AUX_CCDC_A",
        "AUX_CCER_B",
        "AUX_CCGT_B",
        "AUX_CCIA_A",
        "AUX_CCIA_B",
        "AUX_CCIA_M",
        "AUX_CCLI_B",
        "AUX_CCMC_A",
        "AUX_CCOG_A",
        "AUX_CCOG_B",
        "AUX_CCPS_A",
        "AUX_CCPS_B",
        "AUX_CCPS_M",
        "AUX_CCSN_B",
        "AUX_CCSX_A",
        "AUX_CSMO_M",
        "AUX_ORBPRE",
        "AUX_ORBRES",
        "AUX_STAT_M",
        "AUX_TBBL_M",
        "AUX_TSCO_M",
        "AUX_VRAD_M",
        "GEO_ATTOBS",
        "GEO_ORBOBS",
        "MPL_ORBSCT",
    ],
    "JAXAL2InstChecked": [],
}
"""To reduce number of performed API request, this variable is by default hardcoded (last updated on 2025-07-16)."""


def get_collection_product_type_dict(
    collections: list[EOCollection], perform_requests: bool = False
) -> dict[CollectionStr, list[ProductTypeStr]]:
    """Retruns a dictionary mapping collection names to lists of available products."""

    if not perform_requests:
        return COLLECTION_PRODUCT_TYPE_DICT

    collection_product_type_dict: dict[CollectionStr, list[ProductTypeStr]] = {}
    for c in collections:
        params = get_available_parameters(c)
        if len(params) == 0:
            collection_product_type_dict[c.name] = []
            continue

        product_type = [p for p in params if p.name == "productType"]
        if len(product_type) == 0:
            collection_product_type_dict[c.name] = []
            continue

        product_type_enum = product_type[0].enum
        if not isinstance(product_type_enum, list):
            collection_product_type_dict[c.name] = []
            continue

        collection_product_type_dict[c.name] = product_type_enum
    return collection_product_type_dict
