"""Network helper class for the network integration."""

from __future__ import annotations

import logging
from typing import Any

from homeassistant.core import HomeAssistant, callback
from homeassistant.helpers.singleton import singleton
from homeassistant.helpers.storage import Store
from homeassistant.util.async_ import create_eager_task
from homeassistant.util.hass_dict import HassKey

from .const import (
    ATTR_CONFIGURED_ADAPTERS,
    DEFAULT_CONFIGURED_ADAPTERS,
    DOMAIN,
    STORAGE_KEY,
    STORAGE_VERSION,
)
from .models import Adapter
from .util import async_load_adapters, enable_adapters, enable_auto_detected_adapters

_LOGGER = logging.getLogger(__name__)

DATA_NETWORK: HassKey[Network] = HassKey(DOMAIN)


@callback
def async_get_loaded_network(hass: HomeAssistant) -> Network:
    """Get network singleton."""
    return hass.data[DATA_NETWORK]


@singleton(DOMAIN)
async def async_get_network(hass: HomeAssistant) -> Network:
    """Get network singleton."""
    network = Network(hass)
    await network.async_setup()
    network.async_configure()

    _LOGGER.debug("Adapters: %s", network.adapters)
    return network


class Network:
    """Network helper class for the network integration."""

    def __init__(self, hass: HomeAssistant) -> None:
        """Initialize the Network class."""
        self._store = Store[dict[str, list[str]]](
            hass, STORAGE_VERSION, STORAGE_KEY, atomic_writes=True
        )
        self._data: dict[str, list[str]] = {}
        self.adapters: list[Adapter] = []

    @property
    def configured_adapters(self) -> list[str]:
        """Return the configured adapters."""
        return self._data.get(ATTR_CONFIGURED_ADAPTERS, DEFAULT_CONFIGURED_ADAPTERS)

    async def async_setup(self) -> None:
        """Set up the network config."""
        storage_load_task = create_eager_task(self.async_load())
        self.adapters = await async_load_adapters()
        await storage_load_task

    @callback
    def async_configure(self) -> None:
        """Configure from storage."""
        if not enable_adapters(self.adapters, self.configured_adapters):
            enable_auto_detected_adapters(self.adapters)

    async def async_reconfig(self, config: dict[str, Any]) -> None:
        """Reconfigure network."""
        self._data[ATTR_CONFIGURED_ADAPTERS] = config[ATTR_CONFIGURED_ADAPTERS]
        self.async_configure()
        await self._async_save()

    async def async_load(self) -> None:
        """Load config."""
        if stored := await self._store.async_load():
            self._data = stored

    async def _async_save(self) -> None:
        """Save preferences."""
        await self._store.async_save(self._data)
