/*
 * Decompiled with CFR 0.152.
 */
package com.siemens.ct.exi.main.cmd;

import com.siemens.ct.exi.main.cmd.BuiltInSchema;
import com.siemens.ct.exi.main.cmd.EXICodec;
import com.siemens.ct.exi.main.cmd.MessageInfo;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

public class EXICodecQuickTest {
    private static EXICodec exi_codec = new EXICodec();
    private static boolean fullDebug = false;
    private static Map<MessageInfo, String> iso15118_2_messageMappings;
    private static Map<MessageInfo, String> iso15118_2_messageMappings_fragmentGrammar;
    private static Map<MessageInfo, String> iso15118_20_messageMappings;
    private static Map<MessageInfo, String> iso15118_20_messageMappings_fragmentGrammar;

    public static void runFullEncodeDecodeTests() {
        EXICodec exi_codec = new EXICodec();
        System.out.println("Version:" + exi_codec.get_version());
        System.out.println("===== ISO15118-2 Tests=====");
        EXICodecQuickTest.runTests(EXICodecQuickTest.getISO15118_2_MessageMapping());
        System.out.println("\n===FRAGMENT GRAMMAR (ISO15118-2) ===");
        EXICodecQuickTest.runTests(EXICodecQuickTest.getISO15118_2_MessageMapping_FragmentGrammar());
        System.out.println("\n===== ISO15118-20 Tests=====");
        EXICodecQuickTest.runTests(EXICodecQuickTest.getISO15118_20_MessageMapping());
        System.out.println("\n===FRAGMENT GRAMMAR (ISO15118-20) ===");
        EXICodecQuickTest.runTests(EXICodecQuickTest.getISO15118_20_MessageMapping_FragmentGrammar());
    }

    public static void runSingleJSONEncodeDecodeTest(String json, BuiltInSchema schema_to_use) {
        System.out.println("Incoming JSON: \n" + json);
        BuiltInSchema schema = schema_to_use;
        long startTime = System.nanoTime();
        byte[] encoded_message = null;
        encoded_message = schema == BuiltInSchema.XSDCore ? exi_codec.encode_signed_info(json) : exi_codec.encode(json, schema);
        long endTime = System.nanoTime();
        if (fullDebug) {
            System.out.println("Time taken(encode): " + (endTime - startTime) / 1000000L);
        }
        if (encoded_message == null) {
            System.out.println(exi_codec.get_last_encoding_error());
        } else {
            System.out.println("Encoded message: " + EXICodecQuickTest.print(encoded_message));
        }
        String decoded_json = "";
        if (encoded_message != null) {
            startTime = System.nanoTime();
            decoded_json = schema == BuiltInSchema.XSDCore ? exi_codec.decode_signed_info(encoded_message) : exi_codec.decode(encoded_message, schema);
            endTime = System.nanoTime();
            if (fullDebug) {
                System.out.println("Time taken(decode): " + (endTime - startTime) / 1000000L);
            }
            if (decoded_json != null) {
                System.out.println("Decoded Message:\n" + decoded_json);
            } else {
                System.out.println("Error decoding stream: " + exi_codec.get_last_decoding_error());
            }
        }
        System.out.println(exi_codec.compareEncodedAndDecodedJSONFiles(json, decoded_json) ? "OK" : "NEEDS CHECKING");
    }

    static byte[] convertToBytes() {
        String str = "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";
        byte[] val = new byte[str.length() / 2];
        int i = 0;
        while (i < val.length) {
            int index = i * 2;
            int j = Integer.parseInt(str.substring(index, index + 2), 16);
            val[i] = (byte)j;
            ++i;
        }
        return val;
    }

    static void runTests(Map<MessageInfo, String> messageMappings) {
        ArrayList<MessageInfo> keySet = new ArrayList<MessageInfo>(messageMappings.keySet());
        int i = 0;
        while (i < keySet.size()) {
            MessageInfo mi = (MessageInfo)keySet.get(i);
            String json = messageMappings.get(mi);
            if (fullDebug) {
                System.out.println("Incoming JSON: \n" + json);
            }
            long startTime = System.nanoTime();
            byte[] encoded_message = exi_codec.encode(json, mi.schema);
            long endTime = System.nanoTime();
            if (fullDebug) {
                System.out.println("Time taken(encode): " + (endTime - startTime) / 1000000L);
            }
            if (encoded_message == null) {
                System.out.println(exi_codec.get_last_encoding_error());
            } else if (fullDebug) {
                System.out.println("Encoded message: " + EXICodecQuickTest.print(encoded_message));
            }
            if (!mi.fragmentMode) {
                String decoded_json = "";
                if (encoded_message != null) {
                    startTime = System.nanoTime();
                    decoded_json = mi.messageType.equals("SignedInfo") ? exi_codec.decode_signed_info(encoded_message) : exi_codec.decode(encoded_message, ((MessageInfo)keySet.get((int)i)).schema);
                    endTime = System.nanoTime();
                    if (fullDebug) {
                        System.out.println("Time taken(decode): " + (endTime - startTime) / 1000000L);
                    }
                    if (decoded_json != null) {
                        if (fullDebug) {
                            System.out.println("Decoded Message:\n" + decoded_json);
                        }
                    } else {
                        System.out.println("Error decoding stream: " + exi_codec.get_last_decoding_error());
                    }
                }
                System.out.println(String.format("%-40s : %s", ((MessageInfo)keySet.get((int)i)).messageType.toString(), exi_codec.compareEncodedAndDecodedJSONFiles(json, decoded_json) ? "OK" : "NEEDS CHECKING"));
            } else {
                System.out.println(String.format("%-40s : %s", ((MessageInfo)keySet.get((int)i)).messageType.toString(), encoded_message != null ? "OK (Needs validating.)" : "NEEDS CHECKING"));
            }
            ++i;
        }
    }

    static Map<MessageInfo, String> getISO15118_20_MessageMapping() {
        if (iso15118_20_messageMappings == null) {
            iso15118_20_messageMappings = Collections.synchronizedMap(new LinkedHashMap());
            iso15118_20_messageMappings.put(new MessageInfo("SupportedAppProtocolReq", BuiltInSchema.AppProtocol), "{\"supportedAppProtocolReq\": {\"AppProtocol\": [{\"ProtocolNamespace\": \"urn:iso:15118:2:2013:MsgDef\", \"VersionNumberMajor\": 2, \"VersionNumberMinor\": 0, \"SchemaID\": 10, \"Priority\": 1}, {\"ProtocolNamespace\": \"urn:iso:std:iso:15118:-20:AC\", \"VersionNumberMajor\": 1, \"VersionNumberMinor\": 0, \"SchemaID\": 20, \"Priority\": 2}]}}");
            iso15118_20_messageMappings.put(new MessageInfo("SupportedAppProtocolRes", BuiltInSchema.AppProtocol), "{\"supportedAppProtocolRes\": {\"ResponseCode\": \"OK_SuccessfulNegotiation\", \"SchemaID\": 10}}");
            iso15118_20_messageMappings.put(new MessageInfo("SessionSetupReq", BuiltInSchema.ISO15118_20_V2G_CI_CommonMessages), "{\"SessionSetupReq\": {\"Header\": {\"SessionID\": \"00\", \"TimeStamp\": 1635706899}, \"EVCCID\": \"WMIV1234567890ABCDEX\"}}");
            iso15118_20_messageMappings.put(new MessageInfo("SessionSetupRes", BuiltInSchema.ISO15118_20_V2G_CI_CommonMessages), "{\"SessionSetupRes\": {\"Header\": {\"SessionID\": \"834BA2D14D305318\", \"TimeStamp\": 1635965564}, \"ResponseCode\": \"OK_NewSessionEstablished\", \"EVSEID\": \"UK123E1234\"}}");
            iso15118_20_messageMappings.put(new MessageInfo("AuthorizationSetupReq", BuiltInSchema.ISO15118_20_V2G_CI_CommonMessages), "{\"AuthorizationSetupReq\": {\"Header\": {\"SessionID\": \"834BA2D14D305318\", \"TimeStamp\": 1635965564}}}");
            iso15118_20_messageMappings.put(new MessageInfo("AuthorizationSetupRes", BuiltInSchema.ISO15118_20_V2G_CI_CommonMessages), "{\"AuthorizationSetupRes\": {\"Header\": {\"SessionID\": \"834BA2D14D305318\", \"TimeStamp\": 1635965565}, \"ResponseCode\": \"OK\", \"AuthorizationServices\": [\"EIM\", \"PnC\"], \"CertificateInstallationService\": true, \"PnC_ASResAuthorizationMode\": {\"GenChallenge\": \"Zm5UeykzrO8sm7LFB8lVxA==\"}, \"EIM_ASResAuthorizationMode\": {}}}");
            iso15118_20_messageMappings.put(new MessageInfo("CertificateInstallationReq", BuiltInSchema.ISO15118_20_V2G_CI_CommonMessages), "{\"CertificateInstallationReq\": {\"Header\": {\"SessionID\": \"5DC94BF6003C58BA\", \"TimeStamp\": 1636461326, \"Signature\": {\"SignedInfo\": {\"CanonicalizationMethod\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}, \"SignatureMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256\"}, \"Reference\": [{\"Transforms\": [{\"Transform\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}}], \"DigestMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmlenc#sha256\"}, \"DigestValue\": \"aGYZjLIwFCgpNhml1UH5UpIotSE5FpUDcmN+0cgXg6E=\", \"URI\": \"#id1\"}]}, \"SignatureValue\": {\"value\": \"MEQCIHyunb6CKRr8c+IWgAMhHaaK8YJvAOYd9GoPBTAbsNCSAiBpdyUFagKJTZgPdxwNvPrOQNkC8bbYJX69EL4Sz2aHWA==\"}}}, \"OEMProvisioningCertificateChain\": {\"Id\": \"id1\", \"Certificate\": \"MIIBwDCCAWWgAwIBAgICMEAwCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNU3ViQ0EyMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowSTEUMBIGA1UEAwwLT0VNUHJvdkNlcnQxDzANBgNVBAoMBlN3aXRjaDELMAkGA1UEBhMCVUsxEzARBgoJkiaJk/IsZAEZFgNPRU0wWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAAQh28fTnaQ+OlxojuQZT/pjJaKu4N8CB4Uz8dND0sIlfu3HkNBb4mOk2b3zgA7cub7OAAVlb1x4hO7fJlEu3/pToz8wPTAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIDiDAdBgNVHQ4EFgQUniuzBygTPlYGcE3UXiyPdBqtkHAwCgYIKoZIzj0EAwIDSQAwRgIhAIkxKc+8WQHqvoYO+4n9LbIc/QJeAogFwqLKS4Pt0t2BAiEAwWjk4oqLA/BVaWgnN/cSlSom/P0YN+Isf4JHvoBzOzo=\", \"SubCertificates\": [{\"Certificate\": \"MIIBwzCCAWmgAwIBAgICMD8wCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNU3ViQ0ExMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowRzESMBAGA1UEAwwJT0VNU3ViQ0EyMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE/IU2sUXeATj5/HcB5k5L1k2Lg10rvjHPEz3J5VF3GUsvEu40vWpOuirxFxE8j2X1pEN+/WA0J0+YsSXP0KzH4KNFMEMwEgYDVR0TAQH/BAgwBgEB/wIBADAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFCNE2aWcqZ5nrQC8qiNLWNeR1nJpMAoGCCqGSM49BAMCA0gAMEUCIAsEEbMnEjUuZ64Patbzz9B7ouugeavjBdR7jdlqv6DFAiEAjyhqzEGtijXNpCnA/pJcAW4EaR3SmBJ6rVIdYQxCVk8=\"}, {\"Certificate\": \"MIIBwzCCAWmgAwIBAgICMD4wCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNUm9vdENBMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowRzESMBAGA1UEAwwJT0VNU3ViQ0ExMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEKuwJGkmrog2b+h2p11GV9cM3cZn92QFoyuvkBAxl4iSW90HGKkfVuXVRojkzBWOMPzXoElVMEcEpB7hugleNOKNFMEMwEgYDVR0TAQH/BAgwBgEB/wIBATAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFAX0j4MPNB9jVG5RmXRkS3EeI4V1MAoGCCqGSM49BAMCA0gAMEUCIHqLCS8cjb9DY/HXdlCMvKH2pBK2K74TbSkXUlHoMZMIAiEApcmmAO33E+oJGX1N+sV8plQtKU5sja8VZxlHf8X1o+c=\"}]}, \"ListOfRootCertificateIDs\": [{\"RootCertificateID\": {\"X509IssuerName\": \"<Name(CN=V2GRootCA,O=Switch,C=UK,DC=V2G)>\", \"X509SerialNumber\": 12345}}], \"MaximumContractCertificateChains\": 3}}");
            iso15118_20_messageMappings.put(new MessageInfo("CertificateInstallationRes", BuiltInSchema.ISO15118_20_V2G_CI_CommonMessages), "{\"CertificateInstallationRes\": {\"Header\": {\"SessionID\": \"00\", \"TimeStamp\": 1}, \"ResponseCode\": \"FAILED_SequenceError\", \"EVSEProcessing\": \"Finished\", \"CPSCertificateChain\": {\"Certificate\": \"\"}, \"SignedInstallationData\": {\"Id\": \"\", \"ContractCertificateChain\": {\"Certificate\": \"\", \"SubCertificates\": [{\"Certificate\": \"\"}]}, \"ECDHCurve\": \"X448\", \"DHPublicKey\": \"\", \"X448_EncryptedPrivateKey\": \"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\"}, \"RemainingContractCertificateChains\": 0}}");
        }
        return iso15118_20_messageMappings;
    }

    static Map<MessageInfo, String> getISO15118_2_MessageMapping() {
        if (iso15118_2_messageMappings == null) {
            iso15118_2_messageMappings = Collections.synchronizedMap(new LinkedHashMap());
            iso15118_2_messageMappings.put(new MessageInfo("SupportedAppProtocolReq", BuiltInSchema.AppProtocol), "{\"supportedAppProtocolReq\": {\"AppProtocol\": [{\"ProtocolNamespace\": \"urn:iso:15118:2:2013:MsgDef\", \"VersionNumberMajor\": 2, \"VersionNumberMinor\": 0, \"SchemaID\": 10, \"Priority\": 1}, {\"ProtocolNamespace\": \"urn:iso:std:iso:15118:-20:AC\", \"VersionNumberMajor\": 1, \"VersionNumberMinor\": 0, \"SchemaID\": 20, \"Priority\": 2}]}}");
            iso15118_2_messageMappings.put(new MessageInfo("SupportedAppProtocolRes", BuiltInSchema.AppProtocol), "{\"supportedAppProtocolRes\": {\"ResponseCode\": \"OK_SuccessfulNegotiation\", \"SchemaID\": 10}}");
            iso15118_2_messageMappings.put(new MessageInfo("SessionSetupReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"00\"}, \"Body\": {\"SessionSetupReq\": {\"EVCCID\": \"8C859028BFC5\"}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("SessionSetupRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"C08CDF36985A7190\"}, \"Body\": {\"SessionSetupRes\": {\"ResponseCode\": \"OK_NewSessionEstablished\", \"EVSEID\": \"UK123E1234\", \"EVSETimeStamp\": 1632218029}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ServiceDiscoveryReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"C08CDF36985A7190\"}, \"Body\": {\"ServiceDiscoveryReq\": {}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("SessionDiscoveryRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"9E3EA6105AE723C5\"}, \"Body\": {\"ServiceDiscoveryRes\": {\"ResponseCode\": \"OK\", \"PaymentOptionList\": [{\"PaymentOption\": \"ExternalPayment\"}, {\"PaymentOption\": \"Contract\"}], \"ChargeService\": {\"ServiceID\": 1, \"ServiceName\": \"AC_DC_Charging\", \"ServiceCategory\": \"EVCharging\", \"FreeService\": false, \"SupportedEnergyTransferMode\": [{\"EnergyTransferMode\": \"AC_single_phase_core\"}, {\"EnergyTransferMode\": \"AC_three_phase_core\"}]}, \"ServiceList\": [{\"Service\": {\"ServiceID\": 2, \"ServiceName\": \"Certificate\", \"ServiceCategory\": \"ContractCertificate\", \"FreeService\": true}}]}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ServiceDetailReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"A53EF57A8CA57A4C\"}, \"Body\": {\"ServiceDetailReq\": {\"ServiceID\": 2}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ServiceDetailRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"A53EF57A8CA57A4C\"}, \"Body\": {\"ServiceDetailRes\": {\"ResponseCode\": \"OK\", \"ServiceID\": 2, \"ServiceParameterList\": {\"ParameterSet\": [{\"ParameterSetID\": 1, \"Parameter\": [{\"Name\": \"Service\", \"stringValue\": \"Installation\"}]}]}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("PaymentServiceSelectionReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"C08CDF36985A7190\"}, \"Body\": {\"PaymentServiceSelectionReq\": {\"SelectedPaymentOption\": {\"PaymentOption\": \"ExternalPayment\"}, \"SelectedServiceList\": [{\"ServiceID\": 1}]}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("PaymentServiceSelectionRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"C08CDF36985A7190\"}, \"Body\": {\"PaymentServiceSelectionRes\": {\"ResponseCode\": \"OK\"}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("AuthorisationReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"C08CDF36985A7190\"}, \"Body\": {\"AuthorizationReq\": {}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("AuthorizationRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"C08CDF36985A7190\"}, \"Body\": {\"AuthorizationRes\": {\"ResponseCode\": \"OK\", \"EVSEProcessing\": \"Finished\"}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ChargeParameterDiscoveryReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"403CEFED6003428C\"}, \"Body\": {\"ChargeParameterDiscoveryReq\": {\"RequestedEnergyTransferMode\": \"AC_three_phase_core\", \"AC_EVChargeParameter\": {\"DepartureTime\": 0, \"EAmount\": {\"Multiplier\": 0, \"Value\": 60, \"Unit\": \"Wh\"}, \"EVMaxVoltage\": {\"Multiplier\": 0, \"Value\": 400, \"Unit\": \"V\"}, \"EVMaxCurrent\": {\"Multiplier\": 0, \"Value\": 16, \"Unit\": \"A\"}, \"EVMinCurrent\": {\"Multiplier\": 0, \"Value\": 10, \"Unit\": \"A\"}}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ChargeParameterDiscoveryRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"225FFA52C1BDAE2F\", \"Signature\": {\"SignedInfo\": {\"CanonicalizationMethod\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}, \"SignatureMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256\"}, \"Reference\": [{\"Transforms\": [{\"Transform\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}}], \"DigestMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmlenc#sha256\"}, \"DigestValue\": \"FOPYeuJCS2wwX9XxiiT+Ly9ofNZY1Zu+UafOP1zwhvA=\", \"URI\": \"#id1\"}]}, \"SignatureValue\": {\"value\": \"MEUCICSta1OLLMIo3/H59qg4Dareuy/+UDr6p6fb/9TRPlFRAiEAx09IkrD7wL1f220dNKwH83WxRY6Yngx1BjAj3vMTDHw=\"}}}, \"Body\": {\"ChargeParameterDiscoveryRes\": {\"ResponseCode\": \"OK\", \"EVSEProcessing\": \"Finished\", \"SAScheduleList\": [{\"SAScheduleTuple\": {\"SAScheduleTupleID\": 1, \"PMaxSchedule\": [{\"PMaxScheduleEntry\": {\"PMax\": {\"Multiplier\": 0, \"Value\": 11000, \"Unit\": \"W\"}, \"RelativeTimeInterval\": {\"start\": 0, \"duration\": 3600}}}], \"SalesTariff\": {\"Id\": \"id1\", \"SalesTariffID\": 10, \"NumEPriceLevels\": 2, \"SalesTariffEntry\": [{\"EPriceLevel\": 1, \"RelativeTimeInterval\": {\"start\": 0}}, {\"EPriceLevel\": 2, \"RelativeTimeInterval\": {\"start\": 1801, \"duration\": 1799}}]}}}], \"AC_EVSEChargeParameter\": {\"AC_EVSEStatus\": {\"NotificationMaxDelay\": 0, \"EVSENotification\": \"None\", \"RCD\": false}, \"EVSENominalVoltage\": {\"Multiplier\": 0, \"Value\": 400, \"Unit\": \"V\"}, \"EVSEMaxCurrent\": {\"Multiplier\": 0, \"Value\": 32, \"Unit\": \"A\"}}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("PowerDeliveryReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"363AD40B7302E70B\"}, \"Body\": {\"PowerDeliveryReq\": {\"ChargeProgress\": \"Start\", \"SAScheduleTupleID\": 1, \"ChargingProfile\": [{\"ProfileEntry\": {\"ChargingProfileEntryStart\": 0, \"ChargingProfileEntryMaxPower\": {\"Multiplier\": 0, \"Value\": 11000, \"Unit\": \"W\"}}}, {\"ProfileEntry\": {\"ChargingProfileEntryStart\": 3600, \"ChargingProfileEntryMaxPower\": {\"Multiplier\": 0, \"Value\": 0, \"Unit\": \"W\"}}}]}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("PowerDeliveryRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"DE47DF7E4C07981D\"}, \"Body\": {\"PowerDeliveryRes\": {\"ResponseCode\": \"OK\", \"AC_EVSEStatus\": {\"NotificationMaxDelay\": 0, \"EVSENotification\": \"None\", \"RCD\": false}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ChargingStatusReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"602BB3E84737BE58\"}, \"Body\": {\"ChargingStatusReq\": {}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ChargingStatusRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"602BB3E84737BE58\"}, \"Body\": {\"ChargingStatusRes\": {\"ResponseCode\": \"OK\", \"EVSEID\": \"UK123E1234\", \"SAScheduleTupleID\": 1, \"ReceiptRequired\": false, \"AC_EVSEStatus\": {\"NotificationMaxDelay\": 0, \"EVSENotification\": \"None\", \"RCD\": false}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("SessionStopReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"772E089BF54E58BB\"}, \"Body\": {\"SessionStopReq\": {\"ChargingSession\": \"Terminate\"}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("SessionStopRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"23BD6F140AC33301\"}, \"Body\": {\"SessionStopRes\": {\"ResponseCode\": \"OK\"}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("SignedInfo", BuiltInSchema.XSDCore), "{\"SignedInfo\": {\"CanonicalizationMethod\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}, \"SignatureMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256\"}, \"Reference\": [{\"Transforms\": [{\"Transform\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}}], \"DigestMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmlenc#sha256\"}, \"DigestValue\": \"/M+0FWWKts921TfZZw5ApQrujKmbpSdiqcuHHtmfric=\", \"URI\": \"#id1\"}]}}");
            iso15118_2_messageMappings.put(new MessageInfo("PaymentDetailsReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"2978369F9B6D3A80\"}, \"Body\": {\"PaymentDetailsReq\": {\"eMAID\": {\"value\": \"DE1ABCD2EF357A\"}, \"ContractSignatureCertChain\": {\"Certificate\": \"MIIByzCCAXKgAwIBAgIBDzAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fU1VCMl9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTIzMDkwMjEzMjEzOFowRDEXMBUGA1UEAwwOREUxQUJDRDJFRjM1N0ExFTATBgNVBAoMDHZlcmlzY28gR21iSDESMBAGCgmSJomT8ixkARkWAk1PMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEaJJ9IFNgGUzaJxpjcZhOYIal3loOXuKwXly+6FlZkaMPdUclMreEAWHKTzrqXzaolWQW8xBHG17/AyynDIhgPqM/MD0wDAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMCA+gwHQYDVR0OBBYEFBPjJKg9jlTxyTUagyAmt3ej13R4MAoGCCqGSM49BAMCA0cAMEQCIFdZO1XVMtwk9/RuJUZHuCoazTSyexAyzKoURmRwNrHQAiB10WMXVNm+6Y4XXQuawoYABuyyoOXUuV4qG+3us3IS3A==\", \"SubCertificates\": [{\"Certificate\": \"MIIB6DCCAY6gAwIBAgIBDTAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fU1VCMV9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTI1MDkwMTEzMjEzOFowWjEgMB4GA1UEAwwXUEtJLTFfQ1JUX01PX1NVQjJfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEjAQBgoJkiaJk/IsZAEZFgJNTzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABA9h8psbI/mdtOsqHTmQsh+oMbkRrR/I6YNeAl0sjWvo64RZiHPA81nv/TYgUz1ZktjLU9cIElLFACj+CS4xalqjRTBDMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQDAgHGMB0GA1UdDgQWBBSQiNhe8kxPxU2YZShx5Ui+pXt3jjAKBggqhkjOPQQDAgNIADBFAiAF0QxlfjVuqomAL/rmzzl1T/d3V7bb2vW8U/ITAMwmnwIhAPvxFX+FMCWUDdnZ9vsg5UdcAoo0RD0uLgXgjoRSRkqv\"}, {\"Certificate\": \"MIIB6DCCAY6gAwIBAgIBDDAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fUk9PVF9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTI1MDkwMTEzMjEzOFowWjEgMB4GA1UEAwwXUEtJLTFfQ1JUX01PX1NVQjFfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEjAQBgoJkiaJk/IsZAEZFgJNTzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABDVarDqUpBKFNnNkssVwvuxnQn3n5mmzKtTQ77ZW8E/yqR6dcWJGfmmOPHYruYZhPFnOGUOej2M6r/0QFhmr55ajRTBDMBIGA1UdEwEB/wQIMAYBAf8CAQEwDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBSo3fgLPv91YCZfWdnipVvlXkP2pDAKBggqhkjOPQQDAgNIADBFAiEAhvm1ML5evLMHizVr04n4h2SakiNFzgGirk9FieQyVVoCIBqA2ZNiEflgdM8KINThkXwONYxJMctgySB6kl/LQ+qL\"}]}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("PaymentDetailsRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"3D61614E45A0004B\"}, \"Body\": {\"PaymentDetailsRes\": {\"ResponseCode\": \"OK\", \"GenChallenge\": \"6Y6/O21CpGOA64OEudrvSw==\", \"EVSETimeStamp\": 1632752516}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("CertificateInstallationReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"2F82FAE56F4442E2\", \"Signature\": {\"SignedInfo\": {\"CanonicalizationMethod\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}, \"SignatureMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256\"}, \"Reference\": [{\"Transforms\": [{\"Transform\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}}], \"DigestMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmlenc#sha256\"}, \"DigestValue\": \"14rVJzskN1olOaOWRREZgabPJ8NFeilm8kaqSBEEgiA=\", \"URI\": \"#id1\"}]}, \"SignatureValue\": {\"value\": \"MEYCIQDPbg+inQpolY0nnY22n84n9+dHE6bYNSoOYTvZcsxI0wIhAMOqeBoDXcwOacDAKXdv5vUvZQm51DgtGpVLvNL4pyO1\"}}}, \"Body\": {\"CertificateInstallationReq\": {\"Id\": \"id1\", \"OEMProvisioningCert\": \"MIIB0TCCAXigAwIBAgIBCTAKBggqhkjOPQQDAjBcMSEwHwYDVQQDDBhQS0ktMV9DUlRfT0VNX1NVQjJfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEzARBgoJkiaJk/IsZAEZFgNPRU0wHhcNMjEwOTAyMTMyMTM4WhcNMjExMDAzMTMyMTM4WjBIMRowGAYDVQQDDBFXMDBWRVJJU0NPMDAwMDAwMDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMRMwEQYKCZImiZPyLGQBGRYDT0VNMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEePb5sm8lLYfueeitKraDIwqVqb526gFTErGN68/SFnQfjlYP8vulZktRg2G1aqt/XHtmThi9vLcV+OaxKRGE0qM/MD0wDAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMCA4gwHQYDVR0OBBYEFPgm1UQLCJg5tT5MmWGtkyG7Bcd9MAoGCCqGSM49BAMCA0cAMEQCIAL9uxIQOgEOjhMizpkzHEg1ZNQolQ0DWKW54bMzkcd4AiBduIbLFOn5HUkvR5IlI5+41ZV//CfECUoYneLO91yp4Q==\", \"ListOfRootCertificateIDs\": [{\"RootCertificateID\": {\"X509IssuerName\": \"<Name(CN=PKI-1_CRT_V2G_ROOT_VALID,O=verisco GmbH,C=DE,DC=V2G)>\", \"X509SerialNumber\": 1}}]}}}} ");
            iso15118_2_messageMappings.put(new MessageInfo("CertificateInstallationRes", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"3554EEC6F808A383\"}, \"Body\": {\"CertificateInstallationRes\": {\"ResponseCode\": \"OK\", \"SAProvisioningCertificateChain\": {\"Certificate\": \"MIIB5zCCAY6gAwIBAgIBGDAKBggqhkjOPQQDAjBdMSIwIAYDVQQDDBlQS0ktMV9DUlRfUFJPVl9TVUIyX1ZBTElEMRUwEwYDVQQKDAx2ZXJpc2NvIEdtYkgxCzAJBgNVBAYTAkRFMRMwEQYKCZImiZPyLGQBGRYDQ1BTMB4XDTIxMDkwMjEzMjE0MFoXDTIxMTAwMzEzMjE0MFowXTEiMCAGA1UEAwwZUEtJLTFfQ1JUX1BST1ZfTEVBRl9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTETMBEGCgmSJomT8ixkARkWA0NQUzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABDah/kgaA+vKDNHU3Vp+6cw8tIxkN/72kIknAj/99oMP9iRM8FHGNLaPzq2J0pcDpzmZexzbToaqbkdQ6p4Y/AmjPzA9MAwGA1UdEwEB/wQCMAAwDgYDVR0PAQH/BAQDAgeAMB0GA1UdDgQWBBQp31ArJBS/DX68uwG2lum2Wuj6sTAKBggqhkjOPQQDAgNHADBEAiBsuwGK22QitrfSXTywE+d4v5ABq3nKlV7MzaSJ3qpvzwIge2zPajjyCextKm3TMhs303uUwmfm99D6mS4tt1uR7wI=\", \"SubCertificates\": [{\"Certificate\": \"MIIB7zCCAZSgAwIBAgIBFzAKBggqhkjOPQQDAjBdMSIwIAYDVQQDDBlQS0ktMV9DUlRfUFJPVl9TVUIxX1ZBTElEMRUwEwYDVQQKDAx2ZXJpc2NvIEdtYkgxCzAJBgNVBAYTAkRFMRMwEQYKCZImiZPyLGQBGRYDQ1BTMB4XDTIxMDkwMjEzMjEzOVoXDTIzMDkwMjEzMjEzOVowXTEiMCAGA1UEAwwZUEtJLTFfQ1JUX1BST1ZfU1VCMl9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTETMBEGCgmSJomT8ixkARkWA0NQUzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABDBjKGihEkDHZtQZXJYXe3XhjtKvHktoK9ulVVljGRYZZPrLUDQQ2lFIm/DKqRIUsrVhlolfu3BGJQUjpUmtk7ujRTBDMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBTjLL8CwMeGQ4nEVnuAc10mDUEJJTAKBggqhkjOPQQDAgNJADBGAiEAl6t2/7D7AB/6kF5tVqK0IDNM6wPsxGFC2U5IxTzngAsCIQClTvD2kXOp7El1TSfG4eVB83D5YimExEIvvONqUKbuMA==\"}, {\"Certificate\": \"MIIB7TCCAZOgAwIBAgIBFjAKBggqhkjOPQQDAjBcMSEwHwYDVQQDDBhQS0ktMV9DUlRfVjJHX1JPT1RfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEzARBgoJkiaJk/IsZAEZFgNWMkcwHhcNMjEwOTAyMTMyMTM5WhcNMjUwOTAxMTMyMTM5WjBdMSIwIAYDVQQDDBlQS0ktMV9DUlRfUFJPVl9TVUIxX1ZBTElEMRUwEwYDVQQKDAx2ZXJpc2NvIEdtYkgxCzAJBgNVBAYTAkRFMRMwEQYKCZImiZPyLGQBGRYDQ1BTMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEuIkmcmiqntz1RGeq9v/nct1vm+7ygfDqO5/YfTrEpwPDbE1Z1X5rmfUvjZ1N0ZXGiy2CtKTGmhd/naQKGvUj4qNFMEMwEgYDVR0TAQH/BAgwBgEB/wIBATAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFFKoBvqCka1q03Wa8y7Uen6EWjuGMAoGCCqGSM49BAMCA0gAMEUCIQDi/X96c6e7HoIJQ/PCzRA9oebrwA0/e00MJ/dOjvpOdwIgbZyOBu8BtwBsv5HTOGNJzJbiUkf8b7Nj85KJjFeHgIw=\"}]}, \"ContractSignatureCertChain\": {\"Id\": \"id1\", \"Certificate\": \"MIIByzCCAXKgAwIBAgIBDzAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fU1VCMl9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTIzMDkwMjEzMjEzOFowRDEXMBUGA1UEAwwOREUxQUJDRDJFRjM1N0ExFTATBgNVBAoMDHZlcmlzY28gR21iSDESMBAGCgmSJomT8ixkARkWAk1PMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEaJJ9IFNgGUzaJxpjcZhOYIal3loOXuKwXly+6FlZkaMPdUclMreEAWHKTzrqXzaolWQW8xBHG17/AyynDIhgPqM/MD0wDAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMCA+gwHQYDVR0OBBYEFBPjJKg9jlTxyTUagyAmt3ej13R4MAoGCCqGSM49BAMCA0cAMEQCIFdZO1XVMtwk9/RuJUZHuCoazTSyexAyzKoURmRwNrHQAiB10WMXVNm+6Y4XXQuawoYABuyyoOXUuV4qG+3us3IS3A==\", \"SubCertificates\": [{\"Certificate\": \"MIIB6DCCAY6gAwIBAgIBDTAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fU1VCMV9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTI1MDkwMTEzMjEzOFowWjEgMB4GA1UEAwwXUEtJLTFfQ1JUX01PX1NVQjJfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEjAQBgoJkiaJk/IsZAEZFgJNTzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABA9h8psbI/mdtOsqHTmQsh+oMbkRrR/I6YNeAl0sjWvo64RZiHPA81nv/TYgUz1ZktjLU9cIElLFACj+CS4xalqjRTBDMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQDAgHGMB0GA1UdDgQWBBSQiNhe8kxPxU2YZShx5Ui+pXt3jjAKBggqhkjOPQQDAgNIADBFAiAF0QxlfjVuqomAL/rmzzl1T/d3V7bb2vW8U/ITAMwmnwIhAPvxFX+FMCWUDdnZ9vsg5UdcAoo0RD0uLgXgjoRSRkqv\"}, {\"Certificate\": \"MIIB6DCCAY6gAwIBAgIBDDAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fUk9PVF9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTI1MDkwMTEzMjEzOFowWjEgMB4GA1UEAwwXUEtJLTFfQ1JUX01PX1NVQjFfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEjAQBgoJkiaJk/IsZAEZFgJNTzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABDVarDqUpBKFNnNkssVwvuxnQn3n5mmzKtTQ77ZW8E/yqR6dcWJGfmmOPHYruYZhPFnOGUOej2M6r/0QFhmr55ajRTBDMBIGA1UdEwEB/wQIMAYBAf8CAQEwDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBSo3fgLPv91YCZfWdnipVvlXkP2pDAKBggqhkjOPQQDAgNIADBFAiEAhvm1ML5evLMHizVr04n4h2SakiNFzgGirk9FieQyVVoCIBqA2ZNiEflgdM8KINThkXwONYxJMctgySB6kl/LQ+qL\"}]}, \"ContractSignatureEncryptedPrivateKey\": {\"Id\": \"id2\", \"value\": \"AA==\"}, \"DHpublickey\": {\"Id\": \"id3\", \"value\": \"AA==\"}, \"eMAID\": {\"Id\": \"id4\", \"value\": \"DE1ABCD2EF357A\"}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("MeteringReceiptReq", BuiltInSchema.ISO15118_2), "{\"V2G_Message\": {\"Header\": {\"SessionID\": \"49B5AEDA5061A7BC\", \"Signature\": {\"SignedInfo\": {\"CanonicalizationMethod\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}, \"SignatureMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256\"}, \"Reference\": [{\"Transforms\": [{\"Transform\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}}], \"DigestMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmlenc#sha256\"}, \"DigestValue\": \"hgBWyaUAlM5Tjrj8Nn8Z4tdC2nP5ctFb/uqkITTnVX0=\", \"URI\": \"#id1\"}]}, \"SignatureValue\": {\"value\": \"MEUCIQDi6nEn1KpkwuDrQqYN8U0MnuZfguYEEEKCp8cNnHtNqAIgPVMar9Qf1ylBRYrvtToWyDGREbPjGKbGsLmpCs7qfrI=\"}}}, \"Body\": {\"MeteringReceiptReq\": {\"Id\": \"id1\", \"SessionID\": \"49B5AEDA5061A7BC\", \"SAScheduleTupleID\": 1, \"MeterInfo\": {\"MeterID\": \"Switch-Meter-123\", \"MeterReading\": 12345, \"TMeter\": 1635433881}}}}}");
            iso15118_2_messageMappings.put(new MessageInfo("ContractSignatureCertChain", BuiltInSchema.ISO15118_2, true), "{\"ContractSignatureCertChain\": {\"Id\": \"id1\", \"Certificate\": \"MIIByzCCAXKgAwIBAgIBDzAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fU1VCMl9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTIzMDkwMjEzMjEzOFowRDEXMBUGA1UEAwwOREUxQUJDRDJFRjM1N0ExFTATBgNVBAoMDHZlcmlzY28gR21iSDESMBAGCgmSJomT8ixkARkWAk1PMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEaJJ9IFNgGUzaJxpjcZhOYIal3loOXuKwXly+6FlZkaMPdUclMreEAWHKTzrqXzaolWQW8xBHG17/AyynDIhgPqM/MD0wDAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMCA+gwHQYDVR0OBBYEFBPjJKg9jlTxyTUagyAmt3ej13R4MAoGCCqGSM49BAMCA0cAMEQCIFdZO1XVMtwk9/RuJUZHuCoazTSyexAyzKoURmRwNrHQAiB10WMXVNm+6Y4XXQuawoYABuyyoOXUuV4qG+3us3IS3A==\", \"SubCertificates\": [{\"Certificate\": \"MIIB6DCCAY6gAwIBAgIBDTAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fU1VCMV9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTI1MDkwMTEzMjEzOFowWjEgMB4GA1UEAwwXUEtJLTFfQ1JUX01PX1NVQjJfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEjAQBgoJkiaJk/IsZAEZFgJNTzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABA9h8psbI/mdtOsqHTmQsh+oMbkRrR/I6YNeAl0sjWvo64RZiHPA81nv/TYgUz1ZktjLU9cIElLFACj+CS4xalqjRTBDMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQDAgHGMB0GA1UdDgQWBBSQiNhe8kxPxU2YZShx5Ui+pXt3jjAKBggqhkjOPQQDAgNIADBFAiAF0QxlfjVuqomAL/rmzzl1T/d3V7bb2vW8U/ITAMwmnwIhAPvxFX+FMCWUDdnZ9vsg5UdcAoo0RD0uLgXgjoRSRkqv\"}, {\"Certificate\": \"MIIB6DCCAY6gAwIBAgIBDDAKBggqhkjOPQQDAjBaMSAwHgYDVQQDDBdQS0ktMV9DUlRfTU9fUk9PVF9WQUxJRDEVMBMGA1UECgwMdmVyaXNjbyBHbWJIMQswCQYDVQQGEwJERTESMBAGCgmSJomT8ixkARkWAk1PMB4XDTIxMDkwMjEzMjEzOFoXDTI1MDkwMTEzMjEzOFowWjEgMB4GA1UEAwwXUEtJLTFfQ1JUX01PX1NVQjFfVkFMSUQxFTATBgNVBAoMDHZlcmlzY28gR21iSDELMAkGA1UEBhMCREUxEjAQBgoJkiaJk/IsZAEZFgJNTzBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABDVarDqUpBKFNnNkssVwvuxnQn3n5mmzKtTQ77ZW8E/yqR6dcWJGfmmOPHYruYZhPFnOGUOej2M6r/0QFhmr55ajRTBDMBIGA1UdEwEB/wQIMAYBAf8CAQEwDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBSo3fgLPv91YCZfWdnipVvlXkP2pDAKBggqhkjOPQQDAgNIADBFAiEAhvm1ML5evLMHizVr04n4h2SakiNFzgGirk9FieQyVVoCIBqA2ZNiEflgdM8KINThkXwONYxJMctgySB6kl/LQ+qL\"}]}}");
            iso15118_2_messageMappings.put(new MessageInfo("ContractSignatureEncryptedPrivateKey", BuiltInSchema.ISO15118_2, true), "{\"ContractSignatureEncryptedPrivateKey\": {\"Id\": \"id2\", \"value\": \"AA==\"}}");
            iso15118_2_messageMappings.put(new MessageInfo("DHpublickey", BuiltInSchema.ISO15118_2, true), "{\"DHpublickey\": {\"Id\": \"id3\", \"value\": \"AA==\"}}");
            iso15118_2_messageMappings.put(new MessageInfo("eMAID", BuiltInSchema.ISO15118_2, true), "{\"eMAID\": {\"Id\": \"id4\", \"value\": \"DE1ABCD2EF357A\"}}");
            iso15118_2_messageMappings.put(new MessageInfo("SalesTariff", BuiltInSchema.ISO15118_2, true), "{\"SalesTariff\": {\"Id\": \"id1\", \"SalesTariffID\": 10, \"NumEPriceLevels\": 2, \"SalesTariffEntry\": [{\"EPriceLevel\": 1, \"RelativeTimeInterval\": {\"start\": 0}}, {\"EPriceLevel\": 2, \"RelativeTimeInterval\": {\"start\": 1801, \"duration\": 1799}}]}}");
        }
        return iso15118_2_messageMappings;
    }

    static Map<MessageInfo, String> getISO15118_2_MessageMapping_FragmentGrammar() {
        if (iso15118_2_messageMappings_fragmentGrammar == null) {
            iso15118_2_messageMappings_fragmentGrammar = Collections.synchronizedMap(new LinkedHashMap());
            iso15118_2_messageMappings_fragmentGrammar.put(new MessageInfo("AuthorizationReq", BuiltInSchema.ISO15118_2, true), "{\"AuthorizationReq\":{\"Id\":\"ID1\",\"GenChallenge\":\"VayS8Hkd0pNuRE1pRx+h2Q==\"}}");
            iso15118_2_messageMappings_fragmentGrammar.put(new MessageInfo("MeteringReceiptReq", BuiltInSchema.ISO15118_2, true), "{\"MeteringReceiptReq\":{\"Id\":\"id1\",\"SessionID\":\"49B5AEDA5061A7BC\",\"SAScheduleTupleID\":1,\"MeterInfo\":{\"MeterID\":\"Switch-Meter-123\",\"MeterReading\":12345,\"TMeter\":1635433881}}}");
            iso15118_2_messageMappings_fragmentGrammar.put(new MessageInfo("CertificateInstallationReq", BuiltInSchema.ISO15118_2, true), "{\"CertificateInstallationReq\": {\"Header\": {\"SessionID\": \"5DC94BF6003C58BA\", \"TimeStamp\": 1636461326, \"Signature\": {\"SignedInfo\": {\"CanonicalizationMethod\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}, \"SignatureMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256\"}, \"Reference\": [{\"Transforms\": [{\"Transform\": {\"Algorithm\": \"http://www.w3.org/TR/canonical-exi/\"}}], \"DigestMethod\": {\"Algorithm\": \"http://www.w3.org/2001/04/xmlenc#sha256\"}, \"DigestValue\": \"aGYZjLIwFCgpNhml1UH5UpIotSE5FpUDcmN+0cgXg6E=\", \"URI\": \"#id1\"}]}, \"SignatureValue\": {\"value\": \"MEQCIHyunb6CKRr8c+IWgAMhHaaK8YJvAOYd9GoPBTAbsNCSAiBpdyUFagKJTZgPdxwNvPrOQNkC8bbYJX69EL4Sz2aHWA==\"}}}, \"OEMProvisioningCertificateChain\": {\"Id\": \"id1\", \"Certificate\": \"MIIBwDCCAWWgAwIBAgICMEAwCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNU3ViQ0EyMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowSTEUMBIGA1UEAwwLT0VNUHJvdkNlcnQxDzANBgNVBAoMBlN3aXRjaDELMAkGA1UEBhMCVUsxEzARBgoJkiaJk/IsZAEZFgNPRU0wWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAAQh28fTnaQ+OlxojuQZT/pjJaKu4N8CB4Uz8dND0sIlfu3HkNBb4mOk2b3zgA7cub7OAAVlb1x4hO7fJlEu3/pToz8wPTAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIDiDAdBgNVHQ4EFgQUniuzBygTPlYGcE3UXiyPdBqtkHAwCgYIKoZIzj0EAwIDSQAwRgIhAIkxKc+8WQHqvoYO+4n9LbIc/QJeAogFwqLKS4Pt0t2BAiEAwWjk4oqLA/BVaWgnN/cSlSom/P0YN+Isf4JHvoBzOzo=\", \"SubCertificates\": [{\"Certificate\": \"MIIBwzCCAWmgAwIBAgICMD8wCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNU3ViQ0ExMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowRzESMBAGA1UEAwwJT0VNU3ViQ0EyMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE/IU2sUXeATj5/HcB5k5L1k2Lg10rvjHPEz3J5VF3GUsvEu40vWpOuirxFxE8j2X1pEN+/WA0J0+YsSXP0KzH4KNFMEMwEgYDVR0TAQH/BAgwBgEB/wIBADAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFCNE2aWcqZ5nrQC8qiNLWNeR1nJpMAoGCCqGSM49BAMCA0gAMEUCIAsEEbMnEjUuZ64Patbzz9B7ouugeavjBdR7jdlqv6DFAiEAjyhqzEGtijXNpCnA/pJcAW4EaR3SmBJ6rVIdYQxCVk8=\"}, {\"Certificate\": \"MIIBwzCCAWmgAwIBAgICMD4wCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNUm9vdENBMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowRzESMBAGA1UEAwwJT0VNU3ViQ0ExMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEKuwJGkmrog2b+h2p11GV9cM3cZn92QFoyuvkBAxl4iSW90HGKkfVuXVRojkzBWOMPzXoElVMEcEpB7hugleNOKNFMEMwEgYDVR0TAQH/BAgwBgEB/wIBATAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFAX0j4MPNB9jVG5RmXRkS3EeI4V1MAoGCCqGSM49BAMCA0gAMEUCIHqLCS8cjb9DY/HXdlCMvKH2pBK2K74TbSkXUlHoMZMIAiEApcmmAO33E+oJGX1N+sV8plQtKU5sja8VZxlHf8X1o+c=\"}]}, \"ListOfRootCertificateIDs\": [{\"RootCertificateID\": {\"X509IssuerName\": \"<Name(CN=V2GRootCA,O=Switch,C=UK,DC=V2G)>\", \"X509SerialNumber\": 12345}}], \"MaximumContractCertificateChains\": 3}}");
        }
        return iso15118_2_messageMappings_fragmentGrammar;
    }

    private static Map<MessageInfo, String> getISO15118_20_MessageMapping_FragmentGrammar() {
        if (iso15118_20_messageMappings_fragmentGrammar == null) {
            iso15118_20_messageMappings_fragmentGrammar = Collections.synchronizedMap(new LinkedHashMap());
            iso15118_20_messageMappings_fragmentGrammar.put(new MessageInfo("OEMProvisioningCertificateChain", BuiltInSchema.ISO15118_20_V2G_CI_CommonMessages, true), "{\"OEMProvisioningCertificateChain\": {\"Id\": \"id1\", \"Certificate\": \"MIIBwDCCAWWgAwIBAgICMEAwCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNU3ViQ0EyMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowSTEUMBIGA1UEAwwLT0VNUHJvdkNlcnQxDzANBgNVBAoMBlN3aXRjaDELMAkGA1UEBhMCVUsxEzARBgoJkiaJk/IsZAEZFgNPRU0wWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAAQh28fTnaQ+OlxojuQZT/pjJaKu4N8CB4Uz8dND0sIlfu3HkNBb4mOk2b3zgA7cub7OAAVlb1x4hO7fJlEu3/pToz8wPTAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIDiDAdBgNVHQ4EFgQUniuzBygTPlYGcE3UXiyPdBqtkHAwCgYIKoZIzj0EAwIDSQAwRgIhAIkxKc+8WQHqvoYO+4n9LbIc/QJeAogFwqLKS4Pt0t2BAiEAwWjk4oqLA/BVaWgnN/cSlSom/P0YN+Isf4JHvoBzOzo=\", \"SubCertificates\": [{\"Certificate\": \"MIIBwzCCAWmgAwIBAgICMD8wCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNU3ViQ0ExMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowRzESMBAGA1UEAwwJT0VNU3ViQ0EyMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE/IU2sUXeATj5/HcB5k5L1k2Lg10rvjHPEz3J5VF3GUsvEu40vWpOuirxFxE8j2X1pEN+/WA0J0+YsSXP0KzH4KNFMEMwEgYDVR0TAQH/BAgwBgEB/wIBADAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFCNE2aWcqZ5nrQC8qiNLWNeR1nJpMAoGCCqGSM49BAMCA0gAMEUCIAsEEbMnEjUuZ64Patbzz9B7ouugeavjBdR7jdlqv6DFAiEAjyhqzEGtijXNpCnA/pJcAW4EaR3SmBJ6rVIdYQxCVk8=\"}, {\"Certificate\": \"MIIBwzCCAWmgAwIBAgICMD4wCgYIKoZIzj0EAwIwRzESMBAGA1UEAwwJT0VNUm9vdENBMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMB4XDTIxMTEwODE3NTY1MVoXDTI1MTEwNzE3NTY1MVowRzESMBAGA1UEAwwJT0VNU3ViQ0ExMQ8wDQYDVQQKDAZTd2l0Y2gxCzAJBgNVBAYTAlVLMRMwEQYKCZImiZPyLGQBGRYDT0VNMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEKuwJGkmrog2b+h2p11GV9cM3cZn92QFoyuvkBAxl4iSW90HGKkfVuXVRojkzBWOMPzXoElVMEcEpB7hugleNOKNFMEMwEgYDVR0TAQH/BAgwBgEB/wIBATAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFAX0j4MPNB9jVG5RmXRkS3EeI4V1MAoGCCqGSM49BAMCA0gAMEUCIHqLCS8cjb9DY/HXdlCMvKH2pBK2K74TbSkXUlHoMZMIAiEApcmmAO33E+oJGX1N+sV8plQtKU5sja8VZxlHf8X1o+c=\"}]}}");
        }
        return iso15118_20_messageMappings_fragmentGrammar;
    }

    private static String print(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        sb.append("[ ");
        byte[] byArray = bytes;
        int n = bytes.length;
        int n2 = 0;
        while (n2 < n) {
            byte b = byArray[n2];
            sb.append(String.format("%02X ", b));
            ++n2;
        }
        sb.append("]");
        return sb.toString();
    }
}

