"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isTableMatch = exports.getNearestFromTableFromPos = exports.getAllNestedFromNodes = exports.findColumnAtPosition = exports.createTablesFromFromNodes = exports.isPosInLocation = exports.getColumnRefByPos = void 0;
const log4js_1 = __importDefault(require("log4js"));
const logger = log4js_1.default.getLogger();
function isNotEmpty(value) {
    return value === null || value === undefined ? false : true;
}
function getColumnRefByPos(columns, pos) {
    var _a;
    return ((_a = columns.find((v) => 
    // guard against ColumnRefNode that don't have a location,
    // for example sql functions that are not known to the parser
    v.location &&
        v.location.start.line === pos.line + 1 &&
        v.location.start.column <= pos.column &&
        v.location.end.line === pos.line + 1 &&
        v.location.end.column >= pos.column)) !== null && _a !== void 0 ? _a : null);
}
exports.getColumnRefByPos = getColumnRefByPos;
function isPosInLocation(location, pos) {
    return (location.start.line === pos.line + 1 &&
        location.start.column <= pos.column &&
        location.end.line === pos.line + 1 &&
        location.end.column >= pos.column);
}
exports.isPosInLocation = isPosInLocation;
function createTablesFromFromNodes(fromNodes) {
    return fromNodes.reduce((p, c) => {
        var _a;
        if (c.type !== 'subquery') {
            return p;
        }
        if (!Array.isArray(c.subquery.columns)) {
            return p;
        }
        const columns = c.subquery.columns
            .map((v) => {
            if (typeof v === 'string') {
                return null;
            }
            return {
                columnName: v.as || (v.expr.type === 'column_ref' && v.expr.column) || '',
                description: 'alias',
            };
        })
            .filter(isNotEmpty);
        return p.concat({
            database: null,
            catalog: null,
            columns: columns !== null && columns !== void 0 ? columns : [],
            tableName: (_a = c.as) !== null && _a !== void 0 ? _a : '',
        });
    }, []);
}
exports.createTablesFromFromNodes = createTablesFromFromNodes;
function findColumnAtPosition(ast, pos) {
    var _a, _b;
    const columns = ast.columns;
    if (Array.isArray(columns)) {
        // columns in select clause
        const columnRefs = columns
            .map((col) => col.expr)
            .filter((expr) => expr.type === 'column_ref');
        if (ast.type === 'select' && ((_a = ast.where) === null || _a === void 0 ? void 0 : _a.expression)) {
            if (ast.where.expression.type === 'column_ref') {
                // columns in where clause
                columnRefs.push(ast.where.expression);
            }
        }
        // column at position
        const columnRef = getColumnRefByPos(columnRefs, pos);
        if (logger.isDebugEnabled())
            logger.debug(JSON.stringify(columnRef));
        return columnRef !== null && columnRef !== void 0 ? columnRef : null;
    }
    else if (columns.type == 'star') {
        if (ast.type === 'select' && ((_b = ast.where) === null || _b === void 0 ? void 0 : _b.expression)) {
            // columns in where clause
            const columnRefs = ast.where.expression.type === 'column_ref' ? [ast.where.expression] : [];
            // column at position
            const columnRef = getColumnRefByPos(columnRefs, pos);
            if (logger.isDebugEnabled())
                logger.debug(JSON.stringify(columnRef));
            return columnRef !== null && columnRef !== void 0 ? columnRef : null;
        }
    }
    return null;
}
exports.findColumnAtPosition = findColumnAtPosition;
/**
 * Recursively pull out the FROM nodes (including sub-queries)
 * @param tableNodes
 * @returns
 */
function getAllNestedFromNodes(tableNodes) {
    return tableNodes.flatMap((tableNode) => {
        var _a;
        let result = [tableNode];
        if (tableNode.type == 'subquery') {
            const subTableNodes = ((_a = tableNode.subquery.from) === null || _a === void 0 ? void 0 : _a.tables) || [];
            result = result.concat(getAllNestedFromNodes(subTableNodes));
        }
        return result;
    });
}
exports.getAllNestedFromNodes = getAllNestedFromNodes;
/**
 * Finds the most deeply nested FROM node that have a range encompasing the position.
 * In cases such as SELECT * FROM T1 JOIN (SELECT * FROM (SELECT * FROM T2 <pos>))
 * We will get a list of nodes like this
 * SELECT * FROM T1
 * (SELECT * FROM
 *    (SELECT * FROM T2))
 * The idea is to reverse the list so that the most nested queries come first. Then
 * apply a filter to keep only the FROM nodes which encompass the position and take
 * the first one from that resulting list.
 * @param fromNodes
 * @param pos
 * @returns
 */
function getNearestFromTableFromPos(fromNodes, pos) {
    var _a;
    return ((_a = fromNodes
        .reverse()
        .filter((tableNode) => isPosInLocation(tableNode.location, pos))
        .shift()) !== null && _a !== void 0 ? _a : null);
}
exports.getNearestFromTableFromPos = getNearestFromTableFromPos;
/**
 * Test if the given table matches the fromNode.
 * @param fromNode
 * @param table
 * @returns
 */
function isTableMatch(fromNode, table) {
    switch (fromNode.type) {
        case 'subquery': {
            if (fromNode.as && fromNode.as !== table.tableName) {
                return false;
            }
            break;
        }
        case 'table': {
            if (fromNode.table && fromNode.table !== table.tableName) {
                return false;
            }
            if (fromNode.db && fromNode.db !== table.database) {
                return false;
            }
            if (fromNode.catalog && fromNode.catalog !== table.catalog) {
                return false;
            }
            break;
        }
        default: {
            return false;
        }
    }
    return true;
}
exports.isTableMatch = isTableMatch;
//# sourceMappingURL=AstUtils.js.map