"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.complete = void 0;
const sql_parser_1 = require("@joe-re/sql-parser");
const log4js_1 = __importDefault(require("log4js"));
const StringUtils_1 = require("./StringUtils");
const getLastToken_1 = require("./utils/getLastToken");
const AstUtils_1 = require("./AstUtils");
const createBasicKeywordCandidates_1 = require("./candidates/createBasicKeywordCandidates");
const createTableCandidates_1 = require("./candidates/createTableCandidates");
const createJoinCandidates_1 = require("./candidates/createJoinCandidates");
const createColumnCandidates_1 = require("./candidates/createColumnCandidates");
const createAliasCandidates_1 = require("./candidates/createAliasCandidates");
const createSelectAllColumnsCandidates_1 = require("./candidates/createSelectAllColumnsCandidates");
const createFunctionCandidates_1 = require("./candidates/createFunctionCandidates");
const createKeywordCandidatesFromExpectedLiterals_1 = require("./candidates/createKeywordCandidatesFromExpectedLiterals");
const createJoinTableCndidates_1 = require("./candidates/createJoinTableCndidates");
const CompletionItemUtils_1 = require("./CompletionItemUtils");
const logger = log4js_1.default.getLogger();
function getFromNodesFromClause(sql) {
    try {
        return (0, sql_parser_1.parseFromClause)(sql);
    }
    catch (_e) {
        // no-op
        return null;
    }
}
class Completer {
    constructor(schema, sql, pos, jupyterLabMode) {
        this.lastToken = '';
        this.candidates = [];
        this.error = null;
        this.isSpaceTriggerCharacter = false;
        this.isDotTriggerCharacter = false;
        this.schema = schema;
        this.sql = sql;
        this.pos = pos;
        this.jupyterLabMode = jupyterLabMode;
    }
    complete() {
        var _a, _b;
        const target = (0, StringUtils_1.getRidOfAfterPosString)(this.sql, this.pos);
        logger.debug(`target: ${target}`);
        this.lastToken = (0, getLastToken_1.getLastToken)(target);
        const idx = this.lastToken.lastIndexOf('.');
        this.isSpaceTriggerCharacter = this.lastToken === '';
        this.isDotTriggerCharacter =
            !this.isSpaceTriggerCharacter && idx == this.lastToken.length - 1;
        try {
            const ast = (0, sql_parser_1.parse)(target);
            this.addCandidatesForParsedStatement(ast);
        }
        catch (_e) {
            logger.debug('error');
            logger.debug(_e);
            if (!(_e instanceof Error)) {
                throw _e;
            }
            if (_e.name !== 'SyntaxError') {
                throw _e;
            }
            const e = _e;
            const parsedFromClause = getFromNodesFromClause(this.sql);
            if (parsedFromClause) {
                const fromNodes = (0, AstUtils_1.getAllNestedFromNodes)(((_a = parsedFromClause === null || parsedFromClause === void 0 ? void 0 : parsedFromClause.from) === null || _a === void 0 ? void 0 : _a.tables) || []);
                const fromNodeOnCursor = (0, AstUtils_1.getNearestFromTableFromPos)(fromNodes, this.pos);
                if (fromNodeOnCursor &&
                    fromNodeOnCursor.type === 'incomplete_subquery') {
                    // Incomplete sub query 'SELECT sub FROM (SELECT e. FROM employees e) sub'
                    this.addCandidatesForIncompleteSubquery(fromNodeOnCursor);
                }
                else {
                    this.addCandidatesForSelectQuery(e, fromNodes);
                    const expectedLiteralNodes = ((_b = e.expected) === null || _b === void 0 ? void 0 : _b.filter((v) => v.type === 'literal')) || [];
                    this.addCandidatesForJoins(expectedLiteralNodes, fromNodes);
                }
            }
            else if (e.message === 'EXPECTED COLUMN NAME') {
                this.addCandidatesForInsert();
            }
            else {
                this.addCandidatesForError(e);
            }
            this.error = {
                label: e.name,
                detail: e.message,
                line: e.location.start.line,
                offset: e.location.start.offset,
            };
        }
        return this.candidates;
    }
    addCandidatesForBasicKeyword() {
        (0, createBasicKeywordCandidates_1.createBasicKeywordCandidates)().forEach((v) => {
            this.addCandidate(v);
        });
    }
    addCandidatesForExpectedLiterals(expected) {
        (0, createKeywordCandidatesFromExpectedLiterals_1.createKeywordCandidatesFromExpectedLiterals)(expected).forEach((v) => {
            this.addCandidate(v);
        });
    }
    addCandidate(item) {
        // A keyword completion can be occured anyplace and need to suppress them.
        if (item.kind &&
            item.kind === CompletionItemUtils_1.ICONS.KEYWORD &&
            !item.label.startsWith(this.lastToken)) {
            return;
        }
        // JupyterLab requires the dot or space character preceeding the <tab> key pressed
        // If the dot or space character are not added to the label then searching
        // in the list of suggestion does not work.
        // Here we fix this issue by adding the dot or space character
        // to the filterText and insertText.
        // TODO: report this issue to JupyterLab-LSP project.
        if (this.jupyterLabMode) {
            const text = item.insertText || item.label;
            if (this.isSpaceTriggerCharacter) {
                item.insertText = ' ' + text;
                item.filterText = ' ' + text;
            }
            else if (this.isDotTriggerCharacter) {
                item.insertText = '.' + text;
                item.filterText = '.' + text;
            }
        }
        this.candidates.push(item);
    }
    addCandidatesForTables(tables, onFromClause) {
        (0, createTableCandidates_1.createTableCandidates)(tables, this.lastToken, onFromClause).forEach((item) => {
            this.addCandidate(item);
        });
    }
    addCandidatesForColumnsOfAnyTable(tables) {
        (0, createColumnCandidates_1.createCandidatesForColumnsOfAnyTable)(tables, this.lastToken).forEach((item) => {
            this.addCandidate(item);
        });
    }
    addCandidatesForIncompleteSubquery(incompleteSubquery) {
        const parsedFromClause = getFromNodesFromClause(incompleteSubquery.text);
        try {
            (0, sql_parser_1.parse)(incompleteSubquery.text);
        }
        catch (e) {
            if (!(e instanceof Error)) {
                throw e;
            }
            if (e.name !== 'SyntaxError') {
                throw e;
            }
            const fromText = incompleteSubquery.text;
            const newPos = parsedFromClause
                ? {
                    line: this.pos.line - (incompleteSubquery.location.start.line - 1),
                    column: this.pos.column - incompleteSubquery.location.start.column + 1,
                }
                : { line: 0, column: 0 };
            const completer = new Completer(this.schema, fromText, newPos, this.jupyterLabMode);
            completer.complete().forEach((item) => this.addCandidate(item));
        }
    }
    /**
     * INSERT INTO TABLE1 (C
     */
    addCandidatesForInsert() {
        this.addCandidatesForColumnsOfAnyTable(this.schema.tables);
    }
    addCandidatesForError(e) {
        var _a;
        const expectedLiteralNodes = ((_a = e.expected) === null || _a === void 0 ? void 0 : _a.filter((v) => v.type === 'literal')) || [];
        this.addCandidatesForExpectedLiterals(expectedLiteralNodes);
        this.addCandidatesForFunctions();
        this.addCandidatesForTables(this.schema.tables, false);
    }
    addCandidatesForSelectQuery(e, fromNodes) {
        var _a;
        const subqueryTables = (0, AstUtils_1.createTablesFromFromNodes)(fromNodes);
        const schemaAndSubqueries = this.schema.tables.concat(subqueryTables);
        this.addCandidatesForSelectStar(fromNodes, schemaAndSubqueries);
        const expectedLiteralNodes = ((_a = e.expected) === null || _a === void 0 ? void 0 : _a.filter((v) => v.type === 'literal')) || [];
        this.addCandidatesForExpectedLiterals(expectedLiteralNodes);
        this.addCandidatesForFunctions();
        this.addCandidatesForScopedColumns(fromNodes, schemaAndSubqueries);
        this.addCandidatesForAliases(fromNodes);
        this.addCandidatesForTables(schemaAndSubqueries, true);
        if (logger.isDebugEnabled())
            logger.debug(`candidates for error returns: ${JSON.stringify(this.candidates)}`);
    }
    addCandidatesForJoins(expected, fromNodes) {
        (0, createJoinTableCndidates_1.createJoinTablesCandidates)(this.schema.tables, expected, fromNodes, this.lastToken).forEach((v) => {
            this.addCandidate(v);
        });
    }
    addCandidatesForParsedDeleteStatement(ast) {
        if ((0, AstUtils_1.isPosInLocation)(ast.table.location, this.pos)) {
            this.addCandidatesForTables(this.schema.tables, false);
        }
        else if (ast.where &&
            (0, AstUtils_1.isPosInLocation)(ast.where.expression.location, this.pos)) {
            const expr = ast.where.expression;
            if (expr.type === 'column_ref') {
                this.addCandidatesForColumnsOfAnyTable(this.schema.tables);
            }
        }
    }
    addCandidatesForParsedDropStatement(ast) {
        if ((0, AstUtils_1.isPosInLocation)(ast.table.location, this.pos)) {
            this.addCandidatesForTables(this.schema.tables, false);
        }
    }
    addCandidatesForParsedAlterTableStatement(ast) {
        if (ast.command.type === 'alter_table_drop_column') {
            if ((0, AstUtils_1.isPosInLocation)(ast.command.column.location, this.pos)) {
                const table = this.schema.tables.find((v) => v.tableName === ast.table);
                this.addCandidatesForColumnsOfAnyTable(table ? [table] : this.schema.tables);
            }
        }
    }
    addCandidatesForParsedSelectQuery(ast) {
        var _a;
        this.addCandidatesForBasicKeyword();
        if (Array.isArray(ast.columns)) {
            this.addCandidate((0, CompletionItemUtils_1.toCompletionItemForKeyword)('FROM'));
            this.addCandidate((0, CompletionItemUtils_1.toCompletionItemForKeyword)('AS'));
        }
        if (!ast.distinct) {
            this.addCandidate((0, CompletionItemUtils_1.toCompletionItemForKeyword)('DISTINCT'));
        }
        const columnRef = (0, AstUtils_1.findColumnAtPosition)(ast, this.pos);
        if (!columnRef) {
            this.addJoinCondidates(ast);
        }
        else {
            const parsedFromClause = getFromNodesFromClause(this.sql);
            const fromNodes = ((_a = parsedFromClause === null || parsedFromClause === void 0 ? void 0 : parsedFromClause.from) === null || _a === void 0 ? void 0 : _a.tables) || [];
            const subqueryTables = (0, AstUtils_1.createTablesFromFromNodes)(fromNodes);
            const schemaAndSubqueries = this.schema.tables.concat(subqueryTables);
            if (columnRef.table) {
                // We know what table/alias this column belongs to
                // Find the corresponding table and suggest it's columns
                this.addCandidatesForScopedColumns(fromNodes, schemaAndSubqueries);
            }
            else {
                // Column is not scoped to a table/alias yet
                // Could be an alias, a talbe or a function
                this.addCandidatesForAliases(fromNodes);
                this.addCandidatesForTables(schemaAndSubqueries, true);
                this.addCandidatesForFunctions();
            }
        }
        if (logger.isDebugEnabled())
            logger.debug(`parse query returns: ${JSON.stringify(this.candidates)}`);
    }
    addCandidatesForParsedStatement(ast) {
        if (logger.isDebugEnabled())
            logger.debug(`getting candidates for parse query ast: ${JSON.stringify(ast)}`);
        if (!ast.type) {
            this.addCandidatesForBasicKeyword();
        }
        else if (ast.type === 'delete') {
            this.addCandidatesForParsedDeleteStatement(ast);
        }
        else if (ast.type === 'select') {
            this.addCandidatesForParsedSelectQuery(ast);
        }
        else if (ast.type === 'alter_table') {
            this.addCandidatesForParsedAlterTableStatement(ast);
        }
        else if (ast.type === 'drop_table') {
            this.addCandidatesForParsedDropStatement(ast);
        }
        else {
            console.log(`AST type not supported yet: ${ast.type}`);
        }
    }
    addJoinCondidates(ast) {
        (0, createJoinCandidates_1.createJoinCondidates)(ast, this.schema.tables, this.pos, this.lastToken).forEach((v) => {
            this.addCandidate(v);
        });
    }
    addCandidatesForFunctions() {
        console.time('addCandidatesForFunctions');
        (0, createFunctionCandidates_1.createFunctionCandidates)(this.schema.functions, this.lastToken).forEach((v) => {
            this.addCandidate(v);
        });
        console.timeEnd('addCandidatesForFunctions');
    }
    addCandidatesForSelectStar(fromNodes, tables) {
        console.time('addCandidatesForSelectStar');
        (0, createSelectAllColumnsCandidates_1.createSelectAllColumnsCandidates)(fromNodes, tables, this.lastToken).forEach((v) => {
            this.addCandidate(v);
        });
        console.timeEnd('addCandidatesForSelectStar');
    }
    addCandidatesForScopedColumns(fromNodes, tables) {
        console.time('addCandidatesForScopedColumns');
        (0, createColumnCandidates_1.createCandidatesForScopedColumns)(fromNodes, tables, this.lastToken).forEach((v) => {
            this.addCandidate(v);
        });
        console.timeEnd('addCandidatesForScopedColumns');
    }
    addCandidatesForAliases(fromNodes) {
        (0, createAliasCandidates_1.createAliasCandidates)(fromNodes, this.lastToken).forEach((v) => {
            this.addCandidate(v);
        });
    }
}
function complete(sql, pos, schema = { tables: [], functions: [] }, jupyterLabMode = false) {
    console.time('complete');
    if (logger.isDebugEnabled())
        logger.debug(`complete: ${sql}, ${JSON.stringify(pos)}`);
    const completer = new Completer(schema, sql, pos, jupyterLabMode);
    const candidates = completer.complete();
    console.timeEnd('complete');
    return { candidates: candidates, error: completer.error };
}
exports.complete = complete;
//# sourceMappingURL=complete.js.map