"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createServer = exports.createServerWithConnection = void 0;
const fs = __importStar(require("fs"));
const path_1 = __importDefault(require("path"));
const process_1 = __importDefault(require("process"));
const vscode_languageserver_1 = require("vscode-languageserver");
const protocol_1 = require("vscode-languageserver-protocol/lib/common/protocol");
const vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const sqlint_1 = require("sqlint");
const log4js_1 = __importDefault(require("log4js"));
const cache_1 = __importDefault(require("./cache"));
const complete_1 = require("./complete");
const createDiagnostics_1 = __importDefault(require("./createDiagnostics"));
const createConnection_1 = __importDefault(require("./createConnection"));
const SettingStore_1 = __importDefault(require("./SettingStore"));
const getDatabaseClient_1 = __importDefault(require("./database_libs/getDatabaseClient"));
const initializeLogging_1 = __importDefault(require("./initializeLogging"));
const Sqlite3Client_1 = require("./database_libs/Sqlite3Client");
const TRIGGER_CHARATER = '.';
function createServerWithConnection(connection, debug = false) {
    (0, initializeLogging_1.default)(debug);
    const logger = log4js_1.default.getLogger();
    const documents = new vscode_languageserver_1.TextDocuments(vscode_languageserver_textdocument_1.TextDocument);
    documents.listen(connection);
    let schema = { tables: [], functions: [] };
    let hasConfigurationCapability = false;
    let rootPath = '';
    let lintConfig;
    // Read schema file
    function readJsonSchemaFile(filePath) {
        if (filePath[0] === '~') {
            const home = process_1.default.env.HOME || '';
            filePath = path_1.default.join(home, filePath.slice(1));
        }
        logger.info(`loading schema file: ${filePath}`);
        const data = fs.readFileSync(filePath, 'utf8').replace(/^\ufeff/u, '');
        try {
            schema = JSON.parse(data);
        }
        catch (e) {
            const err = e;
            logger.error('failed to read schema file ' + err.message);
            connection.sendNotification('sqlLanguageServer.error', {
                message: 'Failed to read schema file: ' + filePath + ' error: ' + err.message,
            });
            throw e;
        }
    }
    function readAndMonitorJsonSchemaFile(filePath) {
        fs.watchFile(filePath, () => {
            logger.info(`change detected, reloading schema file: ${filePath}`);
            readJsonSchemaFile(filePath);
        });
        // The readJsonSchemaFile function can throw exceptions so
        // read file only after setting up monitoring
        readJsonSchemaFile(filePath);
    }
    async function makeDiagnostics(document) {
        const hasRules = !!lintConfig && Object.prototype.hasOwnProperty.call(lintConfig, 'rules');
        const diagnostics = (0, createDiagnostics_1.default)(document.uri, document.getText(), hasRules ? lintConfig : null);
        connection.sendDiagnostics(diagnostics);
    }
    documents.onDidChangeContent(async (params) => {
        logger.debug(`onDidChangeContent: ${params.document.uri}, ${params.document.version}`);
        makeDiagnostics(params.document);
    });
    connection.onInitialize((params) => {
        const capabilities = params.capabilities;
        // JupyterLab sends didChangeConfiguration information
        // using both the workspace.configuration and
        // workspace.didChangeConfiguration
        hasConfigurationCapability =
            !!capabilities.workspace &&
                (!!capabilities.workspace.configuration ||
                    !!capabilities.workspace.didChangeConfiguration);
        logger.debug(`onInitialize: ${params.rootPath}`);
        rootPath = params.rootPath || '';
        return {
            capabilities: {
                textDocumentSync: 1,
                completionProvider: {
                    resolveProvider: true,
                    triggerCharacters: [TRIGGER_CHARATER],
                },
                renameProvider: true,
                codeActionProvider: true,
                executeCommandProvider: {
                    commands: [
                        'sqlLanguageServer.switchDatabaseConnection',
                        'sqlLanguageServer.fixAllFixableProblems',
                    ],
                },
            },
        };
    });
    connection.onInitialized(async () => {
        var _a;
        SettingStore_1.default.getInstance().on('change', async () => {
            logger.debug('onInitialize: receive change event from SettingStore');
            try {
                try {
                    connection.sendNotification('sqlLanguageServer.finishSetup', {
                        personalConfig: SettingStore_1.default.getInstance().getPersonalConfig(),
                        config: SettingStore_1.default.getInstance().getSetting(),
                    });
                }
                catch (e) {
                    logger.error(e);
                }
                const setting = SettingStore_1.default.getInstance().getSetting();
                if (setting.adapter == 'json') {
                    // Loading schema from json file
                    const path = setting.filename || '';
                    if (path == '') {
                        logger.error('filename must be provided');
                        connection.sendNotification('sqlLanguageServer.error', {
                            message: 'filename must be provided',
                        });
                        throw 'filename must be provided';
                    }
                    readAndMonitorJsonSchemaFile(path);
                }
                else {
                    // Else get schema form database client
                    try {
                        const client = (0, getDatabaseClient_1.default)(SettingStore_1.default.getInstance().getSetting());
                        schema = await client.getSchema();
                        logger.debug('get schema', JSON.stringify(schema));
                    }
                    catch (e) {
                        logger.error('failed to get schema info');
                        if (e instanceof Sqlite3Client_1.RequireSqlite3Error) {
                            connection.sendNotification('sqlLanguageServer.error', {
                                message: 'Need to rebuild sqlite3 module.',
                            });
                        }
                        throw e;
                    }
                }
            }
            catch (e) {
                logger.error(e);
            }
        });
        const connections = (hasConfigurationCapability &&
            ((_a = (await connection.workspace.getConfiguration({
                section: 'sqlLanguageServer',
            }))) === null || _a === void 0 ? void 0 : _a.connections)) ||
            [];
        if (connections.length > 0) {
            SettingStore_1.default.getInstance().setSettingFromWorkspaceConfig(connections);
        }
        else if (rootPath) {
            SettingStore_1.default.getInstance().setSettingFromFile(`${process_1.default.env.HOME}/.config/sql-language-server/.sqllsrc.json`, `${rootPath}/.sqllsrc.json`, rootPath || '');
        }
    });
    connection.onDidChangeConfiguration((change) => {
        var _a;
        logger.debug('onDidChangeConfiguration', JSON.stringify(change));
        if (!hasConfigurationCapability) {
            return;
        }
        if (!Object.prototype.hasOwnProperty.call(change.settings, 'sqlLanguageServer')) {
            logger.debug('onDidChangeConfiguration', "it doesn't have sqlLanguageServer property");
            return;
        }
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        const sqlLanguageServerSetting = change.settings
            .sqlLanguageServer;
        const connections = ((_a = sqlLanguageServerSetting.connections) !== null && _a !== void 0 ? _a : []);
        if (connections.length > 0) {
            SettingStore_1.default.getInstance().setSettingFromWorkspaceConfig(connections);
        }
        // On configuration changes we retrieve the lint config
        const lint = sqlLanguageServerSetting.lint;
        lintConfig = lint;
        if (lint === null || lint === void 0 ? void 0 : lint.rules) {
            documents.all().forEach((v) => {
                makeDiagnostics(v);
            });
        }
    });
    connection.onCompletion((docParams) => {
        var _a, _b, _c;
        // Make sure the client does not send use completion request for characters
        // other than the dot which we asked for.
        if (((_a = docParams.context) === null || _a === void 0 ? void 0 : _a.triggerKind) == protocol_1.CompletionTriggerKind.TriggerCharacter) {
            if (((_b = docParams.context) === null || _b === void 0 ? void 0 : _b.triggerCharacter) != TRIGGER_CHARATER) {
                return [];
            }
        }
        const text = (_c = documents.get(docParams.textDocument.uri)) === null || _c === void 0 ? void 0 : _c.getText();
        if (!text) {
            return [];
        }
        logger.debug(text || '');
        const pos = {
            line: docParams.position.line,
            column: docParams.position.character,
        };
        const setting = SettingStore_1.default.getInstance().getSetting();
        const candidates = (0, complete_1.complete)(text, pos, schema, setting.jupyterLabMode).candidates;
        if (logger.isDebugEnabled())
            logger.debug('onCompletion returns: ' + JSON.stringify(candidates));
        return candidates;
    });
    connection.onCodeAction((params) => {
        const lintResult = cache_1.default.findLintCacheByRange(params.textDocument.uri, params.range);
        if (!lintResult) {
            return [];
        }
        const document = documents.get(params.textDocument.uri);
        if (!document) {
            return [];
        }
        const text = document.getText();
        if (!text) {
            return [];
        }
        function toPosition(text, offset) {
            const lines = text.slice(0, offset).split('\n');
            return vscode_languageserver_types_1.Position.create(lines.length - 1, lines[lines.length - 1].length);
        }
        const fixes = Array.isArray(lintResult.lint.fix)
            ? lintResult.lint.fix
            : [lintResult.lint.fix];
        if (fixes.length === 0) {
            return [];
        }
        const action = vscode_languageserver_types_1.CodeAction.create(`fix: ${lintResult.diagnostic.message}`, {
            documentChanges: [
                vscode_languageserver_types_1.TextDocumentEdit.create({ uri: params.textDocument.uri, version: document.version }, fixes.map((v) => {
                    const edit = v.range.startOffset === v.range.endOffset
                        ? vscode_languageserver_types_1.TextEdit.insert(toPosition(text, v.range.startOffset), v.text)
                        : vscode_languageserver_types_1.TextEdit.replace({
                            start: toPosition(text, v.range.startOffset),
                            end: toPosition(text, v.range.endOffset),
                        }, v.text);
                    return edit;
                })),
            ],
        }, vscode_languageserver_types_1.CodeActionKind.QuickFix);
        action.diagnostics = params.context.diagnostics;
        return [action];
    });
    connection.onCompletionResolve((item) => {
        return item;
    });
    connection.onExecuteCommand((request) => {
        var _a;
        logger.debug(`received executeCommand request: ${request.command}, ${request.arguments}`);
        if (request.command === 'switchDatabaseConnection' ||
            request.command === 'sqlLanguageServer.switchDatabaseConnection') {
            try {
                SettingStore_1.default.getInstance().changeConnection((request.arguments && ((_a = request.arguments[0]) === null || _a === void 0 ? void 0 : _a.toString())) || '');
            }
            catch (e) {
                const err = e;
                connection.sendNotification('sqlLanguageServer.error', {
                    message: err.message,
                });
            }
        }
        else if (request.command === 'fixAllFixableProblems' ||
            request.command === 'sqlLanguageServer.fixAllFixableProblems') {
            const uri = request.arguments ? request.arguments[0] : null;
            if (!uri) {
                connection.sendNotification('sqlLanguageServer.error', {
                    message: 'fixAllFixableProblems: Need to specify uri',
                });
                return;
            }
            const document = documents.get(uri.toString());
            const text = document === null || document === void 0 ? void 0 : document.getText();
            if (!text) {
                logger.debug('Failed to get text');
                return;
            }
            const result = JSON.parse((0, sqlint_1.lint)({ formatType: 'json', text, fix: true }));
            if (result.length === 0 && result[0].fixedText) {
                logger.debug("There's no fixable problems");
                return;
            }
            logger.debug('Fix all fixable problems', text, result[0].fixedText);
            connection.workspace.applyEdit({
                documentChanges: [
                    vscode_languageserver_types_1.TextDocumentEdit.create({ uri: uri.toString(), version: document.version }, [
                        vscode_languageserver_types_1.TextEdit.replace({
                            start: vscode_languageserver_types_1.Position.create(0, 0),
                            end: vscode_languageserver_types_1.Position.create(Number.MAX_VALUE, Number.MAX_VALUE),
                        }, result[0].fixedText),
                    ]),
                ],
            });
        }
    });
    connection.listen();
    logger.info('start sql-languager-server');
    return connection;
}
exports.createServerWithConnection = createServerWithConnection;
function createServer(params = {}) {
    var _a;
    const connection = (0, createConnection_1.default)((_a = params.method) !== null && _a !== void 0 ? _a : 'node-ipc');
    return createServerWithConnection(connection, params.debug);
}
exports.createServer = createServer;
//# sourceMappingURL=createServer.js.map