"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const pg_1 = __importDefault(require("pg"));
const log4js_1 = __importDefault(require("log4js"));
const AbstractClient_1 = __importDefault(require("./AbstractClient"));
const logger = log4js_1.default.getLogger();
class PosgresClient extends AbstractClient_1.default {
    constructor(settings) {
        super(settings);
        this.connection = null;
    }
    get DefaultPort() {
        return 5432;
    }
    get DefaultHost() {
        return '127.0.0.1';
    }
    get DefaultUser() {
        return 'postgres';
    }
    connect() {
        const client = new pg_1.default.Client({
            user: this.settings.user || '',
            host: this.settings.host || '',
            database: this.settings.database || '',
            password: this.settings.password || '',
            port: this.settings.port || 5432,
        });
        return new Promise((resolve, reject) => {
            client.connect((err) => {
                if (err) {
                    logger.debug('Failed to connect to postgresql server');
                    logger.error(err);
                    reject(err);
                    return;
                }
                this.connection = client;
                logger.debug('Success to connect to postgresql server');
                resolve(true);
            });
        });
    }
    disconnect() {
        if (this.connection) {
            this.connection.end();
        }
        this.connection = null;
    }
    async getTables() {
        const sql = `
      SELECT c.relname as table_name FROM pg_class c LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       WHERE n.nspname = 'public'
         AND c.relkind IN ('r','v','m','f')
    `;
        return new Promise((resolve, reject) => {
            if (!this.connection) {
                reject(new Error("Don't have database connection."));
                return;
            }
            this.connection.query(sql, (err, results) => {
                if (err) {
                    reject(new Error(err.message));
                    return;
                }
                const tables = results.rows.map((v) => v[`table_name`]);
                resolve(tables);
            });
        });
    }
    getColumns(tableName) {
        const sql = `
    SELECT
      a.attname as field,
      format_type(a.atttypid, a.atttypmod) as type,
      pg_get_expr(d.adbin, d.adrelid) as default,
      a.attnotnull as null,
      col_description(a.attrelid, a.attnum) AS comment
    FROM pg_attribute a
      LEFT JOIN pg_attrdef d ON a.attrelid = d.adrelid AND a.attnum = d.adnum
      LEFT JOIN pg_type t ON a.atttypid = t.oid
      LEFT JOIN pg_collation c ON a.attcollation = c.oid AND a.attcollation <> t.typcollation
    WHERE a.attrelid = '"${tableName}"'::regclass
      AND a.attnum > 0 AND NOT a.attisdropped
    ORDER BY a.attnum
    `;
        return new Promise((resolve, reject) => {
            if (!this.connection) {
                reject(new Error("Don't have database connection."));
                return;
            }
            this.connection.query(sql, (err, results) => {
                if (err) {
                    reject(new Error(err.message));
                    return;
                }
                resolve(results.rows);
            });
        });
    }
}
exports.default = PosgresClient;
//# sourceMappingURL=PostgresClient.js.map