"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequireSqlite3Error = void 0;
const log4js_1 = __importDefault(require("log4js"));
const AbstractClient_1 = __importDefault(require("./AbstractClient"));
const logger = log4js_1.default.getLogger();
class RequireSqlite3Error extends Error {
    constructor(message) {
        super(message);
        this.name = 'RequireSQLite3Error';
    }
}
exports.RequireSqlite3Error = RequireSqlite3Error;
class Sqlite3Client extends AbstractClient_1.default {
    constructor(settings) {
        super(settings);
        this.connection = null;
    }
    get DefaultPort() {
        return 0;
    }
    get DefaultHost() {
        return '';
    }
    get DefaultUser() {
        return '';
    }
    connect() {
        if (!this.settings.filename) {
            throw new Error('Need to specify filename to use sqlite3 connection.');
        }
        try {
            // use commonjs to avoid dynamic import build error
            // eslint-disable-next-line @typescript-eslint/no-var-requires
            const sqlite3 = require('sqlite3');
            this.connection = new sqlite3.Database(this.settings.filename, sqlite3.OPEN_READONLY);
        }
        catch (e) {
            logger.error('Sqlite3Client: failed to connect to database', e);
            if (e instanceof Error) {
                throw new RequireSqlite3Error(e.message);
            }
        }
        return true;
    }
    disconnect() {
        if (this.connection) {
            this.connection.close();
        }
        this.connection = null;
    }
    getTables() {
        const sql = `SELECT name FROM sqlite_master WHERE type='table'`;
        return new Promise((resolve, reject) => {
            if (!this.connection) {
                reject(new Error("Don't have database connection."));
                return;
            }
            this.connection.all(sql, (err, rows) => {
                if (err) {
                    reject(new Error(err.message));
                    return;
                }
                logger.debug('Sqlite3Clinet: done to get table names', rows);
                const tables = rows.map((v) => v.name);
                resolve(tables);
            });
        });
    }
    getColumns(tableName) {
        const sql = `SELECT * FROM pragma_table_info('${tableName}')`;
        return new Promise((resolve, reject) => {
            if (!this.connection) {
                reject(new Error("Don't have database connection."));
                return;
            }
            this.connection.all(sql, (err, rows) => {
                if (err) {
                    reject(new Error(err.message));
                    return;
                }
                logger.debug('Sqlite3Clinet: done to get column names', rows);
                const columns = rows.map((v) => ({
                    field: v.name,
                    type: v.type,
                    null: v.notnull ? 'Yes' : 'No',
                    default: v.dflt_value,
                    comment: v.pk ? 'PRIMARY KEY' : '',
                }));
                resolve(columns);
            });
        });
    }
}
exports.default = Sqlite3Client;
//# sourceMappingURL=Sqlite3Client.js.map