"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.lint = void 0;
const chalk_1 = __importDefault(require("chalk"));
const rules_1 = require("../rules");
const fixer_1 = require("../fixer");
const utils_1 = require("./utils");
const loadConfig_1 = require("./loadConfig");
function pluralize(word, count) {
    return count === 1 ? word : `${word}s`;
}
function formatStylish(result) {
    const targetResult = result.filter((v) => v.diagnostics.length > 0);
    let output = '\n', errorCount = 0, warningCount = 0;
    if (targetResult.length === 0) {
        return output;
    }
    targetResult.forEach((v) => {
        output += chalk_1.default.underline(v.filepath) + '\n';
        v.diagnostics.forEach((v2) => {
            const position = chalk_1.default.dim(`${v2.location.start.line}:${v2.location.start.offset}`);
            const messageType = v2.errorLevel === rules_1.ErrorLevel.Error
                ? chalk_1.default.red('error')
                : chalk_1.default.yellow('warning');
            const message = v2.message;
            const ruleName = chalk_1.default.dim(v2.rulename);
            output += `  ${position} ${messageType} ${message} ${ruleName}\n`;
            if (v2.errorLevel === rules_1.ErrorLevel.Error)
                errorCount++;
            else if (v2.errorLevel === rules_1.ErrorLevel.Warn)
                warningCount++;
        });
    });
    output += '\n';
    const total = errorCount + warningCount;
    output += chalk_1.default.bold.red([
        `\u2716 ${total} ${pluralize('problem', total)}`,
        `(${errorCount} ${pluralize('error', errorCount)},`,
        `${warningCount} ${pluralize('warning', warningCount)})`,
    ].join(' '));
    return output;
}
function lint(params) {
    const { path, formatType, configPath, outputFile, text, configObject } = params;
    const files = path ? (0, utils_1.getFileList)(path) : [];
    if (files.length === 0 && !text) {
        throw new Error(`No files matching. path: ${path}`);
    }
    const config = configObject
        ? (0, loadConfig_1.convertToConfig)(configObject)
        : (0, loadConfig_1.loadConfig)(configPath || process.cwd());
    let result = text
        ? [{ filepath: 'text', diagnostics: (0, rules_1.execute)(text, config) }]
        : files
            .map((v) => {
            const diagnostics = (0, rules_1.execute)((0, utils_1.readFile)(v), config);
            return { filepath: v, diagnostics: diagnostics };
        })
            .flat();
    let output = '';
    if (params.fix) {
        const fixedResult = result.map((v) => {
            const MAX_AUTOFIX_LOOP = 3;
            function getFixDescriptions(diagnostics) {
                return diagnostics.map((v) => v.fix).flat();
            }
            function fix(text, descriptions, loop = MAX_AUTOFIX_LOOP) {
                if (loop === 0) {
                    return text;
                }
                const fixed = (0, fixer_1.applyFixes)(text, descriptions);
                const nextDiagnostics = (0, rules_1.execute)(fixed, config);
                const newDescriptions = getFixDescriptions(nextDiagnostics);
                if (newDescriptions.length === 0) {
                    return fixed;
                }
                return fix(fixed, newDescriptions, --loop);
            }
            const fixedText = fix(params.text || (0, utils_1.readFile)(v.filepath), getFixDescriptions(v.diagnostics));
            const diagnostics = (0, rules_1.execute)(fixedText, config);
            return { filepath: v.filepath, diagnostics, fixedText };
        });
        if (!text) {
            fixedResult.forEach((v) => {
                (0, utils_1.writeFile)(v.filepath, v.fixedText);
            });
        }
        result = fixedResult;
    }
    switch (formatType) {
        case 'stylish':
            output = formatStylish(result);
            break;
        case 'json':
            output = JSON.stringify(result);
            break;
        default:
            throw new Error(`unsupported formatType: ${formatType}`);
    }
    if (outputFile) {
        (0, utils_1.writeFile)(outputFile, output);
    }
    return output;
}
exports.lint = lint;
//# sourceMappingURL=lint.js.map