"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createContext = exports.execute = exports.ErrorLevel = void 0;
const sql_parser_1 = require("@joe-re/sql-parser");
const log4js_1 = __importDefault(require("log4js"));
const fixer_1 = require("../fixer");
const reservedWordCase_1 = require("./reservedWordCase");
const spaceSurroundingOperators_1 = require("./spaceSurroundingOperators");
const linebreakAfterClauseKeyword_1 = require("./linebreakAfterClauseKeyword");
const columnNewLine_1 = require("./columnNewLine");
const alignColumnToTheFirst_1 = require("./alignColumnToTheFirst");
const whereClauseNewLine_1 = require("./whereClauseNewLine");
const alignWhereClauseToTheFirst_1 = require("./alignWhereClauseToTheFirst");
const requireAsToRenameColumn_1 = require("./requireAsToRenameColumn");
const logger = log4js_1.default.getLogger();
var ErrorLevel;
(function (ErrorLevel) {
    ErrorLevel[ErrorLevel["Off"] = 0] = "Off";
    ErrorLevel[ErrorLevel["Warn"] = 1] = "Warn";
    ErrorLevel[ErrorLevel["Error"] = 2] = "Error";
})(ErrorLevel = exports.ErrorLevel || (exports.ErrorLevel = {}));
let rules = [];
function execute(sql, config) {
    rules = [];
    registerRule(reservedWordCase_1.reservedWordCase, config, sql);
    registerRule(spaceSurroundingOperators_1.spaceSurroundingOperators, config, sql);
    registerRule(linebreakAfterClauseKeyword_1.linebreakAfterClauseKeyword, config, sql);
    registerRule(columnNewLine_1.columnNewLine, config, sql);
    registerRule(alignColumnToTheFirst_1.alignColumnToTheFirst, config, sql);
    registerRule(whereClauseNewLine_1.whereClauseNewLine, config, sql);
    registerRule(alignWhereClauseToTheFirst_1.alignWhereClauseToTheFirst, config, sql);
    registerRule(requireAsToRenameColumn_1.requireAsToRenameColumn, config, sql);
    try {
        const ast = (0, sql_parser_1.parseAll)(sql);
        return ast
            .map((v) => {
            return walk(v);
        })
            .flat();
    }
    catch (e) {
        logger.debug(e);
        return [];
    }
}
exports.execute = execute;
function registerRule(rule, config, sql) {
    if (config.rules[rule.meta.name] &&
        config.rules[rule.meta.name].level >= ErrorLevel.Warn) {
        const _config = {
            level: config.rules[rule.meta.name].level,
            option: config.rules[rule.meta.name].option,
        };
        rules.push({ rule: rule, config: _config, sql });
    }
}
function apply(node) {
    let diagnostics = [];
    rules.forEach(({ rule, config, sql }) => {
        if (config.level === ErrorLevel.Off) {
            return;
        }
        if (node.type === rule.meta.type) {
            let ruleResult = rule.create(createContext(sql, node, config));
            if (!ruleResult) {
                return;
            }
            if (!Array.isArray(ruleResult)) {
                ruleResult = [ruleResult];
            }
            const _diagnostics = ruleResult.map((v) => {
                const fix = v.fix ? v.fix((0, fixer_1.createFixer)()) : [];
                return {
                    location: v.location,
                    message: v.message,
                    errorLevel: config.level,
                    fix: Array.isArray(fix) ? fix : [fix],
                    rulename: rule.meta.name,
                };
            });
            diagnostics = diagnostics.concat(_diagnostics).flat();
        }
    });
    return diagnostics.filter((v) => !!v);
}
function walk(node, diagnostics = []) {
    if (!node || typeof node !== 'object' || !node.type) {
        return diagnostics;
    }
    diagnostics = diagnostics.concat(apply(node));
    Object.values(node).forEach((v) => {
        diagnostics = walk(v, diagnostics);
    });
    return diagnostics;
}
function createContext(sql, node, ruleConfig) {
    return {
        getSQL: function (range, options) {
            if (!range) {
                return sql;
            }
            const start = options && options.before
                ? range.start.offset - options.before
                : range.start.offset;
            const end = options && options.after
                ? range.end.offset + options.after
                : range.end.offset;
            return sql.slice(start, end);
        },
        getAfterSQL: function (range) {
            return sql.slice(range.end.offset);
        },
        getBeforeSQL: function (range) {
            return sql.slice(0, range.start.offset);
        },
        node: node,
        config: ruleConfig,
    };
}
exports.createContext = createContext;
//# sourceMappingURL=index.js.map