"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionsBoundary = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const iam_generated_1 = require("./iam.generated");
/**
 * Modify the Permissions Boundaries of Users and Roles in a construct tree
 *
 * ```ts
 * const policy = iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess');
 * iam.PermissionsBoundary.of(this).apply(policy);
 * ```
 */
class PermissionsBoundary {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Access the Permissions Boundaries of a construct tree
     */
    static of(scope) {
        return new PermissionsBoundary(scope);
    }
    /**
     * Apply the given policy as Permissions Boundary to all Roles and Users in
     * the scope.
     *
     * Will override any Permissions Boundaries configured previously; in case
     * a Permission Boundary is applied in multiple scopes, the Boundary applied
     * closest to the Role wins.
     */
    apply(boundaryPolicy) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_IManagedPolicy(boundaryPolicy);
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyOverride('PermissionsBoundary', boundaryPolicy.managedPolicyArn);
                }
            },
        });
    }
    /**
     * Remove previously applied Permissions Boundaries
     */
    clear() {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyDeletionOverride('PermissionsBoundary');
                }
            },
        });
    }
}
exports.PermissionsBoundary = PermissionsBoundary;
_a = JSII_RTTI_SYMBOL_1;
PermissionsBoundary[_a] = { fqn: "@aws-cdk/aws-iam.PermissionsBoundary", version: "1.145.0" };
//# sourceMappingURL=data:application/json;base64,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