import { Resource, SecretValue } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IGroup } from './group';
import { IIdentity } from './identity-base';
import { IManagedPolicy } from './managed-policy';
import { Policy } from './policy';
import { PolicyStatement } from './policy-statement';
import { AddToPrincipalPolicyResult, IPrincipal, PrincipalPolicyFragment } from './principals';
/**
 * Represents an IAM user
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
 */
export interface IUser extends IIdentity {
    /**
     * The user's name
     * @attribute
     */
    readonly userName: string;
    /**
     * The user's ARN
     * @attribute
     */
    readonly userArn: string;
    /**
     * Adds this user to a group.
     */
    addToGroup(group: IGroup): void;
}
/**
 * Properties for defining an IAM user
 */
export interface UserProps {
    /**
     * Groups to add this user to. You can also use `addToGroup` to add this
     * user to a group.
     *
     * @default - No groups.
     */
    readonly groups?: IGroup[];
    /**
     * A list of managed policies associated with this role.
     *
     * You can add managed policies later using
     * `addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName(policyName))`.
     *
     * @default - No managed policies.
     */
    readonly managedPolicies?: IManagedPolicy[];
    /**
     * The path for the user name. For more information about paths, see IAM
     * Identifiers in the IAM User Guide.
     *
     * @default /
     */
    readonly path?: string;
    /**
     * AWS supports permissions boundaries for IAM entities (users or roles).
     * A permissions boundary is an advanced feature for using a managed policy
     * to set the maximum permissions that an identity-based policy can grant to
     * an IAM entity. An entity's permissions boundary allows it to perform only
     * the actions that are allowed by both its identity-based policies and its
     * permissions boundaries.
     *
     * @link https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html#cfn-iam-role-permissionsboundary
     * @link https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html
     *
     * @default - No permissions boundary.
     */
    readonly permissionsBoundary?: IManagedPolicy;
    /**
     * A name for the IAM user. For valid values, see the UserName parameter for
     * the CreateUser action in the IAM API Reference. If you don't specify a
     * name, AWS CloudFormation generates a unique physical ID and uses that ID
     * for the user name.
     *
     * If you specify a name, you cannot perform updates that require
     * replacement of this resource. You can perform updates that require no or
     * some interruption. If you must replace the resource, specify a new name.
     *
     * If you specify a name, you must specify the CAPABILITY_NAMED_IAM value to
     * acknowledge your template's capabilities. For more information, see
     * Acknowledging IAM Resources in AWS CloudFormation Templates.
     *
     * @default - Generated by CloudFormation (recommended)
     */
    readonly userName?: string;
    /**
     * The password for the user. This is required so the user can access the
     * AWS Management Console.
     *
     * You can use `SecretValue.plainText` to specify a password in plain text or
     * use `secretsmanager.Secret.fromSecretAttributes` to reference a secret in
     * Secrets Manager.
     *
     * @default - User won't be able to access the management console without a password.
     */
    readonly password?: SecretValue;
    /**
     * Specifies whether the user is required to set a new password the next
     * time the user logs in to the AWS Management Console.
     *
     * If this is set to 'true', you must also specify "initialPassword".
     *
     * @default false
     */
    readonly passwordResetRequired?: boolean;
}
/**
 * Represents a user defined outside of this stack.
 */
export interface UserAttributes {
    /**
     * The ARN of the user.
     *
     * Format: arn:<partition>:iam::<account-id>:user/<user-name-with-path>
     */
    readonly userArn: string;
}
/**
 * Define a new IAM user
 */
export declare class User extends Resource implements IIdentity, IUser {
    /**
     * Import an existing user given a username.
     *
     * @param scope construct scope
     * @param id construct id
     * @param userName the username of the existing user to import
     */
    static fromUserName(scope: Construct, id: string, userName: string): IUser;
    /**
     * Import an existing user given a user ARN.
     *
     * If the ARN comes from a Token, the User cannot have a path; if so, any attempt
     * to reference its username will fail.
     *
     * @param scope construct scope
     * @param id construct id
     * @param userArn the ARN of an existing user to import
     */
    static fromUserArn(scope: Construct, id: string, userArn: string): IUser;
    /**
     * Import an existing user given user attributes.
     *
     * If the ARN comes from a Token, the User cannot have a path; if so, any attempt
     * to reference its username will fail.
     *
     * @param scope construct scope
     * @param id construct id
     * @param attrs the attributes of the user to import
     */
    static fromUserAttributes(scope: Construct, id: string, attrs: UserAttributes): IUser;
    readonly grantPrincipal: IPrincipal;
    readonly principalAccount: string | undefined;
    readonly assumeRoleAction: string;
    /**
     * An attribute that represents the user name.
     * @attribute
     */
    readonly userName: string;
    /**
     * An attribute that represents the user's ARN.
     * @attribute
     */
    readonly userArn: string;
    /**
     * Returns the permissions boundary attached  to this user
     */
    readonly permissionsBoundary?: IManagedPolicy;
    readonly policyFragment: PrincipalPolicyFragment;
    private readonly groups;
    private readonly managedPolicies;
    private readonly attachedPolicies;
    private defaultPolicy?;
    constructor(scope: Construct, id: string, props?: UserProps);
    /**
     * Adds this user to a group.
     */
    addToGroup(group: IGroup): void;
    /**
     * Attaches a managed policy to the user.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy: IManagedPolicy): void;
    /**
     * Attaches a policy to this user.
     */
    attachInlinePolicy(policy: Policy): void;
    /**
     * Adds an IAM statement to the default policy.
     *
     * @returns true
     */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
    addToPolicy(statement: PolicyStatement): boolean;
    private parseLoginProfile;
}
