"""

--------------------

This script demonstrates basic operations on object like creating new
object, placing it into a view layer, selecting it and making it active.

```../examples/bpy.types.Object.py```

bpy.types.*

:glob:


--------------------

PropertyGroups are the base class for dynamically defined sets of properties.

They can be used to extend existing blender data with your own types which can
be animated, accessed from the user interface and from python.

[NOTE]
The values assigned to blender data are saved to disk but the class
definitions are not, this means whenever you load blender the class needs
to be registered too.
This is best done by creating an add-on which loads on startup and registers
your properties.

[NOTE]
PropertyGroups must be registered before assigning them to blender data.

Property types used in class declarations are all in bpy.props



```../examples/bpy.types.PropertyGroup.py```


--------------------

Here is an example of a simple menu. Menus differ from panels in that they must
reference from a header, panel or another menu.

Notice the 'CATEGORY_MT_name' in  Menu.bl_idname, this is a naming
convention for menus.

[NOTE]
Menu subclasses must be registered before referencing them from blender.

[NOTE]
Menus have their Layout.operator_context initialized as
'EXEC_REGION_WIN' rather than 'INVOKE_DEFAULT' (see Execution Context <operator-execution_context>).
If the operator context needs to initialize inputs from the
Operator.invoke function, then this needs to be explicitly set.

```../examples/bpy.types.Menu.py```


--------------------

This menu demonstrates some different functions.

```../examples/bpy.types.Menu.1.py```


--------------------

When creating menus for add-ons you can't reference menus
in Blender's default scripts.
Instead, the add-on can add menu items to existing menus.

The function menu_draw acts like Menu.draw.

```../examples/bpy.types.Menu.2.py```


--------------------

Preset menus are simply a convention that uses a menu sub-class
to perform the common task of managing presets.

This example shows how you can add a preset menu.

This example uses the object display options,
however you can use properties defined by your own scripts too.

```../examples/bpy.types.Menu.3.py```


--------------------

This example enables you to insert your own menu entry into the common
right click menu that you get while hovering over a value field,
color, string, etc.

To make the example work, you have to first select an object
then right click on an user interface element (maybe a color in the
material properties) and choose *Execute Custom Action*.

Executing the operator will then print all values.

```../examples/bpy.types.Menu.4.py```


--------------------

The mesh data is accessed in object mode and intended for compact storage,
for more flexible mesh editing from python see bmesh.

Blender stores 4 main arrays to define mesh geometry.

* Mesh.vertices (3 points in space)
* Mesh.edges (reference 2 vertices)
* Mesh.loops (reference a single vertex and edge)
* Mesh.polygons: (reference a range of loops)

Each polygon reference a slice in the loop array, this way, polygons do not store vertices or corner data such as UV's directly,
only a reference to loops that the polygon uses.

Mesh.loops, Mesh.uv_layers Mesh.vertex_colors are all aligned so the same polygon loop
indices can be used to find the UV's and vertex colors as with as the vertices.

To compare mesh API options see: NGons and Tessellation Faces <info_gotcha_mesh_faces>

This example script prints the vertices and UV's for each polygon, assumes the active object is a mesh with UVs.

```../examples/bpy.types.Mesh.py```

```../examples/bpy.types.AddonPreferences.1.py```


--------------------

This script is the UIList subclass used to show material slots, with a bunch of additional commentaries.

Notice the name of the class, this naming convention is similar as the one for panels or menus.

[NOTE]
UIList subclasses must be registered for blender to use them.

```../examples/bpy.types.UIList.1.py```


--------------------

This script is an extended version of the UIList

 subclass used to show vertex groups. It is not used 'as is',
because iterating over all vertices in a 'draw' function is a very bad idea for UI performances! However, it's a good
example of how to create/use filtering/reordering callbacks.

```../examples/bpy.types.UIList.2.py```


--------------------

The NodeTree.poll function determines if a node tree is visible
in the given context (similar to how Panel.poll
and Menu.poll define visibility). If it returns False,
the node tree type will not be selectable in the node editor.

A typical condition for shader nodes would be to check the active render engine
of the scene and only show nodes of the renderer they are designed for.

```../examples/bpy.types.NodeTree.py```


--------------------

The Image data-block is a shallow wrapper around image or video file(s)
(on disk, as packed data, or generated).

All actual data like the pixel buffer, size, resolution etc. is
cached in an imbuf.types.ImBuf image buffer (or several buffers
in some cases, like UDIM textures, multi-views, animations...).

Several properties and functions of the Image data-block are then actually
using/modifying its image buffer, and not the Image data-block itself.

[WARNING]
One key limitation is that image buffers are not shared between different
Image data-blocks, and they are not duplicated when copying an image.
So until a modified image buffer is saved on disk, duplicating its Image
data-block will not propagate the underlying buffer changes to the new Image.

This example script generates an Image data-block with a given size,
change its first pixel, rescale it, and duplicates the image.

The duplicated image still has the same size and colors as the original image
at its creation, all editing in the original image's buffer is 'lost' in its copy.

```../examples/bpy.types.Image.py```


--------------------

This script shows simple operator which prints a message.

Since the operator only has an Operator.execute function it takes no
user input.

[NOTE]
Operator subclasses must be registered before accessing them from blender.

```../examples/bpy.types.Operator.py```


--------------------

Operator.invoke is used to initialize the operator from the context
at the moment the operator is called.
invoke() is typically used to assign properties which are then used by
execute().
Some operators don't have an execute() function, removing the ability to be
repeated from a script or macro.

This example shows how to define an operator which gets mouse input to
execute a function and that this operator can be invoked or executed from
the python api.

Also notice this operator defines its own properties, these are different
to typical class properties because blender registers them with the
operator, to use as arguments when called, saved for operator undo/redo and
automatically added into the user interface.

```../examples/bpy.types.Operator.1.py```


--------------------

This example shows how an operator can use the file selector.

Notice the invoke function calls a window manager method and returns
{'RUNNING_MODAL'}

, this means the file selector stays open and the operator does not
exit immediately after invoke finishes.

The file selector runs the operator, calling Operator.execute when the
user confirms.

The Operator.poll function is optional, used to check if the operator
can run.

```../examples/bpy.types.Operator.2.py```


--------------------

This operator uses its Operator.invoke function to call a popup.

```../examples/bpy.types.Operator.3.py```


--------------------

By default operator properties use an automatic user interface layout.
If you need more control you can create your own layout with a
Operator.draw function.

This works like the Panel and Menu draw functions, its used
for dialogs and file selectors.

```../examples/bpy.types.Operator.4.py```


--------------------

This operator defines a Operator.modal function that will keep being
run to handle events until it returns {'FINISHED'}

 or {'CANCELLED'}

.

Modal operators run every time a new event is detected, such as a mouse click
or key press. Conversely, when no new events are detected, the modal operator
will not run. Modal operators are especially useful for interactive tools, an
operator can have its own state where keys toggle options as the operator runs.
Grab, Rotate, Scale, and Fly-Mode are examples of modal operators.

Operator.invoke is used to initialize the operator as being active
by returning {'RUNNING_MODAL'}

, initializing the modal loop.

Notice __init__()

 and __del__()

 are declared.
For other operator types they are not useful but for modal operators they will
be called before the Operator.invoke and after the operator finishes.

```../examples/bpy.types.Operator.5.py```


--------------------

You may want to have an operator prompt the user to select an item
from a search field, this can be done using bpy.types.Operator.invoke_search_popup.

```../examples/bpy.types.Operator.6.py```


--------------------

```../examples/bpy.types.RenderEngine.py```


--------------------

This script is a simple panel which will draw into the object properties
section.

Notice the 'CATEGORY_PT_name' Panel.bl_idname, this is a naming
convention for panels.

[NOTE]
Panel subclasses must be registered for blender to use them.

```../examples/bpy.types.Panel.py```


--------------------

This panel has a Panel.poll and Panel.draw_header function,
even though the contents is basic this closely resembles blenders panels.

```../examples/bpy.types.Panel.1.py```


--------------------

A mix-in parent class can be used to share common properties and
Menu.poll function.

```../examples/bpy.types.Panel.2.py```


--------------------

This example demonstrates access to the evaluated ID (such as object, material, etc.) state from
an original ID.
This is needed every time one needs to access state with animation, constraints, and modifiers
taken into account.

```../examples/bpy.types.Depsgraph.1.py```


--------------------

This example demonstrates access to the original ID.
Such access is needed to check whether object is selected, or to compare pointers.

```../examples/bpy.types.Depsgraph.2.py```


--------------------

Sometimes it is needed to know all the instances with their matrices (for example, when writing an
exporter or a custom render engine).
This example shows how to access all objects and instances in the scene.

```../examples/bpy.types.Depsgraph.3.py```


--------------------

Function to get a mesh from any object with geometry. It is typically used by exporters, render
engines and tools that need to access the evaluated mesh as displayed in the viewport.

Object.to_mesh() is closely interacting with dependency graph: its behavior depends on whether it
is used on original or evaluated object.

When is used on original object, the result mesh is calculated from the object without taking
animation or modifiers into account:

* For meshes this is similar to duplicating the source mesh.
* For curves this disables own modifiers, and modifiers of objects used as bevel and taper.
* For metaballs this produces an empty mesh since polygonization is done as a modifier evaluation.

When is used on evaluated object all modifiers are taken into account.

[NOTE]
The result mesh is owned by the object. It can be freed by calling object.to_mesh_clear().

[NOTE]
The result mesh must be treated as temporary, and can not be referenced from objects in the main
database. If the mesh intended to be used in a persistent manner use bpy.data.meshes.new_from_object()
instead.

[NOTE]
If object does not have geometry (i.e. camera) the functions returns None.

```../examples/bpy.types.Depsgraph.4.py```


--------------------

Function to copy a new mesh from any object with geometry. The mesh is added to the main
database and can be referenced by objects. Typically used by tools that create new objects
or apply modifiers.

When is used on original object, the result mesh is calculated from the object without taking
animation or modifiers into account:

* For meshes this is similar to duplicating the source mesh.
* For curves this disables own modifiers, and modifiers of objects used as bevel and taper.
* For metaballs this produces an empty mesh since polygonization is done as a modifier evaluation.

When is used on evaluated object all modifiers are taken into account.

All the references (such as materials) are re-mapped to original. This ensures validity and
consistency of the main database.

[NOTE]
If object does not have geometry (i.e. camera) the functions returns None.

```../examples/bpy.types.Depsgraph.5.py```


--------------------

This example is a combination of all previous ones, and shows how to write a simple exporter
script.

```../examples/bpy.types.Depsgraph.6.py```


--------------------

Function to get a curve from text and curve objects. It is typically used by exporters, render
engines, and tools that need to access the curve representing the object.

The function takes the evaluated dependency graph as a required parameter and optionally a boolean
apply_modifiers which defaults to false. If apply_modifiers is true and the object is a curve object,
the spline deform modifiers are applied on the control points. Note that constructive modifiers and
modifiers that are not spline-enabled will not be applied. So modifiers like Array will not be applied
and deform modifiers that have Apply On Spline disabled will not be applied.

If the object is a text object. The text will be converted into a 3D curve and returned. Modifiers are
never applied on text objects and apply_modifiers will be ignored. If the object is neither a curve nor
a text object, an error will be reported.

[NOTE]
The resulting curve is owned by the object. It can be freed by calling object.to_curve_clear().

[NOTE]
The resulting curve must be treated as temporary, and can not be referenced from objects in the main
database.

```../examples/bpy.types.Depsgraph.7.py```

"""

import typing
import collections.abc
import bl_operators.anim
import bl_operators.assets
import bl_operators.clip
import bl_operators.constraint
import bl_operators.file
import bl_operators.freestyle
import bl_operators.node
import bl_operators.object
import bl_operators.presets
import bl_operators.spreadsheet
import bl_operators.userpref
import bl_operators.view3d
import bl_operators.wm
import bl_ui
import bl_ui.properties_collection
import bl_ui.properties_constraint
import bl_ui.properties_data_armature
import bl_ui.properties_data_bone
import bl_ui.properties_data_camera
import bl_ui.properties_data_curve
import bl_ui.properties_data_curves
import bl_ui.properties_data_empty
import bl_ui.properties_data_gpencil
import bl_ui.properties_data_lattice
import bl_ui.properties_data_light
import bl_ui.properties_data_lightprobe
import bl_ui.properties_data_mesh
import bl_ui.properties_data_metaball
import bl_ui.properties_data_modifier
import bl_ui.properties_data_pointcloud
import bl_ui.properties_data_shaderfx
import bl_ui.properties_data_speaker
import bl_ui.properties_data_volume
import bl_ui.properties_freestyle
import bl_ui.properties_grease_pencil_common
import bl_ui.properties_mask_common
import bl_ui.properties_material
import bl_ui.properties_material_gpencil
import bl_ui.properties_object
import bl_ui.properties_output
import bl_ui.properties_paint_common
import bl_ui.properties_particle
import bl_ui.properties_physics_cloth
import bl_ui.properties_physics_common
import bl_ui.properties_physics_dynamicpaint
import bl_ui.properties_physics_field
import bl_ui.properties_physics_fluid
import bl_ui.properties_physics_rigidbody
import bl_ui.properties_physics_rigidbody_constraint
import bl_ui.properties_physics_softbody
import bl_ui.properties_render
import bl_ui.properties_scene
import bl_ui.properties_texture
import bl_ui.properties_view_layer
import bl_ui.properties_workspace
import bl_ui.properties_world
import bl_ui.space_clip
import bl_ui.space_console
import bl_ui.space_dopesheet
import bl_ui.space_filebrowser
import bl_ui.space_graph
import bl_ui.space_image
import bl_ui.space_info
import bl_ui.space_nla
import bl_ui.space_node
import bl_ui.space_outliner
import bl_ui.space_properties
import bl_ui.space_sequencer
import bl_ui.space_spreadsheet
import bl_ui.space_statusbar
import bl_ui.space_text
import bl_ui.space_time
import bl_ui.space_toolsystem_common
import bl_ui.space_toolsystem_toolbar
import bl_ui.space_topbar
import bl_ui.space_userpref
import bl_ui.space_view3d
import bl_ui.space_view3d_toolbar
import mathutils

GenericType1 = typing.TypeVar("GenericType1")
GenericType2 = typing.TypeVar("GenericType2")

class bpy_prop_collection(typing.Generic[GenericType1]):
    """built-in class used for all collections."""

    def find(self, key: str | None) -> int:
        """Returns the index of a key in a collection or -1 when not found
        (matches Python's string find function of the same name).

                :param key: The identifier for the collection member.
                :type key: str | None
                :return: index of the key.
                :rtype: int
        """
        ...

    def foreach_get(
        self,
        attr: str,
        seq: collections.abc.MutableSequence[bool]
        | collections.abc.MutableSequence[int]
        | collections.abc.MutableSequence[float],
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        Multi-dimensional arrays (like array of vectors) will be flattened into seq.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.MutableSequence[bool] | collections.abc.MutableSequence[int] | collections.abc.MutableSequence[float]
        """
        ...

    def foreach_set(
        self,
        attr: str,
        seq: collections.abc.Sequence[bool]
        | collections.abc.Sequence[int]
        | collections.abc.Sequence[float],
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        seq must be uni-dimensional, multi-dimensional arrays (like array of vectors) will be re-created from it.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.Sequence[bool] | collections.abc.Sequence[int] | collections.abc.Sequence[float]
        """
        ...

    def get(
        self, key: str | None, default: GenericType2 = None
    ) -> GenericType1 | GenericType2:
        """Returns the value of the item assigned to key or default when not found
        (matches Python's dictionary function of the same name).

                :param key: The identifier for the collection member.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
                :type default: GenericType2
                :return:
                :rtype: GenericType1 | GenericType2
        """
        ...

    def items(self) -> list[tuple[str, GenericType1]]:
        """Return the identifiers of collection members
        (matching Python's dict.items() functionality).

                :return:
                :rtype: list[tuple[str, GenericType1]]
        """
        ...

    def keys(self) -> list[str]:
        """Return the identifiers of collection members
        (matching Python's dict.keys() functionality).

                :return: the identifiers for each member of this collection.
                :rtype: list[str]
        """
        ...

    def values(self) -> list[GenericType1]:
        """Return the values of collection
        (matching Python's dict.values() functionality).

                :return:
                :rtype: list[GenericType1]
        """
        ...

    @typing.overload
    def __getitem__(self, key: int | str) -> GenericType1:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: GenericType1
        """
        ...

    @typing.overload
    def __getitem__(self, key: slice) -> tuple[GenericType1]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: tuple[GenericType1]
        """
        ...

    def __getitem__(self, key: int | str | slice) -> GenericType1 | tuple[GenericType1]:
        """

        :param key:
        :type key: int | str | slice
        :return:
        :rtype: GenericType1 | tuple[GenericType1]
        """
        ...

    @typing.overload
    def __setitem__(self, key: int | str, value: GenericType1):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: GenericType1
        """
        ...

    @typing.overload
    def __setitem__(self, key: slice, value: tuple[GenericType1]):
        """

        :param key:
        :type key: slice
        :param value:
        :type value: tuple[GenericType1]
        """
        ...

    def __setitem__(
        self, key: int | str | slice, value: GenericType1 | tuple[GenericType1]
    ):
        """

        :param key:
        :type key: int | str | slice
        :param value:
        :type value: GenericType1 | tuple[GenericType1]
        """
        ...

    def __delitem__(self, key: int | str | slice):
        """

        :param key:
        :type key: int | str | slice
        """
        ...

    def __iter__(self) -> collections.abc.Iterator[GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[GenericType1]
        """
        ...

    def __next__(self) -> GenericType1:
        """

        :return:
        :rtype: GenericType1
        """
        ...

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """
        ...

    def __contains__(self, key: str | tuple[str, ...]) -> bool:
        """

        :param key:
        :type key: str | tuple[str, ...]
        :return:
        :rtype: bool
        """
        ...

class bpy_struct(typing.Generic[GenericType1]):
    """built-in base class for all classes in bpy.types."""

    id_data: typing.Any
    """ The `bpy.types.ID` object this datablock is from or None, (not available for all data types)"""

    def as_pointer(self) -> int:
        """Returns the memory address which holds a pointer to Blender's internal data

        :return: int (memory address).
        :rtype: int
        """
        ...

    def driver_add(self, path: str | None, index: int | None = -1) -> FCurve:
        """Adds driver(s) to the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: The driver(s) added.
        :rtype: FCurve
        """
        ...

    def driver_remove(self, path: str | None, index: int | None = -1) -> bool:
        """Remove driver(s) from the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: Success of driver removal.
        :rtype: bool
        """
        ...

    def get(self, key: str | None, default=None):
        """Returns the value of the custom property assigned to key or default
        when not found (matches Python's dictionary function of the same name).

                :param key: The key associated with the custom property.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
        """
        ...

    def id_properties_clear(self):
        """

        :return: Remove the parent group for an RNA struct's custom IDProperties.
        """
        ...

    def id_properties_ensure(self):
        """

        :return: the parent group for an RNA struct's custom IDProperties.
        """
        ...

    def id_properties_ui(self, key):
        """

        :param key: String name of the property.
        :return: Return an object used to manage an IDProperty's UI data.
        """
        ...

    def is_property_hidden(self, property) -> bool:
        """Check if a property is hidden.

        :param property:
        :return: True when the property is hidden.
        :rtype: bool
        """
        ...

    def is_property_overridable_library(self, property) -> bool:
        """Check if a property is overridable.

        :param property:
        :return: True when the property is overridable.
        :rtype: bool
        """
        ...

    def is_property_readonly(self, property) -> bool:
        """Check if a property is readonly.

        :param property:
        :return: True when the property is readonly (not writable).
        :rtype: bool
        """
        ...

    def is_property_set(self, property, ghost: bool | None = True) -> bool:
        """Check if a property is set, use for testing operator properties.

                :param property:
                :param ghost: Used for operators that re-run with previous settings.
        In this case the property is not marked as set,
        yet the value from the previous execution is used.

        In rare cases you may want to set this option to false.
                :type ghost: bool | None
                :return: True when the property has been set.
                :rtype: bool
        """
        ...

    def items(self):
        """Returns the items of this objects custom properties (matches Python's
        dictionary function of the same name).

                :return: custom property key, value pairs.
        """
        ...

    def keyframe_delete(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = None,
        group: str | None = "",
    ) -> bool:
        """Remove a keyframe from this properties fcurve.

        :param data_path: path to the property to remove a key, analogous to the fcurve's data path.
        :type data_path: str | None
        :param index: array index of the property to remove a key. Defaults to -1 removing all indices or a single channel if the property is not an array.
        :type index: int | None
        :param frame: The frame on which the keyframe is deleted, defaulting to the current frame.
        :type frame: float | None
        :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
        :type group: str | None
        :return: Success of keyframe deletion.
        :rtype: bool
        """
        ...

    def keyframe_insert(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = None,
        group: str | None = "",
        options=None(),
    ) -> bool:
        """Insert a keyframe on the property given, adding fcurves and animation data when necessary.This is the most simple example of inserting a keyframe from python.Note that when keying data paths which contain nested properties this must be
        done from the `ID` subclass, in this case the `Armature` rather
        than the bone.

                :param data_path: path to the property to key, analogous to the fcurve's data path.
                :type data_path: str | None
                :param index: array index of the property to key.
        Defaults to -1 which will key all indices or a single channel if the property is not an array.
                :type index: int | None
                :param frame: The frame on which the keyframe is inserted, defaulting to the current frame.
                :type frame: float | None
                :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
                :type group: str | None
                :param options: Optional set of flags:

        INSERTKEY_NEEDED Only insert keyframes where they're needed in the relevant F-Curves.

        INSERTKEY_VISUAL Insert keyframes based on 'visual transforms'.

        INSERTKEY_XYZ_TO_RGB Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis.

        INSERTKEY_REPLACE Only replace already existing keyframes.

        INSERTKEY_AVAILABLE Only insert into already existing F-Curves.

        INSERTKEY_CYCLE_AWARE Take cyclic extrapolation into account (Cycle-Aware Keying option).
                :return: Success of keyframe insertion.
                :rtype: bool
        """
        ...

    def keys(self):
        """Returns the keys of this objects custom properties (matches Python's
        dictionary function of the same name).

                :return: custom property keys.
        """
        ...

    def path_from_id(self, property: str | None = "") -> str:
        """Returns the data path from the ID to this object (string).

                :param property: Optional property name which can be used if the path is
        to a property of this object.
                :type property: str | None
                :return: The path from `bpy.types.bpy_struct.id_data`
        to this struct and property (when given).
                :rtype: str
        """
        ...

    def path_resolve(self, path: str | None, coerce: bool | None = True):
        """Returns the property from the path, raise an exception when not found.

                :param path: path which this property resolves.
                :type path: str | None
                :param coerce: optional argument, when True, the property will be converted
        into its Python representation.
                :type coerce: bool | None
        """
        ...

    def pop(self, key: str | None, default=None):
        """Remove and return the value of the custom property assigned to key or default
        when not found (matches Python's dictionary function of the same name).

                :param key: The key associated with the custom property.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
        """
        ...

    def property_overridable_library_set(self, property, overridable) -> bool:
        """Define a property as overridable or not (only for custom properties!).

        :param property:
        :param overridable:
        :return: True when the overridable status of the property was successfully set.
        :rtype: bool
        """
        ...

    def property_unset(self, property):
        """Unset a property, will use default value afterward.

        :param property:
        """
        ...

    def type_recast(self):
        """Return a new instance, this is needed because types
        such as textures can be changed at runtime.

                :return: a new instance of this object with the type initialized again.
        """
        ...

    def values(self):
        """Returns the values of this objects custom properties (matches Python's
        dictionary function of the same name).

                :return: custom property values.
        """
        ...

    def __getitem__(self, key: int | str) -> typing.Any:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: typing.Any
        """
        ...

    def __setitem__(self, key: int | str, value: typing.Any):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: typing.Any
        """
        ...

    def __delitem__(self, key: int | str):
        """

        :param key:
        :type key: int | str
        """
        ...

class bpy_prop_array(typing.Generic[GenericType1]):
    def __get__(self, instance, owner) -> bpy_prop_array[GenericType1]:
        """

        :param instance:
        :param owner:
        :return:
        :rtype: bpy_prop_array[GenericType1]
        """
        ...

    def __set__(self, instance, value: collections.abc.Iterable[GenericType1]):
        """

        :param instance:
        :param value:
        :type value: collections.abc.Iterable[GenericType1]
        """
        ...

    def foreach_get(self, seq: collections.abc.MutableSequence[GenericType1]):
        """

        :param seq:
        :type seq: collections.abc.MutableSequence[GenericType1]
        """
        ...

    def foreach_set(self, seq: typing.Sequence[GenericType1]):
        """

        :param seq:
        :type seq: typing.Sequence[GenericType1]
        """
        ...

    @typing.overload
    def __getitem__(self, key: int) -> GenericType1:
        """

        :param key:
        :type key: int
        :return:
        :rtype: GenericType1
        """
        ...

    @typing.overload
    def __getitem__(self, key: slice) -> tuple[GenericType1]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: tuple[GenericType1]
        """
        ...

    def __getitem__(self, key: int | slice) -> GenericType1 | tuple[GenericType1]:
        """

        :param key:
        :type key: int | slice
        :return:
        :rtype: GenericType1 | tuple[GenericType1]
        """
        ...

    @typing.overload
    def __setitem__(self, key: int, value: GenericType1):
        """

        :param key:
        :type key: int
        :param value:
        :type value: GenericType1
        """
        ...

    @typing.overload
    def __setitem__(self, key: slice, value: tuple[GenericType1]):
        """

        :param key:
        :type key: slice
        :param value:
        :type value: tuple[GenericType1]
        """
        ...

    def __setitem__(self, key: int | slice, value: GenericType1 | tuple[GenericType1]):
        """

        :param key:
        :type key: int | slice
        :param value:
        :type value: GenericType1 | tuple[GenericType1]
        """
        ...

    def __delitem__(self, key: int | slice):
        """

        :param key:
        :type key: int | slice
        """
        ...

    def __iter__(self) -> collections.abc.Iterator[GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[GenericType1]
        """
        ...

    def __next__(self) -> GenericType1:
        """

        :return:
        :rtype: GenericType1
        """
        ...

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """
        ...

class AOV(bpy_struct):
    is_valid: bool
    """ Is the name of the AOV conflicting

    :type: bool
    """

    name: str
    """ Name of the AOV

    :type: str
    """

    type: str
    """ Data type of the AOV

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AOVs(bpy_prop_collection[AOV], bpy_struct):
    """Collection of AOVs"""

    def add(self) -> AOV:
        """add

        :return: Newly created AOV
        :rtype: AOV
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ActionFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of action F-Curves"""

    def new(
        self,
        data_path: str | typing.Any,
        index: typing.Any | None = 0,
        action_group: str | typing.Any = "",
    ) -> FCurve:
        """Add an F-Curve to the action

        :param data_path: Data Path, F-Curve data path to use
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :param action_group: Action Group, Acton group to add this F-Curve into
        :type action_group: str | typing.Any
        :return: Newly created F-Curve
        :rtype: FCurve
        """
        ...

    def find(self, data_path: str | typing.Any, index: typing.Any | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the action.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """
        ...

    def remove(self, fcurve: FCurve):
        """Remove F-Curve

        :param fcurve: F-Curve to remove
        :type fcurve: FCurve
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ActionGroup(bpy_struct):
    """Groups of F-Curves"""

    channels: bpy_prop_collection[FCurve]
    """ F-Curves in this group

    :type: bpy_prop_collection[FCurve]
    """

    color_set: str
    """ Custom color set to use

    :type: str
    """

    colors: ThemeBoneColorSet
    """ Copy of the colors associated with the group's color set

    :type: ThemeBoneColorSet
    """

    is_custom_color_set: bool
    """ Color set is user-defined instead of a fixed theme color set

    :type: bool
    """

    lock: bool
    """ Action group is locked

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ Action group is selected

    :type: bool
    """

    show_expanded: bool
    """ Action group is expanded except in graph editor

    :type: bool
    """

    show_expanded_graph: bool
    """ Action group is expanded in graph editor

    :type: bool
    """

    use_pin: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ActionGroups(bpy_prop_collection[ActionGroup], bpy_struct):
    """Collection of action groups"""

    def new(self, name: str | typing.Any) -> ActionGroup:
        """Create a new action group and add it to the action

        :param name: New name for the action group
        :type name: str | typing.Any
        :return: Newly created action group
        :rtype: ActionGroup
        """
        ...

    def remove(self, action_group: ActionGroup):
        """Remove action group

        :param action_group: Action group to remove
        :type action_group: ActionGroup
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ActionPoseMarkers(bpy_prop_collection[TimelineMarker], bpy_struct):
    """Collection of timeline markers"""

    active: TimelineMarker | None
    """ Active pose marker for this action

    :type: TimelineMarker | None
    """

    active_index: int | None
    """ Index of active pose marker

    :type: int | None
    """

    def new(self, name: str | typing.Any) -> TimelineMarker:
        """Add a pose marker to the action

        :param name: New name for the marker (not unique)
        :type name: str | typing.Any
        :return: Newly created marker
        :rtype: TimelineMarker
        """
        ...

    def remove(self, marker: TimelineMarker):
        """Remove a timeline marker

        :param marker: Timeline marker to remove
        :type marker: TimelineMarker
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Addon(bpy_struct):
    """Python add-ons to be loaded automatically"""

    module: str
    """ Module name

    :type: str
    """

    preferences: AddonPreferences
    """ 

    :type: AddonPreferences
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AddonPreferences(bpy_struct):
    bl_idname: str
    """ 

    :type: str
    """

    layout: UILayout
    """ 

    :type: UILayout
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Addons(bpy_prop_collection[Addon], bpy_struct):
    """Collection of add-ons"""

    @classmethod
    def new(cls) -> Addon:
        """Add a new add-on

        :return: Add-on data
        :rtype: Addon
        """
        ...

    @classmethod
    def remove(cls, addon: Addon):
        """Remove add-on

        :param addon: Add-on to remove
        :type addon: Addon
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AnimData(bpy_struct):
    """Animation data for data-block"""

    action: Action
    """ Active Action for this data-block

    :type: Action
    """

    action_blend_type: str
    """ Method used for combining Active Action's result with result of NLA stack

    :type: str
    """

    action_extrapolation: str
    """ Action to take for gaps past the Active Action's range (when evaluating with NLA)

    :type: str
    """

    action_influence: float
    """ Amount the Active Action contributes to the result of the NLA stack

    :type: float
    """

    drivers: AnimDataDrivers
    """ The Drivers/Expressions for this data-block

    :type: AnimDataDrivers
    """

    nla_tracks: NlaTracks
    """ NLA Tracks (i.e. Animation Layers)

    :type: NlaTracks
    """

    use_nla: bool
    """ NLA stack is evaluated when evaluating this block

    :type: bool
    """

    use_pin: bool
    """ 

    :type: bool
    """

    use_tweak_mode: bool
    """ Whether to enable or disable tweak mode in NLA

    :type: bool
    """

    def nla_tweak_strip_time_to_scene(
        self, frame: float | None, invert: bool | typing.Any | None = False
    ) -> float:
        """Convert a time value from the local time of the tweaked strip to scene time, exactly as done by built-in key editing tools. Returns the input time unchanged if not tweaking.

        :param frame: Input time
        :type frame: float | None
        :param invert: Invert, Convert scene time to action time
        :type invert: bool | typing.Any | None
        :return: Converted time
        :rtype: float
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AnimDataDrivers(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of Driver F-Curves"""

    def new(self, data_path: str | typing.Any, index: typing.Any | None = 0) -> FCurve:
        """new

        :param data_path: Data Path, F-Curve data path to use
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :return: Newly Driver F-Curve
        :rtype: FCurve
        """
        ...

    def remove(self, driver: FCurve):
        """remove

        :param driver:
        :type driver: FCurve
        """
        ...

    def from_existing(self, src_driver: FCurve | None = None) -> FCurve:
        """Add a new driver given an existing one

        :param src_driver: Existing Driver F-Curve to use as template for a new one
        :type src_driver: FCurve | None
        :return: New Driver F-Curve
        :rtype: FCurve
        """
        ...

    def find(self, data_path: str | typing.Any, index: typing.Any | None = 0) -> FCurve:
        """Find a driver F-Curve. Note that this function performs a linear scan of all driver F-Curves.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AnimViz(bpy_struct):
    """Settings for the visualization of motion"""

    motion_path: AnimVizMotionPaths
    """ Motion Path settings for visualization

    :type: AnimVizMotionPaths
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AnimVizMotionPaths(bpy_struct):
    """Motion Path settings for animation visualization"""

    bake_location: str
    """ When calculating Bone Paths, use Head or Tips

    :type: str
    """

    frame_after: int
    """ Number of frames to show after the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_before: int
    """ Number of frames to show before the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_end: int
    """ End frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_start: int
    """ Starting frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_step: int
    """ Number of frames between paths shown (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    has_motion_paths: bool
    """ Are there any bone paths that will need updating (read-only)

    :type: bool
    """

    range: str
    """ Type of range to calculate for Motion Paths

    :type: str
    """

    show_frame_numbers: bool
    """ Show frame numbers on Motion Paths

    :type: bool
    """

    show_keyframe_action_all: bool
    """ For bone motion paths, search whole Action for keyframes instead of in group with matching name only (is slower)

    :type: bool
    """

    show_keyframe_highlight: bool
    """ Emphasize position of keyframes on Motion Paths

    :type: bool
    """

    show_keyframe_numbers: bool
    """ Show frame numbers of Keyframes on Motion Paths

    :type: bool
    """

    type: str
    """ Type of range to show for Motion Paths

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AnyType(bpy_struct):
    """RNA type used for pointers to any possible data"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Area(bpy_struct):
    """Area in a subdivided screen, containing an editor"""

    height: int
    """ Area height

    :type: int
    """

    regions: bpy_prop_collection[Region]
    """ Regions this area is subdivided in

    :type: bpy_prop_collection[Region]
    """

    show_menus: bool
    """ Show menus in the header

    :type: bool
    """

    spaces: AreaSpaces
    """ Spaces contained in this area, the first being the active space (NOTE: Useful for example to restore a previously used 3D view space in a certain area to get the old view orientation)

    :type: AreaSpaces
    """

    type: str
    """ Current editor type for this area

    :type: str
    """

    ui_type: str
    """ Current editor type for this area

    :type: str
    """

    width: int
    """ Area width

    :type: int
    """

    x: int
    """ The window relative vertical location of the area

    :type: int
    """

    y: int
    """ The window relative horizontal location of the area

    :type: int
    """

    def tag_redraw(self):
        """tag_redraw"""
        ...

    def header_text_set(self, text: str | None):
        """Set the header status text

        :param text: Text, New string for the header, None clears the text
        :type text: str | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AreaSpaces(bpy_prop_collection[Space], bpy_struct):
    """Collection of spaces"""

    active: Space
    """ Space currently being displayed in this area

    :type: Space
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ArmatureBones(bpy_prop_collection[Bone], bpy_struct):
    """Collection of armature bones"""

    active: Bone | None
    """ Armature's active bone

    :type: Bone | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ArmatureConstraintTargets(bpy_prop_collection[ConstraintTargetBone], bpy_struct):
    """Collection of target bones and weights"""

    def new(self) -> ConstraintTargetBone:
        """Add a new target to the constraint

        :return: New target bone
        :rtype: ConstraintTargetBone
        """
        ...

    def remove(self, target: ConstraintTargetBone):
        """Delete target from the constraint

        :param target: Target to remove
        :type target: ConstraintTargetBone
        """
        ...

    def clear(self):
        """Delete all targets from object"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ArmatureEditBones(bpy_prop_collection[EditBone], bpy_struct):
    """Collection of armature edit bones"""

    active: EditBone | None
    """ Armatures active edit bone

    :type: EditBone | None
    """

    def new(self, name: str | typing.Any) -> EditBone:
        """Add a new bone

        :param name: New name for the bone
        :type name: str | typing.Any
        :return: Newly created edit bone
        :rtype: EditBone
        """
        ...

    def remove(self, bone: EditBone):
        """Remove an existing bone from the armature

        :param bone: EditBone to remove
        :type bone: EditBone
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AssetLibraryReference(bpy_struct):
    """Identifier to refer to the asset library"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AssetMetaData(bpy_struct):
    """Additional data stored for an asset data-block"""

    active_tag: int | None
    """ Index of the tag set for editing

    :type: int | None
    """

    author: str
    """ Name of the creator of the asset

    :type: str
    """

    catalog_id: str
    """ Identifier for the asset's catalog, used by Blender to look up the asset's catalog path. Must be a UUID according to RFC4122

    :type: str
    """

    catalog_simple_name: str
    """ Simple name of the asset's catalog, for debugging and data recovery purposes

    :type: str
    """

    description: str
    """ A description of the asset to be displayed for the user

    :type: str
    """

    tags: AssetTags
    """ Custom tags (name tokens) for the asset, used for filtering and general asset management

    :type: AssetTags
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AssetTag(bpy_struct):
    """User defined tag (name token)"""

    name: str
    """ The identifier that makes up this tag

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AssetTags(bpy_prop_collection[AssetTag], bpy_struct):
    """Collection of custom asset tags"""

    def new(
        self, name: str | typing.Any, skip_if_exists: bool | typing.Any | None = False
    ) -> AssetTag:
        """Add a new tag to this asset

        :param name: Name
        :type name: str | typing.Any
        :param skip_if_exists: Skip if Exists, Do not add a new tag if one of the same type already exists
        :type skip_if_exists: bool | typing.Any | None
        :return: New tag
        :rtype: AssetTag
        """
        ...

    def remove(self, tag: AssetTag):
        """Remove an existing tag from this asset

        :param tag: Removed tag
        :type tag: AssetTag
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Attribute(bpy_struct):
    """Geometry attribute"""

    data_type: str
    """ Type of data stored in attribute

    :type: str
    """

    domain: str
    """ Domain of the Attribute

    :type: str
    """

    name: str
    """ Name of the Attribute

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class AttributeGroup(bpy_prop_collection[Attribute], bpy_struct):
    """Group of geometry attributes"""

    active: Attribute | None
    """ Active attribute

    :type: Attribute | None
    """

    active_color: Attribute | None
    """ Active color attribute for display and editing

    :type: Attribute | None
    """

    active_color_index: int | None
    """ Active color attribute index

    :type: int | None
    """

    active_index: int | None
    """ Active attribute index

    :type: int | None
    """

    render_color_index: int
    """ The index of the color attribute used as a fallback for rendering

    :type: int
    """

    def new(
        self, name: str | typing.Any, type: str | None, domain: str | None
    ) -> Attribute:
        """Add attribute to geometry

                :param name: Name, Name of geometry attribute
                :type name: str | typing.Any
                :param type: Type, Attribute type

        FLOAT
        Float -- Floating-point value.

        INT
        Integer -- 32-bit integer.

        FLOAT_VECTOR
        Vector -- 3D vector with floating-point values.

        FLOAT_COLOR
        Color -- RGBA color with 32-bit floating-point values.

        BYTE_COLOR
        Byte Color -- RGBA color with 8-bit positive integer values.

        STRING
        String -- Text string.

        BOOLEAN
        Boolean -- True or false.

        FLOAT2
        2D Vector -- 2D vector with floating-point values.

        INT8
        8-Bit Integer -- Smaller integer with a range from -128 to 127.
                :type type: str | None
                :param domain: Domain, Type of element that attribute is stored on

        POINT
        Point -- Attribute on point.

        EDGE
        Edge -- Attribute on mesh edge.

        FACE
        Face -- Attribute on mesh faces.

        CORNER
        Face Corner -- Attribute on mesh face corner.

        CURVE
        Spline -- Attribute on spline.

        INSTANCE
        Instance -- Attribute on instance.
                :type domain: str | None
                :return: New geometry attribute
                :rtype: Attribute
        """
        ...

    def remove(self, attribute: Attribute):
        """Remove attribute from geometry

        :param attribute: Geometry Attribute
        :type attribute: Attribute
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BakeSettings(bpy_struct):
    """Bake data for a Scene data-block"""

    cage_extrusion: float
    """ Inflate the active object by the specified distance for baking. This helps matching to points nearer to the outside of the selected object meshes

    :type: float
    """

    cage_object: Object
    """ Object to use as cage instead of calculating the cage from the active object with cage extrusion

    :type: Object
    """

    filepath: str
    """ Image filepath to use when saving externally

    :type: str
    """

    height: int
    """ Vertical dimension of the baking map

    :type: int
    """

    image_settings: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    margin: int
    """ Extends the baked result as a post process filter

    :type: int
    """

    margin_type: str
    """ Algorithm to extend the baked result

    :type: str
    """

    max_ray_distance: float
    """ The maximum ray distance for matching points between the active and selected objects. If zero, there is no limit

    :type: float
    """

    normal_b: str
    """ Axis to bake in blue channel

    :type: str
    """

    normal_g: str
    """ Axis to bake in green channel

    :type: str
    """

    normal_r: str
    """ Axis to bake in red channel

    :type: str
    """

    normal_space: str
    """ Choose normal space for baking

    :type: str
    """

    pass_filter: set[str]
    """ Passes to include in the active baking pass

    :type: set[str]
    """

    save_mode: str
    """ Where to save baked image textures

    :type: str
    """

    target: str
    """ Where to output the baked map

    :type: str
    """

    use_automatic_name: bool
    """ Automatically name the output file with the pass type (external only)

    :type: bool
    """

    use_cage: bool
    """ Cast rays to active object from a cage

    :type: bool
    """

    use_clear: bool
    """ Clear Images before baking (internal only)

    :type: bool
    """

    use_pass_color: bool
    """ Color the pass

    :type: bool
    """

    use_pass_diffuse: bool
    """ Add diffuse contribution

    :type: bool
    """

    use_pass_direct: bool
    """ Add direct lighting contribution

    :type: bool
    """

    use_pass_emit: bool
    """ Add emission contribution

    :type: bool
    """

    use_pass_glossy: bool
    """ Add glossy contribution

    :type: bool
    """

    use_pass_indirect: bool
    """ Add indirect lighting contribution

    :type: bool
    """

    use_pass_transmission: bool
    """ Add transmission contribution

    :type: bool
    """

    use_selected_to_active: bool
    """ Bake shading on the surface of selected objects to the active object

    :type: bool
    """

    use_split_materials: bool
    """ Split external images per material (external only)

    :type: bool
    """

    width: int
    """ Horizontal dimension of the baking map

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BezierSplinePoint(bpy_struct):
    """Bezier curve point with two handles"""

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the first handle

    :type: bpy_prop_array[float]
    """

    handle_left_type: str
    """ Handle types

    :type: str
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the second handle

    :type: bpy_prop_array[float]
    """

    handle_right_type: str
    """ Handle types

    :type: str
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Handle 1 selection status

    :type: bool
    """

    select_right_handle: bool
    """ Handle 2 selection status

    :type: bool
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendData(bpy_struct):
    """Main data structure representing a .blend file and all its data-blocks"""

    actions: BlendDataActions
    """ Action data-blocks

    :type: BlendDataActions
    """

    armatures: BlendDataArmatures
    """ Armature data-blocks

    :type: BlendDataArmatures
    """

    brushes: BlendDataBrushes
    """ Brush data-blocks

    :type: BlendDataBrushes
    """

    cache_files: BlendDataCacheFiles
    """ Cache Files data-blocks

    :type: BlendDataCacheFiles
    """

    cameras: BlendDataCameras
    """ Camera data-blocks

    :type: BlendDataCameras
    """

    collections: BlendDataCollections
    """ Collection data-blocks

    :type: BlendDataCollections
    """

    curves: BlendDataCurves
    """ Curve data-blocks

    :type: BlendDataCurves
    """

    filepath: str
    """ Path to the .blend file

    :type: str
    """

    fonts: BlendDataFonts
    """ Vector font data-blocks

    :type: BlendDataFonts
    """

    grease_pencils: BlendDataGreasePencils
    """ Grease Pencil data-blocks

    :type: BlendDataGreasePencils
    """

    images: BlendDataImages
    """ Image data-blocks

    :type: BlendDataImages
    """

    is_dirty: bool
    """ Have recent edits been saved to disk

    :type: bool
    """

    is_saved: bool
    """ Has the current session been saved to disk as a .blend file

    :type: bool
    """

    lattices: BlendDataLattices
    """ Lattice data-blocks

    :type: BlendDataLattices
    """

    libraries: BlendDataLibraries
    """ Library data-blocks

    :type: BlendDataLibraries
    """

    lightprobes: BlendDataProbes
    """ Light Probe data-blocks

    :type: BlendDataProbes
    """

    lights: BlendDataLights
    """ Light data-blocks

    :type: BlendDataLights
    """

    linestyles: BlendDataLineStyles
    """ Line Style data-blocks

    :type: BlendDataLineStyles
    """

    masks: BlendDataMasks
    """ Masks data-blocks

    :type: BlendDataMasks
    """

    materials: BlendDataMaterials
    """ Material data-blocks

    :type: BlendDataMaterials
    """

    meshes: BlendDataMeshes
    """ Mesh data-blocks

    :type: BlendDataMeshes
    """

    metaballs: BlendDataMetaBalls
    """ Metaball data-blocks

    :type: BlendDataMetaBalls
    """

    movieclips: BlendDataMovieClips
    """ Movie Clip data-blocks

    :type: BlendDataMovieClips
    """

    node_groups: BlendDataNodeTrees
    """ Node group data-blocks

    :type: BlendDataNodeTrees
    """

    objects: BlendDataObjects
    """ Object data-blocks

    :type: BlendDataObjects
    """

    paint_curves: BlendDataPaintCurves
    """ Paint Curves data-blocks

    :type: BlendDataPaintCurves
    """

    palettes: BlendDataPalettes
    """ Palette data-blocks

    :type: BlendDataPalettes
    """

    particles: BlendDataParticles
    """ Particle data-blocks

    :type: BlendDataParticles
    """

    pointclouds: BlendDataPointClouds
    """ Point cloud data-blocks

    :type: BlendDataPointClouds
    """

    scenes: BlendDataScenes
    """ Scene data-blocks

    :type: BlendDataScenes
    """

    screens: BlendDataScreens
    """ Screen data-blocks

    :type: BlendDataScreens
    """

    shape_keys: bpy_prop_collection[Key]
    """ Shape Key data-blocks

    :type: bpy_prop_collection[Key]
    """

    sounds: BlendDataSounds
    """ Sound data-blocks

    :type: BlendDataSounds
    """

    speakers: BlendDataSpeakers
    """ Speaker data-blocks

    :type: BlendDataSpeakers
    """

    texts: BlendDataTexts
    """ Text data-blocks

    :type: BlendDataTexts
    """

    textures: BlendDataTextures
    """ Texture data-blocks

    :type: BlendDataTextures
    """

    use_autopack: bool
    """ Automatically pack all external data into .blend file

    :type: bool
    """

    version: bpy_prop_array[int]
    """ File format version the .blend file was saved with

    :type: bpy_prop_array[int]
    """

    volumes: BlendDataVolumes
    """ Volume data-blocks

    :type: BlendDataVolumes
    """

    window_managers: BlendDataWindowManagers
    """ Window manager data-blocks

    :type: BlendDataWindowManagers
    """

    workspaces: BlendDataWorkSpaces
    """ Workspace data-blocks

    :type: BlendDataWorkSpaces
    """

    worlds: BlendDataWorlds
    """ World data-blocks

    :type: BlendDataWorlds
    """

    def batch_remove(self, ids):
        """Remove (delete) several IDs at once.WARNING: Considered experimental feature currently.Note that this function is quicker than individual calls to `remove()` (from `bpy.types.BlendData`
        ID collections), but less safe/versatile (it can break Blender, e.g. by removing all scenes...).

                :param ids: Iterables of IDs (types can be mixed).
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

    def orphans_purge(self):
        """Remove (delete) all IDs with no user.

        :return: The number of deleted IDs.
        """
        ...

    def temp_data(self, filepath: str | None = None) -> BlendData:
        """A context manager that temporarily creates blender file data.

        :param filepath: The file path for the newly temporary data. When None, the path of the currently open file is used.
        :type filepath: str | None
        :return: Blend file data which is freed once the context exists.
        :rtype: BlendData
        """
        ...

    def user_map(
        self,
        subset: collections.abc.Sequence | None,
        key_types: set[str] | None,
        value_types: set[str] | None,
    ) -> dict:
        """Returns a mapping of all ID data-blocks in current bpy.data to a set of all datablocks using them.For list of valid set members for key_types & value_types, see: `bpy.types.KeyingSetPath.id_type`.

        :param subset: When passed, only these data-blocks and their users will be included as keys/values in the map.
        :type subset: collections.abc.Sequence | None
        :param key_types: Filter the keys mapped by ID types.
        :type key_types: set[str] | None
        :param value_types: Filter the values in the set by ID types.
        :type value_types: set[str] | None
        :return: dictionary of `bpy.types.ID` instances, with sets of ID's as their values.
        :rtype: dict
        """
        ...

class BlendDataActions(bpy_prop_collection[Action], bpy_struct):
    """Collection of actions"""

    def new(self, name: str | typing.Any) -> Action:
        """Add a new action to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New action data-block
        :rtype: Action
        """
        ...

    def remove(
        self,
        action: Action,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove an action from the current blendfile

        :param action: Action to remove
        :type action: Action
        :param do_unlink: Unlink all usages of this action before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this action
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this action
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataArmatures(bpy_prop_collection[Armature], bpy_struct):
    """Collection of armatures"""

    def new(self, name: str | typing.Any) -> Armature:
        """Add a new armature to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New armature data-block
        :rtype: Armature
        """
        ...

    def remove(
        self,
        armature: Armature,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove an armature from the current blendfile

        :param armature: Armature to remove
        :type armature: Armature
        :param do_unlink: Unlink all usages of this armature before deleting it (WARNING: will also delete objects instancing that armature data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this armature data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this armature data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataBrushes(bpy_prop_collection[Brush], bpy_struct):
    """Collection of brushes"""

    def new(self, name: str | typing.Any, mode: str | None = "TEXTURE_PAINT") -> Brush:
        """Add a new brush to the main database

                :param name: New name for the data-block
                :type name: str | typing.Any
                :param mode: Paint Mode for the new brush

        OBJECT
        Object Mode.

        EDIT
        Edit Mode.

        POSE
        Pose Mode.

        SCULPT
        Sculpt Mode.

        VERTEX_PAINT
        Vertex Paint.

        WEIGHT_PAINT
        Weight Paint.

        TEXTURE_PAINT
        Texture Paint.

        PARTICLE_EDIT
        Particle Edit.

        EDIT_GPENCIL
        Edit Mode -- Edit Grease Pencil Strokes.

        SCULPT_GPENCIL
        Sculpt Mode -- Sculpt Grease Pencil Strokes.

        PAINT_GPENCIL
        Draw Mode -- Paint Grease Pencil Strokes.

        WEIGHT_GPENCIL
        Weight Paint -- Grease Pencil Weight Paint Strokes.

        VERTEX_GPENCIL
        Vertex Paint -- Grease Pencil Vertex Paint Strokes.

        SCULPT_CURVES
        Sculpt Mode.
                :type mode: str | None
                :return: New brush data-block
                :rtype: Brush
        """
        ...

    def remove(
        self,
        brush: Brush,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a brush from the current blendfile

        :param brush: Brush to remove
        :type brush: Brush
        :param do_unlink: Unlink all usages of this brush before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this brush
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this brush
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def create_gpencil_data(self, brush: Brush):
        """Add grease pencil brush settings

        :param brush: Brush
        :type brush: Brush
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataCacheFiles(bpy_prop_collection[CacheFile], bpy_struct):
    """Collection of cache files"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataCameras(bpy_prop_collection[Camera], bpy_struct):
    """Collection of cameras"""

    def new(self, name: str | typing.Any) -> Camera:
        """Add a new camera to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New camera data-block
        :rtype: Camera
        """
        ...

    def remove(
        self,
        camera: Camera,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a camera from the current blendfile

        :param camera: Camera to remove
        :type camera: Camera
        :param do_unlink: Unlink all usages of this camera before deleting it (WARNING: will also delete objects instancing that camera data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this camera
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this camera
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataCollections(bpy_prop_collection[Collection], bpy_struct):
    """Collection of collections"""

    def new(self, name: str | typing.Any) -> Collection:
        """Add a new collection to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New collection data-block
        :rtype: Collection
        """
        ...

    def remove(
        self,
        collection: Collection,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a collection from the current blendfile

        :param collection: Collection to remove
        :type collection: Collection
        :param do_unlink: Unlink all usages of this collection before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this collection
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this collection
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataCurves(bpy_prop_collection[Curve], bpy_struct):
    """Collection of curves"""

    def new(self, name: str | typing.Any, type: str | None) -> Curve:
        """Add a new curve to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param type: Type, The type of curve to add
        :type type: str | None
        :return: New curve data-block
        :rtype: Curve
        """
        ...

    def remove(
        self,
        curve: Curve,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a curve from the current blendfile

        :param curve: Curve to remove
        :type curve: Curve
        :param do_unlink: Unlink all usages of this curve before deleting it (WARNING: will also delete objects instancing that curve data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this curve data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this curve data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataFonts(bpy_prop_collection[VectorFont], bpy_struct):
    """Collection of fonts"""

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> VectorFont:
        """Load a new font into the main database

        :param filepath: path of the font to load
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New font data-block
        :rtype: VectorFont
        """
        ...

    def remove(
        self,
        vfont: VectorFont,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a font from the current blendfile

        :param vfont: Font to remove
        :type vfont: VectorFont
        :param do_unlink: Unlink all usages of this font before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this font
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this font
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataGreasePencils(bpy_prop_collection[GreasePencil], bpy_struct):
    """Collection of grease pencils"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def new(self, name: str | typing.Any) -> GreasePencil:
        """Add a new grease pencil datablock to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New grease pencil data-block
        :rtype: GreasePencil
        """
        ...

    def remove(
        self,
        grease_pencil: GreasePencil,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a grease pencil instance from the current blendfile

        :param grease_pencil: Grease Pencil to remove
        :type grease_pencil: GreasePencil
        :param do_unlink: Unlink all usages of this grease pencil before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this grease pencil
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this grease pencil
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataImages(bpy_prop_collection[Image], bpy_struct):
    """Collection of images"""

    def new(
        self,
        name: str | typing.Any,
        width: int | None,
        height: int | None,
        alpha: bool | typing.Any | None = False,
        float_buffer: bool | typing.Any | None = False,
        stereo3d: bool | typing.Any | None = False,
        is_data: bool | typing.Any | None = False,
        tiled: bool | typing.Any | None = False,
    ) -> Image:
        """Add a new image to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param width: Width of the image
        :type width: int | None
        :param height: Height of the image
        :type height: int | None
        :param alpha: Alpha, Use alpha channel
        :type alpha: bool | typing.Any | None
        :param float_buffer: Float Buffer, Create an image with floating-point color
        :type float_buffer: bool | typing.Any | None
        :param stereo3d: Stereo 3D, Create left and right views
        :type stereo3d: bool | typing.Any | None
        :param is_data: Is Data, Create image with non-color data color space
        :type is_data: bool | typing.Any | None
        :param tiled: Tiled, Create a tiled image
        :type tiled: bool | typing.Any | None
        :return: New image data-block
        :rtype: Image
        """
        ...

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> Image:
        """Load a new image into the main database

        :param filepath: Path of the file to load
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New image data-block
        :rtype: Image
        """
        ...

    def remove(
        self,
        image: Image,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove an image from the current blendfile

        :param image: Image to remove
        :type image: Image
        :param do_unlink: Unlink all usages of this image before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this image
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this image
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataLattices(bpy_prop_collection[Lattice], bpy_struct):
    """Collection of lattices"""

    def new(self, name: str | typing.Any) -> Lattice:
        """Add a new lattice to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New lattice data-block
        :rtype: Lattice
        """
        ...

    def remove(
        self,
        lattice: Lattice,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a lattice from the current blendfile

        :param lattice: Lattice to remove
        :type lattice: Lattice
        :param do_unlink: Unlink all usages of this lattice before deleting it (WARNING: will also delete objects instancing that lattice data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this lattice data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this lattice data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataLibraries(bpy_prop_collection[Library], bpy_struct):
    """Collection of libraries"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def remove(
        self,
        library: Library,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a library from the current blendfile

        :param library: Library to remove
        :type library: Library
        :param do_unlink: Unlink all usages of this library before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this library
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this library
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

    def load(
        self,
        filepath: str | None,
        link: bool | None = False,
        relative: bool | None = False,
        assets_only: bool | None = False,
    ):
        """Returns a context manager which exposes 2 library objects on entering.
        Each object has attributes matching bpy.data which are lists of strings to be linked.

                :param filepath: The path to a blend file.
                :type filepath: str | None
                :param link: When False reference to the original file is lost.
                :type link: bool | None
                :param relative: When True the path is stored relative to the open blend file.
                :type relative: bool | None
                :param assets_only: If True, only list data-blocks marked as assets.
                :type assets_only: bool | None
        """
        ...

    def write(
        self,
        filepath: str | None,
        datablocks: set | None,
        path_remap: str | None = False,
        fake_user: bool | None = False,
        compress: bool | None = False,
    ):
        """Write data-blocks into a blend file.

                :param filepath: The path to write the blend-file.
                :type filepath: str | None
                :param datablocks: set of data-blocks (`bpy.types.ID` instances).
                :type datablocks: set | None
                :param path_remap: Optionally remap paths when writing the file:

        NONE No path manipulation (default).

        RELATIVE Remap paths that are already relative to the new location.

        RELATIVE_ALL Remap all paths to be relative to the new location.

        ABSOLUTE Make all paths absolute on writing.
                :type path_remap: str | None
                :param fake_user: When True, data-blocks will be written with fake-user flag enabled.
                :type fake_user: bool | None
                :param compress: When True, write a compressed blend file.
                :type compress: bool | None
        """
        ...

class BlendDataLights(bpy_prop_collection[Light], bpy_struct):
    """Collection of lights"""

    def new(self, name: str | typing.Any, type: str | None) -> Light:
        """Add a new light to the main database

                :param name: New name for the data-block
                :type name: str | typing.Any
                :param type: Type, The type of light to add

        POINT
        Point -- Omnidirectional point light source.

        SUN
        Sun -- Constant direction parallel ray light source.

        SPOT
        Spot -- Directional cone light source.

        AREA
        Area -- Directional area light source.
                :type type: str | None
                :return: New light data-block
                :rtype: Light
        """
        ...

    def remove(
        self,
        light: Light,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a light from the current blendfile

        :param light: Light to remove
        :type light: Light
        :param do_unlink: Unlink all usages of this light before deleting it (WARNING: will also delete objects instancing that light data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this light data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this light data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataLineStyles(bpy_prop_collection[FreestyleLineStyle], bpy_struct):
    """Collection of line styles"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def new(self, name: str | typing.Any) -> FreestyleLineStyle:
        """Add a new line style instance to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New line style data-block
        :rtype: FreestyleLineStyle
        """
        ...

    def remove(
        self,
        linestyle: FreestyleLineStyle,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a line style instance from the current blendfile

        :param linestyle: Line style to remove
        :type linestyle: FreestyleLineStyle
        :param do_unlink: Unlink all usages of this line style before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this line style
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this line style
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataMasks(bpy_prop_collection[Mask], bpy_struct):
    """Collection of masks"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def new(self, name: str | typing.Any) -> Mask:
        """Add a new mask with a given name to the main database

        :param name: Mask, Name of new mask data-block
        :type name: str | typing.Any
        :return: New mask data-block
        :rtype: Mask
        """
        ...

    def remove(
        self,
        mask: Mask,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a mask from the current blendfile

        :param mask: Mask to remove
        :type mask: Mask
        :param do_unlink: Unlink all usages of this mask before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this mask
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this mask
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    def new(self, name: str | typing.Any) -> Material:
        """Add a new material to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New material data-block
        :rtype: Material
        """
        ...

    def create_gpencil_data(self, material: Material):
        """Add grease pencil material settings

        :param material: Material
        :type material: Material
        """
        ...

    def remove_gpencil_data(self, material: Material):
        """Remove grease pencil material settings

        :param material: Material
        :type material: Material
        """
        ...

    def remove(
        self,
        material: Material,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a material from the current blendfile

        :param material: Material to remove
        :type material: Material
        :param do_unlink: Unlink all usages of this material before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this material
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this material
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataMeshes(bpy_prop_collection[Mesh], bpy_struct):
    """Collection of meshes"""

    def new(self, name: str | typing.Any) -> Mesh:
        """Add a new mesh to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New mesh data-block
        :rtype: Mesh
        """
        ...

    def new_from_object(
        self,
        object: Object,
        preserve_all_data_layers: bool | typing.Any | None = False,
        depsgraph: Depsgraph | None = None,
    ) -> Mesh:
        """Add a new mesh created from given object (undeformed geometry if object is original, and final evaluated geometry, with all modifiers etc., if object is evaluated)

        :param object: Object to create mesh from
        :type object: Object
        :param preserve_all_data_layers: Preserve all data layers in the mesh, like UV maps and vertex groups. By default Blender only computes the subset of data layers needed for viewport display and rendering, for better performance
        :type preserve_all_data_layers: bool | typing.Any | None
        :param depsgraph: Dependency Graph, Evaluated dependency graph which is required when preserve_all_data_layers is true
        :type depsgraph: Depsgraph | None
        :return: Mesh created from object, remove it if it is only used for export
        :rtype: Mesh
        """
        ...

    def remove(
        self,
        mesh: Mesh,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a mesh from the current blendfile

        :param mesh: Mesh to remove
        :type mesh: Mesh
        :param do_unlink: Unlink all usages of this mesh before deleting it (WARNING: will also delete objects instancing that mesh data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this mesh data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this mesh data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataMetaBalls(bpy_prop_collection[MetaBall], bpy_struct):
    """Collection of metaballs"""

    def new(self, name: str | typing.Any) -> MetaBall:
        """Add a new metaball to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New metaball data-block
        :rtype: MetaBall
        """
        ...

    def remove(
        self,
        metaball: MetaBall,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a metaball from the current blendfile

        :param metaball: Metaball to remove
        :type metaball: MetaBall
        :param do_unlink: Unlink all usages of this metaball before deleting it (WARNING: will also delete objects instancing that metaball data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this metaball data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this metaball data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataMovieClips(bpy_prop_collection[MovieClip], bpy_struct):
    """Collection of movie clips"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def remove(
        self,
        clip: MovieClip,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a movie clip from the current blendfile.

        :param clip: Movie clip to remove
        :type clip: MovieClip
        :param do_unlink: Unlink all usages of this movie clip before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this movie clip
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this movie clip
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> MovieClip:
        """Add a new movie clip to the main database from a file (while check_existing is disabled for consistency with other load functions, behavior with multiple movie-clips using the same file may incorrectly generate proxies)

        :param filepath: path for the data-block
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New movie clip data-block
        :rtype: MovieClip
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataNodeTrees(bpy_prop_collection[NodeTree], bpy_struct):
    """Collection of node trees"""

    def new(self, name: str | typing.Any, type: str | None) -> NodeTree:
        """Add a new node tree to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param type: Type, The type of node_group to add
        :type type: str | None
        :return: New node tree data-block
        :rtype: NodeTree
        """
        ...

    def remove(
        self,
        tree: NodeTree,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a node tree from the current blendfile

        :param tree: Node tree to remove
        :type tree: NodeTree
        :param do_unlink: Unlink all usages of this node tree before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this node tree
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this node tree
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of objects"""

    def new(self, name: str | typing.Any, object_data: ID | None) -> Object:
        """Add a new object to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param object_data: Object data or None for an empty object
        :type object_data: ID | None
        :return: New object data-block
        :rtype: Object
        """
        ...

    def remove(
        self,
        object: Object,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove an object from the current blendfile

        :param object: Object to remove
        :type object: Object
        :param do_unlink: Unlink all usages of this object before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this object
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this object
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataPaintCurves(bpy_prop_collection[PaintCurve], bpy_struct):
    """Collection of paint curves"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataPalettes(bpy_prop_collection[Palette], bpy_struct):
    """Collection of palettes"""

    def new(self, name: str | typing.Any) -> Palette:
        """Add a new palette to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New palette data-block
        :rtype: Palette
        """
        ...

    def remove(
        self,
        palette: Palette,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a palette from the current blendfile

        :param palette: Palette to remove
        :type palette: Palette
        :param do_unlink: Unlink all usages of this palette before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this palette
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this palette
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataParticles(bpy_prop_collection[ParticleSettings], bpy_struct):
    """Collection of particle settings"""

    def new(self, name: str | typing.Any) -> ParticleSettings:
        """Add a new particle settings instance to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New particle settings data-block
        :rtype: ParticleSettings
        """
        ...

    def remove(
        self,
        particle: ParticleSettings,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a particle settings instance from the current blendfile

        :param particle: Particle Settings to remove
        :type particle: ParticleSettings
        :param do_unlink: Unlink all usages of those particle settings before deleting them
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this particle settings
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this particle settings
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataPointClouds(bpy_prop_collection[PointCloud], bpy_struct):
    """Collection of point clouds"""

    def new(self, name: str | typing.Any) -> PointCloud:
        """Add a new point cloud to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New point cloud data-block
        :rtype: PointCloud
        """
        ...

    def remove(
        self,
        pointcloud: PointCloud,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a point cloud from the current blendfile

        :param pointcloud: Point cloud to remove
        :type pointcloud: PointCloud
        :param do_unlink: Unlink all usages of this point cloud before deleting it (WARNING: will also delete objects instancing that point cloud data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this point cloud data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this point cloud data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataProbes(bpy_prop_collection[LightProbe], bpy_struct):
    """Collection of light probes"""

    def new(self, name: str | typing.Any, type: str | None) -> LightProbe:
        """Add a new light probe to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param type: Type, The type of light probe to add
        :type type: str | None
        :return: New light probe data-block
        :rtype: LightProbe
        """
        ...

    def remove(
        self,
        lightprobe: LightProbe,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a light probe from the current blendfile

        :param lightprobe: Light probe to remove
        :type lightprobe: LightProbe
        :param do_unlink: Unlink all usages of this light probe before deleting it (WARNING: will also delete objects instancing that light probe data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this light probe
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this light probe
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataScenes(bpy_prop_collection[Scene], bpy_struct):
    """Collection of scenes"""

    def new(self, name: str | typing.Any) -> Scene:
        """Add a new scene to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New scene data-block
        :rtype: Scene
        """
        ...

    def remove(self, scene: Scene, do_unlink: bool | typing.Any | None = True):
        """Remove a scene from the current blendfile

        :param scene: Scene to remove
        :type scene: Scene
        :param do_unlink: Unlink all usages of this scene before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataScreens(bpy_prop_collection[Screen], bpy_struct):
    """Collection of screens"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataSounds(bpy_prop_collection[Sound], bpy_struct):
    """Collection of sounds"""

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> Sound:
        """Add a new sound to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New text data-block
        :rtype: Sound
        """
        ...

    def remove(
        self,
        sound: Sound,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a sound from the current blendfile

        :param sound: Sound to remove
        :type sound: Sound
        :param do_unlink: Unlink all usages of this sound before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this sound
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this sound
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataSpeakers(bpy_prop_collection[Speaker], bpy_struct):
    """Collection of speakers"""

    def new(self, name: str | typing.Any) -> Speaker:
        """Add a new speaker to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New speaker data-block
        :rtype: Speaker
        """
        ...

    def remove(
        self,
        speaker: Speaker,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a speaker from the current blendfile

        :param speaker: Speaker to remove
        :type speaker: Speaker
        :param do_unlink: Unlink all usages of this speaker before deleting it (WARNING: will also delete objects instancing that speaker data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this speaker data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this speaker data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataTexts(bpy_prop_collection[Text], bpy_struct):
    """Collection of texts"""

    def new(self, name: str | typing.Any) -> Text:
        """Add a new text to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New text data-block
        :rtype: Text
        """
        ...

    def remove(
        self,
        text: Text,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a text from the current blendfile

        :param text: Text to remove
        :type text: Text
        :param do_unlink: Unlink all usages of this text before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this text
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this text
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def load(
        self, filepath: str | typing.Any, internal: bool | typing.Any | None = False
    ) -> Text:
        """Add a new text to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str | typing.Any
        :param internal: Make internal, Make text file internal after loading
        :type internal: bool | typing.Any | None
        :return: New text data-block
        :rtype: Text
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataTextures(bpy_prop_collection[Texture], bpy_struct):
    """Collection of textures"""

    def new(self, name: str | typing.Any, type: str | None) -> Texture:
        """Add a new texture to the main database

                :param name: New name for the data-block
                :type name: str | typing.Any
                :param type: Type, The type of texture to add

        NONE
        None.

        BLEND
        Blend -- Procedural - create a ramp texture.

        CLOUDS
        Clouds -- Procedural - create a cloud-like fractal noise texture.

        DISTORTED_NOISE
        Distorted Noise -- Procedural - noise texture distorted by two noise algorithms.

        IMAGE
        Image or Movie -- Allow for images or movies to be used as textures.

        MAGIC
        Magic -- Procedural - color texture based on trigonometric functions.

        MARBLE
        Marble -- Procedural - marble-like noise texture with wave generated bands.

        MUSGRAVE
        Musgrave -- Procedural - highly flexible fractal noise texture.

        NOISE
        Noise -- Procedural - random noise, gives a different result every time, for every frame, for every pixel.

        STUCCI
        Stucci -- Procedural - create a fractal noise texture.

        VORONOI
        Voronoi -- Procedural - create cell-like patterns based on Worley noise.

        WOOD
        Wood -- Procedural - wave generated bands or rings, with optional noise.
                :type type: str | None
                :return: New texture data-block
                :rtype: Texture
        """
        ...

    def remove(
        self,
        texture: Texture,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a texture from the current blendfile

        :param texture: Texture to remove
        :type texture: Texture
        :param do_unlink: Unlink all usages of this texture before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this texture
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this texture
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataVolumes(bpy_prop_collection[Volume], bpy_struct):
    """Collection of volumes"""

    def new(self, name: str | typing.Any) -> Volume:
        """Add a new volume to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New volume data-block
        :rtype: Volume
        """
        ...

    def remove(
        self,
        volume: Volume,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a volume from the current blendfile

        :param volume: Volume to remove
        :type volume: Volume
        :param do_unlink: Unlink all usages of this volume before deleting it (WARNING: will also delete objects instancing that volume data)
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this volume data
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this volume data
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataWindowManagers(bpy_prop_collection[WindowManager], bpy_struct):
    """Collection of window managers"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataWorkSpaces(bpy_prop_collection[WorkSpace], bpy_struct):
    """Collection of workspaces"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlendDataWorlds(bpy_prop_collection[World], bpy_struct):
    """Collection of worlds"""

    def new(self, name: str | typing.Any) -> World:
        """Add a new world to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New world data-block
        :rtype: World
        """
        ...

    def remove(
        self,
        world: World,
        do_unlink: bool | typing.Any | None = True,
        do_id_user: bool | typing.Any | None = True,
        do_ui_user: bool | typing.Any | None = True,
    ):
        """Remove a world from the current blendfile

        :param world: World to remove
        :type world: World
        :param do_unlink: Unlink all usages of this world before deleting it
        :type do_unlink: bool | typing.Any | None
        :param do_id_user: Decrement user counter of all datablocks used by this world
        :type do_id_user: bool | typing.Any | None
        :param do_ui_user: Make sure interface does not reference this world
        :type do_ui_user: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BlenderRNA(bpy_struct):
    """Blender RNA structure definitions"""

    structs: bpy_prop_collection[Struct]
    """ 

    :type: bpy_prop_collection[Struct]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BoidRule(bpy_struct):
    name: str
    """ Boid rule name

    :type: str
    """

    type: str
    """ 

    :type: str
    """

    use_in_air: bool
    """ Use rule when boid is flying

    :type: bool
    """

    use_on_land: bool
    """ Use rule when boid is on land

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BoidSettings(bpy_struct):
    """Settings for boid physics"""

    accuracy: float
    """ Accuracy of attack

    :type: float
    """

    active_boid_state: BoidRule
    """ 

    :type: BoidRule
    """

    active_boid_state_index: int | None
    """ 

    :type: int | None
    """

    aggression: float
    """ Boid will fight this times stronger enemy

    :type: float
    """

    air_acc_max: float
    """ Maximum acceleration in air (relative to maximum speed)

    :type: float
    """

    air_ave_max: float
    """ Maximum angular velocity in air (relative to 180 degrees)

    :type: float
    """

    air_personal_space: float
    """ Radius of boids personal space in air (% of particle size)

    :type: float
    """

    air_speed_max: float
    """ Maximum speed in air

    :type: float
    """

    air_speed_min: float
    """ Minimum speed in air (relative to maximum speed)

    :type: float
    """

    bank: float
    """ Amount of rotation around velocity vector on turns

    :type: float
    """

    health: float
    """ Initial boid health when born

    :type: float
    """

    height: float
    """ Boid height relative to particle size

    :type: float
    """

    land_acc_max: float
    """ Maximum acceleration on land (relative to maximum speed)

    :type: float
    """

    land_ave_max: float
    """ Maximum angular velocity on land (relative to 180 degrees)

    :type: float
    """

    land_jump_speed: float
    """ Maximum speed for jumping

    :type: float
    """

    land_personal_space: float
    """ Radius of boids personal space on land (% of particle size)

    :type: float
    """

    land_smooth: float
    """ How smoothly the boids land

    :type: float
    """

    land_speed_max: float
    """ Maximum speed on land

    :type: float
    """

    land_stick_force: float
    """ How strong a force must be to start effecting a boid on land

    :type: float
    """

    pitch: float
    """ Amount of rotation around side vector

    :type: float
    """

    range: float
    """ Maximum distance from which a boid can attack

    :type: float
    """

    states: bpy_prop_collection[BoidState]
    """ 

    :type: bpy_prop_collection[BoidState]
    """

    strength: float
    """ Maximum caused damage on attack per second

    :type: float
    """

    use_climb: bool
    """ Allow boids to climb goal objects

    :type: bool
    """

    use_flight: bool
    """ Allow boids to move in air

    :type: bool
    """

    use_land: bool
    """ Allow boids to move on land

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BoidState(bpy_struct):
    """Boid state for boid physics"""

    active_boid_rule: BoidRule
    """ 

    :type: BoidRule
    """

    active_boid_rule_index: int | None
    """ 

    :type: int | None
    """

    falloff: float
    """ 

    :type: float
    """

    name: str
    """ Boid state name

    :type: str
    """

    rule_fuzzy: float
    """ 

    :type: float
    """

    rules: bpy_prop_collection[BoidRule]
    """ 

    :type: bpy_prop_collection[BoidRule]
    """

    ruleset_type: str
    """ How the rules in the list are evaluated

    :type: str
    """

    volume: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Bone(bpy_struct):
    """Bone in an Armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveinz: float
    """ Z-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutz: float
    """ Z-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_custom_handle_end: Bone
    """ Bone that serves as the end handle for the B-Bone curve

    :type: Bone
    """

    bbone_custom_handle_start: Bone
    """ Bone that serves as the start handle for the B-Bone curve

    :type: Bone
    """

    bbone_easein: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_easeout: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_handle_type_end: str
    """ Selects how the end handle of the B-Bone is computed

    :type: str
    """

    bbone_handle_type_start: str
    """ Selects how the start handle of the B-Bone is computed

    :type: str
    """

    bbone_handle_use_ease_end: bool
    """ Multiply the B-Bone Ease Out channel by the local Y scale value of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_ease_start: bool
    """ Multiply the B-Bone Ease In channel by the local Y scale value of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_scale_end: list[bool]
    """ Multiply B-Bone Scale Out channels by the local scale values of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: list[bool]
    """

    bbone_handle_use_scale_start: list[bool]
    """ Multiply B-Bone Scale In channels by the local scale values of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: list[bool]
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: bpy_prop_array[float]
    """ Scale factors for the start of the B-Bone, adjusts thickness (for tapering effects)

    :type: bpy_prop_array[float]
    """

    bbone_scaleout: bpy_prop_array[float]
    """ Scale factors for the end of the B-Bone, adjusts thickness (for tapering effects)

    :type: bpy_prop_array[float]
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    children: bpy_prop_collection[Bone]
    """ Bones which are children of this bone

    :type: bpy_prop_collection[Bone]
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: bpy_prop_array[float]
    """ Location of head end of the bone relative to its parent

    :type: bpy_prop_array[float]
    """

    head_local: bpy_prop_array[float]
    """ Location of head end of the bone relative to armature

    :type: bpy_prop_array[float]
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when it is not in Edit Mode (i.e. in Object or Pose Modes)

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    inherit_scale: str
    """ Specifies how the bone inherits scaling from the parent bone

    :type: str
    """

    layers: list[bool]
    """ Layers bone exists in

    :type: list[bool]
    """

    length: float
    """ Length of the bone

    :type: float
    """

    matrix: list[list[float]] | tuple[
        tuple[float, float, float],
        tuple[float, float, float],
        tuple[float, float, float],
    ]
    """ 3x3 bone matrix

    :type: list[list[float]] | tuple[tuple[float, float, float], tuple[float, float, float], tuple[float, float, float]]
    """

    matrix_local: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ 4x4 bone matrix relative to armature

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    name: str
    """ 

    :type: str
    """

    parent: Bone
    """ Parent bone (in same Armature)

    :type: Bone
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always displayed in wireframe regardless of viewport shading mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: bpy_prop_array[float]
    """ Location of tail end of the bone relative to its parent

    :type: bpy_prop_array[float]
    """

    tail_local: bpy_prop_array[float]
    """ Location of tail end of the bone relative to armature

    :type: bpy_prop_array[float]
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Add Roll Out of the Start Handle bone to the Roll In value

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_inherit_scale: bool
    """ DEPRECATED: Bone inherits scaling from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    use_scale_easing: bool
    """ Multiply the final easing values by the Scale In/Out Y factors

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.(readonly)"""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.(readonly)"""

    def evaluate_envelope(self, point: collections.abc.Iterable[float] | None) -> float:
        """Calculate bone envelope at given point

        :param point: Point, Position in 3d space to evaluate
        :type point: collections.abc.Iterable[float] | None
        :return: Factor, Envelope factor
        :rtype: float
        """
        ...

    def convert_local_to_pose(
        self,
        matrix: list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
        | None,
        matrix_local: list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
        | None,
        parent_matrix: typing.Any | None = (
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
        ),
        parent_matrix_local: typing.Any | None = (
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
        ),
        invert: bool | typing.Any | None = False,
    ) -> (
        list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
    ):
        """Transform a matrix from Local to Pose space (or back), taking into account options like Inherit Scale and Local Location. Unlike Object.convert_space, this uses custom rest and pose matrices provided by the caller. If the parent matrices are omitted, the bone is assumed to have no parent.This method enables conversions between Local and Pose space for bones in
        the middle of updating the armature without having to update dependencies
        after each change, by manually carrying updated matrices in a recursive walk.

                :param matrix: The matrix to transform
                :type matrix: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]] | None
                :param matrix_local: The custom rest matrix of this bone (Bone.matrix_local)
                :type matrix_local: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]] | None
                :param parent_matrix: The custom pose matrix of the parent bone (PoseBone.matrix)
                :type parent_matrix: typing.Any | None
                :param parent_matrix_local: The custom rest matrix of the parent bone (Bone.matrix_local)
                :type parent_matrix_local: typing.Any | None
                :param invert: Convert from Pose to Local space
                :type invert: bool | typing.Any | None
                :return: The transformed matrix
                :rtype: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
        """
        ...

    @classmethod
    def MatrixFromAxisRoll(
        cls, axis: typing.Any, roll: float | None
    ) -> (
        list[list[float]]
        | tuple[
            tuple[float, float, float],
            tuple[float, float, float],
            tuple[float, float, float],
        ]
    ):
        """Convert the axis + roll representation to a matrix

        :param axis: The main axis of the bone (tail - head)
        :type axis: typing.Any
        :param roll: The roll of the bone
        :type roll: float | None
        :return: The resulting orientation matrix
        :rtype: list[list[float]] | tuple[tuple[float, float, float], tuple[float, float, float], tuple[float, float, float]]
        """
        ...

    @classmethod
    def AxisRollFromMatrix(
        cls, matrix: typing.Any, axis: typing.Any | None = (0.0, 0.0, 0.0)
    ):
        """Convert a rotational matrix to the axis + roll representation. Note that the resulting value of the roll may not be as expected if the matrix has shear or negative determinant.

                :param matrix: The orientation matrix of the bone
                :type matrix: typing.Any
                :param axis: The optional override for the axis (finds closest approximation for the matrix)
                :type axis: typing.Any | None
                :return: result_axis, The main axis of the bone, float array of 3 items in [-inf, inf]

        result_roll, The roll of the bone, float in [-inf, inf]
        """
        ...

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """
        ...

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BoneGroup(bpy_struct):
    """Groups of Pose Channels (Bones)"""

    color_set: str
    """ Custom color set to use

    :type: str
    """

    colors: ThemeBoneColorSet
    """ Copy of the colors associated with the group's color set

    :type: ThemeBoneColorSet
    """

    is_custom_color_set: bool
    """ Color set is user-defined instead of a fixed theme color set

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BoneGroups(bpy_prop_collection[BoneGroup], bpy_struct):
    """Collection of bone groups"""

    active: BoneGroup | None
    """ Active bone group for this pose

    :type: BoneGroup | None
    """

    active_index: int | None
    """ Active index in bone groups array

    :type: int | None
    """

    def new(self, name: str | typing.Any = "Group") -> BoneGroup:
        """Add a new bone group to the object

        :param name: Name of the new group
        :type name: str | typing.Any
        :return: New bone group
        :rtype: BoneGroup
        """
        ...

    def remove(self, group: BoneGroup):
        """Remove a bone group from this object

        :param group: Removed bone group
        :type group: BoneGroup
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BoolAttributeValue(bpy_struct):
    """Bool value in geometry attribute"""

    value: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BrushCapabilities(bpy_struct):
    """Read-only indications of supported operations"""

    has_overlay: bool
    """ 

    :type: bool
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_smooth_stroke: bool
    """ 

    :type: bool
    """

    has_spacing: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BrushCapabilitiesImagePaint(bpy_struct):
    """Read-only indications of supported operations"""

    has_accumulate: bool
    """ 

    :type: bool
    """

    has_color: bool
    """ 

    :type: bool
    """

    has_radius: bool
    """ 

    :type: bool
    """

    has_space_attenuation: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BrushCapabilitiesSculpt(bpy_struct):
    """Read-only indications of which brush operations are supported by the current sculpt tool"""

    has_accumulate: bool
    """ 

    :type: bool
    """

    has_auto_smooth: bool
    """ 

    :type: bool
    """

    has_color: bool
    """ 

    :type: bool
    """

    has_direction: bool
    """ 

    :type: bool
    """

    has_gravity: bool
    """ 

    :type: bool
    """

    has_height: bool
    """ 

    :type: bool
    """

    has_jitter: bool
    """ 

    :type: bool
    """

    has_normal_weight: bool
    """ 

    :type: bool
    """

    has_persistence: bool
    """ 

    :type: bool
    """

    has_pinch_factor: bool
    """ 

    :type: bool
    """

    has_plane_offset: bool
    """ 

    :type: bool
    """

    has_rake_factor: bool
    """ 

    :type: bool
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_sculpt_plane: bool
    """ 

    :type: bool
    """

    has_secondary_color: bool
    """ 

    :type: bool
    """

    has_smooth_stroke: bool
    """ 

    :type: bool
    """

    has_space_attenuation: bool
    """ 

    :type: bool
    """

    has_strength_pressure: bool
    """ 

    :type: bool
    """

    has_tilt: bool
    """ 

    :type: bool
    """

    has_topology_rake: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BrushCapabilitiesVertexPaint(bpy_struct):
    """Read-only indications of supported operations"""

    has_color: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BrushCapabilitiesWeightPaint(bpy_struct):
    """Read-only indications of supported operations"""

    has_weight: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BrushCurvesSculptSettings(bpy_struct):
    add_amount: int
    """ Number of curves added by the Add brush

    :type: int
    """

    curve_length: float
    """ Length of newly added curves when it is not interpolated from other curves

    :type: float
    """

    interpolate_length: bool
    """ Use length of the curves in close proximity

    :type: bool
    """

    interpolate_shape: bool
    """ Use shape of the curves in close proximity

    :type: bool
    """

    minimum_length: float
    """ Avoid shrinking curves shorter than this length

    :type: float
    """

    scale_uniform: bool
    """ Grow or shrink curves by changing their size uniformly instead of using trimming or extrapolation

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class BrushGpencilSettings(bpy_struct):
    """Settings for grease pencil brush"""

    active_smooth_factor: float | None
    """ Amount of smoothing while drawing

    :type: float | None
    """

    angle: float
    """ Direction of the stroke at which brush gives maximal thickness (0° for horizontal)

    :type: float
    """

    angle_factor: float
    """ Reduce brush thickness by this factor when stroke is perpendicular to 'Angle' direction

    :type: float
    """

    aspect: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    brush_draw_mode: str
    """ Preselected mode when using this brush

    :type: str
    """

    caps_type: str
    """ The shape of the start and end of the stroke

    :type: str
    """

    curve_jitter: CurveMapping
    """ Curve used for the jitter effect

    :type: CurveMapping
    """

    curve_random_hue: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_pressure: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_saturation: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_strength: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_uv: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_value: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_sensitivity: CurveMapping
    """ Curve used for the sensitivity

    :type: CurveMapping
    """

    curve_strength: CurveMapping
    """ Curve used for the strength

    :type: CurveMapping
    """

    dilate: int
    """ Number of pixels to expand or contract fill area

    :type: int
    """

    direction: str
    """ 

    :type: str
    """

    eraser_mode: str
    """ Eraser Mode

    :type: str
    """

    eraser_strength_factor: float
    """ Amount of erasing for strength

    :type: float
    """

    eraser_thickness_factor: float
    """ Amount of erasing for thickness

    :type: float
    """

    extend_stroke_factor: float
    """ Strokes end extension for closing gaps, use zero to disable

    :type: float
    """

    fill_direction: str
    """ Direction of the fill

    :type: str
    """

    fill_draw_mode: str
    """ Mode to draw boundary limits

    :type: str
    """

    fill_factor: float
    """ Factor for fill boundary accuracy, higher values are more accurate but slower

    :type: float
    """

    fill_layer_mode: str
    """ Layers used as boundaries

    :type: str
    """

    fill_leak: int
    """ Size in pixels to consider the leak closed

    :type: int
    """

    fill_simplify_level: int
    """ Number of simplify steps (large values reduce fill accuracy)

    :type: int
    """

    fill_threshold: float
    """ Threshold to consider color transparent for filling

    :type: float
    """

    gpencil_paint_icon: str
    """ 

    :type: str
    """

    gpencil_sculpt_icon: str
    """ 

    :type: str
    """

    gpencil_vertex_icon: str
    """ 

    :type: str
    """

    gpencil_weight_icon: str
    """ 

    :type: str
    """

    hardness: float
    """ Gradient from the center of Dot and Box strokes (set to 1 for a solid stroke)

    :type: float
    """

    input_samples: int
    """ Generate intermediate points for very fast mouse movements. Set to 0 to disable

    :type: int
    """

    material: Material
    """ Material used for strokes drawn using this brush

    :type: Material
    """

    pen_jitter: float
    """ Jitter factor for new strokes

    :type: float
    """

    pen_smooth_factor: float
    """ Amount of smoothing to apply after finish newly created strokes, to reduce jitter/noise

    :type: float
    """

    pen_smooth_steps: int
    """ Number of times to smooth newly created strokes

    :type: int
    """

    pen_strength: float
    """ Color strength for new strokes (affect alpha factor of color)

    :type: float
    """

    pen_subdivision_steps: int
    """ Number of times to subdivide newly created strokes, for less jagged strokes

    :type: int
    """

    pin_draw_mode: bool
    """ Pin the mode to the brush

    :type: bool
    """

    random_hue_factor: float
    """ Random factor to modify original hue

    :type: float
    """

    random_pressure: float
    """ Randomness factor for pressure in new strokes

    :type: float
    """

    random_saturation_factor: float
    """ Random factor to modify original saturation

    :type: float
    """

    random_strength: float
    """ Randomness factor strength in new strokes

    :type: float
    """

    random_value_factor: float
    """ Random factor to modify original value

    :type: float
    """

    show_fill: bool
    """ Show transparent lines to use as boundary for filling

    :type: bool
    """

    show_fill_boundary: bool
    """ Show help lines for filling to see boundaries

    :type: bool
    """

    show_fill_extend: bool
    """ Show help lines for stroke extension

    :type: bool
    """

    show_lasso: bool
    """ Do not display fill color while drawing the stroke

    :type: bool
    """

    simplify_factor: float
    """ Factor of Simplify using adaptive algorithm

    :type: float
    """

    use_default_eraser: bool
    """ Use this brush when enable eraser with fast switch key

    :type: bool
    """

    use_edit_position: bool
    """ The brush affects the position of the point

    :type: bool
    """

    use_edit_strength: bool
    """ The brush affects the color strength of the point

    :type: bool
    """

    use_edit_thickness: bool
    """ The brush affects the thickness of the point

    :type: bool
    """

    use_edit_uv: bool
    """ The brush affects the UV rotation of the point

    :type: bool
    """

    use_fill_limit: bool
    """ Fill only visible areas in viewport

    :type: bool
    """

    use_jitter_pressure: bool
    """ Use tablet pressure for jitter

    :type: bool
    """

    use_material_pin: bool
    """ Keep material assigned to brush

    :type: bool
    """

    use_occlude_eraser: bool
    """ Erase only strokes visible and not occluded

    :type: bool
    """

    use_pressure: bool
    """ Use tablet pressure

    :type: bool
    """

    use_random_press_hue: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_radius: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_sat: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_strength: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_uv: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_val: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_settings_postprocess: bool
    """ Additional post processing options for new strokes

    :type: bool
    """

    use_settings_random: bool
    """ Random brush settings

    :type: bool
    """

    use_settings_stabilizer: bool
    """ Draw lines with a delay to allow smooth strokes. Press Shift key to override while drawing

    :type: bool
    """

    use_strength_pressure: bool
    """ Use tablet pressure for color strength

    :type: bool
    """

    use_stroke_random_hue: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_radius: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_sat: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_strength: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_uv: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_val: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_trim: bool
    """ Trim intersecting stroke ends

    :type: bool
    """

    uv_random: float
    """ Random factor for auto-generated UV rotation

    :type: float
    """

    vertex_color_factor: float
    """ Factor used to mix vertex color to get final color

    :type: float
    """

    vertex_mode: str
    """ Defines how vertex color affect to the strokes

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ByteColorAttributeValue(bpy_struct):
    """Color value in geometry attribute"""

    color: bpy_prop_array[float]
    """ RGBA color in scene linear color space

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ByteIntAttributeValue(bpy_struct):
    """8-bit value in geometry attribute"""

    value: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CacheFileLayer(bpy_struct):
    """Layer of the cache, used to load or override data from the first the first layer"""

    filepath: str
    """ Path to the archive

    :type: str
    """

    hide_layer: bool
    """ Do not load data from this layer

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CacheFileLayers(bpy_prop_collection[CacheFileLayer], bpy_struct):
    """Collection of cache layers"""

    active: CacheFileLayer | None
    """ Active layer of the CacheFile

    :type: CacheFileLayer | None
    """

    def new(self, filepath: str | typing.Any) -> CacheFileLayer:
        """Add a new layer

        :param filepath: File path to the archive used as a layer
        :type filepath: str | typing.Any
        :return: Newly created layer
        :rtype: CacheFileLayer
        """
        ...

    def remove(self, layer: CacheFileLayer):
        """Remove an existing layer from the cache file

        :param layer: Layer to remove
        :type layer: CacheFileLayer
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CacheObjectPath(bpy_struct):
    """Path of an object inside of an Alembic archive"""

    path: str
    """ Object path

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CacheObjectPaths(bpy_prop_collection[CacheObjectPath], bpy_struct):
    """Collection of object paths"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CameraBackgroundImage(bpy_struct):
    """Image and settings for display in the 3D View background"""

    alpha: float
    """ Image opacity to blend the image against the background color

    :type: float
    """

    clip: MovieClip
    """ Movie clip displayed and edited in this space

    :type: MovieClip
    """

    clip_user: MovieClipUser
    """ Parameters defining which frame of the movie clip is displayed

    :type: MovieClipUser
    """

    display_depth: str
    """ Display under or over everything

    :type: str
    """

    frame_method: str
    """ How the image fits in the camera frame

    :type: str
    """

    image: Image
    """ Image displayed and edited in this space

    :type: Image
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    offset: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    rotation: float
    """ Rotation for the background image (ortho view only)

    :type: float
    """

    scale: float
    """ Scale the background image

    :type: float
    """

    show_background_image: bool
    """ Show this image as background

    :type: bool
    """

    show_expanded: bool
    """ Show the expanded in the user interface

    :type: bool
    """

    show_on_foreground: bool
    """ Show this image in front of objects in viewport

    :type: bool
    """

    source: str
    """ Data source used for background

    :type: str
    """

    use_camera_clip: bool
    """ Use movie clip from active scene camera

    :type: bool
    """

    use_flip_x: bool
    """ Flip the background image horizontally

    :type: bool
    """

    use_flip_y: bool
    """ Flip the background image vertically

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CameraBackgroundImages(bpy_prop_collection[CameraBackgroundImage], bpy_struct):
    """Collection of background images"""

    def new(self) -> CameraBackgroundImage:
        """Add new background image

        :return: Image displayed as viewport background
        :rtype: CameraBackgroundImage
        """
        ...

    def remove(self, image: CameraBackgroundImage):
        """Remove background image

        :param image: Image displayed as viewport background
        :type image: CameraBackgroundImage
        """
        ...

    def clear(self):
        """Remove all background images"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CameraDOFSettings(bpy_struct):
    """Depth of Field settings"""

    aperture_blades: int
    """ Number of blades in aperture for polygonal bokeh (at least 3)

    :type: int
    """

    aperture_fstop: float
    """ F-Stop ratio (lower numbers give more defocus, higher numbers give a sharper image)

    :type: float
    """

    aperture_ratio: float
    """ Distortion to simulate anamorphic lens bokeh

    :type: float
    """

    aperture_rotation: float
    """ Rotation of blades in aperture

    :type: float
    """

    focus_distance: float
    """ Distance to the focus point for depth of field

    :type: float
    """

    focus_object: Object
    """ Use this object to define the depth of field focal point

    :type: Object
    """

    use_dof: bool
    """ Use Depth of Field

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CameraStereoData(bpy_struct):
    """Stereoscopy settings for a Camera data-block"""

    convergence_distance: float
    """ The converge point for the stereo cameras (often the distance between a projector and the projection screen)

    :type: float
    """

    convergence_mode: str
    """ 

    :type: str
    """

    interocular_distance: float
    """ Set the distance between the eyes - the stereo plane distance / 30 should be fine

    :type: float
    """

    pivot: str
    """ 

    :type: str
    """

    pole_merge_angle_from: float
    """ Angle at which interocular distance starts to fade to 0

    :type: float
    """

    pole_merge_angle_to: float
    """ Angle at which interocular distance is 0

    :type: float
    """

    use_pole_merge: bool
    """ Fade interocular distance to 0 after the given cutoff angle

    :type: bool
    """

    use_spherical_stereo: bool
    """ Render every pixel rotating the camera around the middle of the interocular distance

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ChannelDriverVariables(bpy_prop_collection[DriverVariable], bpy_struct):
    """Collection of channel driver Variables"""

    def new(self) -> DriverVariable:
        """Add a new variable for the driver

        :return: Newly created Driver Variable
        :rtype: DriverVariable
        """
        ...

    def remove(self, variable: DriverVariable):
        """Remove an existing variable from the driver

        :param variable: Variable to remove from the driver
        :type variable: DriverVariable
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ChildParticle(bpy_struct):
    """Child particle interpolated from simulated or edited particles"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ClothCollisionSettings(bpy_struct):
    """Cloth simulation settings for self collision and collision with other objects"""

    collection: Collection
    """ Limit colliders to this Collection

    :type: Collection
    """

    collision_quality: int
    """ How many collision iterations should be done. (higher is better quality but slower)

    :type: int
    """

    damping: float
    """ Amount of velocity lost on collision

    :type: float
    """

    distance_min: float
    """ Minimum distance between collision objects before collision response takes effect

    :type: float
    """

    friction: float
    """ Friction force if a collision happened (higher = less movement)

    :type: float
    """

    impulse_clamp: float
    """ Clamp collision impulses to avoid instability (0.0 to disable clamping)

    :type: float
    """

    self_distance_min: float
    """ Minimum distance between cloth faces before collision response takes effect

    :type: float
    """

    self_friction: float
    """ Friction with self contact

    :type: float
    """

    self_impulse_clamp: float
    """ Clamp collision impulses to avoid instability (0.0 to disable clamping)

    :type: float
    """

    use_collision: bool
    """ Enable collisions with other objects

    :type: bool
    """

    use_self_collision: bool
    """ Enable self collisions

    :type: bool
    """

    vertex_group_object_collisions: str
    """ Triangles with all vertices in this group are not used during object collisions

    :type: str
    """

    vertex_group_self_collisions: str
    """ Triangles with all vertices in this group are not used during self collisions

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ClothSettings(bpy_struct):
    """Cloth simulation settings for an object"""

    air_damping: float
    """ Air has normally some thickness which slows falling things down

    :type: float
    """

    bending_damping: float
    """ Amount of damping in bending behavior

    :type: float
    """

    bending_model: str
    """ Physical model for simulating bending forces

    :type: str
    """

    bending_stiffness: float
    """ How much the material resists bending

    :type: float
    """

    bending_stiffness_max: float
    """ Maximum bending stiffness value

    :type: float
    """

    collider_friction: float
    """ 

    :type: float
    """

    compression_damping: float
    """ Amount of damping in compression behavior

    :type: float
    """

    compression_stiffness: float
    """ How much the material resists compression

    :type: float
    """

    compression_stiffness_max: float
    """ Maximum compression stiffness value

    :type: float
    """

    density_strength: float
    """ Influence of target density on the simulation

    :type: float
    """

    density_target: float
    """ Maximum density of hair

    :type: float
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    fluid_density: float
    """ Density (kg/l) of the fluid contained inside the object, used to create a hydrostatic pressure gradient simulating the weight of the internal fluid, or buoyancy from the surrounding fluid if negative

    :type: float
    """

    goal_default: float
    """ Default Goal (vertex target position) value, when no Vertex Group used

    :type: float
    """

    goal_friction: float
    """ Goal (vertex target position) friction

    :type: float
    """

    goal_max: float
    """ Goal maximum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_min: float
    """ Goal minimum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_spring: float
    """ Goal (vertex target position) spring stiffness

    :type: float
    """

    gravity: bpy_prop_array[float]
    """ Gravity or external force vector

    :type: bpy_prop_array[float]
    """

    internal_compression_stiffness: float
    """ How much the material resists compression

    :type: float
    """

    internal_compression_stiffness_max: float
    """ Maximum compression stiffness value

    :type: float
    """

    internal_friction: float
    """ 

    :type: float
    """

    internal_spring_max_diversion: float
    """ How much the rays used to connect the internal points can diverge from the vertex normal

    :type: float
    """

    internal_spring_max_length: float
    """ The maximum length an internal spring can have during creation. If the distance between internal points is greater than this, no internal spring will be created between these points. A length of zero means that there is no length limit

    :type: float
    """

    internal_spring_normal_check: bool
    """ Require the points the internal springs connect to have opposite normal directions

    :type: bool
    """

    internal_tension_stiffness: float
    """ How much the material resists stretching

    :type: float
    """

    internal_tension_stiffness_max: float
    """ Maximum tension stiffness value

    :type: float
    """

    mass: float
    """ The mass of each vertex on the cloth material

    :type: float
    """

    pin_stiffness: float
    """ Pin (vertex target position) spring stiffness

    :type: float
    """

    pressure_factor: float
    """ Ambient pressure (kPa) that balances out between the inside and outside of the object when it has the target volume

    :type: float
    """

    quality: int
    """ Quality of the simulation in steps per frame (higher is better quality but slower)

    :type: int
    """

    rest_shape_key: ShapeKey
    """ Shape key to use the rest spring lengths from

    :type: ShapeKey
    """

    sewing_force_max: float
    """ Maximum sewing force

    :type: float
    """

    shear_damping: float
    """ Amount of damping in shearing behavior

    :type: float
    """

    shear_stiffness: float
    """ How much the material resists shearing

    :type: float
    """

    shear_stiffness_max: float
    """ Maximum shear scaling value

    :type: float
    """

    shrink_max: float
    """ Max amount to shrink cloth by

    :type: float
    """

    shrink_min: float
    """ Factor by which to shrink cloth

    :type: float
    """

    target_volume: float
    """ The mesh volume where the inner/outer pressure will be the same. If set to zero the change in volume will not affect pressure

    :type: float
    """

    tension_damping: float
    """ Amount of damping in stretching behavior

    :type: float
    """

    tension_stiffness: float
    """ How much the material resists stretching

    :type: float
    """

    tension_stiffness_max: float
    """ Maximum tension stiffness value

    :type: float
    """

    time_scale: float
    """ Cloth speed is multiplied by this value

    :type: float
    """

    uniform_pressure_force: float
    """ The uniform pressure that is constantly applied to the mesh, in units of Pressure Scale. Can be negative

    :type: float
    """

    use_dynamic_mesh: bool
    """ Make simulation respect deformations in the base mesh

    :type: bool
    """

    use_internal_springs: bool
    """ Simulate an internal volume structure by creating springs connecting the opposite sides of the mesh

    :type: bool
    """

    use_pressure: bool
    """ Simulate pressure inside a closed cloth mesh

    :type: bool
    """

    use_pressure_volume: bool
    """ Use the Target Volume parameter as the initial volume, instead of calculating it from the mesh itself

    :type: bool
    """

    use_sewing_springs: bool
    """ Pulls loose edges together

    :type: bool
    """

    vertex_group_bending: str
    """ Vertex group for fine control over bending stiffness

    :type: str
    """

    vertex_group_intern: str
    """ Vertex group for fine control over the internal spring stiffness

    :type: str
    """

    vertex_group_mass: str
    """ Vertex Group for pinning of vertices

    :type: str
    """

    vertex_group_pressure: str
    """ Vertex Group for where to apply pressure. Zero weight means no pressure while a weight of one means full pressure. Faces with a vertex that has zero weight will be excluded from the volume calculation

    :type: str
    """

    vertex_group_shear_stiffness: str
    """ Vertex group for fine control over shear stiffness

    :type: str
    """

    vertex_group_shrink: str
    """ Vertex Group for shrinking cloth

    :type: str
    """

    vertex_group_structural_stiffness: str
    """ Vertex group for fine control over structural stiffness

    :type: str
    """

    voxel_cell_size: float
    """ Size of the voxel grid cells for interaction effects

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ClothSolverResult(bpy_struct):
    """Result of cloth solver iteration"""

    avg_error: float
    """ Average error during substeps

    :type: float
    """

    avg_iterations: float
    """ Average iterations during substeps

    :type: float
    """

    max_error: float
    """ Maximum error during substeps

    :type: float
    """

    max_iterations: int
    """ Maximum iterations during substeps

    :type: int
    """

    min_error: float
    """ Minimum error during substeps

    :type: float
    """

    min_iterations: int
    """ Minimum iterations during substeps

    :type: int
    """

    status: set[str]
    """ Status of the solver iteration

    :type: set[str]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CollectionChildren(bpy_prop_collection[Collection], bpy_struct):
    """Collection of child collections"""

    def link(self, child: Collection):
        """Add this collection as child of this collection

        :param child: Collection to add
        :type child: Collection
        """
        ...

    def unlink(self, child: Collection | None):
        """Remove this child collection from a collection

        :param child: Collection to remove
        :type child: Collection | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CollectionObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of collection objects"""

    def link(self, object: Object):
        """Add this object to a collection

        :param object: Object to add
        :type object: Object
        """
        ...

    def unlink(self, object: Object | None):
        """Remove this object from a collection

        :param object: Object to remove
        :type object: Object | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CollisionSettings(bpy_struct):
    """Collision settings for object in physics simulation"""

    absorption: float
    """ How much of effector force gets lost during collision with this object (in percent)

    :type: float
    """

    cloth_friction: float
    """ Friction for cloth collisions

    :type: float
    """

    damping: float
    """ Amount of damping during collision

    :type: float
    """

    damping_factor: float
    """ Amount of damping during particle collision

    :type: float
    """

    damping_random: float
    """ Random variation of damping

    :type: float
    """

    friction_factor: float
    """ Amount of friction during particle collision

    :type: float
    """

    friction_random: float
    """ Random variation of friction

    :type: float
    """

    permeability: float
    """ Chance that the particle will pass through the mesh

    :type: float
    """

    stickiness: float
    """ Amount of stickiness to surface collision

    :type: float
    """

    thickness_inner: float
    """ Inner face thickness (only used by softbodies)

    :type: float
    """

    thickness_outer: float
    """ Outer face thickness

    :type: float
    """

    use: bool
    """ Enable this objects as a collider for physics systems

    :type: bool
    """

    use_culling: bool
    """ Cloth collision acts with respect to the collider normals (improves penetration recovery)

    :type: bool
    """

    use_normal: bool
    """ Cloth collision impulses act in the direction of the collider normals (more reliable in some cases)

    :type: bool
    """

    use_particle_kill: bool
    """ Kill collided particles

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorManagedDisplaySettings(bpy_struct):
    """Color management specific to display device"""

    display_device: str
    """ Display device name

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorManagedInputColorspaceSettings(bpy_struct):
    """Input color space settings"""

    is_data: bool
    """ Treat image as non-color data without color management, like normal or displacement maps

    :type: bool
    """

    name: str
    """ Color space in the image file, to convert to and from when saving and loading the image

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorManagedSequencerColorspaceSettings(bpy_struct):
    """Input color space settings"""

    name: str
    """ Color space that the sequencer operates in

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorManagedViewSettings(bpy_struct):
    """Color management settings used for displaying images on the display"""

    curve_mapping: CurveMapping
    """ Color curve mapping applied before display transform

    :type: CurveMapping
    """

    exposure: float
    """ Exposure (stops) applied before display transform

    :type: float
    """

    gamma: float
    """ Amount of gamma modification applied after display transform

    :type: float
    """

    look: str
    """ Additional transform applied before view transform for artistic needs

    :type: str
    """

    use_curve_mapping: bool
    """ Use RGB curved for pre-display transformation

    :type: bool
    """

    view_transform: str
    """ View used when converting image to a display space

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorMapping(bpy_struct):
    """Color mapping settings"""

    blend_color: bpy_prop_array[float]
    """ Blend color to mix with texture output color

    :type: bpy_prop_array[float]
    """

    blend_factor: float
    """ 

    :type: float
    """

    blend_type: str
    """ Mode used to mix with texture output color

    :type: str
    """

    brightness: float
    """ Adjust the brightness of the texture

    :type: float
    """

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    contrast: float
    """ Adjust the contrast of the texture

    :type: float
    """

    saturation: float
    """ Adjust the saturation of colors in the texture

    :type: float
    """

    use_color_ramp: bool
    """ Toggle color ramp operations

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorRamp(bpy_struct):
    """Color ramp mapping a scalar value to a color"""

    color_mode: str
    """ Set color mode to use for interpolation

    :type: str
    """

    elements: ColorRampElements
    """ 

    :type: ColorRampElements
    """

    hue_interpolation: str
    """ Set color interpolation

    :type: str
    """

    interpolation: str
    """ Set interpolation between color stops

    :type: str
    """

    def evaluate(self, position: float | None) -> bpy_prop_array[float]:
        """Evaluate ColorRamp

        :param position: Position, Evaluate ColorRamp at position
        :type position: float | None
        :return: Color, Color at given position
        :rtype: bpy_prop_array[float]
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorRampElement(bpy_struct):
    """Element defining a color at a position in the color ramp"""

    alpha: float
    """ Set alpha of selected color stop

    :type: float
    """

    color: bpy_prop_array[float]
    """ Set color of selected color stop

    :type: bpy_prop_array[float]
    """

    position: float
    """ Set position of selected color stop

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ColorRampElements(bpy_prop_collection[ColorRampElement], bpy_struct):
    """Collection of Color Ramp Elements"""

    def new(self, position: float | None) -> ColorRampElement:
        """Add element to ColorRamp

        :param position: Position, Position to add element
        :type position: float | None
        :return: New element
        :rtype: ColorRampElement
        """
        ...

    def remove(self, element: ColorRampElement):
        """Delete element from ColorRamp

        :param element: Element to remove
        :type element: ColorRampElement
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CompositorNodeOutputFileFileSlots(
    bpy_prop_collection[NodeOutputFileSlotFile], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str | typing.Any) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all file slots from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CompositorNodeOutputFileLayerSlots(
    bpy_prop_collection[NodeOutputFileSlotLayer], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str | typing.Any) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all file slots from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ConsoleLine(bpy_struct):
    """Input line for the interactive console"""

    body: str
    """ Text in the line

    :type: str
    """

    current_character: int
    """ 

    :type: int
    """

    type: str
    """ Console line type when used in scrollback

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Constraint(bpy_struct):
    """Constraint modifying the transformation of objects and bones"""

    active: bool | None
    """ Constraint is the one being edited

    :type: bool | None
    """

    enabled: bool
    """ Use the results of this constraint

    :type: bool
    """

    error_location: float
    """ Amount of residual error in Blender space unit for constraints that work on position

    :type: float
    """

    error_rotation: float
    """ Amount of residual error in radians for constraints that work on orientation

    :type: float
    """

    influence: float
    """ Amount of influence constraint will have on the final solution

    :type: float
    """

    is_override_data: bool
    """ In a local override object, whether this constraint comes from the linked reference object, or is local to the override

    :type: bool
    """

    is_valid: bool
    """ Constraint has valid settings and can be evaluated

    :type: bool
    """

    mute: bool
    """ Enable/Disable Constraint

    :type: bool
    """

    name: str
    """ Constraint name

    :type: str
    """

    owner_space: str
    """ Space that owner is evaluated in

    :type: str
    """

    show_expanded: bool
    """ Constraint's panel is expanded in UI

    :type: bool
    """

    space_object: Object
    """ Object for Custom Space

    :type: Object
    """

    space_subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target_space: str
    """ Space that target is evaluated in

    :type: str
    """

    type: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ConstraintTarget(bpy_struct):
    """Target object for multi-target constraints"""

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object
    """ Target object

    :type: Object
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ConstraintTargetBone(bpy_struct):
    """Target bone for multi-target constraints"""

    subtarget: str
    """ Target armature bone

    :type: str
    """

    target: Object
    """ Target armature

    :type: Object
    """

    weight: float
    """ Blending weight of this bone

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Context(bpy_struct):
    """Current windowmanager and data context"""

    area: Area
    """ 

    :type: Area
    """

    asset_file_handle: FileSelectEntry
    """ The file of an active asset. Avoid using this, it will be replaced by a proper AssetHandle design

    :type: FileSelectEntry
    """

    blend_data: BlendData
    """ 

    :type: BlendData
    """

    collection: Collection
    """ 

    :type: Collection
    """

    engine: str
    """ 

    :type: str
    """

    gizmo_group: GizmoGroup
    """ 

    :type: GizmoGroup
    """

    layer_collection: LayerCollection
    """ 

    :type: LayerCollection
    """

    mode: str
    """ 

    :type: str
    """

    preferences: Preferences
    """ 

    :type: Preferences
    """

    region: Region
    """ 

    :type: Region
    """

    region_data: RegionView3D
    """ 

    :type: RegionView3D
    """

    scene: Scene
    """ 

    :type: Scene
    """

    screen: Screen
    """ 

    :type: Screen
    """

    space_data: Space
    """ 

    :type: Space
    """

    tool_settings: ToolSettings
    """ 

    :type: ToolSettings
    """

    view_layer: ViewLayer
    """ 

    :type: ViewLayer
    """

    window: Window
    """ 

    :type: Window
    """

    window_manager: WindowManager
    """ 

    :type: WindowManager
    """

    workspace: WorkSpace
    """ 

    :type: WorkSpace
    """

    area: Area
    """ 

    :type: Area
    """

    asset_file_handle: FileSelectEntry
    """ The file of an active asset. Avoid using this, it will be replaced by a proper AssetHandle design

    :type: FileSelectEntry
    """

    blend_data: BlendData
    """ 

    :type: BlendData
    """

    collection: Collection
    """ 

    :type: Collection
    """

    engine: str
    """ 

    :type: str
    """

    gizmo_group: GizmoGroup
    """ 

    :type: GizmoGroup
    """

    layer_collection: LayerCollection
    """ 

    :type: LayerCollection
    """

    mode: str
    """ 

    :type: str
    """

    preferences: Preferences
    """ 

    :type: Preferences
    """

    region: Region
    """ 

    :type: Region
    """

    region_data: RegionView3D
    """ 

    :type: RegionView3D
    """

    scene: Scene
    """ 

    :type: Scene
    """

    screen: Screen
    """ 

    :type: Screen
    """

    space_data: Space
    """ 

    :type: Space
    """

    tool_settings: ToolSettings
    """ 

    :type: ToolSettings
    """

    view_layer: ViewLayer
    """ 

    :type: ViewLayer
    """

    window: Window
    """ 

    :type: Window
    """

    window_manager: WindowManager
    """ 

    :type: WindowManager
    """

    workspace: WorkSpace
    """ 

    :type: WorkSpace
    """

    scene: Scene
    """ 

    :type: Scene
    """

    view_layer: ViewLayer
    """ 

    :type: ViewLayer
    """

    visible_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selectable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_objects: list[Object]
    """ 

    :type: list[Object]
    """

    editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    objects_in_mode: list[Object]
    """ 

    :type: list[Object]
    """

    objects_in_mode_unique_data: list[Object]
    """ 

    :type: list[Object]
    """

    visible_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    visible_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    selected_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    selected_pose_bones_from_active_object: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    active_bone: EditBone | None
    """ 

    :type: EditBone | None
    """

    active_pose_bone: PoseBone | None
    """ 

    :type: PoseBone | None
    """

    active_object: Object | None
    """ 

    :type: Object | None
    """

    object: Object
    """ 

    :type: Object
    """

    edit_object: Object
    """ 

    :type: Object
    """

    sculpt_object: Object
    """ 

    :type: Object
    """

    vertex_paint_object: Object
    """ 

    :type: Object
    """

    weight_paint_object: Object
    """ 

    :type: Object
    """

    image_paint_object: Object
    """ 

    :type: Object
    """

    particle_edit_object: Object
    """ 

    :type: Object
    """

    pose_object: Object
    """ 

    :type: Object
    """

    active_sequence_strip: Sequence | None
    """ 

    :type: Sequence | None
    """

    sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_editable_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    active_nla_track: NlaTrack | None
    """ 

    :type: NlaTrack | None
    """

    active_nla_strip: NlaStrip | None
    """ 

    :type: NlaStrip | None
    """

    selected_nla_strips: list[NlaStrip]
    """ 

    :type: list[NlaStrip]
    """

    selected_movieclip_tracks: list[MovieTrackingTrack]
    """ 

    :type: list[MovieTrackingTrack]
    """

    gpencil_data: GreasePencil
    """ 

    :type: GreasePencil
    """

    gpencil_data_owner: ID
    """ 

    :type: ID
    """

    annotation_data: GreasePencil
    """ 

    :type: GreasePencil
    """

    annotation_data_owner: ID
    """ 

    :type: ID
    """

    visible_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_strokes: list[GPencilStroke]
    """ 

    :type: list[GPencilStroke]
    """

    active_gpencil_layer: list[GPencilLayer] | None
    """ 

    :type: list[GPencilLayer] | None
    """

    active_gpencil_frame: typing.Any
    active_annotation_layer: GPencilLayer | None
    """ 

    :type: GPencilLayer | None
    """

    active_operator: Operator | None
    """ 

    :type: Operator | None
    """

    selected_visible_actions: list[Action]
    """ 

    :type: list[Action]
    """

    selected_editable_actions: list[Action]
    """ 

    :type: list[Action]
    """

    visible_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    editable_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    selected_visible_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    selected_editable_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    active_editable_fcurve: FCurve | None
    """ 

    :type: FCurve | None
    """

    selected_editable_keyframes: list[Keyframe]
    """ 

    :type: list[Keyframe]
    """

    ui_list: UIList
    """ 

    :type: UIList
    """

    asset_library_ref: AssetLibraryReference
    """ 

    :type: AssetLibraryReference
    """

    active_object: Object | None
    """ 

    :type: Object | None
    """

    selected_ids: list[ID]
    """ 

    :type: list[ID]
    """

    texture_slot: TextureSlot
    """ 

    :type: TextureSlot
    """

    scene: Scene
    """ 

    :type: Scene
    """

    world: World
    """ 

    :type: World
    """

    object: Object
    """ 

    :type: Object
    """

    mesh: Mesh
    """ 

    :type: Mesh
    """

    armature: Armature
    """ 

    :type: Armature
    """

    lattice: Lattice
    """ 

    :type: Lattice
    """

    curve: Curve
    """ 

    :type: Curve
    """

    meta_ball: MetaBall
    """ 

    :type: MetaBall
    """

    light: Light
    """ 

    :type: Light
    """

    speaker: Speaker
    """ 

    :type: Speaker
    """

    lightprobe: LightProbe
    """ 

    :type: LightProbe
    """

    camera: Camera
    """ 

    :type: Camera
    """

    material: Material
    """ 

    :type: Material
    """

    material_slot: MaterialSlot
    """ 

    :type: MaterialSlot
    """

    texture: Texture
    """ 

    :type: Texture
    """

    texture_user: ID
    """ 

    :type: ID
    """

    texture_user_property: Property
    """ 

    :type: Property
    """

    bone: Bone
    """ 

    :type: Bone
    """

    edit_bone: EditBone
    """ 

    :type: EditBone
    """

    pose_bone: PoseBone
    """ 

    :type: PoseBone
    """

    particle_system: ParticleSystem
    """ 

    :type: ParticleSystem
    """

    particle_system_editable: ParticleSystem
    """ 

    :type: ParticleSystem
    """

    particle_settings: ParticleSettings
    """ 

    :type: ParticleSettings
    """

    cloth: ClothModifier
    """ 

    :type: ClothModifier
    """

    soft_body: SoftBodyModifier
    """ 

    :type: SoftBodyModifier
    """

    fluid: typing.Any
    collision: CollisionModifier
    """ 

    :type: CollisionModifier
    """

    brush: Brush
    """ 

    :type: Brush
    """

    dynamic_paint: DynamicPaintModifier
    """ 

    :type: DynamicPaintModifier
    """

    line_style: FreestyleLineStyle
    """ 

    :type: FreestyleLineStyle
    """

    collection: LayerCollection
    """ 

    :type: LayerCollection
    """

    gpencil: GreasePencil
    """ 

    :type: GreasePencil
    """

    volume: Volume
    """ 

    :type: Volume
    """

    edit_image: Image
    """ 

    :type: Image
    """

    edit_mask: Mask
    """ 

    :type: Mask
    """

    selected_nodes: list[Node]
    """ 

    :type: list[Node]
    """

    active_node: Node | None
    """ 

    :type: Node | None
    """

    light: Light
    """ 

    :type: Light
    """

    material: Material
    """ 

    :type: Material
    """

    world: World
    """ 

    :type: World
    """

    edit_text: Text
    """ 

    :type: Text
    """

    edit_movieclip: MovieClip
    """ 

    :type: MovieClip
    """

    edit_mask: Mask
    """ 

    :type: Mask
    """

    edit_mask: Mask
    """ 

    :type: Mask
    """

    active_file: FileSelectEntry | None
    """ 

    :type: FileSelectEntry | None
    """

    selected_files: list[FileSelectEntry]
    """ 

    :type: list[FileSelectEntry]
    """

    asset_library_ref: AssetLibraryReference
    """ 

    :type: AssetLibraryReference
    """

    selected_asset_files: list[FileSelectEntry]
    """ 

    :type: list[FileSelectEntry]
    """

    id: ID
    """ 

    :type: ID
    """

    def evaluated_depsgraph_get(self) -> Depsgraph:
        """Get the dependency graph for the current scene and view layer, to access to data-blocks with animation and modifiers applied. If any data-blocks have been edited, the dependency graph will be updated. This invalidates all references to evaluated data-blocks from the dependency graph.

        :return: Evaluated dependency graph
        :rtype: Depsgraph
        """
        ...

    def copy(self): ...
    def path_resolve(self, path: str | None, coerce: bool | None = True):
        """Returns the property from the path, raise an exception when not found.

        :param path: patch which this property resolves.
        :type path: str | None
        :param coerce: optional argument, when True, the property will be converted into its Python representation.
        :type coerce: bool | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

    def temp_override(
        self,
        window: Window | None,
        area: Area | None,
        region: Region | None,
        **keywords,
    ):
        """Context manager to temporarily override members in the context.Overriding the context can be used to temporarily activate another window / area & region,
        as well as other members such as the active_object or bone.Notes:Overriding the context can be useful to set the context after loading files
        (which would otherwise by None). For example:This example shows how it's possible to add an object to the scene in another window.

                :param window: Window override or None.
                :type window: Window | None
                :param area: Area override or None.
                :type area: Area | None
                :param region: Region override or None.
                :type region: Region | None
                :param keywords: Additional keywords override context members.
                :return: The context manager .
        """
        ...

class CryptomatteEntry(bpy_struct):
    encoded_hash: float
    """ 

    :type: float
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveMap(bpy_struct):
    """Curve in a curve mapping"""

    points: CurveMapPoints
    """ 

    :type: CurveMapPoints
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveMapPoint(bpy_struct):
    """Point of a curve used for a curve mapping"""

    handle_type: str
    """ Curve interpolation at this point: Bezier or vector

    :type: str
    """

    location: bpy_prop_array[float]
    """ X/Y coordinates of the curve point

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Selection state of the curve point

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveMapPoints(bpy_prop_collection[CurveMapPoint], bpy_struct):
    """Collection of Curve Map Points"""

    def new(self, position: float | None, value: float | None) -> CurveMapPoint:
        """Add point to CurveMap

        :param position: Position, Position to add point
        :type position: float | None
        :param value: Value, Value of point
        :type value: float | None
        :return: New point
        :rtype: CurveMapPoint
        """
        ...

    def remove(self, point: CurveMapPoint):
        """Delete point from CurveMap

        :param point: PointElement to remove
        :type point: CurveMapPoint
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveMapping(bpy_struct):
    """Curve mapping to map color, vector and scalar values to other values using a user defined curve"""

    black_level: bpy_prop_array[float]
    """ For RGB curves, the color that black is mapped to

    :type: bpy_prop_array[float]
    """

    clip_max_x: float
    """ 

    :type: float
    """

    clip_max_y: float
    """ 

    :type: float
    """

    clip_min_x: float
    """ 

    :type: float
    """

    clip_min_y: float
    """ 

    :type: float
    """

    curves: bpy_prop_collection[CurveMap]
    """ 

    :type: bpy_prop_collection[CurveMap]
    """

    extend: str
    """ Extrapolate the curve or extend it horizontally

    :type: str
    """

    tone: str
    """ Tone of the curve

    :type: str
    """

    use_clip: bool
    """ Force the curve view to fit a defined boundary

    :type: bool
    """

    white_level: bpy_prop_array[float]
    """ For RGB curves, the color that white is mapped to

    :type: bpy_prop_array[float]
    """

    def update(self):
        """Update curve mapping after making changes"""
        ...

    def reset_view(self):
        """Reset the curve mapping grid to its clipping size"""
        ...

    def initialize(self):
        """Initialize curve"""
        ...

    def evaluate(self, curve: CurveMap, position: float | None) -> float:
        """Evaluate curve at given location

        :param curve: curve, Curve to evaluate
        :type curve: CurveMap
        :param position: Position, Position to evaluate curve at
        :type position: float | None
        :return: Value, Value of curve at given location
        :rtype: float
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurvePaintSettings(bpy_struct):
    corner_angle: float
    """ Angles above this are considered corners

    :type: float
    """

    curve_type: str
    """ Type of curve to use for new strokes

    :type: str
    """

    depth_mode: str
    """ Method of projecting depth

    :type: str
    """

    error_threshold: int
    """ Allow deviation for a smoother, less precise line

    :type: int
    """

    fit_method: str
    """ Curve fitting method

    :type: str
    """

    radius_max: float
    """ Radius to use when the maximum pressure is applied (or when a tablet isn't used)

    :type: float
    """

    radius_min: float
    """ Minimum radius when the minimum pressure is applied (also the minimum when tapering)

    :type: float
    """

    radius_taper_end: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    radius_taper_start: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    surface_offset: float
    """ Offset the stroke from the surface

    :type: float
    """

    surface_plane: str
    """ Plane for projected stroke

    :type: str
    """

    use_corners_detect: bool
    """ Detect corners and use non-aligned handles

    :type: bool
    """

    use_offset_absolute: bool
    """ Apply a fixed offset (don't scale by the radius)

    :type: bool
    """

    use_pressure_radius: bool
    """ Map tablet pressure to curve radius

    :type: bool
    """

    use_stroke_endpoints: bool
    """ Use the start of the stroke for the depth

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveProfile(bpy_struct):
    """Profile Path editor used to build a profile path"""

    points: CurveProfilePoints
    """ Profile control points

    :type: CurveProfilePoints
    """

    preset: str
    """ 

    :type: str
    """

    segments: bpy_prop_collection[CurveProfilePoint]
    """ Segments sampled from control points

    :type: bpy_prop_collection[CurveProfilePoint]
    """

    use_clip: bool
    """ Force the path view to fit a defined boundary

    :type: bool
    """

    use_sample_even_lengths: bool
    """ Sample edges with even lengths

    :type: bool
    """

    use_sample_straight_edges: bool
    """ Sample edges with vector handles

    :type: bool
    """

    def update(self):
        """Refresh internal data, remove doubles and clip points"""
        ...

    def reset_view(self):
        """Reset the curve profile grid to its clipping size"""
        ...

    def initialize(self, totsegments: typing.Any):
        """Set the number of display segments and fill tables

        :param totsegments: The number of segment values to initialize the segments table with
        :type totsegments: typing.Any
        """
        ...

    def evaluate(self, length_portion: float | None) -> bpy_prop_array[float]:
        """Evaluate the at the given portion of the path length

        :param length_portion: Length Portion, Portion of the path length to travel before evaluation
        :type length_portion: float | None
        :return: Location, The location at the given portion of the profile
        :rtype: bpy_prop_array[float]
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveProfilePoint(bpy_struct):
    """Point of a path used to define a profile"""

    handle_type_1: str
    """ Path interpolation at this point

    :type: str
    """

    handle_type_2: str
    """ Path interpolation at this point

    :type: str
    """

    location: bpy_prop_array[float]
    """ X/Y coordinates of the path point

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Selection state of the path point

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveProfilePoints(bpy_prop_collection[CurveProfilePoint], bpy_struct):
    """Collection of Profile Points"""

    def add(self, x: float | None, y: float | None) -> CurveProfilePoint:
        """Add point to the profile

        :param x: X Position, X Position for new point
        :type x: float | None
        :param y: Y Position, Y Position for new point
        :type y: float | None
        :return: New point
        :rtype: CurveProfilePoint
        """
        ...

    def remove(self, point: CurveProfilePoint):
        """Delete point from the profile

        :param point: Point to remove
        :type point: CurveProfilePoint
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class CurveSplines(bpy_prop_collection[Spline], bpy_struct):
    """Collection of curve splines"""

    active: Spline | None
    """ Active curve spline

    :type: Spline | None
    """

    def new(self, type: str | None) -> Spline:
        """Add a new spline to the curve

        :param type: type for the new spline
        :type type: str | None
        :return: The newly created spline
        :rtype: Spline
        """
        ...

    def remove(self, spline: Spline):
        """Remove a spline from a curve

        :param spline: The spline to remove
        :type spline: Spline
        """
        ...

    def clear(self):
        """Remove all splines from a curve"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DashGpencilModifierSegment(bpy_struct):
    """Configuration for a single dash segment"""

    dash: int
    """ The number of consecutive points from the original stroke to include in this segment

    :type: int
    """

    gap: int
    """ The number of points skipped after this segment

    :type: int
    """

    material_index: int
    """ Use this index on generated segment. -1 means using the existing material

    :type: int
    """

    name: str
    """ Name of the dash segment

    :type: str
    """

    opacity: float
    """ The factor to apply to the original point's opacity for the new points

    :type: float
    """

    radius: float
    """ The factor to apply to the original point's radius for the new points

    :type: float
    """

    use_cyclic: bool
    """ Enable cyclic on individual stroke dashes

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Depsgraph(bpy_struct):
    ids: bpy_prop_collection[ID]
    """ All evaluated data-blocks

    :type: bpy_prop_collection[ID]
    """

    mode: str
    """ Evaluation mode

    :type: str
    """

    object_instances: bpy_prop_collection[DepsgraphObjectInstance]
    """ All object instances to display or render (Warning: Only use this as an iterator, never as a sequence, and do not keep any references to its items)

    :type: bpy_prop_collection[DepsgraphObjectInstance]
    """

    objects: bpy_prop_collection[Object]
    """ Evaluated objects in the dependency graph

    :type: bpy_prop_collection[Object]
    """

    scene: Scene
    """ Original scene dependency graph is built for

    :type: Scene
    """

    scene_eval: Scene
    """ Original scene dependency graph is built for

    :type: Scene
    """

    updates: bpy_prop_collection[DepsgraphUpdate]
    """ Updates to data-blocks

    :type: bpy_prop_collection[DepsgraphUpdate]
    """

    view_layer: ViewLayer
    """ Original view layer dependency graph is built for

    :type: ViewLayer
    """

    view_layer_eval: ViewLayer
    """ Original view layer dependency graph is built for

    :type: ViewLayer
    """

    def debug_relations_graphviz(self, filename: str | typing.Any):
        """debug_relations_graphviz

        :param filename: File Name, Output path for the graphviz debug file
        :type filename: str | typing.Any
        """
        ...

    def debug_stats_gnuplot(
        self, filename: str | typing.Any, output_filename: str | typing.Any
    ):
        """debug_stats_gnuplot

        :param filename: File Name, Output path for the gnuplot debug file
        :type filename: str | typing.Any
        :param output_filename: Output File Name, File name where gnuplot script will save the result
        :type output_filename: str | typing.Any
        """
        ...

    def debug_tag_update(self):
        """debug_tag_update"""
        ...

    def debug_stats(self) -> str | typing.Any:
        """Report the number of elements in the Dependency Graph

        :return: result
        :rtype: str | typing.Any
        """
        ...

    def update(self):
        """Re-evaluate any modified data-blocks, for example for animation or modifiers. This invalidates all references to evaluated data-blocks from this dependency graph."""
        ...

    def id_eval_get(self, id: ID | None) -> ID:
        """id_eval_get

        :param id: Original ID to get evaluated complementary part for
        :type id: ID | None
        :return: Evaluated ID for the given original one
        :rtype: ID
        """
        ...

    def id_type_updated(self, id_type: str | None) -> bool:
        """id_type_updated

        :param id_type: ID Type
        :type id_type: str | None
        :return: Updated, True if any datablock with this type was added, updated or removed
        :rtype: bool
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DepsgraphObjectInstance(bpy_struct):
    """Extended information about dependency graph object iterator (Warning: All data here is 'evaluated' one, not original .blend IDs)"""

    instance_object: Object
    """ Evaluated object which is being instanced by this iterator

    :type: Object
    """

    is_instance: bool
    """ Denotes if the object is generated by another object

    :type: bool
    """

    matrix_world: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Generated transform matrix in world space

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    object: Object
    """ Evaluated object the iterator points to

    :type: Object
    """

    orco: bpy_prop_array[float]
    """ Generated coordinates in parent object space

    :type: bpy_prop_array[float]
    """

    parent: Object
    """ If the object is an instance, the parent object that generated it

    :type: Object
    """

    particle_system: ParticleSystem
    """ Evaluated particle system that this object was instanced from

    :type: ParticleSystem
    """

    persistent_id: bpy_prop_array[int]
    """ Persistent identifier for inter-frame matching of objects with motion blur

    :type: bpy_prop_array[int]
    """

    random_id: int
    """ Random id for this instance, typically for randomized shading

    :type: int
    """

    show_particles: bool
    """ Particles part of the object should be visible in the render

    :type: bool
    """

    show_self: bool
    """ The object geometry itself should be visible in the render

    :type: bool
    """

    uv: bpy_prop_array[float]
    """ UV coordinates in parent object space

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DepsgraphUpdate(bpy_struct):
    """Information about ID that was updated"""

    id: ID
    """ Updated data-block

    :type: ID
    """

    is_updated_geometry: bool
    """ Object geometry is updated

    :type: bool
    """

    is_updated_shading: bool
    """ Object shading is updated

    :type: bool
    """

    is_updated_transform: bool
    """ Object transformation is updated

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DisplaySafeAreas(bpy_struct):
    """Safe areas used in 3D view and the sequencer"""

    action: bpy_prop_array[float]
    """ Safe area for general elements

    :type: bpy_prop_array[float]
    """

    action_center: bpy_prop_array[float]
    """ Safe area for general elements in a different aspect ratio

    :type: bpy_prop_array[float]
    """

    title: bpy_prop_array[float]
    """ Safe area for text and graphics

    :type: bpy_prop_array[float]
    """

    title_center: bpy_prop_array[float]
    """ Safe area for text and graphics in a different aspect ratio

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DopeSheet(bpy_struct):
    """Settings for filtering the channels shown in animation editors"""

    filter_collection: Collection
    """ Collection that included object should be a member of

    :type: Collection
    """

    filter_fcurve_name: str
    """ F-Curve live filtering string

    :type: str
    """

    filter_text: str
    """ Live filtering string

    :type: str
    """

    show_armatures: bool
    """ Include visualization of armature related animation data

    :type: bool
    """

    show_cache_files: bool
    """ Include visualization of cache file related animation data

    :type: bool
    """

    show_cameras: bool
    """ Include visualization of camera related animation data

    :type: bool
    """

    show_curves: bool
    """ Include visualization of curve related animation data

    :type: bool
    """

    show_datablock_filters: bool
    """ Show options for whether channels related to certain types of data are included

    :type: bool
    """

    show_expanded_summary: bool
    """ Collapse summary when shown, so all other channels get hidden (Dope Sheet editors only)

    :type: bool
    """

    show_gpencil: bool
    """ Include visualization of Grease Pencil related animation data and frames

    :type: bool
    """

    show_hair_curves: bool
    """ Include visualization of hair related animation data

    :type: bool
    """

    show_hidden: bool
    """ Include channels from objects/bone that are not visible

    :type: bool
    """

    show_lattices: bool
    """ Include visualization of lattice related animation data

    :type: bool
    """

    show_lights: bool
    """ Include visualization of light related animation data

    :type: bool
    """

    show_linestyles: bool
    """ Include visualization of Line Style related Animation data

    :type: bool
    """

    show_materials: bool
    """ Include visualization of material related animation data

    :type: bool
    """

    show_meshes: bool
    """ Include visualization of mesh related animation data

    :type: bool
    """

    show_metaballs: bool
    """ Include visualization of metaball related animation data

    :type: bool
    """

    show_missing_nla: bool
    """ Include animation data-blocks with no NLA data (NLA editor only)

    :type: bool
    """

    show_modifiers: bool
    """ Include visualization of animation data related to data-blocks linked to modifiers

    :type: bool
    """

    show_movieclips: bool
    """ Include visualization of movie clip related animation data

    :type: bool
    """

    show_nodes: bool
    """ Include visualization of node related animation data

    :type: bool
    """

    show_only_errors: bool
    """ Only include F-Curves and drivers that are disabled or have errors

    :type: bool
    """

    show_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    show_particles: bool
    """ Include visualization of particle related animation data

    :type: bool
    """

    show_pointclouds: bool
    """ Include visualization of point cloud related animation data

    :type: bool
    """

    show_scenes: bool
    """ Include visualization of scene related animation data

    :type: bool
    """

    show_shapekeys: bool
    """ Include visualization of shape key related animation data

    :type: bool
    """

    show_speakers: bool
    """ Include visualization of speaker related animation data

    :type: bool
    """

    show_summary: bool
    """ Display an additional 'summary' line (Dope Sheet editors only)

    :type: bool
    """

    show_textures: bool
    """ Include visualization of texture related animation data

    :type: bool
    """

    show_transforms: bool
    """ Include visualization of object-level animation data (mostly transforms)

    :type: bool
    """

    show_volumes: bool
    """ Include visualization of volume related animation data

    :type: bool
    """

    show_worlds: bool
    """ Include visualization of world related animation data

    :type: bool
    """

    source: ID
    """ ID-Block representing source data, usually ID_SCE (i.e. Scene)

    :type: ID
    """

    use_datablock_sort: bool
    """ Alphabetically sorts data-blocks - mainly objects in the scene (disable to increase viewport speed)

    :type: bool
    """

    use_filter_invert: bool
    """ Invert filter search

    :type: bool
    """

    use_multi_word_filter: bool
    """ Perform fuzzy/multi-word matching.
Warning: May be slow

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Driver(bpy_struct):
    """Driver for the value of a setting based on an external value"""

    expression: str
    """ Expression to use for Scripted Expression

    :type: str
    """

    is_simple_expression: bool
    """ The scripted expression can be evaluated without using the full python interpreter

    :type: bool
    """

    is_valid: bool
    """ Driver could not be evaluated in past, so should be skipped

    :type: bool
    """

    type: str
    """ Driver type

    :type: str
    """

    use_self: bool
    """ Include a 'self' variable in the name-space, so drivers can easily reference the data being modified (object, bone, etc...)

    :type: bool
    """

    variables: ChannelDriverVariables
    """ Properties acting as inputs for this driver

    :type: ChannelDriverVariables
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DriverTarget(bpy_struct):
    """Source of input values for driver variables"""

    bone_target: str
    """ Name of PoseBone to use as target

    :type: str
    """

    data_path: str
    """ RNA Path (from ID-block) to property used

    :type: str
    """

    id: ID
    """ ID-block that the specific property used can be found from (id_type property must be set first)

    :type: ID
    """

    id_type: str
    """ Type of ID-block that can be used

    :type: str
    """

    rotation_mode: str
    """ Mode for calculating rotation channel values

    :type: str
    """

    transform_space: str
    """ Space in which transforms are used

    :type: str
    """

    transform_type: str
    """ Driver variable type

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DriverVariable(bpy_struct):
    """Variable from some source/target for driver relationship"""

    is_name_valid: bool
    """ Is this a valid name for a driver variable

    :type: bool
    """

    name: str
    """ Name to use in scripted expressions/functions (no spaces or dots are allowed, and must start with a letter)

    :type: str
    """

    targets: bpy_prop_collection[DriverTarget]
    """ Sources of input data for evaluating this variable

    :type: bpy_prop_collection[DriverTarget]
    """

    type: str
    """ Driver variable type

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DynamicPaintBrushSettings(bpy_struct):
    """Brush settings"""

    invert_proximity: bool
    """ Proximity falloff is applied inside the volume

    :type: bool
    """

    paint_alpha: float
    """ Paint alpha

    :type: float
    """

    paint_color: bpy_prop_array[float]
    """ Color of the paint

    :type: bpy_prop_array[float]
    """

    paint_distance: float
    """ Maximum distance from brush to mesh surface to affect paint

    :type: float
    """

    paint_ramp: ColorRamp
    """ Color ramp used to define proximity falloff

    :type: ColorRamp
    """

    paint_source: str
    """ 

    :type: str
    """

    paint_wetness: float
    """ Paint wetness, visible in wetmap (some effects only affect wet paint)

    :type: float
    """

    particle_system: ParticleSystem
    """ The particle system to paint with

    :type: ParticleSystem
    """

    proximity_falloff: str
    """ Proximity falloff type

    :type: str
    """

    ray_direction: str
    """ Ray direction to use for projection (if brush object is located in that direction it's painted)

    :type: str
    """

    smooth_radius: float
    """ Smooth falloff added after solid radius

    :type: float
    """

    smudge_strength: float
    """ Smudge effect strength

    :type: float
    """

    solid_radius: float
    """ Radius that will be painted solid

    :type: float
    """

    use_absolute_alpha: bool
    """ Only increase alpha value if paint alpha is higher than existing

    :type: bool
    """

    use_negative_volume: bool
    """ Negate influence inside the volume

    :type: bool
    """

    use_paint_erase: bool
    """ Erase / remove paint instead of adding it

    :type: bool
    """

    use_particle_radius: bool
    """ Use radius from particle settings

    :type: bool
    """

    use_proximity_project: bool
    """ Brush is projected to canvas from defined direction within brush proximity

    :type: bool
    """

    use_proximity_ramp_alpha: bool
    """ Only read color ramp alpha

    :type: bool
    """

    use_smudge: bool
    """ Make this brush to smudge existing paint as it moves

    :type: bool
    """

    use_velocity_alpha: bool
    """ Multiply brush influence by velocity color ramp alpha

    :type: bool
    """

    use_velocity_color: bool
    """ Replace brush color by velocity color ramp

    :type: bool
    """

    use_velocity_depth: bool
    """ Multiply brush intersection depth (displace, waves) by velocity ramp alpha

    :type: bool
    """

    velocity_max: float
    """ Velocity considered as maximum influence (Blender units per frame)

    :type: float
    """

    velocity_ramp: ColorRamp
    """ Color ramp used to define brush velocity effect

    :type: ColorRamp
    """

    wave_clamp: float
    """ Maximum level of surface intersection used to influence waves (use 0.0 to disable)

    :type: float
    """

    wave_factor: float
    """ Multiplier for wave influence of this brush

    :type: float
    """

    wave_type: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DynamicPaintCanvasSettings(bpy_struct):
    """Dynamic Paint canvas settings"""

    canvas_surfaces: DynamicPaintSurfaces
    """ Paint surface list

    :type: DynamicPaintSurfaces
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DynamicPaintSurface(bpy_struct):
    """A canvas surface layer"""

    brush_collection: Collection
    """ Only use brush objects from this collection

    :type: Collection
    """

    brush_influence_scale: float
    """ Adjust influence brush objects have on this surface

    :type: float
    """

    brush_radius_scale: float
    """ Adjust radius of proximity brushes or particles for this surface

    :type: float
    """

    color_dry_threshold: float
    """ The wetness level when colors start to shift to the background

    :type: float
    """

    color_spread_speed: float
    """ How fast colors get mixed within wet paint

    :type: float
    """

    depth_clamp: float
    """ Maximum level of depth intersection in object space (use 0.0 to disable)

    :type: float
    """

    displace_factor: float
    """ Strength of displace when applied to the mesh

    :type: float
    """

    displace_type: str
    """ 

    :type: str
    """

    dissolve_speed: int
    """ Approximately in how many frames should dissolve happen

    :type: int
    """

    drip_acceleration: float
    """ How much surface acceleration affects dripping

    :type: float
    """

    drip_velocity: float
    """ How much surface velocity affects dripping

    :type: float
    """

    dry_speed: int
    """ Approximately in how many frames should drying happen

    :type: int
    """

    effect_ui: str
    """ 

    :type: str
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    frame_end: int
    """ Simulation end frame

    :type: int
    """

    frame_start: int
    """ Simulation start frame

    :type: int
    """

    frame_substeps: int
    """ Do extra frames between scene frames to ensure smooth motion

    :type: int
    """

    image_fileformat: str
    """ 

    :type: str
    """

    image_output_path: str
    """ Directory to save the textures

    :type: str
    """

    image_resolution: int
    """ Output image resolution

    :type: int
    """

    init_color: bpy_prop_array[float]
    """ Initial color of the surface

    :type: bpy_prop_array[float]
    """

    init_color_type: str
    """ 

    :type: str
    """

    init_layername: str
    """ 

    :type: str
    """

    init_texture: Texture
    """ 

    :type: Texture
    """

    is_active: bool
    """ Toggle whether surface is processed or ignored

    :type: bool
    """

    is_cache_user: bool
    """ 

    :type: bool
    """

    name: str
    """ Surface name

    :type: str
    """

    output_name_a: str
    """ Name used to save output from this surface

    :type: str
    """

    output_name_b: str
    """ Name used to save output from this surface

    :type: str
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    shrink_speed: float
    """ How fast shrink effect moves on the canvas surface

    :type: float
    """

    spread_speed: float
    """ How fast spread effect moves on the canvas surface

    :type: float
    """

    surface_format: str
    """ Surface Format

    :type: str
    """

    surface_type: str
    """ Surface Type

    :type: str
    """

    use_antialiasing: bool
    """ Use 5x multisampling to smooth paint edges

    :type: bool
    """

    use_dissolve: bool
    """ Enable to make surface changes disappear over time

    :type: bool
    """

    use_dissolve_log: bool
    """ Use logarithmic dissolve (makes high values to fade faster than low values)

    :type: bool
    """

    use_drip: bool
    """ Process drip effect (drip wet paint to gravity direction)

    :type: bool
    """

    use_dry_log: bool
    """ Use logarithmic drying (makes high values to dry faster than low values)

    :type: bool
    """

    use_drying: bool
    """ Enable to make surface wetness dry over time

    :type: bool
    """

    use_incremental_displace: bool
    """ New displace is added cumulatively on top of existing

    :type: bool
    """

    use_output_a: bool
    """ Save this output layer

    :type: bool
    """

    use_output_b: bool
    """ Save this output layer

    :type: bool
    """

    use_premultiply: bool
    """ Multiply color by alpha (recommended for Blender input)

    :type: bool
    """

    use_shrink: bool
    """ Process shrink effect (shrink paint areas)

    :type: bool
    """

    use_spread: bool
    """ Process spread effect (spread wet paint around surface)

    :type: bool
    """

    use_wave_open_border: bool
    """ Pass waves through mesh edges

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    wave_damping: float
    """ Wave damping factor

    :type: float
    """

    wave_smoothness: float
    """ Limit maximum steepness of wave slope between simulation points (use higher values for smoother waves at expense of reduced detail)

    :type: float
    """

    wave_speed: float
    """ Wave propagation speed

    :type: float
    """

    wave_spring: float
    """ Spring force that pulls water level back to zero

    :type: float
    """

    wave_timescale: float
    """ Wave time scaling factor

    :type: float
    """

    def output_exists(self, object: Object, index: int | None) -> bool:
        """Checks if surface output layer of given name exists

        :param object:
        :type object: Object
        :param index: Index
        :type index: int | None
        :return:
        :rtype: bool
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class DynamicPaintSurfaces(bpy_prop_collection[DynamicPaintSurface], bpy_struct):
    """Collection of Dynamic Paint Canvas surfaces"""

    active: DynamicPaintSurface
    """ Active Dynamic Paint surface being displayed

    :type: DynamicPaintSurface
    """

    active_index: int | None
    """ 

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class EditBone(bpy_struct):
    """Edit mode bone in an armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveinz: float
    """ Z-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutz: float
    """ Z-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_custom_handle_end: EditBone
    """ Bone that serves as the end handle for the B-Bone curve

    :type: EditBone
    """

    bbone_custom_handle_start: EditBone
    """ Bone that serves as the start handle for the B-Bone curve

    :type: EditBone
    """

    bbone_easein: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_easeout: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_handle_type_end: str
    """ Selects how the end handle of the B-Bone is computed

    :type: str
    """

    bbone_handle_type_start: str
    """ Selects how the start handle of the B-Bone is computed

    :type: str
    """

    bbone_handle_use_ease_end: bool
    """ Multiply the B-Bone Ease Out channel by the local Y scale value of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_ease_start: bool
    """ Multiply the B-Bone Ease In channel by the local Y scale value of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_scale_end: list[bool]
    """ Multiply B-Bone Scale Out channels by the local scale values of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: list[bool]
    """

    bbone_handle_use_scale_start: list[bool]
    """ Multiply B-Bone Scale In channels by the local scale values of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: list[bool]
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: bpy_prop_array[float]
    """ Scale factors for the start of the B-Bone, adjusts thickness (for tapering effects)

    :type: bpy_prop_array[float]
    """

    bbone_scaleout: bpy_prop_array[float]
    """ Scale factors for the end of the B-Bone, adjusts thickness (for tapering effects)

    :type: bpy_prop_array[float]
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: bpy_prop_array[float]
    """ Location of head end of the bone

    :type: bpy_prop_array[float]
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when in Edit Mode

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    inherit_scale: str
    """ Specifies how the bone inherits scaling from the parent bone

    :type: str
    """

    layers: list[bool]
    """ Layers bone exists in

    :type: list[bool]
    """

    length: float
    """ Length of the bone. Changing moves the tail end

    :type: float
    """

    lock: bool
    """ Bone is not able to be transformed when in Edit Mode

    :type: bool
    """

    matrix: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Matrix combining location and rotation of the bone (head position, direction and roll), in armature space (does not include/support bone's length/size)

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    name: str
    """ 

    :type: str
    """

    parent: EditBone
    """ Parent edit bone (in same Armature)

    :type: EditBone
    """

    roll: float
    """ Bone rotation around head-tail axis

    :type: float
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always displayed in wireframe regardless of viewport shading mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: bpy_prop_array[float]
    """ Location of tail end of the bone

    :type: bpy_prop_array[float]
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Add Roll Out of the Start Handle bone to the Roll In value

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_inherit_scale: bool
    """ DEPRECATED: Bone inherits scaling from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    use_scale_easing: bool
    """ Multiply the final easing values by the Scale In/Out Y factors

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.(readonly)"""

    children: typing.Any
    """ A list of all the bones children.(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.(readonly)"""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.(readonly)"""

    def align_roll(self, vector: collections.abc.Iterable[float] | None):
        """Align the bone to a local-space roll so the Z axis points in the direction of the vector given

        :param vector: Vector
        :type vector: collections.abc.Iterable[float] | None
        """
        ...

    def align_orientation(self, other):
        """Align this bone to another by moving its tail and settings its roll
        the length of the other bone is not used.

                :param other:
        """
        ...

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """
        ...

    def transform(
        self,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
        *,
        scale: bool | None = True,
        roll: bool | None = True,
    ):
        """Transform the the bones head, tail, roll and envelope
        (when the matrix has a scale component).

                :param matrix: 3x3 or 4x4 transformation matrix.
                :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param scale: Scale the bone envelope by the matrix.
                :type scale: bool | None
                :param roll: Correct the roll to point in the same relative
        direction to the head and tail.
                :type roll: bool | None
        """
        ...

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class EffectorWeights(bpy_struct):
    """Effector weights for physics simulation"""

    all: float
    """ All effector's weight

    :type: float
    """

    apply_to_hair_growing: bool
    """ Use force fields when growing hair

    :type: bool
    """

    boid: float
    """ Boid effector weight

    :type: float
    """

    charge: float
    """ Charge effector weight

    :type: float
    """

    collection: Collection
    """ Limit effectors to this collection

    :type: Collection
    """

    curve_guide: float
    """ Curve guide effector weight

    :type: float
    """

    drag: float
    """ Drag effector weight

    :type: float
    """

    force: float
    """ Force effector weight

    :type: float
    """

    gravity: float
    """ Global gravity weight

    :type: float
    """

    harmonic: float
    """ Harmonic effector weight

    :type: float
    """

    lennardjones: float
    """ Lennard-Jones effector weight

    :type: float
    """

    magnetic: float
    """ Magnetic effector weight

    :type: float
    """

    smokeflow: float
    """ Fluid Flow effector weight

    :type: float
    """

    texture: float
    """ Texture effector weight

    :type: float
    """

    turbulence: float
    """ Turbulence effector weight

    :type: float
    """

    vortex: float
    """ Vortex effector weight

    :type: float
    """

    wind: float
    """ Wind effector weight

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class EnumPropertyItem(bpy_struct):
    """Definition of a choice in an RNA enum property"""

    description: str
    """ Description of the item's purpose

    :type: str
    """

    icon: str
    """ Icon of the item

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    name: str
    """ Human readable name

    :type: str
    """

    value: int
    """ Value of the item

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Event(bpy_struct):
    """Window Manager Event"""

    alt: bool
    """ True when the Alt/Option key is held

    :type: bool
    """

    ascii: str
    """ Single ASCII character for this event

    :type: str
    """

    ctrl: bool
    """ True when the Ctrl key is held

    :type: bool
    """

    direction: str
    """ The direction (only applies to drag events)

    :type: str
    """

    is_mouse_absolute: bool
    """ The last motion event was an absolute input

    :type: bool
    """

    is_repeat: bool
    """ The event is generated by holding a key down

    :type: bool
    """

    is_tablet: bool
    """ The event has tablet data

    :type: bool
    """

    mouse_prev_press_x: int
    """ The window relative horizontal location of the last press event

    :type: int
    """

    mouse_prev_press_y: int
    """ The window relative vertical location of the last press event

    :type: int
    """

    mouse_prev_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_prev_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    mouse_region_x: int
    """ The region relative horizontal location of the mouse

    :type: int
    """

    mouse_region_y: int
    """ The region relative vertical location of the mouse

    :type: int
    """

    mouse_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    oskey: bool
    """ True when the Cmd key is held

    :type: bool
    """

    pressure: float
    """ The pressure of the tablet or 1.0 if no tablet present

    :type: float
    """

    shift: bool
    """ True when the Shift key is held

    :type: bool
    """

    tilt: bpy_prop_array[float]
    """ The pressure of the tablet or zeroes if no tablet present

    :type: bpy_prop_array[float]
    """

    type: str
    """ 

    :type: str
    """

    unicode: str
    """ Single unicode character for this event

    :type: str
    """

    value: str
    """ The type of event, only applies to some

    :type: str
    """

    xr: XrEventData
    """ XR event data

    :type: XrEventData
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FCurve(bpy_struct):
    """F-Curve defining values of a period of time"""

    array_index: int
    """ Index to the specific property affected by F-Curve if applicable

    :type: int
    """

    auto_smoothing: str
    """ Algorithm used to compute automatic handles

    :type: str
    """

    color: bpy_prop_array[float]
    """ Color of the F-Curve in the Graph Editor

    :type: bpy_prop_array[float]
    """

    color_mode: str
    """ Method used to determine color of F-Curve in Graph Editor

    :type: str
    """

    data_path: str
    """ RNA Path to property affected by F-Curve

    :type: str
    """

    driver: Driver
    """ Channel Driver (only set for Driver F-Curves)

    :type: Driver
    """

    extrapolation: str
    """ Method used for evaluating value of F-Curve outside first and last keyframes

    :type: str
    """

    group: ActionGroup
    """ Action Group that this F-Curve belongs to

    :type: ActionGroup
    """

    hide: bool
    """ F-Curve and its keyframes are hidden in the Graph Editor graphs

    :type: bool
    """

    is_empty: bool
    """ True if the curve contributes no animation due to lack of keyframes or useful modifiers, and should be deleted

    :type: bool
    """

    is_valid: bool
    """ False when F-Curve could not be evaluated in past, so should be skipped when evaluating

    :type: bool
    """

    keyframe_points: FCurveKeyframePoints
    """ User-editable keyframes

    :type: FCurveKeyframePoints
    """

    lock: bool
    """ F-Curve's settings cannot be edited

    :type: bool
    """

    modifiers: FCurveModifiers
    """ Modifiers affecting the shape of the F-Curve

    :type: FCurveModifiers
    """

    mute: bool
    """ Disable F-Curve evaluation

    :type: bool
    """

    sampled_points: bpy_prop_collection[FCurveSample]
    """ Sampled animation data

    :type: bpy_prop_collection[FCurveSample]
    """

    select: bool
    """ F-Curve is selected for editing

    :type: bool
    """

    def evaluate(self, frame: float | None) -> float:
        """Evaluate F-Curve

        :param frame: Frame, Evaluate F-Curve at given frame
        :type frame: float | None
        :return: Value, Value of F-Curve specific frame
        :rtype: float
        """
        ...

    def update(self):
        """Ensure keyframes are sorted in chronological order and handles are set correctly"""
        ...

    def range(self) -> bpy_prop_array[float]:
        """Get the time extents for F-Curve

        :return: Range, Min/Max values
        :rtype: bpy_prop_array[float]
        """
        ...

    def update_autoflags(self, data: typing.Any):
        """Update FCurve flags set automatically from affected property (currently, integer/discrete flags set when the property is not a float)

        :param data: Data, Data containing the property controlled by given FCurve
        :type data: typing.Any
        """
        ...

    def convert_to_samples(self, start: int | None, end: int | None):
        """Convert current FCurve from keyframes to sample points, if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """
        ...

    def convert_to_keyframes(self, start: int | None, end: int | None):
        """Convert current FCurve from sample points to keyframes (linear interpolation), if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FCurveKeyframePoints(bpy_prop_collection[Keyframe], bpy_struct):
    """Collection of keyframe points"""

    def insert(
        self,
        frame: float | None,
        value: float | None,
        options: set[str] | None = {},
        keyframe_type: str | None = "KEYFRAME",
    ) -> Keyframe:
        """Add a keyframe point to a F-Curve

                :param frame: X Value of this keyframe point
                :type frame: float | None
                :param value: Y Value of this keyframe point
                :type value: float | None
                :param options: Keyframe options

        REPLACE
        Replace -- Don't add any new keyframes, but just replace existing ones.

        NEEDED
        Needed -- Only adds keyframes that are needed.

        FAST
        Fast -- Fast keyframe insertion to avoid recalculating the curve each time.
                :type options: set[str] | None
                :param keyframe_type: Type of keyframe to insert

        KEYFRAME
        Keyframe -- Normal keyframe, e.g. for key poses.

        BREAKDOWN
        Breakdown -- A breakdown pose, e.g. for transitions between key poses.

        MOVING_HOLD
        Moving Hold -- A keyframe that is part of a moving hold.

        EXTREME
        Extreme -- An "extreme" pose, or some other purpose as needed.

        JITTER
        Jitter -- A filler or baked keyframe for keying on ones, or some other purpose as needed.
                :type keyframe_type: str | None
                :return: Newly created keyframe
                :rtype: Keyframe
        """
        ...

    def add(self, count: int | None):
        """Add a keyframe point to a F-Curve

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """
        ...

    def remove(self, keyframe: Keyframe, fast: bool | typing.Any | None = False):
        """Remove keyframe from an F-Curve

        :param keyframe: Keyframe to remove
        :type keyframe: Keyframe
        :param fast: Fast, Fast keyframe removal to avoid recalculating the curve each time
        :type fast: bool | typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FCurveModifiers(bpy_prop_collection[FModifier], bpy_struct):
    """Collection of F-Curve Modifiers"""

    active: FModifier | None
    """ Active F-Curve Modifier

    :type: FModifier | None
    """

    def new(self, type: str | None) -> FModifier:
        """Add a constraint to this object

                :param type: Constraint type to add

        NULL
        Invalid.

        GENERATOR
        Generator -- Generate a curve using a factorized or expanded polynomial.

        FNGENERATOR
        Built-In Function -- Generate a curve using standard math functions such as sin and cos.

        ENVELOPE
        Envelope -- Reshape F-Curve values, e.g. change amplitude of movements.

        CYCLES
        Cycles -- Cyclic extend/repeat keyframe sequence.

        NOISE
        Noise -- Add pseudo-random noise on top of F-Curves.

        LIMITS
        Limits -- Restrict maximum and minimum values of F-Curve.

        STEPPED
        Stepped Interpolation -- Snap values to nearest grid step, e.g. for a stop-motion look.
                :type type: str | None
                :return: New fmodifier
                :rtype: FModifier
        """
        ...

    def remove(self, modifier: FModifier):
        """Remove a modifier from this F-Curve

        :param modifier: Removed modifier
        :type modifier: FModifier
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FCurveSample(bpy_struct):
    """Sample point for F-Curve"""

    co: bpy_prop_array[float]
    """ Point coordinates

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Selection status

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FFmpegSettings(bpy_struct):
    """FFmpeg related settings for the scene"""

    audio_bitrate: int
    """ Audio bitrate (kb/s)

    :type: int
    """

    audio_channels: str
    """ Audio channel count

    :type: str
    """

    audio_codec: str
    """ FFmpeg audio codec to use

    :type: str
    """

    audio_mixrate: int
    """ Audio samplerate(samples/s)

    :type: int
    """

    audio_volume: float
    """ Audio volume

    :type: float
    """

    buffersize: int
    """ Rate control: buffer size (kb)

    :type: int
    """

    codec: str
    """ FFmpeg codec to use for video output

    :type: str
    """

    constant_rate_factor: str
    """ Constant Rate Factor (CRF); tradeoff between video quality and file size

    :type: str
    """

    ffmpeg_preset: str
    """ Tradeoff between encoding speed and compression ratio

    :type: str
    """

    format: str
    """ Output file container

    :type: str
    """

    gopsize: int
    """ Distance between key frames, also known as GOP size; influences file size and seekability

    :type: int
    """

    max_b_frames: int
    """ Maximum number of B-frames between non-B-frames; influences file size and seekability

    :type: int
    """

    maxrate: int
    """ Rate control: max rate (kbit/s)

    :type: int
    """

    minrate: int
    """ Rate control: min rate (kbit/s)

    :type: int
    """

    muxrate: int
    """ Mux rate (bits/second)

    :type: int
    """

    packetsize: int
    """ Mux packet size (byte)

    :type: int
    """

    use_autosplit: bool
    """ Autosplit output at 2GB boundary

    :type: bool
    """

    use_lossless_output: bool
    """ Use lossless output for video streams

    :type: bool
    """

    use_max_b_frames: bool
    """ Set a maximum number of B-frames

    :type: bool
    """

    video_bitrate: int
    """ Video bitrate (kbit/s)

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FModifier(bpy_struct):
    """Modifier for values of F-Curve"""

    active: bool | None
    """ F-Curve modifier will show settings in the editor

    :type: bool | None
    """

    blend_in: float
    """ Number of frames from start frame for influence to take effect

    :type: float
    """

    blend_out: float
    """ Number of frames from end frame for influence to fade out

    :type: float
    """

    frame_end: float
    """ Frame that modifier's influence ends (if Restrict Frame Range is in use)

    :type: float
    """

    frame_start: float
    """ Frame that modifier's influence starts (if Restrict Frame Range is in use)

    :type: float
    """

    influence: float
    """ Amount of influence F-Curve Modifier will have when not fading in/out

    :type: float
    """

    is_valid: bool
    """ F-Curve Modifier has invalid settings and will not be evaluated

    :type: bool
    """

    mute: bool
    """ Enable F-Curve modifier evaluation

    :type: bool
    """

    show_expanded: bool
    """ F-Curve Modifier's panel is expanded in UI

    :type: bool
    """

    type: str
    """ F-Curve Modifier Type

    :type: str
    """

    use_influence: bool
    """ F-Curve Modifier's effects will be tempered by a default factor

    :type: bool
    """

    use_restricted_range: bool
    """ F-Curve Modifier is only applied for the specified frame range to help mask off effects in order to chain them

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FModifierEnvelopeControlPoint(bpy_struct):
    """Control point for envelope F-Modifier"""

    frame: float
    """ Frame this control-point occurs on

    :type: float
    """

    max: float
    """ Upper bound of envelope at this control-point

    :type: float
    """

    min: float
    """ Lower bound of envelope at this control-point

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FModifierEnvelopeControlPoints(
    bpy_prop_collection[FModifierEnvelopeControlPoint], bpy_struct
):
    """Control points defining the shape of the envelope"""

    def add(self, frame: float | None) -> FModifierEnvelopeControlPoint:
        """Add a control point to a FModifierEnvelope

        :param frame: Frame to add this control-point
        :type frame: float | None
        :return: Newly created control-point
        :rtype: FModifierEnvelopeControlPoint
        """
        ...

    def remove(self, point: FModifierEnvelopeControlPoint):
        """Remove a control-point from an FModifierEnvelope

        :param point: Control-point to remove
        :type point: FModifierEnvelopeControlPoint
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FaceMap(bpy_struct):
    """Group of faces, each face can only be part of one map"""

    index: int
    """ Index number of the face map

    :type: int
    """

    name: str
    """ Face map name

    :type: str
    """

    select: bool
    """ Face map selection state (for tools to use)

    :type: bool
    """

    def add(self, index: collections.abc.Iterable[int] | None):
        """Add faces to the face-map

        :param index: List of indices
        :type index: collections.abc.Iterable[int] | None
        """
        ...

    def remove(self, index: collections.abc.Iterable[int] | None):
        """Remove faces from the face-map

        :param index: List of indices
        :type index: collections.abc.Iterable[int] | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FaceMaps(bpy_prop_collection[FaceMap], bpy_struct):
    """Collection of face maps"""

    active: FaceMap
    """ Face maps of the object

    :type: FaceMap
    """

    active_index: int | None
    """ Active index in face map array

    :type: int | None
    """

    def new(self, name: str | typing.Any = "Map") -> FaceMap:
        """Add face map to object

        :param name: face map name
        :type name: str | typing.Any
        :return: New face map
        :rtype: FaceMap
        """
        ...

    def remove(self, group: FaceMap):
        """Delete vertex group from object

        :param group: Face map to remove
        :type group: FaceMap
        """
        ...

    def clear(self):
        """Delete all vertex groups from object"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FieldSettings(bpy_struct):
    """Field settings for an object in physics simulation"""

    apply_to_location: bool
    """ Affect particle's location

    :type: bool
    """

    apply_to_rotation: bool
    """ Affect particle's dynamic rotation

    :type: bool
    """

    distance_max: float
    """ Maximum distance for the field to work

    :type: float
    """

    distance_min: float
    """ Minimum distance for the field's fall-off

    :type: float
    """

    falloff_power: float
    """ How quickly strength falls off with distance from the force field

    :type: float
    """

    falloff_type: str
    """ 

    :type: str
    """

    flow: float
    """ Convert effector force into air flow velocity

    :type: float
    """

    guide_clump_amount: float
    """ Amount of clumping

    :type: float
    """

    guide_clump_shape: float
    """ Shape of clumping

    :type: float
    """

    guide_free: float
    """ Guide-free time from particle life's end

    :type: float
    """

    guide_kink_amplitude: float
    """ The amplitude of the offset

    :type: float
    """

    guide_kink_axis: str
    """ Which axis to use for offset

    :type: str
    """

    guide_kink_frequency: float
    """ The frequency of the offset (1/total length)

    :type: float
    """

    guide_kink_shape: float
    """ Adjust the offset to the beginning/end

    :type: float
    """

    guide_kink_type: str
    """ Type of periodic offset on the curve

    :type: str
    """

    guide_minimum: float
    """ The distance from which particles are affected fully

    :type: float
    """

    harmonic_damping: float
    """ Damping of the harmonic force

    :type: float
    """

    inflow: float
    """ Inwards component of the vortex force

    :type: float
    """

    linear_drag: float
    """ Drag component proportional to velocity

    :type: float
    """

    noise: float
    """ Amount of noise for the force strength

    :type: float
    """

    quadratic_drag: float
    """ Drag component proportional to the square of velocity

    :type: float
    """

    radial_falloff: float
    """ Radial falloff power (real gravitational falloff = 2)

    :type: float
    """

    radial_max: float
    """ Maximum radial distance for the field to work

    :type: float
    """

    radial_min: float
    """ Minimum radial distance for the field's fall-off

    :type: float
    """

    rest_length: float
    """ Rest length of the harmonic force

    :type: float
    """

    seed: int
    """ Seed of the noise

    :type: int
    """

    shape: str
    """ Which direction is used to calculate the effector force

    :type: str
    """

    size: float
    """ Size of the turbulence

    :type: float
    """

    source_object: Object
    """ Select domain object of the smoke simulation

    :type: Object
    """

    strength: float
    """ Strength of force field

    :type: float
    """

    texture: Texture
    """ Texture to use as force

    :type: Texture
    """

    texture_mode: str
    """ How the texture effect is calculated (RGB and Curl need a RGB texture, else Gradient will be used instead)

    :type: str
    """

    texture_nabla: float
    """ Defines size of derivative offset used for calculating gradient and curl

    :type: float
    """

    type: str
    """ Type of field

    :type: str
    """

    use_2d_force: bool
    """ Apply force only in 2D

    :type: bool
    """

    use_absorption: bool
    """ Force gets absorbed by collision objects

    :type: bool
    """

    use_global_coords: bool
    """ Use effector/global coordinates for turbulence

    :type: bool
    """

    use_gravity_falloff: bool
    """ Multiply force by 1/distance²

    :type: bool
    """

    use_guide_path_add: bool
    """ Based on distance/falloff it adds a portion of the entire path

    :type: bool
    """

    use_guide_path_weight: bool
    """ Use curve weights to influence the particle influence along the curve

    :type: bool
    """

    use_max_distance: bool
    """ Use a maximum distance for the field to work

    :type: bool
    """

    use_min_distance: bool
    """ Use a minimum distance for the field's fall-off

    :type: bool
    """

    use_multiple_springs: bool
    """ Every point is effected by multiple springs

    :type: bool
    """

    use_object_coords: bool
    """ Use object/global coordinates for texture

    :type: bool
    """

    use_radial_max: bool
    """ Use a maximum radial distance for the field to work

    :type: bool
    """

    use_radial_min: bool
    """ Use a minimum radial distance for the field's fall-off

    :type: bool
    """

    use_root_coords: bool
    """ Texture coordinates from root particle locations

    :type: bool
    """

    use_smoke_density: bool
    """ Adjust force strength based on smoke density

    :type: bool
    """

    wind_factor: float
    """ How much the force is reduced when acting parallel to a surface, e.g. cloth

    :type: float
    """

    z_direction: str
    """ Effect in full or only positive/negative Z direction

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FileAssetSelectIDFilter(bpy_struct):
    """Which asset types to show/hide, when browsing an asset library"""

    experimental_filter_armature: bool
    """ Show Armature data-blocks

    :type: bool
    """

    experimental_filter_brush: bool
    """ Show Brushes data-blocks

    :type: bool
    """

    experimental_filter_cachefile: bool
    """ Show Cache File data-blocks

    :type: bool
    """

    experimental_filter_camera: bool
    """ Show Camera data-blocks

    :type: bool
    """

    experimental_filter_curve: bool
    """ Show Curve data-blocks

    :type: bool
    """

    experimental_filter_curves: bool
    """ Show/hide Curves data-blocks

    :type: bool
    """

    experimental_filter_font: bool
    """ Show Font data-blocks

    :type: bool
    """

    experimental_filter_grease_pencil: bool
    """ Show Grease pencil data-blocks

    :type: bool
    """

    experimental_filter_image: bool
    """ Show Image data-blocks

    :type: bool
    """

    experimental_filter_lattice: bool
    """ Show Lattice data-blocks

    :type: bool
    """

    experimental_filter_light: bool
    """ Show Light data-blocks

    :type: bool
    """

    experimental_filter_light_probe: bool
    """ Show Light Probe data-blocks

    :type: bool
    """

    experimental_filter_linestyle: bool
    """ Show Freestyle's Line Style data-blocks

    :type: bool
    """

    experimental_filter_mask: bool
    """ Show Mask data-blocks

    :type: bool
    """

    experimental_filter_mesh: bool
    """ Show Mesh data-blocks

    :type: bool
    """

    experimental_filter_metaball: bool
    """ Show Metaball data-blocks

    :type: bool
    """

    experimental_filter_movie_clip: bool
    """ Show Movie Clip data-blocks

    :type: bool
    """

    experimental_filter_paint_curve: bool
    """ Show Paint Curve data-blocks

    :type: bool
    """

    experimental_filter_palette: bool
    """ Show Palette data-blocks

    :type: bool
    """

    experimental_filter_particle_settings: bool
    """ Show Particle Settings data-blocks

    :type: bool
    """

    experimental_filter_pointcloud: bool
    """ Show/hide Point Cloud data-blocks

    :type: bool
    """

    experimental_filter_scene: bool
    """ Show Scene data-blocks

    :type: bool
    """

    experimental_filter_simulation: bool
    """ Show Simulation data-blocks

    :type: bool
    """

    experimental_filter_sound: bool
    """ Show Sound data-blocks

    :type: bool
    """

    experimental_filter_speaker: bool
    """ Show Speaker data-blocks

    :type: bool
    """

    experimental_filter_text: bool
    """ Show Text data-blocks

    :type: bool
    """

    experimental_filter_texture: bool
    """ Show Texture data-blocks

    :type: bool
    """

    experimental_filter_volume: bool
    """ Show/hide Volume data-blocks

    :type: bool
    """

    experimental_filter_work_space: bool
    """ Show workspace data-blocks

    :type: bool
    """

    filter_action: bool
    """ Show Action data-blocks

    :type: bool
    """

    filter_group: bool
    """ Show Collection data-blocks

    :type: bool
    """

    filter_material: bool
    """ Show Material data-blocks

    :type: bool
    """

    filter_node_tree: bool
    """ Show Node Tree data-blocks

    :type: bool
    """

    filter_object: bool
    """ Show Object data-blocks

    :type: bool
    """

    filter_world: bool
    """ Show World data-blocks

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FileBrowserFSMenuEntry(bpy_struct):
    """File Select Parameters"""

    icon: int
    """ 

    :type: int
    """

    is_valid: bool
    """ Whether this path is currently reachable

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    path: str
    """ 

    :type: str
    """

    use_save: bool
    """ Whether this path is saved in bookmarks, or generated from OS

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FileSelectEntry(bpy_struct):
    """A file viewable in the File Browser"""

    asset_data: AssetMetaData
    """ Asset data, valid if the file represents an asset

    :type: AssetMetaData
    """

    id_type: str
    """ The type of the data-block, if the file represents one ('NONE' otherwise)

    :type: str
    """

    local_id: ID
    """ The local data-block this file represents; only valid if that is a data-block in this file

    :type: ID
    """

    name: str
    """ 

    :type: str
    """

    preview_icon_id: int
    """ Unique integer identifying the preview of this file as an icon (zero means invalid)

    :type: int
    """

    relative_path: str
    """ Path relative to the directory currently displayed in the File Browser (includes the file name)

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FileSelectIDFilter(bpy_struct):
    """Which ID types to show/hide, when browsing a library"""

    category_animation: bool
    """ Show animation data

    :type: bool
    """

    category_environment: bool
    """ Show worlds, lights, cameras and speakers

    :type: bool
    """

    category_geometry: bool
    """ Show meshes, curves, lattice, armatures and metaballs data

    :type: bool
    """

    category_image: bool
    """ Show images, movie clips, sounds and masks

    :type: bool
    """

    category_misc: bool
    """ Show other data types

    :type: bool
    """

    category_object: bool
    """ Show objects and collections

    :type: bool
    """

    category_scene: bool
    """ Show scenes

    :type: bool
    """

    category_shading: bool
    """ Show materials, nodetrees, textures and Freestyle's linestyles

    :type: bool
    """

    filter_action: bool
    """ Show Action data-blocks

    :type: bool
    """

    filter_armature: bool
    """ Show Armature data-blocks

    :type: bool
    """

    filter_brush: bool
    """ Show Brushes data-blocks

    :type: bool
    """

    filter_cachefile: bool
    """ Show Cache File data-blocks

    :type: bool
    """

    filter_camera: bool
    """ Show Camera data-blocks

    :type: bool
    """

    filter_curve: bool
    """ Show Curve data-blocks

    :type: bool
    """

    filter_curves: bool
    """ Show/hide Curves data-blocks

    :type: bool
    """

    filter_font: bool
    """ Show Font data-blocks

    :type: bool
    """

    filter_grease_pencil: bool
    """ Show Grease pencil data-blocks

    :type: bool
    """

    filter_group: bool
    """ Show Collection data-blocks

    :type: bool
    """

    filter_image: bool
    """ Show Image data-blocks

    :type: bool
    """

    filter_lattice: bool
    """ Show Lattice data-blocks

    :type: bool
    """

    filter_light: bool
    """ Show Light data-blocks

    :type: bool
    """

    filter_light_probe: bool
    """ Show Light Probe data-blocks

    :type: bool
    """

    filter_linestyle: bool
    """ Show Freestyle's Line Style data-blocks

    :type: bool
    """

    filter_mask: bool
    """ Show Mask data-blocks

    :type: bool
    """

    filter_material: bool
    """ Show Material data-blocks

    :type: bool
    """

    filter_mesh: bool
    """ Show Mesh data-blocks

    :type: bool
    """

    filter_metaball: bool
    """ Show Metaball data-blocks

    :type: bool
    """

    filter_movie_clip: bool
    """ Show Movie Clip data-blocks

    :type: bool
    """

    filter_node_tree: bool
    """ Show Node Tree data-blocks

    :type: bool
    """

    filter_object: bool
    """ Show Object data-blocks

    :type: bool
    """

    filter_paint_curve: bool
    """ Show Paint Curve data-blocks

    :type: bool
    """

    filter_palette: bool
    """ Show Palette data-blocks

    :type: bool
    """

    filter_particle_settings: bool
    """ Show Particle Settings data-blocks

    :type: bool
    """

    filter_pointcloud: bool
    """ Show/hide Point Cloud data-blocks

    :type: bool
    """

    filter_scene: bool
    """ Show Scene data-blocks

    :type: bool
    """

    filter_simulation: bool
    """ Show Simulation data-blocks

    :type: bool
    """

    filter_sound: bool
    """ Show Sound data-blocks

    :type: bool
    """

    filter_speaker: bool
    """ Show Speaker data-blocks

    :type: bool
    """

    filter_text: bool
    """ Show Text data-blocks

    :type: bool
    """

    filter_texture: bool
    """ Show Texture data-blocks

    :type: bool
    """

    filter_volume: bool
    """ Show/hide Volume data-blocks

    :type: bool
    """

    filter_work_space: bool
    """ Show workspace data-blocks

    :type: bool
    """

    filter_world: bool
    """ Show World data-blocks

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FileSelectParams(bpy_struct):
    """File Select Parameters"""

    directory: str
    """ Directory displayed in the file browser

    :type: str
    """

    display_size: str
    """ Change the size of the display (width of columns or thumbnails size)

    :type: str
    """

    display_type: str
    """ Display mode for the file list

    :type: str
    """

    filename: str
    """ Active file in the file browser

    :type: str
    """

    filter_glob: str
    """ UNIX shell-like filename patterns matching, supports wildcards ('*') and list of patterns separated by ';'

    :type: str
    """

    filter_id: FileSelectIDFilter
    """ Which ID types to show/hide, when browsing a library

    :type: FileSelectIDFilter
    """

    filter_search: str
    """ Filter by name, supports '*' wildcard

    :type: str
    """

    recursion_level: str
    """ Numbers of dirtree levels to show simultaneously

    :type: str
    """

    show_details_datetime: bool
    """ Show a column listing the date and time of modification for each file

    :type: bool
    """

    show_details_size: bool
    """ Show a column listing the size of each file

    :type: bool
    """

    show_hidden: bool
    """ Show hidden dot files

    :type: bool
    """

    sort_method: str
    """ 

    :type: str
    """

    title: str
    """ Title for the file browser

    :type: str
    """

    use_filter: bool
    """ Enable filtering of files

    :type: bool
    """

    use_filter_asset_only: bool
    """ Hide .blend files items that are not data-blocks with asset metadata

    :type: bool
    """

    use_filter_backup: bool
    """ Show .blend1, .blend2, etc. files

    :type: bool
    """

    use_filter_blender: bool
    """ Show .blend files

    :type: bool
    """

    use_filter_blendid: bool
    """ Show .blend files items (objects, materials, etc.)

    :type: bool
    """

    use_filter_folder: bool
    """ Show folders

    :type: bool
    """

    use_filter_font: bool
    """ Show font files

    :type: bool
    """

    use_filter_image: bool
    """ Show image files

    :type: bool
    """

    use_filter_movie: bool
    """ Show movie files

    :type: bool
    """

    use_filter_script: bool
    """ Show script files

    :type: bool
    """

    use_filter_sound: bool
    """ Show sound files

    :type: bool
    """

    use_filter_text: bool
    """ Show text files

    :type: bool
    """

    use_filter_volume: bool
    """ Show 3D volume files

    :type: bool
    """

    use_library_browsing: bool
    """ Whether we may browse blender files' content or not

    :type: bool
    """

    use_sort_invert: bool
    """ Sort items descending, from highest value to lowest

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Float2AttributeValue(bpy_struct):
    """2D Vector value in geometry attribute"""

    vector: bpy_prop_array[float]
    """ 2D vector

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FloatAttributeValue(bpy_struct):
    """Floating-point value in geometry attribute"""

    value: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FloatColorAttributeValue(bpy_struct):
    """Color value in geometry attribute"""

    color: bpy_prop_array[float]
    """ RGBA color in scene linear color space

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FloatVectorAttributeValue(bpy_struct):
    """Vector value in geometry attribute"""

    vector: bpy_prop_array[float]
    """ 3D vector

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FluidDomainSettings(bpy_struct):
    """Fluid domain settings"""

    adapt_margin: int
    """ Margin added around fluid to minimize boundary interference

    :type: int
    """

    adapt_threshold: float
    """ Minimum amount of fluid a cell can contain before it is considered empty

    :type: float
    """

    additional_res: int
    """ Maximum number of additional cells

    :type: int
    """

    alpha: float
    """ Buoyant force based on smoke density (higher value results in faster rising smoke)

    :type: float
    """

    beta: float
    """ Buoyant force based on smoke heat (higher value results in faster rising smoke)

    :type: float
    """

    burning_rate: float
    """ Speed of the burning reaction (higher value results in smaller flames)

    :type: float
    """

    cache_data_format: str
    """ Select the file format to be used for caching volumetric data

    :type: str
    """

    cache_directory: str
    """ Directory that contains fluid cache files

    :type: str
    """

    cache_frame_end: int
    """ Frame on which the simulation stops. This is the last frame that will be baked

    :type: int
    """

    cache_frame_offset: int
    """ Frame offset that is used when loading the simulation from the cache. It is not considered when baking the simulation, only when loading it

    :type: int
    """

    cache_frame_pause_data: int
    """ 

    :type: int
    """

    cache_frame_pause_guide: int
    """ 

    :type: int
    """

    cache_frame_pause_mesh: int
    """ 

    :type: int
    """

    cache_frame_pause_noise: int
    """ 

    :type: int
    """

    cache_frame_pause_particles: int
    """ 

    :type: int
    """

    cache_frame_start: int
    """ Frame on which the simulation starts. This is the first frame that will be baked

    :type: int
    """

    cache_mesh_format: str
    """ Select the file format to be used for caching surface data

    :type: str
    """

    cache_noise_format: str
    """ Select the file format to be used for caching noise data

    :type: str
    """

    cache_particle_format: str
    """ Select the file format to be used for caching particle data

    :type: str
    """

    cache_resumable: bool
    """ Additional data will be saved so that the bake jobs can be resumed after pausing. Because more data will be written to disk it is recommended to avoid enabling this option when baking at high resolutions

    :type: bool
    """

    cache_type: str
    """ Change the cache type of the simulation

    :type: str
    """

    cell_size: bpy_prop_array[float]
    """ Cell Size

    :type: bpy_prop_array[float]
    """

    cfl_condition: float
    """ Maximal velocity per cell (greater CFL numbers will minimize the number of simulation steps and the computation time.)

    :type: float
    """

    clipping: float
    """ Value under which voxels are considered empty space to optimize rendering

    :type: float
    """

    color_grid: bpy_prop_array[float]
    """ Smoke color grid

    :type: bpy_prop_array[float]
    """

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    color_ramp_field: str
    """ Simulation field to color map

    :type: str
    """

    color_ramp_field_scale: float
    """ Multiplier for scaling the selected field to color map

    :type: float
    """

    delete_in_obstacle: bool
    """ Delete fluid inside obstacles

    :type: bool
    """

    density_grid: bpy_prop_array[float]
    """ Smoke density grid

    :type: bpy_prop_array[float]
    """

    display_interpolation: str
    """ Interpolation method to use for smoke/fire volumes in solid mode

    :type: str
    """

    display_thickness: float
    """ Thickness of smoke display in the viewport

    :type: float
    """

    dissolve_speed: int
    """ Determine how quickly the smoke dissolves (lower value makes smoke disappear faster)

    :type: int
    """

    domain_resolution: bpy_prop_array[int]
    """ Smoke Grid Resolution

    :type: bpy_prop_array[int]
    """

    domain_type: str
    """ Change domain type of the simulation

    :type: str
    """

    effector_group: Collection
    """ Limit effectors to this collection

    :type: Collection
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    export_manta_script: bool
    """ Generate and export Mantaflow script from current domain settings during bake. This is only needed if you plan to analyze the cache (e.g. view grids, velocity vectors, particles) in Mantaflow directly (outside of Blender) after baking the simulation

    :type: bool
    """

    flame_grid: bpy_prop_array[float]
    """ Smoke flame grid

    :type: bpy_prop_array[float]
    """

    flame_ignition: float
    """ Minimum temperature of the flames (higher value results in faster rising flames)

    :type: float
    """

    flame_max_temp: float
    """ Maximum temperature of the flames (higher value results in faster rising flames)

    :type: float
    """

    flame_smoke: float
    """ Amount of smoke created by burning fuel

    :type: float
    """

    flame_smoke_color: bpy_prop_array[float]
    """ Color of smoke emitted from burning fuel

    :type: bpy_prop_array[float]
    """

    flame_vorticity: float
    """ Additional vorticity for the flames

    :type: float
    """

    flip_ratio: float
    """ PIC/FLIP Ratio. A value of 1.0 will result in a completely FLIP based simulation. Use a lower value for simulations which should produce smaller splashes

    :type: float
    """

    fluid_group: Collection
    """ Limit fluid objects to this collection

    :type: Collection
    """

    force_collection: Collection
    """ Limit forces to this collection

    :type: Collection
    """

    fractions_distance: float
    """ Determines how far apart fluid and obstacle are (higher values will result in fluid being further away from obstacles, smaller values will let fluid move towards the inside of obstacles)

    :type: float
    """

    fractions_threshold: float
    """ Determines how much fluid is allowed in an obstacle cell (higher values will tag a boundary cell as an obstacle easier and reduce the boundary smoothening effect)

    :type: float
    """

    gravity: bpy_prop_array[float]
    """ Gravity in X, Y and Z direction

    :type: bpy_prop_array[float]
    """

    gridlines_cell_filter: str
    """ Cell type to be highlighted

    :type: str
    """

    gridlines_color_field: str
    """ Simulation field to color map onto gridlines

    :type: str
    """

    gridlines_lower_bound: float
    """ Lower bound of the highlighting range

    :type: float
    """

    gridlines_range_color: bpy_prop_array[float]
    """ Color used to highlight the range

    :type: bpy_prop_array[float]
    """

    gridlines_upper_bound: float
    """ Upper bound of the highlighting range

    :type: float
    """

    guide_alpha: float
    """ Guiding weight (higher value results in greater lag)

    :type: float
    """

    guide_beta: int
    """ Guiding size (higher value results in larger vortices)

    :type: int
    """

    guide_parent: Object
    """ Use velocities from this object for the guiding effect (object needs to have fluid modifier and be of type domain))

    :type: Object
    """

    guide_source: str
    """ Choose where to get guiding velocities from

    :type: str
    """

    guide_vel_factor: float
    """ Guiding velocity factor (higher value results in greater guiding velocities)

    :type: float
    """

    has_cache_baked_any: bool
    """ 

    :type: bool
    """

    has_cache_baked_data: bool
    """ 

    :type: bool
    """

    has_cache_baked_guide: bool
    """ 

    :type: bool
    """

    has_cache_baked_mesh: bool
    """ 

    :type: bool
    """

    has_cache_baked_noise: bool
    """ 

    :type: bool
    """

    has_cache_baked_particles: bool
    """ 

    :type: bool
    """

    heat_grid: bpy_prop_array[float]
    """ Smoke heat grid

    :type: bpy_prop_array[float]
    """

    highres_sampling: str
    """ Method for sampling the high resolution flow

    :type: str
    """

    is_cache_baking_any: bool
    """ 

    :type: bool
    """

    is_cache_baking_data: bool
    """ 

    :type: bool
    """

    is_cache_baking_guide: bool
    """ 

    :type: bool
    """

    is_cache_baking_mesh: bool
    """ 

    :type: bool
    """

    is_cache_baking_noise: bool
    """ 

    :type: bool
    """

    is_cache_baking_particles: bool
    """ 

    :type: bool
    """

    mesh_concave_lower: float
    """ Lower mesh concavity bound (high values tend to smoothen and fill out concave regions)

    :type: float
    """

    mesh_concave_upper: float
    """ Upper mesh concavity bound (high values tend to smoothen and fill out concave regions)

    :type: float
    """

    mesh_generator: str
    """ Which particle level set generator to use

    :type: str
    """

    mesh_particle_radius: float
    """ Particle radius factor (higher value results in larger (meshed) particles). Needs to be adjusted after changing the mesh scale

    :type: float
    """

    mesh_scale: int
    """ The mesh simulation is scaled up by this factor (compared to the base resolution of the domain). For best meshing, it is recommended to adjust the mesh particle radius alongside this value

    :type: int
    """

    mesh_smoothen_neg: int
    """ Negative mesh smoothening

    :type: int
    """

    mesh_smoothen_pos: int
    """ Positive mesh smoothening

    :type: int
    """

    noise_pos_scale: float
    """ Scale of noise (higher value results in larger vortices)

    :type: float
    """

    noise_scale: int
    """ The noise simulation is scaled up by this factor (compared to the base resolution of the domain)

    :type: int
    """

    noise_strength: float
    """ Strength of noise

    :type: float
    """

    noise_time_anim: float
    """ Animation time of noise

    :type: float
    """

    openvdb_cache_compress_type: str
    """ Compression method to be used

    :type: str
    """

    openvdb_data_depth: str
    """ Bit depth for fluid particles and grids (lower bit values reduce file size)

    :type: str
    """

    particle_band_width: float
    """ Particle (narrow) band width (higher value results in thicker band and more particles)

    :type: float
    """

    particle_max: int
    """ Maximum number of particles per cell (ensures that each cell has at most this amount of particles)

    :type: int
    """

    particle_min: int
    """ Minimum number of particles per cell (ensures that each cell has at least this amount of particles)

    :type: int
    """

    particle_number: int
    """ Particle number factor (higher value results in more particles)

    :type: int
    """

    particle_radius: float
    """ Particle radius factor. Increase this value if the simulation appears to leak volume, decrease it if the simulation seems to gain volume

    :type: float
    """

    particle_randomness: float
    """ Randomness factor for particle sampling

    :type: float
    """

    particle_scale: int
    """ The particle simulation is scaled up by this factor (compared to the base resolution of the domain)

    :type: int
    """

    resolution_max: int
    """ Resolution used for the fluid domain. Value corresponds to the longest domain side (resolution for other domain sides is calculated automatically)

    :type: int
    """

    show_gridlines: bool
    """ Show gridlines

    :type: bool
    """

    show_velocity: bool
    """ Visualize vector fields

    :type: bool
    """

    simulation_method: str
    """ Change the underlying simulation method

    :type: str
    """

    slice_axis: str
    """ 

    :type: str
    """

    slice_depth: float
    """ Position of the slice

    :type: float
    """

    slice_per_voxel: float
    """ How many slices per voxel should be generated

    :type: float
    """

    sndparticle_boundary: str
    """ How particles that left the domain are treated

    :type: str
    """

    sndparticle_bubble_buoyancy: float
    """ Amount of buoyancy force that rises bubbles (high value results in bubble movement mainly upwards)

    :type: float
    """

    sndparticle_bubble_drag: float
    """ Amount of drag force that moves bubbles along with the fluid (high value results in bubble movement mainly along with the fluid)

    :type: float
    """

    sndparticle_combined_export: str
    """ Determines which particle systems are created from secondary particles

    :type: str
    """

    sndparticle_life_max: float
    """ Highest possible particle lifetime

    :type: float
    """

    sndparticle_life_min: float
    """ Lowest possible particle lifetime

    :type: float
    """

    sndparticle_potential_max_energy: float
    """ Upper clamping threshold that indicates the fluid speed where cells no longer emit more particles (higher value results in generally less particles)

    :type: float
    """

    sndparticle_potential_max_trappedair: float
    """ Upper clamping threshold for marking fluid cells where air is trapped (higher value results in less marked cells)

    :type: float
    """

    sndparticle_potential_max_wavecrest: float
    """ Upper clamping threshold for marking fluid cells as wave crests (higher value results in less marked cells)

    :type: float
    """

    sndparticle_potential_min_energy: float
    """ Lower clamping threshold that indicates the fluid speed where cells start to emit particles (lower values result in generally more particles)

    :type: float
    """

    sndparticle_potential_min_trappedair: float
    """ Lower clamping threshold for marking fluid cells where air is trapped (lower value results in more marked cells)

    :type: float
    """

    sndparticle_potential_min_wavecrest: float
    """ Lower clamping threshold for marking fluid cells as wave crests (lower value results in more marked cells)

    :type: float
    """

    sndparticle_potential_radius: int
    """ Radius to compute potential for each cell (higher values are slower but create smoother potential grids)

    :type: int
    """

    sndparticle_sampling_trappedair: int
    """ Maximum number of particles generated per trapped air cell per frame

    :type: int
    """

    sndparticle_sampling_wavecrest: int
    """ Maximum number of particles generated per wave crest cell per frame

    :type: int
    """

    sndparticle_update_radius: int
    """ Radius to compute position update for each particle (higher values are slower but particles move less chaotic)

    :type: int
    """

    start_point: bpy_prop_array[float]
    """ Start point

    :type: bpy_prop_array[float]
    """

    surface_tension: float
    """ Surface tension of liquid (higher value results in greater hydrophobic behavior)

    :type: float
    """

    sys_particle_maximum: int
    """ Maximum number of fluid particles that are allowed in this simulation

    :type: int
    """

    temperature_grid: bpy_prop_array[float]
    """ Smoke temperature grid, range 0 to 1 represents 0 to 1000K

    :type: bpy_prop_array[float]
    """

    time_scale: float
    """ Adjust simulation speed

    :type: float
    """

    timesteps_max: int
    """ Maximum number of simulation steps to perform for one frame

    :type: int
    """

    timesteps_min: int
    """ Minimum number of simulation steps to perform for one frame

    :type: int
    """

    use_adaptive_domain: bool
    """ Adapt simulation resolution and size to fluid

    :type: bool
    """

    use_adaptive_timesteps: bool
    """ 

    :type: bool
    """

    use_bubble_particles: bool
    """ Create bubble particle system

    :type: bool
    """

    use_collision_border_back: bool
    """ Enable collisions with back domain border

    :type: bool
    """

    use_collision_border_bottom: bool
    """ Enable collisions with bottom domain border

    :type: bool
    """

    use_collision_border_front: bool
    """ Enable collisions with front domain border

    :type: bool
    """

    use_collision_border_left: bool
    """ Enable collisions with left domain border

    :type: bool
    """

    use_collision_border_right: bool
    """ Enable collisions with right domain border

    :type: bool
    """

    use_collision_border_top: bool
    """ Enable collisions with top domain border

    :type: bool
    """

    use_color_ramp: bool
    """ Render a simulation field while mapping its voxels values to the colors of a ramp or using a predefined color code

    :type: bool
    """

    use_diffusion: bool
    """ Enable fluid diffusion settings (e.g. viscosity, surface tension)

    :type: bool
    """

    use_dissolve_smoke: bool
    """ Let smoke disappear over time

    :type: bool
    """

    use_dissolve_smoke_log: bool
    """ Dissolve smoke in a logarithmic fashion. Dissolves quickly at first, but lingers longer

    :type: bool
    """

    use_flip_particles: bool
    """ Create liquid particle system

    :type: bool
    """

    use_foam_particles: bool
    """ Create foam particle system

    :type: bool
    """

    use_fractions: bool
    """ Fractional obstacles improve and smoothen the fluid-obstacle boundary

    :type: bool
    """

    use_guide: bool
    """ Enable fluid guiding

    :type: bool
    """

    use_mesh: bool
    """ Enable fluid mesh (using amplification)

    :type: bool
    """

    use_noise: bool
    """ Enable fluid noise (using amplification)

    :type: bool
    """

    use_slice: bool
    """ Perform a single slice of the domain object

    :type: bool
    """

    use_speed_vectors: bool
    """ Caches velocities of mesh vertices. These will be used (automatically) when rendering with motion blur enabled

    :type: bool
    """

    use_spray_particles: bool
    """ Create spray particle system

    :type: bool
    """

    use_tracer_particles: bool
    """ Create tracer particle system

    :type: bool
    """

    use_viscosity: bool
    """ Enable fluid viscosity settings

    :type: bool
    """

    vector_display_type: str
    """ 

    :type: str
    """

    vector_field: str
    """ Vector field to be represented by the display vectors

    :type: str
    """

    vector_scale: float
    """ Multiplier for scaling the vectors

    :type: float
    """

    vector_scale_with_magnitude: bool
    """ Scale vectors with their magnitudes

    :type: bool
    """

    vector_show_mac_x: bool
    """ Show X-component of MAC Grid

    :type: bool
    """

    vector_show_mac_y: bool
    """ Show Y-component of MAC Grid

    :type: bool
    """

    vector_show_mac_z: bool
    """ Show Z-component of MAC Grid

    :type: bool
    """

    velocity_grid: bpy_prop_array[float]
    """ Smoke velocity grid

    :type: bpy_prop_array[float]
    """

    velocity_scale: float
    """ Factor to control the amount of motion blur

    :type: float
    """

    viscosity_base: float
    """ Viscosity setting: value that is multiplied by 10 to the power of (exponent*-1)

    :type: float
    """

    viscosity_exponent: int
    """ Negative exponent for the viscosity value (to simplify entering small values e.g. 5*10^-6)

    :type: int
    """

    viscosity_value: float
    """ Viscosity of liquid (higher values result in more viscous fluids, a value of 0 will still apply some viscosity)

    :type: float
    """

    vorticity: float
    """ Amount of turbulence and rotation in smoke

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FluidEffectorSettings(bpy_struct):
    """Smoke collision settings"""

    effector_type: str
    """ Change type of effector in the simulation

    :type: str
    """

    guide_mode: str
    """ How to create guiding velocities

    :type: str
    """

    subframes: int
    """ Number of additional samples to take between frames to improve quality of fast moving effector objects

    :type: int
    """

    surface_distance: float
    """ Additional distance around mesh surface to consider as effector

    :type: float
    """

    use_effector: bool
    """ Control when to apply the effector

    :type: bool
    """

    use_plane_init: bool
    """ Treat this object as a planar, unclosed mesh

    :type: bool
    """

    velocity_factor: float
    """ Multiplier of obstacle velocity

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FluidFlowSettings(bpy_struct):
    """Fluid flow settings"""

    density: float
    """ 

    :type: float
    """

    density_vertex_group: str
    """ Name of vertex group which determines surface emission rate

    :type: str
    """

    flow_behavior: str
    """ Change flow behavior in the simulation

    :type: str
    """

    flow_source: str
    """ Change how fluid is emitted

    :type: str
    """

    flow_type: str
    """ Change type of fluid in the simulation

    :type: str
    """

    fuel_amount: float
    """ 

    :type: float
    """

    noise_texture: Texture
    """ Texture that controls emission strength

    :type: Texture
    """

    particle_size: float
    """ Particle size in simulation cells

    :type: float
    """

    particle_system: ParticleSystem
    """ Particle systems emitted from the object

    :type: ParticleSystem
    """

    smoke_color: bpy_prop_array[float]
    """ Color of smoke

    :type: bpy_prop_array[float]
    """

    subframes: int
    """ Number of additional samples to take between frames to improve quality of fast moving flows

    :type: int
    """

    surface_distance: float
    """ Controls fluid emission from the mesh surface (higher value results in emission further away from the mesh surface

    :type: float
    """

    temperature: float
    """ Temperature difference to ambient temperature

    :type: float
    """

    texture_map_type: str
    """ Texture mapping type

    :type: str
    """

    texture_offset: float
    """ Z-offset of texture mapping

    :type: float
    """

    texture_size: float
    """ Size of texture mapping

    :type: float
    """

    use_absolute: bool
    """ Only allow given density value in emitter area and will not add up

    :type: bool
    """

    use_inflow: bool
    """ Control when to apply fluid flow

    :type: bool
    """

    use_initial_velocity: bool
    """ Fluid has some initial velocity when it is emitted

    :type: bool
    """

    use_particle_size: bool
    """ Set particle size in simulation cells or use nearest cell

    :type: bool
    """

    use_plane_init: bool
    """ Treat this object as a planar and unclosed mesh. Fluid will only be emitted from the mesh surface and based on the surface emission value

    :type: bool
    """

    use_texture: bool
    """ Use a texture to control emission strength

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    velocity_coord: bpy_prop_array[float]
    """ Additional initial velocity in X, Y and Z direction (added to source velocity)

    :type: bpy_prop_array[float]
    """

    velocity_factor: float
    """ Multiplier of source velocity passed to fluid (source velocity is non-zero only if object is moving)

    :type: float
    """

    velocity_normal: float
    """ Amount of normal directional velocity

    :type: float
    """

    velocity_random: float
    """ Amount of random velocity

    :type: float
    """

    volume_density: float
    """ Controls fluid emission from within the mesh (higher value results in greater emissions from inside the mesh)

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FreestyleLineSet(bpy_struct):
    """Line set for associating lines and style parameters"""

    collection: Collection
    """ A collection of objects based on which feature edges are selected

    :type: Collection
    """

    collection_negation: str
    """ Specify either inclusion or exclusion of feature edges belonging to a collection of objects

    :type: str
    """

    edge_type_combination: str
    """ Specify a logical combination of selection conditions on feature edge types

    :type: str
    """

    edge_type_negation: str
    """ Specify either inclusion or exclusion of feature edges selected by edge types

    :type: str
    """

    exclude_border: bool
    """ Exclude border edges

    :type: bool
    """

    exclude_contour: bool
    """ Exclude contours

    :type: bool
    """

    exclude_crease: bool
    """ Exclude crease edges

    :type: bool
    """

    exclude_edge_mark: bool
    """ Exclude edge marks

    :type: bool
    """

    exclude_external_contour: bool
    """ Exclude external contours

    :type: bool
    """

    exclude_material_boundary: bool
    """ Exclude edges at material boundaries

    :type: bool
    """

    exclude_ridge_valley: bool
    """ Exclude ridges and valleys

    :type: bool
    """

    exclude_silhouette: bool
    """ Exclude silhouette edges

    :type: bool
    """

    exclude_suggestive_contour: bool
    """ Exclude suggestive contours

    :type: bool
    """

    face_mark_condition: str
    """ Specify a feature edge selection condition based on face marks

    :type: str
    """

    face_mark_negation: str
    """ Specify either inclusion or exclusion of feature edges selected by face marks

    :type: str
    """

    linestyle: FreestyleLineStyle
    """ Line style settings

    :type: FreestyleLineStyle
    """

    name: str
    """ Line set name

    :type: str
    """

    qi_end: int
    """ Last QI value of the QI range

    :type: int
    """

    qi_start: int
    """ First QI value of the QI range

    :type: int
    """

    select_border: bool
    """ Select border edges (open mesh edges)

    :type: bool
    """

    select_by_collection: bool
    """ Select feature edges based on a collection of objects

    :type: bool
    """

    select_by_edge_types: bool
    """ Select feature edges based on edge types

    :type: bool
    """

    select_by_face_marks: bool
    """ Select feature edges by face marks

    :type: bool
    """

    select_by_image_border: bool
    """ Select feature edges by image border (less memory consumption)

    :type: bool
    """

    select_by_visibility: bool
    """ Select feature edges based on visibility

    :type: bool
    """

    select_contour: bool
    """ Select contours (outer silhouettes of each object)

    :type: bool
    """

    select_crease: bool
    """ Select crease edges (those between two faces making an angle smaller than the Crease Angle)

    :type: bool
    """

    select_edge_mark: bool
    """ Select edge marks (edges annotated by Freestyle edge marks)

    :type: bool
    """

    select_external_contour: bool
    """ Select external contours (outer silhouettes of occluding and occluded objects)

    :type: bool
    """

    select_material_boundary: bool
    """ Select edges at material boundaries

    :type: bool
    """

    select_ridge_valley: bool
    """ Select ridges and valleys (boundary lines between convex and concave areas of surface)

    :type: bool
    """

    select_silhouette: bool
    """ Select silhouettes (edges at the boundary of visible and hidden faces)

    :type: bool
    """

    select_suggestive_contour: bool
    """ Select suggestive contours (almost silhouette/contour edges)

    :type: bool
    """

    show_render: bool
    """ Enable or disable this line set during stroke rendering

    :type: bool
    """

    visibility: str
    """ Determine how to use visibility for feature edge selection

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FreestyleModuleSettings(bpy_struct):
    """Style module configuration for specifying a style module"""

    script: Text
    """ Python script to define a style module

    :type: Text
    """

    use: bool
    """ Enable or disable this style module during stroke rendering

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FreestyleModules(bpy_prop_collection[FreestyleModuleSettings], bpy_struct):
    """A list of style modules (to be applied from top to bottom)"""

    def new(self) -> FreestyleModuleSettings:
        """Add a style module to scene render layer Freestyle settings

        :return: Newly created style module
        :rtype: FreestyleModuleSettings
        """
        ...

    def remove(self, module: FreestyleModuleSettings):
        """Remove a style module from scene render layer Freestyle settings

        :param module: Style module to remove
        :type module: FreestyleModuleSettings
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class FreestyleSettings(bpy_struct):
    """Freestyle settings for a ViewLayer data-block"""

    as_render_pass: bool
    """ Renders Freestyle output to a separate pass instead of overlaying it on the Combined pass

    :type: bool
    """

    crease_angle: float
    """ Angular threshold for detecting crease edges

    :type: float
    """

    kr_derivative_epsilon: float
    """ Kr derivative epsilon for computing suggestive contours

    :type: float
    """

    linesets: Linesets
    """ 

    :type: Linesets
    """

    mode: str
    """ Select the Freestyle control mode

    :type: str
    """

    modules: FreestyleModules
    """ A list of style modules (to be applied from top to bottom)

    :type: FreestyleModules
    """

    sphere_radius: float
    """ Sphere radius for computing curvatures

    :type: float
    """

    use_culling: bool
    """ If enabled, out-of-view edges are ignored

    :type: bool
    """

    use_material_boundaries: bool
    """ Enable material boundaries

    :type: bool
    """

    use_ridges_and_valleys: bool
    """ Enable ridges and valleys

    :type: bool
    """

    use_smoothness: bool
    """ Take face smoothness into account in view map calculation

    :type: bool
    """

    use_suggestive_contours: bool
    """ Enable suggestive contours

    :type: bool
    """

    use_view_map_cache: bool
    """ Keep the computed view map and avoid recalculating it if mesh geometry is unchanged

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Function(bpy_struct):
    """RNA function definition"""

    description: str
    """ Description of the Function's purpose

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    is_registered: bool
    """ Function is registered as callback as part of type registration

    :type: bool
    """

    is_registered_optional: bool
    """ Function is optionally registered as callback part of type registration

    :type: bool
    """

    parameters: bpy_prop_collection[Property]
    """ Parameters for the function

    :type: bpy_prop_collection[Property]
    """

    use_self: bool
    """ Function does not pass itself as an argument (becomes a static method in python)

    :type: bool
    """

    use_self_type: bool
    """ Function passes itself type as an argument (becomes a class method in python if use_self is false)

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilEditCurve(bpy_struct):
    """Edition Curve"""

    curve_points: bpy_prop_collection[GPencilEditCurvePoint]
    """ Curve data points

    :type: bpy_prop_collection[GPencilEditCurvePoint]
    """

    select: bool
    """ Curve is selected for viewport editing

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilEditCurvePoint(bpy_struct):
    """Bezier curve point with two handles"""

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the first handle

    :type: bpy_prop_array[float]
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the second handle

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    point_index: int
    """ Index of the corresponding grease pencil stroke point

    :type: int
    """

    pressure: float
    """ Pressure of the grease pencil stroke point

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Handle 1 selection status

    :type: bool
    """

    select_right_handle: bool
    """ Handle 2 selection status

    :type: bool
    """

    strength: float
    """ Color intensity (alpha factor) of the grease pencil stroke point

    :type: float
    """

    uv_factor: float
    """ Internal UV factor

    :type: float
    """

    uv_rotation: float
    """ Internal UV factor for dot mode

    :type: float
    """

    vertex_color: bpy_prop_array[float]
    """ Vertex color of the grease pencil stroke point

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilFrame(bpy_struct):
    """Collection of related sketches on a particular frame"""

    frame_number: int
    """ The frame on which this sketch appears

    :type: int
    """

    is_edited: bool
    """ Frame is being edited (painted on)

    :type: bool
    """

    keyframe_type: str
    """ Type of keyframe

    :type: str
    """

    select: bool
    """ Frame is selected for editing in the Dope Sheet

    :type: bool
    """

    strokes: GPencilStrokes
    """ Freehand curves defining the sketch on this frame

    :type: GPencilStrokes
    """

    def clear(self):
        """Remove all the grease pencil frame data"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilFrames(bpy_prop_collection[GPencilFrame], bpy_struct):
    """Collection of grease pencil frames"""

    def new(
        self, frame_number: int | None, active: bool | typing.Any | None = False
    ) -> GPencilFrame:
        """Add a new grease pencil frame

        :param frame_number: Frame Number, The frame on which this sketch appears
        :type frame_number: int | None
        :param active: Active
        :type active: bool | typing.Any | None
        :return: The newly created frame
        :rtype: GPencilFrame
        """
        ...

    def remove(self, frame: GPencilFrame):
        """Remove a grease pencil frame

        :param frame: Frame, The frame to remove
        :type frame: GPencilFrame
        """
        ...

    def copy(self, source: GPencilFrame) -> GPencilFrame:
        """Copy a grease pencil frame

        :param source: Source, The source frame
        :type source: GPencilFrame
        :return: The newly copied frame
        :rtype: GPencilFrame
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilInterpolateSettings(bpy_struct):
    """Settings for Grease Pencil interpolation tools"""

    interpolation_curve: CurveMapping
    """ Custom curve to control 'sequence' interpolation between Grease Pencil frames

    :type: CurveMapping
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilLayer(bpy_struct):
    """Collection of related sketches"""

    active_frame: GPencilFrame
    """ Frame currently being displayed for this layer

    :type: GPencilFrame
    """

    annotation_hide: bool
    """ Set annotation Visibility

    :type: bool
    """

    annotation_onion_after_color: bpy_prop_array[float]
    """ Base color for ghosts after the active frame

    :type: bpy_prop_array[float]
    """

    annotation_onion_after_range: int
    """ Maximum number of frames to show after current frame

    :type: int
    """

    annotation_onion_before_color: bpy_prop_array[float]
    """ Base color for ghosts before the active frame

    :type: bpy_prop_array[float]
    """

    annotation_onion_before_range: int
    """ Maximum number of frames to show before current frame

    :type: int
    """

    annotation_opacity: float
    """ Annotation Layer Opacity

    :type: float
    """

    blend_mode: str
    """ Blend mode

    :type: str
    """

    channel_color: bpy_prop_array[float]
    """ Custom color for animation channel in Dopesheet

    :type: bpy_prop_array[float]
    """

    color: bpy_prop_array[float]
    """ Color for all strokes in this layer

    :type: bpy_prop_array[float]
    """

    frames: GPencilFrames
    """ Sketches for this layer on different frames

    :type: GPencilFrames
    """

    hide: bool
    """ Set layer Visibility

    :type: bool
    """

    info: str
    """ Layer name

    :type: str
    """

    is_parented: bool
    """ True when the layer parent object is set

    :type: bool
    """

    is_ruler: bool
    """ This is a special ruler layer

    :type: bool
    """

    line_change: int
    """ Thickness change to apply to current strokes (in pixels)

    :type: int
    """

    location: bpy_prop_array[float]
    """ Values for change location

    :type: bpy_prop_array[float]
    """

    lock: bool
    """ Protect layer from further editing and/or frame changes

    :type: bool
    """

    lock_frame: bool
    """ Lock current frame displayed by layer

    :type: bool
    """

    lock_material: bool
    """ Avoids editing locked materials in the layer

    :type: bool
    """

    mask_layers: GreasePencilMaskLayers
    """ List of Masking Layers

    :type: GreasePencilMaskLayers
    """

    matrix_inverse: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Parent inverse transformation matrix

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    matrix_inverse_layer: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Local Layer transformation inverse matrix

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    matrix_layer: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Local Layer transformation matrix

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    opacity: float
    """ Layer Opacity

    :type: float
    """

    parent: Object
    """ Parent object

    :type: Object
    """

    parent_bone: str
    """ Name of parent bone in case of a bone parenting relation

    :type: str
    """

    parent_type: str
    """ Type of parent relation

    :type: str
    """

    pass_index: int
    """ Index number for the "Layer Index" pass

    :type: int
    """

    rotation: bpy_prop_array[float]
    """ Values for changes in rotation

    :type: bpy_prop_array[float]
    """

    scale: bpy_prop_array[float]
    """ Values for changes in scale

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Layer is selected for editing in the Dope Sheet

    :type: bool
    """

    show_in_front: bool
    """ Make the layer display in front of objects

    :type: bool
    """

    show_points: bool
    """ Show the points which make up the strokes (for debugging purposes)

    :type: bool
    """

    thickness: int
    """ Thickness of annotation strokes

    :type: int
    """

    tint_color: bpy_prop_array[float]
    """ Color for tinting stroke colors

    :type: bpy_prop_array[float]
    """

    tint_factor: float
    """ Factor of tinting color

    :type: float
    """

    use_annotation_onion_skinning: bool
    """ Display annotation onion skins before and after the current frame

    :type: bool
    """

    use_lights: bool
    """ Enable the use of lights on stroke and fill materials

    :type: bool
    """

    use_mask_layer: bool
    """ The visibility of drawings on this layer is affected by the layers in its masks list

    :type: bool
    """

    use_onion_skinning: bool
    """ Display onion skins before and after the current frame

    :type: bool
    """

    use_solo_mode: bool
    """ In Draw Mode only display layers with keyframe in current frame

    :type: bool
    """

    use_viewlayer_masks: bool
    """ Include the mask layers when rendering the view-layer

    :type: bool
    """

    vertex_paint_opacity: float
    """ Vertex Paint mix factor

    :type: float
    """

    viewlayer_render: str
    """ Only include Layer in this View Layer render output (leave blank to include always)

    :type: str
    """

    def clear(self):
        """Remove all the grease pencil layer data"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilLayerMask(bpy_struct):
    """List of Mask Layers"""

    hide: bool
    """ Set mask Visibility

    :type: bool
    """

    invert: bool
    """ Invert mask

    :type: bool
    """

    name: str
    """ Mask layer name

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilSculptGuide(bpy_struct):
    """Guides for drawing"""

    angle: float
    """ Direction of lines

    :type: float
    """

    angle_snap: float
    """ Angle snapping

    :type: float
    """

    location: bpy_prop_array[float]
    """ Custom reference point for guides

    :type: bpy_prop_array[float]
    """

    reference_object: Object
    """ Object used for reference point

    :type: Object
    """

    reference_point: str
    """ Type of speed guide

    :type: str
    """

    spacing: float
    """ Guide spacing

    :type: float
    """

    type: str
    """ Type of speed guide

    :type: str
    """

    use_guide: bool
    """ Enable speed guides

    :type: bool
    """

    use_snapping: bool
    """ Enable snapping to guides angle or spacing options

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilSculptSettings(bpy_struct):
    """General properties for Grease Pencil stroke sculpting tools"""

    guide: GPencilSculptGuide
    """ 

    :type: GPencilSculptGuide
    """

    intersection_threshold: float
    """ Threshold for stroke intersections

    :type: float
    """

    lock_axis: str
    """ 

    :type: str
    """

    multiframe_falloff_curve: CurveMapping
    """ Custom curve to control falloff of brush effect by Grease Pencil frames

    :type: CurveMapping
    """

    thickness_primitive_curve: CurveMapping
    """ Custom curve to control primitive thickness

    :type: CurveMapping
    """

    use_multiframe_falloff: bool
    """ Use falloff effect when edit in multiframe mode to compute brush effect by frame

    :type: bool
    """

    use_scale_thickness: bool
    """ Scale the stroke thickness when transforming strokes

    :type: bool
    """

    use_thickness_curve: bool
    """ Use curve to define primitive stroke thickness

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilStroke(bpy_struct):
    """Freehand curve defining part of a sketch"""

    aspect: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    bound_box_max: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    bound_box_min: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    display_mode: str
    """ Coordinate space that stroke is in

    :type: str
    """

    edit_curve: GPencilEditCurve
    """ Temporary data for Edit Curve

    :type: GPencilEditCurve
    """

    end_cap_mode: str
    """ Stroke end extreme cap style

    :type: str
    """

    hardness: float
    """ Amount of gradient along section of stroke

    :type: float
    """

    has_edit_curve: bool
    """ Stroke has Curve data to edit shape

    :type: bool
    """

    is_nofill_stroke: bool
    """ Special stroke to use as boundary for filling areas

    :type: bool
    """

    line_width: int
    """ Thickness of stroke (in pixels)

    :type: int
    """

    material_index: int
    """ Material slot index of this stroke

    :type: int
    """

    points: GPencilStrokePoints
    """ Stroke data points

    :type: GPencilStrokePoints
    """

    select: bool
    """ Stroke is selected for viewport editing

    :type: bool
    """

    select_index: int
    """ Index of selection used for interpolation

    :type: int
    """

    start_cap_mode: str
    """ Stroke start extreme cap style

    :type: str
    """

    triangles: bpy_prop_collection[GPencilTriangle]
    """ Triangulation data for HQ fill

    :type: bpy_prop_collection[GPencilTriangle]
    """

    use_cyclic: bool
    """ Enable cyclic drawing, closing the stroke

    :type: bool
    """

    uv_rotation: float
    """ Rotation of the UV

    :type: float
    """

    uv_scale: float
    """ Scale of the UV

    :type: float
    """

    uv_translation: bpy_prop_array[float]
    """ Translation of default UV position

    :type: bpy_prop_array[float]
    """

    vertex_color_fill: bpy_prop_array[float]
    """ Color used to mix with fill color to get final color

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilStrokePoint(bpy_struct):
    """Data point for freehand stroke curve"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    pressure: float
    """ Pressure of tablet at point when drawing it

    :type: float
    """

    select: bool
    """ Point is selected for viewport editing

    :type: bool
    """

    strength: float
    """ Color intensity (alpha factor)

    :type: float
    """

    uv_factor: float
    """ Internal UV factor

    :type: float
    """

    uv_fill: bpy_prop_array[float]
    """ Internal UV factor for filling

    :type: bpy_prop_array[float]
    """

    uv_rotation: float
    """ Internal UV factor for dot mode

    :type: float
    """

    vertex_color: bpy_prop_array[float]
    """ Color used to mix with point color to get final color

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilStrokePoints(bpy_prop_collection[GPencilStrokePoint], bpy_struct):
    """Collection of grease pencil stroke points"""

    def add(
        self,
        count: int | None,
        pressure: typing.Any | None = 1.0,
        strength: typing.Any | None = 1.0,
    ):
        """Add a new grease pencil stroke point

        :param count: Number, Number of points to add to the stroke
        :type count: int | None
        :param pressure: Pressure, Pressure for newly created points
        :type pressure: typing.Any | None
        :param strength: Strength, Color intensity (alpha factor) for newly created points
        :type strength: typing.Any | None
        """
        ...

    def pop(self, index: typing.Any | None = -1):
        """Remove a grease pencil stroke point

        :param index: Index, point index
        :type index: typing.Any | None
        """
        ...

    def update(self):
        """Recalculate internal triangulation data"""
        ...

    def weight_get(
        self,
        vertex_group_index: typing.Any | None = 0,
        point_index: typing.Any | None = 0,
    ) -> float:
        """Get vertex group point weight

        :param vertex_group_index: Vertex Group Index, Index of Vertex Group in the array of groups
        :type vertex_group_index: typing.Any | None
        :param point_index: Point Index, Index of the Point in the array
        :type point_index: typing.Any | None
        :return: Weight, Point Weight
        :rtype: float
        """
        ...

    def weight_set(
        self,
        vertex_group_index: typing.Any | None = 0,
        point_index: typing.Any | None = 0,
        weight: typing.Any | None = 0.0,
    ):
        """Set vertex group point weight

        :param vertex_group_index: Vertex Group Index, Index of Vertex Group in the array of groups
        :type vertex_group_index: typing.Any | None
        :param point_index: Point Index, Index of the Point in the array
        :type point_index: typing.Any | None
        :param weight: Weight, Point Weight
        :type weight: typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilStrokes(bpy_prop_collection[GPencilStroke], bpy_struct):
    """Collection of grease pencil stroke"""

    def new(self) -> GPencilStroke:
        """Add a new grease pencil stroke

        :return: The newly created stroke
        :rtype: GPencilStroke
        """
        ...

    def remove(self, stroke: GPencilStroke):
        """Remove a grease pencil stroke

        :param stroke: Stroke, The stroke to remove
        :type stroke: GPencilStroke
        """
        ...

    def close(self, stroke: GPencilStroke):
        """Close a grease pencil stroke adding geometry

        :param stroke: Stroke, The stroke to close
        :type stroke: GPencilStroke
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GPencilTriangle(bpy_struct):
    """Triangulation data for Grease Pencil fills"""

    v1: int
    """ First triangle vertex index

    :type: int
    """

    v2: int
    """ Second triangle vertex index

    :type: int
    """

    v3: int
    """ Third triangle vertex index

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Gizmo(bpy_struct):
    """Collection of gizmos"""

    alpha: float
    """ 

    :type: float
    """

    alpha_highlight: float
    """ 

    :type: float
    """

    bl_idname: str
    """ 

    :type: str
    """

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color_highlight: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    group: GizmoGroup
    """ Gizmo group this gizmo is a member of

    :type: GizmoGroup
    """

    hide: bool
    """ 

    :type: bool
    """

    hide_keymap: bool
    """ Ignore the key-map for this gizmo

    :type: bool
    """

    hide_select: bool
    """ 

    :type: bool
    """

    is_highlight: bool
    """ 

    :type: bool
    """

    is_modal: bool
    """ 

    :type: bool
    """

    line_width: float
    """ 

    :type: float
    """

    matrix_basis: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ 

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    matrix_offset: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ 

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    matrix_space: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ 

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    matrix_world: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ 

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    properties: GizmoProperties
    """ 

    :type: GizmoProperties
    """

    scale_basis: float
    """ 

    :type: float
    """

    select: bool
    """ 

    :type: bool
    """

    select_bias: float
    """ Depth bias used for selection

    :type: float
    """

    use_draw_hover: bool
    """ 

    :type: bool
    """

    use_draw_modal: bool
    """ Show while dragging

    :type: bool
    """

    use_draw_offset_scale: bool
    """ Scale the offset matrix (use to apply screen-space offset)

    :type: bool
    """

    use_draw_scale: bool
    """ Use scale when calculating the matrix

    :type: bool
    """

    use_draw_value: bool
    """ Show an indicator for the current value while dragging

    :type: bool
    """

    use_event_handle_all: bool
    """ When highlighted, do not pass events through to be handled by other keymaps

    :type: bool
    """

    use_grab_cursor: bool
    """ 

    :type: bool
    """

    use_operator_tool_properties: bool
    """ Merge active tool properties on activation (does not overwrite existing)

    :type: bool
    """

    use_select_background: bool
    """ Don't write into the depth buffer

    :type: bool
    """

    use_tooltip: bool
    """ Use tooltips when hovering over this gizmo

    :type: bool
    """

    def draw(self, context: Context):
        """

        :param context:
        :type context: Context
        """
        ...

    def draw_select(self, context: Context, select_id: typing.Any | None = 0):
        """

        :param context:
        :type context: Context
        :param select_id:
        :type select_id: typing.Any | None
        """
        ...

    def test_select(self, context: Context, location: typing.Any) -> int:
        """

        :param context:
        :type context: Context
        :param location: Location, Region coordinates
        :type location: typing.Any
        :return: Use -1 to skip this gizmo
        :rtype: int
        """
        ...

    def modal(self, context: Context, event: Event, tweak: set[str] | None) -> set[str]:
        """

                :param context:
                :type context: Context
                :param event:
                :type event: Event
                :param tweak: Tweak
                :type tweak: set[str] | None
                :return: result

        RUNNING_MODAL
        Running Modal -- Keep the operator running with blender.

        CANCELLED
        Cancelled -- The operator exited without doing anything, so no undo entry should be pushed.

        FINISHED
        Finished -- The operator exited after completing its action.

        PASS_THROUGH
        Pass Through -- Do nothing and pass the event on.

        INTERFACE
        Interface -- Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def setup(self): ...
    def invoke(self, context: Context, event: Event) -> set[str]:
        """

                :param context:
                :type context: Context
                :param event:
                :type event: Event
                :return: result

        RUNNING_MODAL
        Running Modal -- Keep the operator running with blender.

        CANCELLED
        Cancelled -- The operator exited without doing anything, so no undo entry should be pushed.

        FINISHED
        Finished -- The operator exited after completing its action.

        PASS_THROUGH
        Pass Through -- Do nothing and pass the event on.

        INTERFACE
        Interface -- Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def exit(self, context: Context, cancel: bool | None):
        """

        :param context:
        :type context: Context
        :param cancel: Cancel, otherwise confirm
        :type cancel: bool | None
        """
        ...

    def select_refresh(self): ...
    def draw_preset_box(
        self,
        matrix: list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
        | None,
        select_id: typing.Any | None = -1,
    ):
        """Draw a box

        :param matrix: The matrix to transform
        :type matrix: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]] | None
        :param select_id: Zero when not selecting
        :type select_id: typing.Any | None
        """
        ...

    def draw_preset_arrow(
        self,
        matrix: list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
        | None,
        axis: str | None = "POS_Z",
        select_id: typing.Any | None = -1,
    ):
        """Draw a box

        :param matrix: The matrix to transform
        :type matrix: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]] | None
        :param axis: Arrow Orientation
        :type axis: str | None
        :param select_id: Zero when not selecting
        :type select_id: typing.Any | None
        """
        ...

    def draw_preset_circle(
        self,
        matrix: list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
        | None,
        axis: str | None = "POS_Z",
        select_id: typing.Any | None = -1,
    ):
        """Draw a box

        :param matrix: The matrix to transform
        :type matrix: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]] | None
        :param axis: Arrow Orientation
        :type axis: str | None
        :param select_id: Zero when not selecting
        :type select_id: typing.Any | None
        """
        ...

    def draw_preset_facemap(
        self, object: Object, face_map: int | None, select_id: typing.Any | None = -1
    ):
        """Draw the face-map of a mesh object

        :param object: Object
        :type object: Object
        :param face_map: Face map index
        :type face_map: int | None
        :param select_id: Zero when not selecting
        :type select_id: typing.Any | None
        """
        ...

    def target_set_prop(
        self,
        target: str | typing.Any,
        data: typing.Any,
        property: str | typing.Any,
        index: typing.Any | None = -1,
    ):
        """

        :param target: Target property
        :type target: str | typing.Any
        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param index:
        :type index: typing.Any | None
        """
        ...

    def target_set_operator(
        self, operator: str | typing.Any, index: typing.Any | None = 0
    ) -> OperatorProperties:
        """Operator to run when activating the gizmo (overrides property targets)

        :param operator: Target operator
        :type operator: str | typing.Any
        :param index: Part index
        :type index: typing.Any | None
        :return: Operator properties to fill in
        :rtype: OperatorProperties
        """
        ...

    def target_is_valid(self, property: str | typing.Any) -> bool:
        """

        :param property: Property identifier
        :type property: str | typing.Any
        :return:
        :rtype: bool
        """
        ...

    def draw_custom_shape(
        self,
        shape,
        *,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None = None,
        select_id=None,
    ):
        """Draw a shape created form `bpy.types.Gizmo.draw_custom_shape`.

                :param shape: The cached shape to draw.
                :param matrix: 4x4 matrix, when not given
        `bpy.types.Gizmo.matrix_world` is used.
                :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param select_id: The selection id.
        Only use when drawing within `bpy.types.Gizmo.draw_select`.
        """
        ...

    @staticmethod
    def new_custom_shape(type: str | None, verts):
        """Create a new shape that can be passed to `bpy.types.Gizmo.draw_custom_shape`.

        :param type: The type of shape to create in (POINTS, LINES, TRIS, LINE_STRIP).
        :type type: str | None
        :param verts: Coordinates.
        :return: The newly created shape.
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

    def target_get_range(self, target):
        """Get the range for this target property.

        :param target: Target property name.
        :return: The range of this property (min, max).
        """
        ...

    def target_get_value(self, target: str | None):
        """Get the value of this target property.

        :param target: Target property name.
        :type target: str | None
        :return: The value of the target property.
        """
        ...

    def target_set_handler(
        self,
        target,
        get: collections.abc.Callable | None,
        set: collections.abc.Callable | None,
        range: collections.abc.Callable | None = None,
    ):
        """Assigns callbacks to a gizmos property.

        :param target:
        :param get: Function that returns the value for this property (single value or sequence).
        :type get: collections.abc.Callable | None
        :param set: Function that takes a single value argument and applies it.
        :type set: collections.abc.Callable | None
        :param range: Function that returns a (min, max) tuple for gizmos that use a range.
        :type range: collections.abc.Callable | None
        """
        ...

    def target_set_value(self, target: str | None):
        """Set the value of this target property.

        :param target: Target property name.
        :type target: str | None
        """
        ...

class GizmoGroup(bpy_struct):
    """Storage of an operator being executed, or registered after execution"""

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[str]
    """ Options for this operator type

    :type: set[str]
    """

    bl_owner_id: str
    """ 

    :type: str
    """

    bl_region_type: str
    """ The region where the panel is going to be used in

    :type: str
    """

    bl_space_type: str
    """ The space where the panel is going to be used in

    :type: str
    """

    gizmos: Gizmos
    """ List of gizmos in the Gizmo Map

    :type: Gizmos
    """

    has_reports: bool
    """ GizmoGroup has a set of reports (warnings and errors) from last execution

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Test if the gizmo group can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    @classmethod
    def setup_keymap(cls, keyconfig: KeyConfig) -> KeyMap:
        """Initialize keymaps for this gizmo group, use fallback keymap when not present

        :param keyconfig:
        :type keyconfig: KeyConfig
        :return:
        :rtype: KeyMap
        """
        ...

    def setup(self, context: Context):
        """Create gizmos function for the gizmo group

        :param context:
        :type context: Context
        """
        ...

    def refresh(self, context: Context):
        """Refresh data (called on common state changes such as selection)

        :param context:
        :type context: Context
        """
        ...

    def draw_prepare(self, context: Context):
        """Run before each redraw

        :param context:
        :type context: Context
        """
        ...

    def invoke_prepare(self, context: Context, gizmo: Gizmo):
        """Run before invoke

        :param context:
        :type context: Context
        :param gizmo:
        :type gizmo: Gizmo
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GizmoGroupProperties(bpy_struct):
    """Input properties of a Gizmo Group"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GizmoProperties(bpy_struct):
    """Input properties of an Gizmo"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Gizmos(bpy_prop_collection[Gizmo], bpy_struct):
    """Collection of gizmos"""

    def new(self, type: str | typing.Any) -> Gizmo:
        """Add gizmo

        :param type: Gizmo identifier
        :type type: str | typing.Any
        :return: New gizmo
        :rtype: Gizmo
        """
        ...

    def remove(self, gizmo: Gizmo):
        """Delete gizmo

        :param gizmo: New gizmo
        :type gizmo: Gizmo
        """
        ...

    def clear(self):
        """Delete all gizmos"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GpencilModifier(bpy_struct):
    """Modifier affecting the Grease Pencil object"""

    is_override_data: bool
    """ In a local override object, whether this modifier comes from the linked reference object, or is local to the override

    :type: bool
    """

    name: str
    """ Modifier name

    :type: str
    """

    show_expanded: bool
    """ Set modifier expanded in the user interface

    :type: bool
    """

    show_in_editmode: bool
    """ Display modifier in Edit mode

    :type: bool
    """

    show_render: bool
    """ Use modifier during render

    :type: bool
    """

    show_viewport: bool
    """ Display modifier in viewport

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GpencilVertexGroupElement(bpy_struct):
    """Weight value of a vertex in a vertex group"""

    group: int
    """ 

    :type: int
    """

    weight: float
    """ Vertex Weight

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GreasePencilGrid(bpy_struct):
    """Settings for grid and canvas in 3D viewport"""

    color: bpy_prop_array[float]
    """ Color for grid lines

    :type: bpy_prop_array[float]
    """

    lines: int
    """ Number of subdivisions in each side of symmetry line

    :type: int
    """

    offset: bpy_prop_array[float]
    """ Offset of the canvas

    :type: bpy_prop_array[float]
    """

    scale: bpy_prop_array[float]
    """ Grid scale

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GreasePencilLayers(bpy_prop_collection[GPencilLayer], bpy_struct):
    """Collection of grease pencil layers"""

    active: GPencilLayer | None
    """ Active grease pencil layer

    :type: GPencilLayer | None
    """

    active_index: int | None
    """ Index of active grease pencil layer

    :type: int | None
    """

    active_note: str | None
    """ Note/Layer to add annotation strokes to

    :type: str | None
    """

    def new(
        self, name: str | typing.Any, set_active: bool | typing.Any | None = True
    ) -> GPencilLayer:
        """Add a new grease pencil layer

        :param name: Name, Name of the layer
        :type name: str | typing.Any
        :param set_active: Set Active, Set the newly created layer to the active layer
        :type set_active: bool | typing.Any | None
        :return: The newly created layer
        :rtype: GPencilLayer
        """
        ...

    def remove(self, layer: GPencilLayer):
        """Remove a grease pencil layer

        :param layer: The layer to remove
        :type layer: GPencilLayer
        """
        ...

    def move(self, layer: GPencilLayer, type: str | None):
        """Move a grease pencil layer in the layer stack

        :param layer: The layer to move
        :type layer: GPencilLayer
        :param type: Direction of movement
        :type type: str | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class GreasePencilMaskLayers(bpy_prop_collection[GPencilLayerMask], bpy_struct):
    """Collection of grease pencil masking layers"""

    active_mask_index: int | None
    """ Active index in layer mask array

    :type: int | None
    """

    def add(self, layer: GPencilLayer):
        """Add a layer to mask list

        :param layer: Layer to add as mask
        :type layer: GPencilLayer
        """
        ...

    def remove(self, mask: GPencilLayerMask):
        """Remove a layer from mask list

        :param mask: Mask to remove
        :type mask: GPencilLayerMask
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Header(bpy_struct):
    """Editor header containing UI elements"""

    bl_idname: str
    """ If this is set, the header gets a custom ID, otherwise it takes the name of the class used to define the panel; for example, if the class name is "OBJECT_HT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_HT_hello"

    :type: str
    """

    bl_region_type: str
    """ The region where the header is going to be used in (defaults to header region)

    :type: str
    """

    bl_space_type: str
    """ The space where the header is going to be used in

    :type: str
    """

    layout: UILayout
    """ Structure of the header in the UI

    :type: UILayout
    """

    def draw(self, context: Context | None):
        """Draw UI elements into the header UI layout

        :param context:
        :type context: Context | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Histogram(bpy_struct):
    """Statistical view of the levels of color in an image"""

    mode: str
    """ Channels to display in the histogram

    :type: str
    """

    show_line: bool
    """ Display lines rather than filled shapes

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ID(bpy_struct):
    """Base type for data-blocks, defining a unique name, linking from other libraries and garbage collection"""

    asset_data: AssetMetaData
    """ Additional data for an asset data-block

    :type: AssetMetaData
    """

    is_embedded_data: bool
    """ This data-block is not an independent one, but is actually a sub-data of another ID (typical example: root node trees or master collections)

    :type: bool
    """

    is_evaluated: bool
    """ Whether this ID is runtime-only, evaluated data-block, or actual data from .blend file

    :type: bool
    """

    is_library_indirect: bool
    """ Is this ID block linked indirectly

    :type: bool
    """

    library: Library
    """ Library file the data-block is linked from

    :type: Library
    """

    library_weak_reference: LibraryWeakReference
    """ Weak reference to a data-block in another library .blend file (used to re-use already appended data instead of appending new copies)

    :type: LibraryWeakReference
    """

    name: str
    """ Unique data-block ID name

    :type: str
    """

    name_full: str
    """ Unique data-block ID name, including library one is any

    :type: str
    """

    original: ID
    """ Actual data-block from .blend file (Main database) that generated that evaluated one

    :type: ID
    """

    override_library: IDOverrideLibrary
    """ Library override data

    :type: IDOverrideLibrary
    """

    preview: ImagePreview
    """ Preview image and icon of this data-block (always None if not supported for this type of data)

    :type: ImagePreview
    """

    tag: bool
    """ Tools can use this to tag data for their own purposes (initial state is undefined)

    :type: bool
    """

    use_extra_user: bool
    """ Indicates whether an extra user is set or not (mainly for internal/debug usages)

    :type: bool
    """

    use_fake_user: bool
    """ Save this data-block even if it has no users

    :type: bool
    """

    users: int
    """ Number of times this data-block is referenced

    :type: int
    """

    def evaluated_get(self, depsgraph: Depsgraph) -> ID:
        """Get corresponding evaluated ID from the given dependency graph

        :param depsgraph: Dependency graph to perform lookup in
        :type depsgraph: Depsgraph
        :return: New copy of the ID
        :rtype: ID
        """
        ...

    def copy(self) -> ID:
        """Create a copy of this data-block (not supported for all data-blocks)

        :return: New copy of the ID
        :rtype: ID
        """
        ...

    def asset_mark(self):
        """Enable easier reuse of the data-block through the Asset Browser, with the help of customizable metadata (like previews, descriptions and tags)"""
        ...

    def asset_clear(self):
        """Delete all asset metadata and turn the asset data-block back into a normal data-block"""
        ...

    def asset_generate_preview(self):
        """Generate preview image (might be scheduled in a background thread)"""
        ...

    def override_create(
        self, remap_local_usages: bool | typing.Any | None = False
    ) -> ID:
        """Create an overridden local copy of this linked data-block (not supported for all data-blocks)

        :param remap_local_usages: Whether local usages of the linked ID should be remapped to the new library override of it
        :type remap_local_usages: bool | typing.Any | None
        :return: New overridden local copy of the ID
        :rtype: ID
        """
        ...

    def override_hierarchy_create(
        self, scene: Scene, view_layer: ViewLayer, reference: ID | None = None
    ) -> ID:
        """Create an overridden local copy of this linked data-block, and most of its dependencies when it is a Collection or and Object

        :param scene: In which scene the new overrides should be instantiated
        :type scene: Scene
        :param view_layer: In which view layer the new overrides should be instantiated
        :type view_layer: ViewLayer
        :param reference: Another ID (usually an Object or Collection) used as a hint to decide where to instantiate the new overrides
        :type reference: ID | None
        :return: New overridden local copy of the root ID
        :rtype: ID
        """
        ...

    def override_template_create(self):
        """Create an override template for this ID"""
        ...

    def user_clear(self):
        """Clear the user count of a data-block so its not saved, on reload the data will be removedThis function is for advanced use only, misuse can crash blender since the user
        count is used to prevent data being removed when it is used.

        """
        ...

    def user_remap(self, new_id: ID):
        """Replace all usage in the .blend file of this ID by new given one

        :param new_id: New ID to use
        :type new_id: ID
        """
        ...

    def make_local(self, clear_proxy: bool | typing.Any | None = True) -> ID:
        """Make this datablock local, return local one (may be a copy of the original, in case it is also indirectly used)

        :param clear_proxy: Deprecated, has no effect
        :type clear_proxy: bool | typing.Any | None
        :return: This ID, or the new ID if it was copied
        :rtype: ID
        """
        ...

    def user_of_id(self, id: ID) -> int:
        """Count the number of times that ID uses/references given one

        :param id: ID to count usages
        :type id: ID
        :return: Number of usages/references of given id by current data-block
        :rtype: int
        """
        ...

    def animation_data_create(self) -> AnimData:
        """Create animation data to this ID, note that not all ID types support this

        :return: New animation data or NULL
        :rtype: AnimData
        """
        ...

    def animation_data_clear(self):
        """Clear animation on this this ID"""
        ...

    def update_tag(self, refresh: set[str] | None = {}):
        """Tag the ID to update its display data, e.g. when calling `bpy.types.Scene.update`

        :param refresh: Type of updates to perform
        :type refresh: set[str] | None
        """
        ...

    def preview_ensure(self) -> ImagePreview:
        """Ensure that this ID has preview data (if ID type supports it)

        :return: The existing or created preview
        :rtype: ImagePreview
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IDMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    def append(self, material: Material | None):
        """Add a new material to the data-block

        :param material: Material to add
        :type material: Material | None
        """
        ...

    def pop(self, index: typing.Any | None = -1) -> Material:
        """Remove a material from the data-block

        :param index: Index of material to remove
        :type index: typing.Any | None
        :return: Material to remove
        :rtype: Material
        """
        ...

    def clear(self):
        """Remove all materials from the data-block"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IDOverrideLibrary(bpy_struct):
    """Struct gathering all data needed by overridden linked IDs"""

    hierarchy_root: ID
    """ Library override ID used as root of the override hierarchy this ID is a member of

    :type: ID
    """

    is_in_hierarchy: bool
    """ Whether this library override is defined as part of a library hierarchy, or as a single, isolated and autonomous override

    :type: bool
    """

    is_system_override: bool
    """ Whether this library override exists only for the override hierarchy, or if it is actually editable by the user

    :type: bool
    """

    properties: IDOverrideLibraryProperties
    """ List of overridden properties

    :type: IDOverrideLibraryProperties
    """

    reference: ID
    """ Linked ID used as reference by this override

    :type: ID
    """

    def operations_update(self):
        """Update the library override operations based on the differences between this override ID and its reference"""
        ...

    def reset(
        self,
        do_hierarchy: bool | typing.Any | None = True,
        set_system_override: bool | typing.Any | None = False,
    ):
        """Reset this override to match again its linked reference ID

        :param do_hierarchy: Also reset all the dependencies of this override to match their reference linked IDs
        :type do_hierarchy: bool | typing.Any | None
        :param set_system_override: Reset all user-editable overrides as (non-editable) system overrides
        :type set_system_override: bool | typing.Any | None
        """
        ...

    def destroy(self, do_hierarchy: bool | typing.Any | None = True):
        """Delete this override ID and remap its usages to its linked reference ID instead

        :param do_hierarchy: Also delete all the dependencies of this override and remap their usages to their reference linked IDs
        :type do_hierarchy: bool | typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IDOverrideLibraryProperties(
    bpy_prop_collection[IDOverrideLibraryProperty], bpy_struct
):
    """Collection of override properties"""

    def add(self, rna_path: str | typing.Any) -> IDOverrideLibraryProperty:
        """Add a property to the override library when it doesn't exist yet

        :param rna_path: RNA Path, RNA-Path of the property to add
        :type rna_path: str | typing.Any
        :return: New Property, Newly created override property or existing one
        :rtype: IDOverrideLibraryProperty
        """
        ...

    def remove(self, property: IDOverrideLibraryProperty | None):
        """Remove and delete a property

        :param property: Property, Override property to be deleted
        :type property: IDOverrideLibraryProperty | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IDOverrideLibraryProperty(bpy_struct):
    """Description of an overridden property"""

    operations: IDOverrideLibraryPropertyOperations
    """ List of overriding operations for a property

    :type: IDOverrideLibraryPropertyOperations
    """

    rna_path: str
    """ RNA path leading to that property, from owning ID

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IDOverrideLibraryPropertyOperation(bpy_struct):
    """Description of an override operation over an overridden property"""

    flag: str
    """ Optional flags (NOT USED)

    :type: str
    """

    operation: str
    """ What override operation is performed

    :type: str
    """

    subitem_local_index: int
    """ Used to handle insertions into collection

    :type: int
    """

    subitem_local_name: str
    """ Used to handle insertions into collection

    :type: str
    """

    subitem_reference_index: int
    """ Used to handle insertions into collection

    :type: int
    """

    subitem_reference_name: str
    """ Used to handle insertions into collection

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IDOverrideLibraryPropertyOperations(
    bpy_prop_collection[IDOverrideLibraryPropertyOperation], bpy_struct
):
    """Collection of override operations"""

    def add(
        self,
        operation: str | None,
        subitem_reference_name: str | typing.Any = "",
        subitem_local_name: str | typing.Any = "",
        subitem_reference_index: typing.Any | None = -1,
        subitem_local_index: typing.Any | None = -1,
    ) -> IDOverrideLibraryPropertyOperation:
        """Add a new operation

                :param operation: Operation, What override operation is performed

        NOOP
        No-Op -- Does nothing, prevents adding actual overrides (NOT USED).

        REPLACE
        Replace -- Replace value of reference by overriding one.

        DIFF_ADD
        Differential -- Stores and apply difference between reference and local value (NOT USED).

        DIFF_SUB
        Differential -- Stores and apply difference between reference and local value (NOT USED).

        FACT_MULTIPLY
        Factor -- Stores and apply multiplication factor between reference and local value (NOT USED).

        INSERT_AFTER
        Insert After -- Insert a new item into collection after the one referenced in subitem_reference_name or _index.

        INSERT_BEFORE
        Insert Before -- Insert a new item into collection after the one referenced in subitem_reference_name or _index (NOT USED).
                :type operation: str | None
                :param subitem_reference_name: Subitem Reference Name, Used to handle insertions into collection
                :type subitem_reference_name: str | typing.Any
                :param subitem_local_name: Subitem Local Name, Used to handle insertions into collection
                :type subitem_local_name: str | typing.Any
                :param subitem_reference_index: Subitem Reference Index, Used to handle insertions into collection
                :type subitem_reference_index: typing.Any | None
                :param subitem_local_index: Subitem Local Index, Used to handle insertions into collection
                :type subitem_local_index: typing.Any | None
                :return: New Operation, Created operation
                :rtype: IDOverrideLibraryPropertyOperation
        """
        ...

    def remove(self, operation: IDOverrideLibraryPropertyOperation | None):
        """Remove and delete an operation

        :param operation: Operation, Override operation to be deleted
        :type operation: IDOverrideLibraryPropertyOperation | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IDPropertyWrapPtr(bpy_struct):
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IKParam(bpy_struct):
    """Base type for IK solver parameters"""

    ik_solver: str
    """ IK solver for which these parameters are defined

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ImageFormatSettings(bpy_struct):
    """Settings for image formats"""

    cineon_black: int
    """ Log conversion reference blackpoint

    :type: int
    """

    cineon_gamma: float
    """ Log conversion gamma

    :type: float
    """

    cineon_white: int
    """ Log conversion reference whitepoint

    :type: int
    """

    color_depth: str
    """ Bit depth per channel

    :type: str
    """

    color_management: str
    """ Which color management settings to use for file saving

    :type: str
    """

    color_mode: str
    """ Choose BW for saving grayscale images, RGB for saving red, green and blue channels, and RGBA for saving red, green, blue and alpha channels

    :type: str
    """

    compression: int
    """ Amount of time to determine best compression: 0 = no compression with fast file output, 100 = maximum lossless compression with slow file output

    :type: int
    """

    display_settings: ColorManagedDisplaySettings
    """ Settings of device saved image would be displayed on

    :type: ColorManagedDisplaySettings
    """

    exr_codec: str
    """ Codec settings for OpenEXR

    :type: str
    """

    file_format: str
    """ File format to save the rendered images as

    :type: str
    """

    has_linear_colorspace: bool
    """ File format expects linear color space

    :type: bool
    """

    jpeg2k_codec: str
    """ Codec settings for Jpeg2000

    :type: str
    """

    linear_colorspace_settings: ColorManagedInputColorspaceSettings
    """ Output color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    quality: int
    """ Quality for image formats that support lossy compression

    :type: int
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3D

    :type: Stereo3dFormat
    """

    tiff_codec: str
    """ Compression mode for TIFF

    :type: str
    """

    use_cineon_log: bool
    """ Convert to logarithmic color space

    :type: bool
    """

    use_jpeg2k_cinema_48: bool
    """ Use Openjpeg Cinema Preset (48fps)

    :type: bool
    """

    use_jpeg2k_cinema_preset: bool
    """ Use Openjpeg Cinema Preset

    :type: bool
    """

    use_jpeg2k_ycc: bool
    """ Save luminance-chrominance-chrominance channels instead of RGB colors

    :type: bool
    """

    use_preview: bool
    """ When rendering animations, save JPG preview images in same directory

    :type: bool
    """

    use_zbuffer: bool
    """ Save the z-depth per pixel (32-bit unsigned integer z-buffer)

    :type: bool
    """

    view_settings: ColorManagedViewSettings
    """ Color management settings applied on image before saving

    :type: ColorManagedViewSettings
    """

    views_format: str
    """ Format of multiview media

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ImagePackedFile(bpy_struct):
    filepath: str
    """ 

    :type: str
    """

    packed_file: PackedFile
    """ 

    :type: PackedFile
    """

    def save(self):
        """Save the packed file to its filepath"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ImagePreview(bpy_struct):
    """Preview image and icon"""

    icon_id: int
    """ Unique integer identifying this preview as an icon (zero means invalid)

    :type: int
    """

    icon_pixels: int
    """ Icon pixels, as bytes (always 32-bit RGBA)

    :type: int
    """

    icon_pixels_float: float
    """ Icon pixels components, as floats (RGBA concatenated values)

    :type: float
    """

    icon_size: bpy_prop_array[int]
    """ Width and height in pixels

    :type: bpy_prop_array[int]
    """

    image_pixels: int
    """ Image pixels, as bytes (always 32-bit RGBA)

    :type: int
    """

    image_pixels_float: float
    """ Image pixels components, as floats (RGBA concatenated values)

    :type: float
    """

    image_size: bpy_prop_array[int]
    """ Width and height in pixels

    :type: bpy_prop_array[int]
    """

    is_icon_custom: bool
    """ True if this preview icon has been modified by py script,and is no more auto-generated by Blender

    :type: bool
    """

    is_image_custom: bool
    """ True if this preview image has been modified by py script,and is no more auto-generated by Blender

    :type: bool
    """

    def reload(self):
        """Reload the preview from its source path"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ImageUser(bpy_struct):
    """Parameters defining how an Image data-block is used by another data-block"""

    frame_current: int
    """ Current frame number in image sequence or movie

    :type: int
    """

    frame_duration: int
    """ Number of images of a movie to use

    :type: int
    """

    frame_offset: int
    """ Offset the number of the frame to use in the animation

    :type: int
    """

    frame_start: int
    """ Global starting frame of the movie/sequence, assuming first picture has a #1

    :type: int
    """

    multilayer_layer: int
    """ Layer in multilayer image

    :type: int
    """

    multilayer_pass: int
    """ Pass in multilayer image

    :type: int
    """

    multilayer_view: int
    """ View in multilayer image

    :type: int
    """

    tile: int
    """ Tile in tiled image

    :type: int
    """

    use_auto_refresh: bool
    """ Always refresh image on frame changes

    :type: bool
    """

    use_cyclic: bool
    """ Cycle the images in the movie

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class IntAttributeValue(bpy_struct):
    """Integer value in geometry attribute"""

    value: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyConfig(bpy_struct):
    """Input configuration, including keymaps"""

    is_user_defined: bool
    """ Indicates that a keyconfig was defined by the user

    :type: bool
    """

    keymaps: KeyMaps
    """ Key maps configured as part of this configuration

    :type: KeyMaps
    """

    name: str
    """ Name of the key configuration

    :type: str
    """

    preferences: KeyConfigPreferences
    """ 

    :type: KeyConfigPreferences
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyConfigPreferences(bpy_struct):
    bl_idname: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyConfigurations(bpy_prop_collection[KeyConfig], bpy_struct):
    """Collection of KeyConfigs"""

    active: KeyConfig | None
    """ Active key configuration (preset)

    :type: KeyConfig | None
    """

    addon: KeyConfig
    """ Key configuration that can be extended by add-ons, and is added to the active configuration when handling events

    :type: KeyConfig
    """

    default: KeyConfig
    """ Default builtin key configuration

    :type: KeyConfig
    """

    user: KeyConfig
    """ Final key configuration that combines keymaps from the active and add-on configurations, and can be edited by the user

    :type: KeyConfig
    """

    def new(self, name: str | typing.Any) -> KeyConfig:
        """new

        :param name: Name
        :type name: str | typing.Any
        :return: Key Configuration, Added key configuration
        :rtype: KeyConfig
        """
        ...

    def remove(self, keyconfig: KeyConfig):
        """remove

        :param keyconfig: Key Configuration, Removed key configuration
        :type keyconfig: KeyConfig
        """
        ...

    def find_item_from_operator(
        self,
        idname: str | typing.Any,
        context: str | None = "INVOKE_DEFAULT",
        properties: OperatorProperties | None = None,
        include: set[str] | None = {"ACTIONZONE", "KEYBOARD", "MOUSE", "NDOF"},
        exclude: set[str] | None = {},
    ):
        """find_item_from_operator

                :param idname: Operator Identifier
                :type idname: str | typing.Any
                :param context: context
                :type context: str | None
                :param properties:
                :type properties: OperatorProperties | None
                :param include: Include
                :type include: set[str] | None
                :param exclude: Exclude
                :type exclude: set[str] | None
                :return: keymap, `KeyMap`

        item, `KeyMapItem`
        """
        ...

    def update(self):
        """update"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyMap(bpy_struct):
    """Input configuration, including keymaps"""

    bl_owner_id: str
    """ Internal owner

    :type: str
    """

    is_modal: bool
    """ Indicates that a keymap is used for translate modal events for an operator

    :type: bool
    """

    is_user_modified: bool
    """ Keymap is defined by the user

    :type: bool
    """

    keymap_items: KeyMapItems
    """ Items in the keymap, linking an operator to an input event

    :type: KeyMapItems
    """

    name: str
    """ Name of the key map

    :type: str
    """

    region_type: str
    """ Optional region type keymap is associated with

    :type: str
    """

    show_expanded_children: bool
    """ Children expanded in the user interface

    :type: bool
    """

    show_expanded_items: bool
    """ Expanded in the user interface

    :type: bool
    """

    space_type: str
    """ Optional space type keymap is associated with

    :type: str
    """

    def active(self) -> KeyMap:
        """active

        :return: Key Map, Active key map
        :rtype: KeyMap
        """
        ...

    def restore_to_default(self):
        """restore_to_default"""
        ...

    def restore_item_to_default(self, item: KeyMapItem):
        """restore_item_to_default

        :param item: Item
        :type item: KeyMapItem
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyMapItem(bpy_struct):
    """Item in a Key Map"""

    active: bool | None
    """ Activate or deactivate item

    :type: bool | None
    """

    alt: int
    """ Alt key pressed, -1 for any state

    :type: int
    """

    alt_ui: bool
    """ Alt key pressed

    :type: bool
    """

    any: bool
    """ Any modifier keys pressed

    :type: bool
    """

    ctrl: int
    """ Control key pressed, -1 for any state

    :type: int
    """

    ctrl_ui: bool
    """ Control key pressed

    :type: bool
    """

    direction: str
    """ The direction (only applies to drag events)

    :type: str
    """

    id: int
    """ ID of the item

    :type: int
    """

    idname: str
    """ Identifier of operator to call on input event

    :type: str
    """

    is_user_defined: bool
    """ Is this keymap item user defined (doesn't just replace a builtin item)

    :type: bool
    """

    is_user_modified: bool
    """ Is this keymap item modified by the user

    :type: bool
    """

    key_modifier: str
    """ Regular key pressed as a modifier

    :type: str
    """

    map_type: str
    """ Type of event mapping

    :type: str
    """

    name: str
    """ Name of operator (translated) to call on input event

    :type: str
    """

    oskey: int
    """ Operating system key pressed, -1 for any state

    :type: int
    """

    oskey_ui: bool
    """ Operating system key pressed

    :type: bool
    """

    properties: OperatorProperties
    """ Properties to set when the operator is called

    :type: OperatorProperties
    """

    propvalue: str
    """ The value this event translates to in a modal keymap

    :type: str
    """

    repeat: bool
    """ Active on key-repeat events (when a key is held)

    :type: bool
    """

    shift: int
    """ Shift key pressed, -1 for any state

    :type: int
    """

    shift_ui: bool
    """ Shift key pressed

    :type: bool
    """

    show_expanded: bool
    """ Show key map event and property details in the user interface

    :type: bool
    """

    type: str
    """ Type of event

    :type: str
    """

    value: str
    """ 

    :type: str
    """

    def compare(self, item: KeyMapItem | None) -> bool:
        """compare

        :param item: Item
        :type item: KeyMapItem | None
        :return: Comparison result
        :rtype: bool
        """
        ...

    def to_string(self, compact: bool | typing.Any | None = False) -> str | typing.Any:
        """to_string

        :param compact: Compact
        :type compact: bool | typing.Any | None
        :return: result
        :rtype: str | typing.Any
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyMapItems(bpy_prop_collection[KeyMapItem], bpy_struct):
    """Collection of keymap items"""

    def new(
        self,
        idname: str | typing.Any,
        type: str | None,
        value: str | None,
        any: bool | typing.Any | None = False,
        shift: typing.Any | None = 0,
        ctrl: typing.Any | None = 0,
        alt: typing.Any | None = 0,
        oskey: typing.Any | None = 0,
        key_modifier: str | None = "NONE",
        direction: str | None = "ANY",
        repeat: bool | typing.Any | None = False,
        head: bool | typing.Any | None = False,
    ) -> KeyMapItem:
        """new

                :param idname: Operator Identifier
                :type idname: str | typing.Any
                :param type: Type

        NONE
        Undocumented.

        LEFTMOUSE
        Left Mouse -- LMB.

        MIDDLEMOUSE
        Middle Mouse -- MMB.

        RIGHTMOUSE
        Right Mouse -- RMB.

        BUTTON4MOUSE
        Button4 Mouse -- MB4.

        BUTTON5MOUSE
        Button5 Mouse -- MB5.

        BUTTON6MOUSE
        Button6 Mouse -- MB6.

        BUTTON7MOUSE
        Button7 Mouse -- MB7.

        PEN
        Pen.

        ERASER
        Eraser.

        MOUSEMOVE
        Mouse Move -- MsMov.

        INBETWEEN_MOUSEMOVE
        In-between Move -- MsSubMov.

        TRACKPADPAN
        Mouse/Trackpad Pan -- MsPan.

        TRACKPADZOOM
        Mouse/Trackpad Zoom -- MsZoom.

        MOUSEROTATE
        Mouse/Trackpad Rotate -- MsRot.

        MOUSESMARTZOOM
        Mouse/Trackpad Smart Zoom -- MsSmartZoom.

        WHEELUPMOUSE
        Wheel Up -- WhUp.

        WHEELDOWNMOUSE
        Wheel Down -- WhDown.

        WHEELINMOUSE
        Wheel In -- WhIn.

        WHEELOUTMOUSE
        Wheel Out -- WhOut.

        A
        A.

        B
        B.

        C
        C.

        D
        D.

        E
        E.

        F
        F.

        G
        G.

        H
        H.

        I
        I.

        J
        J.

        K
        K.

        L
        L.

        M
        M.

        N
        N.

        O
        O.

        P
        P.

        Q
        Q.

        R
        R.

        S
        S.

        T
        T.

        U
        U.

        V
        V.

        W
        W.

        X
        X.

        Y
        Y.

        Z
        Z.

        ZERO
        0.

        ONE
        1.

        TWO
        2.

        THREE
        3.

        FOUR
        4.

        FIVE
        5.

        SIX
        6.

        SEVEN
        7.

        EIGHT
        8.

        NINE
        9.

        LEFT_CTRL
        Left Ctrl -- CtrlL.

        LEFT_ALT
        Left Alt -- AltL.

        LEFT_SHIFT
        Left Shift -- ShiftL.

        RIGHT_ALT
        Right Alt -- AltR.

        RIGHT_CTRL
        Right Ctrl -- CtrlR.

        RIGHT_SHIFT
        Right Shift -- ShiftR.

        OSKEY
        OS Key -- Cmd.

        APP
        Application -- App.

        GRLESS
        Grless.

        ESC
        Esc.

        TAB
        Tab.

        RET
        Return -- Enter.

        SPACE
        Spacebar -- Space.

        LINE_FEED
        Line Feed.

        BACK_SPACE
        Backspace -- BkSpace.

        DEL
        Delete -- Del.

        SEMI_COLON
        ;.

        PERIOD
        ..

        COMMA
        ,.

        QUOTE
        ".

        ACCENT_GRAVE
        `.

        MINUS
        -.

        PLUS
        +.

        SLASH
        /.

        BACK_SLASH
        \.

        EQUAL
        =.

        LEFT_BRACKET
        [.

        RIGHT_BRACKET
        ].

        LEFT_ARROW
        Left Arrow -- ←.

        DOWN_ARROW
        Down Arrow -- ↓.

        RIGHT_ARROW
        Right Arrow -- →.

        UP_ARROW
        Up Arrow -- ↑.

        NUMPAD_2
        Numpad 2 -- Pad2.

        NUMPAD_4
        Numpad 4 -- Pad4.

        NUMPAD_6
        Numpad 6 -- Pad6.

        NUMPAD_8
        Numpad 8 -- Pad8.

        NUMPAD_1
        Numpad 1 -- Pad1.

        NUMPAD_3
        Numpad 3 -- Pad3.

        NUMPAD_5
        Numpad 5 -- Pad5.

        NUMPAD_7
        Numpad 7 -- Pad7.

        NUMPAD_9
        Numpad 9 -- Pad9.

        NUMPAD_PERIOD
        Numpad . -- Pad..

        NUMPAD_SLASH
        Numpad / -- Pad/.

        NUMPAD_ASTERIX
        Numpad * -- Pad*.

        NUMPAD_0
        Numpad 0 -- Pad0.

        NUMPAD_MINUS
        Numpad - -- Pad-.

        NUMPAD_ENTER
        Numpad Enter -- PadEnter.

        NUMPAD_PLUS
        Numpad + -- Pad+.

        F1
        F1.

        F2
        F2.

        F3
        F3.

        F4
        F4.

        F5
        F5.

        F6
        F6.

        F7
        F7.

        F8
        F8.

        F9
        F9.

        F10
        F10.

        F11
        F11.

        F12
        F12.

        F13
        F13.

        F14
        F14.

        F15
        F15.

        F16
        F16.

        F17
        F17.

        F18
        F18.

        F19
        F19.

        F20
        F20.

        F21
        F21.

        F22
        F22.

        F23
        F23.

        F24
        F24.

        PAUSE
        Pause.

        INSERT
        Insert -- Ins.

        HOME
        Home.

        PAGE_UP
        Page Up -- PgUp.

        PAGE_DOWN
        Page Down -- PgDown.

        END
        End.

        MEDIA_PLAY
        Media Play/Pause -- >/||.

        MEDIA_STOP
        Media Stop -- Stop.

        MEDIA_FIRST
        Media First -- |<<.

        MEDIA_LAST
        Media Last -- >>|.

        TEXTINPUT
        Text Input -- TxtIn.

        WINDOW_DEACTIVATE
        Window Deactivate.

        TIMER
        Timer -- Tmr.

        TIMER0
        Timer 0 -- Tmr0.

        TIMER1
        Timer 1 -- Tmr1.

        TIMER2
        Timer 2 -- Tmr2.

        TIMER_JOBS
        Timer Jobs -- TmrJob.

        TIMER_AUTOSAVE
        Timer Autosave -- TmrSave.

        TIMER_REPORT
        Timer Report -- TmrReport.

        TIMERREGION
        Timer Region -- TmrReg.

        NDOF_MOTION
        NDOF Motion -- NdofMov.

        NDOF_BUTTON_MENU
        NDOF Menu -- NdofMenu.

        NDOF_BUTTON_FIT
        NDOF Fit -- NdofFit.

        NDOF_BUTTON_TOP
        NDOF Top -- Ndof↑.

        NDOF_BUTTON_BOTTOM
        NDOF Bottom -- Ndof↓.

        NDOF_BUTTON_LEFT
        NDOF Left -- Ndof←.

        NDOF_BUTTON_RIGHT
        NDOF Right -- Ndof→.

        NDOF_BUTTON_FRONT
        NDOF Front -- NdofFront.

        NDOF_BUTTON_BACK
        NDOF Back -- NdofBack.

        NDOF_BUTTON_ISO1
        NDOF Isometric 1 -- NdofIso1.

        NDOF_BUTTON_ISO2
        NDOF Isometric 2 -- NdofIso2.

        NDOF_BUTTON_ROLL_CW
        NDOF Roll CW -- NdofRCW.

        NDOF_BUTTON_ROLL_CCW
        NDOF Roll CCW -- NdofRCCW.

        NDOF_BUTTON_SPIN_CW
        NDOF Spin CW -- NdofSCW.

        NDOF_BUTTON_SPIN_CCW
        NDOF Spin CCW -- NdofSCCW.

        NDOF_BUTTON_TILT_CW
        NDOF Tilt CW -- NdofTCW.

        NDOF_BUTTON_TILT_CCW
        NDOF Tilt CCW -- NdofTCCW.

        NDOF_BUTTON_ROTATE
        NDOF Rotate -- NdofRot.

        NDOF_BUTTON_PANZOOM
        NDOF Pan/Zoom -- NdofPanZoom.

        NDOF_BUTTON_DOMINANT
        NDOF Dominant -- NdofDom.

        NDOF_BUTTON_PLUS
        NDOF Plus -- Ndof+.

        NDOF_BUTTON_MINUS
        NDOF Minus -- Ndof-.

        NDOF_BUTTON_1
        NDOF Button 1 -- NdofB1.

        NDOF_BUTTON_2
        NDOF Button 2 -- NdofB2.

        NDOF_BUTTON_3
        NDOF Button 3 -- NdofB3.

        NDOF_BUTTON_4
        NDOF Button 4 -- NdofB4.

        NDOF_BUTTON_5
        NDOF Button 5 -- NdofB5.

        NDOF_BUTTON_6
        NDOF Button 6 -- NdofB6.

        NDOF_BUTTON_7
        NDOF Button 7 -- NdofB7.

        NDOF_BUTTON_8
        NDOF Button 8 -- NdofB8.

        NDOF_BUTTON_9
        NDOF Button 9 -- NdofB9.

        NDOF_BUTTON_10
        NDOF Button 10 -- NdofB10.

        NDOF_BUTTON_A
        NDOF Button A -- NdofBA.

        NDOF_BUTTON_B
        NDOF Button B -- NdofBB.

        NDOF_BUTTON_C
        NDOF Button C -- NdofBC.

        ACTIONZONE_AREA
        ActionZone Area -- AZone Area.

        ACTIONZONE_REGION
        ActionZone Region -- AZone Region.

        ACTIONZONE_FULLSCREEN
        ActionZone Fullscreen -- AZone FullScr.

        XR_ACTION
        XR Action.
                :type type: str | None
                :param value: Value
                :type value: str | None
                :param any: Any
                :type any: bool | typing.Any | None
                :param shift: Shift
                :type shift: typing.Any | None
                :param ctrl: Ctrl
                :type ctrl: typing.Any | None
                :param alt: Alt
                :type alt: typing.Any | None
                :param oskey: OS Key
                :type oskey: typing.Any | None
                :param key_modifier: Key Modifier

        NONE
        Undocumented.

        LEFTMOUSE
        Left Mouse -- LMB.

        MIDDLEMOUSE
        Middle Mouse -- MMB.

        RIGHTMOUSE
        Right Mouse -- RMB.

        BUTTON4MOUSE
        Button4 Mouse -- MB4.

        BUTTON5MOUSE
        Button5 Mouse -- MB5.

        BUTTON6MOUSE
        Button6 Mouse -- MB6.

        BUTTON7MOUSE
        Button7 Mouse -- MB7.

        PEN
        Pen.

        ERASER
        Eraser.

        MOUSEMOVE
        Mouse Move -- MsMov.

        INBETWEEN_MOUSEMOVE
        In-between Move -- MsSubMov.

        TRACKPADPAN
        Mouse/Trackpad Pan -- MsPan.

        TRACKPADZOOM
        Mouse/Trackpad Zoom -- MsZoom.

        MOUSEROTATE
        Mouse/Trackpad Rotate -- MsRot.

        MOUSESMARTZOOM
        Mouse/Trackpad Smart Zoom -- MsSmartZoom.

        WHEELUPMOUSE
        Wheel Up -- WhUp.

        WHEELDOWNMOUSE
        Wheel Down -- WhDown.

        WHEELINMOUSE
        Wheel In -- WhIn.

        WHEELOUTMOUSE
        Wheel Out -- WhOut.

        A
        A.

        B
        B.

        C
        C.

        D
        D.

        E
        E.

        F
        F.

        G
        G.

        H
        H.

        I
        I.

        J
        J.

        K
        K.

        L
        L.

        M
        M.

        N
        N.

        O
        O.

        P
        P.

        Q
        Q.

        R
        R.

        S
        S.

        T
        T.

        U
        U.

        V
        V.

        W
        W.

        X
        X.

        Y
        Y.

        Z
        Z.

        ZERO
        0.

        ONE
        1.

        TWO
        2.

        THREE
        3.

        FOUR
        4.

        FIVE
        5.

        SIX
        6.

        SEVEN
        7.

        EIGHT
        8.

        NINE
        9.

        LEFT_CTRL
        Left Ctrl -- CtrlL.

        LEFT_ALT
        Left Alt -- AltL.

        LEFT_SHIFT
        Left Shift -- ShiftL.

        RIGHT_ALT
        Right Alt -- AltR.

        RIGHT_CTRL
        Right Ctrl -- CtrlR.

        RIGHT_SHIFT
        Right Shift -- ShiftR.

        OSKEY
        OS Key -- Cmd.

        APP
        Application -- App.

        GRLESS
        Grless.

        ESC
        Esc.

        TAB
        Tab.

        RET
        Return -- Enter.

        SPACE
        Spacebar -- Space.

        LINE_FEED
        Line Feed.

        BACK_SPACE
        Backspace -- BkSpace.

        DEL
        Delete -- Del.

        SEMI_COLON
        ;.

        PERIOD
        ..

        COMMA
        ,.

        QUOTE
        ".

        ACCENT_GRAVE
        `.

        MINUS
        -.

        PLUS
        +.

        SLASH
        /.

        BACK_SLASH
        \.

        EQUAL
        =.

        LEFT_BRACKET
        [.

        RIGHT_BRACKET
        ].

        LEFT_ARROW
        Left Arrow -- ←.

        DOWN_ARROW
        Down Arrow -- ↓.

        RIGHT_ARROW
        Right Arrow -- →.

        UP_ARROW
        Up Arrow -- ↑.

        NUMPAD_2
        Numpad 2 -- Pad2.

        NUMPAD_4
        Numpad 4 -- Pad4.

        NUMPAD_6
        Numpad 6 -- Pad6.

        NUMPAD_8
        Numpad 8 -- Pad8.

        NUMPAD_1
        Numpad 1 -- Pad1.

        NUMPAD_3
        Numpad 3 -- Pad3.

        NUMPAD_5
        Numpad 5 -- Pad5.

        NUMPAD_7
        Numpad 7 -- Pad7.

        NUMPAD_9
        Numpad 9 -- Pad9.

        NUMPAD_PERIOD
        Numpad . -- Pad..

        NUMPAD_SLASH
        Numpad / -- Pad/.

        NUMPAD_ASTERIX
        Numpad * -- Pad*.

        NUMPAD_0
        Numpad 0 -- Pad0.

        NUMPAD_MINUS
        Numpad - -- Pad-.

        NUMPAD_ENTER
        Numpad Enter -- PadEnter.

        NUMPAD_PLUS
        Numpad + -- Pad+.

        F1
        F1.

        F2
        F2.

        F3
        F3.

        F4
        F4.

        F5
        F5.

        F6
        F6.

        F7
        F7.

        F8
        F8.

        F9
        F9.

        F10
        F10.

        F11
        F11.

        F12
        F12.

        F13
        F13.

        F14
        F14.

        F15
        F15.

        F16
        F16.

        F17
        F17.

        F18
        F18.

        F19
        F19.

        F20
        F20.

        F21
        F21.

        F22
        F22.

        F23
        F23.

        F24
        F24.

        PAUSE
        Pause.

        INSERT
        Insert -- Ins.

        HOME
        Home.

        PAGE_UP
        Page Up -- PgUp.

        PAGE_DOWN
        Page Down -- PgDown.

        END
        End.

        MEDIA_PLAY
        Media Play/Pause -- >/||.

        MEDIA_STOP
        Media Stop -- Stop.

        MEDIA_FIRST
        Media First -- |<<.

        MEDIA_LAST
        Media Last -- >>|.

        TEXTINPUT
        Text Input -- TxtIn.

        WINDOW_DEACTIVATE
        Window Deactivate.

        TIMER
        Timer -- Tmr.

        TIMER0
        Timer 0 -- Tmr0.

        TIMER1
        Timer 1 -- Tmr1.

        TIMER2
        Timer 2 -- Tmr2.

        TIMER_JOBS
        Timer Jobs -- TmrJob.

        TIMER_AUTOSAVE
        Timer Autosave -- TmrSave.

        TIMER_REPORT
        Timer Report -- TmrReport.

        TIMERREGION
        Timer Region -- TmrReg.

        NDOF_MOTION
        NDOF Motion -- NdofMov.

        NDOF_BUTTON_MENU
        NDOF Menu -- NdofMenu.

        NDOF_BUTTON_FIT
        NDOF Fit -- NdofFit.

        NDOF_BUTTON_TOP
        NDOF Top -- Ndof↑.

        NDOF_BUTTON_BOTTOM
        NDOF Bottom -- Ndof↓.

        NDOF_BUTTON_LEFT
        NDOF Left -- Ndof←.

        NDOF_BUTTON_RIGHT
        NDOF Right -- Ndof→.

        NDOF_BUTTON_FRONT
        NDOF Front -- NdofFront.

        NDOF_BUTTON_BACK
        NDOF Back -- NdofBack.

        NDOF_BUTTON_ISO1
        NDOF Isometric 1 -- NdofIso1.

        NDOF_BUTTON_ISO2
        NDOF Isometric 2 -- NdofIso2.

        NDOF_BUTTON_ROLL_CW
        NDOF Roll CW -- NdofRCW.

        NDOF_BUTTON_ROLL_CCW
        NDOF Roll CCW -- NdofRCCW.

        NDOF_BUTTON_SPIN_CW
        NDOF Spin CW -- NdofSCW.

        NDOF_BUTTON_SPIN_CCW
        NDOF Spin CCW -- NdofSCCW.

        NDOF_BUTTON_TILT_CW
        NDOF Tilt CW -- NdofTCW.

        NDOF_BUTTON_TILT_CCW
        NDOF Tilt CCW -- NdofTCCW.

        NDOF_BUTTON_ROTATE
        NDOF Rotate -- NdofRot.

        NDOF_BUTTON_PANZOOM
        NDOF Pan/Zoom -- NdofPanZoom.

        NDOF_BUTTON_DOMINANT
        NDOF Dominant -- NdofDom.

        NDOF_BUTTON_PLUS
        NDOF Plus -- Ndof+.

        NDOF_BUTTON_MINUS
        NDOF Minus -- Ndof-.

        NDOF_BUTTON_1
        NDOF Button 1 -- NdofB1.

        NDOF_BUTTON_2
        NDOF Button 2 -- NdofB2.

        NDOF_BUTTON_3
        NDOF Button 3 -- NdofB3.

        NDOF_BUTTON_4
        NDOF Button 4 -- NdofB4.

        NDOF_BUTTON_5
        NDOF Button 5 -- NdofB5.

        NDOF_BUTTON_6
        NDOF Button 6 -- NdofB6.

        NDOF_BUTTON_7
        NDOF Button 7 -- NdofB7.

        NDOF_BUTTON_8
        NDOF Button 8 -- NdofB8.

        NDOF_BUTTON_9
        NDOF Button 9 -- NdofB9.

        NDOF_BUTTON_10
        NDOF Button 10 -- NdofB10.

        NDOF_BUTTON_A
        NDOF Button A -- NdofBA.

        NDOF_BUTTON_B
        NDOF Button B -- NdofBB.

        NDOF_BUTTON_C
        NDOF Button C -- NdofBC.

        ACTIONZONE_AREA
        ActionZone Area -- AZone Area.

        ACTIONZONE_REGION
        ActionZone Region -- AZone Region.

        ACTIONZONE_FULLSCREEN
        ActionZone Fullscreen -- AZone FullScr.

        XR_ACTION
        XR Action.
                :type key_modifier: str | None
                :param direction: Direction
                :type direction: str | None
                :param repeat: Repeat, When set, accept key-repeat events
                :type repeat: bool | typing.Any | None
                :param head: At Head, Force item to be added at start (not end) of key map so that it doesn't get blocked by an existing key map item
                :type head: bool | typing.Any | None
                :return: Item, Added key map item
                :rtype: KeyMapItem
        """
        ...

    def new_modal(
        self,
        propvalue: str | typing.Any,
        type: str | None,
        value: str | None,
        any: bool | typing.Any | None = False,
        shift: typing.Any | None = 0,
        ctrl: typing.Any | None = 0,
        alt: typing.Any | None = 0,
        oskey: typing.Any | None = 0,
        key_modifier: str | None = "NONE",
        direction: str | None = "ANY",
        repeat: bool | typing.Any | None = False,
    ) -> KeyMapItem:
        """new_modal

                :param propvalue: Property Value
                :type propvalue: str | typing.Any
                :param type: Type

        NONE
        Undocumented.

        LEFTMOUSE
        Left Mouse -- LMB.

        MIDDLEMOUSE
        Middle Mouse -- MMB.

        RIGHTMOUSE
        Right Mouse -- RMB.

        BUTTON4MOUSE
        Button4 Mouse -- MB4.

        BUTTON5MOUSE
        Button5 Mouse -- MB5.

        BUTTON6MOUSE
        Button6 Mouse -- MB6.

        BUTTON7MOUSE
        Button7 Mouse -- MB7.

        PEN
        Pen.

        ERASER
        Eraser.

        MOUSEMOVE
        Mouse Move -- MsMov.

        INBETWEEN_MOUSEMOVE
        In-between Move -- MsSubMov.

        TRACKPADPAN
        Mouse/Trackpad Pan -- MsPan.

        TRACKPADZOOM
        Mouse/Trackpad Zoom -- MsZoom.

        MOUSEROTATE
        Mouse/Trackpad Rotate -- MsRot.

        MOUSESMARTZOOM
        Mouse/Trackpad Smart Zoom -- MsSmartZoom.

        WHEELUPMOUSE
        Wheel Up -- WhUp.

        WHEELDOWNMOUSE
        Wheel Down -- WhDown.

        WHEELINMOUSE
        Wheel In -- WhIn.

        WHEELOUTMOUSE
        Wheel Out -- WhOut.

        A
        A.

        B
        B.

        C
        C.

        D
        D.

        E
        E.

        F
        F.

        G
        G.

        H
        H.

        I
        I.

        J
        J.

        K
        K.

        L
        L.

        M
        M.

        N
        N.

        O
        O.

        P
        P.

        Q
        Q.

        R
        R.

        S
        S.

        T
        T.

        U
        U.

        V
        V.

        W
        W.

        X
        X.

        Y
        Y.

        Z
        Z.

        ZERO
        0.

        ONE
        1.

        TWO
        2.

        THREE
        3.

        FOUR
        4.

        FIVE
        5.

        SIX
        6.

        SEVEN
        7.

        EIGHT
        8.

        NINE
        9.

        LEFT_CTRL
        Left Ctrl -- CtrlL.

        LEFT_ALT
        Left Alt -- AltL.

        LEFT_SHIFT
        Left Shift -- ShiftL.

        RIGHT_ALT
        Right Alt -- AltR.

        RIGHT_CTRL
        Right Ctrl -- CtrlR.

        RIGHT_SHIFT
        Right Shift -- ShiftR.

        OSKEY
        OS Key -- Cmd.

        APP
        Application -- App.

        GRLESS
        Grless.

        ESC
        Esc.

        TAB
        Tab.

        RET
        Return -- Enter.

        SPACE
        Spacebar -- Space.

        LINE_FEED
        Line Feed.

        BACK_SPACE
        Backspace -- BkSpace.

        DEL
        Delete -- Del.

        SEMI_COLON
        ;.

        PERIOD
        ..

        COMMA
        ,.

        QUOTE
        ".

        ACCENT_GRAVE
        `.

        MINUS
        -.

        PLUS
        +.

        SLASH
        /.

        BACK_SLASH
        \.

        EQUAL
        =.

        LEFT_BRACKET
        [.

        RIGHT_BRACKET
        ].

        LEFT_ARROW
        Left Arrow -- ←.

        DOWN_ARROW
        Down Arrow -- ↓.

        RIGHT_ARROW
        Right Arrow -- →.

        UP_ARROW
        Up Arrow -- ↑.

        NUMPAD_2
        Numpad 2 -- Pad2.

        NUMPAD_4
        Numpad 4 -- Pad4.

        NUMPAD_6
        Numpad 6 -- Pad6.

        NUMPAD_8
        Numpad 8 -- Pad8.

        NUMPAD_1
        Numpad 1 -- Pad1.

        NUMPAD_3
        Numpad 3 -- Pad3.

        NUMPAD_5
        Numpad 5 -- Pad5.

        NUMPAD_7
        Numpad 7 -- Pad7.

        NUMPAD_9
        Numpad 9 -- Pad9.

        NUMPAD_PERIOD
        Numpad . -- Pad..

        NUMPAD_SLASH
        Numpad / -- Pad/.

        NUMPAD_ASTERIX
        Numpad * -- Pad*.

        NUMPAD_0
        Numpad 0 -- Pad0.

        NUMPAD_MINUS
        Numpad - -- Pad-.

        NUMPAD_ENTER
        Numpad Enter -- PadEnter.

        NUMPAD_PLUS
        Numpad + -- Pad+.

        F1
        F1.

        F2
        F2.

        F3
        F3.

        F4
        F4.

        F5
        F5.

        F6
        F6.

        F7
        F7.

        F8
        F8.

        F9
        F9.

        F10
        F10.

        F11
        F11.

        F12
        F12.

        F13
        F13.

        F14
        F14.

        F15
        F15.

        F16
        F16.

        F17
        F17.

        F18
        F18.

        F19
        F19.

        F20
        F20.

        F21
        F21.

        F22
        F22.

        F23
        F23.

        F24
        F24.

        PAUSE
        Pause.

        INSERT
        Insert -- Ins.

        HOME
        Home.

        PAGE_UP
        Page Up -- PgUp.

        PAGE_DOWN
        Page Down -- PgDown.

        END
        End.

        MEDIA_PLAY
        Media Play/Pause -- >/||.

        MEDIA_STOP
        Media Stop -- Stop.

        MEDIA_FIRST
        Media First -- |<<.

        MEDIA_LAST
        Media Last -- >>|.

        TEXTINPUT
        Text Input -- TxtIn.

        WINDOW_DEACTIVATE
        Window Deactivate.

        TIMER
        Timer -- Tmr.

        TIMER0
        Timer 0 -- Tmr0.

        TIMER1
        Timer 1 -- Tmr1.

        TIMER2
        Timer 2 -- Tmr2.

        TIMER_JOBS
        Timer Jobs -- TmrJob.

        TIMER_AUTOSAVE
        Timer Autosave -- TmrSave.

        TIMER_REPORT
        Timer Report -- TmrReport.

        TIMERREGION
        Timer Region -- TmrReg.

        NDOF_MOTION
        NDOF Motion -- NdofMov.

        NDOF_BUTTON_MENU
        NDOF Menu -- NdofMenu.

        NDOF_BUTTON_FIT
        NDOF Fit -- NdofFit.

        NDOF_BUTTON_TOP
        NDOF Top -- Ndof↑.

        NDOF_BUTTON_BOTTOM
        NDOF Bottom -- Ndof↓.

        NDOF_BUTTON_LEFT
        NDOF Left -- Ndof←.

        NDOF_BUTTON_RIGHT
        NDOF Right -- Ndof→.

        NDOF_BUTTON_FRONT
        NDOF Front -- NdofFront.

        NDOF_BUTTON_BACK
        NDOF Back -- NdofBack.

        NDOF_BUTTON_ISO1
        NDOF Isometric 1 -- NdofIso1.

        NDOF_BUTTON_ISO2
        NDOF Isometric 2 -- NdofIso2.

        NDOF_BUTTON_ROLL_CW
        NDOF Roll CW -- NdofRCW.

        NDOF_BUTTON_ROLL_CCW
        NDOF Roll CCW -- NdofRCCW.

        NDOF_BUTTON_SPIN_CW
        NDOF Spin CW -- NdofSCW.

        NDOF_BUTTON_SPIN_CCW
        NDOF Spin CCW -- NdofSCCW.

        NDOF_BUTTON_TILT_CW
        NDOF Tilt CW -- NdofTCW.

        NDOF_BUTTON_TILT_CCW
        NDOF Tilt CCW -- NdofTCCW.

        NDOF_BUTTON_ROTATE
        NDOF Rotate -- NdofRot.

        NDOF_BUTTON_PANZOOM
        NDOF Pan/Zoom -- NdofPanZoom.

        NDOF_BUTTON_DOMINANT
        NDOF Dominant -- NdofDom.

        NDOF_BUTTON_PLUS
        NDOF Plus -- Ndof+.

        NDOF_BUTTON_MINUS
        NDOF Minus -- Ndof-.

        NDOF_BUTTON_1
        NDOF Button 1 -- NdofB1.

        NDOF_BUTTON_2
        NDOF Button 2 -- NdofB2.

        NDOF_BUTTON_3
        NDOF Button 3 -- NdofB3.

        NDOF_BUTTON_4
        NDOF Button 4 -- NdofB4.

        NDOF_BUTTON_5
        NDOF Button 5 -- NdofB5.

        NDOF_BUTTON_6
        NDOF Button 6 -- NdofB6.

        NDOF_BUTTON_7
        NDOF Button 7 -- NdofB7.

        NDOF_BUTTON_8
        NDOF Button 8 -- NdofB8.

        NDOF_BUTTON_9
        NDOF Button 9 -- NdofB9.

        NDOF_BUTTON_10
        NDOF Button 10 -- NdofB10.

        NDOF_BUTTON_A
        NDOF Button A -- NdofBA.

        NDOF_BUTTON_B
        NDOF Button B -- NdofBB.

        NDOF_BUTTON_C
        NDOF Button C -- NdofBC.

        ACTIONZONE_AREA
        ActionZone Area -- AZone Area.

        ACTIONZONE_REGION
        ActionZone Region -- AZone Region.

        ACTIONZONE_FULLSCREEN
        ActionZone Fullscreen -- AZone FullScr.

        XR_ACTION
        XR Action.
                :type type: str | None
                :param value: Value
                :type value: str | None
                :param any: Any
                :type any: bool | typing.Any | None
                :param shift: Shift
                :type shift: typing.Any | None
                :param ctrl: Ctrl
                :type ctrl: typing.Any | None
                :param alt: Alt
                :type alt: typing.Any | None
                :param oskey: OS Key
                :type oskey: typing.Any | None
                :param key_modifier: Key Modifier

        NONE
        Undocumented.

        LEFTMOUSE
        Left Mouse -- LMB.

        MIDDLEMOUSE
        Middle Mouse -- MMB.

        RIGHTMOUSE
        Right Mouse -- RMB.

        BUTTON4MOUSE
        Button4 Mouse -- MB4.

        BUTTON5MOUSE
        Button5 Mouse -- MB5.

        BUTTON6MOUSE
        Button6 Mouse -- MB6.

        BUTTON7MOUSE
        Button7 Mouse -- MB7.

        PEN
        Pen.

        ERASER
        Eraser.

        MOUSEMOVE
        Mouse Move -- MsMov.

        INBETWEEN_MOUSEMOVE
        In-between Move -- MsSubMov.

        TRACKPADPAN
        Mouse/Trackpad Pan -- MsPan.

        TRACKPADZOOM
        Mouse/Trackpad Zoom -- MsZoom.

        MOUSEROTATE
        Mouse/Trackpad Rotate -- MsRot.

        MOUSESMARTZOOM
        Mouse/Trackpad Smart Zoom -- MsSmartZoom.

        WHEELUPMOUSE
        Wheel Up -- WhUp.

        WHEELDOWNMOUSE
        Wheel Down -- WhDown.

        WHEELINMOUSE
        Wheel In -- WhIn.

        WHEELOUTMOUSE
        Wheel Out -- WhOut.

        A
        A.

        B
        B.

        C
        C.

        D
        D.

        E
        E.

        F
        F.

        G
        G.

        H
        H.

        I
        I.

        J
        J.

        K
        K.

        L
        L.

        M
        M.

        N
        N.

        O
        O.

        P
        P.

        Q
        Q.

        R
        R.

        S
        S.

        T
        T.

        U
        U.

        V
        V.

        W
        W.

        X
        X.

        Y
        Y.

        Z
        Z.

        ZERO
        0.

        ONE
        1.

        TWO
        2.

        THREE
        3.

        FOUR
        4.

        FIVE
        5.

        SIX
        6.

        SEVEN
        7.

        EIGHT
        8.

        NINE
        9.

        LEFT_CTRL
        Left Ctrl -- CtrlL.

        LEFT_ALT
        Left Alt -- AltL.

        LEFT_SHIFT
        Left Shift -- ShiftL.

        RIGHT_ALT
        Right Alt -- AltR.

        RIGHT_CTRL
        Right Ctrl -- CtrlR.

        RIGHT_SHIFT
        Right Shift -- ShiftR.

        OSKEY
        OS Key -- Cmd.

        APP
        Application -- App.

        GRLESS
        Grless.

        ESC
        Esc.

        TAB
        Tab.

        RET
        Return -- Enter.

        SPACE
        Spacebar -- Space.

        LINE_FEED
        Line Feed.

        BACK_SPACE
        Backspace -- BkSpace.

        DEL
        Delete -- Del.

        SEMI_COLON
        ;.

        PERIOD
        ..

        COMMA
        ,.

        QUOTE
        ".

        ACCENT_GRAVE
        `.

        MINUS
        -.

        PLUS
        +.

        SLASH
        /.

        BACK_SLASH
        \.

        EQUAL
        =.

        LEFT_BRACKET
        [.

        RIGHT_BRACKET
        ].

        LEFT_ARROW
        Left Arrow -- ←.

        DOWN_ARROW
        Down Arrow -- ↓.

        RIGHT_ARROW
        Right Arrow -- →.

        UP_ARROW
        Up Arrow -- ↑.

        NUMPAD_2
        Numpad 2 -- Pad2.

        NUMPAD_4
        Numpad 4 -- Pad4.

        NUMPAD_6
        Numpad 6 -- Pad6.

        NUMPAD_8
        Numpad 8 -- Pad8.

        NUMPAD_1
        Numpad 1 -- Pad1.

        NUMPAD_3
        Numpad 3 -- Pad3.

        NUMPAD_5
        Numpad 5 -- Pad5.

        NUMPAD_7
        Numpad 7 -- Pad7.

        NUMPAD_9
        Numpad 9 -- Pad9.

        NUMPAD_PERIOD
        Numpad . -- Pad..

        NUMPAD_SLASH
        Numpad / -- Pad/.

        NUMPAD_ASTERIX
        Numpad * -- Pad*.

        NUMPAD_0
        Numpad 0 -- Pad0.

        NUMPAD_MINUS
        Numpad - -- Pad-.

        NUMPAD_ENTER
        Numpad Enter -- PadEnter.

        NUMPAD_PLUS
        Numpad + -- Pad+.

        F1
        F1.

        F2
        F2.

        F3
        F3.

        F4
        F4.

        F5
        F5.

        F6
        F6.

        F7
        F7.

        F8
        F8.

        F9
        F9.

        F10
        F10.

        F11
        F11.

        F12
        F12.

        F13
        F13.

        F14
        F14.

        F15
        F15.

        F16
        F16.

        F17
        F17.

        F18
        F18.

        F19
        F19.

        F20
        F20.

        F21
        F21.

        F22
        F22.

        F23
        F23.

        F24
        F24.

        PAUSE
        Pause.

        INSERT
        Insert -- Ins.

        HOME
        Home.

        PAGE_UP
        Page Up -- PgUp.

        PAGE_DOWN
        Page Down -- PgDown.

        END
        End.

        MEDIA_PLAY
        Media Play/Pause -- >/||.

        MEDIA_STOP
        Media Stop -- Stop.

        MEDIA_FIRST
        Media First -- |<<.

        MEDIA_LAST
        Media Last -- >>|.

        TEXTINPUT
        Text Input -- TxtIn.

        WINDOW_DEACTIVATE
        Window Deactivate.

        TIMER
        Timer -- Tmr.

        TIMER0
        Timer 0 -- Tmr0.

        TIMER1
        Timer 1 -- Tmr1.

        TIMER2
        Timer 2 -- Tmr2.

        TIMER_JOBS
        Timer Jobs -- TmrJob.

        TIMER_AUTOSAVE
        Timer Autosave -- TmrSave.

        TIMER_REPORT
        Timer Report -- TmrReport.

        TIMERREGION
        Timer Region -- TmrReg.

        NDOF_MOTION
        NDOF Motion -- NdofMov.

        NDOF_BUTTON_MENU
        NDOF Menu -- NdofMenu.

        NDOF_BUTTON_FIT
        NDOF Fit -- NdofFit.

        NDOF_BUTTON_TOP
        NDOF Top -- Ndof↑.

        NDOF_BUTTON_BOTTOM
        NDOF Bottom -- Ndof↓.

        NDOF_BUTTON_LEFT
        NDOF Left -- Ndof←.

        NDOF_BUTTON_RIGHT
        NDOF Right -- Ndof→.

        NDOF_BUTTON_FRONT
        NDOF Front -- NdofFront.

        NDOF_BUTTON_BACK
        NDOF Back -- NdofBack.

        NDOF_BUTTON_ISO1
        NDOF Isometric 1 -- NdofIso1.

        NDOF_BUTTON_ISO2
        NDOF Isometric 2 -- NdofIso2.

        NDOF_BUTTON_ROLL_CW
        NDOF Roll CW -- NdofRCW.

        NDOF_BUTTON_ROLL_CCW
        NDOF Roll CCW -- NdofRCCW.

        NDOF_BUTTON_SPIN_CW
        NDOF Spin CW -- NdofSCW.

        NDOF_BUTTON_SPIN_CCW
        NDOF Spin CCW -- NdofSCCW.

        NDOF_BUTTON_TILT_CW
        NDOF Tilt CW -- NdofTCW.

        NDOF_BUTTON_TILT_CCW
        NDOF Tilt CCW -- NdofTCCW.

        NDOF_BUTTON_ROTATE
        NDOF Rotate -- NdofRot.

        NDOF_BUTTON_PANZOOM
        NDOF Pan/Zoom -- NdofPanZoom.

        NDOF_BUTTON_DOMINANT
        NDOF Dominant -- NdofDom.

        NDOF_BUTTON_PLUS
        NDOF Plus -- Ndof+.

        NDOF_BUTTON_MINUS
        NDOF Minus -- Ndof-.

        NDOF_BUTTON_1
        NDOF Button 1 -- NdofB1.

        NDOF_BUTTON_2
        NDOF Button 2 -- NdofB2.

        NDOF_BUTTON_3
        NDOF Button 3 -- NdofB3.

        NDOF_BUTTON_4
        NDOF Button 4 -- NdofB4.

        NDOF_BUTTON_5
        NDOF Button 5 -- NdofB5.

        NDOF_BUTTON_6
        NDOF Button 6 -- NdofB6.

        NDOF_BUTTON_7
        NDOF Button 7 -- NdofB7.

        NDOF_BUTTON_8
        NDOF Button 8 -- NdofB8.

        NDOF_BUTTON_9
        NDOF Button 9 -- NdofB9.

        NDOF_BUTTON_10
        NDOF Button 10 -- NdofB10.

        NDOF_BUTTON_A
        NDOF Button A -- NdofBA.

        NDOF_BUTTON_B
        NDOF Button B -- NdofBB.

        NDOF_BUTTON_C
        NDOF Button C -- NdofBC.

        ACTIONZONE_AREA
        ActionZone Area -- AZone Area.

        ACTIONZONE_REGION
        ActionZone Region -- AZone Region.

        ACTIONZONE_FULLSCREEN
        ActionZone Fullscreen -- AZone FullScr.

        XR_ACTION
        XR Action.
                :type key_modifier: str | None
                :param direction: Direction
                :type direction: str | None
                :param repeat: Repeat, When set, accept key-repeat events
                :type repeat: bool | typing.Any | None
                :return: Item, Added key map item
                :rtype: KeyMapItem
        """
        ...

    def new_from_item(
        self, item: KeyMapItem, head: bool | typing.Any | None = False
    ) -> KeyMapItem:
        """new_from_item

        :param item: Item, Item to use as a reference
        :type item: KeyMapItem
        :param head: At Head
        :type head: bool | typing.Any | None
        :return: Item, Added key map item
        :rtype: KeyMapItem
        """
        ...

    def remove(self, item: KeyMapItem):
        """remove

        :param item: Item
        :type item: KeyMapItem
        """
        ...

    def from_id(self, id: int | None) -> KeyMapItem:
        """from_id

        :param id: id, ID of the item
        :type id: int | None
        :return: Item
        :rtype: KeyMapItem
        """
        ...

    def find_from_operator(
        self,
        idname: str | typing.Any,
        properties: OperatorProperties | None = None,
        include: set[str] | None = {"ACTIONZONE", "KEYBOARD", "MOUSE", "NDOF"},
        exclude: set[str] | None = {},
    ) -> KeyMapItem:
        """find_from_operator

        :param idname: Operator Identifier
        :type idname: str | typing.Any
        :param properties:
        :type properties: OperatorProperties | None
        :param include: Include
        :type include: set[str] | None
        :param exclude: Exclude
        :type exclude: set[str] | None
        :return:
        :rtype: KeyMapItem
        """
        ...

    def match_event(self, event: Event | None) -> KeyMapItem:
        """match_event

        :param event:
        :type event: Event | None
        :return:
        :rtype: KeyMapItem
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyMaps(bpy_prop_collection[KeyMap], bpy_struct):
    """Collection of keymaps"""

    def new(
        self,
        name: str | typing.Any,
        space_type: str | None = "EMPTY",
        region_type: str | None = "WINDOW",
        modal: bool | typing.Any | None = False,
        tool: bool | typing.Any | None = False,
    ) -> KeyMap:
        """Ensure the keymap exists. This will return the one with the given name/space type/region type, or create a new one if it does not exist yet.

                :param name: Name
                :type name: str | typing.Any
                :param space_type: Space Type

        EMPTY
        Empty.

        VIEW_3D
        3D Viewport -- Manipulate objects in a 3D environment.

        IMAGE_EDITOR
        UV/Image Editor -- View and edit images and UV Maps.

        NODE_EDITOR
        Node Editor -- Editor for node-based shading and compositing tools.

        SEQUENCE_EDITOR
        Video Sequencer -- Video editing tools.

        CLIP_EDITOR
        Movie Clip Editor -- Motion tracking tools.

        DOPESHEET_EDITOR
        Dope Sheet -- Adjust timing of keyframes.

        GRAPH_EDITOR
        Graph Editor -- Edit drivers and keyframe interpolation.

        NLA_EDITOR
        Nonlinear Animation -- Combine and layer Actions.

        TEXT_EDITOR
        Text Editor -- Edit scripts and in-file documentation.

        CONSOLE
        Python Console -- Interactive programmatic console for advanced editing and script development.

        INFO
        Info -- Log of operations, warnings and error messages.

        TOPBAR
        Top Bar -- Global bar at the top of the screen for global per-window settings.

        STATUSBAR
        Status Bar -- Global bar at the bottom of the screen for general status information.

        OUTLINER
        Outliner -- Overview of scene graph and all available data-blocks.

        PROPERTIES
        Properties -- Edit properties of active object and related data-blocks.

        FILE_BROWSER
        File Browser -- Browse for files and assets.

        SPREADSHEET
        Spreadsheet -- Explore geometry data in a table.

        PREFERENCES
        Preferences -- Edit persistent configuration settings.
                :type space_type: str | None
                :param region_type: Region Type
                :type region_type: str | None
                :param modal: Modal, Keymap for modal operators
                :type modal: bool | typing.Any | None
                :param tool: Tool, Keymap for active tools
                :type tool: bool | typing.Any | None
                :return: Key Map, Added key map
                :rtype: KeyMap
        """
        ...

    def remove(self, keymap: KeyMap):
        """remove

        :param keymap: Key Map, Removed key map
        :type keymap: KeyMap
        """
        ...

    def find(
        self,
        name: str | typing.Any,
        space_type: str | None = "EMPTY",
        region_type: str | None = "WINDOW",
    ) -> KeyMap:
        """find

                :param name: Name
                :type name: str | typing.Any
                :param space_type: Space Type

        EMPTY
        Empty.

        VIEW_3D
        3D Viewport -- Manipulate objects in a 3D environment.

        IMAGE_EDITOR
        UV/Image Editor -- View and edit images and UV Maps.

        NODE_EDITOR
        Node Editor -- Editor for node-based shading and compositing tools.

        SEQUENCE_EDITOR
        Video Sequencer -- Video editing tools.

        CLIP_EDITOR
        Movie Clip Editor -- Motion tracking tools.

        DOPESHEET_EDITOR
        Dope Sheet -- Adjust timing of keyframes.

        GRAPH_EDITOR
        Graph Editor -- Edit drivers and keyframe interpolation.

        NLA_EDITOR
        Nonlinear Animation -- Combine and layer Actions.

        TEXT_EDITOR
        Text Editor -- Edit scripts and in-file documentation.

        CONSOLE
        Python Console -- Interactive programmatic console for advanced editing and script development.

        INFO
        Info -- Log of operations, warnings and error messages.

        TOPBAR
        Top Bar -- Global bar at the top of the screen for global per-window settings.

        STATUSBAR
        Status Bar -- Global bar at the bottom of the screen for general status information.

        OUTLINER
        Outliner -- Overview of scene graph and all available data-blocks.

        PROPERTIES
        Properties -- Edit properties of active object and related data-blocks.

        FILE_BROWSER
        File Browser -- Browse for files and assets.

        SPREADSHEET
        Spreadsheet -- Explore geometry data in a table.

        PREFERENCES
        Preferences -- Edit persistent configuration settings.
                :type space_type: str | None
                :param region_type: Region Type
                :type region_type: str | None
                :return: Key Map, Corresponding key map
                :rtype: KeyMap
        """
        ...

    def find_modal(self, name: str | typing.Any) -> KeyMap:
        """find_modal

        :param name: Operator Name
        :type name: str | typing.Any
        :return: Key Map, Corresponding key map
        :rtype: KeyMap
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Keyframe(bpy_struct):
    """Bezier curve point with two handles defining a Keyframe on an F-Curve"""

    amplitude: float
    """ Amount to boost elastic bounces for 'elastic' easing

    :type: float
    """

    back: float
    """ Amount of overshoot for 'back' easing

    :type: float
    """

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    co_ui: bpy_prop_array[float]
    """ Coordinates of the control point. Note: Changing this value also updates the handles similar to using the graph editor transform operator

    :type: bpy_prop_array[float]
    """

    easing: str
    """ Which ends of the segment between this and the next keyframe easing interpolation is applied to

    :type: str
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the left handle (before the control point)

    :type: bpy_prop_array[float]
    """

    handle_left_type: str
    """ Handle types

    :type: str
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the right handle (after the control point)

    :type: bpy_prop_array[float]
    """

    handle_right_type: str
    """ Handle types

    :type: str
    """

    interpolation: str
    """ Interpolation method to use for segment of the F-Curve from this Keyframe until the next Keyframe

    :type: str
    """

    period: float
    """ Time between bounces for elastic easing

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Left handle selection status

    :type: bool
    """

    select_right_handle: bool
    """ Right handle selection status

    :type: bool
    """

    type: str
    """ Type of keyframe (for visual purposes only)

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyingSet(bpy_struct):
    """Settings that should be keyframed together"""

    bl_description: str
    """ A short description of the keying set

    :type: str
    """

    bl_idname: str
    """ If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is "BUILTIN_KSI_location", and bl_idname is not set by the script, then bl_idname = "BUILTIN_KSI_location")

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    is_path_absolute: bool
    """ Keying Set defines specific paths/settings to be keyframed (i.e. is not reliant on context info)

    :type: bool
    """

    paths: KeyingSetPaths
    """ Keying Set Paths to define settings that get keyframed together

    :type: KeyingSetPaths
    """

    type_info: KeyingSetInfo
    """ Callback function defines for built-in Keying Sets

    :type: KeyingSetInfo
    """

    use_insertkey_needed: bool
    """ Only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_needed: bool
    """ Override default setting to only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_visual: bool
    """ Override default setting to insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_override_xyz_to_rgb: bool
    """ Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis

    :type: bool
    """

    use_insertkey_visual: bool
    """ Insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis

    :type: bool
    """

    def refresh(self):
        """Refresh Keying Set to ensure that it is valid for the current context (call before each use of one)"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyingSetInfo(bpy_struct):
    """Callback function defines for builtin Keying Sets"""

    bl_description: str
    """ A short description of the keying set

    :type: str
    """

    bl_idname: str
    """ If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is "BUILTIN_KSI_location", and bl_idname is not set by the script, then bl_idname = "BUILTIN_KSI_location")

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[str]
    """ Keying Set options to use when inserting keyframes

    :type: set[str]
    """

    def poll(self, context: Context | None) -> bool:
        """Test if Keying Set can be used or not

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """
        ...

    def iterator(self, context: Context | None, ks: KeyingSet | None):
        """Call generate() on the structs which have properties to be keyframed

        :param context:
        :type context: Context | None
        :param ks:
        :type ks: KeyingSet | None
        """
        ...

    def generate(self, context: Context | None, ks: KeyingSet | None, data: typing.Any):
        """Add Paths to the Keying Set to keyframe the properties of the given data

        :param context:
        :type context: Context | None
        :param ks:
        :type ks: KeyingSet | None
        :param data:
        :type data: typing.Any
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyingSetPath(bpy_struct):
    """Path to a setting for use in a Keying Set"""

    array_index: int
    """ Index to the specific setting if applicable

    :type: int
    """

    data_path: str
    """ Path to property setting

    :type: str
    """

    group: str
    """ Name of Action Group to assign setting(s) for this path to

    :type: str
    """

    group_method: str
    """ Method used to define which Group-name to use

    :type: str
    """

    id: ID
    """ ID-Block that keyframes for Keying Set should be added to (for Absolute Keying Sets only)

    :type: ID
    """

    id_type: str
    """ Type of ID-block that can be used

    :type: str
    """

    use_entire_array: bool
    """ When an 'array/vector' type is chosen (Location, Rotation, Color, etc.), entire array is to be used

    :type: bool
    """

    use_insertkey_needed: bool
    """ Only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_needed: bool
    """ Override default setting to only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_visual: bool
    """ Override default setting to insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_override_xyz_to_rgb: bool
    """ Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis

    :type: bool
    """

    use_insertkey_visual: bool
    """ Insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyingSetPaths(bpy_prop_collection[KeyingSetPath], bpy_struct):
    """Collection of keying set paths"""

    active: KeyingSetPath | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSetPath | None
    """

    active_index: int | None
    """ Current Keying Set index

    :type: int | None
    """

    def add(
        self,
        target_id: ID | None,
        data_path: str | typing.Any,
        index: typing.Any | None = -1,
        group_method: str | None = "KEYINGSET",
        group_name: str | typing.Any = "",
    ) -> KeyingSetPath:
        """Add a new path for the Keying Set

        :param target_id: Target ID, ID data-block for the destination
        :type target_id: ID | None
        :param data_path: Data-Path, RNA-Path to destination property
        :type data_path: str | typing.Any
        :param index: Index, The index of the destination property (i.e. axis of Location/Rotation/etc.), or -1 for the entire array
        :type index: typing.Any | None
        :param group_method: Grouping Method, Method used to define which Group-name to use
        :type group_method: str | None
        :param group_name: Group Name, Name of Action Group to assign destination to (only if grouping mode is to use this name)
        :type group_name: str | typing.Any
        :return: New Path, Path created and added to the Keying Set
        :rtype: KeyingSetPath
        """
        ...

    def remove(self, path: KeyingSetPath):
        """Remove the given path from the Keying Set

        :param path: Path
        :type path: KeyingSetPath
        """
        ...

    def clear(self):
        """Remove all the paths from the Keying Set"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyingSets(bpy_prop_collection[KeyingSet], bpy_struct):
    """Scene keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

    def new(
        self,
        idname: str | typing.Any = "KeyingSet",
        name: str | typing.Any = "KeyingSet",
    ) -> KeyingSet:
        """Add a new Keying Set to Scene

        :param idname: IDName, Internal identifier of Keying Set
        :type idname: str | typing.Any
        :param name: Name, User visible name of Keying Set
        :type name: str | typing.Any
        :return: Newly created Keying Set
        :rtype: KeyingSet
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class KeyingSetsAll(bpy_prop_collection[KeyingSet], bpy_struct):
    """All available keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LatticePoint(bpy_struct):
    """Point in the lattice grid"""

    co: bpy_prop_array[float]
    """ Original undeformed location used to calculate the strength of the deform effect (edit/animate the Deformed Location instead)

    :type: bpy_prop_array[float]
    """

    co_deform: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    groups: bpy_prop_collection[VertexGroupElement]
    """ Weights for the vertex groups this point is member of

    :type: bpy_prop_collection[VertexGroupElement]
    """

    select: bool
    """ Selection status

    :type: bool
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LayerCollection(bpy_struct):
    """Layer collection"""

    children: bpy_prop_collection[LayerCollection]
    """ Child layer collections

    :type: bpy_prop_collection[LayerCollection]
    """

    collection: Collection
    """ Collection this layer collection is wrapping

    :type: Collection
    """

    exclude: bool
    """ Exclude from view layer

    :type: bool
    """

    hide_viewport: bool
    """ Temporarily hide in viewport

    :type: bool
    """

    holdout: bool
    """ Mask out objects in collection from view layer

    :type: bool
    """

    indirect_only: bool
    """ Objects in collection only contribute indirectly (through shadows and reflections) in the view layer

    :type: bool
    """

    is_visible: bool
    """ Whether this collection is visible for the view layer, take into account the collection parent

    :type: bool
    """

    name: str
    """ Name of this view layer (same as its collection one)

    :type: str
    """

    def visible_get(self) -> bool:
        """Whether this collection is visible, take into account the collection parent and the viewport

        :return:
        :rtype: bool
        """
        ...

    def has_objects(self) -> bool:
        """

        :return:
        :rtype: bool
        """
        ...

    def has_selected_objects(self, view_layer: ViewLayer | None) -> bool:
        """

        :param view_layer: View layer the layer collection belongs to
        :type view_layer: ViewLayer | None
        :return:
        :rtype: bool
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LayerObjects(bpy_prop_collection[Object], bpy_struct):
    """Collections of objects"""

    active: Object | None
    """ Active object for this layer

    :type: Object | None
    """

    selected: bpy_prop_collection[Object]
    """ All the selected objects of this layer

    :type: bpy_prop_collection[Object]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LibraryWeakReference(bpy_struct):
    """Read-only external reference to a linked data-block and its library file"""

    filepath: str
    """ Path to the library .blend file

    :type: str
    """

    id_name: str
    """ Full ID name in the library .blend file (including the two leading 'id type' chars)

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Lightgroup(bpy_struct):
    name: str
    """ Name of the Lightgroup

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Lightgroups(bpy_prop_collection[Lightgroup], bpy_struct):
    """Collection of Lightgroups"""

    def add(self, name: str | typing.Any = "") -> Lightgroup:
        """add

        :param name: Name, Name of newly created lightgroup
        :type name: str | typing.Any
        :return: Newly created Lightgroup
        :rtype: Lightgroup
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LineStyleAlphaModifiers(bpy_prop_collection[LineStyleAlphaModifier], bpy_struct):
    """Alpha modifiers for changing line alphas"""

    def new(self, name: str | typing.Any, type: str | None) -> LineStyleAlphaModifier:
        """Add a alpha modifier to line style

        :param name: New name for the alpha modifier (not unique)
        :type name: str | typing.Any
        :param type: Alpha modifier type to add
        :type type: str | None
        :return: Newly added alpha modifier
        :rtype: LineStyleAlphaModifier
        """
        ...

    def remove(self, modifier: LineStyleAlphaModifier):
        """Remove a alpha modifier from line style

        :param modifier: Alpha modifier to remove
        :type modifier: LineStyleAlphaModifier
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LineStyleColorModifiers(bpy_prop_collection[LineStyleColorModifier], bpy_struct):
    """Color modifiers for changing line colors"""

    def new(self, name: str | typing.Any, type: str | None) -> LineStyleColorModifier:
        """Add a color modifier to line style

        :param name: New name for the color modifier (not unique)
        :type name: str | typing.Any
        :param type: Color modifier type to add
        :type type: str | None
        :return: Newly added color modifier
        :rtype: LineStyleColorModifier
        """
        ...

    def remove(self, modifier: LineStyleColorModifier):
        """Remove a color modifier from line style

        :param modifier: Color modifier to remove
        :type modifier: LineStyleColorModifier
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LineStyleGeometryModifiers(
    bpy_prop_collection[LineStyleGeometryModifier], bpy_struct
):
    """Geometry modifiers for changing line geometries"""

    def new(
        self, name: str | typing.Any, type: str | None
    ) -> LineStyleGeometryModifier:
        """Add a geometry modifier to line style

        :param name: New name for the geometry modifier (not unique)
        :type name: str | typing.Any
        :param type: Geometry modifier type to add
        :type type: str | None
        :return: Newly added geometry modifier
        :rtype: LineStyleGeometryModifier
        """
        ...

    def remove(self, modifier: LineStyleGeometryModifier):
        """Remove a geometry modifier from line style

        :param modifier: Geometry modifier to remove
        :type modifier: LineStyleGeometryModifier
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LineStyleModifier(bpy_struct):
    """Base type to define modifiers"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LineStyleTextureSlots(bpy_prop_collection[LineStyleTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> LineStyleTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """
        ...

    @classmethod
    def create(cls, index: int | None) -> LineStyleTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """
        ...

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LineStyleThicknessModifiers(
    bpy_prop_collection[LineStyleThicknessModifier], bpy_struct
):
    """Thickness modifiers for changing line thickness"""

    def new(
        self, name: str | typing.Any, type: str | None
    ) -> LineStyleThicknessModifier:
        """Add a thickness modifier to line style

        :param name: New name for the thickness modifier (not unique)
        :type name: str | typing.Any
        :param type: Thickness modifier type to add
        :type type: str | None
        :return: Newly added thickness modifier
        :rtype: LineStyleThicknessModifier
        """
        ...

    def remove(self, modifier: LineStyleThicknessModifier):
        """Remove a thickness modifier from line style

        :param modifier: Thickness modifier to remove
        :type modifier: LineStyleThicknessModifier
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Linesets(bpy_prop_collection[FreestyleLineSet], bpy_struct):
    """Line sets for associating lines and style parameters"""

    active: FreestyleLineSet
    """ Active line set being displayed

    :type: FreestyleLineSet
    """

    active_index: int | None
    """ Index of active line set slot

    :type: int | None
    """

    def new(self, name: str | typing.Any) -> FreestyleLineSet:
        """Add a line set to scene render layer Freestyle settings

        :param name: New name for the line set (not unique)
        :type name: str | typing.Any
        :return: Newly created line set
        :rtype: FreestyleLineSet
        """
        ...

    def remove(self, lineset: FreestyleLineSet):
        """Remove a line set from scene render layer Freestyle settings

        :param lineset: Line set to remove
        :type lineset: FreestyleLineSet
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class LoopColors(bpy_prop_collection[MeshLoopColorLayer], bpy_struct):
    """Collection of vertex colors"""

    active: MeshLoopColorLayer | None
    """ Active vertex color layer

    :type: MeshLoopColorLayer | None
    """

    active_index: int | None
    """ Active vertex color index

    :type: int | None
    """

    def new(
        self, name: str | typing.Any = "Col", do_init: bool | typing.Any | None = True
    ) -> MeshLoopColorLayer:
        """Add a vertex color layer to Mesh

        :param name: Vertex color name
        :type name: str | typing.Any
        :param do_init: Whether new layer's data should be initialized by copying current active one
        :type do_init: bool | typing.Any | None
        :return: The newly created layer
        :rtype: MeshLoopColorLayer
        """
        ...

    def remove(self, layer: MeshLoopColorLayer):
        """Remove a vertex color layer

        :param layer: The layer to remove
        :type layer: MeshLoopColorLayer
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Macro(bpy_struct):
    """Storage of a macro operator being executed, or registered after execution"""

    bl_cursor_pending: str
    """ Cursor to use when waiting for the user to select a location to activate the operator (when bl_options has DEPENDS_ON_CURSOR set)

    :type: str
    """

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[str]
    """ Options for this operator type

    :type: set[str]
    """

    bl_translation_context: str
    """ 

    :type: str
    """

    bl_undo_group: str
    """ 

    :type: str
    """

    has_reports: bool
    """ Operator has a set of reports (warnings and errors) from last execution

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    properties: OperatorProperties
    """ 

    :type: OperatorProperties
    """

    def report(self, type: set[str] | None, message: str | typing.Any):
        """report

        :param type: Type
        :type type: set[str] | None
        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    def draw(self, context: Context):
        """Draw function for the operator

        :param context:
        :type context: Context
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskLayer(bpy_struct):
    """Single layer used for masking pixels"""

    alpha: float
    """ Render Opacity

    :type: float
    """

    blend: str
    """ Method of blending mask layers

    :type: str
    """

    falloff: str
    """ Falloff type the feather

    :type: str
    """

    hide: bool
    """ Restrict visibility in the viewport

    :type: bool
    """

    hide_render: bool
    """ Restrict renderability

    :type: bool
    """

    hide_select: bool
    """ Restrict selection in the viewport

    :type: bool
    """

    invert: bool
    """ Invert the mask black/white

    :type: bool
    """

    name: str
    """ Unique name of layer

    :type: str
    """

    select: bool
    """ Layer is selected for editing in the Dope Sheet

    :type: bool
    """

    splines: MaskSplines
    """ Collection of splines which defines this layer

    :type: MaskSplines
    """

    use_fill_holes: bool
    """ Calculate holes when filling overlapping curves

    :type: bool
    """

    use_fill_overlap: bool
    """ Calculate self intersections and overlap before filling

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskLayers(bpy_prop_collection[MaskLayer], bpy_struct):
    """Collection of layers used by mask"""

    active: MaskLayer | None
    """ Active layer in this mask

    :type: MaskLayer | None
    """

    def new(self, name: str | typing.Any = "") -> MaskLayer:
        """Add layer to this mask

        :param name: Name, Name of new layer
        :type name: str | typing.Any
        :return: New mask layer
        :rtype: MaskLayer
        """
        ...

    def remove(self, layer: MaskLayer):
        """Remove layer from this mask

        :param layer: Shape to be removed
        :type layer: MaskLayer
        """
        ...

    def clear(self):
        """Remove all mask layers"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskParent(bpy_struct):
    """Parenting settings for masking element"""

    id: ID
    """ ID-block to which masking element would be parented to or to its property

    :type: ID
    """

    id_type: str
    """ Type of ID-block that can be used

    :type: str
    """

    parent: str
    """ Name of parent object in specified data-block to which parenting happens

    :type: str
    """

    sub_parent: str
    """ Name of parent sub-object in specified data-block to which parenting happens

    :type: str
    """

    type: str
    """ Parent Type

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskSpline(bpy_struct):
    """Single spline used for defining mask shape"""

    offset_mode: str
    """ The method used for calculating the feather offset

    :type: str
    """

    points: MaskSplinePoints
    """ Collection of points

    :type: MaskSplinePoints
    """

    use_cyclic: bool
    """ Make this spline a closed loop

    :type: bool
    """

    use_fill: bool
    """ Make this spline filled

    :type: bool
    """

    use_self_intersection_check: bool
    """ Prevent feather from self-intersections

    :type: bool
    """

    weight_interpolation: str
    """ The type of weight interpolation for spline

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskSplinePoint(bpy_struct):
    """Single point in spline used for defining mask"""

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    feather_points: bpy_prop_collection[MaskSplinePointUW]
    """ Points defining feather

    :type: bpy_prop_collection[MaskSplinePointUW]
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the first handle

    :type: bpy_prop_array[float]
    """

    handle_left_type: str
    """ Handle type

    :type: str
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the second handle

    :type: bpy_prop_array[float]
    """

    handle_right_type: str
    """ Handle type

    :type: str
    """

    handle_type: str
    """ Handle type

    :type: str
    """

    parent: MaskParent
    """ 

    :type: MaskParent
    """

    select: bool
    """ Selection status

    :type: bool
    """

    weight: float
    """ Weight of the point

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskSplinePointUW(bpy_struct):
    """Single point in spline segment defining feather"""

    select: bool
    """ Selection status

    :type: bool
    """

    u: float
    """ U coordinate of point along spline segment

    :type: float
    """

    weight: float
    """ Weight of feather point

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskSplinePoints(bpy_prop_collection[MaskSplinePoint], bpy_struct):
    """Collection of masking spline points"""

    def add(self, count: int | None):
        """Add a number of point to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """
        ...

    def remove(self, point: MaskSplinePoint):
        """Remove a point from a spline

        :param point: The point to remove
        :type point: MaskSplinePoint
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaskSplines(bpy_prop_collection[MaskSpline], bpy_struct):
    """Collection of masking splines"""

    active: MaskSpline | None
    """ Active spline of masking layer

    :type: MaskSpline | None
    """

    active_point: MaskSplinePoint | None
    """ Active spline of masking layer

    :type: MaskSplinePoint | None
    """

    def new(self) -> MaskSpline:
        """Add a new spline to the layer

        :return: The newly created spline
        :rtype: MaskSpline
        """
        ...

    def remove(self, spline: MaskSpline):
        """Remove a spline from a layer

        :param spline: The spline to remove
        :type spline: MaskSpline
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaterialGPencilStyle(bpy_struct):
    alignment_mode: str
    """ Defines how align Dots and Boxes with drawing path and object rotation

    :type: str
    """

    alignment_rotation: float
    """ Additional rotation applied to dots and square texture of strokes. Only applies in texture shading mode

    :type: float
    """

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    fill_color: bpy_prop_array[float]
    """ Color for filling region bounded by each stroke

    :type: bpy_prop_array[float]
    """

    fill_image: Image
    """ 

    :type: Image
    """

    fill_style: str
    """ Select style used to fill strokes

    :type: str
    """

    flip: bool
    """ Flip filling colors

    :type: bool
    """

    ghost: bool
    """ Display strokes using this color when showing onion skins

    :type: bool
    """

    gradient_type: str
    """ Select type of gradient used to fill strokes

    :type: str
    """

    hide: bool
    """ Set color Visibility

    :type: bool
    """

    is_fill_visible: bool
    """ True when opacity of fill is set high enough to be visible

    :type: bool
    """

    is_stroke_visible: bool
    """ True when opacity of stroke is set high enough to be visible

    :type: bool
    """

    lock: bool
    """ Protect color from further editing and/or frame changes

    :type: bool
    """

    mix_color: bpy_prop_array[float]
    """ Color for mixing with primary filling color

    :type: bpy_prop_array[float]
    """

    mix_factor: float
    """ Mix Factor

    :type: float
    """

    mix_stroke_factor: float
    """ Mix Stroke Factor

    :type: float
    """

    mode: str
    """ Select line type for strokes

    :type: str
    """

    pass_index: int
    """ Index number for the "Color Index" pass

    :type: int
    """

    pixel_size: float
    """ Texture Pixel Size factor along the stroke

    :type: float
    """

    show_fill: bool
    """ Show stroke fills of this material

    :type: bool
    """

    show_stroke: bool
    """ Show stroke lines of this material

    :type: bool
    """

    stroke_image: Image
    """ 

    :type: Image
    """

    stroke_style: str
    """ Select style used to draw strokes

    :type: str
    """

    texture_angle: float
    """ Texture Orientation Angle

    :type: float
    """

    texture_clamp: bool
    """ Do not repeat texture and clamp to one instance only

    :type: bool
    """

    texture_offset: bpy_prop_array[float]
    """ Shift Texture in 2d Space

    :type: bpy_prop_array[float]
    """

    texture_scale: bpy_prop_array[float]
    """ Scale Factor for Texture

    :type: bpy_prop_array[float]
    """

    use_fill_holdout: bool
    """ Remove the color from underneath this stroke by using it as a mask

    :type: bool
    """

    use_overlap_strokes: bool
    """ Disable stencil and overlap self intersections with alpha materials

    :type: bool
    """

    use_stroke_holdout: bool
    """ Remove the color from underneath this stroke by using it as a mask

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaterialLineArt(bpy_struct):
    mat_occlusion: int
    """ Faces with this material will behave as if it has set number of layers in occlusion

    :type: int
    """

    use_material_mask: bool
    """ Use material masks to filter out occluded strokes

    :type: bool
    """

    use_material_mask_bits: list[bool]
    """ 

    :type: list[bool]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MaterialSlot(bpy_struct):
    """Material slot in an object"""

    link: str
    """ Link material to object or the object's data

    :type: str
    """

    material: Material
    """ Material data-block used by this material slot

    :type: Material
    """

    name: str
    """ Material slot name

    :type: str
    """

    slot_index: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Menu(bpy_struct):
    """Editor menu containing buttons"""

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ If this is set, the menu gets a custom ID, otherwise it takes the name of the class used to define the menu (for example, if the class name is "OBJECT_MT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_MT_hello")

    :type: str
    """

    bl_label: str
    """ The menu label

    :type: str
    """

    bl_owner_id: str
    """ 

    :type: str
    """

    bl_translation_context: str | typing.Any
    """ 

    :type: str | typing.Any
    """

    layout: UILayout
    """ Defines the structure of the menu in the UI

    :type: UILayout
    """

    @classmethod
    def poll(cls, context: Context | None) -> bool:
        """If this method returns a non-null output, then the menu can be drawn

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """
        ...

    def draw(self, context: Context | None):
        """Draw UI elements into the menu UI layout

        :param context:
        :type context: Context | None
        """
        ...

    def draw_preset(self, _context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_add_operator (string)
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param _context:
        """
        ...

    def path_menu(
        self,
        searchpaths: list[str] | None,
        operator: str | None,
        *,
        props_default: dict | None = None,
        prop_filepath: str | None = "filepath",
        filter_ext: collections.abc.Callable | None = None,
        filter_path=None,
        display_name: collections.abc.Callable | None = None,
        add_operator=None,
    ):
        """Populate a menu from a list of paths.

                :param searchpaths: Paths to scan.
                :type searchpaths: list[str] | None
                :param operator: The operator id to use with each file.
                :type operator: str | None
                :param props_default: Properties to assign to each operator.
                :type props_default: dict | None
                :param prop_filepath: Optional operator filepath property (defaults to "filepath").
                :type prop_filepath: str | None
                :param filter_ext: Optional callback that takes the file extensions.

        Returning false excludes the file from the list.
                :type filter_ext: collections.abc.Callable | None
                :param filter_path:
                :param display_name: Optional callback that takes the full path, returns the name to display.
                :type display_name: collections.abc.Callable | None
                :param add_operator:
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

    @classmethod
    def append(cls, draw_func):
        """

        :param draw_func:
        """
        ...

    @classmethod
    def prepend(cls, draw_func):
        """

        :param draw_func:
        """
        ...

    @classmethod
    def remove(cls, draw_func):
        """

        :param draw_func:
        """
        ...

class MeshEdge(bpy_struct):
    """Edge in a Mesh data-block"""

    bevel_weight: float
    """ Weight used by the Bevel modifier

    :type: float
    """

    crease: float
    """ Weight used by the Subdivision Surface modifier for creasing

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this edge

    :type: int
    """

    is_loose: bool
    """ Loose edge

    :type: bool
    """

    select: bool
    """ 

    :type: bool
    """

    use_edge_sharp: bool
    """ Sharp edge for the Edge Split modifier

    :type: bool
    """

    use_freestyle_mark: bool
    """ Edge mark for Freestyle line rendering

    :type: bool
    """

    use_seam: bool
    """ Seam edge for UV unwrapping

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    key: typing.Any
    """ (readonly)"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshEdges(bpy_prop_collection[MeshEdge], bpy_struct):
    """Collection of mesh edges"""

    def add(self, count: int | None):
        """add

        :param count: Count, Number of edges to add
        :type count: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshFaceMap(bpy_struct):
    value: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshFaceMapLayer(bpy_struct):
    """Per-face map index"""

    data: bpy_prop_collection[MeshFaceMap]
    """ 

    :type: bpy_prop_collection[MeshFaceMap]
    """

    name: str
    """ Name of face map layer

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshFaceMapLayers(bpy_prop_collection[MeshFaceMapLayer], bpy_struct):
    """Collection of mesh face maps"""

    active: MeshFaceMapLayer
    """ 

    :type: MeshFaceMapLayer
    """

    def new(self, name: str | typing.Any = "Face Map") -> MeshFaceMapLayer:
        """Add a float property layer to Mesh

        :param name: Face map name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshFaceMapLayer
        """
        ...

    def remove(self, layer: MeshFaceMapLayer):
        """Remove a face map layer

        :param layer: The layer to remove
        :type layer: MeshFaceMapLayer
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshLoop(bpy_struct):
    """Loop in a Mesh data-block"""

    bitangent: bpy_prop_array[float]
    """ Bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, use it only if really needed, slower access than bitangent_sign)

    :type: bpy_prop_array[float]
    """

    bitangent_sign: float
    """ Sign of the bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, bitangent = bitangent_sign * cross(normal, tangent))

    :type: float
    """

    edge_index: int
    """ Edge index

    :type: int
    """

    index: int
    """ Index of this loop

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length split normal vector of this vertex for this polygon (must be computed beforehand using calc_normals_split or calc_tangents)

    :type: bpy_prop_array[float]
    """

    tangent: bpy_prop_array[float]
    """ Local space unit length tangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents)

    :type: bpy_prop_array[float]
    """

    vertex_index: int
    """ Vertex index

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshLoopColor(bpy_struct):
    """Vertex loop colors in a Mesh"""

    color: bpy_prop_array[float]
    """ Color in sRGB color space

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshLoopColorLayer(bpy_struct):
    """Layer of vertex colors in a Mesh data-block"""

    active: bool | None
    """ Sets the layer as active for display and editing

    :type: bool | None
    """

    active_render: bool | None
    """ Sets the layer as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshLoopColor]
    """ 

    :type: bpy_prop_collection[MeshLoopColor]
    """

    name: str
    """ Name of Vertex color layer

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshLoopTriangle(bpy_struct):
    """Tessellated triangle in a Mesh data-block"""

    area: float
    """ Area of this triangle

    :type: float
    """

    index: int
    """ Index of this loop triangle

    :type: int
    """

    loops: bpy_prop_array[int]
    """ Indices of mesh loops that make up the triangle

    :type: bpy_prop_array[int]
    """

    material_index: int
    """ Material slot index of this triangle

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length normal vector for this triangle

    :type: bpy_prop_array[float]
    """

    polygon_index: int
    """ Index of mesh polygon that the triangle is a part of

    :type: int
    """

    split_normals: list[list[float]] | tuple[
        tuple[float, float, float],
        tuple[float, float, float],
        tuple[float, float, float],
    ]
    """ Local space unit length split normals vectors of the vertices of this triangle (must be computed beforehand using calc_normals_split or calc_tangents)

    :type: list[list[float]] | tuple[tuple[float, float, float], tuple[float, float, float], tuple[float, float, float]]
    """

    use_smooth: bool
    """ 

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Indices of triangle vertices

    :type: bpy_prop_array[int]
    """

    center: typing.Any
    """ The midpoint of the face.(readonly)"""

    edge_keys: typing.Any
    """ (readonly)"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshLoopTriangles(bpy_prop_collection[MeshLoopTriangle], bpy_struct):
    """Tessellation of mesh polygons into triangles"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshLoops(bpy_prop_collection[MeshLoop], bpy_struct):
    """Collection of mesh loops"""

    def add(self, count: int | None):
        """add

        :param count: Count, Number of loops to add
        :type count: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshNormalValue(bpy_struct):
    """Vector in a mesh normal array"""

    vector: bpy_prop_array[float]
    """ 3D vector

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPaintMaskLayer(bpy_struct):
    """Per-vertex paint mask data"""

    data: bpy_prop_collection[MeshPaintMaskProperty]
    """ 

    :type: bpy_prop_collection[MeshPaintMaskProperty]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPaintMaskProperty(bpy_struct):
    """Floating-point paint mask value"""

    value: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygon(bpy_struct):
    """Polygon in a Mesh data-block"""

    area: float
    """ Read only area of this polygon

    :type: float
    """

    center: bpy_prop_array[float]
    """ Center of this polygon

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this polygon

    :type: int
    """

    loop_start: int
    """ Index of the first loop of this polygon

    :type: int
    """

    loop_total: int
    """ Number of loops used by this polygon

    :type: int
    """

    material_index: int
    """ Material slot index of this polygon

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length normal vector for this polygon

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    use_freestyle_mark: bool
    """ Face mark for Freestyle line rendering

    :type: bool
    """

    use_smooth: bool
    """ 

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    edge_keys: typing.Any
    """ (readonly)"""

    loop_indices: typing.Any
    """ (readonly)"""

    def flip(self):
        """Invert winding of this polygon (flip its normal)"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygonFloatProperty(bpy_struct):
    """User defined floating-point number value in a float properties layer"""

    value: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygonFloatPropertyLayer(bpy_struct):
    """User defined layer of floating-point number values"""

    data: bpy_prop_collection[MeshPolygonFloatProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonFloatProperty]
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygonIntProperty(bpy_struct):
    """User defined integer number value in an integer properties layer"""

    value: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygonIntPropertyLayer(bpy_struct):
    """User defined layer of integer number values"""

    data: bpy_prop_collection[MeshPolygonIntProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonIntProperty]
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygonStringProperty(bpy_struct):
    """User defined string text value in a string properties layer"""

    value: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygonStringPropertyLayer(bpy_struct):
    """User defined layer of string text values"""

    data: bpy_prop_collection[MeshPolygonStringProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonStringProperty]
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshPolygons(bpy_prop_collection[MeshPolygon], bpy_struct):
    """Collection of mesh polygons"""

    active: int | None
    """ The active polygon for this mesh

    :type: int | None
    """

    def add(self, count: int | None):
        """add

        :param count: Count, Number of polygons to add
        :type count: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshSkinVertex(bpy_struct):
    """Per-vertex skin data for use with the Skin modifier"""

    radius: bpy_prop_array[float]
    """ Radius of the skin

    :type: bpy_prop_array[float]
    """

    use_loose: bool
    """ If vertex has multiple adjacent edges, it is hulled to them directly

    :type: bool
    """

    use_root: bool
    """ Vertex is a root for rotation calculations and armature generation, setting this flag does not clear other roots in the same mesh island

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshSkinVertexLayer(bpy_struct):
    """Per-vertex skin data for use with the Skin modifier"""

    data: bpy_prop_collection[MeshSkinVertex]
    """ 

    :type: bpy_prop_collection[MeshSkinVertex]
    """

    name: str
    """ Name of skin layer

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshStatVis(bpy_struct):
    distort_max: float
    """ Maximum angle to display

    :type: float
    """

    distort_min: float
    """ Minimum angle to display

    :type: float
    """

    overhang_axis: str
    """ 

    :type: str
    """

    overhang_max: float
    """ Maximum angle to display

    :type: float
    """

    overhang_min: float
    """ Minimum angle to display

    :type: float
    """

    sharp_max: float
    """ Maximum angle to display

    :type: float
    """

    sharp_min: float
    """ Minimum angle to display

    :type: float
    """

    thickness_max: float
    """ Maximum for measuring thickness

    :type: float
    """

    thickness_min: float
    """ Minimum for measuring thickness

    :type: float
    """

    thickness_samples: int
    """ Number of samples to test per face

    :type: int
    """

    type: str
    """ Type of data to visualize/check

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshUVLoop(bpy_struct):
    pin_uv: bool
    """ 

    :type: bool
    """

    select: bool
    """ 

    :type: bool
    """

    select_edge: bool
    """ 

    :type: bool
    """

    uv: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshUVLoopLayer(bpy_struct):
    active: bool | None
    """ Set the map as active for display and editing

    :type: bool | None
    """

    active_clone: bool | None
    """ Set the map as active for cloning

    :type: bool | None
    """

    active_render: bool | None
    """ Set the UV map as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshUVLoop]
    """ 

    :type: bpy_prop_collection[MeshUVLoop]
    """

    name: str
    """ Name of UV map

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertColor(bpy_struct):
    """Vertex colors in a Mesh"""

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertColorLayer(bpy_struct):
    """Layer of sculpt vertex colors in a Mesh data-block"""

    active: bool | None
    """ Sets the sculpt vertex color layer as active for display and editing

    :type: bool | None
    """

    active_render: bool | None
    """ Sets the sculpt vertex color layer as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshVertColor]
    """ 

    :type: bpy_prop_collection[MeshVertColor]
    """

    name: str
    """ Name of Sculpt Vertex color layer

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertex(bpy_struct):
    """Vertex in a Mesh data-block"""

    bevel_weight: float
    """ Weight used by the Bevel modifier 'Only Vertices' option

    :type: float
    """

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    groups: bpy_prop_collection[VertexGroupElement]
    """ Weights for the vertex groups this vertex is member of

    :type: bpy_prop_collection[VertexGroupElement]
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this vertex

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Vertex Normal

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    undeformed_co: bpy_prop_array[float]
    """ For meshes with modifiers applied, the coordinate of the vertex with no deforming modifiers applied, as used for generated texture coordinates

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexCrease(bpy_struct):
    value: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexCreaseLayer(bpy_struct):
    """Per-vertex crease"""

    data: bpy_prop_collection[MeshVertexCrease]
    """ 

    :type: bpy_prop_collection[MeshVertexCrease]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexFloatProperty(bpy_struct):
    """User defined floating-point number value in a float properties layer"""

    value: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexFloatPropertyLayer(bpy_struct):
    """User defined layer of floating-point number values"""

    data: bpy_prop_collection[MeshVertexFloatProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexFloatProperty]
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexIntProperty(bpy_struct):
    """User defined integer number value in an integer properties layer"""

    value: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexIntPropertyLayer(bpy_struct):
    """User defined layer of integer number values"""

    data: bpy_prop_collection[MeshVertexIntProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexIntProperty]
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexStringProperty(bpy_struct):
    """User defined string text value in a string properties layer"""

    value: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertexStringPropertyLayer(bpy_struct):
    """User defined layer of string text values"""

    data: bpy_prop_collection[MeshVertexStringProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexStringProperty]
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MeshVertices(bpy_prop_collection[MeshVertex], bpy_struct):
    """Collection of mesh vertices"""

    def add(self, count: int | None):
        """add

        :param count: Count, Number of vertices to add
        :type count: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MetaBallElements(bpy_prop_collection[MetaElement], bpy_struct):
    """Collection of metaball elements"""

    active: MetaElement
    """ Last selected element

    :type: MetaElement
    """

    def new(self, type: str | None = "BALL") -> MetaElement:
        """Add a new element to the metaball

        :param type: Type for the new metaball element
        :type type: str | None
        :return: The newly created metaball element
        :rtype: MetaElement
        """
        ...

    def remove(self, element: MetaElement):
        """Remove an element from the metaball

        :param element: The element to remove
        :type element: MetaElement
        """
        ...

    def clear(self):
        """Remove all elements from the metaball"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MetaElement(bpy_struct):
    """Blobby element in a metaball data-block"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ Hide element

    :type: bool
    """

    radius: float
    """ 

    :type: float
    """

    rotation: bpy_prop_array[float]
    """ Normalized quaternion rotation

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Select element

    :type: bool
    """

    size_x: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    size_y: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    size_z: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    stiffness: float
    """ Stiffness defines how much of the element to fill

    :type: float
    """

    type: str
    """ Metaball types

    :type: str
    """

    use_negative: bool
    """ Set metaball as negative one

    :type: bool
    """

    use_scale_stiffness: bool
    """ Scale stiffness instead of radius

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Modifier(bpy_struct):
    """Modifier affecting the geometry data of an object"""

    is_active: bool
    """ The active modifier in the list

    :type: bool
    """

    is_override_data: bool
    """ In a local override object, whether this modifier comes from the linked reference object, or is local to the override

    :type: bool
    """

    name: str
    """ Modifier name

    :type: str
    """

    show_expanded: bool
    """ Set modifier expanded in the user interface

    :type: bool
    """

    show_in_editmode: bool
    """ Display modifier in Edit mode

    :type: bool
    """

    show_on_cage: bool
    """ Adjust edit cage to modifier result

    :type: bool
    """

    show_render: bool
    """ Use modifier during render

    :type: bool
    """

    show_viewport: bool
    """ Display modifier in viewport

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

    use_apply_on_spline: bool
    """ Apply this and all preceding deformation modifiers on splines' points rather than on filled curve/surface

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MotionPath(bpy_struct):
    """Cache of the world-space positions of an element over a frame range"""

    color: bpy_prop_array[float]
    """ Custom color for motion path

    :type: bpy_prop_array[float]
    """

    frame_end: int
    """ End frame of the stored range

    :type: int
    """

    frame_start: int
    """ Starting frame of the stored range

    :type: int
    """

    is_modified: bool
    """ Path is being edited

    :type: bool
    """

    length: int
    """ Number of frames cached

    :type: int
    """

    line_thickness: int
    """ Line thickness for motion path

    :type: int
    """

    lines: bool
    """ Use straight lines between keyframe points

    :type: bool
    """

    points: bpy_prop_collection[MotionPathVert]
    """ Cached positions per frame

    :type: bpy_prop_collection[MotionPathVert]
    """

    use_bone_head: bool
    """ For PoseBone paths, use the bone head location when calculating this path

    :type: bool
    """

    use_custom_color: bool
    """ Use custom color for this motion path

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MotionPathVert(bpy_struct):
    """Cached location on path"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Path point is selected for editing

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieClipProxy(bpy_struct):
    """Proxy parameters for a movie clip"""

    build_100: bool
    """ Build proxy resolution 100% of the original footage dimension

    :type: bool
    """

    build_25: bool
    """ Build proxy resolution 25% of the original footage dimension

    :type: bool
    """

    build_50: bool
    """ Build proxy resolution 50% of the original footage dimension

    :type: bool
    """

    build_75: bool
    """ Build proxy resolution 75% of the original footage dimension

    :type: bool
    """

    build_free_run: bool
    """ Build free run time code index

    :type: bool
    """

    build_free_run_rec_date: bool
    """ Build free run time code index using Record Date/Time

    :type: bool
    """

    build_record_run: bool
    """ Build record run time code index

    :type: bool
    """

    build_undistorted_100: bool
    """ Build proxy resolution 100% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_25: bool
    """ Build proxy resolution 25% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_50: bool
    """ Build proxy resolution 50% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_75: bool
    """ Build proxy resolution 75% of the original undistorted footage dimension

    :type: bool
    """

    directory: str
    """ Location to store the proxy files

    :type: str
    """

    quality: int
    """ JPEG quality of proxy images

    :type: int
    """

    timecode: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieClipScopes(bpy_struct):
    """Scopes for statistical view of a movie clip"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieClipUser(bpy_struct):
    """Parameters defining how a MovieClip data-block is used by another data-block"""

    frame_current: int
    """ Current frame number in movie or image sequence

    :type: int
    """

    proxy_render_size: str
    """ Display preview using full resolution or different proxy resolutions

    :type: str
    """

    use_render_undistorted: bool
    """ Render preview using undistorted proxy

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieReconstructedCamera(bpy_struct):
    """Match-moving reconstructed camera data from tracker"""

    average_error: float
    """ Average error of reconstruction

    :type: float
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    matrix: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Worldspace transformation matrix

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTracking(bpy_struct):
    """Match-moving data for tracking"""

    active_object_index: int | None
    """ Index of active object

    :type: int | None
    """

    camera: MovieTrackingCamera
    """ 

    :type: MovieTrackingCamera
    """

    dopesheet: MovieTrackingDopesheet
    """ 

    :type: MovieTrackingDopesheet
    """

    objects: MovieTrackingObjects
    """ Collection of objects in this tracking data object

    :type: MovieTrackingObjects
    """

    plane_tracks: MovieTrackingPlaneTracks
    """ Collection of plane tracks in this tracking data object

    :type: MovieTrackingPlaneTracks
    """

    reconstruction: MovieTrackingReconstruction
    """ 

    :type: MovieTrackingReconstruction
    """

    settings: MovieTrackingSettings
    """ 

    :type: MovieTrackingSettings
    """

    stabilization: MovieTrackingStabilization
    """ 

    :type: MovieTrackingStabilization
    """

    tracks: MovieTrackingTracks
    """ Collection of tracks in this tracking data object

    :type: MovieTrackingTracks
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingCamera(bpy_struct):
    """Match-moving camera data for tracking"""

    brown_k1: float
    """ First coefficient of fourth order Brown-Conrady radial distortion

    :type: float
    """

    brown_k2: float
    """ Second coefficient of fourth order Brown-Conrady radial distortion

    :type: float
    """

    brown_k3: float
    """ Third coefficient of fourth order Brown-Conrady radial distortion

    :type: float
    """

    brown_k4: float
    """ Fourth coefficient of fourth order Brown-Conrady radial distortion

    :type: float
    """

    brown_p1: float
    """ First coefficient of second order Brown-Conrady tangential distortion

    :type: float
    """

    brown_p2: float
    """ Second coefficient of second order Brown-Conrady tangential distortion

    :type: float
    """

    distortion_model: str
    """ Distortion model used for camera lenses

    :type: str
    """

    division_k1: float
    """ First coefficient of second order division distortion

    :type: float
    """

    division_k2: float
    """ Second coefficient of second order division distortion

    :type: float
    """

    focal_length: float
    """ Camera's focal length

    :type: float
    """

    focal_length_pixels: float
    """ Camera's focal length

    :type: float
    """

    k1: float
    """ First coefficient of third order polynomial radial distortion

    :type: float
    """

    k2: float
    """ Second coefficient of third order polynomial radial distortion

    :type: float
    """

    k3: float
    """ Third coefficient of third order polynomial radial distortion

    :type: float
    """

    nuke_k1: float
    """ First coefficient of second order Nuke distortion

    :type: float
    """

    nuke_k2: float
    """ Second coefficient of second order Nuke distortion

    :type: float
    """

    pixel_aspect: float
    """ Pixel aspect ratio

    :type: float
    """

    principal: bpy_prop_array[float]
    """ Optical center of lens

    :type: bpy_prop_array[float]
    """

    sensor_width: float
    """ Width of CCD sensor in millimeters

    :type: float
    """

    units: str
    """ Units used for camera focal length

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingDopesheet(bpy_struct):
    """Match-moving dopesheet data"""

    show_hidden: bool
    """ Include channels from objects/bone that aren't visible

    :type: bool
    """

    show_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    sort_method: str
    """ Method to be used to sort channels in dopesheet view

    :type: str
    """

    use_invert_sort: bool
    """ Invert sort order of dopesheet channels

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingMarker(bpy_struct):
    """Match-moving marker data for tracking"""

    co: bpy_prop_array[float]
    """ Marker position at frame in normalized coordinates

    :type: bpy_prop_array[float]
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    is_keyed: bool
    """ Whether the position of the marker is keyframed or tracked

    :type: bool
    """

    mute: bool
    """ Is marker muted for current frame

    :type: bool
    """

    pattern_bound_box: list[list[float]] | tuple[
        tuple[float, float], tuple[float, float]
    ]
    """ Pattern area bounding box in normalized coordinates

    :type: list[list[float]] | tuple[tuple[float, float], tuple[float, float]]
    """

    pattern_corners: list[list[float]] | tuple[
        tuple[float, float, float, float], tuple[float, float, float, float]
    ]
    """ Array of coordinates which represents pattern's corners in normalized coordinates relative to marker position

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    search_max: bpy_prop_array[float]
    """ Right-bottom corner of search area in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

    search_min: bpy_prop_array[float]
    """ Left-bottom corner of search area in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingMarkers(bpy_prop_collection[MovieTrackingMarker], bpy_struct):
    """Collection of markers for movie tracking track"""

    def find_frame(
        self, frame: int | None, exact: bool | typing.Any | None = True
    ) -> MovieTrackingMarker:
        """Get marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get marker at exact frame number rather than get estimated marker
        :type exact: bool | typing.Any | None
        :return: Marker for specified frame
        :rtype: MovieTrackingMarker
        """
        ...

    def insert_frame(
        self, frame: int | None, co: typing.Any | None = (0.0, 0.0)
    ) -> MovieTrackingMarker:
        """Insert a new marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :param co: Coordinate, Place new marker at the given frame using specified in normalized space coordinates
        :type co: typing.Any | None
        :return: Newly created marker
        :rtype: MovieTrackingMarker
        """
        ...

    def delete_frame(self, frame: int | None):
        """Delete marker at specified frame

        :param frame: Frame, Frame number to delete marker from
        :type frame: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingObject(bpy_struct):
    """Match-moving object tracking and reconstruction data"""

    is_camera: bool
    """ Object is used for camera tracking

    :type: bool
    """

    keyframe_a: int
    """ First keyframe used for reconstruction initialization

    :type: int
    """

    keyframe_b: int
    """ Second keyframe used for reconstruction initialization

    :type: int
    """

    name: str
    """ Unique name of object

    :type: str
    """

    plane_tracks: MovieTrackingObjectPlaneTracks
    """ Collection of plane tracks in this tracking data object

    :type: MovieTrackingObjectPlaneTracks
    """

    reconstruction: MovieTrackingReconstruction
    """ 

    :type: MovieTrackingReconstruction
    """

    scale: float
    """ Scale of object solution in camera space

    :type: float
    """

    tracks: MovieTrackingObjectTracks
    """ Collection of tracks in this tracking data object

    :type: MovieTrackingObjectTracks
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingObjectPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of tracking plane tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingObjectTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    def new(
        self, name: str | typing.Any = "", frame: typing.Any | None = 1
    ) -> MovieTrackingTrack:
        """create new motion track in this movie clip

        :param name: Name of new track
        :type name: str | typing.Any
        :param frame: Frame, Frame number to add tracks on
        :type frame: typing.Any | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingObjects(bpy_prop_collection[MovieTrackingObject], bpy_struct):
    """Collection of movie tracking objects"""

    active: MovieTrackingObject | None
    """ Active object in this tracking data object

    :type: MovieTrackingObject | None
    """

    def new(self, name: str | typing.Any) -> MovieTrackingObject:
        """Add tracking object to this movie clip

        :param name: Name of new object
        :type name: str | typing.Any
        :return: New motion tracking object
        :rtype: MovieTrackingObject
        """
        ...

    def remove(self, object: MovieTrackingObject):
        """Remove tracking object from this movie clip

        :param object: Motion tracking object to be removed
        :type object: MovieTrackingObject
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingPlaneMarker(bpy_struct):
    """Match-moving plane marker data for tracking"""

    corners: list[list[float]] | tuple[
        tuple[float, float, float, float], tuple[float, float, float, float]
    ]
    """ Array of coordinates which represents UI rectangle corners in frame normalized coordinates

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    mute: bool
    """ Is marker muted for current frame

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingPlaneMarkers(
    bpy_prop_collection[MovieTrackingPlaneMarker], bpy_struct
):
    """Collection of markers for movie tracking plane track"""

    def find_frame(
        self, frame: int | None, exact: bool | typing.Any | None = True
    ) -> MovieTrackingPlaneMarker:
        """Get plane marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get plane marker at exact frame number rather than get estimated marker
        :type exact: bool | typing.Any | None
        :return: Plane marker for specified frame
        :rtype: MovieTrackingPlaneMarker
        """
        ...

    def insert_frame(self, frame: int | None) -> MovieTrackingPlaneMarker:
        """Insert a new plane marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :return: Newly created plane marker
        :rtype: MovieTrackingPlaneMarker
        """
        ...

    def delete_frame(self, frame: int | None):
        """Delete plane marker at specified frame

        :param frame: Frame, Frame number to delete plane marker from
        :type frame: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingPlaneTrack(bpy_struct):
    """Match-moving plane track data for tracking"""

    image: Image
    """ Image displayed in the track during editing in clip editor

    :type: Image
    """

    image_opacity: float
    """ Opacity of the image

    :type: float
    """

    markers: MovieTrackingPlaneMarkers
    """ Collection of markers in track

    :type: MovieTrackingPlaneMarkers
    """

    name: str
    """ Unique name of track

    :type: str
    """

    select: bool
    """ Plane track is selected

    :type: bool
    """

    use_auto_keying: bool
    """ Automatic keyframe insertion when moving plane corners

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of movie tracking plane tracks"""

    active: MovieTrackingPlaneTrack | None
    """ Active plane track in this tracking data object

    :type: MovieTrackingPlaneTrack | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingReconstructedCameras(
    bpy_prop_collection[MovieReconstructedCamera], bpy_struct
):
    """Collection of solved cameras"""

    def find_frame(self, frame: typing.Any | None = 1) -> MovieReconstructedCamera:
        """Find a reconstructed camera for a give frame number

        :param frame: Frame, Frame number to find camera for
        :type frame: typing.Any | None
        :return: Camera for a given frame
        :rtype: MovieReconstructedCamera
        """
        ...

    def matrix_from_frame(
        self, frame: typing.Any | None = 1
    ) -> (
        list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
    ):
        """Return interpolated camera matrix for a given frame

        :param frame: Frame, Frame number to find camera for
        :type frame: typing.Any | None
        :return: Matrix, Interpolated camera matrix for a given frame
        :rtype: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingReconstruction(bpy_struct):
    """Match-moving reconstruction data from tracker"""

    average_error: float
    """ Average error of reconstruction

    :type: float
    """

    cameras: MovieTrackingReconstructedCameras
    """ Collection of solved cameras

    :type: MovieTrackingReconstructedCameras
    """

    is_valid: bool
    """ Is tracking data contains valid reconstruction information

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingSettings(bpy_struct):
    """Match moving settings"""

    clean_action: str
    """ Cleanup action to execute

    :type: str
    """

    clean_error: float
    """ Effect on tracks which have a larger re-projection error

    :type: float
    """

    clean_frames: int
    """ Effect on tracks which are tracked less than the specified amount of frames

    :type: int
    """

    default_correlation_min: float
    """ Default minimum value of correlation between matched pattern and reference that is still treated as successful tracking

    :type: float
    """

    default_frames_limit: int
    """ Every tracking cycle, this number of frames are tracked

    :type: int
    """

    default_margin: int
    """ Default distance from image boundary at which marker stops tracking

    :type: int
    """

    default_motion_model: str
    """ Default motion model to use for tracking

    :type: str
    """

    default_pattern_match: str
    """ Track pattern from given frame when tracking marker to next frame

    :type: str
    """

    default_pattern_size: int
    """ Size of pattern area for newly created tracks

    :type: int
    """

    default_search_size: int
    """ Size of search area for newly created tracks

    :type: int
    """

    default_weight: float
    """ Influence of newly created track on a final solution

    :type: float
    """

    distance: float
    """ Distance between two bundles used for scene scaling

    :type: float
    """

    object_distance: float
    """ Distance between two bundles used for object scaling

    :type: float
    """

    refine_intrinsics_focal_length: bool
    """ Refine focal length during camera solving

    :type: bool
    """

    refine_intrinsics_principal_point: bool
    """ Refine principal point during camera solving

    :type: bool
    """

    refine_intrinsics_radial_distortion: bool
    """ Refine radial coefficients of distortion model during camera solving

    :type: bool
    """

    refine_intrinsics_tangential_distortion: bool
    """ Refine tangential coefficients of distortion model during camera solving

    :type: bool
    """

    speed: str
    """ Limit speed of tracking to make visual feedback easier (this does not affect the tracking quality)

    :type: str
    """

    use_default_blue_channel: bool
    """ Use blue channel from footage for tracking

    :type: bool
    """

    use_default_brute: bool
    """ Use a brute-force translation-only initialization when tracking

    :type: bool
    """

    use_default_green_channel: bool
    """ Use green channel from footage for tracking

    :type: bool
    """

    use_default_mask: bool
    """ Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking

    :type: bool
    """

    use_default_normalization: bool
    """ Normalize light intensities while tracking (slower)

    :type: bool
    """

    use_default_red_channel: bool
    """ Use red channel from footage for tracking

    :type: bool
    """

    use_keyframe_selection: bool
    """ Automatically select keyframes when solving camera/object motion

    :type: bool
    """

    use_tripod_solver: bool
    """ Use special solver to track a stable camera position, such as a tripod

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingStabilization(bpy_struct):
    """2D stabilization based on tracking markers"""

    active_rotation_track_index: int | None
    """ Index of active track in rotation stabilization tracks list

    :type: int | None
    """

    active_track_index: int | None
    """ Index of active track in translation stabilization tracks list

    :type: int | None
    """

    anchor_frame: int
    """ Reference point to anchor stabilization (other frames will be adjusted relative to this frame's position)

    :type: int
    """

    filter_type: str
    """ Interpolation to use for sub-pixel shifts and rotations due to stabilization

    :type: str
    """

    influence_location: float
    """ Influence of stabilization algorithm on footage location

    :type: float
    """

    influence_rotation: float
    """ Influence of stabilization algorithm on footage rotation

    :type: float
    """

    influence_scale: float
    """ Influence of stabilization algorithm on footage scale

    :type: float
    """

    rotation_tracks: bpy_prop_collection[MovieTrackingTrack]
    """ Collection of tracks used for 2D stabilization (translation)

    :type: bpy_prop_collection[MovieTrackingTrack]
    """

    scale_max: float
    """ Limit the amount of automatic scaling

    :type: float
    """

    show_tracks_expanded: bool
    """ Show UI list of tracks participating in stabilization

    :type: bool
    """

    target_position: bpy_prop_array[float]
    """ Known relative offset of original shot, will be subtracted (e.g. for panning shot, can be animated)

    :type: bpy_prop_array[float]
    """

    target_rotation: float
    """ Rotation present on original shot, will be compensated (e.g. for deliberate tilting)

    :type: float
    """

    target_scale: float
    """ Explicitly scale resulting frame to compensate zoom of original shot

    :type: float
    """

    tracks: bpy_prop_collection[MovieTrackingTrack]
    """ Collection of tracks used for 2D stabilization (translation)

    :type: bpy_prop_collection[MovieTrackingTrack]
    """

    use_2d_stabilization: bool
    """ Use 2D stabilization for footage

    :type: bool
    """

    use_autoscale: bool
    """ Automatically scale footage to cover unfilled areas when stabilizing

    :type: bool
    """

    use_stabilize_rotation: bool
    """ Stabilize detected rotation around center of frame

    :type: bool
    """

    use_stabilize_scale: bool
    """ Compensate any scale changes relative to center of rotation

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingTrack(bpy_struct):
    """Match-moving track data for tracking"""

    average_error: float
    """ Average error of re-projection

    :type: float
    """

    bundle: bpy_prop_array[float]
    """ Position of bundle reconstructed from this track

    :type: bpy_prop_array[float]
    """

    color: bpy_prop_array[float]
    """ Color of the track in the Movie Clip Editor and the 3D viewport after a solve

    :type: bpy_prop_array[float]
    """

    correlation_min: float
    """ Minimal value of correlation between matched pattern and reference that is still treated as successful tracking

    :type: float
    """

    frames_limit: int
    """ Every tracking cycle, this number of frames are tracked

    :type: int
    """

    grease_pencil: GreasePencil
    """ Grease pencil data for this track

    :type: GreasePencil
    """

    has_bundle: bool
    """ True if track has a valid bundle

    :type: bool
    """

    hide: bool
    """ Track is hidden

    :type: bool
    """

    lock: bool
    """ Track is locked and all changes to it are disabled

    :type: bool
    """

    margin: int
    """ Distance from image boundary at which marker stops tracking

    :type: int
    """

    markers: MovieTrackingMarkers
    """ Collection of markers in track

    :type: MovieTrackingMarkers
    """

    motion_model: str
    """ Default motion model to use for tracking

    :type: str
    """

    name: str
    """ Unique name of track

    :type: str
    """

    offset: bpy_prop_array[float]
    """ Offset of track from the parenting point

    :type: bpy_prop_array[float]
    """

    pattern_match: str
    """ Track pattern from given frame when tracking marker to next frame

    :type: str
    """

    select: bool
    """ Track is selected

    :type: bool
    """

    select_anchor: bool
    """ Track's anchor point is selected

    :type: bool
    """

    select_pattern: bool
    """ Track's pattern area is selected

    :type: bool
    """

    select_search: bool
    """ Track's search area is selected

    :type: bool
    """

    use_alpha_preview: bool
    """ Apply track's mask on displaying preview

    :type: bool
    """

    use_blue_channel: bool
    """ Use blue channel from footage for tracking

    :type: bool
    """

    use_brute: bool
    """ Use a brute-force translation only pre-track before refinement

    :type: bool
    """

    use_custom_color: bool
    """ Use custom color instead of theme-defined

    :type: bool
    """

    use_grayscale_preview: bool
    """ Display what the tracking algorithm sees in the preview

    :type: bool
    """

    use_green_channel: bool
    """ Use green channel from footage for tracking

    :type: bool
    """

    use_mask: bool
    """ Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking

    :type: bool
    """

    use_normalization: bool
    """ Normalize light intensities while tracking. Slower

    :type: bool
    """

    use_red_channel: bool
    """ Use red channel from footage for tracking

    :type: bool
    """

    weight: float
    """ Influence of this track on a final solution

    :type: float
    """

    weight_stab: float
    """ Influence of this track on 2D stabilization

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class MovieTrackingTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    def new(
        self, name: str | typing.Any = "", frame: typing.Any | None = 1
    ) -> MovieTrackingTrack:
        """Create new motion track in this movie clip

        :param name: Name of new track
        :type name: str | typing.Any
        :param frame: Frame, Frame number to add track on
        :type frame: typing.Any | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NlaStrip(bpy_struct):
    """A container referencing an existing Action"""

    action: Action
    """ Action referenced by this strip

    :type: Action
    """

    action_frame_end: float
    """ Last frame from action to use

    :type: float
    """

    action_frame_start: float
    """ First frame from action to use

    :type: float
    """

    active: bool
    """ NLA Strip is active

    :type: bool
    """

    blend_in: float
    """ Number of frames at start of strip to fade in influence

    :type: float
    """

    blend_out: float
    """ 

    :type: float
    """

    blend_type: str
    """ Method used for combining strip's result with accumulated result

    :type: str
    """

    extrapolation: str
    """ Action to take for gaps past the strip extents

    :type: str
    """

    fcurves: NlaStripFCurves
    """ F-Curves for controlling the strip's influence and timing

    :type: NlaStripFCurves
    """

    frame_end: float
    """ 

    :type: float
    """

    frame_start: float
    """ 

    :type: float
    """

    influence: float
    """ Amount the strip contributes to the current result

    :type: float
    """

    modifiers: bpy_prop_collection[FModifier]
    """ Modifiers affecting all the F-Curves in the referenced Action

    :type: bpy_prop_collection[FModifier]
    """

    mute: bool
    """ Disable NLA Strip evaluation

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    repeat: float
    """ Number of times to repeat the action range

    :type: float
    """

    scale: float
    """ Scaling factor for action

    :type: float
    """

    select: bool
    """ NLA Strip is selected

    :type: bool
    """

    strip_time: float
    """ Frame of referenced Action to evaluate

    :type: float
    """

    strips: bpy_prop_collection[NlaStrip]
    """ NLA Strips that this strip acts as a container for (if it is of type Meta)

    :type: bpy_prop_collection[NlaStrip]
    """

    type: str
    """ Type of NLA Strip

    :type: str
    """

    use_animated_influence: bool
    """ Influence setting is controlled by an F-Curve rather than automatically determined

    :type: bool
    """

    use_animated_time: bool
    """ Strip time is controlled by an F-Curve rather than automatically determined

    :type: bool
    """

    use_animated_time_cyclic: bool
    """ Cycle the animated time within the action start and end

    :type: bool
    """

    use_auto_blend: bool
    """ Number of frames for Blending In/Out is automatically determined from overlapping strips

    :type: bool
    """

    use_reverse: bool
    """ NLA Strip is played back in reverse order (only when timing is automatically determined)

    :type: bool
    """

    use_sync_length: bool
    """ Update range of frames referenced from action after tweaking strip and its keyframes

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NlaStripFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of NLA strip F-Curves"""

    def find(self, data_path: str | typing.Any, index: typing.Any | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the NLA strip.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NlaStrips(bpy_prop_collection[NlaStrip], bpy_struct):
    """Collection of Nla Strips"""

    def new(
        self, name: str | typing.Any, start: int | None, action: Action
    ) -> NlaStrip:
        """Add a new Action-Clip strip to the track

        :param name: Name for the NLA Strips
        :type name: str | typing.Any
        :param start: Start Frame, Start frame for this strip
        :type start: int | None
        :param action: Action to assign to this strip
        :type action: Action
        :return: New NLA Strip
        :rtype: NlaStrip
        """
        ...

    def remove(self, strip: NlaStrip):
        """Remove a NLA Strip

        :param strip: NLA Strip to remove
        :type strip: NlaStrip
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NlaTrack(bpy_struct):
    """A animation layer containing Actions referenced as NLA strips"""

    active: bool
    """ NLA Track is active

    :type: bool
    """

    is_override_data: bool
    """ In a local override data, whether this NLA track comes from the linked reference data, or is local to the override

    :type: bool
    """

    is_solo: bool
    """ NLA Track is evaluated itself (i.e. active Action and all other NLA Tracks in the same AnimData block are disabled)

    :type: bool
    """

    lock: bool
    """ NLA Track is locked

    :type: bool
    """

    mute: bool
    """ Disable NLA Track evaluation

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ NLA Track is selected

    :type: bool
    """

    strips: NlaStrips
    """ NLA Strips on this NLA-track

    :type: NlaStrips
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NlaTracks(bpy_prop_collection[NlaTrack], bpy_struct):
    """Collection of NLA Tracks"""

    active: NlaTrack | None
    """ Active NLA Track

    :type: NlaTrack | None
    """

    def new(self, prev: NlaTrack | None = None) -> NlaTrack:
        """Add a new NLA Track

        :param prev: NLA Track to add the new one after
        :type prev: NlaTrack | None
        :return: New NLA Track
        :rtype: NlaTrack
        """
        ...

    def remove(self, track: NlaTrack):
        """Remove a NLA Track

        :param track: NLA Track to remove
        :type track: NlaTrack
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Node(bpy_struct):
    """Node in a node tree"""

    bl_description: str
    """ 

    :type: str
    """

    bl_height_default: float
    """ 

    :type: float
    """

    bl_height_max: float
    """ 

    :type: float
    """

    bl_height_min: float
    """ 

    :type: float
    """

    bl_icon: str
    """ The node icon

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ The node label

    :type: str
    """

    bl_static_type: str
    """ Node type (deprecated, use with care)

    :type: str
    """

    bl_width_default: float
    """ 

    :type: float
    """

    bl_width_max: float
    """ 

    :type: float
    """

    bl_width_min: float
    """ 

    :type: float
    """

    color: bpy_prop_array[float]
    """ Custom color of the node body

    :type: bpy_prop_array[float]
    """

    dimensions: bpy_prop_array[float]
    """ Absolute bounding box dimensions of the node

    :type: bpy_prop_array[float]
    """

    height: float
    """ Height of the node

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    inputs: NodeInputs
    """ 

    :type: NodeInputs
    """

    internal_links: bpy_prop_collection[NodeLink]
    """ Internal input-to-output connections for muting

    :type: bpy_prop_collection[NodeLink]
    """

    label: str
    """ Optional custom node label

    :type: str
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    mute: bool
    """ 

    :type: bool
    """

    name: str
    """ Unique node identifier

    :type: str
    """

    outputs: NodeOutputs
    """ 

    :type: NodeOutputs
    """

    parent: Node
    """ Parent this node is attached to

    :type: Node
    """

    select: bool
    """ Node selection state

    :type: bool
    """

    show_options: bool
    """ 

    :type: bool
    """

    show_preview: bool
    """ 

    :type: bool
    """

    show_texture: bool
    """ Display node in viewport textured shading mode

    :type: bool
    """

    type: str
    """ Node type (deprecated, use bl_static_type or bl_idname for the actual identifier string)

    :type: str
    """

    use_custom_color: bool
    """ Use custom color for the node

    :type: bool
    """

    width: float
    """ Width of the node

    :type: float
    """

    width_hidden: float
    """ Width of the node in hidden state

    :type: float
    """

    def socket_value_update(self, context: Context):
        """Update after property changes

        :param context:
        :type context: Context
        """
        ...

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """
        ...

    @classmethod
    def poll(cls, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node type can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """
        ...

    def poll_instance(self, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """
        ...

    def update(self):
        """Update on node graph topology changes (adding or removing nodes and links)"""
        ...

    def insert_link(self, link: NodeLink):
        """Handle creation of a link to or from the node

        :param link: Link, Node link that will be inserted
        :type link: NodeLink
        """
        ...

    def init(self, context: Context):
        """Initialize a new instance of this node

        :param context:
        :type context: Context
        """
        ...

    def copy(self, node: Node):
        """Initialize a new instance of this node from an existing node

        :param node: Node, Existing node to copy
        :type node: Node
        """
        ...

    def free(self):
        """Clean up node on removal"""
        ...

    def draw_buttons(self, context: Context, layout: UILayout):
        """Draw node buttons

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_buttons_ext(self, context: Context, layout: UILayout):
        """Draw node buttons in the sidebar

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_label(self) -> str | typing.Any:
        """Returns a dynamic label string

        :return: Label
        :rtype: str | typing.Any
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeInputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(
        self,
        type: str | typing.Any,
        name: str | typing.Any,
        identifier: str | typing.Any = "",
    ) -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeInstanceHash(bpy_struct):
    """Hash table containing node instance data"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeInternalSocketTemplate(bpy_struct):
    """Type and default value of a node socket"""

    identifier: str
    """ Identifier of the socket

    :type: str
    """

    name: str
    """ Name of the socket

    :type: str
    """

    type: str
    """ Data type of the socket

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeLink(bpy_struct):
    """Link between nodes in a node tree"""

    from_node: Node
    """ 

    :type: Node
    """

    from_socket: NodeSocket
    """ 

    :type: NodeSocket
    """

    is_hidden: bool
    """ Link is hidden due to invisible sockets

    :type: bool
    """

    is_muted: bool
    """ Link is muted and can be ignored

    :type: bool
    """

    is_valid: bool
    """ Link is valid

    :type: bool
    """

    to_node: Node
    """ 

    :type: Node
    """

    to_socket: NodeSocket
    """ 

    :type: NodeSocket
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeLinks(bpy_prop_collection[NodeLink], bpy_struct):
    """Collection of Node Links"""

    def new(
        self,
        input: NodeSocket,
        output: NodeSocket,
        verify_limits: bool | typing.Any | None = True,
    ) -> NodeLink:
        """Add a node link to this node tree

        :param input: The input socket
        :type input: NodeSocket
        :param output: The output socket
        :type output: NodeSocket
        :param verify_limits: Verify Limits, Remove existing links if connection limit is exceeded
        :type verify_limits: bool | typing.Any | None
        :return: New node link
        :rtype: NodeLink
        """
        ...

    def remove(self, link: NodeLink):
        """remove a node link from the node tree

        :param link: The node link to remove
        :type link: NodeLink
        """
        ...

    def clear(self):
        """remove all node links from the node tree"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeOutputFileSlotFile(bpy_struct):
    """Single layer file slot of the file output node"""

    format: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    path: str
    """ Subpath used for this slot

    :type: str
    """

    save_as_render: bool
    """ Apply render part of display transform when saving byte image

    :type: bool
    """

    use_node_format: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeOutputFileSlotLayer(bpy_struct):
    """Multilayer slot of the file output node"""

    name: str
    """ OpenEXR layer name used for this slot

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeOutputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(
        self,
        type: str | typing.Any,
        name: str | typing.Any,
        identifier: str | typing.Any = "",
    ) -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeSocket(bpy_struct):
    """Input or output socket of a node"""

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ Label to display for the socket type in the UI

    :type: str
    """

    description: str
    """ Socket tooltip

    :type: str
    """

    display_shape: str
    """ Socket shape

    :type: str
    """

    enabled: bool
    """ Enable the socket

    :type: bool
    """

    hide: bool
    """ Hide the socket

    :type: bool
    """

    hide_value: bool
    """ Hide the socket input value

    :type: bool
    """

    identifier: str
    """ Unique identifier for mapping sockets

    :type: str
    """

    is_linked: bool
    """ True if the socket is connected

    :type: bool
    """

    is_multi_input: bool
    """ True if the socket can accept multiple ordered input links

    :type: bool
    """

    is_output: bool
    """ True if the socket is an output, otherwise input

    :type: bool
    """

    label: str
    """ Custom dynamic defined socket label

    :type: str
    """

    link_limit: int
    """ Max number of links allowed for this socket

    :type: int
    """

    name: str
    """ Socket name

    :type: str
    """

    node: Node
    """ Node owning this socket

    :type: Node
    """

    show_expanded: bool
    """ Socket links are expanded in the user interface

    :type: bool
    """

    type: str
    """ Data type

    :type: str
    """

    links: typing.Any
    """ List of node links from or to this socket.(readonly)"""

    def draw(
        self, context: Context, layout: UILayout, node: Node, text: str | typing.Any
    ):
        """Draw socket

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        :param node: Node, Node the socket belongs to
        :type node: Node
        :param text: Text, Text label to draw alongside properties
        :type text: str | typing.Any
        """
        ...

    def draw_color(self, context: Context, node: Node) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context
        :param node: Node, Node the socket belongs to
        :type node: Node
        :return: Color
        :rtype: bpy_prop_array[float]
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeSocketInterface(bpy_struct):
    """Parameters to define node sockets"""

    attribute_domain: str
    """ Attribute domain used by the geometry nodes modifier to create an attribute output

    :type: str
    """

    bl_label: str
    """ Label to display for the socket type in the UI

    :type: str
    """

    bl_socket_idname: str
    """ 

    :type: str
    """

    default_attribute_name: str
    """ The attribute name used by default when the node group is used by a geometry nodes modifier

    :type: str
    """

    description: str
    """ Socket tooltip

    :type: str
    """

    hide_value: bool
    """ Hide the socket input value even when the socket is not connected

    :type: bool
    """

    identifier: str
    """ Unique identifier for mapping sockets

    :type: str
    """

    is_output: bool
    """ True if the socket is an output, otherwise input

    :type: bool
    """

    name: str
    """ Socket name

    :type: str
    """

    def draw(self, context: Context, layout: UILayout):
        """Draw template settings

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_color(self, context: Context) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context
        :return: Color
        :rtype: bpy_prop_array[float]
        """
        ...

    def register_properties(self, data_rna_type: Struct | None):
        """Define RNA properties of a socket

        :param data_rna_type: Data RNA Type, RNA type for special socket properties
        :type data_rna_type: Struct | None
        """
        ...

    def init_socket(self, node: Node, socket: NodeSocket, data_path: str | typing.Any):
        """Initialize a node socket instance

        :param node: Node, Node of the socket to initialize
        :type node: Node
        :param socket: Socket, Socket to initialize
        :type socket: NodeSocket
        :param data_path: Data Path, Path to specialized socket data
        :type data_path: str | typing.Any
        """
        ...

    def from_socket(self, node: Node, socket: NodeSocket):
        """Setup template parameters from an existing socket

        :param node: Node, Node of the original socket
        :type node: Node
        :param socket: Socket, Original socket
        :type socket: NodeSocket
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeTreeInputs(bpy_prop_collection[NodeSocketInterface], bpy_struct):
    """Collection of Node Tree Sockets"""

    def new(
        self, type: str | typing.Any, name: str | typing.Any
    ) -> NodeSocketInterface:
        """Add a socket to this node tree

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocketInterface
        """
        ...

    def remove(self, socket: NodeSocketInterface | None):
        """Remove a socket from this node tree

        :param socket: The socket to remove
        :type socket: NodeSocketInterface | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node tree"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeTreeOutputs(bpy_prop_collection[NodeSocketInterface], bpy_struct):
    """Collection of Node Tree Sockets"""

    def new(
        self, type: str | typing.Any, name: str | typing.Any
    ) -> NodeSocketInterface:
        """Add a socket to this node tree

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocketInterface
        """
        ...

    def remove(self, socket: NodeSocketInterface | None):
        """Remove a socket from this node tree

        :param socket: The socket to remove
        :type socket: NodeSocketInterface | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node tree"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class NodeTreePath(bpy_struct):
    """Element of the node space tree path"""

    node_tree: NodeTree
    """ Base node tree from context

    :type: NodeTree
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Nodes(bpy_prop_collection[Node], bpy_struct):
    """Collection of Nodes"""

    active: Node | None
    """ Active node in this tree

    :type: Node | None
    """

    def new(self, type: str | typing.Any) -> Node:
        """Add a node to this node tree

        :param type: Type, Type of node to add (Warning: should be same as node.bl_idname, not node.type!)
        :type type: str | typing.Any
        :return: New node
        :rtype: Node
        """
        ...

    def remove(self, node: Node):
        """Remove a node from this node tree

        :param node: The node to remove
        :type node: Node
        """
        ...

    def clear(self):
        """Remove all nodes from this node tree"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ObjectBase(bpy_struct):
    """An object instance in a render layer"""

    hide_viewport: bool
    """ Temporarily hide in viewport

    :type: bool
    """

    object: Object
    """ Object this base links to

    :type: Object
    """

    select: bool
    """ Object base selection state

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ObjectConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of object constraints"""

    active: Constraint | None
    """ Active Object constraint

    :type: Constraint | None
    """

    def new(self, type: str | None) -> Constraint:
        """Add a new constraint to this object

                :param type: Constraint type to add

        CAMERA_SOLVER
        Camera Solver.

        FOLLOW_TRACK
        Follow Track.

        OBJECT_SOLVER
        Object Solver.

        COPY_LOCATION
        Copy Location -- Copy the location of a target (with an optional offset), so that they move together.

        COPY_ROTATION
        Copy Rotation -- Copy the rotation of a target (with an optional offset), so that they rotate together.

        COPY_SCALE
        Copy Scale -- Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.

        COPY_TRANSFORMS
        Copy Transforms -- Copy all the transformations of a target, so that they move together.

        LIMIT_DISTANCE
        Limit Distance -- Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).

        LIMIT_LOCATION
        Limit Location -- Restrict movement along each axis within given ranges.

        LIMIT_ROTATION
        Limit Rotation -- Restrict rotation along each axis within given ranges.

        LIMIT_SCALE
        Limit Scale -- Restrict scaling along each axis with given ranges.

        MAINTAIN_VOLUME
        Maintain Volume -- Compensate for scaling one axis by applying suitable scaling to the other two axes.

        TRANSFORM
        Transformation -- Use one transform property from target to control another (or same) property on owner.

        TRANSFORM_CACHE
        Transform Cache -- Look up the transformation matrix from an external file.

        CLAMP_TO
        Clamp To -- Restrict movements to lie along a curve by remapping location along curve's longest axis.

        DAMPED_TRACK
        Damped Track -- Point towards a target by performing the smallest rotation necessary.

        IK
        Inverse Kinematics -- Control a chain of bones by specifying the endpoint target (Bones only).

        LOCKED_TRACK
        Locked Track -- Rotate around the specified ('locked') axis to point towards a target.

        SPLINE_IK
        Spline IK -- Align chain of bones along a curve (Bones only).

        STRETCH_TO
        Stretch To -- Stretch along Y-Axis to point towards a target.

        TRACK_TO
        Track To -- Legacy tracking constraint prone to twisting artifacts.

        ACTION
        Action -- Use transform property of target to look up pose for owner from an Action.

        ARMATURE
        Armature -- Apply weight-blended transformation from multiple bones like the Armature modifier.

        CHILD_OF
        Child Of -- Make target the 'detachable' parent of owner.

        FLOOR
        Floor -- Use position (and optionally rotation) of target to define a 'wall' or 'floor' that the owner can not cross.

        FOLLOW_PATH
        Follow Path -- Use to animate an object/bone following a path.

        PIVOT
        Pivot -- Change pivot point for transforms (buggy).

        SHRINKWRAP
        Shrinkwrap -- Restrict movements to surface of target mesh.
                :type type: str | None
                :return: New constraint
                :rtype: Constraint
        """
        ...

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """
        ...

    def clear(self):
        """Remove all constraint from this object"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a constraint to a different position

        :param from_index: From Index, Index to move
        :type from_index: int | None
        :param to_index: To Index, Target index
        :type to_index: int | None
        """
        ...

    def copy(self, constraint: Constraint) -> Constraint:
        """Add a new constraint that is a copy of the given one

        :param constraint: Constraint to copy - may belong to a different object
        :type constraint: Constraint
        :return: New constraint
        :rtype: Constraint
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ObjectDisplay(bpy_struct):
    """Object display settings for 3D viewport"""

    show_shadows: bool
    """ Object cast shadows in the 3D viewport

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ObjectGpencilModifiers(bpy_prop_collection[GpencilModifier], bpy_struct):
    """Collection of object grease pencil modifiers"""

    def new(self, name: str | typing.Any, type: str | None) -> GpencilModifier:
        """Add a new greasepencil_modifier

                :param name: New name for the greasepencil_modifier
                :type name: str | typing.Any
                :param type: Modifier type to add

        GP_TEXTURE
        Texture Mapping -- Change stroke uv texture values.

        GP_TIME
        Time Offset -- Offset keyframes.

        GP_WEIGHT_ANGLE
        Vertex Weight Angle -- Generate Vertex Weights base on stroke angle.

        GP_WEIGHT_PROXIMITY
        Vertex Weight Proximity -- Generate Vertex Weights base on distance to object.

        GP_ARRAY
        Array -- Create array of duplicate instances.

        GP_BUILD
        Build -- Create duplication of strokes.

        GP_DASH
        Dot Dash -- Generate dot-dash styled strokes.

        GP_ENVELOPE
        Envelope -- Create an envelope shape.

        GP_LENGTH
        Length -- Extend or shrink strokes.

        GP_LINEART
        Line Art -- Generate line art strokes from selected source.

        GP_MIRROR
        Mirror -- Duplicate strokes like a mirror.

        GP_MULTIPLY
        Multiple Strokes -- Produce multiple strokes along one stroke.

        GP_SIMPLIFY
        Simplify -- Simplify stroke reducing number of points.

        GP_SUBDIV
        Subdivide -- Subdivide stroke adding more control points.

        GP_ARMATURE
        Armature -- Deform stroke points using armature object.

        GP_HOOK
        Hook -- Deform stroke points using objects.

        GP_LATTICE
        Lattice -- Deform strokes using lattice.

        GP_NOISE
        Noise -- Add noise to strokes.

        GP_OFFSET
        Offset -- Change stroke location, rotation or scale.

        SHRINKWRAP
        Shrinkwrap -- Project the shape onto another object.

        GP_SMOOTH
        Smooth -- Smooth stroke.

        GP_THICK
        Thickness -- Change stroke thickness.

        GP_COLOR
        Hue/Saturation -- Apply changes to stroke colors.

        GP_OPACITY
        Opacity -- Opacity of the strokes.

        GP_TINT
        Tint -- Tint strokes with new color.
                :type type: str | None
                :return: Newly created modifier
                :rtype: GpencilModifier
        """
        ...

    def remove(self, greasepencil_modifier: GpencilModifier):
        """Remove an existing greasepencil_modifier from the object

        :param greasepencil_modifier: Modifier to remove
        :type greasepencil_modifier: GpencilModifier
        """
        ...

    def clear(self):
        """Remove all grease pencil modifiers from the object"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ObjectLineArt(bpy_struct):
    """Object line art settings"""

    crease_threshold: float
    """ Angles smaller than this will be treated as creases

    :type: float
    """

    usage: str
    """ How to use this object in line art calculation

    :type: str
    """

    use_crease_override: bool
    """ Use this object's crease setting to overwrite scene global

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ObjectModifiers(bpy_prop_collection[Modifier], bpy_struct):
    """Collection of object modifiers"""

    active: Modifier | None
    """ The active modifier in the list

    :type: Modifier | None
    """

    def new(self, name: str | typing.Any, type: str | None) -> Modifier:
        """Add a new modifier

                :param name: New name for the modifier
                :type name: str | typing.Any
                :param type: Modifier type to add

        DATA_TRANSFER
        Data Transfer -- Transfer several types of data (vertex groups, UV maps, vertex colors, custom normals) from one mesh to another.

        MESH_CACHE
        Mesh Cache -- Deform the mesh using an external frame-by-frame vertex transform cache.

        MESH_SEQUENCE_CACHE
        Mesh Sequence Cache -- Deform the mesh or curve using an external mesh cache in Alembic format.

        NORMAL_EDIT
        Normal Edit -- Modify the direction of the surface normals.

        WEIGHTED_NORMAL
        Weighted Normal -- Modify the direction of the surface normals using a weighting method.

        UV_PROJECT
        UV Project -- Project the UV map coordinates from the negative Z axis of another object.

        UV_WARP
        UV Warp -- Transform the UV map using the difference between two objects.

        VERTEX_WEIGHT_EDIT
        Vertex Weight Edit -- Modify of the weights of a vertex group.

        VERTEX_WEIGHT_MIX
        Vertex Weight Mix -- Mix the weights of two vertex groups.

        VERTEX_WEIGHT_PROXIMITY
        Vertex Weight Proximity -- Set the vertex group weights based on the distance to another target object.

        ARRAY
        Array -- Create copies of the shape with offsets.

        BEVEL
        Bevel -- Generate sloped corners by adding geometry to the mesh's edges or vertices.

        BOOLEAN
        Boolean -- Use another shape to cut, combine or perform a difference operation.

        BUILD
        Build -- Cause the faces of the mesh object to appear or disappear one after the other over time.

        DECIMATE
        Decimate -- Reduce the geometry density.

        EDGE_SPLIT
        Edge Split -- Split away joined faces at the edges.

        NODES
        Geometry Nodes.

        MASK
        Mask -- Dynamically hide vertices based on a vertex group or armature.

        MIRROR
        Mirror -- Mirror along the local X, Y and/or Z axes, over the object origin.

        MESH_TO_VOLUME
        Mesh to Volume.

        MULTIRES
        Multiresolution -- Subdivide the mesh in a way that allows editing the higher subdivision levels.

        REMESH
        Remesh -- Generate new mesh topology based on the current shape.

        SCREW
        Screw -- Lathe around an axis, treating the input mesh as a profile.

        SKIN
        Skin -- Create a solid shape from vertices and edges, using the vertex radius to define the thickness.

        SOLIDIFY
        Solidify -- Make the surface thick.

        SUBSURF
        Subdivision Surface -- Split the faces into smaller parts, giving it a smoother appearance.

        TRIANGULATE
        Triangulate -- Convert all polygons to triangles.

        VOLUME_TO_MESH
        Volume to Mesh.

        WELD
        Weld -- Find groups of vertices closer than dist and merge them together.

        WIREFRAME
        Wireframe -- Convert faces into thickened edges.

        ARMATURE
        Armature -- Deform the shape using an armature object.

        CAST
        Cast -- Shift the shape towards a predefined primitive.

        CURVE
        Curve -- Bend the mesh using a curve object.

        DISPLACE
        Displace -- Offset vertices based on a texture.

        HOOK
        Hook -- Deform specific points using another object.

        LAPLACIANDEFORM
        Laplacian Deform -- Deform based a series of anchor points.

        LATTICE
        Lattice -- Deform using the shape of a lattice object.

        MESH_DEFORM
        Mesh Deform -- Deform using a different mesh, which acts as a deformation cage.

        SHRINKWRAP
        Shrinkwrap -- Project the shape onto another object.

        SIMPLE_DEFORM
        Simple Deform -- Deform the shape by twisting, bending, tapering or stretching.

        SMOOTH
        Smooth -- Smooth the mesh by flattening the angles between adjacent faces.

        CORRECTIVE_SMOOTH
        Smooth Corrective -- Smooth the mesh while still preserving the volume.

        LAPLACIANSMOOTH
        Smooth Laplacian -- Reduce the noise on a mesh surface with minimal changes to its shape.

        SURFACE_DEFORM
        Surface Deform -- Transfer motion from another mesh.

        WARP
        Warp -- Warp parts of a mesh to a new location in a very flexible way thanks to 2 specified objects.

        WAVE
        Wave -- Adds a ripple-like motion to an object's geometry.

        VOLUME_DISPLACE
        Volume Displace -- Deform volume based on noise or other vector fields.

        CLOTH
        Cloth.

        COLLISION
        Collision.

        DYNAMIC_PAINT
        Dynamic Paint.

        EXPLODE
        Explode -- Break apart the mesh faces and let them follow particles.

        FLUID
        Fluid.

        OCEAN
        Ocean -- Generate a moving ocean surface.

        PARTICLE_INSTANCE
        Particle Instance.

        PARTICLE_SYSTEM
        Particle System -- Spawn particles from the shape.

        SOFT_BODY
        Soft Body.

        SURFACE
        Surface.
                :type type: str | None
                :return: Newly created modifier
                :rtype: Modifier
        """
        ...

    def remove(self, modifier: Modifier):
        """Remove an existing modifier from the object

        :param modifier: Modifier to remove
        :type modifier: Modifier
        """
        ...

    def clear(self):
        """Remove all modifiers from the object"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ObjectShaderFx(bpy_prop_collection[ShaderFx], bpy_struct):
    """Collection of object effects"""

    def new(self, name: str | typing.Any, type: str | None) -> ShaderFx:
        """Add a new shader fx

                :param name: New name for the effect
                :type name: str | typing.Any
                :param type: Effect type to add

        FX_BLUR
        Blur -- Apply Gaussian Blur to object.

        FX_COLORIZE
        Colorize -- Apply different tint effects.

        FX_FLIP
        Flip -- Flip image.

        FX_GLOW
        Glow -- Create a glow effect.

        FX_PIXEL
        Pixelate -- Pixelate image.

        FX_RIM
        Rim -- Add a rim to the image.

        FX_SHADOW
        Shadow -- Create a shadow effect.

        FX_SWIRL
        Swirl -- Create a rotation distortion.

        FX_WAVE
        Wave Distortion -- Apply sinusoidal deformation.
                :type type: str | None
                :return: Newly created effect
                :rtype: ShaderFx
        """
        ...

    def remove(self, shader_fx: ShaderFx):
        """Remove an existing effect from the object

        :param shader_fx: Effect to remove
        :type shader_fx: ShaderFx
        """
        ...

    def clear(self):
        """Remove all effects from the object"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Operator(bpy_struct):
    """Storage of an operator being executed, or registered after execution"""

    bl_cursor_pending: str
    """ Cursor to use when waiting for the user to select a location to activate the operator (when bl_options has DEPENDS_ON_CURSOR set)

    :type: str
    """

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[str]
    """ Options for this operator type

    :type: set[str]
    """

    bl_translation_context: str
    """ 

    :type: str
    """

    bl_undo_group: str
    """ 

    :type: str
    """

    has_reports: bool
    """ Operator has a set of reports (warnings and errors) from last execution

    :type: bool
    """

    layout: UILayout
    """ 

    :type: UILayout
    """

    macros: bpy_prop_collection[Macro]
    """ 

    :type: bpy_prop_collection[Macro]
    """

    name: str
    """ 

    :type: str
    """

    options: OperatorOptions
    """ Runtime options

    :type: OperatorOptions
    """

    properties: OperatorProperties
    """ 

    :type: OperatorProperties
    """

    bl_property: typing.Any
    """ The name of a property to use as this operators primary property.
Currently this is only used to select the default property when
expanding an operator into a menu.
:type: string"""

    is_registered: bool
    """ 

    :type: bool
    """

    def report(self, type: set[str] | None, message: str | typing.Any):
        """report

        :param type: Type
        :type type: set[str] | None
        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    def is_repeat(self) -> bool:
        """is_repeat

        :return: result
        :rtype: bool
        """
        ...

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    def execute(self, context: Context) -> set[str]:
        """Execute the operator

                :param context:
                :type context: Context
                :return: result

        RUNNING_MODAL
        Running Modal -- Keep the operator running with blender.

        CANCELLED
        Cancelled -- The operator exited without doing anything, so no undo entry should be pushed.

        FINISHED
        Finished -- The operator exited after completing its action.

        PASS_THROUGH
        Pass Through -- Do nothing and pass the event on.

        INTERFACE
        Interface -- Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def check(self, context: Context) -> bool:
        """Check the operator settings, return True to signal a change to redraw

        :param context:
        :type context: Context
        :return: result
        :rtype: bool
        """
        ...

    def invoke(self, context: Context, event: Event) -> set[str]:
        """Invoke the operator

                :param context:
                :type context: Context
                :param event:
                :type event: Event
                :return: result

        RUNNING_MODAL
        Running Modal -- Keep the operator running with blender.

        CANCELLED
        Cancelled -- The operator exited without doing anything, so no undo entry should be pushed.

        FINISHED
        Finished -- The operator exited after completing its action.

        PASS_THROUGH
        Pass Through -- Do nothing and pass the event on.

        INTERFACE
        Interface -- Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def modal(self, context: Context, event: Event) -> set[str]:
        """Modal operator function

                :param context:
                :type context: Context
                :param event:
                :type event: Event
                :return: result

        RUNNING_MODAL
        Running Modal -- Keep the operator running with blender.

        CANCELLED
        Cancelled -- The operator exited without doing anything, so no undo entry should be pushed.

        FINISHED
        Finished -- The operator exited after completing its action.

        PASS_THROUGH
        Pass Through -- Do nothing and pass the event on.

        INTERFACE
        Interface -- Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def draw(self, context: Context):
        """Draw function for the operator

        :param context:
        :type context: Context
        """
        ...

    def cancel(self, context: Context):
        """Called when the operator is canceled

        :param context:
        :type context: Context
        """
        ...

    @classmethod
    def description(cls, context: Context, properties: OperatorProperties) -> str:
        """Compute a description string that depends on parameters

        :param context:
        :type context: Context
        :param properties:
        :type properties: OperatorProperties
        :return: result
        :rtype: str
        """
        ...

    def as_keywords(self, *, ignore=()):
        """Return a copy of the properties as a dictionary

        :param ignore:
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

    def poll_message_set(self, message: str | None, *args):
        """Set the message to show in the tool-tip when poll fails.When message is callable, additional user defined positional arguments are passed to the message function.

        :param message: The message or a function that returns the message.
        :type message: str | None
        :param args:
        """
        ...

class OperatorMacro(bpy_struct):
    """Storage of a sub operator in a macro after it has been added"""

    properties: OperatorProperties
    """ 

    :type: OperatorProperties
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class OperatorOptions(bpy_struct):
    """Runtime options"""

    is_grab_cursor: bool
    """ True when the cursor is grabbed

    :type: bool
    """

    is_invoke: bool
    """ True when invoked (even if only the execute callbacks available)

    :type: bool
    """

    is_repeat: bool
    """ True when run from the 'Adjust Last Operation' panel

    :type: bool
    """

    is_repeat_last: bool
    """ True when run from the operator 'Repeat Last'

    :type: bool
    """

    use_cursor_region: bool
    """ Enable to use the region under the cursor for modal execution

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class OperatorProperties(bpy_struct):
    """Input properties of an operator"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PackedFile(bpy_struct):
    """External file packed into the .blend file"""

    data: str
    """ Raw data (bytes, exact content of the embedded file)

    :type: str
    """

    size: int
    """ Size of packed file in bytes

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Paint(bpy_struct):
    brush: Brush
    """ Active Brush

    :type: Brush
    """

    cavity_curve: CurveMapping
    """ Editable cavity curve

    :type: CurveMapping
    """

    input_samples: int
    """ Average multiple input samples together to smooth the brush stroke

    :type: int
    """

    palette: Palette
    """ Active Palette

    :type: Palette
    """

    show_brush: bool
    """ 

    :type: bool
    """

    show_brush_on_surface: bool
    """ 

    :type: bool
    """

    show_low_resolution: bool
    """ For multires, show low resolution while navigating the view

    :type: bool
    """

    tile_offset: bpy_prop_array[float]
    """ Stride at which tiled strokes are copied

    :type: bpy_prop_array[float]
    """

    tile_x: bool
    """ Tile along X axis

    :type: bool
    """

    tile_y: bool
    """ Tile along Y axis

    :type: bool
    """

    tile_z: bool
    """ Tile along Z axis

    :type: bool
    """

    tool_slots: bpy_prop_collection[PaintToolSlot]
    """ 

    :type: bpy_prop_collection[PaintToolSlot]
    """

    use_cavity: bool
    """ Mask painting according to mesh geometry cavity

    :type: bool
    """

    use_sculpt_delay_updates: bool
    """ Update the geometry when it enters the view, providing faster view navigation

    :type: bool
    """

    use_symmetry_feather: bool
    """ Reduce the strength of the brush where it overlaps symmetrical daubs

    :type: bool
    """

    use_symmetry_x: bool
    """ Mirror brush across the X axis

    :type: bool
    """

    use_symmetry_y: bool
    """ Mirror brush across the Y axis

    :type: bool
    """

    use_symmetry_z: bool
    """ Mirror brush across the Z axis

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PaintModeSettings(bpy_struct):
    """Properties of paint mode"""

    canvas_image: Image
    """ Image used as as painting target

    :type: Image
    """

    canvas_source: str
    """ Source to select canvas from

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PaintToolSlot(bpy_struct):
    brush: Brush
    """ 

    :type: Brush
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PaletteColor(bpy_struct):
    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    strength: float
    """ 

    :type: float
    """

    weight: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PaletteColors(bpy_prop_collection[PaletteColor], bpy_struct):
    """Collection of palette colors"""

    active: PaletteColor | None
    """ 

    :type: PaletteColor | None
    """

    def new(self) -> PaletteColor:
        """Add a new color to the palette

        :return: The newly created color
        :rtype: PaletteColor
        """
        ...

    def remove(self, color: PaletteColor):
        """Remove a color from the palette

        :param color: The color to remove
        :type color: PaletteColor
        """
        ...

    def clear(self):
        """Remove all colors from the palette"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Panel(bpy_struct):
    """Panel containing UI elements"""

    bl_category: str
    """ The category (tab) in which the panel will be displayed, when applicable

    :type: str
    """

    bl_context: str
    """ The context in which the panel belongs to. (TODO: explain the possible combinations bl_context/bl_region_type/bl_space_type)

    :type: str
    """

    bl_description: str
    """ The panel tooltip

    :type: str
    """

    bl_idname: str
    """ If this is set, the panel gets a custom ID, otherwise it takes the name of the class used to define the panel. For example, if the class name is "OBJECT_PT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_PT_hello"

    :type: str
    """

    bl_label: str
    """ The panel label, shows up in the panel header at the right of the triangle used to collapse the panel

    :type: str
    """

    bl_options: set[str]
    """ Options for this panel type

    :type: set[str]
    """

    bl_order: int
    """ Panels with lower numbers are default ordered before panels with higher numbers

    :type: int
    """

    bl_owner_id: str
    """ The ID owning the data displayed in the panel, if any

    :type: str
    """

    bl_parent_id: str
    """ If this is set, the panel becomes a sub-panel

    :type: str
    """

    bl_region_type: str
    """ The region where the panel is going to be used in

    :type: str
    """

    bl_space_type: str
    """ The space where the panel is going to be used in

    :type: str
    """

    bl_translation_context: str | typing.Any
    """ Specific translation context, only define when the label needs to be disambiguated from others using the exact same label

    :type: str | typing.Any
    """

    bl_ui_units_x: int
    """ When set, defines popup panel width

    :type: int
    """

    custom_data: Constraint
    """ Panel data

    :type: Constraint
    """

    is_popover: bool
    """ 

    :type: bool
    """

    layout: UILayout
    """ Defines the structure of the panel in the UI

    :type: UILayout
    """

    text: str
    """ XXX todo

    :type: str
    """

    use_pin: bool
    """ Show the panel on all tabs

    :type: bool
    """

    is_registered: bool
    """ 

    :type: bool
    """

    @classmethod
    def poll(cls, context: Context) -> bool:
        """If this method returns a non-null output, then the panel can be drawn

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    def draw(self, context: Context):
        """Draw UI elements into the panel UI layout

        :param context:
        :type context: Context
        """
        ...

    def draw_header(self, context: Context):
        """Draw UI elements into the panel's header UI layout

        :param context:
        :type context: Context
        """
        ...

    def draw_header_preset(self, context: Context):
        """Draw UI elements for presets in the panel's header

        :param context:
        :type context: Context
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Particle(bpy_struct):
    """Particle in a particle system"""

    alive_state: str
    """ 

    :type: str
    """

    angular_velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    birth_time: float
    """ 

    :type: float
    """

    die_time: float
    """ 

    :type: float
    """

    hair_keys: bpy_prop_collection[ParticleHairKey]
    """ 

    :type: bpy_prop_collection[ParticleHairKey]
    """

    is_exist: bool
    """ 

    :type: bool
    """

    is_visible: bool
    """ 

    :type: bool
    """

    lifetime: float
    """ 

    :type: float
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    particle_keys: bpy_prop_collection[ParticleKey]
    """ 

    :type: bpy_prop_collection[ParticleKey]
    """

    prev_angular_velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    prev_location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    prev_rotation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    prev_velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    rotation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    size: float
    """ 

    :type: float
    """

    velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    def uv_on_emitter(self, modifier: ParticleSystemModifier) -> bpy_prop_array[float]:
        """Obtain UV coordinates for a particle on an evaluated mesh.

        :param modifier: Particle modifier from an evaluated object
        :type modifier: ParticleSystemModifier
        :return: uv
        :rtype: bpy_prop_array[float]
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleBrush(bpy_struct):
    """Particle editing brush"""

    count: int
    """ Particle count

    :type: int
    """

    curve: CurveMapping
    """ 

    :type: CurveMapping
    """

    length_mode: str
    """ 

    :type: str
    """

    puff_mode: str
    """ 

    :type: str
    """

    size: int
    """ Radius of the brush in pixels

    :type: int
    """

    steps: int
    """ Brush steps

    :type: int
    """

    strength: float
    """ Brush strength

    :type: float
    """

    use_puff_volume: bool
    """ Apply puff to unselected end-points (helps maintain hair volume when puffing root)

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleDupliWeight(bpy_struct):
    """Weight of a particle instance object in a collection"""

    count: int
    """ The number of times this object is repeated with respect to other objects

    :type: int
    """

    name: str
    """ Particle instance object name

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleEdit(bpy_struct):
    """Properties of particle editing mode"""

    brush: ParticleBrush
    """ 

    :type: ParticleBrush
    """

    default_key_count: int
    """ How many keys to make new particles with

    :type: int
    """

    display_step: int
    """ How many steps to display the path with

    :type: int
    """

    emitter_distance: float
    """ Distance to keep particles away from the emitter

    :type: float
    """

    fade_frames: int
    """ How many frames to fade

    :type: int
    """

    is_editable: bool
    """ A valid edit mode exists

    :type: bool
    """

    is_hair: bool
    """ Editing hair

    :type: bool
    """

    object: Object
    """ The edited object

    :type: Object
    """

    select_mode: str
    """ Particle select and display mode

    :type: str
    """

    shape_object: Object
    """ Outer shape to use for tools

    :type: Object
    """

    show_particles: bool
    """ Display actual particles

    :type: bool
    """

    tool: str
    """ 

    :type: str
    """

    type: str
    """ 

    :type: str
    """

    use_auto_velocity: bool
    """ Calculate point velocities automatically

    :type: bool
    """

    use_default_interpolate: bool
    """ Interpolate new particles from the existing ones

    :type: bool
    """

    use_emitter_deflect: bool
    """ Keep paths from intersecting the emitter

    :type: bool
    """

    use_fade_time: bool
    """ Fade paths and keys further away from current frame

    :type: bool
    """

    use_preserve_length: bool
    """ Keep path lengths constant

    :type: bool
    """

    use_preserve_root: bool
    """ Keep root keys unmodified

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleHairKey(bpy_struct):
    """Particle key for hair particle system"""

    co: bpy_prop_array[float]
    """ Location of the hair key in object space

    :type: bpy_prop_array[float]
    """

    co_local: bpy_prop_array[float]
    """ Location of the hair key in its local coordinate system, relative to the emitting face

    :type: bpy_prop_array[float]
    """

    time: float
    """ Relative time of key over hair length

    :type: float
    """

    weight: float
    """ Weight for cloth simulation

    :type: float
    """

    def co_object(
        self, object: Object, modifier: ParticleSystemModifier, particle: Particle
    ) -> bpy_prop_array[float]:
        """Obtain hairkey location with particle and modifier data

        :param object: Object
        :type object: Object
        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :param particle: hair particle
        :type particle: Particle
        :return: Co, Exported hairkey location
        :rtype: bpy_prop_array[float]
        """
        ...

    def co_object_set(
        self,
        object: Object,
        modifier: ParticleSystemModifier,
        particle: Particle,
        co: collections.abc.Iterable[float] | None,
    ):
        """Set hairkey location with particle and modifier data

        :param object: Object
        :type object: Object
        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :param particle: hair particle
        :type particle: Particle
        :param co: Co, Specified hairkey location
        :type co: collections.abc.Iterable[float] | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleKey(bpy_struct):
    """Key location for a particle over time"""

    angular_velocity: bpy_prop_array[float]
    """ Key angular velocity

    :type: bpy_prop_array[float]
    """

    location: bpy_prop_array[float]
    """ Key location

    :type: bpy_prop_array[float]
    """

    rotation: bpy_prop_array[float]
    """ Key rotation quaternion

    :type: bpy_prop_array[float]
    """

    time: float
    """ Time of key over the simulation

    :type: float
    """

    velocity: bpy_prop_array[float]
    """ Key velocity

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleSettingsTextureSlots(
    bpy_prop_collection[ParticleSettingsTextureSlot], bpy_struct
):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> ParticleSettingsTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """
        ...

    @classmethod
    def create(cls, index: int | None) -> ParticleSettingsTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """
        ...

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleSystem(bpy_struct):
    """Particle system in an object"""

    active_particle_target: ParticleTarget
    """ 

    :type: ParticleTarget
    """

    active_particle_target_index: int | None
    """ 

    :type: int | None
    """

    child_particles: bpy_prop_collection[ChildParticle]
    """ Child particles generated by the particle system

    :type: bpy_prop_collection[ChildParticle]
    """

    child_seed: int
    """ Offset in the random number table for child particles, to get a different randomized result

    :type: int
    """

    cloth: ClothModifier
    """ Cloth dynamics for hair

    :type: ClothModifier
    """

    dt_frac: float
    """ The current simulation time step size, as a fraction of a frame

    :type: float
    """

    has_multiple_caches: bool
    """ Particle system has multiple point caches

    :type: bool
    """

    invert_vertex_group_clump: bool
    """ Negate the effect of the clump vertex group

    :type: bool
    """

    invert_vertex_group_density: bool
    """ Negate the effect of the density vertex group

    :type: bool
    """

    invert_vertex_group_field: bool
    """ Negate the effect of the field vertex group

    :type: bool
    """

    invert_vertex_group_kink: bool
    """ Negate the effect of the kink vertex group

    :type: bool
    """

    invert_vertex_group_length: bool
    """ Negate the effect of the length vertex group

    :type: bool
    """

    invert_vertex_group_rotation: bool
    """ Negate the effect of the rotation vertex group

    :type: bool
    """

    invert_vertex_group_roughness_1: bool
    """ Negate the effect of the roughness 1 vertex group

    :type: bool
    """

    invert_vertex_group_roughness_2: bool
    """ Negate the effect of the roughness 2 vertex group

    :type: bool
    """

    invert_vertex_group_roughness_end: bool
    """ Negate the effect of the roughness end vertex group

    :type: bool
    """

    invert_vertex_group_size: bool
    """ Negate the effect of the size vertex group

    :type: bool
    """

    invert_vertex_group_tangent: bool
    """ Negate the effect of the tangent vertex group

    :type: bool
    """

    invert_vertex_group_twist: bool
    """ Negate the effect of the twist vertex group

    :type: bool
    """

    invert_vertex_group_velocity: bool
    """ Negate the effect of the velocity vertex group

    :type: bool
    """

    is_editable: bool
    """ Particle system can be edited in particle mode

    :type: bool
    """

    is_edited: bool
    """ Particle system has been edited in particle mode

    :type: bool
    """

    is_global_hair: bool
    """ Hair keys are in global coordinate space

    :type: bool
    """

    name: str
    """ Particle system name

    :type: str
    """

    parent: Object
    """ Use this object's coordinate system instead of global coordinate system

    :type: Object
    """

    particles: bpy_prop_collection[Particle]
    """ Particles generated by the particle system

    :type: bpy_prop_collection[Particle]
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    reactor_target_object: Object
    """ For reactor systems, the object that has the target particle system (empty if same object)

    :type: Object
    """

    reactor_target_particle_system: int
    """ For reactor systems, index of particle system on the target object

    :type: int
    """

    seed: int
    """ Offset in the random number table, to get a different randomized result

    :type: int
    """

    settings: ParticleSettings
    """ Particle system settings

    :type: ParticleSettings
    """

    targets: bpy_prop_collection[ParticleTarget]
    """ Target particle systems

    :type: bpy_prop_collection[ParticleTarget]
    """

    use_hair_dynamics: bool
    """ Enable hair dynamics using cloth simulation

    :type: bool
    """

    use_keyed_timing: bool
    """ Use key times

    :type: bool
    """

    vertex_group_clump: str
    """ Vertex group to control clump

    :type: str
    """

    vertex_group_density: str
    """ Vertex group to control density

    :type: str
    """

    vertex_group_field: str
    """ Vertex group to control field

    :type: str
    """

    vertex_group_kink: str
    """ Vertex group to control kink

    :type: str
    """

    vertex_group_length: str
    """ Vertex group to control length

    :type: str
    """

    vertex_group_rotation: str
    """ Vertex group to control rotation

    :type: str
    """

    vertex_group_roughness_1: str
    """ Vertex group to control roughness 1

    :type: str
    """

    vertex_group_roughness_2: str
    """ Vertex group to control roughness 2

    :type: str
    """

    vertex_group_roughness_end: str
    """ Vertex group to control roughness end

    :type: str
    """

    vertex_group_size: str
    """ Vertex group to control size

    :type: str
    """

    vertex_group_tangent: str
    """ Vertex group to control tangent

    :type: str
    """

    vertex_group_twist: str
    """ Vertex group to control twist

    :type: str
    """

    vertex_group_velocity: str
    """ Vertex group to control velocity

    :type: str
    """

    def co_hair(
        self,
        object: Object,
        particle_no: typing.Any | None = 0,
        step: typing.Any | None = 0,
    ) -> bpy_prop_array[float]:
        """Obtain cache hair data

        :param object: Object
        :type object: Object
        :param particle_no: Particle no
        :type particle_no: typing.Any | None
        :param step: step no
        :type step: typing.Any | None
        :return: Co, Exported hairkey location
        :rtype: bpy_prop_array[float]
        """
        ...

    def uv_on_emitter(
        self,
        modifier: ParticleSystemModifier,
        particle: Particle,
        particle_no: typing.Any | None = 0,
        uv_no: typing.Any | None = 0,
    ) -> bpy_prop_array[float]:
        """Obtain uv for all particles

        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :param particle: Particle
        :type particle: Particle
        :param particle_no: Particle no
        :type particle_no: typing.Any | None
        :param uv_no: UV no
        :type uv_no: typing.Any | None
        :return: uv
        :rtype: bpy_prop_array[float]
        """
        ...

    def mcol_on_emitter(
        self,
        modifier: ParticleSystemModifier,
        particle: Particle,
        particle_no: typing.Any | None = 0,
        vcol_no: typing.Any | None = 0,
    ) -> bpy_prop_array[float]:
        """Obtain mcol for all particles

        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :param particle: Particle
        :type particle: Particle
        :param particle_no: Particle no
        :type particle_no: typing.Any | None
        :param vcol_no: vcol no
        :type vcol_no: typing.Any | None
        :return: mcol
        :rtype: bpy_prop_array[float]
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleSystems(bpy_prop_collection[ParticleSystem], bpy_struct):
    """Collection of particle systems"""

    active: ParticleSystem
    """ Active particle system being displayed

    :type: ParticleSystem
    """

    active_index: int | None
    """ Index of active particle system slot

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ParticleTarget(bpy_struct):
    """Target particle system"""

    alliance: str
    """ 

    :type: str
    """

    duration: float
    """ 

    :type: float
    """

    is_valid: bool
    """ Keyed particles target is valid

    :type: bool
    """

    name: str
    """ Particle target name

    :type: str
    """

    object: Object
    """ The object that has the target particle system (empty if same object)

    :type: Object
    """

    system: int
    """ The index of particle system on the target object

    :type: int
    """

    time: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PathCompare(bpy_struct):
    """Match paths against this value"""

    path: str
    """ 

    :type: str
    """

    use_glob: bool
    """ Enable wildcard globbing

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PathCompareCollection(bpy_prop_collection[PathCompare], bpy_struct):
    """Collection of paths"""

    @classmethod
    def new(cls) -> PathCompare:
        """Add a new path

        :return:
        :rtype: PathCompare
        """
        ...

    @classmethod
    def remove(cls, pathcmp: PathCompare):
        """Remove path

        :param pathcmp:
        :type pathcmp: PathCompare
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Point(bpy_struct):
    """Point in a point cloud"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    index: int
    """ Index of this points

    :type: int
    """

    radius: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PointCache(bpy_struct):
    """Active point cache for physics simulations"""

    compression: str
    """ Compression method to be used

    :type: str
    """

    filepath: str
    """ Cache file path

    :type: str
    """

    frame_end: int
    """ Frame on which the simulation stops

    :type: int
    """

    frame_start: int
    """ Frame on which the simulation starts

    :type: int
    """

    frame_step: int
    """ Number of frames between cached frames

    :type: int
    """

    index: int
    """ Index number of cache files

    :type: int
    """

    info: str
    """ Info on current cache status

    :type: str
    """

    is_baked: bool
    """ The cache is baked

    :type: bool
    """

    is_baking: bool
    """ The cache is being baked

    :type: bool
    """

    is_frame_skip: bool
    """ Some frames were skipped while baking/saving that cache

    :type: bool
    """

    is_outdated: bool
    """ 

    :type: bool
    """

    name: str
    """ Cache name

    :type: str
    """

    point_caches: PointCaches
    """ 

    :type: PointCaches
    """

    use_disk_cache: bool
    """ Save cache files to disk (.blend file must be saved first)

    :type: bool
    """

    use_external: bool
    """ Read cache from an external location

    :type: bool
    """

    use_library_path: bool
    """ Use this file's path for the disk cache when library linked into another file (for local bakes per scene file, disable this option)

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PointCacheItem(bpy_struct):
    """Point cache for physics simulations"""

    compression: str
    """ Compression method to be used

    :type: str
    """

    filepath: str
    """ Cache file path

    :type: str
    """

    frame_end: int
    """ Frame on which the simulation stops

    :type: int
    """

    frame_start: int
    """ Frame on which the simulation starts

    :type: int
    """

    frame_step: int
    """ Number of frames between cached frames

    :type: int
    """

    index: int
    """ Index number of cache files

    :type: int
    """

    info: str
    """ Info on current cache status

    :type: str
    """

    is_baked: bool
    """ The cache is baked

    :type: bool
    """

    is_baking: bool
    """ The cache is being baked

    :type: bool
    """

    is_frame_skip: bool
    """ Some frames were skipped while baking/saving that cache

    :type: bool
    """

    is_outdated: bool
    """ 

    :type: bool
    """

    name: str
    """ Cache name

    :type: str
    """

    use_disk_cache: bool
    """ Save cache files to disk (.blend file must be saved first)

    :type: bool
    """

    use_external: bool
    """ Read cache from an external location

    :type: bool
    """

    use_library_path: bool
    """ Use this file's path for the disk cache when library linked into another file (for local bakes per scene file, disable this option)

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PointCaches(bpy_prop_collection[PointCacheItem], bpy_struct):
    """Collection of point caches"""

    active_index: int | None
    """ 

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PolygonFloatProperties(
    bpy_prop_collection[MeshPolygonFloatPropertyLayer], bpy_struct
):
    """Collection of float properties"""

    def new(
        self, name: str | typing.Any = "Float Prop"
    ) -> MeshPolygonFloatPropertyLayer:
        """Add a float property layer to Mesh

        :param name: Float property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshPolygonFloatPropertyLayer
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PolygonIntProperties(
    bpy_prop_collection[MeshPolygonIntPropertyLayer], bpy_struct
):
    """Collection of int properties"""

    def new(self, name: str | typing.Any = "Int Prop") -> MeshPolygonIntPropertyLayer:
        """Add a integer property layer to Mesh

        :param name: Int property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshPolygonIntPropertyLayer
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PolygonStringProperties(
    bpy_prop_collection[MeshPolygonStringPropertyLayer], bpy_struct
):
    """Collection of string properties"""

    def new(
        self, name: str | typing.Any = "String Prop"
    ) -> MeshPolygonStringPropertyLayer:
        """Add a string property layer to Mesh

        :param name: String property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshPolygonStringPropertyLayer
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Pose(bpy_struct):
    """A collection of pose channels, including settings for animating bones"""

    animation_visualization: AnimViz
    """ Animation data for this data-block

    :type: AnimViz
    """

    bone_groups: BoneGroups
    """ Groups of the bones

    :type: BoneGroups
    """

    bones: bpy_prop_collection[PoseBone]
    """ Individual pose bones for the armature

    :type: bpy_prop_collection[PoseBone]
    """

    ik_param: IKParam
    """ Parameters for IK solver

    :type: IKParam
    """

    ik_solver: str
    """ Selection of IK solver for IK chain

    :type: str
    """

    use_auto_ik: bool
    """ Add temporary IK constraints while grabbing bones in Pose Mode

    :type: bool
    """

    use_mirror_relative: bool
    """ Apply relative transformations in X-mirror mode (not supported with Auto IK)

    :type: bool
    """

    use_mirror_x: bool
    """ Apply changes to matching bone on opposite side of X-Axis

    :type: bool
    """

    @classmethod
    def apply_pose_from_action(
        cls, action: Action | None, evaluation_time: typing.Any | None = 0.0
    ):
        """Apply the given action to this pose by evaluating it at a specific time. Only updates the pose of selected bones, or all bones if none are selected.

        :param action: Action, The Action containing the pose
        :type action: Action | None
        :param evaluation_time: Evaluation Time, Time at which the given action is evaluated to obtain the pose
        :type evaluation_time: typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PoseBone(bpy_struct):
    """Channel defining pose data for a bone in a Pose"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveinz: float
    """ Z-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutz: float
    """ Z-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_custom_handle_end: PoseBone
    """ Bone that serves as the end handle for the B-Bone curve

    :type: PoseBone
    """

    bbone_custom_handle_start: PoseBone
    """ Bone that serves as the start handle for the B-Bone curve

    :type: PoseBone
    """

    bbone_easein: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_easeout: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: bpy_prop_array[float]
    """ Scale factors for the start of the B-Bone, adjusts thickness (for tapering effects)

    :type: bpy_prop_array[float]
    """

    bbone_scaleout: bpy_prop_array[float]
    """ Scale factors for the end of the B-Bone, adjusts thickness (for tapering effects)

    :type: bpy_prop_array[float]
    """

    bone: Bone
    """ Bone associated with this PoseBone

    :type: Bone
    """

    bone_group: BoneGroup
    """ Bone group this pose channel belongs to

    :type: BoneGroup
    """

    bone_group_index: int
    """ Bone group this pose channel belongs to (0 means no group)

    :type: int
    """

    child: PoseBone
    """ Child of this pose bone

    :type: PoseBone
    """

    constraints: PoseBoneConstraints
    """ Constraints that act on this pose channel

    :type: PoseBoneConstraints
    """

    custom_shape: Object
    """ Object that defines custom display shape for this bone

    :type: Object
    """

    custom_shape_rotation_euler: bpy_prop_array[float]
    """ Adjust the rotation of the custom shape

    :type: bpy_prop_array[float]
    """

    custom_shape_scale_xyz: bpy_prop_array[float]
    """ Adjust the size of the custom shape

    :type: bpy_prop_array[float]
    """

    custom_shape_transform: PoseBone
    """ Bone that defines the display transform of this custom shape

    :type: PoseBone
    """

    custom_shape_translation: bpy_prop_array[float]
    """ Adjust the location of the custom shape

    :type: bpy_prop_array[float]
    """

    head: bpy_prop_array[float]
    """ Location of head of the channel's bone

    :type: bpy_prop_array[float]
    """

    ik_linear_weight: float
    """ Weight of scale constraint for IK

    :type: float
    """

    ik_max_x: float
    """ Maximum angles for IK Limit

    :type: float
    """

    ik_max_y: float
    """ Maximum angles for IK Limit

    :type: float
    """

    ik_max_z: float
    """ Maximum angles for IK Limit

    :type: float
    """

    ik_min_x: float
    """ Minimum angles for IK Limit

    :type: float
    """

    ik_min_y: float
    """ Minimum angles for IK Limit

    :type: float
    """

    ik_min_z: float
    """ Minimum angles for IK Limit

    :type: float
    """

    ik_rotation_weight: float
    """ Weight of rotation constraint for IK

    :type: float
    """

    ik_stiffness_x: float
    """ IK stiffness around the X axis

    :type: float
    """

    ik_stiffness_y: float
    """ IK stiffness around the Y axis

    :type: float
    """

    ik_stiffness_z: float
    """ IK stiffness around the Z axis

    :type: float
    """

    ik_stretch: float
    """ Allow scaling of the bone for IK

    :type: float
    """

    is_in_ik_chain: bool
    """ Is part of an IK chain

    :type: bool
    """

    length: float
    """ Length of the bone

    :type: float
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    lock_ik_x: bool
    """ Disallow movement around the X axis

    :type: bool
    """

    lock_ik_y: bool
    """ Disallow movement around the Y axis

    :type: bool
    """

    lock_ik_z: bool
    """ Disallow movement around the Z axis

    :type: bool
    """

    lock_location: list[bool]
    """ Lock editing of location when transforming

    :type: list[bool]
    """

    lock_rotation: list[bool]
    """ Lock editing of rotation when transforming

    :type: list[bool]
    """

    lock_rotation_w: bool
    """ Lock editing of 'angle' component of four-component rotations when transforming

    :type: bool
    """

    lock_rotations_4d: bool
    """ Lock editing of four component rotations by components (instead of as Eulers)

    :type: bool
    """

    lock_scale: list[bool]
    """ Lock editing of scale when transforming

    :type: list[bool]
    """

    matrix: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Final 4x4 matrix after constraints and drivers are applied (object space)

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    matrix_basis: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Alternative access to location/scale/rotation relative to the parent and own rest bone

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    matrix_channel: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ 4x4 matrix, before constraints

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    motion_path: MotionPath
    """ Motion Path for this element

    :type: MotionPath
    """

    name: str
    """ 

    :type: str
    """

    parent: PoseBone
    """ Parent of this pose bone

    :type: PoseBone
    """

    rotation_axis_angle: bpy_prop_array[float]
    """ Angle of Rotation for Axis-Angle rotation representation

    :type: bpy_prop_array[float]
    """

    rotation_euler: bpy_prop_array[float]
    """ Rotation in Eulers

    :type: bpy_prop_array[float]
    """

    rotation_mode: str
    """ 

    :type: str
    """

    rotation_quaternion: bpy_prop_array[float]
    """ Rotation in Quaternions

    :type: bpy_prop_array[float]
    """

    scale: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tail: bpy_prop_array[float]
    """ Location of tail of the channel's bone

    :type: bpy_prop_array[float]
    """

    use_custom_shape_bone_size: bool
    """ Scale the custom object by the bone length

    :type: bool
    """

    use_ik_limit_x: bool
    """ Limit movement around the X axis

    :type: bool
    """

    use_ik_limit_y: bool
    """ Limit movement around the Y axis

    :type: bool
    """

    use_ik_limit_z: bool
    """ Limit movement around the Z axis

    :type: bool
    """

    use_ik_linear_control: bool
    """ Apply channel size as IK constraint if stretching is enabled

    :type: bool
    """

    use_ik_rotation_control: bool
    """ Apply channel rotation as IK constraint

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.(readonly)"""

    children: typing.Any
    """ (readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.(readonly)"""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.(readonly)"""

    def evaluate_envelope(self, point: collections.abc.Iterable[float] | None) -> float:
        """Calculate bone envelope at given point

        :param point: Point, Position in 3d space to evaluate
        :type point: collections.abc.Iterable[float] | None
        :return: Factor, Envelope factor
        :rtype: float
        """
        ...

    def bbone_segment_matrix(
        self, index: int | None, rest: bool | typing.Any | None = False
    ) -> (
        list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
    ):
        """Retrieve the matrix of the joint between B-Bone segments if available

        :param index: Index of the segment endpoint
        :type index: int | None
        :param rest: Return the rest pose matrix
        :type rest: bool | typing.Any | None
        :return: The resulting matrix in bone local space
        :rtype: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
        """
        ...

    def compute_bbone_handles(
        self,
        rest: bool | typing.Any | None = False,
        ease: bool | typing.Any | None = False,
        offsets: bool | typing.Any | None = False,
    ):
        """Retrieve the vectors and rolls coming from B-Bone custom handles

                :param rest: Return the rest pose state
                :type rest: bool | typing.Any | None
                :param ease: Apply scale from ease values
                :type ease: bool | typing.Any | None
                :param offsets: Apply roll and curve offsets from bone properties
                :type offsets: bool | typing.Any | None
                :return: handle1, The direction vector of the start handle in bone local space, float array of 3 items in [-inf, inf]

        roll1, Roll of the start handle, float in [-inf, inf]

        handle2, The direction vector of the end handle in bone local space, float array of 3 items in [-inf, inf]

        roll2, Roll of the end handle, float in [-inf, inf]
        """
        ...

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """
        ...

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PoseBoneConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of pose bone constraints"""

    active: Constraint | None
    """ Active PoseChannel constraint

    :type: Constraint | None
    """

    def new(self, type: str | None) -> Constraint:
        """Add a constraint to this object

                :param type: Constraint type to add

        CAMERA_SOLVER
        Camera Solver.

        FOLLOW_TRACK
        Follow Track.

        OBJECT_SOLVER
        Object Solver.

        COPY_LOCATION
        Copy Location -- Copy the location of a target (with an optional offset), so that they move together.

        COPY_ROTATION
        Copy Rotation -- Copy the rotation of a target (with an optional offset), so that they rotate together.

        COPY_SCALE
        Copy Scale -- Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.

        COPY_TRANSFORMS
        Copy Transforms -- Copy all the transformations of a target, so that they move together.

        LIMIT_DISTANCE
        Limit Distance -- Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).

        LIMIT_LOCATION
        Limit Location -- Restrict movement along each axis within given ranges.

        LIMIT_ROTATION
        Limit Rotation -- Restrict rotation along each axis within given ranges.

        LIMIT_SCALE
        Limit Scale -- Restrict scaling along each axis with given ranges.

        MAINTAIN_VOLUME
        Maintain Volume -- Compensate for scaling one axis by applying suitable scaling to the other two axes.

        TRANSFORM
        Transformation -- Use one transform property from target to control another (or same) property on owner.

        TRANSFORM_CACHE
        Transform Cache -- Look up the transformation matrix from an external file.

        CLAMP_TO
        Clamp To -- Restrict movements to lie along a curve by remapping location along curve's longest axis.

        DAMPED_TRACK
        Damped Track -- Point towards a target by performing the smallest rotation necessary.

        IK
        Inverse Kinematics -- Control a chain of bones by specifying the endpoint target (Bones only).

        LOCKED_TRACK
        Locked Track -- Rotate around the specified ('locked') axis to point towards a target.

        SPLINE_IK
        Spline IK -- Align chain of bones along a curve (Bones only).

        STRETCH_TO
        Stretch To -- Stretch along Y-Axis to point towards a target.

        TRACK_TO
        Track To -- Legacy tracking constraint prone to twisting artifacts.

        ACTION
        Action -- Use transform property of target to look up pose for owner from an Action.

        ARMATURE
        Armature -- Apply weight-blended transformation from multiple bones like the Armature modifier.

        CHILD_OF
        Child Of -- Make target the 'detachable' parent of owner.

        FLOOR
        Floor -- Use position (and optionally rotation) of target to define a 'wall' or 'floor' that the owner can not cross.

        FOLLOW_PATH
        Follow Path -- Use to animate an object/bone following a path.

        PIVOT
        Pivot -- Change pivot point for transforms (buggy).

        SHRINKWRAP
        Shrinkwrap -- Restrict movements to surface of target mesh.
                :type type: str | None
                :return: New constraint
                :rtype: Constraint
        """
        ...

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a constraint to a different position

        :param from_index: From Index, Index to move
        :type from_index: int | None
        :param to_index: To Index, Target index
        :type to_index: int | None
        """
        ...

    def copy(self, constraint: Constraint) -> Constraint:
        """Add a new constraint that is a copy of the given one

        :param constraint: Constraint to copy - may belong to a different object
        :type constraint: Constraint
        :return: New constraint
        :rtype: Constraint
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Preferences(bpy_struct):
    """Global preferences"""

    active_section: str | None
    """ Active section of the preferences shown in the user interface

    :type: str | None
    """

    addons: Addons
    """ 

    :type: Addons
    """

    app_template: str
    """ 

    :type: str
    """

    apps: PreferencesApps
    """ Preferences that work only for apps

    :type: PreferencesApps
    """

    autoexec_paths: PathCompareCollection
    """ 

    :type: PathCompareCollection
    """

    edit: PreferencesEdit
    """ Settings for interacting with Blender data

    :type: PreferencesEdit
    """

    experimental: PreferencesExperimental
    """ Settings for features that are still early in their development stage

    :type: PreferencesExperimental
    """

    filepaths: PreferencesFilePaths
    """ Default paths for external files

    :type: PreferencesFilePaths
    """

    inputs: PreferencesInput
    """ Settings for input devices

    :type: PreferencesInput
    """

    is_dirty: bool
    """ Preferences have changed

    :type: bool
    """

    keymap: PreferencesKeymap
    """ Shortcut setup for keyboards and other input devices

    :type: PreferencesKeymap
    """

    studio_lights: StudioLights
    """ 

    :type: StudioLights
    """

    system: PreferencesSystem
    """ Graphics driver and operating system settings

    :type: PreferencesSystem
    """

    themes: bpy_prop_collection[Theme]
    """ 

    :type: bpy_prop_collection[Theme]
    """

    ui_styles: bpy_prop_collection[ThemeStyle]
    """ 

    :type: bpy_prop_collection[ThemeStyle]
    """

    use_preferences_save: bool
    """ Save preferences on exit when modified (unless factory settings have been loaded)

    :type: bool
    """

    version: bpy_prop_array[int]
    """ Version of Blender the userpref.blend was saved with

    :type: bpy_prop_array[int]
    """

    view: PreferencesView
    """ Preferences related to viewing data

    :type: PreferencesView
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesApps(bpy_struct):
    """Preferences that work only for apps"""

    show_corner_split: bool
    """ Split and join editors by dragging from corners

    :type: bool
    """

    show_edge_resize: bool
    """ Resize editors by dragging from the edges

    :type: bool
    """

    show_regions_visibility_toggle: bool
    """ Header and side bars visibility toggles

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesEdit(bpy_struct):
    """Settings for interacting with Blender data"""

    auto_keying_mode: str
    """ Mode of automatic keyframe insertion for Objects and Bones (default setting used for new Scenes)

    :type: str
    """

    collection_instance_empty_size: float
    """ Display size of the empty when new collection instances are created

    :type: float
    """

    fcurve_new_auto_smoothing: str
    """ Auto Handle Smoothing mode used for newly added F-Curves

    :type: str
    """

    fcurve_unselected_alpha: float
    """ The opacity of unselected F-Curves against the background of the Graph Editor

    :type: float
    """

    grease_pencil_default_color: bpy_prop_array[float]
    """ Color of new annotation layers

    :type: bpy_prop_array[float]
    """

    grease_pencil_eraser_radius: int
    """ Radius of eraser 'brush'

    :type: int
    """

    grease_pencil_euclidean_distance: int
    """ Distance moved by mouse when drawing stroke to include

    :type: int
    """

    grease_pencil_manhattan_distance: int
    """ Pixels moved by mouse per axis when drawing stroke

    :type: int
    """

    keyframe_new_handle_type: str
    """ Handle type for handles of new keyframes

    :type: str
    """

    keyframe_new_interpolation_type: str
    """ Interpolation mode used for first keyframe on newly added F-Curves (subsequent keyframes take interpolation from preceding keyframe)

    :type: str
    """

    material_link: str
    """ Toggle whether the material is linked to object data or the object block

    :type: str
    """

    node_margin: int
    """ Minimum distance between nodes for Auto-offsetting nodes

    :type: int
    """

    object_align: str
    """ The default alignment for objects added from a 3D viewport menu

    :type: str
    """

    sculpt_paint_overlay_color: bpy_prop_array[float]
    """ Color of texture overlay

    :type: bpy_prop_array[float]
    """

    undo_memory_limit: int
    """ Maximum memory usage in megabytes (0 means unlimited)

    :type: int
    """

    undo_steps: int
    """ Number of undo steps available (smaller values conserve memory)

    :type: int
    """

    use_anim_channel_group_colors: bool
    """ Use animation channel group colors; generally this is used to show bone group colors

    :type: bool
    """

    use_auto_keying: bool
    """ Automatic keyframe insertion for Objects and Bones (default setting used for new Scenes)

    :type: bool
    """

    use_auto_keying_warning: bool
    """ Show warning indicators when transforming objects and bones if auto keying is enabled

    :type: bool
    """

    use_cursor_lock_adjust: bool
    """ Place the cursor without 'jumping' to the new location (when lock-to-cursor is used)

    :type: bool
    """

    use_duplicate_action: bool
    """ Causes actions to be duplicated with the data-blocks

    :type: bool
    """

    use_duplicate_armature: bool
    """ Causes armature data to be duplicated with the object

    :type: bool
    """

    use_duplicate_camera: bool
    """ Causes camera data to be duplicated with the object

    :type: bool
    """

    use_duplicate_curve: bool
    """ Causes curve data to be duplicated with the object

    :type: bool
    """

    use_duplicate_curves: bool
    """ Causes curves data to be duplicated with the object

    :type: bool
    """

    use_duplicate_grease_pencil: bool
    """ Causes grease pencil data to be duplicated with the object

    :type: bool
    """

    use_duplicate_lattice: bool
    """ Causes lattice data to be duplicated with the object

    :type: bool
    """

    use_duplicate_light: bool
    """ Causes light data to be duplicated with the object

    :type: bool
    """

    use_duplicate_lightprobe: bool
    """ Causes light probe data to be duplicated with the object

    :type: bool
    """

    use_duplicate_material: bool
    """ Causes material data to be duplicated with the object

    :type: bool
    """

    use_duplicate_mesh: bool
    """ Causes mesh data to be duplicated with the object

    :type: bool
    """

    use_duplicate_metaball: bool
    """ Causes metaball data to be duplicated with the object

    :type: bool
    """

    use_duplicate_particle: bool
    """ Causes particle systems to be duplicated with the object

    :type: bool
    """

    use_duplicate_pointcloud: bool
    """ Causes point cloud data to be duplicated with the object

    :type: bool
    """

    use_duplicate_speaker: bool
    """ Causes speaker data to be duplicated with the object

    :type: bool
    """

    use_duplicate_surface: bool
    """ Causes surface data to be duplicated with the object

    :type: bool
    """

    use_duplicate_text: bool
    """ Causes text data to be duplicated with the object

    :type: bool
    """

    use_duplicate_volume: bool
    """ Causes volume data to be duplicated with the object

    :type: bool
    """

    use_enter_edit_mode: bool
    """ Enter edit mode automatically after adding a new object

    :type: bool
    """

    use_global_undo: bool
    """ Global undo works by keeping a full copy of the file itself in memory, so takes extra memory

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) and also Color is based on the transform axis

    :type: bool
    """

    use_keyframe_insert_available: bool
    """ Automatic keyframe insertion in available F-Curves

    :type: bool
    """

    use_keyframe_insert_needed: bool
    """ Keyframe insertion only when keyframe needed

    :type: bool
    """

    use_mouse_depth_cursor: bool
    """ Use the surface depth for cursor placement

    :type: bool
    """

    use_negative_frames: bool
    """ Current frame number can be manually set to a negative value

    :type: bool
    """

    use_text_edit_auto_close: bool
    """ Automatically close relevant character pairs when typing in the text editor

    :type: bool
    """

    use_visual_keying: bool
    """ Use Visual keying automatically for constrained objects

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesExperimental(bpy_struct):
    """Experimental features"""

    enable_eevee_next: bool
    """ Enable the new EEVEE codebase, requires restart

    :type: bool
    """

    override_auto_resync: bool
    """ Enable library overrides automatic resync detection and process on file load. Disable when dealing with older .blend files that need manual Resync (Enforce) handling

    :type: bool
    """

    show_asset_debug_info: bool
    """ Enable some extra fields in the Asset Browser to aid in debugging

    :type: bool
    """

    use_asset_indexing: bool
    """ Disabling the asset indexer forces every asset library refresh to completely reread assets from disk

    :type: bool
    """

    use_cycles_debug: bool
    """ Enable Cycles debugging options for developers

    :type: bool
    """

    use_draw_manager_acquire_lock: bool
    """ Don't lock UI during background rendering

    :type: bool
    """

    use_extended_asset_browser: bool
    """ Enable Asset Browser editor and operators to manage regular data-blocks as assets, not just poses

    :type: bool
    """

    use_full_frame_compositor: bool
    """ Enable compositor full frame execution mode option (no tiling, reduces execution time and memory usage)

    :type: bool
    """

    use_new_curves_type: bool
    """ Enable the new curves data type in the UI

    :type: bool
    """

    use_new_point_cloud_type: bool
    """ Enable the new point cloud type in the ui

    :type: bool
    """

    use_override_templates: bool
    """ Enable library override template in the python API

    :type: bool
    """

    use_sculpt_texture_paint: bool
    """ Use texture painting in Sculpt Mode

    :type: bool
    """

    use_sculpt_tools_tilt: bool
    """ Support for pen tablet tilt events in Sculpt Mode

    :type: bool
    """

    use_undo_legacy: bool
    """ Use legacy undo (slower than the new default one, but may be more stable in some cases)

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesFilePaths(bpy_struct):
    """Default paths for external files"""

    animation_player: str
    """ Path to a custom animation/frame sequence player

    :type: str
    """

    animation_player_preset: str
    """ Preset configs for external animation players

    :type: str
    """

    asset_libraries: bpy_prop_collection[UserAssetLibrary]
    """ 

    :type: bpy_prop_collection[UserAssetLibrary]
    """

    auto_save_time: int
    """ The time (in minutes) to wait between automatic temporary saves

    :type: int
    """

    file_preview_type: str
    """ What type of blend preview to create

    :type: str
    """

    font_directory: str
    """ The default directory to search for loading fonts

    :type: str
    """

    i18n_branches_directory: str
    """ The path to the '/branches' directory of your local svn-translation copy, to allow translating from the UI

    :type: str
    """

    image_editor: str
    """ Path to an image editor

    :type: str
    """

    recent_files: int
    """ Maximum number of recently opened files to remember

    :type: int
    """

    render_cache_directory: str
    """ Where to cache raw render results

    :type: str
    """

    render_output_directory: str
    """ The default directory for rendering output, for new scenes

    :type: str
    """

    save_version: int
    """ The number of old versions to maintain in the current directory, when manually saving

    :type: int
    """

    script_directory: str
    """ Alternate script path, matching the default layout with subdirectories: startup, add-ons, modules, and presets (requires restart)

    :type: str
    """

    show_hidden_files_datablocks: bool
    """ Show files and data-blocks that are normally hidden

    :type: bool
    """

    show_recent_locations: bool
    """ Show Recent locations list in the File Browser

    :type: bool
    """

    show_system_bookmarks: bool
    """ Show System locations list in the File Browser

    :type: bool
    """

    sound_directory: str
    """ The default directory to search for sounds

    :type: str
    """

    temporary_directory: str
    """ The directory for storing temporary save files

    :type: str
    """

    texture_directory: str
    """ The default directory to search for textures

    :type: str
    """

    use_auto_save_temporary_files: bool
    """ Automatic saving of temporary files in temp directory, uses process ID.
Warning: Sculpt and edit mode data won't be saved

    :type: bool
    """

    use_file_compression: bool
    """ Enable file compression when saving .blend files

    :type: bool
    """

    use_filter_files: bool
    """ Enable filtering of files in the File Browser

    :type: bool
    """

    use_load_ui: bool
    """ Load user interface setup when loading .blend files

    :type: bool
    """

    use_relative_paths: bool
    """ Default relative path option for the file selector, when no path is defined yet

    :type: bool
    """

    use_scripts_auto_execute: bool
    """ Allow any .blend file to run scripts automatically (unsafe with blend files from an untrusted source)

    :type: bool
    """

    use_tabs_as_spaces: bool
    """ Automatically convert all new tabs into spaces for new and loaded text files

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesInput(bpy_struct):
    """Settings for input devices"""

    drag_threshold: int
    """ Number of pixels to drag before a drag event is triggered for keyboard and other non mouse/tablet input (otherwise click events are detected)

    :type: int
    """

    drag_threshold_mouse: int
    """ Number of pixels to drag before a drag event is triggered for mouse/track-pad input (otherwise click events are detected)

    :type: int
    """

    drag_threshold_tablet: int
    """ Number of pixels to drag before a drag event is triggered for tablet input (otherwise click events are detected)

    :type: int
    """

    invert_mouse_zoom: bool
    """ Invert the axis of mouse movement for zooming

    :type: bool
    """

    invert_zoom_wheel: bool
    """ Swap the Mouse Wheel zoom direction

    :type: bool
    """

    mouse_double_click_time: int
    """ Time/delay (in ms) for a double click

    :type: int
    """

    mouse_emulate_3_button_modifier: str
    """ Hold this modifier to emulate the middle mouse button

    :type: str
    """

    move_threshold: int
    """ Number of pixels to before the cursor is considered to have moved (used for cycling selected items on successive clicks)

    :type: int
    """

    navigation_mode: str
    """ Which method to use for viewport navigation

    :type: str
    """

    ndof_deadzone: float
    """ Threshold of initial movement needed from the device's rest position

    :type: float
    """

    ndof_fly_helicopter: bool
    """ Device up/down directly controls the Z position of the 3D viewport

    :type: bool
    """

    ndof_lock_camera_pan_zoom: bool
    """ Pan/zoom the camera view instead of leaving the camera view when orbiting

    :type: bool
    """

    ndof_lock_horizon: bool
    """ Keep horizon level while flying with 3D Mouse

    :type: bool
    """

    ndof_orbit_sensitivity: float
    """ Overall sensitivity of the 3D Mouse for orbiting

    :type: float
    """

    ndof_pan_yz_swap_axis: bool
    """ Pan using up/down on the device (otherwise forward/backward)

    :type: bool
    """

    ndof_panx_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_pany_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_panz_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_rotx_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_roty_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_rotz_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_sensitivity: float
    """ Overall sensitivity of the 3D Mouse for panning

    :type: float
    """

    ndof_show_guide: bool
    """ Display the center and axis during rotation

    :type: bool
    """

    ndof_view_navigate_method: str
    """ Navigation style in the viewport

    :type: str
    """

    ndof_view_rotate_method: str
    """ Rotation style in the viewport

    :type: str
    """

    ndof_zoom_invert: bool
    """ Zoom using opposite direction

    :type: bool
    """

    pressure_softness: float
    """ Adjusts softness of the low pressure response onset using a gamma curve

    :type: float
    """

    pressure_threshold_max: float
    """ Raw input pressure value that is interpreted as 100% by Blender

    :type: float
    """

    tablet_api: str
    """ Select the tablet API to use for pressure sensitivity (may require restarting Blender for changes to take effect)

    :type: str
    """

    use_auto_perspective: bool
    """ Automatically switch between orthographic and perspective when changing from top/front/side views

    :type: bool
    """

    use_drag_immediately: bool
    """ Moving things with a mouse drag confirms when releasing the button

    :type: bool
    """

    use_emulate_numpad: bool
    """ Main 1 to 0 keys act as the numpad ones (useful for laptops)

    :type: bool
    """

    use_mouse_continuous: bool
    """ Let the mouse wrap around the view boundaries so mouse movements are not limited by the screen size (used by transform, dragging of UI controls, etc.)

    :type: bool
    """

    use_mouse_depth_navigate: bool
    """ Use the depth under the mouse to improve view pan/rotate/zoom functionality

    :type: bool
    """

    use_mouse_emulate_3_button: bool
    """ Emulate Middle Mouse with Alt+Left Mouse

    :type: bool
    """

    use_ndof: bool
    """ 

    :type: bool
    """

    use_numeric_input_advanced: bool
    """ When entering numbers while transforming, default to advanced mode for full math expression evaluation

    :type: bool
    """

    use_rotate_around_active: bool
    """ Use selection as the pivot point

    :type: bool
    """

    use_zoom_to_mouse: bool
    """ Zoom in towards the mouse pointer's position in the 3D view, rather than the 2D window center

    :type: bool
    """

    view_rotate_method: str
    """ Orbit method in the viewport

    :type: str
    """

    view_rotate_sensitivity_trackball: float
    """ Scale trackball orbit sensitivity

    :type: float
    """

    view_rotate_sensitivity_turntable: float
    """ Rotation amount per pixel to control how fast the viewport orbits

    :type: float
    """

    view_zoom_axis: str
    """ Axis of mouse movement to zoom in or out on

    :type: str
    """

    view_zoom_method: str
    """ Which style to use for viewport scaling

    :type: str
    """

    walk_navigation: WalkNavigation
    """ Settings for walk navigation mode

    :type: WalkNavigation
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesKeymap(bpy_struct):
    """Shortcut setup for keyboards and other input devices"""

    active_keyconfig: str
    """ The name of the active key configuration

    :type: str
    """

    show_ui_keyconfig: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesSystem(bpy_struct):
    """Graphics driver and operating system settings"""

    anisotropic_filter: str
    """ Quality of anisotropic filtering

    :type: str
    """

    audio_channels: str
    """ Audio channel count

    :type: str
    """

    audio_device: str
    """ Audio output device

    :type: str
    """

    audio_mixing_buffer: str
    """ Number of samples used by the audio mixing buffer

    :type: str
    """

    audio_sample_format: str
    """ Audio sample format

    :type: str
    """

    audio_sample_rate: str
    """ Audio sample rate

    :type: str
    """

    dpi: int
    """ 

    :type: int
    """

    gl_clip_alpha: float
    """ Clip alpha below this threshold in the 3D textured view

    :type: float
    """

    gl_texture_limit: str
    """ Limit the texture size to save graphics memory

    :type: str
    """

    image_draw_method: str
    """ Method used for displaying images on the screen

    :type: str
    """

    legacy_compute_device_type: int
    """ For backwards compatibility only

    :type: int
    """

    light_ambient: bpy_prop_array[float]
    """ Color of the ambient light that uniformly lit the scene

    :type: bpy_prop_array[float]
    """

    memory_cache_limit: int
    """ Memory cache limit (in megabytes)

    :type: int
    """

    opensubdiv_compute_type: str
    """ Type of computer back-end used with OpenSubdiv

    :type: str
    """

    pixel_size: float
    """ 

    :type: float
    """

    scrollback: int
    """ Maximum number of lines to store for the console buffer

    :type: int
    """

    sequencer_disk_cache_compression: str
    """ Smaller compression will result in larger files, but less decoding overhead

    :type: str
    """

    sequencer_disk_cache_dir: str
    """ Override default directory

    :type: str
    """

    sequencer_disk_cache_size_limit: int
    """ Disk cache limit (in gigabytes)

    :type: int
    """

    sequencer_proxy_setup: str
    """ When and how proxies are created

    :type: str
    """

    solid_lights: bpy_prop_collection[UserSolidLight]
    """ Lights used to display objects in solid shading mode

    :type: bpy_prop_collection[UserSolidLight]
    """

    texture_collection_rate: int
    """ Number of seconds between each run of the GL texture garbage collector

    :type: int
    """

    texture_time_out: int
    """ Time since last access of a GL texture in seconds after which it is freed (set to 0 to keep textures allocated)

    :type: int
    """

    ui_line_width: float
    """ Suggested line thickness and point size in pixels, for add-ons displaying custom user interface elements, based on operating system settings and Blender UI scale

    :type: float
    """

    ui_scale: float
    """ Size multiplier to use when displaying custom user interface elements, so that they are scaled correctly on screens with different DPI. This value is based on operating system DPI settings and Blender display scale

    :type: float
    """

    use_edit_mode_smooth_wire: bool
    """ Enable edit mode edge smoothing, reducing aliasing (requires restart)

    :type: bool
    """

    use_gpu_subdivision: bool
    """ Enable GPU acceleration for evaluating the last subdivision surface modifiers in the stack

    :type: bool
    """

    use_overlay_smooth_wire: bool
    """ Enable overlay smooth wires, reducing aliasing

    :type: bool
    """

    use_region_overlap: bool
    """ Display tool/property regions over the main region

    :type: bool
    """

    use_select_pick_depth: bool
    """ When making a selection in 3D View, use the GPU depth buffer to ensure the frontmost object is selected first

    :type: bool
    """

    use_sequencer_disk_cache: bool
    """ Store cached images to disk

    :type: bool
    """

    use_studio_light_edit: bool
    """ View the result of the studio light editor in the viewport

    :type: bool
    """

    vbo_collection_rate: int
    """ Number of seconds between each run of the GL Vertex buffer object garbage collector

    :type: int
    """

    vbo_time_out: int
    """ Time since last access of a GL Vertex buffer object in seconds after which it is freed (set to 0 to keep vbo allocated)

    :type: int
    """

    viewport_aa: str
    """ Method of anti-aliasing in 3d viewport

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PreferencesView(bpy_struct):
    """Preferences related to viewing data"""

    color_picker_type: str
    """ Different styles of displaying the color picker widget

    :type: str
    """

    factor_display_type: str
    """ How factor values are displayed

    :type: str
    """

    filebrowser_display_type: str
    """ Default location where the File Editor will be displayed in

    :type: str
    """

    font_path_ui: str
    """ Path to interface font

    :type: str
    """

    font_path_ui_mono: str
    """ Path to interface monospaced Font

    :type: str
    """

    gizmo_size: int
    """ Diameter of the gizmo

    :type: int
    """

    gizmo_size_navigate_v3d: int
    """ The Navigate Gizmo size

    :type: int
    """

    header_align: str
    """ Default header position for new space-types

    :type: str
    """

    language: str
    """ Language used for translation

    :type: str
    """

    lookdev_sphere_size: int
    """ Diameter of the HDRI preview spheres

    :type: int
    """

    mini_axis_brightness: int
    """ Brightness of the icon

    :type: int
    """

    mini_axis_size: int
    """ The axes icon's size

    :type: int
    """

    mini_axis_type: str
    """ Show a small rotating 3D axes in the top right corner of the 3D viewport

    :type: str
    """

    open_sublevel_delay: int
    """ Time delay in 1/10 seconds before automatically opening sub level menus

    :type: int
    """

    open_toplevel_delay: int
    """ Time delay in 1/10 seconds before automatically opening top level menus

    :type: int
    """

    pie_animation_timeout: int
    """ Time needed to fully animate the pie to unfolded state (in 1/100ths of sec)

    :type: int
    """

    pie_initial_timeout: int
    """ Pie menus will use the initial mouse position as center for this amount of time (in 1/100ths of sec)

    :type: int
    """

    pie_menu_confirm: int
    """ Distance threshold after which selection is made (zero to disable)

    :type: int
    """

    pie_menu_radius: int
    """ Pie menu size in pixels

    :type: int
    """

    pie_menu_threshold: int
    """ Distance from center needed before a selection can be made

    :type: int
    """

    pie_tap_timeout: int
    """ Pie menu button held longer than this will dismiss menu on release.(in 1/100ths of sec)

    :type: int
    """

    render_display_type: str
    """ Default location where rendered images will be displayed in

    :type: str
    """

    rotation_angle: float
    """ Rotation step for numerical pad keys (2 4 6 8)

    :type: float
    """

    show_addons_enabled_only: bool
    """ Only show enabled add-ons. Un-check to see all installed add-ons

    :type: bool
    """

    show_column_layout: bool
    """ Use a column layout for toolbox

    :type: bool
    """

    show_developer_ui: bool
    """ Show options for developers (edit source in context menu, geometry indices)

    :type: bool
    """

    show_gizmo: bool
    """ Use transform gizmos by default

    :type: bool
    """

    show_navigate_ui: bool
    """ Show navigation controls in 2D and 3D views which do not have scroll bars

    :type: bool
    """

    show_object_info: bool
    """ Include the name of the active object and the current frame number in the text info overlay

    :type: bool
    """

    show_playback_fps: bool
    """ Include the number of frames displayed per second in the text info overlay while animation is played back

    :type: bool
    """

    show_splash: bool
    """ Display splash screen on startup

    :type: bool
    """

    show_statusbar_memory: bool
    """ Show Blender memory usage

    :type: bool
    """

    show_statusbar_stats: bool
    """ Show scene statistics

    :type: bool
    """

    show_statusbar_version: bool
    """ Show Blender version string

    :type: bool
    """

    show_statusbar_vram: bool
    """ Show GPU video memory usage

    :type: bool
    """

    show_tooltips: bool
    """ Display tooltips (when disabled, hold Alt to force display)

    :type: bool
    """

    show_tooltips_python: bool
    """ Show Python references in tooltips

    :type: bool
    """

    show_view_name: bool
    """ Include the name of the view orientation in the text info overlay

    :type: bool
    """

    smooth_view: int
    """ Time to animate the view in milliseconds, zero to disable

    :type: int
    """

    text_hinting: str
    """ Method for making user interface text render sharp

    :type: str
    """

    timecode_style: str
    """ Format of Time Codes displayed when not displaying timing in terms of frames

    :type: str
    """

    ui_line_width: str
    """ Changes the thickness of widget outlines, lines and dots in the interface

    :type: str
    """

    ui_scale: float
    """ Changes the size of the fonts and widgets in the interface

    :type: float
    """

    use_directional_menus: bool
    """ Otherwise menus, etc will always be top to bottom, left to right, no matter opening direction

    :type: bool
    """

    use_mouse_over_open: bool
    """ Open menu buttons and pulldowns automatically when the mouse is hovering

    :type: bool
    """

    use_save_prompt: bool
    """ Ask for confirmation when quitting with unsaved changes

    :type: bool
    """

    use_text_antialiasing: bool
    """ Smooth jagged edges of user interface text

    :type: bool
    """

    use_translate_interface: bool
    """ Translate all labels in menus, buttons and panels (note that this might make it hard to follow tutorials or the manual)

    :type: bool
    """

    use_translate_new_dataname: bool
    """ Translate the names of new data-blocks (objects, materials...)

    :type: bool
    """

    use_translate_tooltips: bool
    """ Translate the descriptions when hovering UI elements (recommended)

    :type: bool
    """

    use_weight_color_range: bool
    """ Enable color range used for weight visualization in weight painting mode

    :type: bool
    """

    view2d_grid_spacing_min: int
    """ Minimum number of pixels between each gridline in 2D Viewports

    :type: int
    """

    view_frame_keyframes: int
    """ Keyframes around cursor that we zoom around

    :type: int
    """

    view_frame_seconds: float
    """ Seconds around cursor that we zoom around

    :type: float
    """

    view_frame_type: str
    """ How zooming to frame focuses around current frame

    :type: str
    """

    weight_color_range: ColorRamp
    """ Color range used for weight visualization in weight painting mode

    :type: ColorRamp
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Property(bpy_struct):
    """RNA property definition"""

    description: str
    """ Description of the property for tooltips

    :type: str
    """

    icon: str
    """ Icon of the item

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    is_animatable: bool
    """ Property is animatable through RNA

    :type: bool
    """

    is_argument_optional: bool
    """ True when the property is optional in a Python function implementing an RNA function

    :type: bool
    """

    is_enum_flag: bool
    """ True when multiple enums

    :type: bool
    """

    is_hidden: bool
    """ True when the property is hidden

    :type: bool
    """

    is_library_editable: bool
    """ Property is editable from linked instances (changes not saved)

    :type: bool
    """

    is_never_none: bool
    """ True when this value can't be set to None

    :type: bool
    """

    is_output: bool
    """ True when this property is an output value from an RNA function

    :type: bool
    """

    is_overridable: bool
    """ Property is overridable through RNA

    :type: bool
    """

    is_readonly: bool
    """ Property is editable through RNA

    :type: bool
    """

    is_registered: bool
    """ Property is registered as part of type registration

    :type: bool
    """

    is_registered_optional: bool
    """ Property is optionally registered as part of type registration

    :type: bool
    """

    is_required: bool
    """ False when this property is an optional argument in an RNA function

    :type: bool
    """

    is_runtime: bool
    """ Property has been dynamically created at runtime

    :type: bool
    """

    is_skip_save: bool
    """ True when the property is not saved in presets

    :type: bool
    """

    name: str
    """ Human readable name

    :type: str
    """

    srna: Struct
    """ Struct definition used for properties assigned to this item

    :type: Struct
    """

    subtype: str
    """ Semantic interpretation of the property

    :type: str
    """

    tags: set[str]
    """ Subset of tags (defined in parent struct) that are set for this property

    :type: set[str]
    """

    translation_context: str
    """ Translation context of the property's name

    :type: str
    """

    type: str
    """ Data type of the property

    :type: str
    """

    unit: str
    """ Type of units for this property

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PropertyGroup(bpy_struct):
    """Group of ID properties"""

    name: str
    """ Unique name used in the code and scripting

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class PropertyGroupItem(bpy_struct):
    """Property that stores arbitrary, user defined properties"""

    collection: bpy_prop_collection[PropertyGroup]
    """ 

    :type: bpy_prop_collection[PropertyGroup]
    """

    double: float
    """ 

    :type: float
    """

    double_array: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    float: float
    """ 

    :type: float
    """

    float_array: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    group: PropertyGroup
    """ 

    :type: PropertyGroup
    """

    id: ID
    """ 

    :type: ID
    """

    idp_array: bpy_prop_collection[PropertyGroup]
    """ 

    :type: bpy_prop_collection[PropertyGroup]
    """

    int: int
    """ 

    :type: int
    """

    int_array: bpy_prop_array[int]
    """ 

    :type: bpy_prop_array[int]
    """

    string: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Region(bpy_struct):
    """Region in a subdivided screen area"""

    alignment: str
    """ Alignment of the region within the area

    :type: str
    """

    data: typing.Any
    """ Region specific data (the type depends on the region type)

    :type: typing.Any
    """

    height: int
    """ Region height

    :type: int
    """

    type: str
    """ Type of this region

    :type: str
    """

    view2d: View2D
    """ 2D view of the region

    :type: View2D
    """

    width: int
    """ Region width

    :type: int
    """

    x: int
    """ The window relative vertical location of the region

    :type: int
    """

    y: int
    """ The window relative horizontal location of the region

    :type: int
    """

    def tag_redraw(self):
        """tag_redraw"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RegionView3D(bpy_struct):
    """3D View region data"""

    clip_planes: list[list[float]] | tuple[
        tuple[float, float, float, float, float, float],
        tuple[float, float, float, float, float, float],
        tuple[float, float, float, float, float, float],
        tuple[float, float, float, float, float, float],
    ]
    """ 

    :type: list[list[float]] | tuple[tuple[float, float, float, float, float, float], tuple[float, float, float, float, float, float], tuple[float, float, float, float, float, float], tuple[float, float, float, float, float, float]]
    """

    is_orthographic_side_view: bool
    """ Is current view aligned to an axis (does not check the view is orthographic use "is_perspective" for that). Assignment sets the "view_rotation" to the closest axis aligned view

    :type: bool
    """

    is_perspective: bool
    """ 

    :type: bool
    """

    lock_rotation: bool
    """ Lock view rotation of side views to Top/Front/Right

    :type: bool
    """

    perspective_matrix: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Current perspective matrix (window_matrix * view_matrix)

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    show_sync_view: bool
    """ Sync view position between side views

    :type: bool
    """

    use_box_clip: bool
    """ Clip view contents based on what is visible in other side views

    :type: bool
    """

    use_clip_planes: bool
    """ 

    :type: bool
    """

    view_camera_offset: bpy_prop_array[float]
    """ View shift in camera view

    :type: bpy_prop_array[float]
    """

    view_camera_zoom: float
    """ Zoom factor in camera view

    :type: float
    """

    view_distance: float
    """ Distance to the view location

    :type: float
    """

    view_location: bpy_prop_array[float]
    """ View pivot location

    :type: bpy_prop_array[float]
    """

    view_matrix: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Current view matrix

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    view_perspective: str
    """ View Perspective

    :type: str
    """

    view_rotation: bpy_prop_array[float]
    """ Rotation in quaternions (keep normalized)

    :type: bpy_prop_array[float]
    """

    window_matrix: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ Current window matrix

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    def update(self):
        """Recalculate the view matrices"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderEngine(bpy_struct):
    """Render engine"""

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_use_alembic_procedural: bool
    """ Support loading Alembic data at render time

    :type: bool
    """

    bl_use_custom_freestyle: bool
    """ Handles freestyle rendering on its own, instead of delegating it to EEVEE

    :type: bool
    """

    bl_use_eevee_viewport: bool
    """ Uses Eevee for viewport shading in LookDev shading mode

    :type: bool
    """

    bl_use_gpu_context: bool
    """ Enable OpenGL context for the render method, for engines that render using OpenGL

    :type: bool
    """

    bl_use_image_save: bool
    """ Save images/movie to disk while rendering an animation. Disabling image saving is only supported when bl_use_postprocess is also disabled

    :type: bool
    """

    bl_use_postprocess: bool
    """ Apply compositing on render results

    :type: bool
    """

    bl_use_preview: bool
    """ Render engine supports being used for rendering previews of materials, lights and worlds

    :type: bool
    """

    bl_use_shading_nodes_custom: bool
    """ Don't expose Cycles and Eevee shading nodes in the node editor user interface, so own nodes can be used instead

    :type: bool
    """

    bl_use_spherical_stereo: bool
    """ Support spherical stereo camera models

    :type: bool
    """

    bl_use_stereo_viewport: bool
    """ Support rendering stereo 3D viewport

    :type: bool
    """

    camera_override: Object
    """ 

    :type: Object
    """

    is_animation: bool
    """ 

    :type: bool
    """

    is_preview: bool
    """ 

    :type: bool
    """

    layer_override: list[bool]
    """ 

    :type: list[bool]
    """

    render: RenderSettings
    """ 

    :type: RenderSettings
    """

    resolution_x: int
    """ 

    :type: int
    """

    resolution_y: int
    """ 

    :type: int
    """

    temporary_directory: str
    """ 

    :type: str
    """

    use_highlight_tiles: bool
    """ 

    :type: bool
    """

    def update(self, data: BlendData | None = None, depsgraph: Depsgraph | None = None):
        """Export scene data for render

        :param data:
        :type data: BlendData | None
        :param depsgraph:
        :type depsgraph: Depsgraph | None
        """
        ...

    def render(self, depsgraph: Depsgraph | None):
        """Render scene into an image

        :param depsgraph:
        :type depsgraph: Depsgraph | None
        """
        ...

    def render_frame_finish(self):
        """Perform finishing operations after all view layers in a frame were rendered"""
        ...

    def draw(self, context: Context | None, depsgraph: Depsgraph | None):
        """Draw render image

        :param context:
        :type context: Context | None
        :param depsgraph:
        :type depsgraph: Depsgraph | None
        """
        ...

    def bake(
        self,
        depsgraph: Depsgraph | None,
        object: Object | None,
        pass_type: str | None,
        pass_filter: int | None,
        width: int | None,
        height: int | None,
    ):
        """Bake passes

        :param depsgraph:
        :type depsgraph: Depsgraph | None
        :param object:
        :type object: Object | None
        :param pass_type: Pass, Pass to bake
        :type pass_type: str | None
        :param pass_filter: Pass Filter, Filter to combined, diffuse, glossy and transmission passes
        :type pass_filter: int | None
        :param width: Width, Image width
        :type width: int | None
        :param height: Height, Image height
        :type height: int | None
        """
        ...

    def view_update(self, context: Context | None, depsgraph: Depsgraph | None):
        """Update on data changes for viewport render

        :param context:
        :type context: Context | None
        :param depsgraph:
        :type depsgraph: Depsgraph | None
        """
        ...

    def view_draw(self, context: Context | None, depsgraph: Depsgraph | None):
        """Draw viewport render

        :param context:
        :type context: Context | None
        :param depsgraph:
        :type depsgraph: Depsgraph | None
        """
        ...

    def update_script_node(self, node: Node | None = None):
        """Compile shader script node

        :param node:
        :type node: Node | None
        """
        ...

    def update_render_passes(
        self, scene: Scene | None = None, renderlayer: ViewLayer | None = None
    ):
        """Update the render passes that will be generated

        :param scene:
        :type scene: Scene | None
        :param renderlayer:
        :type renderlayer: ViewLayer | None
        """
        ...

    def tag_redraw(self):
        """Request redraw for viewport rendering"""
        ...

    def tag_update(self):
        """Request update call for viewport rendering"""
        ...

    def begin_result(
        self,
        x: int | None,
        y: int | None,
        w: int | None,
        h: int | None,
        layer: str | typing.Any = "",
        view: str | typing.Any = "",
    ) -> RenderResult:
        """Create render result to write linear floating-point render layers and passes

        :param x: X
        :type x: int | None
        :param y: Y
        :type y: int | None
        :param w: Width
        :type w: int | None
        :param h: Height
        :type h: int | None
        :param layer: Layer, Single layer to get render result for
        :type layer: str | typing.Any
        :param view: View, Single view to get render result for
        :type view: str | typing.Any
        :return: Result
        :rtype: RenderResult
        """
        ...

    def update_result(self, result: RenderResult | None):
        """Signal that pixels have been updated and can be redrawn in the user interface

        :param result: Result
        :type result: RenderResult | None
        """
        ...

    def end_result(
        self,
        result: RenderResult | None,
        cancel: bool | typing.Any | None = False,
        highlight: bool | typing.Any | None = False,
        do_merge_results: bool | typing.Any | None = False,
    ):
        """All pixels in the render result have been set and are final

        :param result: Result
        :type result: RenderResult | None
        :param cancel: Cancel, Don't mark tile as done, don't merge results unless forced
        :type cancel: bool | typing.Any | None
        :param highlight: Highlight, Don't mark tile as done yet
        :type highlight: bool | typing.Any | None
        :param do_merge_results: Merge Results, Merge results even if cancel=true
        :type do_merge_results: bool | typing.Any | None
        """
        ...

    def add_pass(
        self,
        name: str | typing.Any,
        channels: int | None,
        chan_id: str | typing.Any,
        layer: str | typing.Any = "",
    ):
        """Add a pass to the render layer

        :param name: Name, Name of the Pass, without view or channel tag
        :type name: str | typing.Any
        :param channels: Channels
        :type channels: int | None
        :param chan_id: Channel IDs, Channel names, one character per channel
        :type chan_id: str | typing.Any
        :param layer: Layer, Single layer to add render pass to
        :type layer: str | typing.Any
        """
        ...

    def get_result(self) -> RenderResult:
        """Get final result for non-pixel operations

        :return: Result
        :rtype: RenderResult
        """
        ...

    def test_break(self) -> bool:
        """Test if the render operation should been canceled, this is a fast call that should be used regularly for responsiveness

        :return: Break
        :rtype: bool
        """
        ...

    def pass_by_index_get(
        self, layer: str | typing.Any, index: int | None
    ) -> RenderPass:
        """pass_by_index_get

        :param layer: Layer, Name of render layer to get pass for
        :type layer: str | typing.Any
        :param index: Index, Index of pass to get
        :type index: int | None
        :return: Index, Index of pass to get
        :rtype: RenderPass
        """
        ...

    def active_view_get(self) -> str | typing.Any:
        """active_view_get

        :return: View, Single view active
        :rtype: str | typing.Any
        """
        ...

    def active_view_set(self, view: str | typing.Any):
        """active_view_set

        :param view: View, Single view to set as active
        :type view: str | typing.Any
        """
        ...

    def camera_shift_x(
        self,
        camera: Object | None,
        use_spherical_stereo: bool | typing.Any | None = False,
    ) -> float:
        """camera_shift_x

        :param camera:
        :type camera: Object | None
        :param use_spherical_stereo: Spherical Stereo
        :type use_spherical_stereo: bool | typing.Any | None
        :return: Shift X
        :rtype: float
        """
        ...

    def camera_model_matrix(
        self,
        camera: Object | None,
        use_spherical_stereo: bool | typing.Any | None = False,
    ) -> (
        list[list[float]]
        | tuple[
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
            tuple[float, float, float, float],
        ]
    ):
        """camera_model_matrix

        :param camera:
        :type camera: Object | None
        :param use_spherical_stereo: Spherical Stereo
        :type use_spherical_stereo: bool | typing.Any | None
        :return: Model Matrix, Normalized camera model matrix
        :rtype: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
        """
        ...

    def use_spherical_stereo(self, camera: Object | None) -> bool:
        """use_spherical_stereo

        :param camera:
        :type camera: Object | None
        :return: Spherical Stereo
        :rtype: bool
        """
        ...

    def update_stats(self, stats: str | typing.Any, info: str | typing.Any):
        """Update and signal to redraw render status text

        :param stats: Stats
        :type stats: str | typing.Any
        :param info: Info
        :type info: str | typing.Any
        """
        ...

    def frame_set(self, frame: int | None, subframe: float | None):
        """Evaluate scene at a different frame (for motion blur)

        :param frame: Frame
        :type frame: int | None
        :param subframe: Subframe
        :type subframe: float | None
        """
        ...

    def update_progress(self, progress: float | None):
        """Update progress percentage of render

        :param progress: Percentage of render that's done
        :type progress: float | None
        """
        ...

    def update_memory_stats(
        self, memory_used: typing.Any | None = 0.0, memory_peak: typing.Any | None = 0.0
    ):
        """Update memory usage statistics

        :param memory_used: Current memory usage in megabytes
        :type memory_used: typing.Any | None
        :param memory_peak: Peak memory usage in megabytes
        :type memory_peak: typing.Any | None
        """
        ...

    def report(self, type: set[str] | None, message: str | typing.Any):
        """Report info, warning or error messages

        :param type: Type
        :type type: set[str] | None
        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    def error_set(self, message: str | typing.Any):
        """Set error message displaying after the render is finished

        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    def bind_display_space_shader(self, scene: Scene | None):
        """Bind GLSL fragment shader that converts linear colors to display space colors using scene color management settings

        :param scene:
        :type scene: Scene | None
        """
        ...

    def unbind_display_space_shader(self):
        """Unbind GLSL display space shader, must always be called after binding the shader"""
        ...

    def support_display_space_shader(self, scene: Scene | None) -> bool:
        """Test if GLSL display space shader is supported for the combination of graphics card and scene settings

        :param scene:
        :type scene: Scene | None
        :return: Supported
        :rtype: bool
        """
        ...

    def get_preview_pixel_size(self, scene: Scene | None) -> int:
        """Get the pixel size that should be used for preview rendering

        :param scene:
        :type scene: Scene | None
        :return: Pixel Size
        :rtype: int
        """
        ...

    def free_blender_memory(self):
        """Free Blender side memory of render engine"""
        ...

    def tile_highlight_set(
        self,
        x: int | None,
        y: int | None,
        width: int | None,
        height: int | None,
        highlight: bool | None,
    ):
        """Set highlighted state of the given tile

        :param x: X
        :type x: int | None
        :param y: Y
        :type y: int | None
        :param width: Width
        :type width: int | None
        :param height: Height
        :type height: int | None
        :param highlight: Highlight
        :type highlight: bool | None
        """
        ...

    def tile_highlight_clear_all(self):
        """The temp directory used by Blender"""
        ...

    def register_pass(
        self,
        scene: Scene | None,
        view_layer: ViewLayer | None,
        name: str | typing.Any,
        channels: int | None,
        chanid: str | typing.Any,
        type: str | None,
    ):
        """Register a render pass that will be part of the render with the current settings

        :param scene:
        :type scene: Scene | None
        :param view_layer:
        :type view_layer: ViewLayer | None
        :param name: Name
        :type name: str | typing.Any
        :param channels: Channels
        :type channels: int | None
        :param chanid: Channel IDs
        :type chanid: str | typing.Any
        :param type: Type
        :type type: str | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderLayer(bpy_struct):
    name: str
    """ View layer name

    :type: str
    """

    passes: RenderPasses
    """ 

    :type: RenderPasses
    """

    use_ao: bool
    """ Render Ambient Occlusion in this Layer

    :type: bool
    """

    use_motion_blur: bool
    """ Render motion blur in this Layer, if enabled in the scene

    :type: bool
    """

    use_pass_ambient_occlusion: bool
    """ Deliver Ambient Occlusion pass

    :type: bool
    """

    use_pass_combined: bool
    """ Deliver full combined RGBA buffer

    :type: bool
    """

    use_pass_diffuse_color: bool
    """ Deliver diffuse color pass

    :type: bool
    """

    use_pass_diffuse_direct: bool
    """ Deliver diffuse direct pass

    :type: bool
    """

    use_pass_diffuse_indirect: bool
    """ Deliver diffuse indirect pass

    :type: bool
    """

    use_pass_emit: bool
    """ Deliver emission pass

    :type: bool
    """

    use_pass_environment: bool
    """ Deliver environment lighting pass

    :type: bool
    """

    use_pass_glossy_color: bool
    """ Deliver glossy color pass

    :type: bool
    """

    use_pass_glossy_direct: bool
    """ Deliver glossy direct pass

    :type: bool
    """

    use_pass_glossy_indirect: bool
    """ Deliver glossy indirect pass

    :type: bool
    """

    use_pass_material_index: bool
    """ Deliver material index pass

    :type: bool
    """

    use_pass_mist: bool
    """ Deliver mist factor pass (0.0 to 1.0)

    :type: bool
    """

    use_pass_normal: bool
    """ Deliver normal pass

    :type: bool
    """

    use_pass_object_index: bool
    """ Deliver object index pass

    :type: bool
    """

    use_pass_position: bool
    """ Deliver position pass

    :type: bool
    """

    use_pass_shadow: bool
    """ Deliver shadow pass

    :type: bool
    """

    use_pass_subsurface_color: bool
    """ Deliver subsurface color pass

    :type: bool
    """

    use_pass_subsurface_direct: bool
    """ Deliver subsurface direct pass

    :type: bool
    """

    use_pass_subsurface_indirect: bool
    """ Deliver subsurface indirect pass

    :type: bool
    """

    use_pass_transmission_color: bool
    """ Deliver transmission color pass

    :type: bool
    """

    use_pass_transmission_direct: bool
    """ Deliver transmission direct pass

    :type: bool
    """

    use_pass_transmission_indirect: bool
    """ Deliver transmission indirect pass

    :type: bool
    """

    use_pass_uv: bool
    """ Deliver texture UV pass

    :type: bool
    """

    use_pass_vector: bool
    """ Deliver speed vector pass

    :type: bool
    """

    use_pass_z: bool
    """ Deliver Z values pass

    :type: bool
    """

    use_sky: bool
    """ Render Sky in this Layer

    :type: bool
    """

    use_solid: bool
    """ Render Solid faces in this Layer

    :type: bool
    """

    use_strand: bool
    """ Render Strands in this Layer

    :type: bool
    """

    use_volumes: bool
    """ Render volumes in this Layer

    :type: bool
    """

    def load_from_file(
        self,
        filename: str | typing.Any,
        x: typing.Any | None = 0,
        y: typing.Any | None = 0,
    ):
        """Copies the pixels of this renderlayer from an image file

        :param filename: Filename, Filename to load into this render tile, must be no smaller than the renderlayer
        :type filename: str | typing.Any
        :param x: Offset X, Offset the position to copy from if the image is larger than the render layer
        :type x: typing.Any | None
        :param y: Offset Y, Offset the position to copy from if the image is larger than the render layer
        :type y: typing.Any | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderPass(bpy_struct):
    channel_id: str
    """ 

    :type: str
    """

    channels: int
    """ 

    :type: int
    """

    fullname: str
    """ 

    :type: str
    """

    name: str
    """ 

    :type: str
    """

    rect: float
    """ 

    :type: float
    """

    view_id: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderPasses(bpy_prop_collection[RenderPass], bpy_struct):
    """Collection of render passes"""

    def find_by_type(self, pass_type: str | None, view: str | typing.Any) -> RenderPass:
        """Get the render pass for a given type and view

        :param pass_type: Pass
        :type pass_type: str | None
        :param view: View, Render view to get pass from
        :type view: str | typing.Any
        :return: The matching render pass
        :rtype: RenderPass
        """
        ...

    def find_by_name(
        self, name: str | typing.Any, view: str | typing.Any
    ) -> RenderPass:
        """Get the render pass for a given name and view

        :param name: Pass
        :type name: str | typing.Any
        :param view: View, Render view to get pass from
        :type view: str | typing.Any
        :return: The matching render pass
        :rtype: RenderPass
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderResult(bpy_struct):
    """Result of rendering, including all layers and passes"""

    layers: bpy_prop_collection[RenderLayer]
    """ 

    :type: bpy_prop_collection[RenderLayer]
    """

    resolution_x: int
    """ 

    :type: int
    """

    resolution_y: int
    """ 

    :type: int
    """

    views: bpy_prop_collection[RenderView]
    """ 

    :type: bpy_prop_collection[RenderView]
    """

    def load_from_file(self, filename: str | typing.Any):
        """Copies the pixels of this render result from an image file

        :param filename: File Name, Filename to load into this render tile, must be no smaller than the render result
        :type filename: str | typing.Any
        """
        ...

    def stamp_data_add_field(self, field: str | typing.Any, value: str | typing.Any):
        """Add engine-specific stamp data to the result

        :param field: Field, Name of the stamp field to add
        :type field: str | typing.Any
        :param value: Value, Value of the stamp data
        :type value: str | typing.Any
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderSettings(bpy_struct):
    """Rendering settings for a Scene data-block"""

    bake: BakeSettings
    """ 

    :type: BakeSettings
    """

    bake_bias: float
    """ Bias towards faces further away from the object (in blender units)

    :type: float
    """

    bake_margin: int
    """ Extends the baked result as a post process filter

    :type: int
    """

    bake_margin_type: str
    """ Algorithm to generate the margin

    :type: str
    """

    bake_samples: int
    """ Number of samples used for ambient occlusion baking from multires

    :type: int
    """

    bake_type: str
    """ Choose shading information to bake into the image

    :type: str
    """

    bake_user_scale: float
    """ Instead of automatically normalizing to the range 0 to 1, apply a user scale to the derivative map

    :type: float
    """

    border_max_x: float
    """ Maximum X value for the render region

    :type: float
    """

    border_max_y: float
    """ Maximum Y value for the render region

    :type: float
    """

    border_min_x: float
    """ Minimum X value for the render region

    :type: float
    """

    border_min_y: float
    """ Minimum Y value for the render region

    :type: float
    """

    dither_intensity: float
    """ Amount of dithering noise added to the rendered image to break up banding

    :type: float
    """

    engine: str
    """ Engine to use for rendering

    :type: str
    """

    ffmpeg: FFmpegSettings
    """ FFmpeg related settings for the scene

    :type: FFmpegSettings
    """

    file_extension: str
    """ The file extension used for saving renders

    :type: str
    """

    filepath: str
    """ Directory/name to save animations, # characters defines the position and length of frame numbers

    :type: str
    """

    film_transparent: bool
    """ World background is transparent, for compositing the render over another background

    :type: bool
    """

    filter_size: float
    """ Width over which the reconstruction filter combines samples

    :type: float
    """

    fps: int
    """ Framerate, expressed in frames per second

    :type: int
    """

    fps_base: float
    """ Framerate base

    :type: float
    """

    frame_map_new: int
    """ How many frames the Map Old will last

    :type: int
    """

    frame_map_old: int
    """ Old mapping value in frames

    :type: int
    """

    hair_subdiv: int
    """ Additional subdivision along the curves

    :type: int
    """

    hair_type: str
    """ Curves shape type

    :type: str
    """

    has_multiple_engines: bool
    """ More than one rendering engine is available

    :type: bool
    """

    image_settings: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    is_movie_format: bool
    """ When true the format is a movie

    :type: bool
    """

    line_thickness: float
    """ Line thickness in pixels

    :type: float
    """

    line_thickness_mode: str
    """ Line thickness mode for Freestyle line drawing

    :type: str
    """

    metadata_input: str
    """ Where to take the metadata from

    :type: str
    """

    motion_blur_shutter: float
    """ Time taken in frames between shutter open and close

    :type: float
    """

    motion_blur_shutter_curve: CurveMapping
    """ Curve defining the shutter's openness over time

    :type: CurveMapping
    """

    pixel_aspect_x: float
    """ Horizontal aspect ratio - for anamorphic or non-square pixel output

    :type: float
    """

    pixel_aspect_y: float
    """ Vertical aspect ratio - for anamorphic or non-square pixel output

    :type: float
    """

    preview_pixel_size: str
    """ Pixel size for viewport rendering

    :type: str
    """

    resolution_percentage: int
    """ Percentage scale for render resolution

    :type: int
    """

    resolution_x: int
    """ Number of horizontal pixels in the rendered image

    :type: int
    """

    resolution_y: int
    """ Number of vertical pixels in the rendered image

    :type: int
    """

    sequencer_gl_preview: str
    """ Display method used in the sequencer view

    :type: str
    """

    simplify_child_particles: float
    """ Global child particles percentage

    :type: float
    """

    simplify_child_particles_render: float
    """ Global child particles percentage during rendering

    :type: float
    """

    simplify_gpencil: bool
    """ Simplify Grease Pencil drawing

    :type: bool
    """

    simplify_gpencil_antialiasing: bool
    """ Use Antialiasing to smooth stroke edges

    :type: bool
    """

    simplify_gpencil_modifier: bool
    """ Display modifiers

    :type: bool
    """

    simplify_gpencil_onplay: bool
    """ Simplify Grease Pencil only during animation playback

    :type: bool
    """

    simplify_gpencil_shader_fx: bool
    """ Display Shader Effects

    :type: bool
    """

    simplify_gpencil_tint: bool
    """ Display layer tint

    :type: bool
    """

    simplify_gpencil_view_fill: bool
    """ Display fill strokes in the viewport

    :type: bool
    """

    simplify_subdivision: int
    """ Global maximum subdivision level

    :type: int
    """

    simplify_subdivision_render: int
    """ Global maximum subdivision level during rendering

    :type: int
    """

    simplify_volumes: float
    """ Resolution percentage of volume objects in viewport

    :type: float
    """

    stamp_background: bpy_prop_array[float]
    """ Color to use behind stamp text

    :type: bpy_prop_array[float]
    """

    stamp_font_size: int
    """ Size of the font used when rendering stamp text

    :type: int
    """

    stamp_foreground: bpy_prop_array[float]
    """ Color to use for stamp text

    :type: bpy_prop_array[float]
    """

    stamp_note_text: str
    """ Custom text to appear in the stamp note

    :type: str
    """

    stereo_views: bpy_prop_collection[SceneRenderView]
    """ 

    :type: bpy_prop_collection[SceneRenderView]
    """

    threads: int
    """ Maximum number of CPU cores to use simultaneously while rendering (for multi-core/CPU systems)

    :type: int
    """

    threads_mode: str
    """ Determine the amount of render threads used

    :type: str
    """

    use_bake_clear: bool
    """ Clear Images before baking

    :type: bool
    """

    use_bake_lores_mesh: bool
    """ Calculate heights against unsubdivided low resolution mesh

    :type: bool
    """

    use_bake_multires: bool
    """ Bake directly from multires object

    :type: bool
    """

    use_bake_selected_to_active: bool
    """ Bake shading on the surface of selected objects to the active object

    :type: bool
    """

    use_bake_user_scale: bool
    """ Use a user scale for the derivative map

    :type: bool
    """

    use_border: bool
    """ Render a user-defined render region, within the frame size

    :type: bool
    """

    use_compositing: bool
    """ Process the render result through the compositing pipeline, if compositing nodes are enabled

    :type: bool
    """

    use_crop_to_border: bool
    """ Crop the rendered frame to the defined render region size

    :type: bool
    """

    use_file_extension: bool
    """ Add the file format extensions to the rendered file name (eg: filename + .jpg)

    :type: bool
    """

    use_freestyle: bool
    """ Draw stylized strokes using Freestyle

    :type: bool
    """

    use_high_quality_normals: bool
    """ Use high quality tangent space at the cost of lower performance

    :type: bool
    """

    use_lock_interface: bool
    """ Lock interface during rendering in favor of giving more memory to the renderer

    :type: bool
    """

    use_motion_blur: bool
    """ Use multi-sampled 3D scene motion blur

    :type: bool
    """

    use_multiview: bool
    """ Use multiple views in the scene

    :type: bool
    """

    use_overwrite: bool
    """ Overwrite existing files while rendering

    :type: bool
    """

    use_persistent_data: bool
    """ Keep render data around for faster re-renders and animation renders, at the cost of increased memory usage

    :type: bool
    """

    use_placeholder: bool
    """ Create empty placeholder files while rendering frames (similar to Unix 'touch')

    :type: bool
    """

    use_render_cache: bool
    """ Save render cache to EXR files (useful for heavy compositing, Note: affects indirectly rendered scenes)

    :type: bool
    """

    use_sequencer: bool
    """ Process the render (and composited) result through the video sequence editor pipeline, if sequencer strips exist

    :type: bool
    """

    use_sequencer_override_scene_strip: bool
    """ Use workbench render settings from the sequencer scene, instead of each individual scene used in the strip

    :type: bool
    """

    use_simplify: bool
    """ Enable simplification of scene for quicker preview renders

    :type: bool
    """

    use_single_layer: bool
    """ Only render the active layer. Only affects rendering from the interface, ignored for rendering from command line

    :type: bool
    """

    use_spherical_stereo: bool
    """ Active render engine supports spherical stereo rendering

    :type: bool
    """

    use_stamp: bool
    """ Render the stamp info text in the rendered image

    :type: bool
    """

    use_stamp_camera: bool
    """ Include the name of the active camera in image metadata

    :type: bool
    """

    use_stamp_date: bool
    """ Include the current date in image/video metadata

    :type: bool
    """

    use_stamp_filename: bool
    """ Include the .blend filename in image/video metadata

    :type: bool
    """

    use_stamp_frame: bool
    """ Include the frame number in image metadata

    :type: bool
    """

    use_stamp_frame_range: bool
    """ Include the rendered frame range in image/video metadata

    :type: bool
    """

    use_stamp_hostname: bool
    """ Include the hostname of the machine that rendered the frame

    :type: bool
    """

    use_stamp_labels: bool
    """ Display stamp labels ("Camera" in front of camera name, etc.)

    :type: bool
    """

    use_stamp_lens: bool
    """ Include the active camera's lens in image metadata

    :type: bool
    """

    use_stamp_marker: bool
    """ Include the name of the last marker in image metadata

    :type: bool
    """

    use_stamp_memory: bool
    """ Include the peak memory usage in image metadata

    :type: bool
    """

    use_stamp_note: bool
    """ Include a custom note in image/video metadata

    :type: bool
    """

    use_stamp_render_time: bool
    """ Include the render time in image metadata

    :type: bool
    """

    use_stamp_scene: bool
    """ Include the name of the active scene in image/video metadata

    :type: bool
    """

    use_stamp_sequencer_strip: bool
    """ Include the name of the foreground sequence strip in image metadata

    :type: bool
    """

    use_stamp_time: bool
    """ Include the rendered frame timecode as HH:MM:SS.FF in image metadata

    :type: bool
    """

    views: RenderViews
    """ 

    :type: RenderViews
    """

    views_format: str
    """ 

    :type: str
    """

    def frame_path(
        self,
        frame: typing.Any | None = -2147483648,
        preview: bool | typing.Any | None = False,
        view: str | typing.Any = "",
    ) -> str | typing.Any:
        """Return the absolute path to the filename to be written for a given frame

        :param frame: Frame number to use, if unset the current frame will be used
        :type frame: typing.Any | None
        :param preview: Preview, Use preview range
        :type preview: bool | typing.Any | None
        :param view: View, The name of the view to use to replace the "%" chars
        :type view: str | typing.Any
        :return: File Path, The resulting filepath from the scenes render settings
        :rtype: str | typing.Any
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderSlot(bpy_struct):
    """Parameters defining the render slot"""

    name: str
    """ Render slot name

    :type: str
    """

    def clear(self, iuser: ImageUser | None):
        """Clear the render slot

        :param iuser: ImageUser
        :type iuser: ImageUser | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderSlots(bpy_prop_collection[RenderSlot], bpy_struct):
    """Collection of render layers"""

    active: RenderSlot | None
    """ Active render slot of the image

    :type: RenderSlot | None
    """

    active_index: int | None
    """ Active render slot of the image

    :type: int | None
    """

    def new(self, name: str | typing.Any = "") -> RenderSlot:
        """Add a render slot to the image

        :param name: Name, New name for the render slot
        :type name: str | typing.Any
        :return: Newly created render layer
        :rtype: RenderSlot
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderView(bpy_struct):
    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RenderViews(bpy_prop_collection[SceneRenderView], bpy_struct):
    """Collection of render views"""

    active: SceneRenderView
    """ Active Render View

    :type: SceneRenderView
    """

    active_index: int | None
    """ Active index in render view array

    :type: int | None
    """

    def new(self, name: str | typing.Any) -> SceneRenderView:
        """Add a render view to scene

        :param name: New name for the marker (not unique)
        :type name: str | typing.Any
        :return: Newly created render view
        :rtype: SceneRenderView
        """
        ...

    def remove(self, view: SceneRenderView):
        """Remove a render view

        :param view: Render view to remove
        :type view: SceneRenderView
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RigidBodyConstraint(bpy_struct):
    """Constraint influencing Objects inside Rigid Body Simulation"""

    breaking_threshold: float
    """ Impulse threshold that must be reached for the constraint to break

    :type: float
    """

    disable_collisions: bool
    """ Disable collisions between constrained rigid bodies

    :type: bool
    """

    enabled: bool
    """ Enable this constraint

    :type: bool
    """

    limit_ang_x_lower: float
    """ Lower limit of X axis rotation

    :type: float
    """

    limit_ang_x_upper: float
    """ Upper limit of X axis rotation

    :type: float
    """

    limit_ang_y_lower: float
    """ Lower limit of Y axis rotation

    :type: float
    """

    limit_ang_y_upper: float
    """ Upper limit of Y axis rotation

    :type: float
    """

    limit_ang_z_lower: float
    """ Lower limit of Z axis rotation

    :type: float
    """

    limit_ang_z_upper: float
    """ Upper limit of Z axis rotation

    :type: float
    """

    limit_lin_x_lower: float
    """ Lower limit of X axis translation

    :type: float
    """

    limit_lin_x_upper: float
    """ Upper limit of X axis translation

    :type: float
    """

    limit_lin_y_lower: float
    """ Lower limit of Y axis translation

    :type: float
    """

    limit_lin_y_upper: float
    """ Upper limit of Y axis translation

    :type: float
    """

    limit_lin_z_lower: float
    """ Lower limit of Z axis translation

    :type: float
    """

    limit_lin_z_upper: float
    """ Upper limit of Z axis translation

    :type: float
    """

    motor_ang_max_impulse: float
    """ Maximum angular motor impulse

    :type: float
    """

    motor_ang_target_velocity: float
    """ Target angular motor velocity

    :type: float
    """

    motor_lin_max_impulse: float
    """ Maximum linear motor impulse

    :type: float
    """

    motor_lin_target_velocity: float
    """ Target linear motor velocity

    :type: float
    """

    object1: Object
    """ First Rigid Body Object to be constrained

    :type: Object
    """

    object2: Object
    """ Second Rigid Body Object to be constrained

    :type: Object
    """

    solver_iterations: int
    """ Number of constraint solver iterations made per simulation step (higher values are more accurate but slower)

    :type: int
    """

    spring_damping_ang_x: float
    """ Damping on the X rotational axis

    :type: float
    """

    spring_damping_ang_y: float
    """ Damping on the Y rotational axis

    :type: float
    """

    spring_damping_ang_z: float
    """ Damping on the Z rotational axis

    :type: float
    """

    spring_damping_x: float
    """ Damping on the X axis

    :type: float
    """

    spring_damping_y: float
    """ Damping on the Y axis

    :type: float
    """

    spring_damping_z: float
    """ Damping on the Z axis

    :type: float
    """

    spring_stiffness_ang_x: float
    """ Stiffness on the X rotational axis

    :type: float
    """

    spring_stiffness_ang_y: float
    """ Stiffness on the Y rotational axis

    :type: float
    """

    spring_stiffness_ang_z: float
    """ Stiffness on the Z rotational axis

    :type: float
    """

    spring_stiffness_x: float
    """ Stiffness on the X axis

    :type: float
    """

    spring_stiffness_y: float
    """ Stiffness on the Y axis

    :type: float
    """

    spring_stiffness_z: float
    """ Stiffness on the Z axis

    :type: float
    """

    spring_type: str
    """ Which implementation of spring to use

    :type: str
    """

    type: str
    """ Type of Rigid Body Constraint

    :type: str
    """

    use_breaking: bool
    """ Constraint can be broken if it receives an impulse above the threshold

    :type: bool
    """

    use_limit_ang_x: bool
    """ Limit rotation around X axis

    :type: bool
    """

    use_limit_ang_y: bool
    """ Limit rotation around Y axis

    :type: bool
    """

    use_limit_ang_z: bool
    """ Limit rotation around Z axis

    :type: bool
    """

    use_limit_lin_x: bool
    """ Limit translation on X axis

    :type: bool
    """

    use_limit_lin_y: bool
    """ Limit translation on Y axis

    :type: bool
    """

    use_limit_lin_z: bool
    """ Limit translation on Z axis

    :type: bool
    """

    use_motor_ang: bool
    """ Enable angular motor

    :type: bool
    """

    use_motor_lin: bool
    """ Enable linear motor

    :type: bool
    """

    use_override_solver_iterations: bool
    """ Override the number of solver iterations for this constraint

    :type: bool
    """

    use_spring_ang_x: bool
    """ Enable spring on X rotational axis

    :type: bool
    """

    use_spring_ang_y: bool
    """ Enable spring on Y rotational axis

    :type: bool
    """

    use_spring_ang_z: bool
    """ Enable spring on Z rotational axis

    :type: bool
    """

    use_spring_x: bool
    """ Enable spring on X axis

    :type: bool
    """

    use_spring_y: bool
    """ Enable spring on Y axis

    :type: bool
    """

    use_spring_z: bool
    """ Enable spring on Z axis

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RigidBodyObject(bpy_struct):
    """Settings for object participating in Rigid Body Simulation"""

    angular_damping: float
    """ Amount of angular velocity that is lost over time

    :type: float
    """

    collision_collections: list[bool]
    """ Collision collections rigid body belongs to

    :type: list[bool]
    """

    collision_margin: float
    """ Threshold of distance near surface where collisions are still considered (best results when non-zero)

    :type: float
    """

    collision_shape: str
    """ Collision Shape of object in Rigid Body Simulations

    :type: str
    """

    deactivate_angular_velocity: float
    """ Angular Velocity below which simulation stops simulating object

    :type: float
    """

    deactivate_linear_velocity: float
    """ Linear Velocity below which simulation stops simulating object

    :type: float
    """

    enabled: bool
    """ Rigid Body actively participates to the simulation

    :type: bool
    """

    friction: float
    """ Resistance of object to movement

    :type: float
    """

    kinematic: bool
    """ Allow rigid body to be controlled by the animation system

    :type: bool
    """

    linear_damping: float
    """ Amount of linear velocity that is lost over time

    :type: float
    """

    mass: float
    """ How much the object 'weighs' irrespective of gravity

    :type: float
    """

    mesh_source: str
    """ Source of the mesh used to create collision shape

    :type: str
    """

    restitution: float
    """ Tendency of object to bounce after colliding with another (0 = stays still, 1 = perfectly elastic)

    :type: float
    """

    type: str
    """ Role of object in Rigid Body Simulations

    :type: str
    """

    use_deactivation: bool
    """ Enable deactivation of resting rigid bodies (increases performance and stability but can cause glitches)

    :type: bool
    """

    use_deform: bool
    """ Rigid body deforms during simulation

    :type: bool
    """

    use_margin: bool
    """ Use custom collision margin (some shapes will have a visible gap around them)

    :type: bool
    """

    use_start_deactivated: bool
    """ Deactivate rigid body at the start of the simulation

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class RigidBodyWorld(bpy_struct):
    """Self-contained rigid body simulation environment and settings"""

    collection: Collection
    """ Collection containing objects participating in this simulation

    :type: Collection
    """

    constraints: Collection
    """ Collection containing rigid body constraint objects

    :type: Collection
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    enabled: bool
    """ Simulation will be evaluated

    :type: bool
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    solver_iterations: int
    """ Number of constraint solver iterations made per simulation step (higher values are more accurate but slower)

    :type: int
    """

    substeps_per_frame: int
    """ Number of simulation steps taken per frame (higher values are more accurate but slower)

    :type: int
    """

    time_scale: float
    """ Change the speed of the simulation

    :type: float
    """

    use_split_impulse: bool
    """ Reduce extra velocity that can build up when objects collide (lowers simulation stability a little so use only when necessary)

    :type: bool
    """

    def convex_sweep_test(
        self,
        object: Object,
        start: collections.abc.Iterable[float] | None,
        end: collections.abc.Iterable[float] | None,
    ):
        """Sweep test convex rigidbody against the current rigidbody world

                :param object: Rigidbody object with a convex collision shape
                :type object: Object
                :param start:
                :type start: collections.abc.Iterable[float] | None
                :param end:
                :type end: collections.abc.Iterable[float] | None
                :return: object_location, The hit location of this sweep test, float array of 3 items in [-inf, inf]

        hitpoint, The hit location of this sweep test, float array of 3 items in [-inf, inf]

        normal, The face normal at the sweep test hit location, float array of 3 items in [-inf, inf]

        has_hit, If the function has found collision point, value is 1, otherwise 0, int in [-inf, inf]
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SPHFluidSettings(bpy_struct):
    """Settings for particle fluids physics"""

    buoyancy: float
    """ Artificial buoyancy force in negative gravity direction based on pressure differences inside the fluid

    :type: float
    """

    fluid_radius: float
    """ Fluid interaction radius

    :type: float
    """

    linear_viscosity: float
    """ Linear viscosity

    :type: float
    """

    plasticity: float
    """ How much the spring rest length can change after the elastic limit is crossed

    :type: float
    """

    repulsion: float
    """ How strongly the fluid tries to keep from clustering (factor of stiffness)

    :type: float
    """

    rest_density: float
    """ Fluid rest density

    :type: float
    """

    rest_length: float
    """ Spring rest length (factor of particle radius)

    :type: float
    """

    solver: str
    """ The code used to calculate internal forces on particles

    :type: str
    """

    spring_force: float
    """ Spring force

    :type: float
    """

    spring_frames: int
    """ Create springs for this number of frames since particles birth (0 is always)

    :type: int
    """

    stiff_viscosity: float
    """ Creates viscosity for expanding fluid

    :type: float
    """

    stiffness: float
    """ How incompressible the fluid is (speed of sound)

    :type: float
    """

    use_factor_density: bool
    """ Density is calculated as a factor of default density (depends on particle size)

    :type: bool
    """

    use_factor_radius: bool
    """ Interaction radius is a factor of 4 * particle size

    :type: bool
    """

    use_factor_repulsion: bool
    """ Repulsion is a factor of stiffness

    :type: bool
    """

    use_factor_rest_length: bool
    """ Spring rest length is a factor of 2 * particle size

    :type: bool
    """

    use_factor_stiff_viscosity: bool
    """ Stiff viscosity is a factor of normal viscosity

    :type: bool
    """

    use_initial_rest_length: bool
    """ Use the initial length as spring rest length instead of 2 * particle size

    :type: bool
    """

    use_viscoelastic_springs: bool
    """ Use viscoelastic springs instead of Hooke's springs

    :type: bool
    """

    yield_ratio: float
    """ How much the spring has to be stretched/compressed in order to change its rest length

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SceneDisplay(bpy_struct):
    """Scene display settings for 3D viewport"""

    light_direction: bpy_prop_array[float]
    """ Direction of the light for shadows and highlights

    :type: bpy_prop_array[float]
    """

    matcap_ssao_attenuation: float
    """ Attenuation constant

    :type: float
    """

    matcap_ssao_distance: float
    """ Distance of object that contribute to the Cavity/Edge effect

    :type: float
    """

    matcap_ssao_samples: int
    """ Number of samples

    :type: int
    """

    render_aa: str
    """ Method of anti-aliasing when rendering final image

    :type: str
    """

    shading: View3DShading
    """ Shading settings for OpenGL render engine

    :type: View3DShading
    """

    shadow_focus: float
    """ Shadow factor hardness

    :type: float
    """

    shadow_shift: float
    """ Shadow termination angle

    :type: float
    """

    viewport_aa: str
    """ Method of anti-aliasing when rendering 3d viewport

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SceneEEVEE(bpy_struct):
    """Scene display settings for 3D viewport"""

    bloom_clamp: float
    """ Maximum intensity a bloom pixel can have (0 to disabled)

    :type: float
    """

    bloom_color: bpy_prop_array[float]
    """ Color applied to the bloom effect

    :type: bpy_prop_array[float]
    """

    bloom_intensity: float
    """ Blend factor

    :type: float
    """

    bloom_knee: float
    """ Makes transition between under/over-threshold gradual

    :type: float
    """

    bloom_radius: float
    """ Bloom spread distance

    :type: float
    """

    bloom_threshold: float
    """ Filters out pixels under this level of brightness

    :type: float
    """

    bokeh_denoise_fac: float
    """ Amount of flicker removal applied to bokeh highlights

    :type: float
    """

    bokeh_max_size: float
    """ Max size of the bokeh shape for the depth of field (lower is faster)

    :type: float
    """

    bokeh_neighbor_max: float
    """ Maximum brightness to consider when rejecting bokeh sprites based on neighborhood (lower is faster)

    :type: float
    """

    bokeh_overblur: float
    """ Apply blur to each jittered sample to reduce under-sampling artifacts

    :type: float
    """

    bokeh_threshold: float
    """ Brightness threshold for using sprite base depth of field

    :type: float
    """

    gi_auto_bake: bool
    """ Auto bake indirect lighting when editing probes

    :type: bool
    """

    gi_cache_info: str
    """ Info on current cache status

    :type: str
    """

    gi_cubemap_display_size: float
    """ Size of the cubemap spheres to debug captured light

    :type: float
    """

    gi_cubemap_resolution: str
    """ Size of every cubemaps

    :type: str
    """

    gi_diffuse_bounces: int
    """ Number of time the light is reinjected inside light grids, 0 disable indirect diffuse light

    :type: int
    """

    gi_filter_quality: float
    """ Take more samples during cubemap filtering to remove artifacts

    :type: float
    """

    gi_glossy_clamp: float
    """ Clamp pixel intensity to reduce noise inside glossy reflections from reflection cubemaps (0 to disabled)

    :type: float
    """

    gi_irradiance_display_size: float
    """ Size of the irradiance sample spheres to debug captured light

    :type: float
    """

    gi_irradiance_smoothing: float
    """ Smoother irradiance interpolation but introduce light bleeding

    :type: float
    """

    gi_show_cubemaps: bool
    """ Display captured cubemaps in the viewport

    :type: bool
    """

    gi_show_irradiance: bool
    """ Display irradiance samples in the viewport

    :type: bool
    """

    gi_visibility_resolution: str
    """ Size of the shadow map applied to each irradiance sample

    :type: str
    """

    gtao_distance: float
    """ Distance of object that contribute to the ambient occlusion effect

    :type: float
    """

    gtao_factor: float
    """ Factor for ambient occlusion blending

    :type: float
    """

    gtao_quality: float
    """ Precision of the horizon search

    :type: float
    """

    light_threshold: float
    """ Minimum light intensity for a light to contribute to the lighting

    :type: float
    """

    motion_blur_depth_scale: float
    """ Lower values will reduce background bleeding onto foreground elements

    :type: float
    """

    motion_blur_max: int
    """ Maximum blur distance a pixel can spread over

    :type: int
    """

    motion_blur_position: str
    """ Offset for the shutter's time interval, allows to change the motion blur trails

    :type: str
    """

    motion_blur_shutter: float
    """ Time taken in frames between shutter open and close

    :type: float
    """

    motion_blur_steps: int
    """ Controls accuracy of motion blur, more steps means longer render time

    :type: int
    """

    overscan_size: float
    """ Percentage of render size to add as overscan to the internal render buffers

    :type: float
    """

    shadow_cascade_size: str
    """ Size of sun light shadow maps

    :type: str
    """

    shadow_cube_size: str
    """ Size of point and area light shadow maps

    :type: str
    """

    ssr_border_fade: float
    """ Screen percentage used to fade the SSR

    :type: float
    """

    ssr_firefly_fac: float
    """ Clamp pixel intensity to remove noise (0 to disabled)

    :type: float
    """

    ssr_max_roughness: float
    """ Do not raytrace reflections for roughness above this value

    :type: float
    """

    ssr_quality: float
    """ Precision of the screen space ray-tracing

    :type: float
    """

    ssr_thickness: float
    """ Pixel thickness used to detect intersection

    :type: float
    """

    sss_jitter_threshold: float
    """ Rotate samples that are below this threshold

    :type: float
    """

    sss_samples: int
    """ Number of samples to compute the scattering effect

    :type: int
    """

    taa_render_samples: int
    """ Number of samples per pixels for rendering

    :type: int
    """

    taa_samples: int
    """ Number of samples, unlimited if 0

    :type: int
    """

    use_bloom: bool
    """ High brightness pixels generate a glowing effect

    :type: bool
    """

    use_bokeh_high_quality_slight_defocus: bool
    """ Sample all pixels in almost in-focus regions to eliminate noise

    :type: bool
    """

    use_bokeh_jittered: bool
    """ Jitter camera position to create accurate blurring using render samples

    :type: bool
    """

    use_gtao: bool
    """ Enable ambient occlusion to simulate medium scale indirect shadowing

    :type: bool
    """

    use_gtao_bent_normals: bool
    """ Compute main non occluded direction to sample the environment

    :type: bool
    """

    use_gtao_bounce: bool
    """ An approximation to simulate light bounces giving less occlusion on brighter objects

    :type: bool
    """

    use_motion_blur: bool
    """ Enable motion blur effect (only in camera view)

    :type: bool
    """

    use_overscan: bool
    """ Internally render past the image border to avoid screen-space effects disappearing

    :type: bool
    """

    use_shadow_high_bitdepth: bool
    """ Use 32-bit shadows

    :type: bool
    """

    use_soft_shadows: bool
    """ Randomize shadowmaps origin to create soft shadows

    :type: bool
    """

    use_ssr: bool
    """ Enable screen space reflection

    :type: bool
    """

    use_ssr_halfres: bool
    """ Raytrace at a lower resolution

    :type: bool
    """

    use_ssr_refraction: bool
    """ Enable screen space Refractions

    :type: bool
    """

    use_taa_reprojection: bool
    """ Denoise image using temporal reprojection (can leave some ghosting)

    :type: bool
    """

    use_volumetric_lights: bool
    """ Enable scene light interactions with volumetrics

    :type: bool
    """

    use_volumetric_shadows: bool
    """ Generate shadows from volumetric material (Very expensive)

    :type: bool
    """

    volumetric_end: float
    """ End distance of the volumetric effect

    :type: float
    """

    volumetric_light_clamp: float
    """ Maximum light contribution, reducing noise

    :type: float
    """

    volumetric_sample_distribution: float
    """ Distribute more samples closer to the camera

    :type: float
    """

    volumetric_samples: int
    """ Number of samples to compute volumetric effects

    :type: int
    """

    volumetric_shadow_samples: int
    """ Number of samples to compute volumetric shadowing

    :type: int
    """

    volumetric_start: float
    """ Start distance of the volumetric effect

    :type: float
    """

    volumetric_tile_size: str
    """ Control the quality of the volumetric effects (lower size increase vram usage and quality)

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SceneGpencil(bpy_struct):
    """Render settings"""

    antialias_threshold: float
    """ Threshold for edge detection algorithm (higher values might over-blur some part of the image)

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SceneObjects(bpy_prop_collection[Object], bpy_struct):
    """All of the scene objects"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SceneRenderView(bpy_struct):
    """Render viewpoint for 3D stereo and multiview rendering"""

    camera_suffix: str
    """ Suffix to identify the cameras to use, and added to the render images for this view

    :type: str
    """

    file_suffix: str
    """ Suffix added to the render images for this view

    :type: str
    """

    name: str
    """ Render view name

    :type: str
    """

    use: bool
    """ Disable or enable the render view

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Scopes(bpy_struct):
    """Scopes for statistical view of an image"""

    accuracy: float
    """ Proportion of original image source pixel lines to sample

    :type: float
    """

    histogram: Histogram
    """ Histogram for viewing image statistics

    :type: Histogram
    """

    use_full_resolution: bool
    """ Sample every pixel of the image

    :type: bool
    """

    vectorscope_alpha: float
    """ Opacity of the points

    :type: float
    """

    waveform_alpha: float
    """ Opacity of the points

    :type: float
    """

    waveform_mode: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Sequence(bpy_struct):
    """Sequence strip in the sequence editor"""

    blend_alpha: float
    """ Percentage of how much the strip's colors affect other strips

    :type: float
    """

    blend_type: str
    """ Method for controlling how the strip combines with other strips

    :type: str
    """

    channel: int
    """ Y position of the sequence strip

    :type: int
    """

    color_tag: str
    """ Color tag for a strip

    :type: str
    """

    effect_fader: float
    """ Custom fade value

    :type: float
    """

    frame_duration: int
    """ The length of the contents of this strip before the handles are applied

    :type: int
    """

    frame_final_duration: int
    """ The length of the contents of this strip after the handles are applied

    :type: int
    """

    frame_final_end: int
    """ End frame displayed in the sequence editor after offsets are applied

    :type: int
    """

    frame_final_start: int
    """ Start frame displayed in the sequence editor after offsets are applied, setting this is equivalent to moving the handle, not the actual start frame

    :type: int
    """

    frame_offset_end: int
    """ 

    :type: int
    """

    frame_offset_start: int
    """ 

    :type: int
    """

    frame_start: int
    """ X position where the strip begins

    :type: int
    """

    frame_still_end: int
    """ 

    :type: int
    """

    frame_still_start: int
    """ 

    :type: int
    """

    lock: bool
    """ Lock strip so that it cannot be transformed

    :type: bool
    """

    modifiers: SequenceModifiers
    """ Modifiers affecting this strip

    :type: SequenceModifiers
    """

    mute: bool
    """ Disable strip so that it cannot be viewed in the output

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    override_cache_settings: bool
    """ Override global cache settings

    :type: bool
    """

    select: bool
    """ 

    :type: bool
    """

    select_left_handle: bool
    """ 

    :type: bool
    """

    select_right_handle: bool
    """ 

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

    use_cache_composite: bool
    """ Cache intermediate composited images, for faster tweaking of stacked strips at the cost of memory usage

    :type: bool
    """

    use_cache_preprocessed: bool
    """ Cache preprocessed images, for faster tweaking of effects at the cost of memory usage

    :type: bool
    """

    use_cache_raw: bool
    """ Cache raw images read from disk, for faster tweaking of strip parameters at the cost of memory usage

    :type: bool
    """

    use_default_fade: bool
    """ Fade effect using the built-in default (usually make transition as long as effect strip)

    :type: bool
    """

    use_linear_modifiers: bool
    """ Calculate modifiers in linear space instead of sequencer's space

    :type: bool
    """

    def update(self, data: bool | typing.Any | None = False):
        """Update the strip dimensions

        :param data: Data, Update strip data
        :type data: bool | typing.Any | None
        """
        ...

    def strip_elem_from_frame(self, frame: int | None) -> SequenceElement:
        """Return the strip element from a given frame or None

        :param frame: Frame, The frame to get the strip element from
        :type frame: int | None
        :return: strip element of the current frame
        :rtype: SequenceElement
        """
        ...

    def swap(self, other: Sequence):
        """swap

        :param other: Other
        :type other: Sequence
        """
        ...

    def move_to_meta(self, meta_sequence: Sequence):
        """move_to_meta

        :param meta_sequence: Destination Meta Sequence, Meta to move the strip into
        :type meta_sequence: Sequence
        """
        ...

    def parent_meta(self) -> Sequence:
        """Parent meta

        :return: Parent Meta
        :rtype: Sequence
        """
        ...

    def invalidate_cache(self, type: str):
        """Invalidate cached images for strip and all dependent strips

        :param type: Type, Cache Type
        :type type: str
        """
        ...

    def split(self, frame: int | None, split_method: str) -> Sequence:
        """Split Sequence

        :param frame: Frame where to split the strip
        :type frame: int | None
        :param split_method:
        :type split_method: str
        :return: Right side Sequence
        :rtype: Sequence
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceColorBalanceData(bpy_struct):
    """Color balance parameters for a sequence strip and its modifiers"""

    correction_method: str
    """ 

    :type: str
    """

    gain: bpy_prop_array[float]
    """ Color balance gain (highlights)

    :type: bpy_prop_array[float]
    """

    gamma: bpy_prop_array[float]
    """ Color balance gamma (midtones)

    :type: bpy_prop_array[float]
    """

    invert_gain: bool
    """ Invert the gain color`

    :type: bool
    """

    invert_gamma: bool
    """ Invert the gamma color

    :type: bool
    """

    invert_lift: bool
    """ Invert the lift color

    :type: bool
    """

    invert_offset: bool
    """ Invert the offset color

    :type: bool
    """

    invert_power: bool
    """ Invert the power color

    :type: bool
    """

    invert_slope: bool
    """ Invert the slope color`

    :type: bool
    """

    lift: bpy_prop_array[float]
    """ Color balance lift (shadows)

    :type: bpy_prop_array[float]
    """

    offset: bpy_prop_array[float]
    """ Correction for entire tonal range

    :type: bpy_prop_array[float]
    """

    power: bpy_prop_array[float]
    """ Correction for midtones

    :type: bpy_prop_array[float]
    """

    slope: bpy_prop_array[float]
    """ Correction for highlights

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceCrop(bpy_struct):
    """Cropping parameters for a sequence strip"""

    max_x: int
    """ Number of pixels to crop from the right side

    :type: int
    """

    max_y: int
    """ Number of pixels to crop from the top

    :type: int
    """

    min_x: int
    """ Number of pixels to crop from the left side

    :type: int
    """

    min_y: int
    """ Number of pixels to crop from the bottom

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceEditor(bpy_struct):
    """Sequence editing data for a Scene data-block"""

    active_strip: Sequence | None
    """ Sequencer's active strip

    :type: Sequence | None
    """

    channels: bpy_prop_collection[SequenceTimelineChannel]
    """ 

    :type: bpy_prop_collection[SequenceTimelineChannel]
    """

    meta_stack: bpy_prop_collection[Sequence]
    """ Meta strip stack, last is currently edited meta strip

    :type: bpy_prop_collection[Sequence]
    """

    overlay_frame: int
    """ Number of frames to offset

    :type: int
    """

    proxy_dir: str
    """ 

    :type: str
    """

    proxy_storage: str
    """ How to store proxies for this project

    :type: str
    """

    sequences: SequencesTopLevel
    """ Top-level strips only

    :type: SequencesTopLevel
    """

    sequences_all: bpy_prop_collection[Sequence]
    """ All strips, recursively including those inside metastrips

    :type: bpy_prop_collection[Sequence]
    """

    show_cache: bool
    """ Visualize cached images on the timeline

    :type: bool
    """

    show_cache_composite: bool
    """ Visualize cached composite images

    :type: bool
    """

    show_cache_final_out: bool
    """ Visualize cached complete frames

    :type: bool
    """

    show_cache_preprocessed: bool
    """ Visualize cached pre-processed images

    :type: bool
    """

    show_cache_raw: bool
    """ Visualize cached raw images

    :type: bool
    """

    show_overlay_frame: bool
    """ Partial overlay on top of the sequencer with a frame offset

    :type: bool
    """

    use_cache_composite: bool
    """ Cache intermediate composited images, for faster tweaking of stacked strips at the cost of memory usage

    :type: bool
    """

    use_cache_final: bool
    """ Cache final image for each frame

    :type: bool
    """

    use_cache_preprocessed: bool
    """ Cache pre-processed images, for faster tweaking of effects at the cost of memory usage

    :type: bool
    """

    use_cache_raw: bool
    """ Cache raw images read from disk, for faster tweaking of strip parameters at the cost of memory usage

    :type: bool
    """

    use_overlay_frame_lock: bool
    """ 

    :type: bool
    """

    use_prefetch: bool
    """ Render frames ahead of current frame in the background for faster playback

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceElement(bpy_struct):
    """Sequence strip data for a single frame"""

    filename: str
    """ Name of the source file

    :type: str
    """

    orig_fps: float
    """ Original frames per second

    :type: float
    """

    orig_height: int
    """ Original image height

    :type: int
    """

    orig_width: int
    """ Original image width

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceElements(bpy_prop_collection[SequenceElement], bpy_struct):
    """Collection of SequenceElement"""

    def append(self, filename: str | typing.Any) -> SequenceElement:
        """Push an image from ImageSequence.directory

        :param filename: Filepath to image
        :type filename: str | typing.Any
        :return: New SequenceElement
        :rtype: SequenceElement
        """
        ...

    def pop(self, index: int | None):
        """Pop an image off the collection

        :param index: Index of image to remove
        :type index: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceModifier(bpy_struct):
    """Modifier for sequence strip"""

    input_mask_id: Mask
    """ Mask ID used as mask input for the modifier

    :type: Mask
    """

    input_mask_strip: Sequence
    """ Strip used as mask input for the modifier

    :type: Sequence
    """

    input_mask_type: str
    """ Type of input data used for mask

    :type: str
    """

    mask_time: str
    """ Time to use for the Mask animation

    :type: str
    """

    mute: bool
    """ Mute this modifier

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    show_expanded: bool
    """ Mute expanded settings for the modifier

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceModifiers(bpy_prop_collection[SequenceModifier], bpy_struct):
    """Collection of strip modifiers"""

    def new(self, name: str | typing.Any, type: str | None) -> SequenceModifier:
        """Add a new modifier

        :param name: New name for the modifier
        :type name: str | typing.Any
        :param type: Modifier type to add
        :type type: str | None
        :return: Newly created modifier
        :rtype: SequenceModifier
        """
        ...

    def remove(self, modifier: SequenceModifier):
        """Remove an existing modifier from the sequence

        :param modifier: Modifier to remove
        :type modifier: SequenceModifier
        """
        ...

    def clear(self):
        """Remove all modifiers from the sequence"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceProxy(bpy_struct):
    """Proxy parameters for a sequence strip"""

    build_100: bool
    """ Build 100% proxy resolution

    :type: bool
    """

    build_25: bool
    """ Build 25% proxy resolution

    :type: bool
    """

    build_50: bool
    """ Build 50% proxy resolution

    :type: bool
    """

    build_75: bool
    """ Build 75% proxy resolution

    :type: bool
    """

    build_free_run: bool
    """ Build free run time code index

    :type: bool
    """

    build_free_run_rec_date: bool
    """ Build free run time code index using Record Date/Time

    :type: bool
    """

    build_record_run: bool
    """ Build record run time code index

    :type: bool
    """

    directory: str
    """ Location to store the proxy files

    :type: str
    """

    filepath: str
    """ Location of custom proxy file

    :type: str
    """

    quality: int
    """ Quality of proxies to build

    :type: int
    """

    timecode: str
    """ Method for reading the inputs timecode

    :type: str
    """

    use_overwrite: bool
    """ Overwrite existing proxy files when building

    :type: bool
    """

    use_proxy_custom_directory: bool
    """ Use a custom directory to store data

    :type: bool
    """

    use_proxy_custom_file: bool
    """ Use a custom file to read proxy data from

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceTimelineChannel(bpy_struct):
    lock: bool
    """ 

    :type: bool
    """

    mute: bool
    """ 

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequenceTransform(bpy_struct):
    """Transform parameters for a sequence strip"""

    filter: str
    """ Type of filter to use for image transformation

    :type: str
    """

    offset_x: float
    """ Move along X axis

    :type: float
    """

    offset_y: float
    """ Move along Y axis

    :type: float
    """

    origin: bpy_prop_array[float]
    """ Origin of image for transformation

    :type: bpy_prop_array[float]
    """

    rotation: float
    """ Rotate around image center

    :type: float
    """

    scale_x: float
    """ Scale along X axis

    :type: float
    """

    scale_y: float
    """ Scale along Y axis

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequencerPreviewOverlay(bpy_struct):
    show_annotation: bool
    """ Show annotations for this view

    :type: bool
    """

    show_cursor: bool
    """ 

    :type: bool
    """

    show_image_outline: bool
    """ 

    :type: bool
    """

    show_metadata: bool
    """ Show metadata of first visible strip

    :type: bool
    """

    show_safe_areas: bool
    """ Show TV title safe and action safe areas in preview

    :type: bool
    """

    show_safe_center: bool
    """ Show safe areas to fit content in a different aspect ratio

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequencerTimelineOverlay(bpy_struct):
    show_fcurves: bool
    """ Display strip opacity/volume curve

    :type: bool
    """

    show_grid: bool
    """ Show vertical grid lines

    :type: bool
    """

    show_strip_duration: bool
    """ 

    :type: bool
    """

    show_strip_name: bool
    """ 

    :type: bool
    """

    show_strip_offset: bool
    """ Display strip in/out offsets

    :type: bool
    """

    show_strip_source: bool
    """ Display path to source file, or name of source datablock

    :type: bool
    """

    show_strip_tag_color: bool
    """ Display the strip color tags in the sequencer

    :type: bool
    """

    show_thumbnails: bool
    """ Show strip thumbnails

    :type: bool
    """

    waveform_display_type: str
    """ How Waveforms are displayed

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequencerToolSettings(bpy_struct):
    fit_method: str
    """ Scale fit method

    :type: str
    """

    overlap_mode: str
    """ How to resolve overlap after transformation

    :type: str
    """

    pivot_point: str
    """ Rotation or scaling pivot point

    :type: str
    """

    snap_distance: int
    """ Maximum distance for snapping in pixels

    :type: int
    """

    snap_ignore_muted: bool
    """ Don't snap to hidden strips

    :type: bool
    """

    snap_ignore_sound: bool
    """ Don't snap to sound strips

    :type: bool
    """

    snap_to_current_frame: bool
    """ Snap to current frame

    :type: bool
    """

    snap_to_hold_offset: bool
    """ Snap to strip hold offsets

    :type: bool
    """

    use_snap_current_frame_to_strips: bool
    """ Snap current frame to strip start or end

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequencesMeta(bpy_prop_collection[Sequence], bpy_struct):
    """Collection of Sequences"""

    def new_clip(
        self,
        name: str | typing.Any,
        clip: MovieClip,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new movie clip sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param clip: Movie clip to add
        :type clip: MovieClip
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_mask(
        self,
        name: str | typing.Any,
        mask: Mask,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new mask sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param mask: Mask to add
        :type mask: Mask
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_scene(
        self,
        name: str | typing.Any,
        scene: Scene,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new scene sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param scene: Scene to add
        :type scene: Scene
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_image(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
        fit_method: str | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new image sequence

                :param name: Name for the new sequence
                :type name: str | typing.Any
                :param filepath: Filepath to image
                :type filepath: str | typing.Any
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: str | None
                :return: New Sequence
                :rtype: Sequence
        """
        ...

    def new_movie(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
        fit_method: str | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new movie sequence

                :param name: Name for the new sequence
                :type name: str | typing.Any
                :param filepath: Filepath to movie
                :type filepath: str | typing.Any
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: str | None
                :return: New Sequence
                :rtype: Sequence
        """
        ...

    def new_sound(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new sound sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param filepath: Filepath to movie
        :type filepath: str | typing.Any
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_meta(
        self, name: str | typing.Any, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new meta sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_effect(
        self,
        name: str | typing.Any,
        type: str | None,
        channel: int | None,
        frame_start: int | None,
        frame_end: typing.Any | None = 0,
        seq1: Sequence | None = None,
        seq2: Sequence | None = None,
        seq3: Sequence | None = None,
    ) -> Sequence:
        """Add a new effect sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param type: Type, type for the new sequence
        :type type: str | None
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :param frame_end: The end frame for the new sequence
        :type frame_end: typing.Any | None
        :param seq1: Sequence 1 for effect
        :type seq1: Sequence | None
        :param seq2: Sequence 2 for effect
        :type seq2: Sequence | None
        :param seq3: Sequence 3 for effect
        :type seq3: Sequence | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def remove(self, sequence: Sequence):
        """Remove a Sequence

        :param sequence: Sequence to remove
        :type sequence: Sequence
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SequencesTopLevel(bpy_prop_collection[Sequence], bpy_struct):
    """Collection of Sequences"""

    def new_clip(
        self,
        name: str | typing.Any,
        clip: MovieClip,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new movie clip sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param clip: Movie clip to add
        :type clip: MovieClip
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_mask(
        self,
        name: str | typing.Any,
        mask: Mask,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new mask sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param mask: Mask to add
        :type mask: Mask
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_scene(
        self,
        name: str | typing.Any,
        scene: Scene,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new scene sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param scene: Scene to add
        :type scene: Scene
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_image(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
        fit_method: str | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new image sequence

                :param name: Name for the new sequence
                :type name: str | typing.Any
                :param filepath: Filepath to image
                :type filepath: str | typing.Any
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: str | None
                :return: New Sequence
                :rtype: Sequence
        """
        ...

    def new_movie(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
        fit_method: str | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new movie sequence

                :param name: Name for the new sequence
                :type name: str | typing.Any
                :param filepath: Filepath to movie
                :type filepath: str | typing.Any
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: str | None
                :return: New Sequence
                :rtype: Sequence
        """
        ...

    def new_sound(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new sound sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param filepath: Filepath to movie
        :type filepath: str | typing.Any
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_meta(
        self, name: str | typing.Any, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new meta sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_effect(
        self,
        name: str | typing.Any,
        type: str | None,
        channel: int | None,
        frame_start: int | None,
        frame_end: typing.Any | None = 0,
        seq1: Sequence | None = None,
        seq2: Sequence | None = None,
        seq3: Sequence | None = None,
    ) -> Sequence:
        """Add a new effect sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param type: Type, type for the new sequence
        :type type: str | None
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :param frame_end: The end frame for the new sequence
        :type frame_end: typing.Any | None
        :param seq1: Sequence 1 for effect
        :type seq1: Sequence | None
        :param seq2: Sequence 2 for effect
        :type seq2: Sequence | None
        :param seq3: Sequence 3 for effect
        :type seq3: Sequence | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def remove(self, sequence: Sequence):
        """Remove a Sequence

        :param sequence: Sequence to remove
        :type sequence: Sequence
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ShaderFx(bpy_struct):
    """Effect affecting the grease pencil object"""

    name: str
    """ Effect name

    :type: str
    """

    show_expanded: bool
    """ Set effect expansion in the user interface

    :type: bool
    """

    show_in_editmode: bool
    """ Display effect in Edit mode

    :type: bool
    """

    show_render: bool
    """ Use effect during render

    :type: bool
    """

    show_viewport: bool
    """ Display effect in viewport

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ShapeKey(bpy_struct):
    """Shape key in a shape keys data-block"""

    data: bpy_prop_collection[UnknownType]
    """ 

    :type: bpy_prop_collection[UnknownType]
    """

    frame: float
    """ Frame for absolute keys

    :type: float
    """

    interpolation: str
    """ Interpolation type for absolute shape keys

    :type: str
    """

    mute: bool
    """ Toggle this shape key

    :type: bool
    """

    name: str
    """ Name of Shape Key

    :type: str
    """

    relative_key: ShapeKey
    """ Shape used as a relative key

    :type: ShapeKey
    """

    slider_max: float
    """ Maximum for slider

    :type: float
    """

    slider_min: float
    """ Minimum for slider

    :type: float
    """

    value: float
    """ Value of shape key at the current frame

    :type: float
    """

    vertex_group: str
    """ Vertex weight group, to blend with basis shape

    :type: str
    """

    def normals_vertex_get(self) -> float:
        """Compute local space vertices' normals for this shape key

        :return: normals
        :rtype: float
        """
        ...

    def normals_polygon_get(self) -> float:
        """Compute local space faces' normals for this shape key

        :return: normals
        :rtype: float
        """
        ...

    def normals_split_get(self) -> float:
        """Compute local space face corners' normals for this shape key

        :return: normals
        :rtype: float
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ShapeKeyBezierPoint(bpy_struct):
    """Point in a shape key for Bezier curves"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_left: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_right: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ShapeKeyCurvePoint(bpy_struct):
    """Point in a shape key for curves"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class ShapeKeyPoint(bpy_struct):
    """Point in a shape key"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SoftBodySettings(bpy_struct):
    """Soft body simulation settings for an object"""

    aero: int
    """ Make edges 'sail'

    :type: int
    """

    aerodynamics_type: str
    """ Method of calculating aerodynamic interaction

    :type: str
    """

    ball_damp: float
    """ Blending to inelastic collision

    :type: float
    """

    ball_size: float
    """ Absolute ball size or factor if not manually adjusted

    :type: float
    """

    ball_stiff: float
    """ Ball inflating pressure

    :type: float
    """

    bend: float
    """ Bending Stiffness

    :type: float
    """

    choke: int
    """ 'Viscosity' inside collision target

    :type: int
    """

    collision_collection: Collection
    """ Limit colliders to this collection

    :type: Collection
    """

    collision_type: str
    """ Choose Collision Type

    :type: str
    """

    damping: float
    """ Edge spring friction

    :type: float
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    error_threshold: float
    """ The Runge-Kutta ODE solver error limit, low value gives more precision, high values speed

    :type: float
    """

    friction: float
    """ General media friction for point movements

    :type: float
    """

    fuzzy: int
    """ Fuzziness while on collision, high values make collision handling faster but less stable

    :type: int
    """

    goal_default: float
    """ Default Goal (vertex target position) value

    :type: float
    """

    goal_friction: float
    """ Goal (vertex target position) friction

    :type: float
    """

    goal_max: float
    """ Goal maximum, vertex weights are scaled to match this range

    :type: float
    """

    goal_min: float
    """ Goal minimum, vertex weights are scaled to match this range

    :type: float
    """

    goal_spring: float
    """ Goal (vertex target position) spring stiffness

    :type: float
    """

    gravity: float
    """ Apply gravitation to point movement

    :type: float
    """

    location_mass_center: bpy_prop_array[float]
    """ Location of center of mass

    :type: bpy_prop_array[float]
    """

    mass: float
    """ General Mass value

    :type: float
    """

    plastic: int
    """ Permanent deform

    :type: int
    """

    pull: float
    """ Edge spring stiffness when longer than rest length

    :type: float
    """

    push: float
    """ Edge spring stiffness when shorter than rest length

    :type: float
    """

    rotation_estimate: list[list[float]] | tuple[
        tuple[float, float, float],
        tuple[float, float, float],
        tuple[float, float, float],
    ]
    """ Estimated rotation matrix

    :type: list[list[float]] | tuple[tuple[float, float, float], tuple[float, float, float], tuple[float, float, float]]
    """

    scale_estimate: list[list[float]] | tuple[
        tuple[float, float, float],
        tuple[float, float, float],
        tuple[float, float, float],
    ]
    """ Estimated scale matrix

    :type: list[list[float]] | tuple[tuple[float, float, float], tuple[float, float, float], tuple[float, float, float]]
    """

    shear: float
    """ Shear Stiffness

    :type: float
    """

    speed: float
    """ Tweak timing for physics to control frequency and speed

    :type: float
    """

    spring_length: int
    """ Alter spring length to shrink/blow up (unit %) 0 to disable

    :type: int
    """

    step_max: int
    """ Maximal # solver steps/frame

    :type: int
    """

    step_min: int
    """ Minimal # solver steps/frame

    :type: int
    """

    use_auto_step: bool
    """ Use velocities for automagic step sizes

    :type: bool
    """

    use_diagnose: bool
    """ Turn on SB diagnose console prints

    :type: bool
    """

    use_edge_collision: bool
    """ Edges collide too

    :type: bool
    """

    use_edges: bool
    """ Use Edges as springs

    :type: bool
    """

    use_estimate_matrix: bool
    """ Store the estimated transforms in the soft body settings

    :type: bool
    """

    use_face_collision: bool
    """ Faces collide too, can be very slow

    :type: bool
    """

    use_goal: bool
    """ Define forces for vertices to stick to animated position

    :type: bool
    """

    use_self_collision: bool
    """ Enable naive vertex ball self collision

    :type: bool
    """

    use_stiff_quads: bool
    """ Add diagonal springs on 4-gons

    :type: bool
    """

    vertex_group_goal: str
    """ Control point weight values

    :type: str
    """

    vertex_group_mass: str
    """ Control point mass values

    :type: str
    """

    vertex_group_spring: str
    """ Control point spring strength values

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Space(bpy_struct):
    """Space data for a screen area"""

    show_locked_time: bool
    """ Synchronize the visible timeline range with other time-based editors

    :type: bool
    """

    show_region_header: bool
    """ 

    :type: bool
    """

    type: str
    """ Space data type

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

    def draw_handler_add(
        self,
        callback: typing.Any | None,
        args: tuple | None,
        region_type: str | None,
        draw_type: str | None,
    ) -> typing.Any:
        """Add a new draw handler to this space type.
        It will be called every time the specified region in the space type will be drawn.
        Note: All arguments are positional only for now.

                :param callback: A function that will be called when the region is drawn.
        It gets the specified arguments as input.
                :type callback: typing.Any | None
                :param args: Arguments that will be passed to the callback.
                :type args: tuple | None
                :param region_type: The region type the callback draws in; usually WINDOW. (`bpy.types.Region.type`)
                :type region_type: str | None
                :param draw_type: Usually POST_PIXEL for 2D drawing and POST_VIEW for 3D drawing. In some cases PRE_VIEW can be used. BACKDROP can be used for backdrops in the node editor.
                :type draw_type: str | None
                :return: Handler that can be removed later on.
                :rtype: typing.Any
        """
        ...

    def draw_handler_remove(self, handler: typing.Any | None, region_type: str | None):
        """Remove a draw handler that was added previously.

        :param handler: The draw handler that should be removed.
        :type handler: typing.Any | None
        :param region_type: Region type the callback was added to.
        :type region_type: str | None
        """
        ...

class SpaceImageOverlay(bpy_struct):
    """Settings for display of overlays in the UV/Image editor"""

    show_grid_background: bool
    """ Show the grid background and borders

    :type: bool
    """

    show_overlays: bool
    """ Display overlays like UV Maps and Metadata

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpaceNodeEditorPath(bpy_prop_collection[NodeTreePath], bpy_struct):
    """Get the node tree path as a string"""

    to_string: str
    """ 

    :type: str
    """

    def clear(self):
        """Reset the node tree path"""
        ...

    def start(self, node_tree: NodeTree | None):
        """Set the root node tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        """
        ...

    def append(self, node_tree: NodeTree | None, node: Node | None = None):
        """Append a node group tree to the path

        :param node_tree: Node Tree, Node tree to append to the node editor path
        :type node_tree: NodeTree | None
        :param node: Node, Group node linking to this node tree
        :type node: Node | None
        """
        ...

    def pop(self):
        """Remove the last node tree from the path"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpaceNodeOverlay(bpy_struct):
    """Settings for display of overlays in the Node Editor"""

    show_context_path: bool
    """ Display breadcrumbs for the editor's context

    :type: bool
    """

    show_named_attributes: bool
    """ Show when nodes are using named attributes

    :type: bool
    """

    show_overlays: bool
    """ Display overlays like colored or dashed wires

    :type: bool
    """

    show_timing: bool
    """ Display each node's last execution time

    :type: bool
    """

    show_wire_color: bool
    """ Color node links based on their connected sockets

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpaceUVEditor(bpy_struct):
    """UV editor data for the image editor space"""

    custom_grid_subdivisions: int
    """ Number of grid units in UV space that make one UV Unit

    :type: int
    """

    display_stretch_type: str
    """ Type of stretch to display

    :type: str
    """

    edge_display_type: str
    """ Display style for UV edges

    :type: str
    """

    lock_bounds: bool
    """ Constraint to stay within the image bounds while editing

    :type: bool
    """

    pixel_snap_mode: str
    """ Snap UVs to pixels while editing

    :type: str
    """

    show_faces: bool
    """ Display faces over the image

    :type: bool
    """

    show_metadata: bool
    """ Display metadata properties of the image

    :type: bool
    """

    show_modified_edges: bool
    """ Display edges after modifiers are applied

    :type: bool
    """

    show_pixel_coords: bool
    """ Display UV coordinates in pixels rather than from 0.0 to 1.0

    :type: bool
    """

    show_stretch: bool
    """ Display faces colored according to the difference in shape between UVs and their 3D coordinates (blue for low distortion, red for high distortion)

    :type: bool
    """

    show_texpaint: bool
    """ Display overlay of texture paint uv layer

    :type: bool
    """

    tile_grid_shape: bpy_prop_array[int]
    """ How many tiles will be shown in the background

    :type: bpy_prop_array[int]
    """

    use_custom_grid: bool
    """ Use a grid with a user-defined number of steps

    :type: bool
    """

    use_live_unwrap: bool
    """ Continuously unwrap the selected UV island while transforming pinned vertices

    :type: bool
    """

    uv_opacity: float
    """ Opacity of UV overlays

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Spline(bpy_struct):
    """Element of a curve, either NURBS, Bezier or Polyline or a character with text objects"""

    bezier_points: SplineBezierPoints
    """ Collection of points for Bezier curves only

    :type: SplineBezierPoints
    """

    character_index: int
    """ Location of this character in the text data (only for text curves)

    :type: int
    """

    hide: bool
    """ Hide this curve in Edit mode

    :type: bool
    """

    material_index: int
    """ Material slot index of this curve

    :type: int
    """

    order_u: int
    """ NURBS order in the U direction (for splines and surfaces, higher values let points influence a greater area)

    :type: int
    """

    order_v: int
    """ NURBS order in the V direction (for surfaces only, higher values let points influence a greater area)

    :type: int
    """

    point_count_u: int
    """ Total number points for the curve or surface in the U direction

    :type: int
    """

    point_count_v: int
    """ Total number points for the surface on the V direction

    :type: int
    """

    points: SplinePoints
    """ Collection of points that make up this poly or nurbs spline

    :type: SplinePoints
    """

    radius_interpolation: str
    """ The type of radius interpolation for Bezier curves

    :type: str
    """

    resolution_u: int
    """ Curve or Surface subdivisions per segment

    :type: int
    """

    resolution_v: int
    """ Surface subdivisions per segment

    :type: int
    """

    tilt_interpolation: str
    """ The type of tilt interpolation for 3D, Bezier curves

    :type: str
    """

    type: str
    """ The interpolation type for this curve element

    :type: str
    """

    use_bezier_u: bool
    """ Make this nurbs curve or surface act like a Bezier spline in the U direction

    :type: bool
    """

    use_bezier_v: bool
    """ Make this nurbs surface act like a Bezier spline in the V direction

    :type: bool
    """

    use_cyclic_u: bool
    """ Make this curve or surface a closed loop in the U direction

    :type: bool
    """

    use_cyclic_v: bool
    """ Make this surface a closed loop in the V direction

    :type: bool
    """

    use_endpoint_u: bool
    """ Make this nurbs curve or surface meet the endpoints in the U direction

    :type: bool
    """

    use_endpoint_v: bool
    """ Make this nurbs surface meet the endpoints in the V direction

    :type: bool
    """

    use_smooth: bool
    """ Smooth the normals of the surface or beveled curve

    :type: bool
    """

    def calc_length(self, resolution: typing.Any | None = 0) -> float:
        """Calculate spline length

        :param resolution: Resolution, Spline resolution to be used, 0 defaults to the resolution_u
        :type resolution: typing.Any | None
        :return: Length, Length of the polygonaly approximated spline
        :rtype: float
        """
        ...

    def valid_message(self, direction: int | None) -> str:
        """Return the message

        :param direction: Direction, The direction where 0-1 maps to U-V
        :type direction: int | None
        :return: Return value, The message or an empty string when there is no error
        :rtype: str
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SplineBezierPoints(bpy_prop_collection[BezierSplinePoint], bpy_struct):
    """Collection of spline Bezier points"""

    def add(self, count: int | None):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SplinePoint(bpy_struct):
    """Spline point without handles"""

    co: bpy_prop_array[float]
    """ Point coordinates

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    select: bool
    """ Selection status

    :type: bool
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    weight: float
    """ NURBS weight

    :type: float
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SplinePoints(bpy_prop_collection[SplinePoint], bpy_struct):
    """Collection of spline points"""

    def add(self, count: int | None):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpreadsheetColumn(bpy_struct):
    """Persistent data associated with a spreadsheet column"""

    data_type: str
    """ The data type of the corresponding column visible in the spreadsheet

    :type: str
    """

    id: SpreadsheetColumnID
    """ Data used to identify the corresponding data from the data source

    :type: SpreadsheetColumnID
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpreadsheetColumnID(bpy_struct):
    """Data used to identify a spreadsheet column"""

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpreadsheetContext(bpy_struct):
    """Element of spreadsheet context path"""

    show_region_channels: bool
    """ 

    :type: bool
    """

    show_region_footer: bool
    """ 

    :type: bool
    """

    type: str
    """ Type of the context

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpreadsheetContextPath(bpy_prop_collection[SpreadsheetContext], bpy_struct):
    def append(self, type: str | None) -> SpreadsheetContext:
        """Append a context path element

        :param type: type
        :type type: str | None
        :return: Newly created context path element
        :rtype: SpreadsheetContext
        """
        ...

    def clear(self):
        """Clear entire context path"""
        ...

    def guess(self):
        """Guess the context path from the current context"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class SpreadsheetRowFilter(bpy_struct):
    column_name: str
    """ 

    :type: str
    """

    enabled: bool
    """ 

    :type: bool
    """

    operation: str
    """ 

    :type: str
    """

    show_expanded: bool
    """ 

    :type: bool
    """

    threshold: float
    """ How close float values need to be to be equal

    :type: float
    """

    value_boolean: bool
    """ 

    :type: bool
    """

    value_byte_color: bpy_prop_array[int]
    """ 

    :type: bpy_prop_array[int]
    """

    value_color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    value_float: float
    """ 

    :type: float
    """

    value_float2: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    value_float3: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    value_int: int
    """ 

    :type: int
    """

    value_int8: int
    """ 

    :type: int
    """

    value_string: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Stereo3dDisplay(bpy_struct):
    """Settings for stereo 3D display"""

    anaglyph_type: str
    """ 

    :type: str
    """

    display_mode: str
    """ 

    :type: str
    """

    interlace_type: str
    """ 

    :type: str
    """

    use_interlace_swap: bool
    """ Swap left and right stereo channels

    :type: bool
    """

    use_sidebyside_crosseyed: bool
    """ Right eye should see left image and vice versa

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Stereo3dFormat(bpy_struct):
    """Settings for stereo output"""

    anaglyph_type: str
    """ 

    :type: str
    """

    display_mode: str
    """ 

    :type: str
    """

    interlace_type: str
    """ 

    :type: str
    """

    use_interlace_swap: bool
    """ Swap left and right stereo channels

    :type: bool
    """

    use_sidebyside_crosseyed: bool
    """ Right eye should see left image and vice versa

    :type: bool
    """

    use_squeezed_frame: bool
    """ Combine both views in a squeezed image

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class StringAttributeValue(bpy_struct):
    """String value in geometry attribute"""

    value: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class Struct(bpy_struct):
    """RNA structure definition"""

    base: Struct
    """ Struct definition this is derived from

    :type: Struct
    """

    description: str
    """ Description of the Struct's purpose

    :type: str
    """

    functions: bpy_prop_collection[Function]
    """ 

    :type: bpy_prop_collection[Function]
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    name: str
    """ Human readable name

    :type: str
    """

    name_property: StringProperty
    """ Property that gives the name of the struct

    :type: StringProperty
    """

    nested: Struct
    """ Struct in which this struct is always nested, and to which it logically belongs

    :type: Struct
    """

    properties: bpy_prop_collection[Property]
    """ Properties in the struct

    :type: bpy_prop_collection[Property]
    """

    property_tags: bpy_prop_collection[EnumPropertyItem]
    """ Tags that properties can use to influence behavior

    :type: bpy_prop_collection[EnumPropertyItem]
    """

    translation_context: str
    """ Translation context of the struct's name

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class StudioLight(bpy_struct):
    """Studio light"""

    has_specular_highlight_pass: bool
    """ Studio light image file has separate "diffuse" and "specular" passes

    :type: bool
    """

    index: int
    """ 

    :type: int
    """

    is_user_defined: bool
    """ 

    :type: bool
    """

    light_ambient: bpy_prop_array[float]
    """ Color of the ambient light that uniformly lit the scene

    :type: bpy_prop_array[float]
    """

    name: str
    """ 

    :type: str
    """

    path: str
    """ 

    :type: str
    """

    path_irr_cache: str
    """ Path where the irradiance cache is stored

    :type: str
    """

    path_sh_cache: str
    """ Path where the spherical harmonics cache is stored

    :type: str
    """

    solid_lights: bpy_prop_collection[UserSolidLight]
    """ Lights user to display objects in solid draw mode

    :type: bpy_prop_collection[UserSolidLight]
    """

    spherical_harmonics_coefficients: list[list[float]] | tuple[
        tuple[float, float, float, float],
        tuple[float, float, float, float],
        tuple[float, float, float, float],
    ]
    """ 

    :type: list[list[float]] | tuple[tuple[float, float, float, float], tuple[float, float, float, float], tuple[float, float, float, float]]
    """

    type: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class StudioLights(bpy_prop_collection[StudioLight], bpy_struct):
    """Collection of studio lights"""

    def load(self, path: str | typing.Any, type: str | None) -> StudioLight:
        """Load studiolight from file

        :param path: File Path, File path where the studio light file can be found
        :type path: str | typing.Any
        :param type: Type, The type for the new studio light
        :type type: str | None
        :return: Newly created StudioLight
        :rtype: StudioLight
        """
        ...

    def new(self, path: str | typing.Any) -> StudioLight:
        """Create studiolight from default lighting

        :param path: Path, Path to the file that will contain the lighting info (without extension)
        :type path: str | typing.Any
        :return: Newly created StudioLight
        :rtype: StudioLight
        """
        ...

    def remove(self, studio_light: StudioLight):
        """Remove a studio light

        :param studio_light: The studio light to remove
        :type studio_light: StudioLight
        """
        ...

    def refresh(self):
        """Refresh Studio Lights from disk"""
        ...

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class TexMapping(bpy_struct):
    """Texture coordinate mapping settings"""

    mapping: str
    """ 

    :type: str
    """

    mapping_x: str
    """ 

    :type: str
    """

    mapping_y: str
    """ 

    :type: str
    """

    mapping_z: str
    """ 

    :type: str
    """

    max: bpy_prop_array[float]
    """ Maximum value for clipping

    :type: bpy_prop_array[float]
    """

    min: bpy_prop_array[float]
    """ Minimum value for clipping

    :type: bpy_prop_array[float]
    """

    rotation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    scale: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    translation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    use_max: bool
    """ Whether to use maximum clipping value

    :type: bool
    """

    use_min: bool
    """ Whether to use minimum clipping value

    :type: bool
    """

    vector_type: str
    """ Type of vector that the mapping transforms

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class TexPaintSlot(bpy_struct):
    """Slot that contains information about texture painting"""

    icon_value: int
    """ Paint slot icon

    :type: int
    """

    is_valid: bool
    """ Slot has a valid image and UV map

    :type: bool
    """

    name: str
    """ Name of the slot

    :type: str
    """

    uv_layer: str
    """ Name of UV map

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class TextBox(bpy_struct):
    """Text bounding box for layout"""

    height: float
    """ 

    :type: float
    """

    width: float
    """ 

    :type: float
    """

    x: float
    """ 

    :type: float
    """

    y: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """
        ...

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """
        ...

class TextCharacterFormat(bpy_struct):
    """Text character formatting settings"""

    kerning: int
    """ Spacing between characters

    :type: int
    """

    material_index: int
    """ Material slot index of this character

    :type: int
    """

    use_bold: bool
    """ 

    :type: bool
    """

    use_italic: bool
    """ 

    :type: bool
    """

    use_small_caps: bool
    """ 

    :type: bool
    """

    use_underline: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
 