"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedCodeDeployedFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_synthetics_alpha_1 = require("@aws-cdk/aws-synthetics-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_codedeploy_1 = require("aws-cdk-lib/aws-codedeploy");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_ecs_patterns_1 = require("aws-cdk-lib/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const api_canary_1 = require("../api-canary");
const ecs_deployment_1 = require("../ecs-deployment");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer and deployed by CodeDeploy.
 */
class ApplicationLoadBalancedCodeDeployedFargateService extends aws_ecs_patterns_1.ApplicationLoadBalancedFargateService {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedCodeDeployedFargateService class.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            deploymentController: {
                type: aws_ecs_1.DeploymentControllerType.CODE_DEPLOY,
            },
        });
        if (props.deregistrationDelay) {
            this.targetGroup.setAttribute('deregistration_delay.timeout_seconds', props.deregistrationDelay.toSeconds().toString());
        }
        if (props.healthCheck) {
            this.targetGroup.configureHealthCheck(props.healthCheck);
        }
        this.accessLogBucket = props.accessLogBucket ??
            new aws_s3_1.Bucket(this, 'AccessLogBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                enforceSSL: true,
            });
        this.loadBalancer.logAccessLogs(this.accessLogBucket, props.accessLogPrefix);
        const alarms = [];
        if (props.responseTimeAlarmThreshold) {
            const responseTimeAlarm = new aws_cloudwatch_1.Alarm(this, 'ResponseTimeAlarm', {
                metric: this.loadBalancer.metrics.targetResponseTime({
                    period: aws_cdk_lib_1.Duration.minutes(1),
                    statistic: 'p95',
                }),
                evaluationPeriods: 2,
                threshold: props.responseTimeAlarmThreshold.toSeconds(),
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            });
            alarms.push(responseTimeAlarm);
        }
        const protocol = props.protocol ?? (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        const testHostName = props.domainName ? props.domainName : this.loadBalancer.loadBalancerDnsName;
        if (props.apiTestSteps?.length) {
            this.apiCanary = new api_canary_1.ApiCanary(this, 'Canary', {
                baseUrl: `${protocol.toLowerCase()}://${testHostName}`,
                durationAlarmThreshold: props.apiCanaryTimeout,
                schedule: aws_synthetics_alpha_1.Schedule.rate(props.apiCanarySchedule ?? aws_cdk_lib_1.Duration.minutes(5)),
                threadCount: props.apiCanaryThreadCount,
                steps: props.apiTestSteps,
            });
            this.apiCanary.node.addDependency(this.service);
            alarms.push(this.apiCanary.successAlarm);
            if (this.apiCanary.durationAlarm) {
                alarms.push(this.apiCanary.durationAlarm);
            }
        }
        if (alarms.length > 0) {
            this.healthAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'HealthAlarm', {
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...alarms),
            });
        }
        else {
            this.healthAlarm = undefined;
        }
        let testPort;
        if (props.testPort) {
            testPort = props.testPort;
        }
        else if (props.listenerPort) {
            testPort = props.listenerPort + 1;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            testPort = 8080;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            testPort = 8443;
        }
        else {
            throw new Error('Unable to determine port for test listener');
        }
        let certificates;
        if (props.certificate) {
            certificates = [props.certificate];
        }
        this.testListener = this.loadBalancer.addListener('TestListener', {
            protocol,
            port: testPort,
            open: props.openListener ?? true,
            sslPolicy: props.sslPolicy,
            certificates: certificates,
        });
        this.greenTargetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(this, 'GreenTargetGroup', {
            vpc: this.cluster.vpc,
            port: testPort,
            protocol: props.targetProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            protocolVersion: props.protocolVersion,
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            targetType: aws_elasticloadbalancingv2_1.TargetType.IP,
        });
        this.listener.node.addDependency(this.greenTargetGroup);
        this.testListener.addTargetGroups('ECS', {
            targetGroups: [this.greenTargetGroup],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'Application');
        this.deploymentGroup = new aws_codedeploy_1.EcsDeploymentGroup(this, 'DeploymentGroup', {
            application: this.application,
            alarms: this.healthAlarm ? [this.healthAlarm] : undefined,
            service: this.service,
            blueGreenDeploymentConfig: {
                blueTargetGroup: this.targetGroup,
                greenTargetGroup: this.greenTargetGroup,
                listener: this.listener,
                testListener: this.testListener,
                terminationWaitTime: props.terminationWaitTime ?? aws_cdk_lib_1.Duration.minutes(10),
            },
            deploymentConfig: props.deploymentConfig ?? aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE,
            autoRollback: {
                stoppedDeployment: true,
            },
        });
        this.deployment = new ecs_deployment_1.EcsDeployment({
            deploymentGroup: this.deploymentGroup,
            timeout: props.deploymentTimeout ?? aws_cdk_lib_1.Duration.minutes(60),
            targetService: {
                taskDefinition: this.taskDefinition,
                containerName: this.taskDefinition.defaultContainer.containerName,
                containerPort: this.taskDefinition.defaultContainer.containerPort,
            },
        });
    }
    addServiceAsTarget(service) {
        super.addServiceAsTarget(service);
    }
}
exports.ApplicationLoadBalancedCodeDeployedFargateService = ApplicationLoadBalancedCodeDeployedFargateService;
_a = JSII_RTTI_SYMBOL_1;
ApplicationLoadBalancedCodeDeployedFargateService[_a] = { fqn: "@cdklabs/cdk-ecs-codedeploy.ApplicationLoadBalancedCodeDeployedFargateService", version: "0.0.121" };
//# sourceMappingURL=data:application/json;base64,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