from dataclasses import dataclass, field
from typing import Any, Dict, Optional

from .mcp_tool import McpTool


@dataclass
class McpServer:
    """Represents an MCP server configuration."""

    name: str
    enabled: Optional[bool] = True
    command: Optional[str] = None
    args: Optional[list[str]] = None
    permissions: Dict[str, str] = field(default_factory=dict)
    env: Dict[str, str] = field(default_factory=dict)
    tools: Dict[str, McpTool] = field(default_factory=dict)

    def add_tool(
        self,
        name: str,
        title: str,
        description: Optional[str] = None,
        required: Optional[list[str]] = None,
        input_schema: Optional[Dict[str, Any]] = None,
        output_schema: Optional[Dict[str, Any]] = None,
        annotations: Optional[list[str]] = None,
        meta: Optional[str] = None,
        permission: str = "manual",
    ) -> None:
        """Add a tool to this server."""
        self.tools[name] = McpTool(
            name,
            title,
            description,
            required,
            input_schema,
            output_schema,
            annotations,
            meta,
            permission,
        )

    def set_tool_permission(self, tool_name: str, permission: str) -> bool:
        """Set the permission for a tool. Returns True if successful, False if tool not found."""
        if tool_name in self.tools:
            self.tools[tool_name].permission = permission
            return True
        return False

    def get_env_dict(self) -> Dict[str, str]:
        """Get a dictionary of environment variables for this server."""
        return self.env

    def is_valid(self) -> bool:
        """Check if the server configuration is valid."""
        return self.command is not None
