"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceProvider = exports.CustomResourceProviderRuntime = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("../construct-compat");
/**
 * The lambda runtime to use for the resource provider.
 *
 * This also indicates
 * which language is used for the handler.
 *
 * @stability stable
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    CustomResourceProviderRuntime["NODEJS_12"] = "nodejs12.x";
    /**
     * Node.js 12.x
     */
    CustomResourceProviderRuntime["NODEJS_12_X"] = "nodejs12.x";
    CustomResourceProviderRuntime["NODEJS_14_X"] = "nodejs14.x";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * An AWS-Lambda backed custom resource provider, for CDK Construct Library constructs.
 *
 * This is a provider for `CustomResource` constructs, backed by an AWS Lambda
 * Function. It only supports NodeJS runtimes.
 *
 * **This is not a generic custom resource provider class**. It is specifically
 * intended to be used only by constructs in the AWS CDK Construct Library, and
 * only exists here because of reverse dependency issues (for example, it cannot
 * use `iam.PolicyStatement` objects, since the `iam` library already depends on
 * the CDK `core` library and we cannot have cyclic dependencies).
 *
 * If you are not writing constructs for the AWS Construct Library, you should
 * use the `Provider` class in the `custom-resources` module instead, which has
 * a better API and supports all Lambda runtimes, not just Node.
 *
 * N.B.: When you are writing Custom Resource Providers, there are a number of
 * lifecycle events you have to pay attention to. These are documented in the
 * README of the `custom-resources` module. Be sure to give the documentation
 * in that module a read, regardless of whether you end up using the Provider
 * class in there or this one.
 *
 * @stability stable
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const assetFileName = staging.relativeStagedPath(stack);
        const asset = stack.synthesizer.addFileAsset({
            fileName: assetFileName,
            sourceHash: staging.assetHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        this.roleArn = token_1.Token.asString(role.getAtt('Arn'));
        const timeout = (_b = props.timeout) !== null && _b !== void 0 ? _b : duration_1.Duration.minutes(15);
        const memory = (_c = props.memorySize) !== null && _c !== void 0 ? _c : size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: props.runtime,
                Environment: this.renderEnvironmentVariables(props.environment),
                Description: (_d = props.description) !== null && _d !== void 0 ? _d : undefined,
            },
        });
        handler.addDependsOn(role);
        if (this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PATH_KEY, assetFileName);
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY, 'Code');
        }
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource provider.
     *
     * @param scope Construct scope.
     * @param uniqueid A globally unique id that will be used for the stack-level construct.
     * @param props Provider properties which will only be applied when the provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     * @stability stable
     */
    static getOrCreate(scope, uniqueid, props) {
        jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        return this.getOrCreateProvider(scope, uniqueid, props).serviceToken;
    }
    /**
     * Returns a stack-level singleton for the custom resource provider.
     *
     * @param scope Construct scope.
     * @param uniqueid A globally unique id that will be used for the stack-level construct.
     * @param props Provider properties which will only be applied when the provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     * @stability stable
     */
    static getOrCreateProvider(scope, uniqueid, props) {
        var _b;
        jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = (_b = stack.node.tryFindChild(id)) !== null && _b !== void 0 ? _b : new CustomResourceProvider(stack, id, props);
        return provider;
    }
    renderEnvironmentVariables(env) {
        if (!env || Object.keys(env).length === 0) {
            return undefined;
        }
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it)
        const variables = {};
        const keys = Object.keys(env).sort();
        for (const key of keys) {
            variables[key] = env[key];
        }
        return { Variables: variables };
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
_a = JSII_RTTI_SYMBOL_1;
CustomResourceProvider[_a] = { fqn: "@aws-cdk/core.CustomResourceProvider", version: "1.138.2" };
//# sourceMappingURL=data:application/json;base64,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