# pylint: disable=line-too-long, unused-wildcard-import
"""
The |sw1d.DOCNAME.long| model family member |sw1d_lias_sluice| extends the routing
submodel |sw1d_lias| with simple sluice functionalities.

The main model |sw1d_channel| documentation explains |sw1d_lias| depth.  Here, we limit
the discussion to some of the additional features of |sw1d_lias_sluice|.  Please also
read the documentation on method |Update_Discharge_V2|, which shows how to define
seasonal control schemes and additional and "closed gate" functionality, which helps
to prevent source catchment from running dry during low flow periods.

Integration tests
=================

.. how_to_understand_integration_tests::

We create a test setting identical to the one of the |sw1d_channel| documentation,
except replacing the |sw1d_lias| submodel at the central location with a
|sw1d_lias_sluice| submodel:

>>> from hydpy import pub
>>> pub.timegrids = "2000-01-01 00:00", "2000-01-01 05:00", "5m"

>>> from hydpy.models.sw1d_channel import *
>>> parameterstep()
>>> nmbsegments(8)
>>> lengths = 2.0, 3.0, 2.0, 3.0, 2.0, 3.0, 2.0, 3.0

>>> for i, length_ in enumerate(lengths):
...     with model.add_storagemodel_v1("sw1d_storage", position=i):
...         length(length_)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(5.0)
...             bottomwidths(5.0)
...             sideslopes(0.0)

>>> for i in range(1, nmbsegments.value):
...     lias = "sw1d_lias_sluice" if i == 4 else "sw1d_lias"
...     with model.add_routingmodel_v2(lias, position=i):
...         lengthupstream(2.0 if i % 2 else 3.0)
...         lengthdownstream(3.0 if i % 2 else 2.0)
...         stricklercoefficient(1.0/0.03)
...         timestepfactor(0.7)
...         diffusionfactor(0.2)
...         if i == 4:
...             bottomlowwaterthreshold(0.0)
...             upperlowwaterthreshold(0.0)
...             bottomhighwaterthreshold(5.0)
...             upperhighwaterthreshold(5.0)
...         with model.add_crosssection_v2("wq_trapeze"):
...             nmbtrapezes(1)
...             bottomlevels(5.0)
...             bottomwidths(5.0)
...             sideslopes(0.0)

>>> from hydpy import Element
>>> channel = Element("channel")
>>> channel.model = model

>>> from hydpy.core.testtools import IntegrationTest
>>> test = IntegrationTest(channel)
>>> test.plotting_options.axis1 = (factors.waterlevels,)

>>> def prepare_inits(hs):
...     if isinstance(hs, float):
...         hs = nmbsegments.value * [hs]
...     inits = []
...     for h, s in zip(hs, model.storagemodels):
...         length = s.parameters.control.length
...         c = s.crosssection.parameters.control
...         v = h * (c.bottomwidths[0] + h * c.sideslopes[0]) * length
...         inits.append((s.sequences.states.watervolume, v))
...     for r in model.routingmodels[1:-1]:
...         inits.append((r.sequences.states.discharge, 0.0))
...     test.inits = inits

LIAS comparison
_______________

We start with a close comparison with |sw1d_lias| by repeating the
:ref:`sw1d_channel_zero_inflow_and_outflow` example.  So, we neglect any inflows and
outflows and set the same uneven water depths:

>>> prepare_inits(hs=[3.0, 3.0, 3.0, 3.0, 1.0, 1.0, 1.0, 1.0])

Due to the identical setting of all common parameters, the following results are
identical to those of the :ref:`sw1d_channel_zero_inflow_and_outflow` example for the
first four hours of the simulation period.  The water levelling takes place by an
unhindered downstream water movement.  However, due to setting the additional
parameters |BottomHighWaterThreshold| and |UpperHighWaterThreshold| to 5 m (the channel
bottom), the sluice functionality is potentially active all the time, which is
noticeable in the fifth hour.  The excess water in the last four channel segments
cannot flow back, so two different water levels (one above and one below the sluice)
start to set in:

.. integration-test::

    >>> conditions = test("sw1d_lias_sluice_lias_comparison", get_conditions="2000-01-01 00:00")
    |                date |  timestep |                                                                       waterlevels |                                                                                   discharges |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 41.932744 |      8.0       8.0   7.99689  7.659739   6.51142  6.001387       6.0          6.0 | 0.0        0.0        0.0   0.103665  17.116697   0.069362        0.0        0.0         0.0 |
    | 2000-01-01 00:05:00 | 41.932744 | 7.999992  7.996361  7.824697  7.649492  6.587536  6.077925  6.002109          6.0 | 0.0    0.00028   0.182249   5.922032   6.434402   3.897203   0.070336   0.000021         0.0 |
    | 2000-01-01 00:10:00 |  41.86989 | 7.993352  7.931422   7.78946  7.595997  6.715088   6.12184  6.026127      6.00139 | 0.0   0.221329   3.468262   4.642813   7.317577   3.065842   0.870068   0.069492         0.0 |
    | 2000-01-01 00:15:00 | 40.570615 | 7.942995  7.887898  7.761077  7.556577  6.807278   6.17491   6.04788     6.007794 | 0.0   1.678558   3.854736   4.800826   6.771817   3.698807    1.04529   0.320178         0.0 |
    | 2000-01-01 00:20:00 | 38.464288 | 7.876758   7.85486  7.731448  7.524701  6.879592    6.2307  6.069909     6.017934 | 0.0   2.207898   3.859806   4.847457   6.441228   4.030779   1.241302   0.506993         0.0 |
    | 2000-01-01 00:25:00 | 36.060713 | 7.818283   7.81743  7.702268  7.498342  6.934081  6.288067  6.093754      6.03057 | 0.0   1.949184   3.820718   4.793387   6.111343   4.295028    1.42665   0.631814         0.0 |
    | 2000-01-01 00:30:00 | 33.726739 | 7.771332   7.77554  7.671519  7.475733  6.974537  6.345257  6.120048      6.04518 | 0.0   1.565018   3.659513   4.684456   5.814913   4.466393   1.606941   0.730501         0.0 |
    | 2000-01-01 00:35:00 | 31.578426 |  7.73188  7.733155  7.638895  7.455195    7.0041  6.400921  6.148744     6.061651 | 0.0   1.315079   3.434327   4.521823   5.548742   4.563291   1.780091   0.823557         0.0 |
    | 2000-01-01 00:40:00 | 29.590331 | 7.695544  7.693043  7.605781  7.435459  7.025353  6.454008  6.179562     6.079997 | 0.0   1.211187   3.216789   4.320583   5.307354   4.598933   1.944567   0.917288         0.0 |
    | 2000-01-01 00:45:00 | 27.697934 | 7.660109  7.655911  7.573675  7.415932  7.040288  6.503779  6.212122     6.100249 | 0.0   1.181168   3.037773   4.107951   5.084325   4.586478   2.097941   1.012595         0.0 |
    | 2000-01-01 00:50:00 | 25.854608 | 7.625278  7.621259  7.543316  7.396526  7.050449  6.549766  6.245998     6.122422 | 0.0   1.161016   2.893595   3.905593   4.875904   4.537222    2.23787   1.108668         0.0 |
    | 2000-01-01 00:55:00 | 24.044617 | 7.591516  7.588403  7.514686  7.377366  7.057039  6.591734  6.280746     6.146505 | 0.0   1.125422   2.768196   3.722499   4.680467   4.460811     2.3624   1.204147         0.0 |
    | 2000-01-01 01:00:00 | 22.271186 | 7.559223  7.556936  7.487453  7.358578  7.060989  6.629639  6.315927     6.172454 | 0.0   1.076444   2.649834     3.5576   4.497034   4.365364    2.47012   1.297422         0.0 |
    | 2000-01-01 01:05:00 | 20.541621 | 7.528507  7.526715  7.461316  7.340213  7.062994  6.663585  6.351131     6.200189 | 0.0   1.023846   2.534864   3.406128   4.324343   4.257504   2.560214   1.386749         0.0 |
    | 2000-01-01 01:10:00 | 18.859766 | 7.499272  7.497721  7.436118  7.322279  7.063549  6.693783  6.385994     6.229596 | 0.0   0.974495   2.424218   3.264124   4.160846   4.142345   2.632447    1.47035         0.0 |
    | 2000-01-01 01:15:00 | 17.225223 | 7.471356   7.46994  7.411816  7.304769  7.062995  6.720512  6.420212     6.260525 | 0.0   0.930529   2.319569   3.129639    4.00511   4.023586   2.687091   1.546498         0.0 |
    | 2000-01-01 01:20:00 | 15.635338 | 7.444619  7.443332  7.388403  7.287689  7.061565  6.744091  6.453548     6.292798 | 0.0   0.891234   2.221637   3.002079   3.856098   3.903739    2.72482   1.613616         0.0 |
    | 2000-01-01 01:25:00 | 14.087165 | 7.418965  7.417833  7.365868  7.271051  7.059429  6.764847  6.485835     6.326205 | 0.0   0.855147   2.130122   2.881273   3.713189   3.784411   2.746582    1.67035         0.0 |
    | 2000-01-01 01:30:00 | 12.578396 |  7.39433  7.393371  7.344187  7.254869  7.056713  6.783109   6.51697     6.360518 | 0.0    0.82117   2.044235    2.76695   3.576035   3.666568   2.753481   1.715636         0.0 |
    | 2000-01-01 01:35:00 | 11.107466 | 7.370666  7.369884  7.323322  7.239154  7.053523   6.79919  6.546909     6.395492 | 0.0   0.788783   1.963163   2.658648   3.444403   3.550737   2.746682   1.748731         0.0 |
    | 2000-01-01 01:40:00 |   9.67329 | 7.347931  7.347315  7.303237  7.223909   7.04995  6.813386  6.575652     6.430877 | 0.0   0.757842   1.886273   2.555805   3.318058   3.437146   2.727335   1.769232         0.0 |
    | 2000-01-01 01:45:00 |  8.274999 | 7.326081  7.325619  7.283894  7.209132  7.046077  6.825972  6.603235     6.466418 | 0.0   0.728329   1.813123   2.457882   3.196719   3.325814   2.696526   1.777071         0.0 |
    | 2000-01-01 01:50:00 |   6.91179 | 7.305075  7.304755  7.265263  7.194819   7.04198    6.8372  6.629718     6.501868 | 0.0   0.700211   1.743393   2.364414   3.080056   3.216616   2.655249   1.772492         0.0 |
    | 2000-01-01 01:55:00 |  5.582889 | 7.284873  7.284687  7.247317  7.180966  7.037731  6.847298  6.655169     6.536988 | 0.0   0.673412    1.67682   2.275012   2.967695   3.109324   2.604399   1.756018         0.0 |
    | 2000-01-01 02:00:00 |  4.287568 | 7.265438   7.26538  7.230032  7.167568  7.033399  6.856475  6.679661     6.571556 | 0.0    0.64782    1.61316   2.189331   2.859236   3.003643   2.544779   1.728405         0.0 |
    | 2000-01-01 02:05:00 |  3.025166 | 7.246738  7.246803  7.213386  7.154623   7.02905  6.864918  6.703256     6.605368 | 0.0   0.623323   1.552172   2.107049   2.754256   2.899238   2.477123   1.690598         0.0 |
    | 2000-01-01 02:10:00 |  1.795106 | 7.228744  7.228927  7.197359  7.142134  7.024747   6.87279   6.72601     6.638242 | 0.0   0.599814   1.493619   2.027846   2.652318   2.795754   2.402115   1.643674         0.0 |
    | 2000-01-01 02:15:00 |  0.596896 | 7.211428  7.211726  7.181935  7.130102  7.020551  6.880239  6.747958     6.670018 | 0.0   0.577198   1.437266   1.951396   2.552973   2.692831   2.320423   1.588801         0.0 |
    | 2000-01-01 02:20:00 |     300.0 | 7.194761  7.195181  7.167095  7.118539  7.016521  6.887386  6.769131     6.700556 | 0.0   0.555555   1.382809   1.877496   2.455693   2.590029   2.232663   1.526904         0.0 |
    | 2000-01-01 02:25:00 |     300.0 | 7.179143  7.178767  7.153074  7.107295  7.012697  6.894305  6.789218     6.730212 | 0.0   0.520605    1.34131   1.808668   2.370851   2.498317   2.152358   1.482801         0.0 |
    | 2000-01-01 02:30:00 |     300.0 | 7.164041  7.163107  7.139327  7.096616  7.009094  6.901059  6.808633     6.758487 | 0.0   0.503419    1.28639   1.744609   2.278535   2.398625   2.060935   1.413763         0.0 |
    | 2000-01-01 02:35:00 |     300.0 |  7.14924  7.148327  7.125966  7.086446  7.005787  6.907742  6.827426     6.785206 | 0.0   0.493359   1.232396   1.677772   2.186283   2.296518   1.962357   1.335919         0.0 |
    | 2000-01-01 02:40:00 |     300.0 | 7.134803   7.13427  7.113223  7.076738  7.002818  6.914435  6.845526      6.81031 | 0.0   0.481213   1.184035   1.608792   2.094209   2.193177   1.858547   1.255216         0.0 |
    | 2000-01-01 02:45:00 |     300.0 | 7.120879  7.120776  7.101168   7.06752  7.000211  6.921182  6.862846     6.833787 | 0.0   0.464144    1.13886   1.540716   2.001626   2.088528   1.751174   1.173856         0.0 |
    | 2000-01-01 02:50:00 |     300.0 | 7.107569  7.107779  7.089733  7.058843  6.997996  6.927994  6.879314     6.855642 | 0.0   0.443671   1.093526   1.474669   1.908479   1.982306   1.641693   1.092748         0.0 |
    | 2000-01-01 02:55:00 |     300.0 |   7.0949  7.095291  7.078844   7.05074  6.996207  6.934858   6.89488      6.87589 | 0.0   0.422306   1.046697   1.409677   1.814849   1.874506   1.531302   1.012416         0.0 |
    | 2000-01-01 03:00:00 |     300.0 |  7.08285  7.083351  7.068477  7.043214  6.994861  6.941745  6.909512     6.894556 | 0.0   0.401668   0.998683   1.344234   1.720515   1.765371   1.421011   0.933302         0.0 |
    | 2000-01-01 03:05:00 |     300.0 | 7.071386   7.07199  7.058659  7.036262  6.993955  6.948615  6.923187     6.911674 | 0.0    0.38214   0.950157   1.277423   1.625017   1.655202   1.311701   0.855859         0.0 |
    | 2000-01-01 03:10:00 |     300.0 | 7.060487  7.061228   7.04943  7.029884  6.993466  6.955417  6.935895     6.927284 | 0.0   0.363293   0.901399   1.209035   1.527933   1.544235   1.204145    0.78054         0.0 |
    | 2000-01-01 03:15:00 |     300.0 | 7.050154  7.051073  7.040821  7.024087  6.993358   6.96209  6.947632      6.94144 | 0.0   0.344439   0.852209   1.139185   1.429057   1.432655   1.098997   0.707776         0.0 |
    | 2000-01-01 03:20:00 |     300.0 | 7.040403   7.04153  7.032847  7.018878  6.993591  6.968568  6.958397     6.954198 | 0.0   0.325005   0.802146   1.067939   1.328408   1.320664   0.996774   0.637926         0.0 |
    | 2000-01-01 03:25:00 |     300.0 | 7.031263  7.032609  7.025514  7.014258  6.994119  6.974782  6.968196     6.965623 | 0.0   0.304682   0.750747   0.995179    1.22614   1.208538   0.897862   0.571242         0.0 |
    | 2000-01-01 03:30:00 |     300.0 | 7.022761  7.024324  7.018824  7.010222  6.994894  6.980664  6.977036      6.97578 | 0.0   0.283391   0.697651   0.920653   1.122462   1.096635   0.802523   0.507842         0.0 |
    | 2000-01-01 03:35:00 |     300.0 | 7.014926  7.016695   7.01278  7.006752  6.995861  6.986153  6.984932     6.984735 | 0.0   0.261178   0.642619   0.844086   1.017589   0.985357   0.710905    0.44772         0.0 |
    | 2000-01-01 03:40:00 |     300.0 | 7.007783  7.009747  7.007387  7.003823   6.99696  6.991194    6.9919      6.99255 | 0.0   0.238099   0.585506   0.765278   0.911751   0.875101   0.623051   0.390767         0.0 |
    | 2000-01-01 03:45:00 |     300.0 | 7.001358  7.003505  7.002649  7.001402   6.99813   6.99574  6.997963     6.999286 | 0.0   0.214166   0.526232   0.684152   0.805203   0.766215   0.538901   0.336802         0.0 |
    | 2000-01-01 03:50:00 |     300.0 | 6.995678  6.997997   6.99857  6.999452  6.999308  6.999754  7.003144     7.004998 | 0.0   0.189335   0.464764    0.60075   0.698243   0.658976   0.458301   0.285605         0.0 |
    | 2000-01-01 03:55:00 |     300.0 | 6.990772  6.993245  6.995147  6.997932  7.000436  7.003205  7.007467     7.009737 | 0.0   0.163546    0.40111   0.515219   0.591202   0.553589   0.381022   0.236926         0.0 |
    | 2000-01-01 04:00:00 |     300.0 | 6.986669  6.989274  6.992373  7.006488   6.98693  7.006073  7.010955     7.013547 | 0.0    0.13676   0.335337   0.427783        0.0   0.450191   0.306782   0.190502         0.0 |
    | 2000-01-01 04:05:00 |     300.0 | 6.983399  6.986102  6.994673  7.005714  6.989306  7.004376  7.013632     7.016468 | 0.0   0.108995   0.267588   0.190935   0.229626   0.150432   0.235278   0.146056         0.0 |
    | 2000-01-01 04:10:00 |     300.0 | 6.980989  6.985004  6.996644  6.999306  7.001298  7.002123  7.013618     7.018534 | 0.0   0.080337   0.135218   0.069495   0.389911  -0.009812   0.102848   0.103317         0.0 |
    | 2000-01-01 04:15:00 |     300.0 | 6.980078   6.98587  6.993989  7.000818  7.003299  7.001616  7.011362     7.019211 | 0.0   0.030361  -0.012916   0.075605        0.0  -0.066696  -0.041367   0.033823         0.0 |
    | 2000-01-01 04:20:00 |     300.0 | 6.981329   6.98735  6.991062  7.000455  7.004412  7.004011    7.0085     7.017983 | 0.0  -0.041707  -0.115716  -0.018166        0.0  -0.037094   -0.15682  -0.061391         0.0 |
    | 2000-01-01 04:25:00 |     300.0 | 6.984753  6.987834  6.990987  6.997738   7.00562  7.006841  7.008286     7.014489 | 0.0  -0.114128  -0.138316  -0.135812        0.0  -0.040287  -0.181798  -0.174675         0.0 |
    | 2000-01-01 04:30:00 |     300.0 | 6.988878  6.988428  6.991838  6.993827  7.007547   7.00909   7.00966      7.01004 | 0.0  -0.137492  -0.167207  -0.195557        0.0  -0.064233  -0.176664  -0.222478         0.0 |
    | 2000-01-01 04:35:00 |     300.0 | 6.992408   6.99006  6.991499  6.990067  7.010128  7.010767  7.010277      7.00623 | 0.0   -0.11769  -0.199284  -0.187983        0.0  -0.086023  -0.169898  -0.190467         0.0 |
    | 2000-01-01 04:40:00 |     300.0 | 6.994936  6.992329  6.989981  6.987125  7.012875  7.011973  7.009214     7.003901 | 0.0  -0.084253  -0.197714  -0.147126        0.0  -0.091576  -0.151884  -0.116449         0.0 |
    | 2000-01-01 04:45:00 |     300.0 | 6.996601   6.99416  6.988588  6.985113  7.015161   7.01257  7.007367     7.003013 | 0.0  -0.055498  -0.147048  -0.100597        0.0  -0.076179  -0.105989  -0.044399         0.0 |
    | 2000-01-01 04:50:00 |     300.0 | 6.997502  6.994873  6.988215  6.984047  7.016365  7.012457  7.006176     7.003118 | 0.0  -0.030048  -0.065707  -0.053293        0.0  -0.040131  -0.034475   0.005219         0.0 |
    | 2000-01-01 04:55:00 |     300.0 | 6.997494  6.994542  6.988735  6.984038  7.016044  7.011834   7.00612     7.003991 | 0.0   0.000283   0.016866  -0.000463        0.0   0.010696   0.041814   0.043659         0.0 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_lias_sluice_sharp_targets:

Sharp targets
_____________

Next, we set the lower and the upper high water threshold to 6.5 m (corresponding to a
water depth of 1.5 m) and reverse the initial water depths:

>>> model.routingmodels[4].parameters.control.upperhighwaterthreshold(6.5)
>>> model.routingmodels[4].parameters.control.bottomhighwaterthreshold(6.5)
>>> prepare_inits(hs=[1.0, 1.0, 1.0, 1.0, 3.0, 3.0, 3.0, 3.0])

The water should initially stream from the last to the first four segments until the
upstream segments' water levels reach 6.5 m.  However, the finally reached water levels
lie between 6.537 m and 6.544 m, and the third segment's trajectory shows a strong
sawtooth pattern caused by numerical problems:

.. integration-test::

    >>> conditions = test("sw1d_lias_sluice_sharp_targets", get_conditions="2000-01-01 00:00")
    |                date |  timestep |                                                                       waterlevels |                                                                                    discharges |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 41.932744 |      6.0       6.0  6.001525  6.414195  7.382449  7.996489       8.0          8.0 | 0.0        0.0        0.0  -0.050835  -20.760562  -0.175539        0.0        0.0         0.0 |
    | 2000-01-01 00:05:00 | 41.932744 |      6.0  6.001261  6.095305  6.468775  7.465454  7.827362  7.993156     7.999991 | 0.0  -0.000015  -0.063089  -3.189091   -5.918125  -8.684938  -0.228593  -0.000474         0.0 |
    | 2000-01-01 00:10:00 | 41.755789 | 6.001783  6.017947  6.138545  6.551658  7.448998  7.782868  7.891368     7.993732 | 0.0  -0.059402  -0.893659  -2.334978   -6.479105  -5.930589  -3.705853  -0.312916         0.0 |
    | 2000-01-01 00:15:00 | 38.971274 | 6.009936  6.034382  6.173784  6.506293  7.607183  7.751033  7.851022     7.947008 | 0.0  -0.271774  -1.093564  -2.268215         0.0   -5.27284  -3.681112  -2.336229         0.0 |
    | 2000-01-01 00:20:00 | 37.040731 | 6.022396  6.049518  6.200974  6.469642  7.712627  7.747493  7.830762     7.888841 | 0.0  -0.415328  -1.172128  -2.078459   -0.245882  -3.760657  -3.583645  -2.908319         0.0 |
    | 2000-01-01 00:25:00 |   35.4139 | 6.036805  6.064739   6.22756  6.643149  7.472425  7.756931  7.809819     7.837441 | 0.0  -0.480307  -1.241338  -2.127542  -10.802919  -2.796206  -3.268132  -2.570017         0.0 |
    | 2000-01-01 00:30:00 | 33.864882 | 6.052197  6.082329   6.26113  6.592918  7.614202  7.719793  7.784908     7.796669 | 0.0  -0.513077  -1.392561  -2.511558         0.0  -4.725905  -2.868966  -2.038601         0.0 |
    | 2000-01-01 00:35:00 | 32.378118 | 6.068796  6.100154  6.286441  6.547153  7.711757  7.712221  7.747315     7.764266 | 0.0  -0.553295  -1.444583  -2.288284         0.0   -3.25183  -2.873229  -1.620143         0.0 |
    | 2000-01-01 00:40:00 | 30.658027 | 6.086158  6.118453  6.302069  6.506861   7.76735  7.721905     7.722     7.734396 | 0.0  -0.578723  -1.493661  -2.014593         0.0  -1.853092  -2.337325  -1.493486         0.0 |
    | 2000-01-01 00:45:00 |  30.41696 | 6.104112  6.136405  6.310309  6.475037  7.783246  7.739483  7.713212     7.708489 | 0.0  -0.598468  -1.496078  -1.770742    -0.17956  -0.709428  -1.588303  -1.295387         0.0 |
    | 2000-01-01 00:50:00 | 31.225664 | 6.122382  6.153537  6.321195  6.676713  7.437235  7.747681  7.715788     7.691002 | 0.0   -0.60901  -1.465603  -1.828478  -11.912284  -0.378565  -0.788492  -0.874339         0.0 |
    | 2000-01-01 00:55:00 | 29.399898 | 6.140716  6.172271  6.349988  6.626561  7.573413  7.664556  7.712534      7.68551 | 0.0  -0.611143  -1.547846  -2.507614         0.0  -4.539296  -0.383036   -0.27458         0.0 |
    | 2000-01-01 01:00:00 | 28.060061 | 6.159725  6.191262  6.369317  6.582012  7.658322  7.639391  7.661114      7.68835 | 0.0  -0.633609   -1.58315  -2.227431         0.0  -2.830269  -1.571996   0.142004         0.0 |
    | 2000-01-01 01:05:00 | 26.552092 | 6.179173  6.210274  6.379515  6.543236  7.698412  7.642508    7.6184     7.686982 | 0.0  -0.648266  -1.598879  -1.938809         0.0  -1.336344  -1.492204  -0.068409         0.0 |
    | 2000-01-01 01:10:00 | 26.758662 |  6.19888  6.228486  6.383059  6.509524  7.704603  7.652664  7.634945     7.661669 | 0.0  -0.656919  -1.567474  -1.685623         0.0  -0.206357  -0.714174  -1.265656         0.0 |
    | 2000-01-01 01:15:00 | 27.154837 | 6.218513  6.245443   6.38201  6.482748  7.684858  7.671284  7.657431     7.638651 | 0.0   -0.65443  -1.502306  -1.467321   -0.128516   0.529634  -0.401336  -1.150881         0.0 |
    | 2000-01-01 01:20:00 | 26.930957 |  6.23769  6.260952  6.384804  6.678242  7.322742  7.693075   7.66231     7.629368 | 0.0  -0.639218  -1.414663  -1.507811   -11.28251    0.78804  -0.301541  -0.464159         0.0 |
    | 2000-01-01 01:25:00 | 26.685404 | 6.256106  6.277594  6.412473  6.630876  7.416139  7.630042  7.659243     7.632181 | 0.0  -0.613897  -1.445984  -2.368291         0.0  -3.113238   0.038394   0.140628         0.0 |
    | 2000-01-01 01:30:00 | 25.336926 | 6.274494  6.295284  6.428769  6.590064  7.523752  7.555635   7.64767     7.642561 | 0.0  -0.612917  -1.497415  -2.040591         0.0  -3.587114   0.133254    0.51902         0.0 |
    | 2000-01-01 01:35:00 | 25.192054 | 6.293006  6.312818  6.436246  6.555204  7.581727  7.529752  7.615587     7.651182 | 0.0  -0.617066  -1.493761  -1.743003         0.0  -1.932491  -0.638358   0.431059         0.0 |
    | 2000-01-01 01:40:00 | 24.367713 | 6.311461  6.329317  6.437692  6.525437  7.596528  7.556332  7.565203     7.648324 | 0.0  -0.615155  -1.440141  -1.488353         0.0  -0.493382  -1.822374  -0.142894         0.0 |
    | 2000-01-01 01:45:00 | 22.708709 |  6.32947   6.34435    6.4353  6.499992   7.59695  7.579521  7.576188     7.617531 | 0.0  -0.600316  -1.351974  -1.272243         0.0  -0.014055  -1.173514  -1.539684         0.0 |
    | 2000-01-01 01:50:00 | 22.402658 | 6.346602  6.357729  6.433907  6.714745   7.23515  7.598914  7.595826     7.587622 | 0.0  -0.571066  -1.239991  -1.193539  -11.931179   0.128816  -0.840815   -1.49542         0.0 |
    | 2000-01-01 01:55:00 | 22.634502 | 6.362468  6.371916  6.468937  6.666627   7.33196  7.550016  7.597145     7.571101 | 0.0  -0.528859  -1.238216  -2.405889         0.0  -3.226982  -0.782099  -0.826053         0.0 |
    | 2000-01-01 02:00:00 | 22.081019 | 6.377981  6.388862  6.488476  6.626314  7.440133  7.510516  7.556814     7.565373 | 0.0  -0.517099  -1.364381  -2.015672         0.0  -3.605766  -1.630754   -0.28639         0.0 |
    | 2000-01-01 02:05:00 | 20.646965 | 6.394036  6.405832  6.497407  6.592686  7.512657  7.499911  7.517589     7.553779 | 0.0  -0.535169  -1.383693  -1.681403         0.0  -2.417467  -1.887222   -0.57973         0.0 |
    | 2000-01-01 02:10:00 | 19.097171 | 6.410525  6.421469  6.499413  6.564718  7.550705  7.505461  7.504669     7.531477 | 0.0  -0.549652  -1.331498   -1.39836         0.0   -1.26829  -1.545779  -1.115112         0.0 |
    | 2000-01-01 02:15:00 | 19.127266 | 6.426847  6.435251  6.497238  6.541505  7.561356  7.520881   7.50618     7.507949 | 0.0  -0.544052  -1.233136  -1.160645         0.0  -0.355035  -1.126017  -1.176381         0.0 |
    | 2000-01-01 02:20:00 | 19.821072 | 6.442278  6.447086   6.49292  6.522262  7.556196  7.537549  7.511287     7.491316 | 0.0  -0.514373  -1.106121  -0.962172         0.0   0.172011  -0.661412  -0.831639         0.0 |
    | 2000-01-01 02:25:00 | 20.017428 |  6.45618  6.457073  6.487886  6.506363  7.541876  7.551022  7.514943     7.484953 | 0.0  -0.463412  -0.962756  -0.794965         0.0   0.477336  -0.196309  -0.318173         0.0 |
    | 2000-01-01 02:30:00 | 19.841341 | 6.468084  6.465362  6.483068  6.494487  7.534275  7.549573  7.517367     7.488716 | 0.0  -0.396788  -0.811236  -0.650646    -0.05687   0.196506   0.268965   0.188155         0.0 |
    | 2000-01-01 02:35:00 | 19.558036 | 6.477701  6.472106  6.483673  6.687266  7.221931  7.537844  7.520761     7.500073 | 0.0  -0.320552  -0.657795  -0.677975  -10.316895    0.09455   0.680976   0.567846         0.0 |
    | 2000-01-01 02:40:00 | 18.848856 | 6.484962  6.480338   6.52439  6.647049  7.307064  7.464749  7.523644     7.514491 | 0.0  -0.242032  -0.653589  -2.010815         0.0  -2.837749   0.817012   0.720908         0.0 |
    | 2000-01-01 02:45:00 | 18.424574 | 6.491651  6.494237  6.544388  6.615359  7.394405  7.401763  7.510408     7.528074 | 0.0  -0.222975  -0.917942  -1.584519         0.0  -2.911388   0.237929   0.679131         0.0 |
    | 2000-01-01 02:50:00 | 18.284318 | 6.500688  6.508171  6.551377  6.590741  7.437789  7.388593  7.480092     7.532531 | 0.0   -0.30124  -0.997925  -1.230892         0.0  -1.446123  -0.787628   0.222889         0.0 |
    | 2000-01-01 02:55:00 | 17.361846 | 6.512224  6.519553  6.551339  6.571693  7.447546  7.418355  7.439671     7.523213 | 0.0  -0.384541  -0.953642  -0.952384         0.0  -0.325217  -1.813317  -0.465937         0.0 |
    | 2000-01-01 03:00:00 | 15.767469 | 6.524655  6.528229  6.547914  6.557013  7.450178  7.442632  7.449664     7.490519 | 0.0  -0.414363  -0.848177  -0.734026         0.0  -0.087746  -1.301579  -1.634667         0.0 |
    | 2000-01-01 03:05:00 | 15.359639 | 6.536197  6.534778  6.543414  6.545769   7.45524  7.459916  7.463743     7.460474 | 0.0  -0.384741  -0.712175   -0.56217         0.0  -0.168718  -1.032944  -1.502262         0.0 |
    | 2000-01-01 03:10:00 | 15.153359 | 6.545595  6.539741  6.539279  6.537298  7.465514  7.470349   7.46496      7.44238 | 0.0  -0.313251  -0.561393  -0.423553         0.0  -0.342489  -0.864122   -0.90469         0.0 |
    | 2000-01-01 03:15:00 | 15.442833 | 6.552205  6.543446  6.536282  6.531184  7.477913  7.474286  7.457469     7.435172 | 0.0   -0.22033  -0.405627  -0.305735         0.0  -0.413286  -0.610133  -0.360415         0.0 |
    | 2000-01-01 03:20:00 | 15.961177 | 6.555878  6.546038  6.534688  6.527207  7.486814  7.473546  7.450199     7.434824 | 0.0  -0.122442  -0.252002  -0.198848         0.0  -0.296706  -0.259717  -0.017411         0.0 |
    | 2000-01-01 03:25:00 | 16.250043 | 6.556817  6.547534  6.534412  6.525269  7.488033  7.470526  7.448609     7.438091 | 0.0  -0.031293  -0.106107  -0.096908         0.0   -0.04065    0.11035    0.16336         0.0 |
    | 2000-01-01 03:30:00 | 16.222807 |  6.55545  6.547906  6.535168  6.525303  7.481888  7.466444  7.452796     7.443478 | 0.0   0.045564   0.026944   0.001721         0.0   0.204832   0.408923    0.26937         0.0 |
    | 2000-01-01 03:35:00 | 15.918086 | 6.552415  6.547158  6.536508  6.527182  7.469597  7.464404   7.45732     7.450696 | 0.0   0.101167   0.138575   0.093927         0.0   0.409717   0.511715     0.3609         0.0 |
    | 2000-01-01 03:40:00 | 15.446975 | 6.548447  6.545631    6.5379  6.530427  7.457537  7.461999  7.461183     7.458566 | 0.0   0.132277   0.208645   0.162252         0.0   0.401995   0.522247   0.393478         0.0 |
    | 2000-01-01 03:45:00 |  15.09979 | 6.544238  6.543689   6.53914  6.534347  7.449465  7.458434   7.46383     7.465748 | 0.0   0.140283   0.237395   0.196045         0.0   0.269062   0.447312   0.359083         0.0 |
    | 2000-01-01 03:50:00 |   15.2783 | 6.540342   6.54164  6.540215  6.538277  7.445947  7.454834  7.464791     7.471052 | 0.0    0.12988   0.232321   0.196501         0.0   0.117283    0.29727   0.265237         0.0 |
    | 2000-01-01 03:55:00 | 15.434043 | 6.537129   6.53974  6.541116  6.541718  7.445998  7.452622  7.464291     7.473564 | 0.0   0.107077   0.202062   0.172005         0.0  -0.001719   0.108912   0.125578         0.0 |
    | 2000-01-01 04:00:00 | 15.474669 | 6.534811  6.538192  6.541797  6.544358  7.449362  7.451857  7.463099     7.472881 | 0.0    0.07729   0.154711   0.132014         0.0  -0.112127  -0.073878  -0.034126         0.0 |
    | 2000-01-01 04:05:00 |  15.41767 |  6.53347   6.53713  6.542198  6.546046  7.452964  7.454063  7.461431     7.469385 | 0.0   0.044707   0.097791   0.084421         0.0  -0.120046  -0.230373    -0.1748         0.0 |
    | 2000-01-01 04:10:00 | 15.279217 | 6.533096  6.536616  6.542287   6.54675  7.456885  7.457344  7.460355     7.464208 | 0.0   0.012436   0.038157   0.035196         0.0  -0.130715  -0.294754  -0.258884         0.0 |
    | 2000-01-01 04:15:00 | 15.122164 | 6.533608  6.536637  6.542075  6.546529  7.461042  7.460321  7.459586     7.458972 | 0.0  -0.017062  -0.018135  -0.011078         0.0   -0.13858  -0.287422  -0.261789         0.0 |
    | 2000-01-01 04:20:00 | 15.056042 | 6.534843  6.537114  6.541631  6.545524  7.464806  7.462471  7.458625     7.454954 | 0.0   -0.04116  -0.065012  -0.050209         0.0  -0.125442  -0.232945  -0.200893         0.0 |
    | 2000-01-01 04:25:00 | 15.236394 | 6.536563  6.537888  6.541066  6.543981  7.467366  7.463671  7.457658     7.452692 | 0.0  -0.057318  -0.096003  -0.077148         0.0  -0.085357  -0.145347  -0.113113         0.0 |
    | 2000-01-01 04:30:00 | 15.352843 | 6.538485  6.538802  6.540469  6.542183  7.468149  7.463953  7.457231     7.452173 | 0.0  -0.064088  -0.109822  -0.089918         0.0   -0.02609  -0.040198   -0.02596         0.0 |
    | 2000-01-01 04:35:00 | 15.386456 | 6.540335  6.539729  6.539908  6.540397  7.467043   7.46346   7.45761      7.45315 | 0.0   -0.06166  -0.108003  -0.089306         0.0   0.036866   0.061526   0.048863         0.0 |
    | 2000-01-01 04:40:00 | 15.336774 | 6.541887  6.540565   6.53944  6.538838  7.464476  7.462428  7.458569     7.455254 | 0.0  -0.051726  -0.093522  -0.077934         0.0   0.085569   0.137138   0.105202         0.0 |
    | 2000-01-01 04:45:00 | 15.223978 | 6.542991   6.54123  6.539112  6.537656  7.461271  7.461106  7.459662     7.457984 | 0.0   -0.03681  -0.070037  -0.059118         0.0   0.106828   0.172965    0.13653         0.0 |
    | 2000-01-01 04:50:00 | 15.082158 | 6.543579  6.541667  6.538953  6.536933   7.45829  7.459705  7.460585     7.460757 | 0.0  -0.019585  -0.041459  -0.036144         0.0   0.099365     0.1694   0.138624         0.0 |
    | 2000-01-01 04:55:00 | 15.050247 | 6.543652  6.541851  6.538964  6.536693  7.456138  7.458466  7.461198     7.463022 | 0.0  -0.002444  -0.011632  -0.011997         0.0   0.071746   0.133715   0.113265         0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_lias_sluice_higher_precision:

Higher precision
________________

One possible solution is increasing numerical precision (like in the
:ref:`sw1d_channel_higher_precision` example), eventually accompanied by reducing
numerical diffusion (like in the :ref:`sw1d_channel_no_additional_diffusion` example):

>>> for routingmodel in model.routingmodels.submodels[1:-1]:
...     routingmodel.parameters.control.timestepfactor(0.1)
...     routingmodel.parameters.control.diffusionfactor(0.0)

Despite the significant increase in computation time, some trajectories still appear a
little unsmooth, and the reached water levels range from 6.504 m to 6.515 m, which is
less of a numerical error than to due the physical effect of the water sloshing back
and forth until all levels are equal:

.. integration-test::

    >>> test("sw1d_lias_sluice_higher_precision")
    |                date |  timestep |                                                                       waterlevels |                                                                                   discharges |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  5.065455 |      6.0  6.000016  6.003548  6.163941  7.765552  7.990114  7.999795     7.999999 | 0.0  -0.000002  -0.000807  -0.119079  -8.316126  -0.501205   -0.00691  -0.000065         0.0 |
    | 2000-01-01 00:05:00 |  5.006625 |  6.00001  6.000673  6.029531  6.334195  7.585149   7.92721  7.992553      7.99976 | 0.0  -0.000337  -0.033186  -0.899273  -9.411965  -3.398527  -0.253339  -0.011935         0.0 |
    | 2000-01-01 00:10:00 |  4.471659 | 6.000183  6.004629  6.075716  6.465475  7.503515  7.841512  7.958749     7.996276 | 0.0  -0.005755  -0.203547  -1.743042  -8.307032  -5.585905     -1.301  -0.174201         0.0 |
    | 2000-01-01 00:15:00 |   2.94126 | 6.001212  6.014604  6.126577  6.501253  7.558221  7.778636  7.902287     7.979976 | 0.0  -0.034305  -0.533064  -2.228432   -4.01736   -5.84088  -2.697044  -0.814982         0.0 |
    | 2000-01-01 00:20:00 |  0.742352 | 6.004535  6.029822  6.169205  6.504694  7.633492  7.748562  7.854336     7.942544 | 0.0  -0.110781  -0.871658  -2.292608  -2.464631  -4.973661  -3.469973  -1.871615         0.0 |
    | 2000-01-01 00:25:00 | 36.372941 | 6.011609  6.047213  6.203222  6.500661  7.690191  7.741546  7.826391     7.889638 | 0.0  -0.235796  -1.105352  -2.239232  -2.037594  -3.927576  -3.576804  -2.645288         0.0 |
    | 2000-01-01 00:30:00 | 34.706462 | 6.022803  6.064728  6.230943  6.499066  7.719576  7.747489  7.807453     7.834867 | 0.0   -0.37313  -1.248894  -2.172948  -2.093196   -3.07269   -3.36982  -2.738574         0.0 |
    | 2000-01-01 00:35:00 | 33.187935 | 6.037309  6.081906  6.253939  6.499088  7.731427  7.757337  7.788059     7.787846 | 0.0   -0.48354  -1.342435  -2.108975  -2.110064  -2.505093  -2.997513  -2.351053         0.0 |
    | 2000-01-01 00:40:00 | 31.793426 | 6.053964  6.098943  6.273193  6.500346  7.733948  7.763284  7.767065     7.751981 | 0.0  -0.555157  -1.406988  -2.048772  -2.111716  -2.195734  -2.493055   -1.79323         0.0 |
    | 2000-01-01 00:45:00 | 30.986328 | 6.071855  6.116032  6.289216  6.495515  7.742021  7.761278  7.747162     7.727006 | 0.0  -0.596381  -1.450829  -1.984922  -1.743347  -2.012452  -1.912167  -1.248739         0.0 |
    | 2000-01-01 00:50:00 | 30.330341 | 6.090426  6.133203  6.303147   6.49861  7.735992  7.750176  7.730605     7.711231 | 0.0  -0.619006  -1.477579  -1.941956  -2.096725  -1.895779   -1.34067  -0.788775         0.0 |
    | 2000-01-01 00:55:00 | 29.812583 | 6.109361  6.150414  6.315322  6.501218  7.726836  7.733134  7.717415     7.702611 | 0.0  -0.631193  -1.491736  -1.897582  -2.027978  -1.722757  -0.870645  -0.430976         0.0 |
    | 2000-01-01 01:00:00 | 29.256068 | 6.128482  6.167572   6.32598  6.498529  7.719017  7.715097  7.706351     7.698915 | 0.0  -0.637367  -1.495272  -1.850522  -1.716064  -1.455428  -0.553605  -0.184805         0.0 |
    | 2000-01-01 01:05:00 | 28.437291 | 6.147668  6.184572  6.335574  6.494979  7.706035  7.698795  7.696795     7.697605 | 0.0  -0.639523  -1.489539  -1.809332  -1.631827  -1.199114  -0.384001  -0.065493         0.0 |
    | 2000-01-01 01:10:00 | 27.768876 | 6.166819  6.201336  6.344387  6.499215  7.677849  7.684623   7.68903     7.696103 | 0.0  -0.638361  -1.476535  -1.770323  -1.982111  -1.042583  -0.333962  -0.075118         0.0 |
    | 2000-01-01 01:15:00 | 27.462547 | 6.185841  6.217818  6.352665   6.49897  7.657337  7.671597  7.682821     7.692505 | 0.0  -0.634073  -1.458181  -1.734098   -1.72186  -1.038121  -0.386844  -0.179884         0.0 |
    | 2000-01-01 01:20:00 | 27.182196 | 6.204644  6.233991  6.360491  6.497001  7.643188  7.659726  7.676451     7.686098 | 0.0  -0.626776  -1.435444  -1.696327  -1.597868  -1.126224  -0.532684  -0.320354         0.0 |
    | 2000-01-01 01:25:00 |  26.81393 | 6.223144  6.249837  6.367918  6.497585  7.630023  7.649449   7.66798     7.677086 | 0.0   -0.61665  -1.408926  -1.656474  -1.685667  -1.246839  -0.732981  -0.450596         0.0 |
    | 2000-01-01 01:30:00 | 26.320745 | 6.241263  6.265342  6.375177  6.495883  7.624759   7.64062   7.65714      7.66593 | 0.0  -0.603955  -1.379191   -1.62117  -1.536092   -1.36063  -0.919167  -0.557843         0.0 |
    | 2000-01-01 01:35:00 | 25.725279 | 6.258934  6.280504  6.382376   6.49815  7.616705  7.632687  7.645364     7.653073 | 0.0  -0.589043  -1.347179  -1.587152  -1.700498  -1.432022  -1.035365  -0.642827         0.0 |
    | 2000-01-01 01:40:00 | 25.085942 | 6.276105   6.29533  6.389377  6.497668  7.613993  7.625453  7.634121     7.639151 | 0.0  -0.572368  -1.313662  -1.547031  -1.522919  -1.432513  -1.070848  -0.696077         0.0 |
    | 2000-01-01 01:45:00 | 24.452482 | 6.292738    6.3098  6.396139  6.496352   7.61184  7.618986  7.623891     7.625123 | 0.0  -0.554449  -1.277951  -1.503354  -1.437585  -1.365829  -1.042438  -0.701428         0.0 |
    | 2000-01-01 01:50:00 | 23.851271 | 6.308809  6.323884  6.402736  6.495783  7.607394  7.612729  7.614383     7.612056 | 0.0  -0.535696  -1.239875  -1.459748  -1.431287  -1.283094   -0.97025  -0.653345         0.0 |
    | 2000-01-01 01:55:00 | 23.346911 |   6.3243  6.337557  6.409478  6.500689  7.594224  7.605554  7.605155     7.600762 | 0.0  -0.516366   -1.20001   -1.42476  -1.670036  -1.231036  -0.872305   -0.56468         0.0 |
    | 2000-01-01 02:00:00 | 22.891033 |   6.3392   6.35082  6.416112  6.501336  7.586931  7.597508   7.59595     7.591541 | 0.0  -0.496659   -1.15983  -1.380952  -1.413338  -1.170233  -0.767923  -0.461084         0.0 |
    | 2000-01-01 02:05:00 | 22.419949 | 6.353505  6.363661  6.422706  6.502638  7.578048  7.588735  7.586985     7.584136 | 0.0  -0.476829  -1.118898  -1.338717  -1.403779  -1.107675  -0.669053   -0.37025         0.0 |
    | 2000-01-01 02:10:00 |  21.94552 | 6.367217  6.376067  6.429247   6.49698  7.579043  7.579535  7.578727     7.577927 | 0.0  -0.457082  -1.077383  -1.295412  -1.012531  -1.045705  -0.585708  -0.310429         0.0 |
    | 2000-01-01 02:15:00 | 21.500564 | 6.380344   6.38802  6.435556  6.501831  7.563393  7.570763  7.571433     7.572234 | 0.0  -0.437544  -1.035193  -1.245497  -1.488065  -0.966393  -0.527785  -0.284655         0.0 |
    | 2000-01-01 02:20:00 | 21.138879 | 6.392891  6.399496  6.441724  6.497745  7.560831  7.562585  7.564976      7.56656 | 0.0  -0.418235  -0.992005  -1.197608  -0.993264  -0.907865  -0.498965  -0.283716         0.0 |
    | 2000-01-01 02:25:00 | 20.809326 | 6.404862  6.410473  6.447671   6.50022  7.548627  7.555267  7.558891     7.560673 | 0.0   -0.39905  -0.947917  -1.146143  -1.269928  -0.863111  -0.497186  -0.294347         0.0 |
    | 2000-01-01 02:30:00 | 20.479241 | 6.416257  6.420936  6.453533  6.497711  7.545081  7.548423  7.552732     7.554527 | 0.0  -0.379824   -0.90298  -1.098378  -0.972931  -0.854745  -0.512571   -0.30728         0.0 |
    | 2000-01-01 02:35:00 | 20.136361 | 6.427069  6.430884  6.459131  6.501775  7.533062  7.542168   7.54629     7.548139 | 0.0  -0.360399  -0.857778  -1.044359  -1.247558  -0.846909  -0.534158  -0.319419         0.0 |
    | 2000-01-01 02:40:00 | 19.778619 | 6.437291  6.440313   6.46455  6.504558  7.524479  7.536262    7.5397     7.541521 | 0.0  -0.340722  -0.812201  -0.992839  -1.131979  -0.845891  -0.550557  -0.330902         0.0 |
    | 2000-01-01 02:45:00 | 19.413522 | 6.446916  6.449223  6.469765  6.499542   7.52868  7.530776  7.533259     7.534713 | 0.0   -0.32083  -0.766317  -0.940164  -0.689371  -0.829404  -0.555105    -0.3404         0.0 |
    | 2000-01-01 02:50:00 | 19.052328 |  6.45594  6.457605  6.474562  6.500292  7.524443  7.526148  7.527221     7.527844 | 0.0  -0.300813  -0.719904   -0.87981   -0.91732  -0.776077  -0.544702   -0.34344         0.0 |
    | 2000-01-01 02:55:00 | 18.721735 | 6.464362  6.465433  6.478929  6.498708  7.523488  7.522346  7.521756     7.521157 | 0.0  -0.280731   -0.67214  -0.817685  -0.738447  -0.706642   -0.51653  -0.334352         0.0 |
    | 2000-01-01 03:00:00 |  18.46289 | 6.472177  6.472674  6.483069  6.499192  7.519474  7.518733  7.516918     7.514975 | 0.0   -0.26049  -0.622531  -0.760552  -0.784784  -0.650982  -0.470335  -0.309092         0.0 |
    | 2000-01-01 03:05:00 | 18.249225 | 6.479372   6.47932  6.487061  6.499692  7.515858  7.514816  7.512534     7.509622 | 0.0  -0.239844   -0.57216  -0.705202   -0.73018  -0.609649  -0.413817  -0.267676         0.0 |
    | 2000-01-01 03:10:00 | 18.041736 | 6.485931  6.485394  6.490876  6.498627  7.515024  7.510595  7.508305     7.505292 | 0.0  -0.218644  -0.522345  -0.649524  -0.596299  -0.568492  -0.357423  -0.216458         0.0 |
    | 2000-01-01 03:15:00 | 17.878547 | 6.491843  6.490923  6.494284   6.49877  7.512282  7.506657  7.504148     7.501945 | 0.0   -0.19706  -0.473507  -0.587108  -0.594242  -0.502844  -0.305941  -0.167359         0.0 |
    | 2000-01-01 03:20:00 | 17.672013 | 6.497109  6.495905  6.497295  6.500077    7.5075  7.503164  7.500412      7.49931 | 0.0  -0.175528  -0.424623     -0.525  -0.590371  -0.430947  -0.256315  -0.131784         0.0 |
    | 2000-01-01 03:25:00 | 17.521545 | 6.501742  6.500316  6.499895  6.499883  7.504697  7.500147  7.497562     7.497057 | 0.0  -0.154424  -0.374952  -0.461609   -0.45187  -0.358452  -0.207606  -0.112609         0.0 |
    | 2000-01-01 03:30:00 | 17.305747 | 6.505756  6.504124  6.502249  6.500826  7.500399  7.497317  7.495681     7.495009 | 0.0  -0.133825  -0.324259  -0.402722  -0.449899  -0.306619  -0.165087  -0.102399         0.0 |
    | 2000-01-01 03:35:00 | 17.179202 | 6.509161  6.507324  6.504332  6.502943  7.494852  7.494792  7.494274     7.493195 | 0.0  -0.113491  -0.273479  -0.342915  -0.448773  -0.263885  -0.137651  -0.090741         0.0 |
    | 2000-01-01 03:40:00 | 17.022259 | 6.511954  6.509931  6.506064  6.501799  7.494833  7.492855  7.492712     7.491706 | 0.0  -0.093102  -0.223426  -0.281176  -0.223951  -0.223308  -0.126476  -0.074428         0.0 |
    | 2000-01-01 03:45:00 | 16.935089 | 6.514129  6.511961  6.507402  6.501899   7.49368  7.491567  7.490879     7.490512 | 0.0  -0.072504  -0.174043   -0.21863  -0.223647  -0.185229    -0.1208  -0.059688         0.0 |
    | 2000-01-01 03:50:00 | 16.919275 | 6.515682  6.513426   6.50828  6.503287  7.491285  7.490856  7.489243     7.489438 | 0.0  -0.051749  -0.124999  -0.154257  -0.223631  -0.143803   -0.10825  -0.053723         0.0 |
    | 2000-01-01 03:55:00 | 16.841058 | 6.516612  6.514318  6.508836  6.501405  7.494554  7.490359  7.488373     7.488336 | 0.0  -0.031003  -0.075563  -0.094109        0.0  -0.108939  -0.084125  -0.055099         0.0 |
    | 2000-01-01 04:00:00 |   16.9239 | 6.516923  6.514625    6.5089  6.500847   7.49603  7.490465  7.488331     7.487274 | 0.0  -0.010376  -0.025739  -0.027855        0.0  -0.049215  -0.054486  -0.053097         0.0 |
    | 2000-01-01 04:05:00 | 16.952807 | 6.516619  6.514324  6.508397  6.501686  7.495175  7.491482  7.488772     7.486532 | 0.0   0.010126   0.025178   0.041925        0.0   0.028502  -0.022363  -0.037081         0.0 |
    | 2000-01-01 04:10:00 | 16.925938 | 6.515706  6.513404  6.507741  6.503652  7.492778  7.492732  7.489436     7.486438 | 0.0   0.030435   0.076401   0.098292        0.0   0.079892   0.017421  -0.004703         0.0 |
    | 2000-01-01 04:15:00 | 16.861578 | 6.514198  6.511968  6.507527  6.506236   7.49036  7.493061  7.490213     7.487203 | 0.0   0.050265   0.122095    0.12922        0.0   0.080595   0.064132   0.038224         0.0 |
    | 2000-01-01 04:20:00 | 16.813911 | 6.512142  6.510272  6.508096  6.508923  7.489052  7.491875  7.490884     7.488814 | 0.0   0.068555    0.15333   0.134364        0.0   0.043611   0.102927   0.080568         0.0 |
    | 2000-01-01 04:25:00 | 16.793735 | 6.509664  6.508674   6.50933  6.511351  7.488907  7.489675  7.491083     7.490977 | 0.0   0.082592   0.162497   0.121358        0.0   0.004834   0.114792   0.108161         0.0 |
    | 2000-01-01 04:30:00 | 16.809744 |  6.50702  6.507488  6.510777  6.513334  7.489095  7.487682  7.490674     7.493118 | 0.0   0.088121    0.14744   0.099189        0.0  -0.006281   0.093379   0.107027         0.0 |
    | 2000-01-01 04:35:00 | 16.834465 | 6.504593  6.506862  6.511933  6.514808   7.48873  7.486917  7.490073     7.494527 | 0.0   0.080911   0.112223   0.073704        0.0    0.01217   0.050442   0.070453         0.0 |
    | 2000-01-01 04:40:00 | 16.840552 | 6.502824  6.506744  6.512479  6.515741  7.487651  7.487535  7.490015     7.494667 | 0.0    0.05895   0.064823   0.046613        0.0    0.03599   0.005068   0.007008         0.0 |
    | 2000-01-01 04:45:00 | 16.841455 | 6.502104  6.506948  6.512382  6.516082  7.486556  7.488893  7.490822     7.493501 | 0.0   0.024012   0.013812   0.017064        0.0   0.036503  -0.031409  -0.058304         0.0 |
    | 2000-01-01 04:50:00 | 16.840628 | 6.502646  6.507266  6.511836  6.515767   7.48645   7.49016  7.491887     7.491595 | 0.0   -0.01806  -0.033949  -0.015763        0.0   0.003516    -0.0598  -0.095296         0.0 |
    | 2000-01-01 04:55:00 | 16.827808 | 6.504369  6.507582  6.511136  6.514769  7.487892  7.490877  7.492092      7.48978 | 0.0  -0.057437  -0.073249  -0.049903        0.0  -0.048067  -0.083906  -0.090748         0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _sw1d_lias_sluice_fuzzy_targets:

Fuzzy targets
_____________

One might reach more satisfactory results by setting two different thresholds and,
thus, a more fuzzy target water level.  Here, we reduce the bottom threshold to 6.3 m:

>>> model.routingmodels[4].parameters.control.bottomhighwaterthreshold(6.3)
>>> model.routingmodels[4].parameters.control.upperhighwaterthreshold(6.5)

Also, we reset the initial time step and diffusion factors to reduce computation time:

>>> for routingmodel in model.routingmodels.submodels[1:-1]:
...     routingmodel.parameters.control.timestepfactor(0.7)
...     routingmodel.parameters.control.diffusionfactor(0.2)

Due to the introduced fuzziness, the third segment water level trajectory's
fluctuations disappear, and all of the first four segments' final water levels exceed
the upper threshold of 6.5 m by less than 3 mm:

.. integration-test::

    >>> test("sw1d_lias_sluice_fuzzy_targets")
    |                date |  timestep |                                                                       waterlevels |                                                                                    discharges |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 41.932744 |      6.0       6.0  6.001525  6.400901   7.40239  7.996489       8.0          8.0 | 0.0        0.0        0.0  -0.050835  -20.095867  -0.175539        0.0        0.0         0.0 |
    | 2000-01-01 00:05:00 | 41.932744 |      6.0  6.001097   6.08119  6.410704  7.548818  7.839196  7.993511     7.999991 | 0.0  -0.000015   -0.05487   -2.71038   -3.200533   -8.08146  -0.216778  -0.000474         0.0 |
    | 2000-01-01 00:10:00 | 41.764954 | 6.001532  6.015575  6.116757  6.454861  7.581637  7.803378  7.898199     7.994102 | 0.0  -0.051051  -0.774957  -1.960521   -4.168393  -5.262356  -3.471477  -0.294426         0.0 |
    | 2000-01-01 00:15:00 | 39.160078 | 6.008686  6.029981  6.149068  6.461312  7.648816  7.779526  7.862033     7.950113 | 0.0  -0.238461  -0.958767  -2.035782   -2.358315  -4.597618  -3.405024  -2.199462         0.0 |
    | 2000-01-01 00:20:00 |  37.39714 | 6.019867  6.043918   6.17692  6.466341  7.692252  7.771646  7.845098     7.895337 | 0.0  -0.372711  -1.069565  -1.997971   -2.249425  -3.697285  -3.303268  -2.738794         0.0 |
    | 2000-01-01 00:25:00 | 35.931538 | 6.033142  6.058189  6.201015  6.468668  7.720843  7.773397  7.823685     7.847289 | 0.0  -0.442492  -1.156041  -1.959204   -2.075551   -3.02859  -3.116173  -2.402411         0.0 |
    | 2000-01-01 00:30:00 | 34.441301 | 6.047515  6.073016  6.221851  6.470562  7.737297  7.776223  7.800374     7.808841 | 0.0  -0.479096  -1.220415  -1.914968   -2.009681  -2.558162   -2.69944  -1.922379         0.0 |
    | 2000-01-01 00:35:00 | 33.054214 | 6.062564  6.088332  6.239975  6.471885  7.745714  7.775285  7.779914     7.779053 | 0.0  -0.501631  -1.267429  -1.871564     -1.9377  -2.218261  -2.171357  -1.489379         0.0 |
    | 2000-01-01 00:40:00 | 31.895114 | 6.078107  6.103974  6.255829  6.472923  7.747231  7.769925    7.7634     7.756802 | 0.0  -0.518096  -1.300205  -1.828645   -1.880548  -1.931092  -1.663075  -1.112587         0.0 |
    | 2000-01-01 00:45:00 | 31.238726 | 6.094039  6.119777  6.269801   6.47362  7.742904  7.761092  7.749975     7.741031 | 0.0  -0.531095  -1.321265  -1.787026   -1.821863  -1.677651  -1.236043  -0.788543         0.0 |
    | 2000-01-01 00:50:00 | 30.673979 | 6.110274  6.135606  6.282222  6.474149  7.733618  7.749953   7.73864     7.730456 | 0.0   -0.54116   -1.33258  -1.746589   -1.773049  -1.463525  -0.906553  -0.528726         0.0 |
    | 2000-01-01 00:55:00 | 30.121951 | 6.126723  6.151355  6.293377  6.474528  7.720909  7.737412  7.728795     7.723502 | 0.0  -0.548293  -1.335767  -1.707615   -1.726554  -1.302902  -0.675868  -0.347702         0.0 |
    | 2000-01-01 01:00:00 | 29.583015 | 6.143296  6.166949  6.303508  6.474821  7.706357  7.724305  7.719983     7.718496 | 0.0  -0.552438  -1.332129   -1.66982   -1.684477  -1.199428  -0.544053  -0.250341         0.0 |
    | 2000-01-01 01:05:00 | 29.060422 | 6.159904  6.182332  6.312818  6.475066  7.691475  7.711418  7.711707     7.713913 | 0.0  -0.553604  -1.322777  -1.633129   -1.645411  -1.149327  -0.504997  -0.229125         0.0 |
    | 2000-01-01 01:10:00 | 28.605222 | 6.176461  6.197467  6.321479  6.475299  7.677483  7.699368  7.703495     7.708587 | 0.0  -0.551903  -1.308638   -1.59734    -1.60895  -1.142561  -0.540031  -0.266293         0.0 |
    | 2000-01-01 01:15:00 | 28.251473 | 6.192887  6.212327  6.329632  6.475544  7.665183  7.688458  7.695045      7.70184 | 0.0  -0.547525  -1.290504  -1.562269    -1.57452  -1.164497  -0.619002  -0.337342         0.0 |
    | 2000-01-01 01:20:00 | 27.860484 | 6.209108  6.226895  6.337392  6.475846   7.65483  7.678687  7.686251     7.693519 | 0.0    -0.5407  -1.269092  -1.527759   -1.542881  -1.197801  -0.709244  -0.416089         0.0 |
    | 2000-01-01 01:25:00 | 27.425398 |  6.22506  6.241159  6.344853  6.476206  7.646283  7.669849  7.677139     7.683895 | 0.0  -0.531732  -1.244974  -1.493657   -1.511664  -1.226769  -0.784892  -0.481161         0.0 |
    | 2000-01-01 01:30:00 | 26.952007 | 6.240689  6.255113  6.352084  6.476621  7.639083  7.661669  7.667821     7.673479 | 0.0  -0.520951  -1.218622   -1.45967   -1.480397    -1.2404  -0.831389  -0.520794         0.0 |
    | 2000-01-01 01:35:00 | 26.453574 | 6.255949  6.268747  6.359138  6.477075  7.632676  7.653874  7.658463     7.662819 | 0.0  -0.508689  -1.190417  -1.425546   -1.448275  -1.234692  -0.844975  -0.533037         0.0 |
    | 2000-01-01 01:40:00 | 25.946112 | 6.270807  6.282056  6.366049  6.477556   7.62656  7.646248  7.649239     7.652371 | 0.0  -0.495256  -1.160663  -1.391026   -1.415068  -1.211197  -0.829874  -0.522404         0.0 |
    | 2000-01-01 01:45:00 | 25.444385 | 6.285235  6.295029  6.372838  6.478052  7.620379  7.638646  7.640287     7.642447 | 0.0  -0.480934  -1.129602  -1.355907   -1.380689  -1.174669  -0.794564  -0.496189         0.0 |
    | 2000-01-01 01:50:00 |  24.95934 | 6.299214  6.307658  6.379517  6.478556  7.613946  7.631001  7.631691     7.633207 | 0.0  -0.465961  -1.097427  -1.320038   -1.345236   -1.13079  -0.748524  -0.461971         0.0 |
    | 2000-01-01 01:55:00 | 24.497179 |  6.31273  6.319933  6.386088  6.479066  7.607221  7.623309  7.623476     7.624682 | 0.0  -0.450529  -1.064294  -1.283329    -1.30884   -1.08467  -0.700099  -0.426285         0.0 |
    | 2000-01-01 02:00:00 | 24.059753 | 6.325773  6.331844   6.39255  6.479584  7.600274  7.615617  7.615632     7.616802 | 0.0  -0.434788  -1.030334  -1.245738   -1.271635  -1.040078  -0.655451  -0.393968         0.0 |
    | 2000-01-01 02:05:00 | 23.645619 | 6.338339  6.343381  6.398898  6.480113  7.593239  7.607997  7.608122     7.609444 | 0.0  -0.418847  -0.995663  -1.207265   -1.233722  -0.999238  -0.618237  -0.367915         0.0 |
    | 2000-01-01 02:10:00 |  23.25133 | 6.350422  6.354533  6.405124  6.480658  7.586273  7.600532  7.600904     7.602462 | 0.0  -0.402783  -0.960383  -1.167938   -1.195165  -0.962961  -0.589744  -0.349121         0.0 |
    | 2000-01-01 02:15:00 | 22.872656 | 6.362022  6.365292  6.411221  6.481221  7.579522    7.5933  7.593934     7.595722 | 0.0  -0.386649  -0.924587  -1.127803   -1.155994   -0.93094  -0.569308  -0.336986         0.0 |
    | 2000-01-01 02:20:00 | 22.505605 | 6.373136  6.375649  6.417178  6.481807  7.573098  7.586356  7.587181     7.589126 | 0.0  -0.370482  -0.888363  -1.086918   -1.116215  -0.902104  -0.554919  -0.329811         0.0 |
    | 2000-01-01 02:25:00 | 22.147146 | 6.383765  6.385599  6.422984  6.482417  7.567073  7.579734  7.580625     7.582618 | 0.0  -0.354307  -0.851789  -1.045346   -1.075821  -0.874986  -0.543912  -0.325376         0.0 |
    | 2000-01-01 02:30:00 |  21.79558 |  6.39391  6.395135  6.428631   6.48305  7.561471  7.573446  7.574261     7.576188 | 0.0  -0.338144  -0.814939  -1.003154   -1.034803  -0.848062  -0.533617  -0.321489         0.0 |
    | 2000-01-01 02:35:00 | 21.450597 |  6.40357  6.404252  6.434106  6.483705  7.556278  7.567482  7.568097     7.569861 | 0.0  -0.322008  -0.777884  -0.960408   -0.993164  -0.820046  -0.521849  -0.316386         0.0 |
    | 2000-01-01 02:40:00 | 21.113056 | 6.412748  6.412948  6.439401   6.48438  7.551452  7.561824   7.56215     7.563682 | 0.0  -0.305917  -0.740687  -0.917179   -0.950926  -0.790075  -0.507187  -0.308945         0.0 |
    | 2000-01-01 02:45:00 |  20.78461 | 6.421444  6.421218  6.444505  6.485072  7.546941  7.556449   7.55644     7.557707 | 0.0  -0.289883  -0.703407  -0.873538   -0.908135  -0.757778  -0.489041  -0.298721         0.0 |
    | 2000-01-01 02:50:00 | 20.467266 | 6.429662  6.429062  6.449409  6.485778  7.542693  7.551335  7.550989      7.55199 | 0.0  -0.273921  -0.666105  -0.829563   -0.864861  -0.723235  -0.467545  -0.285844         0.0 |
    | 2000-01-01 02:55:00 | 20.162997 | 6.437403  6.436477  6.454104  6.486495  7.538662  7.546465  7.545814     7.546573 | 0.0  -0.258048  -0.628836  -0.785338   -0.821194  -0.686859  -0.443352  -0.270849         0.0 |
    | 2000-01-01 03:00:00 |  19.87344 | 6.444671  6.443465  6.458583  6.487221  7.534823  7.541828  7.540927     7.541484 | 0.0  -0.242278  -0.591661  -0.740955    -0.77724  -0.649249  -0.417396  -0.254486         0.0 |
    | 2000-01-01 03:05:00 | 19.607329 |  6.45147  6.450025  6.462839  6.487952  7.531162  7.537421  7.536333     7.536733 | 0.0   -0.22663   -0.55464  -0.696514   -0.733077  -0.611048  -0.390674  -0.237544         0.0 |
    | 2000-01-01 03:10:00 | 19.357676 | 6.457804   6.45616  6.466867  6.488688   7.52768  7.533245  7.532033     7.532318 | 0.0  -0.211123  -0.517842  -0.652121   -0.688903  -0.572826  -0.364064  -0.220724         0.0 |
    | 2000-01-01 03:15:00 |  19.12228 | 6.463677  6.461871  6.470664  6.489426  7.524387  7.529309  7.528022     7.528228 | 0.0  -0.195778   -0.48134  -0.607896   -0.644794  -0.535048   -0.33822  -0.204538         0.0 |
    | 2000-01-01 03:20:00 | 18.901262 | 6.469096  6.467163  6.474226  6.490164  7.521301  7.525619  7.524295     7.524442 | 0.0  -0.180624  -0.445216  -0.563959   -0.600873     -0.498  -0.313527  -0.189279         0.0 |
    | 2000-01-01 03:25:00 | 18.694688 | 6.474067   6.47204  6.477553    6.4909  7.518438  7.522185  7.520842     7.520942 | 0.0  -0.165693  -0.409556  -0.520436   -0.557256  -0.461807  -0.290104  -0.175024         0.0 |
    | 2000-01-01 03:30:00 | 18.502573 | 6.478598  6.476509  6.480643  6.491632   7.51581  7.519013  7.517657     7.517708 | 0.0  -0.151021  -0.374457   -0.47746   -0.514066  -0.426469  -0.267842  -0.161675         0.0 |
    | 2000-01-01 03:35:00 |  18.32489 | 6.482697  6.480576  6.483497  6.492358  7.513424  7.516104  7.514733     7.514728 | 0.0   -0.13665  -0.340022  -0.435166   -0.471428  -0.391903  -0.246489  -0.149015         0.0 |
    | 2000-01-01 03:40:00 | 18.161559 | 6.486376  6.484251  6.486117  6.493073  7.511279  7.513459  7.512064     7.511992 | 0.0  -0.122622  -0.306358  -0.393699   -0.429473  -0.357994  -0.225732  -0.136775         0.0 |
    | 2000-01-01 03:45:00 | 18.012452 | 6.489645  6.487543  6.488506  6.493776  7.509369  7.511072  7.509647     7.509498 | 0.0  -0.108986  -0.273581  -0.353205   -0.388339  -0.324646  -0.205277  -0.124698         0.0 |
    | 2000-01-01 03:50:00 |  17.87738 | 6.492519  6.490463  6.490667  6.494462  7.507678  7.508934  7.507477     7.507246 | 0.0  -0.095789   -0.24181  -0.313838   -0.348171  -0.291815  -0.184913  -0.112595         0.0 |
    | 2000-01-01 03:55:00 | 17.762865 | 6.495011  6.493025  6.492605  6.495129  7.506191  7.507034  7.505552     7.505239 | 0.0  -0.083085   -0.21117  -0.275759   -0.309107  -0.259529  -0.164547   -0.10037         0.0 |
    | 2000-01-01 04:00:00 | 17.669858 | 6.497139  6.495242  6.494325  6.495773  7.504888  7.505361  7.503867     7.503478 | 0.0  -0.070925  -0.181789   -0.23913   -0.271326  -0.227897  -0.144219  -0.088037         0.0 |
    | 2000-01-01 04:05:00 | 17.588095 |  6.49892  6.497131  6.495834  6.496391  7.503751    7.5039  7.502415     7.501964 | 0.0  -0.059364  -0.153799  -0.204122    -0.23501  -0.197106  -0.124085  -0.075703         0.0 |
    | 2000-01-01 04:10:00 | 17.516682 | 6.500374  6.498708  6.497142  6.496979  7.502764   7.50264   7.50119     7.500693 | 0.0  -0.048456  -0.127333  -0.170908   -0.200307  -0.167407  -0.104396  -0.063545         0.0 |
    | 2000-01-01 04:15:00 | 17.454819 | 6.501522  6.499994  6.498256  6.497534  7.501914  7.501568   7.50018     7.499658 | 0.0  -0.038258  -0.102526  -0.139659   -0.167395  -0.139065  -0.085456  -0.051786         0.0 |
    | 2000-01-01 04:20:00 |  17.40181 | 6.502386  6.501008  6.499187  6.498052  7.501191  7.500672  7.499372     7.498844 | 0.0  -0.028826  -0.079511  -0.110544   -0.136446  -0.112351  -0.067573  -0.040657         0.0 |
    | 2000-01-01 04:25:00 | 17.357071 | 6.502993  6.501772  6.499946   6.49853  7.500587  7.499942  7.498752     7.498237 | 0.0  -0.020213  -0.058416  -0.083724   -0.107629  -0.087512  -0.051021  -0.030365         0.0 |
    | 2000-01-01 04:30:00 | 17.320107 | 6.503367  6.502309  6.500546  6.498965  7.500097  7.499368  7.498304     7.497816 | 0.0   -0.01247  -0.039358   -0.05935   -0.081106  -0.064756  -0.036015  -0.021069         0.0 |
    | 2000-01-01 04:35:00 | 17.290468 | 6.503536  6.502645  6.500999  6.499354  7.499713  7.498937  7.498009     7.497558 | 0.0  -0.005643  -0.022441  -0.037554   -0.057025   -0.04424  -0.022693  -0.012866         0.0 |
    | 2000-01-01 04:40:00 | 17.267706 | 6.503526  6.502807   6.50132  6.499696   7.49943  7.498638   7.49785     7.497443 | 0.0   0.000325  -0.007748  -0.018441   -0.035518  -0.026068  -0.011113  -0.005791         0.0 |
    | 2000-01-01 04:45:00 | 17.251317 | 6.503293  6.502871  6.501522  6.499987  7.499238  7.498457  7.497806     7.497446 | 0.0   0.007766   0.004576  -0.002147    -0.01669  -0.010289  -0.001265   0.000191         0.0 |
    | 2000-01-01 04:50:00 | 17.240715 | 6.503006  6.502734  6.501721  6.500197  7.499123  7.498383  7.497843     7.497558 | 0.0   0.009568   0.016412   0.009768   -0.000723   0.003092   0.006815   0.005572         0.0 |
    | 2000-01-01 04:55:00 | 17.235145 | 6.502666  6.502506  6.501734  6.500643  7.498705  7.498385  7.497989     7.497736 | 0.0    0.01136   0.022732   0.022306         0.0   0.013935   0.013799   0.008926         0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import modeltools
from hydpy.interfaces import routinginterfaces

# ...from musk
from hydpy.models.sw1d import sw1d_model


class Model(sw1d_model.Main_CrossSectionModel_V2, routinginterfaces.RoutingModel_V2):
    """|sw1d_lias_sluice.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="SW1D-LIAS/Sluice",
        description=(
            "submodel that extends HydPy-SW1D-LIAS with sluice functionalities"
        ),
    )
    __HYDPY_ROOTMODEL__ = False

    INLET_METHODS = (sw1d_model.Reset_DischargeVolume_V1,)
    RECEIVER_METHODS = ()
    RUN_METHODS = ()
    INTERFACE_METHODS = (
        sw1d_model.Determine_MaxTimeStep_V1,
        sw1d_model.Determine_Discharge_V5,
        sw1d_model.Get_MaxTimeStep_V1,
        sw1d_model.Get_Discharge_V1,
        sw1d_model.Get_PartialDischargeUpstream_V1,
        sw1d_model.Get_PartialDischargeDownstream_V1,
        sw1d_model.Get_DischargeVolume_V1,
        sw1d_model.Set_TimeStep_V1,
    )
    ADD_METHODS = (
        sw1d_model.Calc_WaterVolumeUpstream_V1,
        sw1d_model.Calc_WaterVolumeDownstream_V1,
        sw1d_model.Calc_WaterLevelUpstream_V1,
        sw1d_model.Calc_WaterLevelDownstream_V1,
        sw1d_model.Calc_WaterLevel_V1,
        sw1d_model.Calc_WaterDepth_WettedArea_WettedPerimeter_CrossSectionModel_V2,
        sw1d_model.Calc_WaterDepth_WettedArea_WettedPerimeter_V1,
        sw1d_model.Calc_MaxTimeStep_V1,
        sw1d_model.Calc_DischargeUpstream_V1,
        sw1d_model.Calc_DischargeDownstream_V1,
        sw1d_model.Calc_Discharge_V1,
        sw1d_model.Update_Discharge_V1,
        sw1d_model.Update_Discharge_V2,
        sw1d_model.Update_DischargeVolume_V1,
    )
    OUTLET_METHODS = (sw1d_model.Pass_Discharge_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (
        routinginterfaces.CrossSectionModel_V2,
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        routinginterfaces.RoutingModel_V3,
        routinginterfaces.StorageModel_V1,
    )
    SUBMODELS = ()

    crosssection = modeltools.SubmodelProperty(routinginterfaces.CrossSectionModel_V2)

    storagemodelupstream = modeltools.SubmodelProperty(
        routinginterfaces.StorageModel_V1, sidemodel=True
    )
    storagemodelupstream_is_mainmodel = modeltools.SubmodelIsMainmodelProperty()
    storagemodelupstream_typeid = modeltools.SubmodelTypeIDProperty()

    storagemodeldownstream = modeltools.SubmodelProperty(
        routinginterfaces.StorageModel_V1, sidemodel=True
    )
    storagemodeldownstream_is_mainmodel = modeltools.SubmodelIsMainmodelProperty()
    storagemodeldownstream_typeid = modeltools.SubmodelTypeIDProperty()

    routingmodelsupstream = modeltools.SubmodelsProperty(
        routinginterfaces.RoutingModel_V1,
        routinginterfaces.RoutingModel_V2,
        sidemodels=True,
    )

    routingmodelsdownstream = modeltools.SubmodelsProperty(
        routinginterfaces.RoutingModel_V2,
        routinginterfaces.RoutingModel_V3,
        sidemodels=True,
    )


tester = Tester()
cythonizer = Cythonizer()
