"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToTranslate = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToTranslate class.
 */
class LambdaToTranslate extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToTranslate class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToTranslateProps} props - user provided props for the construct.
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // All our tests are based upon this behavior being on, so we're setting
        // context here rather than assuming the client will set it
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        // Check props
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        defaults.CheckTranslateProps(props);
        // Setup VPC if required
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.TRANSLATE);
            if (props.asyncJobs) {
                defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.S3);
            }
        }
        const lambdaEnvironmentVariables = [];
        const translateConfiguration = defaults.ConfigureTranslateSupport(this, id, props);
        if (translateConfiguration.sourceBucket) {
            // Incorporate all the configuration created (to support async jobs)
            this.sourceBucket = translateConfiguration.sourceBucket.bucket;
            this.sourceLoggingBucket = translateConfiguration.sourceBucket.loggingBucket;
            this.sourceBucketInterface = translateConfiguration.sourceBucket.bucketInterface;
            this.destinationBucket = translateConfiguration.destinationBucket?.bucket;
            this.destinationLoggingBucket = translateConfiguration.destinationBucket?.loggingBucket;
            this.destinationBucketInterface = translateConfiguration.destinationBucket?.bucketInterface;
            lambdaEnvironmentVariables.push({
                defaultName: "SOURCE_BUCKET_NAME",
                clientNameOverride: props.sourceBucketEnvironmentVariableName,
                value: this.sourceBucketInterface.bucketName
            });
            lambdaEnvironmentVariables.push({
                defaultName: "DESTINATION_BUCKET_NAME",
                clientNameOverride: props.destinationBucketEnvironmentVariableName,
                value: this.destinationBucketInterface?.bucketName
            });
            lambdaEnvironmentVariables.push({
                defaultName: "DATA_ACCESS_ROLE_ARN",
                clientNameOverride: props.dataAccessRoleArnEnvironmentVariableName,
                value: translateConfiguration.translateRole?.roleArn
            });
        }
        // Now we know everything the Lambda Function needs, we can configure it
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        // Add all actions from translate configuration and client to the Lambda function
        // PassRole is handled separately, because it must specify role being passed as the resource
        const lambdaFunctionRoleActions = [];
        translateConfiguration.lambdaIamActionsRequired.forEach(action => {
            lambdaFunctionRoleActions.push(action);
        });
        if (props.additionalPermissions) {
            props.additionalPermissions.forEach(permission => {
                if (!lambdaFunctionRoleActions.includes(permission)) {
                    lambdaFunctionRoleActions.push(permission);
                }
            });
        }
        this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: lambdaFunctionRoleActions,
            resources: ['*']
        }));
        // Add PassRole in it's own statement
        if (translateConfiguration.translateRole) {
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ["iam:PassRole"],
                resources: [translateConfiguration.translateRole.roleArn]
            }));
        }
        // Configure environment variables
        lambdaEnvironmentVariables.forEach(variable => {
            const varName = variable.clientNameOverride || variable.defaultName;
            this.lambdaFunction.addEnvironment(varName, variable.value);
        });
    }
}
exports.LambdaToTranslate = LambdaToTranslate;
_a = JSII_RTTI_SYMBOL_1;
LambdaToTranslate[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-translate.LambdaToTranslate", version: "2.96.0" };
//# sourceMappingURL=data:application/json;base64,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