from dataclasses import dataclass
from deep_translator import GoogleTranslator
from gtts import gTTS
import threading
import subprocess
import sys
import os
import glob
import tempfile
import wave
import pydub
import pydub.playback as pb
import noisereduce as nr
from scipy.io import wavfile
import ChatAIStreamer as casr

ChatAIStreamer = casr.ChatAIStreamer
streamParams = casr.streamParams
userMessage = casr.userMessage
aiParams = casr.aiParams

TMPFILE_POSTFIX = "_GttsAIStreamer"

# Data type of voice used by GttsAIStreamer 
@dataclass
class voiceData:
  content: bytes=None
  sample_width: int=2
  frame_rate: int=2400
  channels: int=1
  duration_seconds: int=0

# Concrete VoiceGanarator of GttsAIStreamer
# 'generate' is overridden.
class GttsGenerator(casr.voiceGenerator):
  def __init__(self, lang='en'):
    self.lang = lang
    super(GttsGenerator, self).__init__()
  def generate(self, text):
    # Translate into target language to make TTS voice fluent.
    text_en = GoogleTranslator(target=self.lang).translate(text=text)
    # Create TTS instance.
    tts = gTTS(text_en, lang=self.lang)
    # Create path of temporary file.
    with tempfile.NamedTemporaryFile() as file:
      file_path = file.name + TMPFILE_POSTFIX
    # Save voice to temporary file in order to get wave binary.
    # This process may not be necessary by other TTS.
    tts.save(file_path)
    # Create Pydub AudioSegment.
    segment = pydub.AudioSegment.from_mp3(file_path)
    os.remove(file_path)
    # Pack in viceData format.
    voice = voiceData(
      content=segment.get_array_of_samples(),
      sample_width=segment.sample_width,
      frame_rate=segment.frame_rate,
      channels=segment.channels,
      duration_seconds=segment.duration_seconds )

    # return voice with translated text.
    return text_en, voice

# Extend streamerParams and params to hold voiceGenerator instance.
# voice_generator is defaultly intialized with English voiceGenerator.
@dataclass
class streamerParams(casr.streamerParams):
  voice_generator : casr.voiceGenerator = GttsGenerator()
@dataclass
class params(casr.params):
  streamer_params : streamerParams = streamerParams()

# VoicePlayer class which plays the voice generated by GttsGenerator.
class VoicePlayer(threading.Thread):
  def __init__(self, voice, volume=100, python_command="python3"):
    self.__voice = voice
    self.__volume = volume
    self.__python_command = python_command
    super(VoicePlayer, self).__init__()

  def run(self):
    # Create path of temporary file.
    with tempfile.NamedTemporaryFile() as file:
      file_path = file.name + TMPFILE_POSTFIX
    # Create wave writer.
    writer = wave.open(file_path, mode='wb')

    # Write voice to Temporary wav file in order to receive it on subprocess.
    writer.setsampwidth(self.__voice.sample_width)
    writer.setframerate(self.__voice.frame_rate)
    writer.setnchannels(self.__voice.channels)
    writer.writeframesraw(self.__voice.content)

    # Play voice on subprocess in order to suppress pydub log.
    subprocess.run(
      [ self.__python_command,
         __file__,
        file_path,
        str(self.__voice.sample_width),
        str(self.__voice.frame_rate),
        str(self.__voice.channels),
        str(self.__voice.duration_seconds),
        str(self.__volume) ],
        stdout=subprocess.PIPE, stderr=subprocess.PIPE )

# GttsAIStreamer as inheritence from ChatAIStreamer
class GttsAIStreamer(casr.ChatAIStreamer):
  def __init__(self, params):
    # Remove garbage of temporary files.
    with tempfile.NamedTemporaryFile() as file:
      tmp_dir_path = os.path.dirname(file.name)
    for file_path in glob.glob(f"{tmp_dir_path}/*{TMPFILE_POSTFIX}*"):
      os.remove(file_path)
    super(GttsAIStreamer, self).__init__(params)

def play(file_path, sample_width, frame_rate, channels, duration_seconds, volume):
  _ = duration_seconds
  # Open wav file generated by GttsGenerator.
  rate, data = wavfile.read(file_path)
  os.remove(file_path)
  # Reduce noise.
  data_reduced = nr.reduce_noise(y=data, sr=rate)
  # Create pydub AudioSegment.
  segment = pydub.AudioSegment( data_reduced.astype("int16").tobytes(), 
                                sample_width=sample_width, 
                                frame_rate=frame_rate, 
                                channels=channels )

  # Add silent section to the top of the voice.
  silent = lambda msec: pydub.AudioSegment.silent(duration=msec)
  segment = silent(2000) + segment

  # Adjust volume.
  delta = pydub.utils.ratio_to_db(volume * 2 / 100.)
  segment = segment + delta
  pb.play(segment)

if __name__ == '__main__':
    # Enable to execute from command line in order to call from subprocess.
    play(sys.argv[1], int(sys.argv[2]), int(sys.argv[3]), int(sys.argv[4]), float(sys.argv[5]), int(sys.argv[6]))
