"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.launchBrowserServer = launchBrowserServer;
exports.printApiJson = printApiJson;
exports.runDriver = runDriver;
exports.runServer = runServer;

var _fs = _interopRequireDefault(require("fs"));

var playwright = _interopRequireWildcard(require("../.."));

var _server = require("../server");

var _transport = require("../protocol/transport");

var _playwrightServer = require("../remote/playwrightServer");

var _processLauncher = require("../utils/processLauncher");

var _recorder = require("../server/recorder");

var _recorderApp = require("../server/recorder/recorderApp");

var _instrumentation = require("../server/instrumentation");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* eslint-disable no-console */
function printApiJson() {
  // Note: this file is generated by build-playwright-driver.sh
  console.log(JSON.stringify(require('../../api.json')));
}

function runDriver() {
  const dispatcherConnection = new _server.DispatcherConnection();
  new _server.Root(dispatcherConnection, async (rootScope, {
    sdkLanguage
  }) => {
    const playwright = (0, _server.createPlaywright)(sdkLanguage);
    return new _server.PlaywrightDispatcher(rootScope, playwright);
  });
  const transport = process.send ? new _transport.IpcTransport(process) : new _transport.PipeTransport(process.stdout, process.stdin);

  transport.onmessage = message => dispatcherConnection.dispatch(JSON.parse(message));

  dispatcherConnection.onmessage = message => transport.send(JSON.stringify(message));

  transport.onclose = () => {
    // Drop any messages during shutdown on the floor.
    dispatcherConnection.onmessage = () => {};

    selfDestruct();
  };
}

async function runServer(port, path = '/', maxClients = Infinity, enableSocksProxy = true, reuseBrowser = false) {
  const maxIncomingConnections = maxClients;
  const maxConcurrentConnections = reuseBrowser ? 1 : maxClients;
  const server = new _playwrightServer.PlaywrightServer(reuseBrowser ? 'reuse-browser' : 'auto', {
    path,
    maxIncomingConnections,
    maxConcurrentConnections,
    enableSocksProxy
  });
  const wsEndpoint = await server.listen(port);
  process.on('exit', () => server.close().catch(console.error));
  console.log('Listening on ' + wsEndpoint); // eslint-disable-line no-console

  process.stdin.on('close', () => selfDestruct());
  if (process.send && server.preLaunchedPlaywright()) wireController(server.preLaunchedPlaywright(), wsEndpoint);
}

async function launchBrowserServer(browserName, configFile) {
  let options = {};
  if (configFile) options = JSON.parse(_fs.default.readFileSync(configFile).toString());
  const browserType = playwright[browserName];
  const server = await browserType.launchServer(options);
  console.log(server.wsEndpoint());
}

function selfDestruct() {
  // Force exit after 30 seconds.
  setTimeout(() => process.exit(0), 30000); // Meanwhile, try to gracefully close all browsers.

  (0, _processLauncher.gracefullyCloseAll)().then(() => {
    process.exit(0);
  });
}

const internalMetadata = (0, _instrumentation.serverSideCallMetadata)();

class ProtocolHandler {
  constructor(playwright) {
    this._playwright = void 0;
    this._autoCloseTimer = void 0;
    this._playwright = playwright;
    playwright.instrumentation.addListener({
      onPageOpen: () => this._sendSnapshot(),
      onPageNavigated: () => this._sendSnapshot(),
      onPageClose: () => this._sendSnapshot()
    }, null);
  }

  _sendSnapshot() {
    const browsers = [];

    for (const browser of this._playwright.allBrowsers()) {
      const b = {
        name: browser.options.name,
        guid: browser.guid,
        contexts: []
      };
      browsers.push(b);

      for (const context of browser.contexts()) {
        const c = {
          guid: context.guid,
          pages: []
        };
        b.contexts.push(c);

        for (const page of context.pages()) {
          const p = {
            guid: page.guid,
            url: page.mainFrame().url()
          };
          c.pages.push(p);
        }
      }
    }

    process.send({
      method: 'browsersChanged',
      params: {
        browsers
      }
    });
  }

  async resetForReuse() {
    const contexts = new Set();

    for (const page of this._playwright.allPages()) contexts.add(page.context());

    for (const context of contexts) await context.resetForReuse(internalMetadata, null);
  }

  async navigate(params) {
    for (const p of this._playwright.allPages()) await p.mainFrame().goto(internalMetadata, params.url);
  }

  async setMode(params) {
    await gc(this._playwright);

    if (params.mode === 'none') {
      for (const recorder of await allRecorders(this._playwright)) {
        recorder.setHighlightedSelector('');
        recorder.setMode('none');
      }

      this.setAutoClose({
        enabled: true
      });
      return;
    }

    const browsers = this._playwright.allBrowsers();

    if (!browsers.length) await this._playwright.chromium.launch(internalMetadata, {
      headless: false
    }); // Create page if none.

    const pages = this._playwright.allPages();

    if (!pages.length) {
      const [browser] = this._playwright.allBrowsers();

      const {
        context
      } = await browser.newContextForReuse({}, internalMetadata);
      await context.newPage(internalMetadata);
    } // Toggle the mode.


    for (const recorder of await allRecorders(this._playwright)) {
      recorder.setHighlightedSelector('');
      if (params.mode === 'recording') recorder.setOutput(params.language, params.file);
      recorder.setMode(params.mode);
    }

    this.setAutoClose({
      enabled: true
    });
  }

  async setAutoClose(params) {
    if (this._autoCloseTimer) clearTimeout(this._autoCloseTimer);
    if (!params.enabled) return;

    const heartBeat = () => {
      if (!this._playwright.allPages().length) selfDestruct();else this._autoCloseTimer = setTimeout(heartBeat, 5000);
    };

    this._autoCloseTimer = setTimeout(heartBeat, 30000);
  }

  async highlight(params) {
    for (const recorder of await allRecorders(this._playwright)) recorder.setHighlightedSelector(params.selector);
  }

  async hideHighlight() {
    await this._playwright.hideHighlight();
  }

  async kill() {
    selfDestruct();
  }

}

function wireController(playwright, wsEndpoint) {
  process.send({
    method: 'ready',
    params: {
      wsEndpoint
    }
  });
  const handler = new ProtocolHandler(playwright);
  process.on('message', async message => {
    try {
      const result = await handler[message.method](message.params);
      process.send({
        id: message.id,
        result
      });
    } catch (e) {
      process.send({
        id: message.id,
        error: e.toString()
      });
    }
  });
}

async function gc(playwright) {
  for (const browser of playwright.allBrowsers()) {
    for (const context of browser.contexts()) {
      if (!context.pages().length) await context.close((0, _instrumentation.serverSideCallMetadata)());
    }

    if (!browser.contexts()) await browser.close();
  }
}

async function allRecorders(playwright) {
  const contexts = new Set();

  for (const page of playwright.allPages()) contexts.add(page.context());

  const result = await Promise.all([...contexts].map(c => _recorder.Recorder.show(c, {
    omitCallTracking: true
  }, () => Promise.resolve(new InspectingRecorderApp()))));
  return result.filter(Boolean);
}

class InspectingRecorderApp extends _recorderApp.EmptyRecorderApp {
  async setSelector(selector) {
    process.send({
      method: 'inspectRequested',
      params: {
        selector
      }
    });
  }

}