import os
import json
import csv
import requests
import time
import git
import re

class PyExploitDb:
    EXPLOIT_DB_REPO = "https://gitlab.com/exploit-database/exploitdb.git"
    USER_AGENT = {'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_1) Chrome/39.0 Safari/537.36'}
    
    def __init__(self):
        self.cveToExploitMap = {}
        self.currentPath = os.path.dirname(os.path.abspath(__file__))
        self.edbidToCveFile = os.path.join(self.currentPath, "edbidToCve.json")
        self.cveToEdbidFile = os.path.join(self.currentPath, "cveToEdbid.json")
        self.exploitDbPath = os.path.join(self.currentPath, "exploit-database")
        self.requestCoolOffTime = 1
        self.debug = False

    def logDebug(self, message):
        if self.debug:
            print(message)

    def cloneOrUpdateRepo(self):
        if not os.path.isdir(self.exploitDbPath):
            self.logDebug("Cloning exploit-database repository")
            git.Repo.clone_from(self.EXPLOIT_DB_REPO, self.exploitDbPath)
        else:
            self.pullLatestUpdates()

    def pullLatestUpdates(self):
        try:
            self.logDebug("Pulling exploit-database updates...")
            git.Git(self.exploitDbPath).pull('origin', 'main')
        except git.exc.GitCommandError:
            self.logDebug("Repo broken, re-cloning...")
            self.deleteAndRecloneRepo()

    def deleteAndRecloneRepo(self):
        os.rmdir(self.exploitDbPath)
        git.Repo.clone_from(self.EXPLOIT_DB_REPO, self.exploitDbPath)

    def openFile(self, exploitMap="cveToEdbid.json", encoding="utf-8"):
        self.cloneOrUpdateRepo()
        with open(os.path.join(self.currentPath, exploitMap), encoding=encoding) as fileData:
            self.cveToExploitMap = json.load(fileData)
            self.logDebug(self.cveToExploitMap)

    def getCveDetails(self, cveSearch):
        with open(os.path.join(self.exploitDbPath, "files_exploits.csv"), encoding="utf-8") as file:
            reader = csv.reader(file)
            next(reader)
            for row in reader:
                if row[0] in self.cveToExploitMap.get(cveSearch, []):
                    return self.extractCveDetails(row)
        self.logDebug("ERROR - No EDB Id found")
        return {}

    def extractCveDetails(self, row):
        details = {
            'id': row[0],
            'file': row[1],
            'description': row[2],
            'date': row[3],
            'author': row[4],
            'type': row[5],
            'platform': row[6],
            'port': row[7],
            'date_updated': row[9],
            'verified': row[10],
            'codes': row[11],
            'tags': row[12],
            'aliases': row[13],
            'app_url': row[14],
            'src_url': row[15],
        }
        self.logDebug(f"CVE Details: {details}")
        return details

    def searchCve(self, cveSearch):
        if not cveSearch:
            return []
        cveSearch = cveSearch.upper()
        self.logDebug(f"Searching for CVE: {cveSearch}")
        if cveSearch in self.cveToExploitMap:
            return self.getCveDetails(cveSearch)
        return []

    def updateDb(self):
        data = self.loadExistingData(self.edbidToCveFile)
        exploits = self.loadExploitCsv()
        
        for i, row in enumerate(exploits):
            edb_id = row[0]
            if edb_id not in data:
                data[edb_id] = self.fetchCvesForExploit(edb_id)

        self.writeJson(self.edbidToCveFile, data)
        self.createCveToExploitMap(data)
        self.writeJson(self.cveToEdbidFile, self.cveToExploitMap)

    def loadExistingData(self, file_path):
        if os.path.exists(file_path):
            with open(file_path, encoding="utf-8") as fileData:
                try:
                    return json.load(fileData)
                except json.JSONDecodeError:
                    self.logDebug(f"Corrupt file detected at {file_path}, recreating...")
        return {}

    def loadExploitCsv(self):
        with open(os.path.join(self.exploitDbPath, "files_exploits.csv"), encoding="utf-8") as file:
            reader = csv.reader(file)
            next(reader)
            return list(reader)

    def fetchCvesForExploit(self, edb_id):
        requestUri = f"https://www.exploit-db.com/exploits/{edb_id}"
        self.logDebug(f"Requesting {requestUri}")
        while True:
            try:
                response = requests.get(requestUri, headers=self.USER_AGENT, timeout=10)
                return self.parseCvesFromContent(response.content)
            except requests.RequestException as e:
                self.logDebug(f"Request error: {e}, retrying after cool-off")
                time.sleep(self.requestCoolOffTime)

    def parseCvesFromContent(self, content):
        indexes = [m.start() for m in re.finditer('https://nvd.nist.gov/vuln/detail/CVE-', content)]
        return {content[pos + 33: pos + 47].decode("ISO-8859-1").strip() for pos in indexes}

    def writeJson(self, file_path, data):
        with open(file_path, "w", encoding="utf-8") as fileData:
            json.dump(data, fileData, indent=2)

    def createCveToExploitMap(self, data):
        for edb_id, cves in data.items():
            for cve in cves:
                if cve not in self.cveToExploitMap:
                    self.cveToExploitMap[cve] = []
                self.cveToExploitMap[cve].append(edb_id)

def test():
    pEdb = PyExploitDb()
    pEdb.debug = False
    pEdb.openFile()
    results = pEdb.searchCve("CVE-2018-14592")
    print('PASS' if results else 'FAIL')

if __name__ == "__main__":
    test()
