"""Default class for joints, i.e. points in space at a specific timestamp. The methods in this class are mainly
handled by the methods in the class Pose, but some of them can be directly accessed."""

from math import cos, sin
from classes.exceptions import InvalidParameterValueException


class Joint(object):
    """Creates a Joint instance, with a joint label and x, y, and z coordinates.

    .. versionadded:: 2.0

    Parameters
    ----------
    joint_label: str, optional
        The label of the joint (e.g. ``"Head"``).
    x: float, optional
        The position of the joint on the x axis (in meters).
    y: float, optional
        The position of the joint on the y axis (in meters).
    z: float, optional
        The position of the joint on the z axis (in meters).

    Attributes
    ----------
    joint_label: str
        The label of the joint (e.g. ``"Head"``).
    x: float
        The position of the joint on the x axis (in meters).
    y: float
        The position of the joint on the y axis (in meters).
    z: float
        The position of the joint on the z axis (in meters).
    position: list(float)
        The coordinates of the joint, equivalent to [x, y, z].
    _has_velocity_over_threshold: bool
        Defines if the velocity of this joint, compared to the previous joint, has been found to be over threshold
        defined in the parameters :meth:`.Sequence.correct_jitter`.
    _is_corrected: bool
        Defines if the coordinates of this joint have been modified by :meth:`.Sequence.correct_jitter`,
        :meth:`.Sequence.re_reference` or :meth:`.Sequence.correct_zeros`.
    _is_randomized: bool
        Defines if the coordinates of this joint have been randomly generated by (typically, by
        :meth:`.Sequence.randomize`).
    """

    def __init__(self, joint_label=None, x=None, y=None, z=None):

        self.joint_label = joint_label
        self.x = float(x)
        self.y = float(y)
        self.z = float(z)
        self.position = [self.x, self.y, self.z]

        self._has_velocity_over_threshold = False
        self._is_corrected = False
        self._is_randomized = False

    # === Setter functions ===

    def set_joint_label(self, joint_label):
        """Sets the :attr:`joint_label` attribute of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        joint_label: str
            The label of the joint (e.g. ``"Head"``).
        """
        self.joint_label = joint_label

    def set_x(self, x):
        """Sets the :attr:`x` coordinate of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        x: float
            The position of the joint on the x axis.
        """
        self.x = float(x)
        self.position[0] = self.x

    def set_y(self, y):
        """Sets the :attr:`y` coordinate of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        y: float
            The position of the joint on the y axis.
        """
        self.y = float(y)
        self.position[1] = self.y

    def set_z(self, z):
        """Sets the :attr:`z` coordinate of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        z: float
            The position of the joint on the z axis.
        """
        self.z = float(z)
        self.position[2] = self.z

    def set_position(self, x, y, z):
        """Sets the :attr:`x`, :attr:`y` and :attr:`z` coordinates of the joint.

        .. versionadded:: 2.0

        Parameters
        ----------
        x: float
            The position of the joint on the x axis.
        y: float
            The position of the joint on the y axis.
        z: float
            The position of the joint on the z axis.
        """

        self.x = float(x)
        self.y = float(y)
        self.z = float(z)
        self.position = [self.x, self.y, self.z]

    def set_to_zero(self):
        """Sets the joints coordinates to (0, 0, 0).

        .. versionadded:: 2.0
        """
        self.x = 0.0
        self.y = 0.0
        self.z = 0.0
        self.position = [self.x, self.y, self.z]

    # === Getter functions ===

    def get_joint_label(self):
        """Returns the :attr:`joint_label` attribute.

        .. versionadded:: 2.0

        Returns
        -------
        str
            The label of the joint (e.g. ``"Head"``).
        """
        return self.joint_label

    def get_x(self):
        """Returns the :attr:`x` coordinate of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The x coordinate of the joint (in meters).
        """
        return self.x

    def get_y(self):
        """Returns the :attr:`y` coordinate of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The y coordinate of the joint (in meters).
        """
        return self.y

    def get_z(self):
        """Returns the :attr:`z` coordinate of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The z coordinate of the joint (in meters).
        """
        return self.z

    def get_coordinate(self, axis):
        """Returns the coordinate on the specified axis (x, y, or z).

        .. versionadded:: 2.0

        Parameters
        ----------
        axis: str
            The axis (``"x"``, ``"y"``, or ``"z"``) from which to get the coordinate.

        Returns
        -------
        float
            The x, y or z coordinate of the joint (in meters).
        """
        if axis == "x":
            return self.x
        elif axis == "y":
            return self.y
        elif axis == "z":
            return self.z
        else:
            raise InvalidParameterValueException("axis", axis, ["x", "y", "z"])

    def get_position(self):
        """Returns a list containing the :attr:`x`, :attr:`y` and :attr:`z` coordinates of the joint.

        .. versionadded:: 2.0

        Returns
        -------
        float
            The x coordinate of the joint (in meters).
        """
        return self.position

    def get_has_velocity_over_threshold(self):
        """Returns the value of the attribute :attr:`_has_velocity_over_threshold`, which will be `True` if the
        velocity of this joint, compared to the previous joint, has been found to be over threshold defined in the
        parameters of :meth:`.Sequence.correct_jitter()`.

        .. versionadded:: 2.0

        Returns
        -------
        bool
            Value of the attribute :attr:`_has_velocity_over_threshold`.
        """
        return self._has_velocity_over_threshold

    def get_is_corrected(self):
        """Returns the value of the attribute :attr:`_is_corrected`, which will be `True` if the coordinates of this
        joint have been modified by :meth:`.Sequence.correct_jitter()`, :meth:`.Sequence.re_reference()` or
        :meth:`.Sequence.correct_zeros()`.

        .. versionadded:: 2.0

        Returns
        -------
        bool
            Value of the attribute :attr:`_is_corrected`.
        """
        return self._is_corrected

    def get_is_randomized(self):
        """Returns the value of the attribute :attr:`_is_randomized`, which will be `True` if the coordinates of this
        joint have been randomly generated by :meth:`.Sequence.randomize()`.

        .. versionadded:: 2.0

        Returns
        -------
        bool
            Value of the attribute :attr:`_is_randomized`.
        """
        return self._is_randomized

    def get_copy(self):
        """Returns a deep copy of itself.

        .. versionadded:: 2.0

        Returns
        -------
        Joint
            A deep copy of the Joint instance.
        """

        j = Joint(self.joint_label, self.x, self.y, self.z)
        j._set_has_velocity_over_threshold(self._has_velocity_over_threshold)
        j._set_is_corrected(self._is_corrected)
        j._is_randomized = self._is_randomized
        return j

    # === Rotation function ===
    def convert_rotation(self, yaw=0, pitch=0, roll=0):
        """Returns converted coordinates given three rotations: yaw, pitch and roll.

        .. versionadded:: 2.0

        Warning
        -------
        This function is experimental as of version 2.0.

        Parameters
        ----------
        yaw: float, optional
            The angle of yaw, or rotation on the x axis, in degrees (default: 0).
        pitch: float, optional
            The angle of pitch, or rotation on the y axis, in degrees (default: 0).
        roll: float, optional
            The angle of roll, or rotation on the z axis, in degrees (default: 0).

        Returns
        -------
        float
            The converted x coordinate.
        float
            The converted y coordinate.
        float
            The converted z coordinate.
        """
        x = self.x * cos(yaw) * cos(pitch) + \
            self.y * (cos(yaw) * sin(pitch) * sin(roll) - sin(yaw) * cos(roll)) + \
            self.z * (cos(yaw) * sin(pitch) * cos(roll) - sin(yaw) * sin(roll))
        y = self.x * sin(yaw) * cos(pitch) + \
            self.y * (sin(yaw) * sin(pitch) * sin(roll) + cos(yaw) * cos(roll)) + \
            self.z * (sin(yaw) * sin(pitch) * sin(roll) * cos(yaw) * sin(roll))
        z = self.x * -sin(pitch) + \
            self.y * cos(pitch) * sin(roll) + \
            self.z * cos(pitch) * cos(roll)
        return x, y, z

    # === Modification functions ===
    def _correct_joint(self, x, y, z):
        """Assigns new x, y and z coordinates to the joint and marks it as corrected.

        .. versionadded:: 2.0

        Parameters
        ----------
        x: float
            The position of the joint on the x axis.
        y: float
            The position of the joint on the y axis.
        z: float
            The position of the joint on the z axis.
        """
        self.set_position(x, y, z)
        self._is_corrected = True

    def _randomize_coordinates_keep_movement(self, joint_pose_0, joint_random, verbosity=1):
        """Modifies the coordinates of the joint while keeping the relative position compared to the position of the
        joint at the beginning of the sequence. This function is used by `.Sequence.randomize()`.

        .. versionadded:: 2.0

        Parameters
        ----------
        joint_pose_0: Joint
            The joint from the first pose of the sequence, with the same :attr:`joint_label`.
        joint_random: Joint
            A joint with random coordinates, generated by :meth:`.Sequence.randomize()`
        verbosity: int, optional
            Sets how much feedback the code will provide in the console output:

            • *0: Silent mode.* The code won’t provide any feedback, apart from error messages.
            • *1: Normal mode* (default). The code will provide essential feedback such as progression markers and
              current steps.
            • *2: Chatty mode.* The code will provide all possible information on the events happening. Note that this
              may clutter the output and slow down the execution.
        """

        if verbosity > 1:
            print("Coordinates before randomization: "+str(self.x)+", "+str(self.y)+", "+str(self.z))
        self.x = joint_random.x + (self.x - joint_pose_0.x)
        self.y = joint_random.y + (self.y - joint_pose_0.y)
        self.z = joint_random.z + (self.z - joint_pose_0.z)
        self._is_randomized = True
        if verbosity > 1:
            print("Coordinates after randomization: "+str(self.x)+", "+str(self.y)+", "+str(self.z))

    def _set_has_velocity_over_threshold(self, has_velocity_over_threshold):
        """Sets the value of the attribute :attr:`_has_velocity_over_threshold`. This function is used by
        :meth:`.Sequence.correct_jitter()`.

        .. versionadded:: 2.0

        Parameters
        ----------
        has_velocity_over_threshold: bool
            If `True`, the velocity of this joint, compared to the previous joint, has been found to be over threshold
            defined in the parameters of :meth:`.Sequence.correct_jitter()`.
        """
        self._has_velocity_over_threshold = has_velocity_over_threshold

    def _set_is_corrected(self, is_corrected):
        """Sets the value of the attribute :attr:`_is_corrected`.

        .. versionadded:: 2.0

        Parameters
        ----------
        is_corrected: bool
            If `True`, the coordinates of this joint have been modified by :meth:`.Sequence.correct_jitter()`,
             :meth:`.Sequence.re_reference()` or :meth:`.Sequence.correct_zeros()`.
        """
        self._is_corrected = is_corrected

    # === Magic methods ===

    def __repr__(self):
        """Returns a string containing the joint label, the x, y ans z coordinates, and adds information if one or more
        of the private attributes :attr:`_has_velocity_over_threshold`, :attr:`_is_corrected` or :attr:`_is_randomized`
        are `True`.

        Returns
        -------
        str
            A formatted string of the information contained in all the attributes of the object.

        Examples
        --------
        >>> sequence = Sequence("C:/Users/Adrian/Sequences/seq1/")
        >>> joint = sequence.get_pose(4).get_joint("Head")
        >>> print(joint)
        Head: (4.8, 15.16, 23.42) CORRECTED
        """
        txt = self.joint_label + ": (" + str(self.x) + ", " + str(self.y) + ", " + str(self.z) + ")"
        if self._has_velocity_over_threshold:
            txt += " OVER THRESHOLD"
        if self._is_corrected:
            txt += " CORRECTED"
        if self._is_randomized:
            txt += " RANDOMIZED"
        return txt

    def __eq__(self, other):
        """Returns `True` if the attributes :attr:`x`, :attr:`y`, :attr:`z` and :attr:`joint_label` are identical
        between the two :class:`Joint` objects.

        .. versionadded:: 2.0

        Parameters
        ----------
        other: Joint
            Another :class:`Joint` object.
        """
        if self.x == other.x and self.y == other.y and self.z == other.z and self.joint_label == other.joint_label:
            return True
        return False
