"""
Tools for generating toy P2CPs.
"""

from typing import Tuple

import numpy as np
import pandas as pd

from hiveplotlib import (
    P2CP,
    p2cp_n_axes,
)


def four_gaussian_blobs_3d(
    num_points: int = 50, noise: float = 0.5, random_seed: int = 0
) -> pd.DataFrame:
    """
    Generate a ``pandas`` dataframe of four Gaussian blobs in 3d.

    This dataset serves as a simple example for showing 3d viz using Polar Parallel Coordinates Plots (P2CPs) instead
    of 3d plotting.

    :param num_points: number of points in each blob.
    :param noise: noisiness of Gaussian blobs.
    :param random_seed: random seed to generate consistent data between calls.
    :return: ``(num_points * 4, 4)`` ``pd.DataFrame`` of X, Y, Z, and blob labels.
    """
    # dimension of data (e.g. 3 => 3d data)
    dim = 3

    # keeping a subset of the corner blobs to plot
    corners_to_keep = [0, 1, 2, 4]

    # name of the features we will create for each set of data generated below
    feature_names = ["X", "Y", "Z", "Label"]

    # set seed for consistent data
    rng = np.random.default_rng(random_seed)

    # build list of arrays of Gaussian blobs at each corner of a cube
    b_list = []
    coords = []
    for i in [0, 5]:
        for j in [0, 5]:
            for k in [0, 5]:
                b = rng.normal(scale=noise, size=num_points * dim).reshape(
                    num_points, dim
                )
                b[:, 0] += i
                b[:, 1] += j
                b[:, 2] += k
                b = np.c_[b, np.repeat(len(b_list), b.shape[0])]
                b_list.append(b)
                coords.append((i, j, k))

    # put our 4 blobs of interest into a single dataframe
    df = pd.DataFrame(
        np.vstack([b_list[i] for i in corners_to_keep]), columns=feature_names
    )

    # make the labels ints
    df.Label = df.Label.astype(int)
    # replace the 4s with 3s so our labels are just range(4)
    df.Label.to_numpy()[df.Label.to_numpy() == 4] = 3

    return df


def example_p2cp(
    num_points: int = 50,
    noise: float = 0.5,
    random_seed: int = 0,
    four_colors: Tuple[str, str, str, str] = (
        "#de8f05",
        "#029e73",
        "#cc78bc",
        "#0173b2",
    ),
    **p2cp_n_axes_kwargs,
) -> P2CP:
    """
    Generate example P2CP of four gaussian blobs.

    Points will be generated by calling :py:func:`hiveplotlib.datasets.toy_p2cps.four_gaussian_blobs_3d()` and turned
    into a P2CP via :py:func:`hiveplotlib.p2cp_n_axes()`.

    :param num_points: number of points in each Gaussian blob.
    :param noise: noisiness of Gaussian blobs.
    :param random_seed: random seed to generate consistent data between calls.
    :param four_colors: four colors to use for four Gaussian blobs.
    :param p2cp_n_axes_kwargs: additional keyword arguments for the underlying
        :py:func:`hiveplotlib.p2cp_n_axes()` call.
    :return: resulting ``P2CP`` instance.
    """
    # color palette for the blobs
    df = four_gaussian_blobs_3d(
        num_points=num_points,
        noise=noise,
        random_seed=random_seed,
    )

    # the first (0) blob needs to be plotted last, so let's give it the last label for pandas to sort to plot last
    df.loc[df["Label"] == 0, "Label"] = 4

    # align the colors with the figure from the paper
    indices_dict = [{"color": color} for color in four_colors]

    return p2cp_n_axes(
        data=df,
        split_on="Label",
        vmins=[-1] * 3,
        vmaxes=[6] * 3,
        indices_list_kwargs=indices_dict,
        **p2cp_n_axes_kwargs,
    )
