from __future__ import annotations

from abc import ABC
from typing import TYPE_CHECKING, Tuple

import nested_pandas as npd
import numpy as np
import numpy.typing as npt
import pandas as pd
from hats.catalog import TableProperties
from hats.pixel_math.spatial_index import SPATIAL_INDEX_COLUMN

if TYPE_CHECKING:
    from lsdb.catalog import Catalog


# pylint: disable=too-many-instance-attributes, too-many-arguments
class AbstractCrossmatchAlgorithm(ABC):
    """Abstract class used to write a crossmatch algorithm

    To specify a custom function, write a class that subclasses the
    `AbstractCrossmatchAlgorithm` class, and overwrite the `perform_crossmatch` function.

    The function should be able to perform a crossmatch on two pandas DataFrames
    from a partition from each catalog. It should return two 1d numpy arrays of equal lengths
    with the indices of the matching rows from the left and right dataframes, and a dataframe
    with any extra columns generated by the crossmatch algorithm, also with the same length.
    These columns are specified in {AbstractCrossmatchAlgorithm.extra_columns}, with
    their respective data types, by means of an empty pandas dataframe. As an example,
    the KdTreeCrossmatch algorithm outputs a "_dist_arcsec" column with the distance between
    data points. Its extra_columns attribute is specified as follows::

        pd.DataFrame({"_dist_arcsec": pd.Series(dtype=np.dtype("float64"))})

    The class will have been initialized with the following parameters, which the
    crossmatch function should use:
        - left: npd.NestedFrame,
        - right: npd.NestedFrame,
        - left_order: int,
        - left_pixel: int,
        - right_order: int,
        - right_pixel: int,
        - left_metadata: hc.catalog.Catalog,
        - right_metadata: hc.catalog.Catalog,
        - right_margin_hc_structure: hc.margin.MarginCatalog,
        - suffixes: Tuple[str, str]

    You may add any additional keyword argument parameters to the crossmatch
    function definition, and the user will be able to pass them in as kwargs in the
    `Catalog.crossmatch` method. Any additional keyword arguments must also be added to the
    `CrossmatchAlgorithm.validate` classmethod by overwriting the method.
    """

    extra_columns: pd.DataFrame | None = None
    """The metadata for the columns generated by the crossmatch algorithm"""

    def __init__(
        self,
        left: npd.NestedFrame,
        right: npd.NestedFrame,
        left_order: int,
        left_pixel: int,
        right_order: int,
        right_pixel: int,
        left_catalog_info: TableProperties,
        right_catalog_info: TableProperties,
        right_margin_catalog_info: TableProperties | None,
        suffixes: Tuple[str, str],
    ):
        """Initializes a crossmatch algorithm

        Args:
            left (pd.DataFrame): Data from the pixel in the left tree
            right (pd.DataFrame): Data from the pixel in the right tree
            left_order (int): The HEALPix order of the left pixel
            left_pixel (int): The HEALPix pixel number in NESTED ordering of the left pixel
            right_order (int): The HEALPix order of the right pixel
            right_pixel (int): The HEALPix pixel number in NESTED ordering of the right pixel
            left_catalog_info (hats.TableProperties): The hats TableProperties object with the
                metadata of the left catalog
            right_catalog_info (hats.TableProperties): The hats TableProperties object with the
                metadata of the right catalog
            right_margin_catalog_info (hats.TableProperties): The hats TableProperties
                objects with the metadata of the right **margin** catalog
            suffixes (Tuple[str,str]): A pair of suffixes to be appended to the end of each column
                name, with the first appended to the left columns and the second to the right
                columns
        """
        self.left = left.copy(deep=False)
        self.right = right.copy(deep=False)
        self.left_order = left_order
        self.left_pixel = left_pixel
        self.right_order = right_order
        self.right_pixel = right_pixel
        self.left_catalog_info = left_catalog_info
        self.right_catalog_info = right_catalog_info
        self.right_margin_catalog_info = right_margin_catalog_info
        self.suffixes = suffixes

    def crossmatch(self, **kwargs) -> npd.NestedFrame:
        """Perform a crossmatch"""
        l_inds, r_inds, extra_cols = self.perform_crossmatch(**kwargs)
        if not len(l_inds) == len(r_inds) == len(extra_cols):
            raise ValueError(
                "Crossmatch algorithm must return left and right indices and extra columns with same length"
            )
        return self._create_crossmatch_df(l_inds, r_inds, extra_cols)

    def perform_crossmatch(self) -> Tuple[np.ndarray, np.ndarray, pd.DataFrame]:
        """Performs a crossmatch to get the indices of the matching rows and any extra columns

        Any additional keyword arguments needed can be added to this method in the subclass, and the user
        will be able to pass them through the `Catalog.crossmatch` method.

        Returns:
            A tuple of:
                - a numpy array with the indices of the matching rows from the left table
                - a numpy array with the indices of the matching rows from the right table
                - a pandas dataframe with any additional columns generated by the algorithm

            These all must have the same lengths
        """
        raise NotImplementedError(
            "CrossmatchAlgorithm must either implement `perform_crossmatch` or overwrite `crossmatch`"
        )

    # pylint: disable=protected-access
    @classmethod
    def validate(cls, left: Catalog, right: Catalog):
        """Validate the metadata and arguments.

        This method will be called **once**, after the algorithm object has
        been initialized, during the lazy construction of the execution graph.
        This can be used to catch simple errors without waiting for an
        expensive ``.compute()`` call.

        This must accept any additional arguments the `crossmatch` method accepts.
        """
        # Check that we have the appropriate columns in our dataset.
        if left._ddf.index.name != SPATIAL_INDEX_COLUMN:
            raise ValueError(f"index of left table must be {SPATIAL_INDEX_COLUMN}")
        if right._ddf.index.name != SPATIAL_INDEX_COLUMN:
            raise ValueError(f"index of right table must be {SPATIAL_INDEX_COLUMN}")
        column_names = left._ddf.columns
        if left.hc_structure.catalog_info.ra_column not in column_names:
            raise ValueError(f"left table must have column {left.hc_structure.catalog_info.ra_column}")
        if left.hc_structure.catalog_info.dec_column not in column_names:
            raise ValueError(f"left table must have column {left.hc_structure.catalog_info.dec_column}")

        column_names = right._ddf.columns
        if right.hc_structure.catalog_info.ra_column not in column_names:
            raise ValueError(f"right table must have column {right.hc_structure.catalog_info.ra_column}")
        if right.hc_structure.catalog_info.dec_column not in column_names:
            raise ValueError(f"right table must have column {right.hc_structure.catalog_info.dec_column}")

    @staticmethod
    def _rename_columns_with_suffix(dataframe, suffix):
        columns_renamed = {name: name + suffix for name in dataframe.columns}
        dataframe.rename(columns=columns_renamed, inplace=True)

    @classmethod
    def _append_extra_columns(cls, dataframe: npd.NestedFrame, extra_columns: pd.DataFrame | None = None):
        """Adds crossmatch extra columns to the resulting Dataframe."""
        if cls.extra_columns is None:
            return
        if extra_columns is None:
            raise ValueError("No extra column values were provided")
        # Check if the provided columns are in the specification
        for col in extra_columns.columns:
            if col not in cls.extra_columns.columns:
                raise ValueError(f"Provided extra column '{col}' not found in definition")
        # Update columns according to crossmatch algorithm specification
        columns_to_update = []
        for col, col_type in cls.extra_columns.dtypes.items():
            if col not in extra_columns:
                raise ValueError(f"Missing extra column '{col} of type {col_type}'")
            if col_type != extra_columns[col].dtype:
                raise ValueError(f"Invalid type '{col_type}' for extra column '{col}'")
            columns_to_update.append(col)
        for col in columns_to_update:
            new_col = extra_columns[col]
            new_col.index = dataframe.index
            dataframe[col] = new_col

    def _create_crossmatch_df(
        self,
        left_idx: npt.NDArray[np.int64],
        right_idx: npt.NDArray[np.int64],
        extra_cols: pd.DataFrame,
    ) -> npd.NestedFrame:
        """Creates a df containing the crossmatch result from matching indices and additional columns

        Args:
            left_idx (np.ndarray): indices of the matching rows from the left table
            right_idx (np.ndarray): indices of the matching rows from the right table
            extra_cols (pd.DataFrame): dataframe containing additional columns from crossmatching

        Returns:
            A dataframe with the matching rows from the left and right table concatenated together, with the
            additional columns added
        """
        # rename columns so no same names during merging
        self._rename_columns_with_suffix(self.left, self.suffixes[0])
        self._rename_columns_with_suffix(self.right, self.suffixes[1])
        # concat dataframes together
        self.left.index.name = SPATIAL_INDEX_COLUMN
        left_join_part = self.left.iloc[left_idx].reset_index()
        right_join_part = self.right.iloc[right_idx].reset_index(drop=True)
        out = pd.concat(
            [
                left_join_part,
                right_join_part,
            ],
            axis=1,
        )
        out.set_index(SPATIAL_INDEX_COLUMN, inplace=True)
        extra_cols.index = out.index
        self._append_extra_columns(out, extra_cols)
        return npd.NestedFrame(out)
