import logging
import redis

from __init__ import __version__


class Locli:
    def __init__(
        self,
        key=None,
        handler=None,
        host="localhost",
        port=6379,
        db=1,
        sleep_time=0.1,
    ):
        logging.debug(f"{Locli.__name__=}, {__version__=}")

        self._redis = redis.Redis(host=host, port=port, db=db)

        if None in (key, handler):
            logging.debug("setting: key or handler is None!")
            return

        self._user_handler = handler
        self._pubsub = self._redis.pubsub(ignore_subscribe_messages=True)
        pkey = key + "*"
        self._pubsub.psubscribe(**{pkey: self._user_handler})
        self._event_thread = self._pubsub.run_in_thread(sleep_time=sleep_time)

        self._get_to_pub(key, pkey)

    def __del__(self):
        if hasattr(self, "_event_thread"):
            self._event_thread.stop()
        # self._pubsub.close()

    def _get_to_pub(self, key, pkey):
        _, keys = self._redis.scan(
            cursor=0,
            match=key + "*",
            count=1000,
        )

        for key_search in keys:
            data = self._redis.hgetall(key_search)
            for k, v in data.items():
                message = {
                    "type": "init",
                    "pattern": pkey.encode(),
                    "channel": (
                        key_search.decode() + "/" + k.decode()
                    ).encode(),
                    "data": v,
                }
                self._user_handler(message)

    def put(self, key, field, value=None):
        with self._redis.pipeline() as p:
            if value:
                p.hset(key, field, value)
                p.publish(key + "/" + field, str(value))
            else:
                p.hdel(key, field)
                p.publish(key + "/" + field, "")
            p.execute()


if __name__ == "__main__":

    def handler(msg):
        logging.info(msg)

    locli = Locli(key="usbman", handler=handler)
    locli._event_thread.join()
