"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const os = require("os");
const path = require("path");
const sinon = require("sinon");
const lib_1 = require("../lib");
const STUB_INPUT_FILE = '/tmp/docker-stub.input';
var DockerStubCommand;
(function (DockerStubCommand) {
    DockerStubCommand["SUCCESS"] = "DOCKER_STUB_SUCCESS";
    DockerStubCommand["FAIL"] = "DOCKER_STUB_FAIL";
    DockerStubCommand["SUCCESS_NO_OUTPUT"] = "DOCKER_STUB_SUCCESS_NO_OUTPUT";
})(DockerStubCommand || (DockerStubCommand = {}));
const userInfo = os.userInfo();
const USER_ARG = `-u ${userInfo.uid}:${userInfo.gid}`;
// this is a way to provide a custom "docker" command for staging.
process.env.CDK_DOCKER = `${__dirname}/docker-stub.sh`;
function readDockerStubInput() {
    const out = fs.readFileSync(STUB_INPUT_FILE, 'utf-8').trim();
    return out
        .replace(/-v ([^:]+):\/asset-input/, '-v /input:/asset-input')
        .replace(/-v ([^:]+):\/asset-output/, '-v /output:/asset-output');
}
module.exports = {
    'tearDown'(cb) {
        if (fs.existsSync(STUB_INPUT_FILE)) {
            fs.unlinkSync(STUB_INPUT_FILE);
        }
        cb();
        sinon.restore();
    },
    'base case'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), 'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.done();
    },
    'staging can be disabled through context'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.node.setContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT, true);
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), sourcePath);
        test.done();
    },
    'files are copied to the output directory during synth'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const file = path.join(__dirname, 'fs', 'fixtures.tar.gz');
        // WHEN
        new lib_1.AssetStaging(stack, 's1', { sourcePath: directory });
        new lib_1.AssetStaging(stack, 'file', { sourcePath: file });
        // THEN
        const assembly = app.synth();
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00',
            'asset.af10ac04b3b607b0f8659c8f0cee8c343025ee75baf0b146f10f0e5311d2c46b.gz',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'allow specifying extra data to include in the source hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const withoutExtra = new lib_1.AssetStaging(stack, 'withoutExtra', { sourcePath: directory });
        const withExtra = new lib_1.AssetStaging(stack, 'withExtra', { sourcePath: directory, extraHash: 'boom' });
        // THEN
        test.notEqual(withoutExtra.sourceHash, withExtra.sourceHash);
        test.deepEqual(withoutExtra.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(withExtra.sourceHash, 'c95c915a5722bb9019e2c725d11868e5a619b55f36172f76bcbcaa8bb2d10c5f');
        test.done();
    },
    'with bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const ensureDirSyncSpy = sinon.spy(fs, 'ensureDirSync');
        const mkdtempSyncSpy = sinon.spy(fs, 'mkdtempSync');
        const chmodSyncSpy = sinon.spy(fs, 'chmodSync');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        // asset is bundled in a directory inside .cdk.staging
        const stagingTmp = path.join('.', '.cdk.staging');
        test.ok(ensureDirSyncSpy.calledWith(stagingTmp));
        test.ok(mkdtempSyncSpy.calledWith(sinon.match(path.join(stagingTmp, 'asset-bundle-'))));
        test.ok(chmodSyncSpy.calledWith(sinon.match(path.join(stagingTmp, 'asset-bundle-')), 0o777));
        test.done();
    },
    'bundling throws when /asset-ouput is empty'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS_NO_OUTPUT],
            },
        }), /Bundling did not produce any output/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS_NO_OUTPUT`);
        test.done();
    },
    'bundling with BUNDLE asset hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHashType: lib_1.AssetHashType.BUNDLE,
        });
        // THEN
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.equal(asset.assetHash, '33cbf2cae5432438e0f046bc45ba8c3cef7b6afcf47b59d1c183775c1918fb1f');
        test.done();
    },
    'custom hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHash: 'my-custom-hash',
        });
        // THEN
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.equal(asset.assetHash, 'b9c77053f5b83bbe5ba343bc18e92db939a49017010813225fea91fa892c4823'); // hash of 'my-custom-hash'
        test.done();
    },
    'throws with assetHash and not CUSTOM hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHash: 'my-custom-hash',
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot specify `bundle` for `assetHashType`/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.done();
    },
    'throws with BUNDLE hash type and no bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.done();
    },
    'throws with CUSTOM and no hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.CUSTOM,
        }), /`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false); // "docker" not executed
        test.done();
    },
    'throws when bundling fails'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('this-is-an-invalid-docker-image'),
                command: [DockerStubCommand.FAIL],
            },
        }), /Failed to run bundling Docker image for asset stack\/Asset/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input this-is-an-invalid-docker-image DOCKER_STUB_FAIL`);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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