"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckOpenApiProps = CheckOpenApiProps;
exports.ObtainApiDefinition = ObtainApiDefinition;
exports.MapApiIntegrationsToApiFunction = MapApiIntegrationsToApiFunction;
exports.ExtractFunctionInterface = ExtractFunctionInterface;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
/*
 * This file is core openapi functionality and should ideally be in the core library. Since
 * that causes a circular reference with the resources library we have left it here for now
 * in the interest of getting these updates out faster
 */
const apigateway = require("aws-cdk-lib/aws-apigateway");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const resources = require("@aws-solutions-constructs/resources");
const defaults = require("@aws-solutions-constructs/core");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CheckOpenApiProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if ((props.apiDefinitionBucket && !props.apiDefinitionKey) || (!props.apiDefinitionBucket && props.apiDefinitionKey)) {
        errorMessages += 'apiDefinitionBucket and apiDefinitionKey must be specified together.\n';
        errorFound = true;
    }
    const definitionCount = (props.apiDefinitionAsset ? 1 : 0) +
        (props.apiDefinitionBucket ? 1 : 0) +
        (props.apiDefinitionJson ? 1 : 0);
    if (definitionCount !== 1) {
        errorMessages += 'Exactly one of apiDefinitionAsset, apiDefinitionJson or (apiDefinitionBucket/apiDefinitionKey) must be provided\n';
        errorFound = true;
    }
    if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
        errorMessages += 'At least one ApiIntegration must be specified in the apiIntegrations property\n';
        errorFound = true;
    }
    else {
        props.apiIntegrations.forEach((apiIntegration) => {
            if (!apiIntegration.id) {
                errorMessages += 'Each ApiIntegration must have a non-empty id property\n';
                errorFound = true;
            }
            let functionDefCount = 0;
            if (apiIntegration.lambdaFunctionProps) {
                functionDefCount++;
            }
            if (apiIntegration.existingLambdaObj) {
                functionDefCount++;
            }
            if (functionDefCount !== 1) {
                errorMessages += `ApiIntegration id:${apiIntegration.id} must have exactly one of lambdaFunctionProps or existingLambdaObj\n`;
                errorFound = true;
            }
        });
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function ObtainApiDefinition(scope, props) {
    const apiRawInlineSpec = props.apiJsonDefinition;
    const meldedDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
    const meldedDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
    // Map each id and lambda function pair to the required format for the template writer custom resource
    const apiIntegrationUris = props.tokenToFunctionMap.map(apiLambdaFunction => {
        // the placeholder string that will be replaced in the OpenAPI Definition
        const uriPlaceholderString = apiLambdaFunction.id;
        // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        // We know that either functionAlias or lambdaFunction must be defined, so we can use ! to satisfy Typescript
        const targetArn = apiLambdaFunction.functionAlias ? apiLambdaFunction.functionAlias.functionArn : apiLambdaFunction.lambdaFunction.functionArn;
        const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${targetArn}/invocations`;
        return {
            id: uriPlaceholderString,
            value: uriResolvedValue
        };
    });
    let apiDefinitionWriter;
    let newApiDefinition;
    if (props.apiDefinitionAsset || props.apiDefinitionBucket) {
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        apiDefinitionWriter = resources.createTemplateWriterCustomResource(scope, 'Api', {
            // CheckOpenapiProps() has confirmed the existence of these values
            templateBucket: meldedDefinitionBucket,
            templateKey: meldedDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? aws_cdk_lib_1.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        newApiDefinition = apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key);
    }
    else if (apiRawInlineSpec) {
        newApiDefinition = InlineTemplateWriter(apiRawInlineSpec, apiIntegrationUris);
    }
    else {
        throw new Error("No definition provided (this code should be unreachable)");
    }
    return newApiDefinition;
}
function InlineTemplateWriter(rawInlineSpec, templateValues) {
    let template = JSON.stringify(rawInlineSpec);
    // This replicates logic in the template writer custom resource (resources/lib/template-writer-custom-resource/index.ts),
    // any logic changes should be made to both locations every time
    templateValues.forEach((templateValue) => {
        template = template?.replace(new RegExp(templateValue.id, 'g'), templateValue.value);
    });
    return new apigateway.InlineApiDefinition(JSON.parse(template));
}
function MapApiIntegrationsToApiFunction(scope, apiIntegrations) {
    // store a counter to be able to uniquely name lambda functions to avoid naming collisions
    let lambdaCounter = 0;
    return apiIntegrations.map(rawApiIntegration => {
        if (rawApiIntegration.existingLambdaObj && isResourceAnAlias(rawApiIntegration.existingLambdaObj)) {
            return {
                id: rawApiIntegration.id,
                functionAlias: rawApiIntegration.existingLambdaObj
            };
        }
        else {
            return {
                id: rawApiIntegration.id,
                lambdaFunction: defaults.buildLambdaFunction(scope, {
                    existingLambdaObj: rawApiIntegration.existingLambdaObj,
                    lambdaFunctionProps: rawApiIntegration.lambdaFunctionProps
                }, `${rawApiIntegration.id}ApiFunction${lambdaCounter++}`),
            };
        }
    });
}
function ExtractFunctionInterface(apiFunction) {
    return (apiFunction.lambdaFunction ?? apiFunction.functionAlias);
}
function isResourceAnAlias(lambdaResource) {
    return lambdaResource.aliasName !== undefined;
}
//# sourceMappingURL=data:application/json;base64,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