"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, poetryIncludeHashes, commandHooks, assetExcludes = [], } = props;
        const outputPath = path.posix.join(aws_cdk_lib_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: aws_cdk_lib_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
            poetryIncludeHashes,
            commandHooks,
            assetExcludes,
        });
        this.image = image ?? aws_cdk_lib_1.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = props.command ?? ['bash', '-c', chain(bundlingCommands)];
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
    }
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry, options.poetryIncludeHashes);
        let bundlingCommands = [];
        bundlingCommands.push(...options.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []);
        const exclusionStr = options.assetExcludes?.map(item => `--exclude='${item}'`).join(' ');
        bundlingCommands.push([
            'rsync', '-rLv', exclusionStr ?? '', `${options.inputDir}/`, options.outputDir,
        ].filter(item => item).join(' '));
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(...options.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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