from typing import Any, Tuple

import etrobo_python

from . import connector


def create_device(device_type: str, port: str) -> Any:
    if device_type == 'hub':
        return Hub()
    elif device_type == 'motor':
        return Motor(get_raspike_port(port))
    elif device_type == 'reversed_motor':
        return ReversedMotor(get_raspike_port(port))
    elif device_type == 'color_sensor':
        return ColorSensor()
    elif device_type == 'touch_sensor':
        return TouchSensor()
    elif device_type == 'sonar_sensor':
        return SonarSensor()
    elif device_type == 'gyro_sensor':
        return GyroSensor()
    else:
        raise NotImplementedError(f'Unsupported device: {device_type}')


def get_raspike_port(port: str) -> int:
    port_names = ('A', 'B', 'C', 'D', '1', '2', '3', '4')
    port_values = (0, 1, 2, -1, -1, -1, -1, -1)

    if port in port_names:
        return port_values[port_names.index(port)]
    else:
        raise Exception(f'Unknown port: {port}')


class Hub(etrobo_python.Hub):
    def __init__(self):
        self.hub = connector.Hub()
        self.log = bytearray(5)

    def set_led(self, color: str) -> None:
        if color.startswith('bla'):  # black
            color_code = 0
        elif color.startswith('p'):  # pink
            color_code = 1
        elif color.startswith('blu'):  # blue
            color_code = 3
        elif color.startswith('g'):  # green
            color_code = 6
        elif color.startswith('y'):  # yellow
            color_code = 7
        elif color.startswith('o'):  # orange
            color_code = 8
        elif color.startswith('r'):  # red
            color_code = 9
        else:
            return

        self.hub.set_led(color_code)

    def get_time(self) -> float:
        return self.hub.get_time() / 1000

    def get_battery_voltage(self) -> int:
        return self.hub.get_battery_voltage()

    def get_battery_current(self) -> int:
        return self.hub.get_battery_current()

    def play_speaker_tone(self, frequency: int, duration: float) -> None:
        self.hub.play_speaker_tone(frequency, int(duration * 1000))

    def set_speaker_volume(self, volume: int) -> None:
        self.hub.set_speaker_volume(min(max(volume // 10, 0), 10))

    def is_left_button_pressed(self) -> bool:
        return (self.hub.get_button_pressed() & 0x02) != 0

    def is_right_button_pressed(self) -> bool:
        return (self.hub.get_button_pressed() & 0x04) != 0

    def is_up_button_pressed(self) -> bool:
        return False

    def is_down_button_pressed(self) -> bool:
        return False

    def get_log(self) -> bytes:
        self.log[:4] = int.to_bytes(int(self.get_time() * 1000), 4, 'big')
        self.log[4] = (
            int(self.is_left_button_pressed())
            | int(self.is_right_button_pressed()) << 1
        )

        return self.log


class _Motor(etrobo_python.Motor):
    def __init__(self, port: int, reversed: bool) -> None:
        self.motor = connector.Motor(port)
        self.sign = -1 if reversed else 1
        self.log = bytearray(4)

    def get_count(self) -> int:
        return self.motor.get_count() * self.sign

    def reset_count(self) -> None:
        self.motor.reset_count()

    def set_power(self, power: int) -> None:
        self.motor.set_pwm(power * self.sign)

    def set_brake(self, brake: bool) -> None:
        self.motor.set_brake(brake)

    def get_log(self) -> bytes:
        self.log[:] = int.to_bytes(self.get_count(), 4, 'big', signed=True)
        return self.log


class Motor(_Motor):
    def __init__(self, port: int) -> None:
        super().__init__(port, False)


class ReversedMotor(_Motor):
    def __init__(self, port: int) -> None:
        super().__init__(port, True)


class ColorSensor(etrobo_python.ColorSensor):
    def __init__(self) -> None:
        self.color_sensor = connector.ColorSensor()
        self.mode = -1
        self.log = bytearray(5)

    def get_brightness(self) -> int:
        self.mode = 0
        return self.color_sensor.get_brightness()

    def get_ambient(self) -> int:
        self.mode = 1
        return self.color_sensor.get_ambient()

    def get_raw_color(self) -> Tuple[int, int, int]:
        self.mode = 2
        return self.color_sensor.get_raw_color()

    def get_log(self) -> bytes:
        if self.mode == 0:
            self.log[0] = self.get_brightness()
            self.log[1] = 0
            self.log[2], self.log[3], self.log[4] = 0, 0, 0
        elif self.mode == 1:
            self.log[0] = 0
            self.log[1] = self.get_ambient()
            self.log[2], self.log[3], self.log[4] = 0, 0, 0
        elif self.mode == 2:
            self.log[0] = 0
            self.log[1] = 0
            self.log[2], self.log[3], self.log[4] = self.get_raw_color()

        return self.log


class TouchSensor(etrobo_python.TouchSensor):
    def __init__(self):
        self.hub = connector.Hub()
        self.log = bytearray(1)

    def is_pressed(self) -> bool:
        return (self.hub.get_button_pressed() & 0x01) != 0

    def get_log(self) -> bytes:
        self.log[0] = int(self.is_pressed())
        return self.log


class SonarSensor(etrobo_python.SonarSensor):
    def __init__(self) -> None:
        self.sonar_sensor = connector.SonarSensor()
        self.log = bytearray(2)

    def listen(self) -> bool:
        return self.sonar_sensor.listen()

    def get_distance(self) -> int:
        return self.sonar_sensor.get_distance()

    def get_log(self) -> bytes:
        self.log[:] = int.to_bytes(self.get_distance(), 2, 'big')
        return self.log


class GyroSensor(etrobo_python.GyroSensor):
    def __init__(self) -> None:
        self.gyro_sensor = connector.GyroSensor()
        self.log = bytearray(4)

    def reset(self) -> None:
        self.gyro_sensor.reset()

    def get_angle(self) -> int:
        return self.gyro_sensor.get_angle()

    def get_angler_velocity(self) -> int:
        return self.gyro_sensor.get_angler_velocity()

    def get_log(self) -> bytes:
        self.log[:2] = int.to_bytes(self.get_angle(), 2, 'big', signed=True)
        self.log[2:] = int.to_bytes(self.get_angler_velocity(), 2, 'big', signed=True)
        return self.log
