"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createBackupConfig = exports.createProcessingConfig = exports.createEncryptionConfig = exports.createBufferingHints = exports.createLoggingOptions = void 0;
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const constructs_1 = require("constructs");
function createLoggingOptions(scope, props) {
    var _a, _b;
    if (props.logging === false && props.logGroup) {
        throw new Error('logging cannot be set to false when logGroup is provided');
    }
    if (props.logging !== false || props.logGroup) {
        const logGroup = (_b = (_a = props.logGroup) !== null && _a !== void 0 ? _a : constructs_1.Node.of(scope).tryFindChild('LogGroup')) !== null && _b !== void 0 ? _b : new logs.LogGroup(scope, 'LogGroup');
        const logGroupGrant = logGroup.grantWrite(props.role);
        return {
            loggingOptions: {
                enabled: true,
                logGroupName: logGroup.logGroupName,
                logStreamName: logGroup.addStream(props.streamId).logStreamName,
            },
            dependables: [logGroupGrant],
        };
    }
    return undefined;
}
exports.createLoggingOptions = createLoggingOptions;
function createBufferingHints(interval, size) {
    var _a, _b;
    if (!interval && !size) {
        return undefined;
    }
    const intervalInSeconds = (_a = interval === null || interval === void 0 ? void 0 : interval.toSeconds()) !== null && _a !== void 0 ? _a : 300;
    if (intervalInSeconds < 60 || intervalInSeconds > 900) {
        throw new Error(`Buffering interval must be between 60 and 900 seconds. Buffering interval provided was ${intervalInSeconds} seconds.`);
    }
    const sizeInMBs = (_b = size === null || size === void 0 ? void 0 : size.toMebibytes()) !== null && _b !== void 0 ? _b : 5;
    if (sizeInMBs < 1 || sizeInMBs > 128) {
        throw new Error(`Buffering size must be between 1 and 128 MiBs. Buffering size provided was ${sizeInMBs} MiBs.`);
    }
    return { intervalInSeconds, sizeInMBs };
}
exports.createBufferingHints = createBufferingHints;
function createEncryptionConfig(role, encryptionKey) {
    encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.grantEncryptDecrypt(role);
    return encryptionKey
        ? { kmsEncryptionConfig: { awskmsKeyArn: encryptionKey.keyArn } }
        : undefined;
}
exports.createEncryptionConfig = createEncryptionConfig;
function createProcessingConfig(scope, role, dataProcessor) {
    return dataProcessor
        ? {
            enabled: true,
            processors: [renderDataProcessor(dataProcessor, scope, role)],
        }
        : undefined;
}
exports.createProcessingConfig = createProcessingConfig;
function renderDataProcessor(processor, scope, role) {
    const processorConfig = processor.bind(scope, { role });
    const parameters = [{ parameterName: 'RoleArn', parameterValue: role.roleArn }];
    parameters.push(processorConfig.processorIdentifier);
    if (processor.props.bufferInterval) {
        parameters.push({ parameterName: 'BufferIntervalInSeconds', parameterValue: processor.props.bufferInterval.toSeconds().toString() });
    }
    if (processor.props.bufferSize) {
        parameters.push({ parameterName: 'BufferSizeInMBs', parameterValue: processor.props.bufferSize.toMebibytes().toString() });
    }
    if (processor.props.retries) {
        parameters.push({ parameterName: 'NumberOfRetries', parameterValue: processor.props.retries.toString() });
    }
    return {
        type: processorConfig.processorType,
        parameters,
    };
}
function createBackupConfig(scope, role, props) {
    var _a, _b, _c;
    if (!props || (props.mode === undefined && !props.bucket)) {
        return undefined;
    }
    const bucket = (_a = props.bucket) !== null && _a !== void 0 ? _a : new s3.Bucket(scope, 'BackupBucket');
    const bucketGrant = bucket.grantReadWrite(role);
    const { loggingOptions, dependables: loggingDependables } = (_b = createLoggingOptions(scope, {
        logging: props.logging,
        logGroup: props.logGroup,
        role,
        streamId: 'S3Backup',
    })) !== null && _b !== void 0 ? _b : {};
    return {
        backupConfig: {
            bucketArn: bucket.bucketArn,
            roleArn: role.roleArn,
            prefix: props.dataOutputPrefix,
            errorOutputPrefix: props.errorOutputPrefix,
            bufferingHints: createBufferingHints(props.bufferingInterval, props.bufferingSize),
            compressionFormat: (_c = props.compression) === null || _c === void 0 ? void 0 : _c.value,
            encryptionConfiguration: createEncryptionConfig(role, props.encryptionKey),
            cloudWatchLoggingOptions: loggingOptions,
        },
        dependables: [bucketGrant, ...(loggingDependables !== null && loggingDependables !== void 0 ? loggingDependables : [])],
    };
}
exports.createBackupConfig = createBackupConfig;
//# sourceMappingURL=data:application/json;base64,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