import subprocess
import time
import os
from typing import List, Optional


class MalwareLauncher:
    """
    Class to launch malware executables.

    This class provides a simple interface to run malware executables, killing the process if it runs for too long.

    Attributes:
        filepath (str): The path to the malware executable.
        timeout (float): The time in seconds before the malware process is forcefully terminated.

    Methods:
        run(): Launches the malware executable and waits for it to complete, with a timeout specified by the `timeout`
        attribute. Returns `True` if the process exited on its own, and `False` if it had to be forcefully terminated.

    Example:
        >>> launcher = MalwareLauncher('samples/malware1', timeout=1)
        >>> launcher.run() # Returns True is malware executed successfully.

        False

    Todo:
        Make `Malware` its own internal class.
    """

    def __init__(self, filepath: str, timeout: float = 2.0):
        """
        Initializes a `MalwareLauncher` object.

        Args:
            filepath (str): The path to the malware executable.
            timeout (float): The time in seconds before the malware process is forcefully terminated.
        """

        self.filepath: str = filepath
        self.timeout: float = timeout
        self.name: str = os.path.basename(filepath)
        self.detected: Optional[bool] = None

    def run(self) -> bool:
        """
        Launches the malware executable and waits for it to complete.

        The method launches the malware executable specified by the `filepath` attribute and waits for it to complete,
        with a timeout specified by the `timeout` attribute.

        Returns:
            bool: `True` if the malware was detected, `False` if it went undetected.
        """

        if os.name == "nt":
            process = subprocess.Popen(self.filepath, creationflags=subprocess.CREATE_NEW_PROCESS_GROUP |
                                       subprocess.CREATE_BREAKAWAY_FROM_JOB)
        else:
            process = subprocess.Popen(self.filepath)

        timestamp = time.monotonic()
        while process.poll() is None:
            # If process is running for more than two seconds, kill the process and mark it as detected.
            if time.monotonic() - timestamp >= self.timeout:
                process.kill()

                self.detected = True
                return True

            time.sleep(.1)

        # If the process finished without error, assume it completed without detection. Otherwise assume it was detected.
        if process.poll() == 0:
            self.detected = False
            return False
        else:
            self.detected = True
            return True


class MalwareMetrics:
    """
    Class to calculate metrics on a list of MalwareLauncher objects.

    This class provides methods to calculate detection rate and list failed samples from a list of MalwareLauncher objects.

    Attributes:
        samples (List[MalwareLauncher]): A list of MalwareLauncher objects.

    Methods:
        detection_rate(): Calculates the detection rate of the samples.
        failed_samples(): Returns a list of MalwareLauncher objects that were not detected.

    Example:
        >>> malware1 = MalwareLauncher('samples/malware1', timeout=1)
        >>> malware2 = MalwareLauncher('samples/malware2', timeout=1)
        >>> metrics = MalwareMetrics([malware1, malware2])
        >>> metrics.detection_rate() # Returns the detection rate of the samples.

        0.5
    """

    def __init__(self, samples: List[MalwareLauncher]):
        """
        Initializes a `MalwareMetrics` object.

        Args:
            samples (List[MalwareLauncher]): A list of MalwareLauncher objects.
        """

        self.samples: List[MalwareLauncher] = samples

    def detection_rate(self) -> float:
        """
        Calculates the detection rate of the samples.

        Returns:
            float: The detection rate of the samples.
        """

        failed = 0

        for sample in self.samples:
            if sample.detected is True:
                failed += 1

        if failed == 0:
            return 0

        return failed / len(self.samples)

    def failed_samples(self) -> List[MalwareLauncher]:
        """
        Returns a list of MalwareLauncher objects that were not detected.

        Returns:
            List[MalwareLauncher]: A list of MalwareLauncher objects that were not detected.
        """

        samples = []

        for sample in self.samples:
            if sample.detected is False:
                samples.append(sample)

        return samples
