<%def
  name="malware_scan_step(job_step, job_variant, cfg_set, indent)",
  filter="indent_func(indent),trim"
>
<%
from makoutil import indent_func
from concourse.steps import step_lib
import dataclasses
import ci.util
main_repo = job_variant.main_repository()
repo_name = main_repo.logical_name().upper()

image_scan_trait = job_variant.trait('image_scan')
filter_cfg = image_scan_trait.matching_config()
clam_av = image_scan_trait.clam_av()
age_threshold = clam_av.virus_db_max_age_days()
aws_cfg_name = clam_av.aws_cfg_name()

issue_tgt_repo_url = image_scan_trait.overwrite_github_issues_tgt_repository_url()
if issue_tgt_repo_url:
  parsed_repo_url = ci.util.urlparse(issue_tgt_repo_url)
  tgt_repo_org, tgt_repo_name = parsed_repo_url.path.strip('/').split('/')

github_issue_templates = image_scan_trait.github_issue_templates()
github_issue_labels_to_preserve = image_scan_trait.github_issue_labels_to_preserve()
issue_policies = image_scan_trait.issue_policies()

component_trait = job_variant.trait('component_descriptor')
ocm_repository_mappings = component_trait.ocm_repository_mappings()
component_descriptor_dir = job_step.input('component_descriptor_dir')
%>
import dacite
import datetime
import enum
import functools
import logging
import os
import sys
import tabulate
import textwrap
import uuid

import requests.exceptions

import gci.componentmodel as cm

logger = logging.getLogger('malware_scan.step')

# debugging (dump stacktrace on error-signals)
import faulthandler
faulthandler.enable() # print stacktraces upon fatal signals
# end of debugging block

import ccc.aws
import ccc.clamav
import ccc.delivery
import ccc.github
import ccc.oci
import ci.log
import ci.util
import clamav.cnudie
import clamav.model
import clamav.report
import cnudie.iter
import cnudie.retrieve
import concourse.model.traits.filter
import dso.model
import github.compliance.model as gcm
import github.compliance.report
import mailutil

ci.log.configure_default_logging()

from concourse.model.traits.image_scan import Notify

${step_lib('scan_container_images')}
${step_lib('component_descriptor_util')}

<%
import concourse.steps
template = concourse.steps.step_template('component_descriptor')
ocm_repository_lookup = template.get_def('ocm_repository_lookup').render
%>
${ocm_repository_lookup(ocm_repository_mappings)}

component_descriptor_lookup = cnudie.retrieve.create_default_component_descriptor_lookup(
    ocm_repository_lookup=cnudie.retrieve.ocm_repository_lookup(
        ocm_repository_lookup,
    ),
)

cfg_factory = ci.util.ctx().cfg_factory()
cfg_set = cfg_factory.cfg_set("${cfg_set.name()}")

component_descriptor = component_descriptor_from_dir(dir_path='${component_descriptor_dir}')
component = component_descriptor.component

matching_configs = concourse.model.traits.filter.matching_configs_from_dicts(
  dicts=${filter_cfg}
)
filter_function = concourse.model.traits.filter.filter_for_matching_configs(
  configs=matching_configs
)

resource_nodes = cnudie.iter.iter(
  component=component,
  lookup=component_descriptor_lookup,
  node_filter=cnudie.iter.Filter.resources,
)

def resource_type_is_tar(resource):
  if isinstance(resource.access, cm.OciAccess):
    return True
  if isinstance(resource.type, enum.Enum):
    rtype = resource.type.value
  else:
    rtype = resource.type
  return rtype.startswith('application/tar')


resource_nodes = (
  node for node in resource_nodes
  if isinstance(node.resource.access, cm.OciAccess) \
  or resource_type_is_tar(node.resource)
)
resource_nodes = (node for node in resource_nodes
  if filter_function(node)
)

clamav_client = ccc.clamav.client(cfg='${clam_av.clamav_cfg_name()}')
oci_client = ccc.oci.oci_client()

logger.info('running malware scan for all container images')

scanned_results = []
skipped_results = []
findings_count = 0
err_count = 0

if (delivery_client := ccc.delivery.default_client_if_available()):
  logger.info('uploading results to deliverydb')
else:
  logger.warning('not uploading results to deliverydb, client not available')

if '${aws_cfg_name}':
  aws_cfg = cfg_set.aws('${aws_cfg_name}')
  aws_session = ccc.aws.session(aws_cfg)
else:
  aws_session = ccc.aws.default_session(cfg_set=cfg_set)

if aws_session:
  s3_client = aws_session.resource('s3')
else:
  s3_client = None

rescoring_raw = [raw for raw in
  ${[dataclasses.asdict(r) for r in clam_av.rescorings()]}
]
rescoring_entries = [
  dacite.from_dict(
    data=raw,
    data_class=image_scan.ClamAVRescoringEntry,
    config=dacite.Config(
      type_hooks={gcm.Severity: gcm.Severity.parse},
    ),
  ) for raw in rescoring_raw
]


def filter_findings_rescored_to_none(scan_result):
  for result in scan_result.findings:
    for entry in rescoring_entries:
      if not entry.digest == result.meta.scanned_content_digest:
        continue
      if not entry.malware_name.lower() in result.details.lower():
        continue

      # rescore
      if entry.severity is gcm.Severity.NONE:
        break
    else:
      # no rescoring -> keep finding
      yield result


# Note: this assumes that all k8s-pods have the same clamav-
# version and that the version does not change during the scan.
clamav_version_info = clamav_client.clamav_version_info()

for result in clamav.cnudie.scan_resources(
  resource_nodes=resource_nodes,
  oci_client=oci_client,
  clamav_client=clamav_client,
  delivery_client=delivery_client,
  virus_db_max_age_days=${age_threshold},
  clamav_version_info=clamav_version_info,
  s3_client=s3_client,
  max_workers=${clam_av.parallel_jobs()},
):

  if delivery_client and result.state is not gcm.ScanState.SKIPPED:
    if not result.severity is gcm.Severity.NONE:
      logger.info(f'checking for rescoring for {result.scan_result.name}')
      result.scan_result.findings = list(filter_findings_rescored_to_none(result.scan_result))

      if not result.scan_result.findings:
        result.scan_result.malware_status = clamav.model.MalwareStatus.OK

    findings_data = clamav.cnudie.resource_scan_result_to_artefact_metadata(
        resource_scan_result=result,
        datasource=dso.model.Datasource.CLAMAV,
        datatype=dso.model.Datatype.MALWARE,
    )
    delivery_client.update_metadata(data=[findings_data])

  if (scan_result := result.scan_result):
    scanned_results.append(result)
    logger.info(f'{scan_result}')
    if scan_result.malware_status is clamav.model.MalwareStatus.FOUND_MALWARE:
      findings_count += 1
    elif scan_result.malware_status is clamav.model.MalwareStatus.UNKNOWN:
      err_count += 1
  else:
    skipped_results.append(result)

# order so that "Skipped" and "OK"-results are displayed at the end
results = sorted(scanned_results, key=lambda r: r.scan_result.malware_status, reverse=True)
results.extend(skipped_results)

logger.info(f'summary: {findings_count=} {err_count=}')
logger.info('findings and errors are printed on top of table\n')

% if issue_tgt_repo_url:
gh_api = ccc.github.github_api(repo_url='${issue_tgt_repo_url}')
overwrite_repository = gh_api.repository('${tgt_repo_org}', '${tgt_repo_name}')
% else:
gh_api = None
overwrite_repository = None
% endif


result_group_collection = scan_result_group_collection_for_malware(
  results=results,
  rescoring_entries=rescoring_entries,
)

print('Scan Results (honouring rescorings)')
print(clamav.report.as_table(
  scan_result_group_collection=result_group_collection,
  tablefmt='fancy_grid',
))


max_processing_days = dacite.from_dict(
  data_class=gcm.MaxProcessingTimesDays,
  data=${dataclasses.asdict(issue_policies.max_processing_time_days)},
)

% if github_issue_templates:
github_issue_template_cfgs = [dacite.from_dict(
    data_class=image_scan.GithubIssueTemplateCfg,
    data=raw
    ) for raw in ${[dataclasses.asdict(ghit) for ghit in github_issue_templates]}
]
% endif

github.compliance.report.create_or_update_github_issues(
    result_group_collection=result_group_collection,
    max_processing_days=max_processing_days,
    gh_api=gh_api,
    overwrite_repository=overwrite_repository,
% if github_issue_labels_to_preserve:
    preserve_labels_regexes=${github_issue_labels_to_preserve},
% endif
% if github_issue_templates:
    github_issue_template_cfgs=github_issue_template_cfgs,
% endif
    delivery_svc_client=delivery_client,
    delivery_svc_endpoints=ccc.delivery.endpoints(cfg_set=cfg_set),
)

notification_policy = Notify('${image_scan_trait.notify().value}')


if notification_policy is Notify.NOBODY:
  logger.info("Notification policy set to 'nobody', exiting")
  sys.exit(0)


# component_name identifies the landscape that has been scanned
component_name = "${component_trait.component_name()}"

if result_group_collection.result_groups_with_findings:
  have_findings = True
else:
  have_findings = False

if result_group_collection.result_groups_with_scan_errors:
  have_errors = True
else:
  have_errors = False

if not have_findings and not have_errors:
  subject = f'[CLEAN] - Malware-scan results for landscape {component_name}'
  header = f'No malware was found, successfully scanned {len(results)=} artifacts'
elif not have_findings and have_errors:
  subject = f'[Action Required] {component_name=}: malware-scan-errors'
  header = f'Did not find malware. However, there were {err_count} scan-errors (see table below)'
elif have_findings and not have_errors:
  subject = f'[Action Required] {component_name=}: found malware'
  header = f'Found {findings_count=} resources potentially containing malware (see table below)'
elif have_findings and have_errors:
  subject = 'f[Action Required] {component_name=}: scan-errors and found malware'
  header = f'''Found {findings_count=} reosurces potentially containing malware. Also,
    {err_count} scanning errors occurred. See table below.
  '''

body = f'''
  <em>{header}</em>
  <p>
  {clamav.report.as_table(scan_result_group_collection=result_group_collection, tablefmt='html')}
'''

email_addresses = ${image_scan_trait.email_recipients() or ()}

try:
  mailutil._send_mail(
    email_cfg=cfg_set.email(),
    recipients=email_addresses,
    mail_template=body,
    subject=subject,
    mimetype='html',
  )
  logger.info(f'sent notifications to: {email_addresses=}')
except:
  import traceback
  traceback.print_exc()
  logger.warning(f'error whilst trying to send notification-mails for {component_name=}')
</%def>
