"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../../lib");
const physical_name_generator_1 = require("../../lib/private/physical-name-generator");
class TestResource extends lib_1.Resource {
}
module.exports = nodeunit.testCase({
    generatePhysicalName: {
        'generates correct physical names'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
            const testResourceA = new TestResource(stack, 'A');
            const testResourceB = new TestResource(testResourceA, 'B');
            test.equal(physical_name_generator_1.generatePhysicalName(testResourceA), 'teststackteststackaa164c141d59b37c1b663');
            test.equal(physical_name_generator_1.generatePhysicalName(testResourceB), 'teststackteststackab27595cd34d8188283a1f');
            test.done();
        },
        'generates different names in different accounts'(test) {
            const appA = new lib_1.App();
            const stackA = new lib_1.Stack(appA, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
            const resourceA = new TestResource(stackA, 'Resource');
            const appB = new lib_1.App();
            const stackB = new lib_1.Stack(appB, 'TestStack', { env: { account: '012345678913', region: 'bermuda-triangle-1' } });
            const resourceB = new TestResource(stackB, 'Resource');
            test.notEqual(physical_name_generator_1.generatePhysicalName(resourceA), physical_name_generator_1.generatePhysicalName(resourceB));
            test.done();
        },
        'generates different names in different regions'(test) {
            const appA = new lib_1.App();
            const stackA = new lib_1.Stack(appA, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-1' } });
            const resourceA = new TestResource(stackA, 'Resource');
            const appB = new lib_1.App();
            const stackB = new lib_1.Stack(appB, 'TestStack', { env: { account: '012345678912', region: 'bermuda-triangle-2' } });
            const resourceB = new TestResource(stackB, 'Resource');
            test.notEqual(physical_name_generator_1.generatePhysicalName(resourceA), physical_name_generator_1.generatePhysicalName(resourceB));
            test.done();
        },
        'fails when the region is an unresolved token'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912', region: lib_1.Aws.REGION } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the region is un-resolved or missing/);
            test.done();
        },
        'fails when the region is not provided'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: '012345678912' } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the region is un-resolved or missing/);
            test.done();
        },
        'fails when the account is an unresolved token'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { account: lib_1.Aws.ACCOUNT_ID, region: 'bermuda-triangle-1' } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the account is un-resolved or missing/);
            test.done();
        },
        'fails when the account is not provided'(test) {
            const app = new lib_1.App();
            const stack = new lib_1.Stack(app, 'TestStack', { env: { region: 'bermuda-triangle-1' } });
            const testResource = new TestResource(stack, 'A');
            test.throws(() => physical_name_generator_1.generatePhysicalName(testResource), /Cannot generate a physical name for TestStack\/A, because the account is un-resolved or missing/);
            test.done();
        },
    },
    GeneratedWhenNeededMarker: {
        'is correctly recognized'(test) {
            const marker = new physical_name_generator_1.GeneratedWhenNeededMarker();
            const asString = lib_1.Token.asString(marker);
            test.ok(physical_name_generator_1.isGeneratedWhenNeededMarker(asString));
            test.done();
        },
        'throws when resolved'(test) {
            const marker = new physical_name_generator_1.GeneratedWhenNeededMarker();
            const asString = lib_1.Token.asString(marker);
            test.throws(() => new lib_1.Stack().resolve(asString), /Use "this.physicalName" instead/);
            test.done();
        },
    },
    isGeneratedWhenNeededMarker: {
        'correctly response for other tokens'(test) {
            test.ok(!physical_name_generator_1.isGeneratedWhenNeededMarker('this is not even a token!'));
            test.ok(!physical_name_generator_1.isGeneratedWhenNeededMarker(lib_1.Lazy.stringValue({ produce: () => 'Bazinga!' })));
            test.done();
        }
    },
});
//# sourceMappingURL=data:application/json;base64,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