# pylint: disable=line-too-long, unused-wildcard-import
"""
The GR6J model (modèle du Génie Rural à 6 parametres Journalier) is a daily
lumped six-parameter rainfall-runoff model and belongs to the family of soil moisture
accounting models.  It was published by :cite:t:`ref-Pushpalatha2011` and is a
modification of GR4 and GR5.  Our implementation, |gland_gr6|, follows the one of the R
package airGR :cite:p:`ref-airGR2017` but with a few extensions that we explain in the
documentation of |gland_gr4|, which implements and extends GR4J.

GR6J resembles GR4J in the simulation of runoff concentration (two Unit
Hydrographs instead of one) and GR5J in the calculation of groundwater exchange
(additional parameter `X5`). Its unique feature is an exponential store that is
connected the new parameter `X6`.

The following figure :cite:p:`ref-airGRManual` shows the general structure of
|gland_gr6|:

.. image:: HydPy-G-GR6.png

.. _gland_gr6_integration_tests:

Integration tests
=================

.. how_to_understand_integration_tests::

The settings of the following tests are similar to the ones of the application models
|gland_gr4| and |gland_gr5|.  Hence, comparing the test results gives a good impression
of the functional differences between the three models.

The following settings are identical:

>>> from hydpy import pub
>>> pub.timegrids = "1990-01-01", "1990-02-20", "1d"

>>> from hydpy.models.gland_gr6 import *
>>> parameterstep("1d")
>>> from hydpy import Element
>>> land = Element("land", outlets="outlet")
>>> land.model = model

>>> area(360.0)

>>> with model.add_petmodel_v1("evap_ret_io"):
...     evapotranspirationfactor(1.0)

>>> from hydpy import IntegrationTest
>>> test = IntegrationTest(land)
>>> test.plotting_options.axis1 = inputs.p, fluxes.e, fluxes.qh
>>> test.dateformat = "%d.%m."

>>> inputs.p.series = (
...     0.0, 9.3, 3.2, 7.3, 0.0, 0.0, 0.0, 0.0, 0.1, 0.2, 2.9, 0.2, 0.0, 0.0, 0.0, 3.3,
...     4.6, 0.8, 1.8, 1.1, 0.0, 5.0, 13.1, 14.6, 4.0, 0.8, 0.1, 3.3, 7.7, 10.3, 3.7,
...     15.3, 3.2, 2.7, 2.2, 8.0, 14.3, 6.3, 0.0, 5.9, 9.2, 6.1, 0.1, 0.0, 2.8, 10.6,
...     8.8, 7.2, 4.9, 1.8
... )
>>> model.petmodel.sequences.inputs.referenceevapotranspiration.series = (
...     0.3, 0.4, 0.4, 0.3, 0.1, 0.1, 0.1, 0.2, 0.2, 0.3, 0.3, 0.2, 0.2, 0.2, 0.2, 0.3,
...     0.3, 0.2, 0.2, 0.3, 0.2, 0.2, 0.3, 0.6, 0.4, 0.4, 0.4, 0.5, 0.4, 0.3, 0.3, 0.5,
...     0.5, 0.3, 0.3, 0.4, 0.4, 0.3, 0.2, 0.1, 0.1, 0.0, 0.1, 0.1, 0.0, 0.2, 0.9, 0.9,
...     0.5, 0.9
... )


.. _gland_gr6_base_example:

base example
____________

The following parameter values do not agree with the ones of the
:ref:`gland_gr4_base_example` on application model |gland_gr4| but with the ones of
the example given by :cite:t:`ref-airGRManual`:

>>> imax(0.0)
>>> x1(242.257)
>>> x2(0.637)
>>> x3(53.517)
>>> x5(0.424)
>>> x6(4.759)

>>> with model.add_rconcmodel_directflow_v1("rconc_uh"):
...     uh("gr_uh2", x4=2.218)
>>> with model.add_rconcmodel_routingstore_v1("rconc_uh"):
...     uh("gr_uh1", x4=2.218)

>>> test.inits = (
...     (states.i, 0.0),
...     (states.s, 0.3 * x1),
...     (states.r, 0.5 * x3),
...     (states.r2, 0),
...     (model.rconcmodel_routingstore.sequences.logs.quh, 0.0),
...     (model.rconcmodel_directflow.sequences.logs.quh, 0.0),
... )

.. integration-test::

    >>> conditions = test("gland_gr6_base_example", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |       fd |       fr |      fr2 |       qr |      qr2 |       qd |       qh |        qv |   i |          s |         r |         r2 |    outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 | 0.152867 | 0.152867 | 0.005681 | 0.005113 | 0.000568 | 0.005681 | 0.000698 | 0.000039 | 0.048412 | 0.048412 | 0.048412 | 0.406074 | 3.323095 | 0.048451 |  3.77762 | 15.740083 | 0.0 |  72.518551 | 26.401257 |  -3.274404 | 15.740083 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  8.9 |   8.01079 | 0.4 |      0.0 |      0.4 | 0.898798 | 0.808918 |  0.08988 | 0.009588 | 0.113658 | 0.006314 |  0.04416 |  0.04416 |  0.04416 | 0.384929 | 1.967881 | 0.050474 | 2.403284 | 10.013685 | 0.0 |  80.519753 | 26.128682 |  -5.152662 | 10.013685 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  2.8 |  2.481037 | 0.4 |      0.0 |      0.4 | 0.330116 | 0.297104 | 0.033012 | 0.011152 | 0.555871 |  0.03107 | 0.040915 | 0.040915 | 0.040915 |  0.38419 | 1.456093 | 0.071985 | 1.912268 |  7.967783 | 0.0 |  82.989638 |  26.11893 |  -6.345491 |  7.967783 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |  6.116283 | 0.3 |      0.0 |      0.3 | 0.899618 | 0.809657 | 0.089962 | 0.015902 | 0.483706 | 0.056746 | 0.040799 | 0.040799 | 0.040799 | 0.380461 | 1.163228 | 0.097546 | 1.641234 |  6.838476 | 0.0 |  89.090019 | 26.069493 |  -7.274437 |  6.838476 |
    | 05.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 |  0.06001 |  0.06001 | 0.015834 | 0.014251 | 0.001583 | 0.015834 |  0.58428 | 0.057791 | 0.040211 | 0.040211 | 0.040211 | 0.381182 | 0.983998 | 0.098002 | 1.463182 |  6.096592 | 0.0 |  89.014175 | 26.079089 |  -7.984512 |  6.096592 |
    | 06.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.059971 | 0.059971 | 0.015767 |  0.01419 | 0.001577 | 0.015767 | 0.195519 | 0.046726 | 0.040325 | 0.040325 | 0.040325 | 0.365784 | 0.833841 | 0.087051 | 1.286676 |  5.361152 | 0.0 |  88.938437 | 25.870942 |  -8.699821 |  5.361152 |
    | 07.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.059931 | 0.059931 |   0.0157 |  0.01413 |  0.00157 |   0.0157 | 0.014196 | 0.015996 | 0.037848 | 0.037848 | 0.037848 |  0.34462 | 0.715344 | 0.053844 | 1.113807 |  4.640864 | 0.0 |  88.862806 | 25.572688 |  -9.371639 |  4.640864 |
    | 08.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.119752 | 0.119752 | 0.015581 | 0.014023 | 0.001558 | 0.015581 | 0.014129 | 0.002331 | 0.034298 | 0.034298 | 0.034298 | 0.325503 | 0.626646 | 0.036629 | 0.988777 |  4.119906 | 0.0 |  88.727474 |  25.28996 |  -9.958335 |  4.119906 |
    | 09.01. |  0.1 | 0.2 | 0.1 |  0.0 |       0.0 | 0.1 | 0.059821 | 0.159821 | 0.015515 | 0.013963 | 0.001551 | 0.015515 | 0.014039 | 0.001566 | 0.030932 | 0.030932 | 0.030932 | 0.308148 | 0.557678 | 0.032498 | 0.898325 |   3.74302 | 0.0 |  88.652138 | 25.021167 | -10.479466 |   3.74302 |
    | 10.01. |  0.2 | 0.3 | 0.1 |  0.0 |       0.0 | 0.2 | 0.059781 | 0.259781 | 0.015449 | 0.013904 | 0.001545 | 0.015449 | 0.013969 | 0.001557 | 0.027733 | 0.027733 | 0.027733 | 0.292324 | 0.502456 |  0.02929 |  0.82407 |  3.433627 | 0.0 |  88.576908 | 24.764957 | -10.948602 |  3.433627 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  2.6 |  2.243524 | 0.3 |      0.0 |      0.3 | 0.373966 |  0.33657 | 0.037397 | 0.017491 | 0.057958 | 0.003997 | 0.024683 | 0.024683 | 0.024683 | 0.279286 | 0.458816 |  0.02868 | 0.766782 |  3.194925 | 0.0 |  90.802941 | 24.545129 | -11.359551 |  3.194925 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.017474 | 0.015727 | 0.001747 | 0.017474 | 0.219241 | 0.012954 | 0.022067 | 0.022067 | 0.022067 |  0.27244 | 0.427782 |  0.03502 | 0.735243 |  3.063511 | 0.0 |  90.785467 | 24.426301 |  -11.67757 |  3.063511 |
    | 13.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.121749 | 0.121749 | 0.017341 | 0.015607 | 0.001734 | 0.017341 | 0.088832 | 0.017588 | 0.020652 | 0.020652 | 0.020652 | 0.261894 | 0.396953 |  0.03824 | 0.697087 |  2.904529 | 0.0 |  90.646377 | 24.238358 | -12.018337 |  2.904529 |
    | 14.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.121606 | 0.121606 | 0.017208 | 0.015487 | 0.001721 | 0.017208 | 0.015618 | 0.007446 | 0.018415 | 0.018415 | 0.018415 | 0.249868 | 0.368208 | 0.025861 | 0.643937 |  2.683072 | 0.0 |  90.507563 | 24.016277 | -12.361883 |  2.683072 |
    | 15.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.121462 | 0.121462 | 0.017077 | 0.015369 | 0.001708 | 0.017077 | 0.015498 | 0.002036 | 0.015772 | 0.015772 | 0.015772 | 0.238729 |  0.34328 | 0.017808 | 0.599816 |  2.499235 | 0.0 |  90.369024 | 23.802619 | -12.683192 |  2.499235 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  3.0 |  2.570541 | 0.3 |      0.0 |      0.3 | 0.449087 | 0.404178 | 0.044909 | 0.019628 | 0.068464 | 0.004666 | 0.013229 | 0.013229 | 0.013229 | 0.229883 | 0.322829 | 0.017895 | 0.570607 |  2.377529 | 0.0 |  92.919937 | 23.627043 | -12.965406 |  2.377529 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  3.642215 | 0.3 |      0.0 |      0.3 | 0.681547 | 0.613392 | 0.068155 | 0.023761 | 0.344122 | 0.019974 | 0.011139 | 0.011139 | 0.011139 | 0.229304 | 0.311607 | 0.031113 | 0.572024 |  2.383433 | 0.0 |  96.538391 | 23.615352 | -13.128225 |  2.383433 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |  0.504222 | 0.2 |      0.0 |      0.2 | 0.120136 | 0.108123 | 0.012014 | 0.024358 | 0.496748 | 0.042766 |    0.011 |    0.011 |    0.011 | 0.233089 | 0.305212 | 0.053766 | 0.592068 |   2.46695 | 0.0 |  97.018254 | 23.691311 | -13.223739 |   2.46695 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |  1.339826 | 0.2 |      0.0 |      0.2 | 0.286227 | 0.257605 | 0.028623 | 0.026054 | 0.243678 | 0.043325 | 0.011904 | 0.011904 | 0.011904 | 0.229529 | 0.293279 | 0.055229 | 0.578036 |  2.408485 | 0.0 |  98.332027 | 23.619893 | -13.407642 |  2.408485 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |    0.6673 | 0.3 |      0.0 |      0.3 | 0.159614 | 0.143653 | 0.015961 | 0.026914 | 0.207984 |  0.02683 | 0.011054 | 0.011054 | 0.011054 |  0.22514 | 0.281615 | 0.037884 | 0.544638 |  2.269325 | 0.0 |  98.972413 | 23.530598 |  -13.59501 |  2.269325 |
    | 21.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.129972 | 0.129972 | 0.026701 | 0.024031 |  0.00267 | 0.026701 | 0.153296 | 0.020508 | 0.009991 | 0.009991 | 0.009991 | 0.219469 | 0.269779 | 0.030499 | 0.519747 |  2.165614 | 0.0 |   98.81574 | 23.413097 |  -13.79348 |  2.165614 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.8 |  3.968784 | 0.2 |      0.0 |      0.2 |  0.86368 | 0.777312 | 0.086368 | 0.032464 | 0.154107 | 0.018535 | 0.008592 | 0.008592 | 0.008592 | 0.214124 | 0.258997 | 0.027128 | 0.500249 |   2.08437 | 0.0 |  102.75206 |  23.30003 | -13.982241 |   2.08437 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 10.257869 | 0.3 |      0.0 |      0.3 | 2.594273 | 2.334846 | 0.259427 | 0.052142 | 0.818221 | 0.049142 | 0.007247 | 0.007247 | 0.007247 | 0.227121 | 0.263029 | 0.056388 | 0.546538 |  2.277243 | 0.0 | 112.957787 | 23.571088 | -13.910735 |  2.277243 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 |  10.65723 | 0.6 |      0.0 |      0.6 |  3.42438 | 3.081942 | 0.342438 |  0.08161 | 2.081847 | 0.144841 | 0.010473 | 0.010473 | 0.010473 |  0.27963 | 0.296091 | 0.155314 | 0.731036 |  3.045983 | 0.0 | 123.533407 | 24.551039 | -13.363614 |  3.045983 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |   2.64368 | 0.4 |      0.0 |      0.4 | 1.046732 | 0.942059 | 0.104673 | 0.090413 | 2.619606 | 0.245346 | 0.022137 | 0.022137 | 0.022137 | 0.359602 | 0.346515 | 0.267483 | 0.973601 |  4.056669 | 0.0 | 126.086674 | 25.785338 |  -12.64015 |  4.056669 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.291395 | 0.4 |      0.0 |      0.4 | 0.199739 | 0.179765 | 0.019974 | 0.091134 | 1.325703 | 0.229388 | 0.036829 | 0.036829 | 0.036829 | 0.392315 | 0.362345 | 0.266217 | 1.020877 |  4.253652 | 0.0 | 126.286936 | 26.225274 | -12.435385 |  4.253652 |
    | 27.01. |  0.1 | 0.4 | 0.3 |  0.0 |       0.0 | 0.1 | 0.231115 | 0.331115 |  0.08998 | 0.080982 | 0.008998 |  0.08998 | 0.340012 | 0.110611 | 0.042065 | 0.042065 | 0.042065 | 0.381958 | 0.349075 | 0.152676 | 0.883709 |   3.68212 | 0.0 | 125.965841 | 26.089388 | -12.606389 |   3.68212 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.8 |  2.030679 | 0.5 |      0.0 |      0.5 | 0.866432 | 0.779789 | 0.086643 | 0.097112 | 0.198874 | 0.037356 | 0.040448 | 0.040448 | 0.040448 | 0.366627 | 0.333231 | 0.077803 | 0.777661 |  3.240256 | 0.0 | 127.899409 | 25.882533 | -12.819623 |  3.240256 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  5.181272 | 0.4 |      0.0 |      0.4 | 2.236684 | 2.013015 | 0.223668 | 0.117955 | 0.788849 |   0.0509 | 0.037986 | 0.037986 | 0.037986 | 0.376554 | 0.334606 | 0.088886 | 0.800046 |  3.333525 | 0.0 | 132.962726 | 26.017274 | -12.800704 |  3.333525 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  6.829043 | 0.3 |      0.0 |      0.3 | 3.321736 | 2.989562 | 0.332174 | 0.150779 | 1.865245 | 0.133947 | 0.039589 | 0.039589 | 0.039589 | 0.433868 | 0.366623 | 0.173536 | 0.974026 |  4.058443 | 0.0 |  139.64099 | 26.742143 | -12.381639 |  4.058443 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  2.251963 | 0.3 |      0.0 |      0.3 | 1.310467 |  1.17942 | 0.131047 | 0.162429 | 2.519939 | 0.228058 | 0.048217 | 0.048217 | 0.048217 | 0.527892 | 0.421325 | 0.276276 | 1.225492 |  5.106219 | 0.0 | 141.730523 | 27.774432 | -11.746771 |  5.106219 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 |  9.387156 | 0.5 |      0.0 |      0.5 | 5.635217 | 5.071695 | 0.563522 | 0.222373 |  2.12321 | 0.264571 | 0.060504 | 0.060504 | 0.060504 | 0.604234 | 0.464711 | 0.325076 |  1.39402 |  5.808418 | 0.0 | 150.895307 | 28.504628 | -11.301694 |  5.808418 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |  1.641019 | 0.5 |      0.0 |      0.5 | 1.291956 | 1.162761 | 0.129196 | 0.232976 | 3.651104 |  0.30135 | 0.069196 | 0.069196 | 0.069196 | 0.786878 | 0.574469 | 0.370546 | 1.731893 |  7.216219 | 0.0 |  152.30335 | 29.977608 | -10.346525 |  7.216219 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |  1.442382 | 0.3 |      0.0 |      0.3 | 1.199948 | 1.079953 | 0.119995 |  0.24233 | 2.042321 | 0.324015 | 0.086728 | 0.086728 | 0.086728 | 0.852663 | 0.613069 | 0.410743 | 1.876475 |  7.818645 | 0.0 | 153.503402 | 30.437067 | -10.055937 |  7.818645 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |  1.131508 | 0.3 |      0.0 |      0.3 | 1.017888 | 0.916099 | 0.101789 | 0.249396 | 1.076461 |  0.19394 | 0.092197 | 0.092197 | 0.092197 | 0.837973 | 0.602246 | 0.286137 | 1.726357 |  7.193152 | 0.0 | 154.385514 | 30.337168 | -10.135402 |  7.193152 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |  4.423556 | 0.4 |      0.0 |      0.4 | 3.461238 | 3.115115 | 0.346124 | 0.284794 | 1.253583 | 0.134913 | 0.091008 | 0.091008 | 0.091008 | 0.838634 | 0.601082 | 0.225921 | 1.665637 |  6.940152 | 0.0 | 158.524276 | 30.341692 | -10.144042 |  6.940152 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 |  7.652417 | 0.4 |      0.0 |      0.4 | 6.604543 | 5.944088 | 0.660454 |  0.35696 | 3.000072 | 0.220714 | 0.091062 | 0.091062 | 0.091062 | 0.980952 | 0.688381 | 0.311776 | 1.981109 |  8.254621 | 0.0 | 165.819733 | 31.251844 |  -9.541333 |  8.254621 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |  3.135145 | 0.3 |      0.0 |      0.3 | 3.252526 | 2.927273 | 0.325253 |  0.38767 | 4.887592 | 0.403547 | 0.101895 | 0.101895 | 0.101895 | 1.308894 | 0.897271 | 0.505442 | 2.711606 |  11.29836 | 0.0 | 168.567207 | 32.977402 |  -8.381671 |  11.29836 |
    | 08.02. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.181449 | 0.181449 | 0.381214 | 0.343093 | 0.038121 | 0.381214 | 3.262107 | 0.456355 | 0.122434 | 0.122434 | 0.122434 | 1.450416 | 0.992647 | 0.578789 | 3.021852 | 12.591049 | 0.0 | 168.004543 | 33.606684 |  -7.947041 | 12.591049 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.8 |  2.960832 | 0.1 |      0.0 |      0.1 | 3.250345 | 2.925311 | 0.325035 | 0.411178 | 1.284482 | 0.287821 | 0.129924 | 0.129924 | 0.129924 | 1.348402 |  0.92888 | 0.417745 | 2.695027 | 11.229281 | 0.0 | 170.554198 | 33.158896 |  -8.232204 | 11.229281 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  4.469321 | 0.1 |      0.0 |      0.1 | 5.092757 | 4.583481 | 0.509276 | 0.462077 | 2.563133 | 0.212749 | 0.124594 | 0.124594 | 0.124594 |  1.40602 | 0.968817 | 0.337344 | 2.712181 | 11.300756 | 0.0 | 174.561441 |  33.41535 |  -8.051174 | 11.300756 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |  2.879951 | 0.0 |      0.0 |      0.0 | 3.714754 | 3.343278 | 0.371475 | 0.494705 | 4.036303 | 0.340832 | 0.127647 | 0.127647 | 0.127647 | 1.630772 | 1.121029 | 0.468479 |  3.22028 | 13.417833 | 0.0 | 176.946688 | 34.334006 |  -7.430034 | 13.417833 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.487884 | 0.439096 | 0.048788 | 0.487884 | 3.229538 | 0.400529 | 0.138581 | 0.138581 | 0.138581 | 1.724893 | 1.187645 | 0.539111 | 3.451648 | 14.381867 | 0.0 | 176.458803 | 34.685418 |  -7.187283 | 14.381867 |
    | 13.02. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.092613 | 0.092613 | 0.479977 | 0.431979 | 0.047998 | 0.479977 |      1.1 | 0.268509 | 0.142764 | 0.142764 | 0.142764 | 1.534191 | 1.060512 | 0.411273 | 3.005976 | 12.524902 | 0.0 | 175.886214 | 33.953991 |  -7.665031 | 12.524902 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.8 |  1.312983 | 0.0 |      0.0 |      0.0 | 1.978373 | 1.780536 | 0.197837 | 0.491356 | 0.617664 | 0.114306 | 0.134058 | 0.134058 | 0.134058 | 1.339632 | 0.932329 | 0.248364 | 2.520325 | 10.501355 | 0.0 |  176.70784 | 33.119016 |  -8.216236 | 10.501355 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |  4.716023 | 0.2 |      0.0 |      0.2 | 6.236389 |  5.61275 | 0.623639 | 0.552412 | 1.996247 | 0.137799 |  0.12412 |  0.12412 |  0.12412 |  1.33644 | 0.930603 | 0.261919 | 2.528961 | 10.537339 | 0.0 | 180.871451 | 33.104444 |   -8.22422 | 10.537339 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  3.412051 | 0.9 |      0.0 |      0.9 | 5.084999 | 4.576499 |   0.5085 |  0.59705 | 4.597936 | 0.329098 | 0.123946 | 0.123946 | 0.123946 |  1.63541 | 1.131123 | 0.453044 | 3.219577 | 13.414904 | 0.0 | 183.686452 | 34.351742 |  -7.392223 | 13.414904 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |  2.625687 | 0.9 |      0.0 |      0.9 | 4.304729 | 3.874256 | 0.430473 | 0.630416 | 4.716817 | 0.500776 | 0.138793 | 0.138793 | 0.138793 |    1.929 | 1.334853 | 0.639568 | 3.903421 | 16.264254 | 0.0 | 185.681723 | 35.391625 |  -6.701556 | 16.264254 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  1.790017 | 0.5 |      0.0 |      0.5 | 3.260126 | 2.934113 | 0.326013 | 0.650142 | 3.905982 | 0.487018 |  0.15117 |  0.15117 |  0.15117 | 2.064025 | 1.430302 | 0.638188 | 4.132514 |  17.21881 | 0.0 | 186.821597 | 35.822359 |  -6.418295 |  17.21881 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.363722 | 0.9 |      0.0 |      0.9 | 1.181503 | 1.063353 |  0.11815 | 0.645225 | 2.893038 | 0.390105 | 0.156297 | 0.156297 | 0.156297 | 2.022325 | 1.400259 | 0.546402 | 3.968987 | 16.537444 | 0.0 | 186.540094 | 35.692154 |  -6.505042 | 16.537444 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _gland_gr6_groundwater_loss:

groundwater loss
________________

This integration test corresponds to the :ref:`gland_gr4_groundwater_loss` example on
|gland_gr4|:

>>> x2(-0.637)
>>> test.inits.s = 0.0
>>> test.inits.r = 0.0
>>> test.inits.r2 = 0.0

.. integration-test::

    >>> conditions = test("gland_gr6_groundwater_loss", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |        fd |        fr |       fr2 |       qr |      qr2 |       qd |       qh |        qv |   i |          s |         r |         r2 |    outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 |  0.270088 |  0.270088 |  0.270088 |      0.0 | 3.435647 | 0.270088 | 3.705735 | 15.440563 | 0.0 |        0.0 |  0.270088 |  -3.165559 | 15.440563 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  8.9 |  8.895998 | 0.4 |      0.0 |      0.4 | 0.004002 | 0.003602 |   0.0004 |      0.0 | 0.000492 | 0.000027 |  0.266873 |  0.266873 |  0.266873 |      0.0 | 2.066781 | 0.266901 | 2.333682 |  9.723674 | 0.0 |   8.895998 |  0.537256 |  -4.965271 |  9.723674 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  2.8 |  2.794914 | 0.4 |      0.0 |      0.4 | 0.005087 | 0.004578 | 0.000509 | 0.000001 | 0.002914 | 0.000162 |  0.263693 |  0.263693 |  0.263693 |      0.0 |  1.50664 | 0.263855 | 1.770495 |  7.377064 | 0.0 |  11.690911 |  0.802698 |  -6.207052 |  7.377064 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |  6.972036 | 0.3 |      0.0 |      0.3 | 0.027971 | 0.025174 | 0.002797 | 0.000006 | 0.007167 | 0.000531 |  0.260534 |  0.260534 |  0.260534 |      0.0 | 1.200066 | 0.261064 | 1.461131 |  6.088045 | 0.0 |   18.66294 |  1.067532 |  -7.143718 |  6.088045 |
    | 05.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.014808 | 0.014808 | 0.000006 | 0.000006 | 0.000001 | 0.000006 | 0.017045 |  0.00118 |  0.257381 |  0.257381 |  0.257381 |      0.0 | 1.006834 | 0.258561 | 1.265395 |  5.272478 | 0.0 |  18.648126 |   1.33514 |  -7.886352 |  5.272478 |
    | 06.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.014797 | 0.014797 | 0.000006 | 0.000006 | 0.000001 | 0.000006 | 0.005742 | 0.001331 |  0.254196 |  0.254196 |  0.254196 |      0.0 |  0.87259 | 0.255527 | 1.128117 |  4.700489 | 0.0 |  18.633322 |   1.59278 |   -8.50245 |  4.700489 |
    | 07.01. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.014786 | 0.014786 | 0.000006 | 0.000006 | 0.000001 | 0.000006 | 0.000006 | 0.000453 |   0.25113 |   0.25113 |   0.25113 | 0.000001 | 0.773975 | 0.251582 | 1.025558 |  4.273158 | 0.0 |   18.61853 |  1.843913 |  -9.025293 |  4.273158 |
    | 08.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.029538 | 0.029538 | 0.000006 | 0.000006 | 0.000001 | 0.000006 | 0.000006 | 0.000025 |   0.24814 |   0.24814 |   0.24814 | 0.000001 |  0.69866 | 0.248165 | 0.946826 |  3.945109 | 0.0 |  18.588986 |  2.092055 |   -9.47581 |  3.945109 |
    | 09.01. |  0.1 | 0.2 | 0.1 |  0.0 |       0.0 | 0.1 | 0.014752 | 0.114752 | 0.000006 | 0.000006 | 0.000001 | 0.000006 | 0.000006 | 0.000001 |  0.245187 |  0.245187 |  0.245187 | 0.000002 | 0.639232 | 0.245187 | 0.884421 |  3.685089 | 0.0 |  18.574227 |  2.337243 |  -9.869853 |  3.685089 |
    | 10.01. |  0.2 | 0.3 | 0.1 |  0.0 |       0.0 | 0.2 | 0.014741 | 0.214741 | 0.000006 | 0.000006 | 0.000001 | 0.000006 | 0.000006 | 0.000001 |  0.242268 |  0.242268 |  0.242268 | 0.000003 |  0.59112 | 0.242269 | 0.833392 |  3.472467 | 0.0 |   18.55948 |  2.579512 | -10.218702 |  3.472467 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  2.6 |  2.582518 | 0.3 |      0.0 |      0.3 | 0.017494 | 0.015745 | 0.001749 | 0.000012 | 0.002154 |  0.00012 |  0.239385 |  0.239385 |  0.239385 | 0.000005 | 0.551445 | 0.239505 | 0.790955 |  3.295645 | 0.0 |  21.141986 |  2.820183 | -10.529901 |  3.295645 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.000012 | 0.000011 | 0.000001 | 0.000012 |  0.01001 | 0.000556 |   0.23652 |   0.23652 |   0.23652 | 0.000008 | 0.518399 | 0.237076 | 0.755484 |  3.147849 | 0.0 |  21.141974 |  3.062701 | -10.807776 |  3.147849 |
    | 13.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 |  0.03336 |  0.03336 | 0.000012 | 0.000011 | 0.000001 | 0.000012 | 0.003597 |  0.00078 |  0.233633 |  0.233633 |  0.233633 | 0.000012 | 0.489926 | 0.234413 | 0.724351 |   3.01813 | 0.0 |  21.108602 |  3.298481 |  -11.06263 |   3.01813 |
    | 14.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 |  0.03331 |  0.03331 | 0.000012 | 0.000011 | 0.000001 | 0.000012 | 0.000011 | 0.000281 |  0.230827 |  0.230827 |  0.230827 | 0.000017 | 0.465194 | 0.231108 | 0.696318 |  2.901326 | 0.0 |   21.07528 |  3.529297 | -11.296992 |  2.901326 |
    | 15.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.033259 | 0.033259 | 0.000012 | 0.000011 | 0.000001 | 0.000012 | 0.000011 | 0.000016 |   0.22808 |   0.22808 |   0.22808 | 0.000023 | 0.443605 | 0.228096 | 0.671724 |  2.798848 | 0.0 |  21.042009 |  3.757361 | -11.512513 |  2.798848 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  3.0 |  2.974016 | 0.3 |      0.0 |      0.3 | 0.026007 | 0.023406 | 0.002601 | 0.000023 | 0.003204 | 0.000179 |  0.225365 |  0.225365 |  0.225365 | 0.000031 | 0.424691 | 0.225544 | 0.650266 |   2.70944 | 0.0 |  24.016003 |  3.984617 | -11.710558 |   2.70944 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  4.249817 | 0.3 |      0.0 |      0.3 | 0.050234 | 0.045211 | 0.005023 | 0.000051 |  0.02105 |  0.00117 |   0.22266 |   0.22266 |   0.22266 | 0.000041 | 0.408466 |  0.22383 | 0.632337 |  2.634738 | 0.0 |  28.265768 |  4.219867 | -11.887944 |  2.634738 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |   0.59166 | 0.2 |      0.0 |      0.2 | 0.008397 | 0.007557 |  0.00084 | 0.000057 | 0.035102 | 0.002812 |   0.21986 |   0.21986 |   0.21986 | 0.000054 | 0.394347 | 0.222672 | 0.617073 |  2.571138 | 0.0 |  28.857372 |  4.460734 | -12.048391 |  2.571138 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |  1.576034 | 0.2 |      0.0 |      0.2 |  0.02404 | 0.021636 | 0.002404 | 0.000074 |  0.01806 | 0.003084 |  0.216993 |  0.216993 |  0.216993 | 0.000069 | 0.381038 | 0.220077 | 0.601185 |  2.504936 | 0.0 |  30.433332 |  4.688494 | -12.205212 |  2.504936 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |  0.787045 | 0.3 |      0.0 |      0.3 | 0.013039 | 0.011735 | 0.001304 | 0.000084 | 0.017076 | 0.002053 |  0.214282 |  0.214282 |  0.214282 | 0.000087 | 0.368922 | 0.216335 | 0.585344 |  2.438935 | 0.0 |  31.220293 |  4.912934 | -12.353022 |  2.438935 |
    | 21.01. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.048193 | 0.048193 | 0.000083 | 0.000075 | 0.000008 | 0.000083 |   0.0124 | 0.001663 |  0.211611 |  0.211611 |  0.211611 | 0.000108 | 0.357724 | 0.213273 | 0.571106 |   2.37961 | 0.0 |  31.172017 |  5.131876 | -12.494176 |   2.37961 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.8 |  4.707908 | 0.2 |      0.0 |      0.2 |  0.09226 | 0.083034 | 0.009226 | 0.000168 | 0.014055 | 0.001605 |  0.209005 |  0.209005 |  0.209005 | 0.000133 | 0.347506 | 0.210609 | 0.558249 |  2.326036 | 0.0 |  35.879757 |   5.34918 | -12.627055 |  2.326036 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 12.410562 | 0.3 |      0.0 |      0.3 | 0.390182 | 0.351163 | 0.039018 | 0.000744 | 0.100724 | 0.005828 |  0.206418 |  0.206418 |  0.206418 |  0.00017 | 0.340477 | 0.212246 | 0.552893 |  2.303722 | 0.0 |  48.289576 |  5.615862 | -12.720824 |  2.303722 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 | 13.276027 | 0.6 |      0.0 |      0.6 | 0.726478 |  0.65383 | 0.072648 | 0.002505 | 0.331366 | 0.021479 |  0.203244 |  0.203244 |  0.203244 | 0.000241 | 0.340154 | 0.224723 | 0.565117 |  2.354653 | 0.0 |  61.563098 |  6.017685 | -12.725188 |  2.354653 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |  3.354602 | 0.4 |      0.0 |      0.4 | 0.248663 | 0.223796 | 0.024866 | 0.003265 | 0.526156 | 0.043637 |  0.198461 |  0.198461 |  0.198461 | 0.000362 |  0.34493 | 0.242098 |  0.58739 |  2.447457 | 0.0 |  64.914435 |  6.531478 | -12.661194 |  2.447457 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.371115 | 0.4 |      0.0 |      0.4 | 0.032243 | 0.029019 | 0.003224 | 0.003358 | 0.295218 | 0.046798 |  0.192345 |  0.192345 |  0.192345 | 0.000477 | 0.342526 | 0.239144 | 0.582147 |  2.425611 | 0.0 |  65.282192 |  6.900477 | -12.693287 |  2.425611 |
    | 27.01. |  0.1 | 0.4 | 0.3 |  0.0 |       0.0 | 0.1 | 0.139773 | 0.239773 | 0.003322 |  0.00299 | 0.000332 | 0.003322 | 0.069857 | 0.024084 |  0.187953 |  0.187953 |  0.187953 | 0.000562 |  0.33384 | 0.212037 | 0.546439 |  2.276827 | 0.0 |  65.139097 |  7.129783 | -12.811231 |  2.276827 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.8 |  2.589401 | 0.5 |      0.0 |      0.5 | 0.214635 | 0.193171 | 0.021463 | 0.004035 | 0.034879 | 0.007609 |  0.185224 |  0.185224 |  0.185224 | 0.000647 | 0.324836 | 0.192833 | 0.518317 |  2.159652 | 0.0 |  67.724462 |  7.335287 | -12.936892 |  2.159652 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  6.671275 | 0.4 |      0.0 |      0.4 | 0.635178 |  0.57166 | 0.063518 | 0.006453 | 0.201487 | 0.012033 |  0.182778 |  0.182778 |  0.182778 | 0.000793 | 0.320805 | 0.194811 | 0.516409 |  2.151703 | 0.0 |  74.389285 |  7.638164 | -12.994324 |  2.151703 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  8.938718 | 0.3 |      0.0 |      0.3 | 1.072656 | 0.965391 | 0.107266 | 0.011374 | 0.539141 | 0.037145 |  0.179173 |  0.179173 |  0.179173 | 0.001089 | 0.325666 | 0.216318 | 0.543074 |  2.262806 | 0.0 |  83.316628 |  8.139732 | -12.925162 |  2.262806 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  2.983253 | 0.3 |      0.0 |      0.3 | 0.430298 | 0.387268 |  0.04303 | 0.013551 | 0.796751 |  0.06875 |  0.173203 |  0.173203 |  0.173203 | 0.001599 | 0.336843 | 0.241953 | 0.580395 |  2.418311 | 0.0 |   86.28633 |  8.789386 | -12.770101 |  2.418311 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 | 12.631868 | 0.5 |      0.0 |      0.5 | 2.194936 | 1.975442 | 0.219494 | 0.026803 | 0.735792 | 0.086772 |   0.16547 |   0.16547 |   0.16547 | 0.002231 | 0.345346 | 0.252242 | 0.599819 |  2.499247 | 0.0 |  98.891394 |    9.3941 |  -12.65566 |  2.499247 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |  2.239805 | 0.5 |      0.0 |      0.5 | 0.490132 | 0.441119 | 0.049013 | 0.029937 | 1.404073 | 0.109977 |  0.158272 |  0.158272 |  0.158272 | 0.003695 | 0.372545 | 0.268249 | 0.644489 |  2.685372 | 0.0 | 101.101262 | 10.391121 | -12.308303 |  2.685372 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |  1.973779 | 0.3 |      0.0 |      0.3 | 0.459146 | 0.413231 | 0.045915 | 0.032925 | 0.786992 | 0.124261 |  0.146405 |  0.146405 |  0.146405 | 0.004925 | 0.379278 | 0.270665 | 0.654868 |  2.728618 | 0.0 | 103.042116 | 11.004796 |  -12.22638 |  2.728618 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |  1.551054 | 0.3 |      0.0 |      0.3 | 0.384367 |  0.34593 | 0.038437 |  0.03542 | 0.410401 | 0.074544 |  0.139101 |  0.139101 |  0.139101 | 0.005835 | 0.373498 | 0.213645 | 0.592978 |  2.470741 | 0.0 | 104.557749 | 11.384302 | -12.296617 |  2.470741 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |  6.099702 | 0.4 |      0.0 |      0.4 | 1.547238 | 1.392514 | 0.154724 |  0.04694 | 0.504115 | 0.052947 |  0.134583 |  0.134583 |  0.134583 | 0.007025 | 0.370695 |  0.18753 |  0.56525 |  2.355208 | 0.0 | 110.610511 | 11.814329 | -12.331082 |  2.355208 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 | 10.709966 | 0.4 |      0.0 |      0.4 | 3.264355 | 2.937919 | 0.326435 | 0.074321 | 1.364923 | 0.096335 |  0.129465 |  0.129465 |  0.129465 |   0.0103 |  0.39422 |   0.2258 | 0.630319 |   2.62633 | 0.0 | 121.246157 | 12.752449 | -12.049868 |   2.62633 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |  4.441126 | 0.3 |      0.0 |      0.3 | 1.647547 | 1.482793 | 0.164755 | 0.088674 | 2.387107 | 0.190433 |  0.118299 |  0.118299 |  0.118299 | 0.018186 | 0.451882 | 0.308731 | 0.778799 |  3.244996 | 0.0 | 125.598609 | 14.284826 | -11.428608 |  3.244996 |
    | 08.02. |  0.0 | 0.2 | 0.2 |  0.0 |       0.0 | 0.0 | 0.153561 | 0.153561 | 0.087824 | 0.079041 | 0.008782 | 0.087824 | 1.622826 | 0.223422 |  0.100059 |  0.100059 |  0.100059 | 0.025935 | 0.479592 | 0.323481 | 0.829008 |    3.4542 | 0.0 | 125.357224 | 15.332645 |  -11.15901 |    3.4542 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.8 |  4.194228 | 0.1 |      0.0 |      0.1 | 1.708917 | 1.538026 | 0.170892 | 0.103146 | 0.598097 | 0.141399 |  0.087587 |  0.087587 |  0.087587 | 0.029671 | 0.465158 | 0.228987 | 0.723816 |  3.015898 | 0.0 | 129.448307 | 15.749419 | -11.297342 |  3.015898 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  6.370876 | 0.1 |      0.0 |      0.1 | 2.859701 | 2.573731 |  0.28597 | 0.130577 | 1.346879 | 0.106905 |  0.082626 |  0.082626 |  0.082626 | 0.038658 | 0.479923 | 0.189532 | 0.708112 |  2.950469 | 0.0 | 135.688607 | 16.601515 | -11.155887 |  2.950469 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |  4.127264 | 0.0 |      0.0 |      0.0 | 2.123645 | 1.911281 | 0.212365 | 0.150909 | 2.247272 | 0.184293 |  0.072484 |  0.072484 |  0.072484 | 0.057486 | 0.529331 | 0.256777 | 0.843594 |  3.514975 | 0.0 | 139.664961 | 17.964877 | -10.713825 |  3.514975 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.150098 | 0.135088 |  0.01501 | 0.150098 | 1.819826 | 0.223291 |  0.056256 |  0.056256 |  0.056256 | 0.076955 |  0.55681 | 0.279548 | 0.913313 |  3.805471 | 0.0 | 139.514864 | 19.036074 | -10.486449 |  3.805471 |
    | 13.02. |  0.0 | 0.1 | 0.1 |  0.0 |       0.0 | 0.0 | 0.081999 | 0.081999 | 0.148857 | 0.133972 | 0.014886 | 0.148857 | 0.539738 | 0.147595 |  0.043506 |  0.043506 |  0.043506 |  0.08293 |  0.52487 | 0.191101 |   0.7989 |   3.32875 | 0.0 | 139.284007 | 19.320493 | -10.751918 |   3.32875 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.8 |  1.861977 | 0.0 |      0.0 |      0.0 | 1.096231 | 0.986608 | 0.109623 | 0.158208 | 0.250601 | 0.055339 |  0.040121 |  0.040121 |  0.040121 | 0.085233 | 0.486139 |  0.09546 | 0.666832 |  2.778468 | 0.0 | 140.987776 | 19.425741 | -11.097696 |  2.778468 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |  6.705901 | 0.2 |      0.0 |      0.2 | 3.892459 | 3.503213 | 0.389246 |  0.19836 | 1.135776 | 0.072253 |  0.038868 |  0.038868 |  0.038868 |  0.09989 | 0.486823 | 0.111121 | 0.697834 |  2.907643 | 0.0 | 147.495316 | 20.046185 |  -11.09134 |  2.907643 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  4.873118 | 0.9 |      0.0 |      0.9 | 3.258583 | 2.932724 | 0.325858 |   0.2317 | 2.851803 |  0.19727 |  0.031483 |  0.031483 |  0.031483 | 0.147151 | 0.557971 | 0.228753 | 0.933876 |  3.891149 | 0.0 | 152.136734 | 21.641599 | -10.477107 |  3.891149 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |  3.753261 | 0.9 |      0.0 |      0.9 | 2.806388 | 2.525749 | 0.280639 | 0.259648 | 3.007194 | 0.313723 |  0.012493 |  0.012493 |  0.012493 | 0.211643 | 0.635332 | 0.326216 | 1.173191 |  4.888297 | 0.0 | 155.630346 | 23.246766 |  -9.897069 |  4.888297 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  2.554033 | 0.5 |      0.0 |      0.5 | 2.125224 | 1.912701 | 0.212522 | 0.279256 | 2.534826 | 0.312047 | -0.006613 | -0.006613 | -0.006613 | 0.275818 | 0.683442 | 0.305434 | 1.264694 |  5.269557 | 0.0 | 157.905122 | 24.485231 |  -9.573192 |  5.269557 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.516379 | 0.9 |      0.0 |      0.9 |  0.66497 | 0.598473 | 0.066497 | 0.281348 |  1.87304 | 0.252546 | -0.021354 | -0.021354 | -0.021354 | 0.323787 | 0.689408 | 0.231192 | 1.244387 |  5.184944 | 0.0 | 158.140153 | 25.263915 |  -9.534738 |  5.184944 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _gland_gr6_interception:

interception
____________

This integration test corresponds to the :ref:`gland_gr4_interception` example on
|gland_gr4|:

>>> imax(10.0)
>>> x2(0.637)
>>> test.inits.s = 0.3 * x1
>>> test.inits.r = 0.5 * x3

.. integration-test::

    >>> conditions = test("gland_gr6_interception", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |       fd |       fr |      fr2 |       qr |      qr2 |       qd |       qh |        qv |    i |          s |         r |         r2 |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 | 0.152867 | 0.152867 | 0.005681 | 0.005113 | 0.000568 | 0.005681 | 0.000698 | 0.000039 | 0.048412 | 0.048412 | 0.048412 | 0.406074 | 3.323095 | 0.048451 |  3.77762 | 15.740083 |  0.0 |  72.518551 | 26.401257 |  -3.274404 | 15.740083 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.005679 | 0.005111 | 0.000568 | 0.005679 | 0.003947 | 0.000219 |  0.04416 |  0.04416 |  0.04416 | 0.380309 | 1.953064 | 0.044379 | 2.377752 |    9.9073 |  8.9 |  72.512873 | 26.067476 |  -5.181729 |    9.9073 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  1.7 |  1.544421 | 0.4 |      0.0 |      0.4 | 0.161886 | 0.145697 | 0.016189 | 0.006307 |   0.0243 | 0.001538 | 0.040187 | 0.040187 | 0.040187 | 0.358086 | 1.393381 | 0.041725 | 1.793192 |  7.471635 | 10.0 |  74.050986 | 25.764157 |  -6.525204 |  7.471635 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |  6.288662 | 0.3 |      0.0 |      0.3 | 0.720814 | 0.648733 | 0.072081 | 0.009476 | 0.182316 | 0.010408 | 0.036577 | 0.036577 | 0.036577 | 0.344155 | 1.098834 | 0.046984 | 1.489973 |  6.208221 | 10.0 |  80.330173 | 25.565968 |  -7.514535 |  6.208221 |
    | 05.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 |  0.00947 | 0.008523 | 0.000947 |  0.00947 | 0.446707 | 0.030277 | 0.034218 | 0.034218 | 0.034218 | 0.341452 |  0.92918 | 0.064495 | 1.335127 |  5.563029 |  9.9 |  80.320702 | 25.526758 |  -8.230814 |  5.563029 |
    | 06.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009465 | 0.008518 | 0.000946 | 0.009465 | 0.154429 | 0.035061 | 0.033751 | 0.033751 | 0.033751 | 0.327845 | 0.791572 | 0.068812 | 1.188228 |  4.950949 |  9.8 |  80.311237 | 25.325322 |  -8.926863 |  4.950949 |
    | 07.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009459 | 0.008513 | 0.000946 | 0.009459 | 0.008519 | 0.012464 | 0.031353 | 0.031353 | 0.031353 | 0.310081 | 0.683127 | 0.043817 | 1.037025 |  4.320937 |  9.7 |  80.301778 | 25.051705 |   -9.57523 |  4.320937 |
    | 08.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.009454 | 0.008508 | 0.000945 | 0.009454 | 0.008514 | 0.001556 | 0.028096 | 0.028096 | 0.028096 | 0.293902 | 0.600991 | 0.029652 | 0.924545 |   3.85227 |  9.5 |  80.292324 | 24.791008 | -10.144719 |   3.85227 |
    | 09.01. |  0.1 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.009448 | 0.008503 | 0.000945 | 0.009448 | 0.008509 | 0.000946 | 0.024993 | 0.024993 | 0.024993 | 0.279104 | 0.536554 | 0.025939 | 0.841597 |  3.506654 |  9.4 |  80.282876 | 24.542003 | -10.652875 |  3.506654 |
    | 10.01. |  0.2 | 0.3 | 0.0 |  0.0 |       0.0 | 0.3 |      0.0 |      0.3 | 0.009443 | 0.008498 | 0.000944 | 0.009443 | 0.008504 | 0.000945 |  0.02203 |  0.02203 |  0.02203 | 0.265517 | 0.484599 | 0.022975 | 0.773091 |  3.221212 |  9.3 |  80.273434 | 24.303618 | -11.112043 |  3.221212 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  1.9 |  1.686967 | 0.3 |      0.0 |      0.3 | 0.223504 | 0.201154 |  0.02235 | 0.010471 | 0.034795 | 0.002406 | 0.019192 | 0.019192 | 0.019192 | 0.253802 | 0.442714 | 0.021598 | 0.718114 |   2.99214 | 10.0 |  81.949929 | 24.089884 | -11.521647 |   2.99214 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010464 | 0.009418 | 0.001046 | 0.010464 | 0.131077 | 0.007754 | 0.016648 | 0.016648 | 0.016648 |  0.24581 | 0.410685 | 0.024402 | 0.680898 |  2.837074 | 10.0 |  81.939465 | 23.939368 | -11.863253 |  2.837074 |
    | 13.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010458 | 0.009412 | 0.001046 | 0.010458 | 0.053114 | 0.010516 | 0.014857 | 0.014857 | 0.014857 | 0.236051 | 0.380809 | 0.025373 | 0.642233 |  2.675972 |  9.8 |  81.929007 | 23.750042 |  -12.20796 |  2.675972 |
    | 14.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010451 | 0.009406 | 0.001045 | 0.010451 | 0.009412 | 0.004455 | 0.012603 | 0.012603 | 0.012603 | 0.225724 | 0.353753 | 0.017058 | 0.596535 |  2.485565 |  9.6 |  81.918557 | 23.542569 | -12.545345 |  2.485565 |
    | 15.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010444 |   0.0094 | 0.001044 | 0.010444 | 0.009406 | 0.001228 | 0.010134 | 0.010134 | 0.010134 | 0.216107 | 0.330197 | 0.011362 | 0.557665 |  2.323606 |  9.4 |  81.908112 |  23.34224 | -12.861646 |  2.323606 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  2.4 |   2.11848 | 0.3 |      0.0 |      0.3 | 0.293379 | 0.264041 | 0.029338 | 0.011859 | 0.044157 | 0.002976 | 0.007749 | 0.007749 | 0.007749 | 0.208037 | 0.310366 | 0.010725 | 0.529128 |  2.204702 | 10.0 |  84.014734 | 23.168445 |   -13.1466 |  2.204702 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  3.759302 | 0.3 |      0.0 |      0.3 | 0.555446 | 0.499902 | 0.055545 | 0.014749 | 0.238199 | 0.013756 |  0.00568 |  0.00568 |  0.00568 | 0.205452 | 0.297414 | 0.019436 | 0.522302 |  2.176259 | 10.0 |  87.759287 | 23.111593 | -13.343055 |  2.176259 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |  0.520794 | 0.2 |      0.0 |      0.2 | 0.094385 | 0.084946 | 0.009438 | 0.015179 | 0.389511 | 0.031537 | 0.005004 | 0.005004 | 0.005004 | 0.206896 | 0.289249 | 0.036541 | 0.532685 |  2.219523 | 10.0 |  88.264902 | 23.143408 | -13.471495 |  2.219523 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |  1.384254 | 0.2 |      0.0 |      0.2 | 0.232138 | 0.208924 | 0.023214 | 0.016392 | 0.196438 | 0.034024 | 0.005382 | 0.005382 | 0.005382 |  0.20328 | 0.277386 | 0.039406 | 0.520072 |  2.166965 | 10.0 |  89.632765 | 23.063374 | -13.664923 |  2.166965 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |  0.689641 | 0.3 |      0.0 |      0.3 | 0.127376 | 0.114639 | 0.012738 | 0.017017 |   0.1678 | 0.021592 |  0.00443 |  0.00443 |  0.00443 |   0.1991 | 0.265966 | 0.026022 | 0.491087 |  2.046198 | 10.0 |  90.305388 | 22.969384 | -13.859339 |  2.046198 |
    | 21.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.017001 | 0.015301 |   0.0017 | 0.017001 | 0.122568 | 0.016488 | 0.003311 | 0.003311 | 0.003311 | 0.193988 | 0.254598 | 0.019799 | 0.468385 |  1.951606 |  9.8 |  90.288388 | 22.852248 | -14.061599 |  1.951606 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.6 |  3.932743 | 0.2 |      0.0 |      0.2 | 0.688276 | 0.619448 | 0.068828 | 0.021019 | 0.120399 | 0.014706 | 0.001917 | 0.001917 | 0.001917 | 0.189079 | 0.244158 | 0.016623 |  0.44986 |  1.874416 | 10.0 |  94.200112 | 22.737326 |  -14.25568 |  1.874416 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 10.636229 | 0.3 |      0.0 |      0.3 | 2.199605 | 1.979645 | 0.219961 | 0.035834 | 0.667411 |  0.03988 | 0.000549 | 0.000549 | 0.000549 | 0.197829 | 0.245329 | 0.040428 | 0.483586 |  2.014941 | 10.0 | 104.800507 | 22.940493 | -14.233495 |  2.014941 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 | 11.090393 | 0.6 |      0.0 |      0.6 | 2.968736 | 2.671862 | 0.296874 | 0.059129 |  1.76413 | 0.121195 | 0.002967 | 0.002967 | 0.002967 | 0.236817 | 0.269715 | 0.124163 | 0.630694 |   2.62789 | 10.0 | 115.831771 | 23.765121 | -13.794591 |   2.62789 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |  2.757196 | 0.4 |      0.0 |      0.4 | 0.909135 | 0.818221 | 0.090913 | 0.066331 | 2.261102 | 0.209535 | 0.012783 | 0.012783 | 0.012783 | 0.296754 |  0.30778 | 0.222318 | 0.826852 |  3.445215 | 10.0 | 118.522636 | 24.837811 | -13.185147 |  3.445215 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.304011 | 0.4 |      0.0 |      0.4 | 0.162987 | 0.146689 | 0.016299 | 0.066998 | 1.149018 | 0.198006 | 0.025551 | 0.025551 | 0.025551 | 0.321637 | 0.319084 | 0.223557 | 0.864278 |   3.60116 | 10.0 | 118.759649 | 25.231135 | -13.019074 |   3.60116 |
    | 27.01. |  0.1 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.066809 | 0.060129 | 0.006681 | 0.066809 | 0.287919 | 0.095526 | 0.030232 | 0.030232 | 0.030232 | 0.314419 | 0.308011 | 0.125758 | 0.748188 |   3.11745 |  9.7 | 118.692839 |   25.1197 | -13.181685 |   3.11745 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.5 |  1.890258 | 0.5 |      0.0 |      0.5 | 0.681834 | 0.613651 | 0.068183 | 0.072092 | 0.155405 |  0.03113 | 0.028906 | 0.028906 | 0.028906 |  0.30299 | 0.294702 | 0.060036 | 0.657727 |  2.740529 | 10.0 | 120.511005 | 24.938859 | -13.385319 |  2.740529 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  5.410815 | 0.4 |      0.0 |      0.4 | 1.978688 | 1.780819 | 0.197869 | 0.089503 | 0.646806 | 0.041535 | 0.026753 | 0.026753 | 0.026753 | 0.309598 | 0.294148 | 0.068288 | 0.672035 |  2.800145 | 10.0 | 125.832317 | 25.044097 | -13.393991 |  2.800145 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  7.144816 | 0.3 |      0.0 |      0.3 | 2.972682 | 2.675414 | 0.297268 | 0.117498 | 1.636918 | 0.114742 | 0.028006 | 0.028006 | 0.028006 | 0.353574 | 0.318357 | 0.142748 | 0.814679 |  3.394498 | 10.0 | 132.859635 |  25.70068 | -13.029575 |  3.394498 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  2.359072 | 0.3 |      0.0 |      0.3 | 1.168649 | 1.051784 | 0.116865 | 0.127721 | 2.249925 | 0.201534 | 0.035821 | 0.035821 | 0.035821 | 0.426986 | 0.360827 | 0.237355 | 1.025168 |  4.271532 | 10.0 | 135.090986 |  26.65947 | -12.454611 |  4.271532 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 |  9.850026 | 0.5 |      0.0 |      0.5 |  5.13057 | 4.617513 | 0.513057 | 0.180596 | 1.908497 | 0.236788 | 0.047233 | 0.047233 | 0.047233 | 0.488584 | 0.395148 | 0.284022 | 1.167754 |  4.865642 | 10.0 | 144.760417 | 27.363218 | -12.039127 |  4.865642 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |   1.72437 | 0.5 |      0.0 |      0.5 | 1.166025 | 1.049422 | 0.116602 | 0.190395 | 3.317864 | 0.272329 |  0.05561 |  0.05561 |  0.05561 |  0.63474 | 0.481806 | 0.327939 | 1.444485 |  6.018686 | 10.0 | 146.294392 | 28.774807 | -11.138178 |  6.018686 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |   1.51567 | 0.3 |      0.0 |      0.3 | 1.083471 | 0.975124 | 0.108347 | 0.199141 | 1.852465 | 0.294176 | 0.072412 | 0.072412 | 0.072412 | 0.693445 | 0.514757 | 0.366588 |  1.57479 |  6.561625 | 10.0 | 147.610921 | 29.265252 | -10.839537 |  6.561625 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |   1.18889 | 0.3 |      0.0 |      0.3 | 0.916989 |  0.82529 | 0.091699 | 0.205879 | 0.971606 | 0.175742 | 0.078249 | 0.078249 | 0.078249 |  0.68989 | 0.509872 | 0.253992 | 1.453754 |   6.05731 | 10.0 | 148.593931 | 29.236575 | -10.882518 |   6.05731 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |  4.649681 | 0.4 |      0.0 |      0.4 | 3.188729 | 2.869856 | 0.318873 | 0.238411 | 1.138498 | 0.122212 | 0.077908 | 0.077908 | 0.077908 | 0.697752 | 0.512258 |  0.20012 |  1.41013 |  5.875541 | 10.0 | 153.005202 | 29.299829 | -10.861469 |  5.875541 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 |  8.054622 | 0.4 |      0.0 |      0.4 | 6.150855 |  5.53577 | 0.615086 | 0.305477 | 2.767758 | 0.202492 | 0.078661 | 0.078661 | 0.078661 | 0.820785 | 0.585524 | 0.281153 | 1.687462 |   7.03109 | 10.0 | 160.754347 | 30.218359 | -10.261229 |   7.03109 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |  3.303093 | 0.3 |      0.0 |      0.3 | 3.031768 | 2.728592 | 0.303177 | 0.334861 | 4.545049 | 0.373772 | 0.089594 | 0.089594 | 0.089594 | 1.100864 |  0.75876 | 0.463366 |  2.32299 |  9.679125 | 10.0 | 163.722579 | 31.934118 |  -9.112375 |  9.679125 |
    | 08.02. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.331471 | 0.298324 | 0.033147 | 0.331471 | 3.036656 | 0.424348 | 0.110016 | 0.110016 | 0.110016 | 1.236104 |  0.84651 | 0.534364 | 2.616978 | 10.904076 |  9.8 | 163.391107 | 32.630024 |  -8.634207 | 10.904076 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.6 |   3.00524 | 0.1 |      0.0 |      0.1 | 2.954075 | 2.658667 | 0.295407 | 0.359315 | 1.174352 | 0.266892 | 0.118299 | 0.118299 | 0.118299 |  1.16744 | 0.805337 | 0.385191 | 2.357969 |  9.824872 | 10.0 | 166.037033 | 32.285494 |  -8.851505 |  9.824872 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  4.702102 | 0.1 |      0.0 |      0.1 | 4.806376 | 4.325738 | 0.480638 | 0.408477 | 2.348271 | 0.195242 | 0.114198 | 0.114198 | 0.114198 | 1.226382 | 0.844833 |  0.30944 | 2.380655 |  9.919398 | 10.0 | 170.330657 | 32.582273 |  -8.642831 |  9.919398 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |  3.030189 | 0.0 |      0.0 |      0.0 | 3.510462 | 3.159416 | 0.351046 | 0.440651 | 3.786616 | 0.316158 | 0.117731 | 0.117731 | 0.117731 |  1.43427 | 0.982143 | 0.433889 | 2.850301 | 11.876255 | 10.0 | 172.920195 | 33.537704 |  -7.992597 | 11.876255 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.435107 | 0.391596 | 0.043511 | 0.435107 |  3.04745 | 0.376138 | 0.129103 | 0.129103 | 0.129103 | 1.535598 |  1.05255 | 0.505242 |  3.09339 | 12.889123 | 10.0 | 172.485088 | 33.959679 |  -7.697063 | 12.889123 |
    | 13.02. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.429687 | 0.386719 | 0.042969 | 0.429687 | 1.021728 | 0.252553 | 0.134126 | 0.134126 | 0.134126 | 1.384808 | 0.955657 | 0.386679 | 2.727144 |   11.3631 |  9.9 | 172.055401 | 33.322034 |  -8.109903 |   11.3631 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.7 |  1.327529 | 0.0 |      0.0 |      0.0 | 1.813401 | 1.632061 |  0.18134 |  0.44093 | 0.557805 | 0.105716 | 0.126536 | 0.126536 | 0.126536 | 1.222102 | 0.850995 | 0.232253 |  2.30535 |  9.605623 | 10.0 |    172.942 | 32.561151 |   -8.61124 |  9.605623 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |  4.945335 | 0.2 |      0.0 |      0.2 | 5.955582 | 5.360024 | 0.595558 | 0.500917 | 1.857066 |  0.12738 |  0.11748 |  0.11748 |  0.11748 | 1.223721 | 0.852521 |  0.24486 | 2.321102 |  9.671258 | 10.0 | 177.386418 | 32.569149 |  -8.603454 |  9.671258 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  3.577702 | 0.9 |      0.0 |      0.9 | 4.867788 | 4.381009 | 0.486779 |  0.54549 | 4.376781 | 0.310495 | 0.117575 | 0.117575 | 0.117575 |   1.4997 | 1.034699 | 0.428069 | 2.962468 | 12.343618 | 10.0 |  180.41863 | 33.813093 |  -7.769866 | 12.343618 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |   2.75186 | 0.9 |      0.0 |      0.9 | 4.127488 | 3.714739 | 0.412749 | 0.579349 | 4.513193 | 0.477474 | 0.132381 | 0.132381 | 0.132381 | 1.777831 | 1.225133 | 0.609855 | 3.612819 | 15.053413 | 10.0 | 182.591142 | 34.875558 |   -7.05734 | 15.053413 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  1.874596 | 0.5 |      0.0 |      0.5 | 3.125393 | 2.812853 | 0.312539 | 0.599989 | 3.743488 | 0.466157 | 0.145027 | 0.145027 | 0.145027 | 1.916456 | 1.323105 | 0.611184 | 3.850745 | 16.044772 | 10.0 | 183.865749 | 35.350223 |  -6.738023 | 16.044772 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.380494 | 0.9 |      0.0 |      0.9 | 1.115957 | 1.004361 | 0.111596 | 0.596451 | 2.771559 | 0.373766 | 0.150677 | 0.150677 | 0.150677 | 1.892699 | 1.307697 | 0.524443 | 3.724839 | 15.520163 | 10.0 | 183.649792 | 35.271136 |  -6.786419 | 15.520163 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _gland_gr6_no_rconc_submodels:

no rconc submodels
__________________

This integration test corresponds to the :ref:`gland_gr4_no_rconc_submodels` example on
|gland_gr4|:

>>> model.rconcmodel_directflow = None
>>> model.rconcmodel_routingstore = None

.. integration-test::

    >>> conditions = test("gland_gr6_no_rconc_submodels", get_conditions="1990-01-01")
    |   date |    p |   e |  en |   pn |        ps |  ei |       es |       ae |       pr |      pr9 |      pr1 |     perc |       q9 |       q1 |       fd |       fr |      fr2 |       qr |      qr2 |       qd |       qh |        qv |    i |          s |         r |         r2 |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |  0.0 | 0.3 | 0.3 |  0.0 |       0.0 | 0.0 | 0.152867 | 0.152867 | 0.005681 | 0.005113 | 0.000568 | 0.005681 | 0.005113 | 0.000568 | 0.048412 | 0.048412 | 0.048412 | 0.406269 | 3.323983 |  0.04898 | 3.779232 | 15.746799 |  0.0 |  72.518551 | 26.403711 |  -3.273526 | 15.746799 |
    | 02.01. |  9.3 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.005679 | 0.005111 | 0.000568 | 0.005679 | 0.005111 | 0.000568 | 0.044189 | 0.044189 | 0.044189 | 0.380531 | 1.953526 | 0.044757 | 2.378814 |  9.911727 |  8.9 |  72.512873 | 26.070435 |  -5.180818 |  9.911727 |
    | 03.01. |  3.2 | 0.4 | 0.0 |  1.7 |  1.544421 | 0.4 |      0.0 |      0.4 | 0.161886 | 0.145697 | 0.016189 | 0.006307 | 0.145697 | 0.016189 | 0.040222 | 0.040222 | 0.040222 | 0.363171 | 1.405996 | 0.056411 | 1.825577 |  7.606571 | 10.0 |  74.050986 | 25.834905 |  -6.488313 |  7.606571 |
    | 04.01. |  7.3 | 0.3 | 0.0 |  7.0 |  6.288662 | 0.3 |      0.0 |      0.3 | 0.720814 | 0.648733 | 0.072081 | 0.009476 | 0.648733 | 0.072081 | 0.037419 | 0.037419 | 0.037419 |  0.36747 | 1.145953 |   0.1095 | 1.622923 |   6.76218 | 10.0 |  80.330173 | 25.894093 |  -7.337354 |   6.76218 |
    | 05.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 |  0.00947 | 0.008523 | 0.000947 |  0.00947 | 0.008523 | 0.000947 | 0.038123 | 0.038123 | 0.038123 | 0.345918 | 0.930211 |  0.03907 | 1.315199 |  5.479998 |  9.9 |  80.320702 | 25.591413 |  -8.226033 |  5.479998 |
    | 06.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009465 | 0.008518 | 0.000946 | 0.009465 | 0.008518 | 0.000946 |  0.03452 |  0.03452 |  0.03452 | 0.326469 | 0.783517 | 0.035467 | 1.145453 |  4.772719 |  9.8 |  80.311237 | 25.304575 |  -8.971622 |  4.772719 |
    | 07.01. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.009459 | 0.008513 | 0.000946 | 0.009459 | 0.008513 | 0.000946 | 0.031106 | 0.031106 | 0.031106 |  0.30883 | 0.677133 | 0.032052 | 1.018016 |  4.241732 |  9.7 |  80.301778 | 25.031959 |  -9.614244 |  4.241732 |
    | 08.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.009454 | 0.008508 | 0.000945 | 0.009454 | 0.008508 | 0.000945 | 0.027861 | 0.027861 | 0.027861 |  0.29276 | 0.596351 | 0.028807 | 0.917918 |  3.824657 |  9.5 |  80.292324 | 24.772166 |  -10.17933 |  3.824657 |
    | 09.01. |  0.1 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.009448 | 0.008503 | 0.000945 | 0.009448 | 0.008503 | 0.000945 | 0.024769 | 0.024769 | 0.024769 | 0.278057 | 0.532851 | 0.025714 | 0.836622 |  3.485927 |  9.4 |  80.282876 |  24.52398 | -10.684011 |  3.485927 |
    | 10.01. |  0.2 | 0.3 | 0.0 |  0.0 |       0.0 | 0.3 |      0.0 |      0.3 | 0.009443 | 0.008498 | 0.000944 | 0.009443 | 0.008498 | 0.000944 | 0.021815 | 0.021815 | 0.021815 | 0.264554 | 0.481572 | 0.022759 | 0.768886 |  3.203691 |  9.3 |  80.273434 |  24.28634 | -11.140369 |  3.203691 |
    | 11.01. |  2.9 | 0.3 | 0.0 |  1.9 |  1.686967 | 0.3 |      0.0 |      0.3 | 0.223504 | 0.201154 |  0.02235 | 0.010471 | 0.201154 |  0.02235 | 0.018986 | 0.018986 | 0.018986 | 0.258033 | 0.446103 | 0.041337 | 0.745472 |  3.106135 | 10.0 |  81.949929 | 24.167986 | -11.487024 |  3.106135 |
    | 12.01. |  0.2 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010464 | 0.009418 | 0.001046 | 0.010464 | 0.009418 | 0.001046 | 0.017578 | 0.017578 | 0.017578 | 0.246111 | 0.409603 | 0.018624 | 0.674338 |   2.80974 | 10.0 |  81.939465 | 23.945103 | -11.875281 |   2.80974 |
    | 13.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010458 | 0.009412 | 0.001046 | 0.010458 | 0.009412 | 0.001046 | 0.014925 | 0.014925 | 0.014925 | 0.235068 | 0.378552 | 0.015971 |  0.62959 |  2.623292 |  9.8 |  81.929007 | 23.730607 | -12.235144 |  2.623292 |
    | 14.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010451 | 0.009406 | 0.001045 | 0.010451 | 0.009406 | 0.001045 | 0.012372 | 0.012372 | 0.012372 | 0.224809 | 0.351794 | 0.013417 |  0.59002 |  2.458417 |  9.6 |  81.918557 | 23.523814 | -12.570804 |  2.458417 |
    | 15.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.010444 |   0.0094 | 0.001044 | 0.010444 |   0.0094 | 0.001044 |  0.00991 |  0.00991 |  0.00991 | 0.215253 |  0.32848 | 0.010955 | 0.554688 |  2.311199 |  9.4 |  81.908112 | 23.324111 | -12.885613 |  2.311199 |
    | 16.01. |  3.3 | 0.3 | 0.0 |  2.4 |   2.11848 | 0.3 |      0.0 |      0.3 | 0.293379 | 0.264041 | 0.029338 | 0.011859 | 0.264041 | 0.029338 | 0.007533 | 0.007533 | 0.007533 | 0.213049 | 0.314418 | 0.036871 | 0.564338 |  2.351408 | 10.0 |  84.014734 | 23.277019 | -13.086881 |  2.351408 |
    | 17.01. |  4.6 | 0.3 | 0.0 |  4.3 |  3.759302 | 0.3 |      0.0 |      0.3 | 0.555446 | 0.499902 | 0.055545 | 0.014749 | 0.499902 | 0.055545 | 0.006973 | 0.006973 | 0.006973 | 0.217261 | 0.307618 | 0.062517 | 0.587397 |  2.447487 | 10.0 |  87.759287 | 23.366672 | -13.187566 |  2.447487 |
    | 18.01. |  0.8 | 0.2 | 0.0 |  0.6 |  0.520794 | 0.2 |      0.0 |      0.2 | 0.094385 | 0.084946 | 0.009438 | 0.015179 | 0.084946 | 0.009438 |  0.00804 |  0.00804 |  0.00804 | 0.210197 | 0.291421 | 0.017478 | 0.519096 |    2.1629 | 10.0 |  88.264902 | 23.215483 | -13.436968 |    2.1629 |
    | 19.01. |  1.8 | 0.2 | 0.0 |  1.6 |  1.384254 | 0.2 |      0.0 |      0.2 | 0.232138 | 0.208924 | 0.023214 | 0.016392 | 0.208924 | 0.023214 |  0.00624 |  0.00624 |  0.00624 | 0.206755 | 0.279681 | 0.029454 |  0.51589 |  2.149544 | 10.0 |  89.632765 | 23.140323 |  -13.62684 |  2.149544 |
    | 20.01. |  1.1 | 0.3 | 0.0 |  0.8 |  0.689641 | 0.3 |      0.0 |      0.3 | 0.127376 | 0.114639 | 0.012738 | 0.017017 | 0.114639 | 0.012738 | 0.005346 | 0.005346 | 0.005346 | 0.201048 | 0.266932 | 0.018083 | 0.486063 |  2.025264 | 10.0 |  90.305388 | 23.013403 |  -13.84257 |  2.025264 |
    | 21.01. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.017001 | 0.015301 |   0.0017 | 0.017001 | 0.015301 |   0.0017 | 0.003835 | 0.003835 | 0.003835 |  0.19317 | 0.253267 | 0.005535 | 0.451972 |  1.883215 |  9.8 |  90.288388 | 22.833249 | -14.085882 |  1.883215 |
    | 22.01. |  5.0 | 0.2 | 0.0 |  4.6 |  3.932743 | 0.2 |      0.0 |      0.2 | 0.688276 | 0.619448 | 0.068828 | 0.021019 | 0.619448 | 0.068828 | 0.001691 | 0.001691 | 0.001691 | 0.200715 |  0.25307 | 0.070518 | 0.524303 |  2.184596 | 10.0 |  94.200112 | 23.005893 | -14.089482 |  2.184596 |
    | 23.01. | 13.1 | 0.3 | 0.0 | 12.8 | 10.636229 | 0.3 |      0.0 |      0.3 | 2.199605 | 1.979645 | 0.219961 | 0.035834 | 1.979645 | 0.219961 | 0.003746 | 0.003746 | 0.003746 |  0.24642 | 0.282738 | 0.223706 | 0.752865 |  3.136937 | 10.0 | 104.800507 | 23.951005 | -13.576617 |  3.136937 |
    | 24.01. | 14.6 | 0.6 | 0.0 | 14.0 | 11.090393 | 0.6 |      0.0 |      0.6 | 2.968736 | 2.671862 | 0.296874 | 0.059129 | 2.671862 | 0.296874 | 0.014995 | 0.014995 | 0.014995 | 0.322642 | 0.332792 | 0.311869 | 0.967303 |  4.030428 | 10.0 | 115.831771 | 25.246476 | -12.825669 |  4.030428 |
    | 25.01. |  4.0 | 0.4 | 0.0 |  3.6 |  2.757196 | 0.4 |      0.0 |      0.4 | 0.909135 | 0.818221 | 0.090913 | 0.066331 | 0.818221 | 0.090913 | 0.030415 | 0.030415 | 0.030415 | 0.335047 | 0.334479 | 0.121328 | 0.790854 |  3.295225 | 10.0 | 118.522636 | 25.432777 | -12.802445 |  3.295225 |
    | 26.01. |  0.8 | 0.4 | 0.0 |  0.4 |  0.304011 | 0.4 |      0.0 |      0.4 | 0.162987 | 0.146689 | 0.016299 | 0.066998 | 0.146689 | 0.016299 | 0.032632 | 0.032632 | 0.032632 | 0.321677 | 0.318362 | 0.048931 |  0.68897 |   2.87071 | 10.0 | 118.759649 | 25.231745 |   -13.0295 |   2.87071 |
    | 27.01. |  0.1 | 0.4 | 0.0 |  0.0 |       0.0 | 0.4 |      0.0 |      0.4 | 0.066809 | 0.060129 | 0.006681 | 0.066809 | 0.060129 | 0.006681 | 0.030239 | 0.030239 | 0.030239 | 0.306297 | 0.301711 |  0.03692 | 0.644929 |  2.687202 |  9.7 | 118.692839 | 24.991765 |  -13.27692 |  2.687202 |
    | 28.01. |  3.3 | 0.5 | 0.0 |  2.5 |  1.890258 | 0.5 |      0.0 |      0.5 | 0.681834 | 0.613651 | 0.068183 | 0.072092 | 0.613651 | 0.068183 | 0.027383 | 0.027383 | 0.027383 | 0.311608 | 0.299943 | 0.095566 | 0.707117 |   2.94632 | 10.0 | 120.511005 |  25.07573 | -13.304019 |   2.94632 |
    | 29.01. |  7.7 | 0.4 | 0.0 |  7.3 |  5.410815 | 0.4 |      0.0 |      0.4 | 1.978688 | 1.780819 | 0.197869 | 0.089503 | 1.780819 | 0.197869 | 0.028382 | 0.028382 | 0.028382 | 0.361457 | 0.328068 | 0.226251 | 0.915776 |  3.815733 | 10.0 | 125.832317 | 25.811147 | -12.891377 |  3.815733 |
    | 30.01. | 10.3 | 0.3 | 0.0 | 10.0 |  7.144816 | 0.3 |      0.0 |      0.3 | 2.972682 | 2.675414 | 0.297268 | 0.117498 | 2.675414 | 0.297268 | 0.037136 | 0.037136 | 0.037136 | 0.455736 | 0.384136 | 0.334404 | 1.174276 |  4.892816 | 10.0 | 132.859635 | 26.997796 | -12.168212 |  4.892816 |
    | 31.01. |  3.7 | 0.3 | 0.0 |  3.4 |  2.359072 | 0.3 |      0.0 |      0.3 | 1.168649 | 1.051784 | 0.116865 | 0.127721 | 1.051784 | 0.116865 |  0.05126 |  0.05126 |  0.05126 |  0.47421 | 0.391006 | 0.168125 | 1.033341 |  4.305586 | 10.0 | 135.090986 | 27.205917 | -12.087243 |  4.305586 |
    | 01.02. | 15.3 | 0.5 | 0.0 | 14.8 |  9.850026 | 0.5 |      0.0 |      0.5 |  5.13057 | 4.617513 | 0.513057 | 0.180596 | 4.617513 | 0.513057 | 0.053737 | 0.053737 | 0.053737 | 0.701376 |  0.52912 | 0.566794 |  1.79729 |   7.48871 | 10.0 | 144.760417 | 29.328786 | -10.715621 |   7.48871 |
    | 02.02. |  3.2 | 0.5 | 0.0 |  2.7 |   1.72437 | 0.5 |      0.0 |      0.5 | 1.166025 | 1.049422 | 0.116602 | 0.190395 | 1.049422 | 0.116602 | 0.079005 | 0.079005 | 0.079005 | 0.702188 | 0.525936 | 0.195608 | 1.423732 |  5.932216 | 10.0 | 146.294392 | 29.335257 | -10.742782 |  5.932216 |
    | 03.02. |  2.7 | 0.3 | 0.0 |  2.4 |   1.51567 | 0.3 |      0.0 |      0.3 | 1.083471 | 0.975124 | 0.108347 | 0.199141 | 0.975124 | 0.108347 | 0.079083 | 0.079083 | 0.079083 | 0.697955 | 0.520025 |  0.18743 | 1.405409 |  5.855873 | 10.0 | 147.610921 | 29.301459 | -10.793675 |  5.855873 |
    | 04.02. |  2.2 | 0.3 | 0.0 |  1.9 |   1.18889 | 0.3 |      0.0 |      0.3 | 0.916989 |  0.82529 | 0.091699 | 0.205879 |  0.82529 | 0.091699 |  0.07868 |  0.07868 |  0.07868 | 0.684277 | 0.508631 | 0.170379 | 1.363287 |  5.680364 | 10.0 | 148.593931 | 29.191036 | -10.893509 |  5.680364 |
    | 05.02. |  8.0 | 0.4 | 0.0 |  7.6 |  4.649681 | 0.4 |      0.0 |      0.4 | 3.188729 | 2.869856 | 0.318873 | 0.238411 | 2.869856 | 0.318873 | 0.077366 | 0.077366 | 0.077366 | 0.814679 | 0.586393 | 0.396239 | 1.797311 |  7.488794 | 10.0 | 153.005202 | 30.175637 | -10.254594 |  7.488794 |
    | 06.02. | 14.3 | 0.4 | 0.0 | 13.9 |  8.054622 | 0.4 |      0.0 |      0.4 | 6.150855 |  5.53577 | 0.615086 | 0.305477 |  5.53577 | 0.615086 | 0.089085 | 0.089085 | 0.089085 | 1.189271 | 0.820247 | 0.704171 | 2.713689 | 11.307038 | 10.0 | 160.754347 | 32.396912 |  -8.771448 | 11.307038 |
    | 07.02. |  6.3 | 0.3 | 0.0 |  6.0 |  3.303093 | 0.3 |      0.0 |      0.3 | 3.031768 | 2.728592 | 0.303177 | 0.334861 | 2.728592 | 0.303177 | 0.115525 | 0.115525 | 0.115525 | 1.284879 | 0.883605 | 0.418702 | 2.587185 | 10.779939 | 10.0 | 163.722579 | 32.864713 |  -8.448091 | 10.779939 |
    | 08.02. |  0.0 | 0.2 | 0.0 |  0.0 |       0.0 | 0.2 |      0.0 |      0.2 | 0.331471 | 0.298324 | 0.033147 | 0.331471 | 0.298324 | 0.033147 | 0.121093 | 0.121093 | 0.121093 | 1.121237 | 0.780552 |  0.15424 | 2.056029 |  8.566788 |  9.8 | 163.391107 | 32.043563 |  -8.988221 |  8.566788 |
    | 09.02. |  5.9 | 0.1 | 0.0 |  5.6 |   3.00524 | 0.1 |      0.0 |      0.1 | 2.954075 | 2.658667 | 0.295407 | 0.359315 | 2.658667 | 0.295407 | 0.111319 | 0.111319 | 0.111319 | 1.216531 | 0.842326 | 0.406726 | 2.465583 | 10.273261 | 10.0 | 166.037033 | 32.533552 |  -8.655761 | 10.273261 |
    | 10.02. |  9.2 | 0.1 | 0.0 |  9.1 |  4.702102 | 0.1 |      0.0 |      0.1 | 4.806376 | 4.325738 | 0.480638 | 0.408477 | 4.325738 | 0.480638 | 0.117151 | 0.117151 | 0.117151 |  1.48674 | 1.020494 | 0.597789 | 3.105022 | 12.937592 | 10.0 | 170.330657 | 33.759406 |  -7.828808 | 12.937592 |
    | 11.02. |  6.1 | 0.0 | 0.0 |  6.1 |  3.030189 | 0.0 |      0.0 |      0.0 | 3.510462 | 3.159416 | 0.351046 | 0.440651 | 3.159416 | 0.351046 | 0.131742 | 0.131742 | 0.131742 | 1.594213 | 1.095211 | 0.482788 | 3.172212 |  13.21755 | 10.0 | 172.920195 | 34.192584 |  -7.528511 |  13.21755 |
    | 12.02. |  0.1 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.435107 | 0.391596 | 0.043511 | 0.435107 | 0.391596 | 0.043511 | 0.136898 | 0.136898 | 0.136898 | 1.358756 | 0.941072 | 0.180409 | 2.480236 | 10.334318 | 10.0 | 172.485088 | 33.205684 |  -8.176046 | 10.334318 |
    | 13.02. |  0.0 | 0.1 | 0.0 |  0.0 |       0.0 | 0.1 |      0.0 |      0.1 | 0.429687 | 0.386719 | 0.042969 | 0.429687 | 0.386719 | 0.042969 | 0.125151 | 0.125151 | 0.125151 |  1.18543 | 0.828996 |  0.16812 | 2.182546 |  9.093942 |  9.9 | 172.055401 | 32.377437 |  -8.725203 |  9.093942 |
    | 14.02. |  2.8 | 0.0 | 0.0 |  2.7 |  1.327529 | 0.0 |      0.0 |      0.0 | 1.813401 | 1.632061 |  0.18134 |  0.44093 | 1.632061 |  0.18134 | 0.115293 | 0.115293 | 0.115293 |  1.17054 | 0.819316 | 0.296633 | 2.286489 |  9.527038 | 10.0 |    172.942 | 32.301426 |  -8.776402 |  9.527038 |
    | 15.02. | 10.6 | 0.2 | 0.0 | 10.4 |  4.945335 | 0.2 |      0.0 |      0.2 | 5.955582 | 5.360024 | 0.595558 | 0.500917 | 5.360024 | 0.595558 | 0.114388 | 0.114388 | 0.114388 | 1.562859 | 1.077921 | 0.709946 | 3.350726 | 13.961359 | 10.0 | 177.386418 | 34.068969 |  -7.595925 | 13.961359 |
    | 16.02. |  8.8 | 0.9 | 0.0 |  7.9 |  3.577702 | 0.9 |      0.0 |      0.9 | 4.867788 | 4.381009 | 0.486779 |  0.54549 | 4.381009 | 0.486779 | 0.135427 | 0.135427 | 0.135427 | 1.817675 | 1.253588 | 0.622206 | 3.693468 | 15.389451 | 10.0 |  180.41863 | 35.015327 |  -6.961682 | 15.389451 |
    | 17.02. |  7.2 | 0.9 | 0.0 |  6.3 |   2.75186 | 0.9 |      0.0 |      0.9 | 4.127488 | 3.714739 | 0.412749 | 0.579349 | 3.714739 | 0.412749 | 0.146691 | 0.146691 | 0.146691 | 1.945418 | 1.344079 |  0.55944 | 3.848937 | 16.037237 | 10.0 | 182.591142 | 35.445443 |  -6.673174 | 16.037237 |
    | 18.02. |  4.9 | 0.5 | 0.0 |  4.4 |  1.874596 | 0.5 |      0.0 |      0.5 | 3.125393 | 2.812853 | 0.312539 | 0.599989 | 2.812853 | 0.312539 | 0.151811 | 0.151811 | 0.151811 | 1.920702 |  1.32765 |  0.46435 | 3.712701 | 15.469589 | 10.0 | 183.865749 | 35.364264 |  -6.723872 | 15.469589 |
    | 19.02. |  1.8 | 0.9 | 0.0 |  0.9 |  0.380494 | 0.9 |      0.0 |      0.9 | 1.115957 | 1.004361 | 0.111596 | 0.596451 | 1.004361 | 0.111596 | 0.150844 | 0.150844 | 0.150844 | 1.662662 | 1.150265 |  0.26244 | 3.075367 | 12.814029 | 10.0 | 183.649792 | 34.455063 |  -7.321548 | 12.814029 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""

# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.interfaces import petinterfaces
from hydpy.interfaces import rconcinterfaces

# ...from  gland
from hydpy.models.gland import gland_model


class Model(gland_model.Main_PETModel_V1, gland_model.Main_RConcModel_V2):
    """|gland_gr6.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="G-GR6", description="Génie Rural model with 6 parameters"
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        gland_model.Calc_E_V1,
        gland_model.Calc_EI_V1,
        gland_model.Calc_PN_V1,
        gland_model.Calc_EN_V1,
        gland_model.Update_I_V1,
        gland_model.Calc_PS_V1,
        gland_model.Calc_ES_V1,
        gland_model.Update_S_V1,
        gland_model.Calc_Perc_V1,
        gland_model.Update_S_V2,
        gland_model.Calc_AE_V1,
        gland_model.Calc_Pr_V1,
        gland_model.Calc_PR1_PR9_V1,
        gland_model.Calc_Q9_V1,
        gland_model.Calc_Q1_V1,
        gland_model.Calc_FR_V2,
        gland_model.Calc_FR2_V1,
        gland_model.Update_R_V2,
        gland_model.Calc_QR_V1,
        gland_model.Update_R_V3,
        gland_model.Update_R2_V1,
        gland_model.Calc_QR2_R2_V1,
        gland_model.Calc_FD_V1,
        gland_model.Calc_QD_V1,
        gland_model.Calc_QH_V2,
        gland_model.Calc_QV_V1,
    )
    INTERFACE_METHODS = ()
    ADD_METHODS = (
        gland_model.Calc_E_PETModel_V1,
        gland_model.Calc_Q_RConcModel_V1,
        gland_model.Calc_Q_RConcModel_V1,
    )
    OUTLET_METHODS = (gland_model.Pass_Q_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (petinterfaces.PETModel_V1, rconcinterfaces.RConcModel_V1)
    SUBMODELS = ()

    petmodel = modeltools.SubmodelProperty(petinterfaces.PETModel_V1)

    rconcmodel_routingstore = modeltools.SubmodelProperty(
        rconcinterfaces.RConcModel_V1, optional=True
    )
    rconcmodel_directflow = modeltools.SubmodelProperty(
        rconcinterfaces.RConcModel_V1, optional=True
    )

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            Error = \Sigma InOut - \Delta Vol - \Delta Rconc
            \\ \\
            \Sigma InOut = \sum_{t=t0}^{t1} P_t - AE_t + FR_t + FR2_t + FD_t - QH_t
            \\
            \Delta Vol  =  \left( I_{t1} - I_{t0} \right)
            + \left( S_{t1} - S_{t0} \right) + \left( R_{t1} - R_{t0} \right)
            \\
            \Delta Rconc  = get\_waterbalance_{direct \ flow}(*)
            + get\_waterbalance_{routing \ store}(*)
        """
        fluxes = self.sequences.fluxes
        inputs = self.sequences.inputs
        last = self.sequences.states
        first = initial_conditions["model"]["states"]

        return (
            numpy.sum(inputs.p.evalseries)
            - numpy.sum(fluxes.ae.evalseries)
            + numpy.sum(fluxes.fr.evalseries)
            + numpy.sum(fluxes.fr2.evalseries)
            + numpy.sum(fluxes.fd.evalseries)
            - numpy.sum(fluxes.qh.evalseries)
            - (last.i - first["i"])
            - (last.s - first["s"])
            - (last.r - first["r"])
            - (last.r2 - first["r2"])
            - self._get_rconcmodel_waterbalance_directflow(initial_conditions)
            - self._get_rconcmodel_waterbalance_routingstore(initial_conditions)
        )


tester = Tester()
cythonizer = Cythonizer()
