from enum import Enum
from dataclasses import dataclass


@dataclass
class Unit:
    """
    A class representing units for TMT (Corresponds to the CSW Units Scala class)
    """
    name: str
    description: str


class Units(Enum):
    # SI units
    angstrom = Unit("Angstrom", "angstrom")
    alpha = Unit("alpha", "alpha: fine structure constant")
    ampere = Unit("A", "ampere: unit of electric current")
    arcmin = Unit("arcmin", "arc minute; angular measurement")
    arcsec = Unit("arcsec", "arc second: angular measurement")
    bar = Unit("bar", "bar: metric ton of pressure")
    candela = Unit("candela", "candela(lumen/sr)")
    day = Unit("d", "day")
    degree = Unit("deg", "degree: angular measurement 1/360 of full rotation")
    degC = Unit("degC", "Degree Celsius K")
    degF = Unit("degF", "Fahrenheit")
    elvolt = Unit("eV", "electron volt")
    gauss = Unit("gauss", "gauss")
    gram = Unit("g", "gram")
    hertz = Unit("Hz", "frequency")
    henry = Unit("henry", "Henry")
    hour = Unit("h", "hour")
    joule = Unit("J", "Joule: energy")
    kelvin = Unit("K", "Kelvin: temperature with a null point at absolute zero")
    kilogram = Unit("kg", "kilogram, base unit of mass in SI")
    kilometer = Unit("km", "kilometers")
    liter = Unit("l", "liter, metric unit of volume")
    lm = Unit("lm", "lumen")
    lsun = Unit("lsun", "solar luminosity")
    lx = Unit("lx", "lux(lm/m2)")
    mas = Unit("mas", "milli arc second")
    me = Unit("me", "me(electron_mass)")
    meter = Unit("m", "meter: base unit of length in SI")
    microarcsec = Unit("µas", "micro arcsec: angular measurement")
    millimeter = Unit("mm", "millimeters")
    millisecond = Unit("ms", "milliseconds")
    micron = Unit("µm", "micron: alias for micrometer")
    micrometer = Unit("µm", "micron")
    minute = Unit("min", "minute")
    MJD = Unit("MJD", "Mod. Julian Date")
    mol = Unit("mol", "mole- unit of substance")
    month = Unit("month", "Month name or number")
    mmyy = Unit("mmyy", "mmyy: Month/Year")
    mu0 = Unit("mu0", "mu0: magnetic constant")
    muB = Unit("muB", "Bohr magneton")
    nanometer = Unit("nm", "nanometers")
    newton = Unit("N", "Newton: force")
    ohm = Unit("ohm", "Ohm")
    pascal = Unit("Pa", "Pascal: pressure")
    pi = Unit("pi", "pi")
    pc = Unit("pc", "parsec")
    ppm = Unit("ppm", "part per million")
    radian = Unit("rad", "radian: angular measurement of the ratio between the length of an arc and its radius")
    second = Unit("s", "second: base unit of time in SI")
    sday = Unit("sday", "sidereal day is the time of one rotation of the Earth")
    steradian = Unit("sr", "steradian: unit of solid angle in SI")
    volt = Unit("V", "Volt: electric potential or electromotive force")
    watt = Unit("W", "Watt: power")
    Wb = Unit("Wb", "Weber")
    week = Unit("wk", "week")
    year = Unit("yr", "year")

    # CGS units
    coulomb = Unit("C", "coulomb: electric charge")
    centimeter = Unit("cm", "centimeter")
    D = Unit("Debye", "Debye(dipole) A electric dipole moment ")
    dyn = Unit("dyne", "dyne: Unit of force ")
    erg = Unit("erg", "erg: CGS unit of energy")

    # Astropyhsics units
    au = Unit("AU", "astronomical unit: approximately the mean Earth-Sun distance")
    a0 = Unit("a0",
              "bohr radius: probable distance between the nucleus and the electron in a hydrogen atom in its ground state")
    c = Unit("c", "c: speed of light")
    cKayser = Unit("cKayser", "cKayser")
    crab = Unit("crab", "Crab: astrophotometrical unit for measurement of the intensity of Astrophysical X-ray sources")
    damas = Unit("d:m:s", "damas: degree arcminute arcsecond (sexagesimal angle from degree)")
    e = Unit("e", "electron charge")
    earth = Unit("earth", "earth (geo) unit")
    F = Unit("F", "Farad: F")
    G = Unit("G", "gravitation constant")
    geoMass = Unit("geoMass", "Earth Mass")
    hm = Unit("hm", "hour minutes (sexagesimal time from hours)")
    hms = Unit("hms", "hour minutes seconds (sexagesimal time from hours)")
    hhmmss = Unit("HH:MM:SS", "hour minutes seconds (sexagesimal time)")
    jansky = Unit("Jy", "Jansky: spectral flux density")
    jd = Unit("jd", "Julian Day")
    jovmass = Unit("jovMass", "Jupiter mass")
    lightyear = Unit("lyr", "light year")
    mag = Unit("mag", "stellar magnitude")
    mjup = Unit("Mjup", "Jupiter mass")
    mp = Unit("mp", "proton_mass")
    minsec = Unit("m:s", "minutes seconds (sexagesimal time from minutes)")
    msun = Unit("Msun", "solar mass")
    photon = Unit("photon", "photon")
    rgeo = Unit("Rgeo", "Earth radius (eq)")
    rjup = Unit("Rjup", "Jupiter Radius(eq)")
    rsun = Unit("Rsun", "solar radius")
    rydberg = Unit("Rydberg", "energy of the photon whose wavenumber is the Rydberg constant")
    seimens = Unit("seimens", "Seimens")
    tesla = Unit("tesla", "Tesla")
    u = Unit("u", "atomic mass unit")

    # Imperial units
    barn = Unit("barn", "barn: metric unit of area")
    cal = Unit("cal", "thermochemical calorie: pre-SI metric unit of energy")
    foot = Unit("ft", "international foot")
    inch = Unit("inch", "international inch")
    pound = Unit("lb", "international avoirdupois pound")
    mile = Unit("mi", "international mile")
    ounce = Unit("oz", "international avoirdupois ounce")
    yard = Unit("yd", "international yard")

    # Others - engineering
    NoUnits = Unit("none", "scalar - no units specified")
    bit = Unit("bit", "bit: binary value of 0 or 1")
    encoder = Unit("enc", "encoder counts")
    count = Unit("ct", "counts as for an encoder or detector")
    mmhg = Unit("mmHg", "millimetre of mercury is a manometric unit of pressure")
    percent = Unit("percent", "percentage")
    pix = Unit("pix", "pixel")

    # Datetime units
    tai = Unit("TAI", "TAI time unit")
    utc = Unit("UTC", "UTC time unit")
    date = Unit("date", "date")
    datetime = Unit("datetime", "date/time")
