"""
Report generation logic for code review.
"""

import datetime
import json
import re
from collections import defaultdict
from pathlib import Path
from typing import List, Dict, Any, Optional

class ReportGenerator:
    def __init__(self, io):
        self.io = io

    def generate_report(
        self,
        output_path: Path,
        format: str,
        review_content: str,
        batch_info: Optional[Dict[str, Any]] = None,
        metadata: Optional[Dict[str, Any]] = None
    ) -> None:
        """Generate a code review report in the specified format."""
        if format == 'md':
            self._generate_markdown_report(output_path, review_content, batch_info, metadata)
        elif format == 'json':
            self._generate_json_report(output_path, review_content, batch_info, metadata)
        elif format == 'html':
            self._generate_html_report(output_path, review_content, batch_info, metadata)
        else:
            raise ValueError(f"Unsupported format: {format}")

    def _generate_markdown_report(
        self,
        output_path: Path,
        review_content: str,
        batch_info: Optional[Dict[str, Any]] = None,
        metadata: Optional[Dict[str, Any]] = None
    ) -> None:
        """Generate a markdown format report."""
        # When processing a batch, append to the file
        # When processing a full report, overwrite the file
        write_mode = "a" if batch_info else "w"
        
        with open(output_path, write_mode, encoding='utf-8') as f:
            if batch_info:
                f.write(f"\n\n### Batch {batch_info['batch_idx']} Review: {batch_info['description']}\n\n")
                f.write(f"Files reviewed in this batch:\n")
                for rel_path in batch_info['files']:
                    f.write(f"- `{rel_path}`\n")
                f.write("\n")
            else:
                f.write("# Code Review TODOs\n\n")
                f.write("Generated by CodeCraft\n\n")
                f.write("## Review Summary\n\n")
                if metadata:
                    for key, value in metadata.items():
                        if isinstance(value, list):
                            f.write(f"- {key}: {', '.join(value)}\n")
                        else:
                            f.write(f"- {key}: {value}\n")
                f.write(f"- Review completed at: {datetime.datetime.now().isoformat()}\n")
                f.write("\n## Review Findings\n\n")

            # Extract only the review content with TODOs
            review_lines = review_content.split('\n')
            start_idx = 0
            for i, line in enumerate(review_lines):
                if line.strip().startswith('- [ ]') or line.strip().startswith('- [x]'):
                    start_idx = i
                    break
            f.write('\n'.join(review_lines[start_idx:]))

    def _generate_json_report(
        self,
        output_path: Path,
        review_content: str,
        batch_info: Optional[Dict[str, Any]] = None,
        metadata: Optional[Dict[str, Any]] = None
    ) -> None:
        """Generate a JSON format report."""
        todos = self._parse_todos(review_content, batch_info)
        
        if batch_info:
            # For batch processing, append to existing JSON file
            try:
                with open(output_path.with_suffix('.json'), 'r', encoding='utf-8') as f:
                    report_data = json.load(f)
            except FileNotFoundError:
                report_data = {
                    'metadata': metadata or {},
                    'todos': []
                }
            
            report_data['todos'].extend(todos)
        else:
            # For single batch, create new JSON file
            report_data = {
                'metadata': {
                    **(metadata or {}),
                    'generated_at': datetime.datetime.now().isoformat()
                },
                'todos': todos
            }

        with open(output_path.with_suffix('.json'), 'w', encoding='utf-8') as f:
            json.dump(report_data, f, indent=2)

    def _generate_html_report(
        self,
        output_path: Path,
        review_content: str,
        batch_info: Optional[Dict[str, Any]] = None,
        metadata: Optional[Dict[str, Any]] = None
    ) -> None:
        """Generate an HTML format report."""
        try:
            import markdown
        except ImportError:
            self.io.tool_error("Python-Markdown package is required for HTML output. Please install it with: pip install markdown")
            return

        metadata_html = ""
        if metadata:
            metadata_html = "<div class='metadata'>\n"
            metadata_html += "<h2>Review Summary</h2>\n"
            metadata_html += "<table class='metadata-table'>\n"
            for key, value in metadata.items():
                metadata_html += f"<tr><th>{key}</th><td>"
                if isinstance(value, list):
                    metadata_html += f"{', '.join(value)}"
                else:
                    metadata_html += f"{value}"
                metadata_html += "</td></tr>\n"
            metadata_html += "</table>\n</div>\n"

        batch_html = ""
        if batch_info:
            batch_html = f"<div class='batch-info'>\n"
            batch_html += f"<h3>Batch {batch_info['batch_idx']} Review: {batch_info['description']}</h3>\n"
            batch_html += f"<p>Files reviewed in this batch:</p>\n<ul class='file-list'>\n"
            for rel_path in batch_info['files']:
                batch_html += f"<li><code>{rel_path}</code></li>\n"
            batch_html += "</ul>\n</div>\n"

        # For batch processing, check if HTML file exists and append content
        if batch_info and output_path.with_suffix('.html').exists():
            try:
                with open(output_path.with_suffix('.html'), 'r', encoding='utf-8') as f:
                    existing_content = f.read()
                
                # Insert batch content before the closing body tag
                if "</body>" in existing_content:
                    # Format review content for better display
                    formatted_review_content = markdown.markdown(
                        review_content, 
                        extensions=['fenced_code', 'tables', 'nl2br']
                    )
                    
                    # Enhance TODO formatting
                    formatted_review_content = self._enhance_todo_formatting(formatted_review_content)
                    
                    updated_content = existing_content.replace(
                        "</body>",
                        f"{batch_html}<div class='batch-content'>{formatted_review_content}</div>\n</body>"
                    )
                    
                    with open(output_path.with_suffix('.html'), 'w', encoding='utf-8') as f:
                        f.write(updated_content)
                    return
            except Exception as e:
                self.io.tool_error(f"Error updating HTML report: {str(e)}")

        # Either it's not batch processing or the HTML file doesn't exist yet
        # Format review content for better display
        formatted_review_content = markdown.markdown(
            review_content, 
            extensions=['fenced_code', 'tables', 'nl2br']
        )
        
        # Enhance TODO formatting
        formatted_review_content = self._enhance_todo_formatting(formatted_review_content)
        
        html_content = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <title>Code Review Report</title>
            <style>
                :root {{
                    --primary-color: #4a6da7;
                    --secondary-color: #f8f9fa;
                    --text-color: #333;
                    --accent-color: #28a745;
                    --warning-color: #ffc107;
                    --error-color: #dc3545;
                    --info-color: #17a2b8;
                    --border-color: #dee2e6;
                    --heading-color: #2c3e50;
                }}
                
                body {{
                    font-family: 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
                    line-height: 1.6;
                    color: var(--text-color);
                    max-width: 1200px;
                    margin: 0 auto;
                    padding: 20px;
                    background-color: #fff;
                }}
                
                h1, h2, h3, h4, h5, h6 {{
                    color: var(--heading-color);
                    margin-top: 1.5em;
                    margin-bottom: 0.5em;
                    font-weight: 600;
                }}
                
                h1 {{ 
                    font-size: 2.2em; 
                    padding-bottom: 0.3em;
                    border-bottom: 1px solid var(--border-color);
                }}
                
                h2 {{ font-size: 1.8em; }}
                h3 {{ font-size: 1.5em; }}
                
                p {{ margin: 1em 0; }}
                
                .metadata {{
                    background: var(--secondary-color);
                    padding: 20px;
                    border-radius: 8px;
                    margin-bottom: 30px;
                    box-shadow: 0 2px 4px rgba(0,0,0,0.05);
                }}
                
                .metadata-table {{
                    width: 100%;
                    border-collapse: collapse;
                }}
                
                .metadata-table th {{
                    text-align: left;
                    padding: 8px 15px;
                    width: 200px;
                    font-weight: 600;
                }}
                
                .metadata-table td {{
                    padding: 8px 15px;
                }}
                
                .metadata-table tr:nth-child(even) {{
                    background-color: rgba(0,0,0,0.02);
                }}
                
                .todo-item {{
                    border-left: 4px solid var(--primary-color);
                    padding: 15px 20px;
                    margin: 25px 0;
                    background-color: rgba(74, 109, 167, 0.05);
                    border-radius: 0 8px 8px 0;
                }}
                
                .todo-item h4 {{
                    margin-top: 0;
                    color: var(--primary-color);
                }}
                
                .priority-high {{
                    border-left-color: var(--error-color);
                    background-color: rgba(220, 53, 69, 0.05);
                }}
                
                .priority-high h4 {{
                    color: var(--error-color);
                }}
                
                .priority-medium {{
                    border-left-color: var(--warning-color);
                    background-color: rgba(255, 193, 7, 0.05);
                }}
                
                .priority-medium h4 {{
                    color: var(--warning-color);
                }}
                
                .priority-low {{
                    border-left-color: var(--accent-color);
                    background-color: rgba(40, 167, 69, 0.05);
                }}
                
                .priority-low h4 {{
                    color: var(--accent-color);
                }}
                
                .file-list {{
                    list-style-type: none;
                    padding-left: 10px;
                }}
                
                .file-list li {{
                    margin: 5px 0;
                }}
                
                .file-list li::before {{
                    content: "📄";
                    margin-right: 8px;
                }}
                
                code {{
                    font-family: 'SFMono-Regular', Consolas, 'Liberation Mono', Menlo, monospace;
                    background: var(--secondary-color);
                    padding: 3px 5px;
                    border-radius: 3px;
                    font-size: 0.9em;
                    color: #e83e8c;
                    word-wrap: break-word;
                }}
                
                pre {{
                    background: var(--secondary-color);
                    padding: 15px;
                    border-radius: 8px;
                    overflow-x: auto;
                    border: 1px solid var(--border-color);
                    margin: 1.5em 0;
                }}
                
                pre code {{
                    background: transparent;
                    color: var(--text-color);
                    padding: 0;
                }}
                
                .batch-info {{
                    margin-top: 40px;
                    padding: 20px;
                    border-top: 1px solid var(--border-color);
                    background-color: rgba(23, 162, 184, 0.05);
                    border-radius: 8px;
                }}
                
                .batch-content {{
                    margin-top: 20px;
                    margin-left: 15px;
                    padding-left: 15px;
                    border-left: 1px dashed var(--border-color);
                }}
                
                .todo-metadata {{
                    display: grid;
                    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                    gap: 10px;
                    margin: 10px 0;
                }}
                
                .todo-metadata-item {{
                    margin: 5px 0;
                }}
                
                .todo-metadata-label {{
                    font-weight: bold;
                    margin-right: 5px;
                }}
                
                .todo-details {{
                    margin-top: 15px;
                }}
                
                strong {{
                    font-weight: 600;
                }}
                
                em {{
                    font-style: italic;
                }}
            </style>
        </head>
        <body>
            <h1>Code Review TODOs</h1>
            {metadata_html}
            {batch_html if batch_info else ""}
            <div class='batch-content'>
                {formatted_review_content}
            </div>
        </body>
        </html>
        """

        with open(output_path.with_suffix('.html'), 'w', encoding='utf-8') as f:
            f.write(html_content)
            
    def _enhance_todo_formatting(self, html_content: str) -> str:
        """Enhance TODO formatting in HTML content with better structure."""
        # Replace todo checkboxes with better formatted elements
        html_content = re.sub(
            r'<li>\[[ x]\]',
            r'<li class="todo-item">',
            html_content
        )
        
        # Add priority-based styling
        html_content = re.sub(
            r'<li class="todo-item">.*?<strong>Priority</strong>: High',
            r'<li class="todo-item priority-high"><h4>High Priority Issue</h4>',
            html_content,
            flags=re.DOTALL
        )
        
        html_content = re.sub(
            r'<li class="todo-item">.*?<strong>Priority</strong>: Medium',
            r'<li class="todo-item priority-medium"><h4>Medium Priority Issue</h4>',
            html_content,
            flags=re.DOTALL
        )
        
        html_content = re.sub(
            r'<li class="todo-item">.*?<strong>Priority</strong>: Low',
            r'<li class="todo-item priority-low"><h4>Low Priority Issue</h4>',
            html_content,
            flags=re.DOTALL
        )
        
        # Structure metadata elements
        html_content = re.sub(
            r'<strong>(Issue|Location|Category|Effort|Dependencies)</strong>: (.*?)<br />',
            r'<div class="todo-metadata-item"><span class="todo-metadata-label">\1:</span> \2</div>',
            html_content
        )
        
        # Wrap metadata in a container
        html_content = re.sub(
            r'(<div class="todo-metadata-item">.*?</div>)\s*(<div class="todo-metadata-item">.*?</div>)\s*(<div class="todo-metadata-item">.*?</div>)',
            r'<div class="todo-metadata">\1\2\3</div>',
            html_content
        )
        
        # Enhance details section
        html_content = re.sub(
            r'<strong>Details</strong>: (.*?)((?:<div class="todo-metadata-item">|<\/li>))',
            r'<div class="todo-details"><strong>Details</strong>:<p>\1</p></div>\2',
            html_content,
            flags=re.DOTALL
        )
        
        return html_content

    def _parse_todos(self, review_content: str, batch_info: Optional[Dict[str, Any]] = None) -> List[Dict[str, Any]]:
        """Parse TODOs from the review content."""
        todos = []
        current_todo = {}

        for line in review_content.split('\n'):
            # Check for both completed and uncompleted TODOs
            if line.startswith('- [ ]') or line.startswith('- [x]'):
                if current_todo:
                    todos.append(current_todo)
                
                # Initialize new TODO with completion status
                current_todo = {
                    'completed': line.startswith('- [x]'),
                    **(batch_info or {})
                }
            elif '**' in line:
                key = re.search(r'\*\*(.*?)\*\*:', line)
                if key:
                    key = key.group(1).lower().replace(' ', '_')
                    value = line.split(':', 1)[1].strip()
                    current_todo[key] = value

        if current_todo:
            todos.append(current_todo)

        return todos 