"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserialize = exports.serialize = void 0;
const yaml = require("yaml");
const yaml_types = require("yaml/types");
/**
 * Serializes the given data structure into valid YAML.
 *
 * @param obj the data structure to serialize
 * @returns a string containing the YAML representation of {@param obj}
 */
function serialize(obj) {
    const oldFold = yaml_types.strOptions.fold.lineWidth;
    try {
        yaml_types.strOptions.fold.lineWidth = 0;
        return yaml.stringify(obj, { schema: 'yaml-1.1' });
    }
    finally {
        yaml_types.strOptions.fold.lineWidth = oldFold;
    }
}
exports.serialize = serialize;
/**
 * Deserialize the YAML into the appropriate data structure.
 *
 * @param str the string containing YAML
 * @returns the data structure the YAML represents
 *   (most often in case of CloudFormation, an object)
 */
function deserialize(str) {
    return parseYamlStrWithCfnTags(str);
}
exports.deserialize = deserialize;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: (_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        },
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join', 'Sub',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or', 'GetAtt',
].map(name => makeTagForCfnIntrinsic(name, true)).concat(makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('Condition', false));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'core',
    });
}
//# sourceMappingURL=data:application/json;base64,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