#!/usr/bin/env python3
"""
NovaLang Command Line Interface
Complete build system and development tools for NovaLang
"""

import argparse
import os
import sys
import json
import time
import subprocess
from pathlib import Path
from typing import List, Dict, Any, Optional
from dataclasses import dataclass, asdict

from lexer import NovaLangLexer
from parser import NovaLangParser, parse_novalang
from compiler import NovaLangCompiler, CompilerOptions, CompilationTarget
from novalang_auto_main import generate_auto_main_file

@dataclass
class ProjectConfig:
    """NovaLang project configuration"""
    name: str = "novalang-project"
    version: str = "1.0.0"
    description: str = ""
    author: str = ""
    license: str = "MIT"
    main: str = "src/main.nova"
    target: str = CompilationTarget.JVM
    output_dir: str = "build"
    source_dirs: List[str] = None
    dependencies: List[str] = None
    dev_dependencies: List[str] = None
    features: Dict[str, bool] = None
    
    def __post_init__(self):
        if self.source_dirs is None:
            self.source_dirs = ["src"]
        if self.dependencies is None:
            self.dependencies = []
        if self.dev_dependencies is None:
            self.dev_dependencies = []
        if self.features is None:
            self.features = {
                "ai_integration": True,
                "blockchain": True,
                "cloud_native": True,
                "performance_optimization": True
            }

class NovaLangCLI:
    """Command line interface for NovaLang"""
    
    def __init__(self):
        self.config: Optional[ProjectConfig] = None
        self.config_file = "nova.json"
    
    def load_config(self) -> ProjectConfig:
        """Load project configuration"""
        if os.path.exists(self.config_file):
            try:
                with open(self.config_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                return ProjectConfig(**data)
            except Exception as e:
                print(f"⚠️  Error loading config: {e}")
                return ProjectConfig()
        return ProjectConfig()
    
    def save_config(self, config: ProjectConfig):
        """Save project configuration"""
        with open(self.config_file, 'w', encoding='utf-8') as f:
            json.dump(asdict(config), f, indent=2)
    
    def init_project(self, args):
        """Initialize a new NovaLang project"""
        print("🚀 Initializing NovaLang project...")
        
        # Create project structure
        directories = [
            "src",
            "src/main",
            "src/test",
            "build",
            "docs",
            "examples"
        ]
        
        for directory in directories:
            os.makedirs(directory, exist_ok=True)
            print(f"✅ Created directory: {directory}")
        
        # Create configuration
        config = ProjectConfig(
            name=args.name or os.path.basename(os.getcwd()),
            description=args.description or "",
            target=args.target or CompilationTarget.JVM
        )
        
        self.save_config(config)
        print(f"✅ Created configuration: {self.config_file}")
        
        # Create sample files
        self.create_sample_files()
        
        print("🎉 Project initialized successfully!")
        print(f"📁 Project: {config.name}")
        print(f"🎯 Target: {config.target}")
        print("📝 Next steps:")
        print("   nova build    - Build the project")
        print("   nova run      - Run the project")
        print("   nova test     - Run tests")
    
    def create_sample_files(self):
        """Create sample NovaLang files"""
        
        # Main application file
        main_content = '''// NovaLang Main Application
@Component
@Application
class MainApplication {
    
    @Inject
    private service: UserService;
    
    @PostConstruct
    function initialize() {
        console.log("🚀 NovaLang application starting...");
    }
    
    function main(args: string[]): void {
        service.processUsers();
        console.log("✅ Application completed successfully!");
    }
}
'''
        
        # Service class
        service_content = '''// User Service with AI and Cloud features
@Service
@Component
@MLModel(framework: "tensorflow", modelPath: "models/user_model.pb")
class UserService {
    
    @Inject
    private repository: UserRepository;
    
    @Predict
    @Cached(ttl: "5m")
    async function getUserRecommendations(
        @Input user: User,
        @Parameter count: int = 10
    ): Observable<Product> {
        let features = user.extractFeatures();
        let predictions = await model.predict(features);
        return predictions.topK(count);
    }
    
    @Transaction
    @Retryable(attempts: 3)
    function processUsers(): void {
        let users = repository.findAll();
        for (user in users) {
            let recommendations = await getUserRecommendations(user, 5);
            user.setRecommendations(recommendations);
            repository.save(user);
        }
    }
}
'''
        
        # Test file
        test_content = '''// NovaLang Test Example
@TestSuite
class UserServiceTest {
    
    @MockBean
    private repository: UserRepository;
    
    @InjectMock
    private service: UserService;
    
    @Test
    async function testUserRecommendations() {
        // Arrange
        let user = new User("test@example.com");
        let mockProducts = [new Product("Product 1"), new Product("Product 2")];
        
        // Act
        let recommendations = await service.getUserRecommendations(user);
        
        // Assert
        assert(recommendations.length > 0);
        assert(recommendations.contains(mockProducts[0]));
    }
    
    @Test
    @Performance(maxTime: "100ms")
    function testProcessUsersPerformance() {
        service.processUsers();
    }
}
'''
        
        # Write files
        with open("src/main/MainApplication.nova", 'w', encoding='utf-8') as f:
            f.write(main_content)
        print("✅ Created: src/main/MainApplication.nova")
        
        with open("src/main/UserService.nova", 'w', encoding='utf-8') as f:
            f.write(service_content)
        print("✅ Created: src/main/UserService.nova")
        
        with open("src/test/UserServiceTest.nova", 'w', encoding='utf-8') as f:
            f.write(test_content)
        print("✅ Created: src/test/UserServiceTest.nova")
        
        # Create README
        readme_content = f'''# NovaLang Project

## Overview
This is a NovaLang project with enterprise-grade features including AI/ML integration, blockchain support, and cloud-native architecture.

## Features
- 🤖 AI/ML Integration with TensorFlow and PyTorch
- ⛓️ Blockchain and Smart Contract support
- ☁️ Cloud-native microservices architecture
- 🚀 High-performance optimizations
- 💡 Modern type system with generics and pattern matching

## Quick Start

### Build the project:
```bash
nova build
```

### Run the application:
```bash
nova run
```

### Run tests:
```bash
nova test
```

### Deploy to cloud:
```bash
nova deploy --platform kubernetes
```

## Project Structure
```
src/
  main/           # Main application code
  test/           # Test files
build/            # Compiled output
docs/             # Documentation
examples/         # Code examples
nova.json         # Project configuration
```

## Dependencies
This project uses NovaLang's advanced features and may require additional runtime libraries for AI, blockchain, and cloud features.
'''
        
        with open("README.md", 'w', encoding='utf-8') as f:
            f.write(readme_content)
        print("✅ Created: README.md")
    
    def build_project(self, args):
        """Build the NovaLang project"""
        print("🔨 Building NovaLang project...")
        
        config = self.load_config()
        start_time = time.time()
        
        # Create build directory
        os.makedirs(config.output_dir, exist_ok=True)
        
        # Find all .nova files
        nova_files = []
        for source_dir in config.source_dirs:
            if os.path.exists(source_dir):
                for root, dirs, files in os.walk(source_dir):
                    for file in files:
                        if file.endswith('.nova'):
                            nova_files.append(os.path.join(root, file))
        
        if not nova_files:
            print("❌ No .nova files found to compile")
            return False
        
        print(f"📁 Found {len(nova_files)} NovaLang files")
        
        # Setup compiler options
        compiler_options = CompilerOptions(
            target=config.target,
            optimization_level=args.optimization if hasattr(args, 'optimization') else 2,
            enable_ai_integration=config.features.get('ai_integration', True),
            enable_blockchain=config.features.get('blockchain', True),
            enable_cloud_native=config.features.get('cloud_native', True),
            enable_performance_opt=config.features.get('performance_optimization', True),
            output_directory=config.output_dir
        )
        
        compiler = NovaLangCompiler(compiler_options)
        
        # Compile each file
        compiled_files = 0
        for nova_file in nova_files:
            try:
                print(f"🔧 Compiling: {nova_file}")
                
                # Read and parse file
                with open(nova_file, 'r', encoding='utf-8') as f:
                    source_code = f.read()
                
                ast = parse_novalang(source_code)
                
                # Generate output filename
                rel_path = os.path.relpath(nova_file, '.')
                output_base = os.path.join(config.output_dir, rel_path)
                
                # Compile to target
                compiler.compile_to_file(ast, output_base)
                compiled_files += 1
                
            except Exception as e:
                print(f"❌ Error compiling {nova_file}: {e}")
                if args.verbose if hasattr(args, 'verbose') else False:
                    import traceback
                    traceback.print_exc()
                continue
        
        build_time = time.time() - start_time
        
        if compiled_files == len(nova_files):
            print(f"✅ Build successful! ({compiled_files} files, {build_time:.2f}s)")
            
            # Generate build info
            build_info = {
                "project": config.name,
                "version": config.version,
                "target": config.target,
                "files_compiled": compiled_files,
                "build_time": build_time,
                "timestamp": time.time(),
                "features": config.features
            }
            
            with open(os.path.join(config.output_dir, "build-info.json"), 'w') as f:
                json.dump(build_info, f, indent=2)
            
            return True
        else:
            print(f"⚠️  Build completed with errors ({compiled_files}/{len(nova_files)} files)")
            return False
    
    def run_project(self, args):
        """Run the NovaLang project"""
        print("🚀 Running NovaLang project...")
        
        config = self.load_config()
        
        # Check if build exists
        build_info_path = os.path.join(config.output_dir, "build-info.json")
        if not os.path.exists(build_info_path):
            print("❌ Project not built. Run 'nova build' first.")
            return False
        
        # Run based on target
        if config.target == CompilationTarget.JVM:
            self.run_java_project(config)
        elif config.target == CompilationTarget.JAVASCRIPT:
            self.run_javascript_project(config)
        else:
            print(f"❌ Running {config.target} projects not implemented yet")
            return False
    
    def run_java_project(self, config: ProjectConfig):
        """Run Java/JVM project"""
        print("☕ Running Java application...")
        
        # Find main class
        main_java = config.main.replace('.nova', '.java').replace('src/', f'{config.output_dir}/src/')
        
        if os.path.exists(main_java):
            try:
                # Compile Java (simplified - in real implementation would use proper Java build)
                subprocess.run(['javac', main_java], check=True)
                print("✅ Java compilation successful")
                
                # Run Java application
                main_class = os.path.basename(main_java).replace('.java', '')
                subprocess.run(['java', '-cp', config.output_dir, main_class], check=True)
            except subprocess.CalledProcessError as e:
                print(f"❌ Error running Java application: {e}")
        else:
            print(f"❌ Main class not found: {main_java}")
    
    def run_javascript_project(self, config: ProjectConfig):
        """Run JavaScript/TypeScript project"""
        print("🌐 Running JavaScript application...")
        
        main_js = config.main.replace('.nova', '.ts').replace('src/', f'{config.output_dir}/src/')
        
        if os.path.exists(main_js):
            try:
                # Run with Node.js (would need TypeScript compilation in real implementation)
                subprocess.run(['node', '-r', 'ts-node/register', main_js], check=True)
            except subprocess.CalledProcessError as e:
                print(f"❌ Error running JavaScript application: {e}")
        else:
            print(f"❌ Main file not found: {main_js}")
    
    def test_project(self, args):
        """Run tests for the project"""
        print("🧪 Running NovaLang tests...")
        
        config = self.load_config()
        
        # Find test files
        test_files = []
        for source_dir in config.source_dirs:
            test_dir = os.path.join(source_dir, 'test')
            if os.path.exists(test_dir):
                for root, dirs, files in os.walk(test_dir):
                    for file in files:
                        if file.endswith('.nova') and 'test' in file.lower():
                            test_files.append(os.path.join(root, file))
        
        if not test_files:
            print("❌ No test files found")
            return False
        
        print(f"🧪 Found {len(test_files)} test files")
        
        # Run tests (simplified - would integrate with test framework)
        passed_tests = 0
        for test_file in test_files:
            try:
                print(f"🔬 Testing: {test_file}")
                
                # Parse test file (basic validation)
                with open(test_file, 'r', encoding='utf-8') as f:
                    source_code = f.read()
                
                ast = parse_novalang(source_code)
                
                # Count test methods
                test_methods = 0
                for stmt in ast.statements:
                    if hasattr(stmt, 'body'):
                        for member in getattr(stmt, 'body', []):
                            if (hasattr(member, 'annotations') and 
                                any('@Test' in str(ann) for ann in getattr(member, 'annotations', []))):
                                test_methods += 1
                
                print(f"  ✅ {test_methods} test methods found")
                passed_tests += 1
                
            except Exception as e:
                print(f"  ❌ Test file error: {e}")
        
        if passed_tests == len(test_files):
            print(f"✅ All tests passed! ({passed_tests}/{len(test_files)} files)")
            return True
        else:
            print(f"⚠️  Some tests failed ({passed_tests}/{len(test_files)} files)")
            return False
    
    def clean_project(self, args):
        """Clean build artifacts"""
        print("🧹 Cleaning project...")
        
        config = self.load_config()
        
        # Remove build directory
        if os.path.exists(config.output_dir):
            import shutil
            shutil.rmtree(config.output_dir)
            print(f"✅ Removed build directory: {config.output_dir}")
        
        # Remove other artifacts
        artifacts = ['*.class', '*.js.map', 'node_modules', '.nova-cache']
        for pattern in artifacts:
            # In real implementation, would use glob to remove matching files
            pass
        
        print("✅ Project cleaned successfully!")
    
    def deploy_project(self, args):
        """Deploy project to cloud platform"""
        print("☁️ Deploying NovaLang project...")
        
        config = self.load_config()
        platform = args.platform if hasattr(args, 'platform') else 'kubernetes'
        
        print(f"🎯 Target platform: {platform}")
        
        # Generate deployment files
        if platform == 'kubernetes':
            self.generate_kubernetes_deployment(config)
        elif platform == 'docker':
            self.generate_docker_deployment(config)
        else:
            print(f"❌ Platform {platform} not supported yet")
            return False
        
        print("✅ Deployment configuration generated!")
        print("📝 Review the generated files and apply them to your platform")
    
    def generate_kubernetes_deployment(self, config: ProjectConfig):
        """Generate Kubernetes deployment files"""
        deployment_yaml = f'''apiVersion: apps/v1
kind: Deployment
metadata:
  name: {config.name}
  labels:
    app: {config.name}
spec:
  replicas: 3
  selector:
    matchLabels:
      app: {config.name}
  template:
    metadata:
      labels:
        app: {config.name}
    spec:
      containers:
      - name: {config.name}
        image: {config.name}:latest
        ports:
        - containerPort: 8080
        env:
        - name: SPRING_PROFILES_ACTIVE
          value: "production"
        resources:
          requests:
            memory: "256Mi"
            cpu: "250m"
          limits:
            memory: "512Mi"
            cpu: "500m"
---
apiVersion: v1
kind: Service
metadata:
  name: {config.name}-service
spec:
  selector:
    app: {config.name}
  ports:
  - port: 80
    targetPort: 8080
  type: LoadBalancer
'''
        
        os.makedirs('deploy', exist_ok=True)
        with open('deploy/kubernetes.yaml', 'w') as f:
            f.write(deployment_yaml)
        print("✅ Created: deploy/kubernetes.yaml")
    
    def generate_docker_deployment(self, config: ProjectConfig):
        """Generate Docker deployment files"""
        dockerfile = f'''FROM openjdk:11-jre-slim

WORKDIR /app

COPY {config.output_dir}/ .

EXPOSE 8080

CMD ["java", "-jar", "{config.name}.jar"]
'''
        
        with open('Dockerfile', 'w') as f:
            f.write(dockerfile)
        print("✅ Created: Dockerfile")
    
    def main(self):
        """Main CLI entry point"""
        parser = argparse.ArgumentParser(
            description="NovaLang - Advanced Programming Language",
            formatter_class=argparse.RawDescriptionHelpFormatter,
            epilog="""
Examples:
  nova init my-project           Initialize new project
  nova build                     Build the project  
  nova run                       Run the application
  nova test                      Run tests
  nova deploy --platform k8s     Deploy to Kubernetes
            """
        )
        
        subparsers = parser.add_subparsers(dest='command', help='Available commands')
        
        # Init command
        init_parser = subparsers.add_parser('init', help='Initialize new project')
        init_parser.add_argument('name', nargs='?', help='Project name')
        init_parser.add_argument('--description', help='Project description')
        init_parser.add_argument('--target', choices=[CompilationTarget.JVM, CompilationTarget.JAVASCRIPT], 
                                default=CompilationTarget.JVM, help='Compilation target')
        
        # Build command
        build_parser = subparsers.add_parser('build', help='Build the project')
        build_parser.add_argument('--optimization', '-O', type=int, choices=[0, 1, 2], 
                                 default=2, help='Optimization level')
        build_parser.add_argument('--verbose', '-v', action='store_true', help='Verbose output')
        
        # Run command
        run_parser = subparsers.add_parser('run', help='Run the application')
        
        # Test command
        test_parser = subparsers.add_parser('test', help='Run tests')
        
        # Clean command
        clean_parser = subparsers.add_parser('clean', help='Clean build artifacts')
        
        # Deploy command
        deploy_parser = subparsers.add_parser('deploy', help='Deploy to cloud platform')
        deploy_parser.add_argument('--platform', choices=['kubernetes', 'docker', 'aws', 'gcp'], 
                                  default='kubernetes', help='Deployment platform')
        
        args = parser.parse_args()
        
        if not args.command:
            parser.print_help()
            return
        
        # Execute command
        if args.command == 'init':
            self.init_project(args)
        elif args.command == 'build':
            self.build_project(args)
        elif args.command == 'run':
            self.run_project(args)
        elif args.command == 'test':
            self.test_project(args)
        elif args.command == 'clean':
            self.clean_project(args)
        elif args.command == 'deploy':
            self.deploy_project(args)

if __name__ == "__main__":
    cli = NovaLangCLI()
    cli.main()
