-- ===================================================
-- NovaLang Spring Boot Database Creation Script
-- Generated by NovaLang Database Creator
-- ===================================================

-- 1. Create Database
CREATE DATABASE IF NOT EXISTS novalang_app;
USE novalang_app;

-- 2. Create Users Table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(255) UNIQUE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email)
);

-- 3. Create Products Table
CREATE TABLE IF NOT EXISTS products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    category VARCHAR(50),
    stock INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_category (category),
    INDEX idx_name (name)
);

-- 4. Create Orders Table
CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    product_id INT,
    quantity INT NOT NULL,
    total DECIMAL(10,2) NOT NULL,
    status VARCHAR(20) DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_product_id (product_id),
    INDEX idx_status (status)
);

-- 5. Insert Sample Users
INSERT INTO users (name, email) VALUES
    ('John Doe', 'john@company.com'),
    ('Jane Smith', 'jane@company.com'),
    ('Bob Wilson', 'bob@company.com'),
    ('Alice Johnson', 'alice@company.com'),
    ('Charlie Brown', 'charlie@company.com');

-- 6. Insert Sample Products
INSERT INTO products (name, price, category, stock) VALUES
    ('MacBook Pro 16"', 2499.99, 'Electronics', 15),
    ('Dell XPS 13', 1299.99, 'Electronics', 20),
    ('iPhone 15 Pro', 999.99, 'Electronics', 50),
    ('Samsung Galaxy S24', 899.99, 'Electronics', 35),
    ('Premium Coffee Beans', 24.99, 'Food', 100),
    ('Organic Tea', 18.99, 'Food', 80),
    ('Gaming Chair', 299.99, 'Furniture', 25),
    ('Standing Desk', 599.99, 'Furniture', 10),
    ('Wireless Mouse', 79.99, 'Electronics', 60),
    ('Mechanical Keyboard', 149.99, 'Electronics', 40);

-- 7. Insert Sample Orders
INSERT INTO orders (user_id, product_id, quantity, total, status) VALUES
    (1, 1, 1, 2499.99, 'completed'),
    (2, 3, 2, 1999.98, 'shipped'),
    (3, 5, 5, 124.95, 'pending'),
    (1, 9, 1, 79.99, 'completed'),
    (4, 7, 1, 299.99, 'processing'),
    (2, 2, 1, 1299.99, 'shipped'),
    (5, 6, 3, 56.97, 'pending'),
    (3, 8, 1, 599.99, 'completed'),
    (4, 4, 1, 899.99, 'processing'),
    (5, 10, 1, 149.99, 'completed');

-- 8. Create Views for Common Queries
CREATE VIEW user_order_summary AS
SELECT 
    u.id as user_id,
    u.name as user_name,
    u.email,
    COUNT(o.id) as total_orders,
    COALESCE(SUM(o.total), 0) as total_spent,
    MAX(o.created_at) as last_order_date
FROM users u
LEFT JOIN orders o ON u.id = o.user_id
GROUP BY u.id, u.name, u.email;

CREATE VIEW product_sales_summary AS
SELECT 
    p.id as product_id,
    p.name as product_name,
    p.category,
    p.price,
    p.stock,
    COUNT(o.id) as times_ordered,
    COALESCE(SUM(o.quantity), 0) as total_quantity_sold,
    COALESCE(SUM(o.total), 0) as total_revenue
FROM products p
LEFT JOIN orders o ON p.id = o.product_id
GROUP BY p.id, p.name, p.category, p.price, p.stock;

-- 9. Sample Queries to Test the Database

-- Get all users with their order count
SELECT * FROM user_order_summary ORDER BY total_spent DESC;

-- Get products by category with sales data
SELECT * FROM product_sales_summary WHERE category = 'Electronics' ORDER BY total_revenue DESC;

-- Get recent orders with user and product details
SELECT 
    o.id as order_id,
    u.name as customer_name,
    p.name as product_name,
    o.quantity,
    o.total,
    o.status,
    o.created_at
FROM orders o
JOIN users u ON o.user_id = u.id
JOIN products p ON o.product_id = p.id
ORDER BY o.created_at DESC
LIMIT 10;

-- Get category-wise sales summary
SELECT 
    p.category,
    COUNT(DISTINCT p.id) as product_count,
    COUNT(o.id) as total_orders,
    SUM(o.quantity) as total_quantity,
    SUM(o.total) as total_revenue,
    AVG(p.price) as avg_product_price
FROM products p
LEFT JOIN orders o ON p.id = o.product_id
GROUP BY p.category
ORDER BY total_revenue DESC;

-- Get top customers by spending
SELECT 
    u.name,
    u.email,
    COUNT(o.id) as order_count,
    SUM(o.total) as total_spent
FROM users u
JOIN orders o ON u.id = o.user_id
GROUP BY u.id, u.name, u.email
ORDER BY total_spent DESC
LIMIT 5;

-- ===================================================
-- Database Setup Complete!
-- 
-- Your NovaLang database now contains:
-- ✅ 3 tables (users, products, orders)
-- ✅ 5 sample users
-- ✅ 10 sample products across 3 categories
-- ✅ 10 sample orders
-- ✅ 2 views for common queries
-- ✅ Foreign key relationships
-- ✅ Indexes for performance
-- ✅ Sample queries to test functionality
--
-- You can now use this database with your NovaLang
-- Spring Boot application!
-- ===================================================
