from __future__ import absolute_import

from six import iteritems
from werkzeug.test import Client as WerkzeugClient, EnvironBuilder
from werkzeug.wrappers import Request as WerkzeugSpecificRequest

from .serving import Service
from .request import WerkzeugRequest
from .response import Response


class Client(WerkzeugClient):
    """The class used as a test client.

    Example::

        client = Client(api)
        # GET /examples
        response = client.get('/examples')
        # POST /examples
        response = client.post('/examples', data='{"name": "test"}',
                               content_type='application/json')

    :param api: the RESTArt API object.
    """

    def __init__(self, api):
        service = Service(api)
        super(Client, self).__init__(service)

    def wrap_response(self, response):
        _data, _status, _headers = response
        data = b''.join(_data)
        status_code = int(_status.split(None, 1)[0])
        headers = {key: value for key, value in iteritems(_headers)}
        return Response(data, status_code, headers)

    def open(self, *args, **kwargs):
        response = super(Client, self).open(*args, **kwargs)
        return self.wrap_response(response)


class RequestFactory(WerkzeugClient):
    """The class used to generate request objects.

    Example::

        factory = RequestFactory()
        # GET /examples
        request = factory.get('/examples')
        # POST /examples
        request = factory.post('/examples', data='{"name": "test"}',
                               content_type='application/json')

    :param keep_initial_request: a boolean value. If set to `True`, the
                                 request object generated by the factory
                                 will be the initial request object, which
                                 is framework-specific. If not specified,
                                 defaults to `False`, then the final adapted
                                 request object will be generated.
    """

    def __init__(self, keep_initial_request=False):
        self.keep_initial_request = keep_initial_request

    def open(self, *args, **kwargs):
        builder = EnvironBuilder(*args, **kwargs)
        environ = builder.get_environ()

        initial_request = WerkzeugSpecificRequest(environ)
        if self.keep_initial_request:
            return initial_request
        return WerkzeugRequest(initial_request)
