import sys
sys.path.append('..')

import numpy as np
from numpy import pi, sqrt

from composites import laminated_plate
from composites.kassapoglou import (calc_Nxx_crit,
                                    calc_Nxy_crit,
                                    calc_Nxx_crit_combined_shear,
                                    calc_Nxx_crit_combined_shear_full,
                                    calc_beff,
                                    )


def test_calc_Nxx_crit():
    """Verificatoin based on Kassapoglou's Figs. 6.4 and 6.5

    """
    alu = laminated_plate(stack=[0], plyt=0.5715e-3, laminaprop=(70e9, 0.3))
    D11, D12, D22, D66 = 0.66, 0.47, 0.66, 0.49
    AR_values = np.linspace(0.6, 2.8)
    res1 = []
    res2 = []
    res3 = []
    res_alu1 = []
    res_alu2 = []
    res_alu3 = []
    b = 0.508/2
    for AR in AR_values:
        a = AR*b
        res1.append(calc_Nxx_crit(a, b, 1, 1, D11, D12, D22, D66))
        res2.append(calc_Nxx_crit(a, b, 2, 1, D11, D12, D22, D66))
        res3.append(calc_Nxx_crit(a, b, 3, 1, D11, D12, D22, D66))

        res_alu1.append(calc_Nxx_crit(a, b, 1, 1, alu.D11, alu.D12, alu.D22, alu.D66))
        res_alu2.append(calc_Nxx_crit(a, b, 2, 1, alu.D11, alu.D12, alu.D22, alu.D66))
        res_alu3.append(calc_Nxx_crit(a, b, 3, 1, alu.D11, alu.D12, alu.D22, alu.D66))

    res1_expected = [760.4492762134736, 728.400368245419, 703.8761399336366, 685.191127113606, 671.1307610092099, 660.8018833650219, 653.5358314008664, 648.8239785928772, 646.2738786053692, 645.5788159767615, 646.4962762781008, 648.8324700409954, 652.4310404371092, 657.1647103179126, 662.9290256450245, 669.6376148748768, 677.2185585936345, 685.6115818818151, 694.7658630211979, 704.6383086268357, 715.192185097175, 726.3960246767871, 738.2227449149805, 750.6489352387343, 763.6542753519922, 777.2210583401934, 791.3337974784167, 805.9789003643929, 821.1443975169391, 836.8197152790613, 852.995484948758, 869.6633816801207, 886.8159879639275, 904.4466774933751, 922.5495160089763, 941.119176343723, 960.1508653910068, 979.6402611206446, 999.5834580935449, 1019.9769201892686, 1040.817439475499, 1062.102100324078, 1083.828248022461, 1105.9934612482962, 1128.5955278731656, 1151.6324236431244, 1175.1022933516867, 1199.003434177715, 1223.3342809083138, 1248.093392806889]
    res2_expected = [1574.5738453392305, 1425.2156696076072, 1304.4297339276068, 1205.4741816411472, 1123.4907376437611, 1054.9067689137432, 997.0476243303884, 947.8787977982346, 905.8305039745393, 869.67588101298, 838.4448711977442, 811.3623166552635, 787.8027900721946, 767.2571828544187, 749.3076788484112, 733.6087918799004, 719.8728442935457, 707.8587364114169, 697.3631813606328, 688.2138056014026, 680.2636747275131, 673.3869177172485, 667.4752047678446, 662.4348935872179, 658.1847029911407, 654.6538053213732, 651.7802536782276, 649.5096784526276, 647.7942017198433, 646.5915288518977, 645.8641850407839, 645.5788709028703, 645.7059164012659, 646.2188163087604, 647.0938335874052, 648.3096595691644, 649.8471218276071, 651.6889322420013, 653.819469055981, 656.224587787787, 658.8914567081547, 661.8084133044526, 664.9648387265003, 668.3510476848932, 671.9581916659561, 675.7781736539106, 679.8035728228145, 684.0275778881181, 688.4439279982369, 693.0468602067966]
    res3_expected = [2971.8346291977537, 2633.2313905592655, 2358.731949245679, 2133.1671277763076, 1945.6058071661016, 1788.0095631162233, 1654.360430605087, 1540.0807709179026, 1441.638567033499, 1356.2733797976512, 1281.8025793497804, 1216.4820604743586, 1158.9046117163628, 1107.9247387390258, 1062.602355196917, 1022.1601171979253, 985.9507500175343, 953.4317793804016, 924.1458098225442, 897.7050008794315, 873.7787501370861, 852.0838477966455, 832.3765518018913, 814.446166995635, 798.1098107133652, 783.2081207240418, 769.6017165033668, 757.1682664309185, 745.8000451760562, 735.4018898258297, 725.8894820622249, 717.1878982720699, 709.2303808723597, 701.9572931018719, 695.315226625222, 689.2562369393289, 683.737186084723, 678.7191757897579, 674.1670571026093, 670.0490049393197, 666.3361479090431, 663.0022453583597, 660.0234048743622, 657.3778345558914, 655.0456252471773, 653.0085586627191, 651.2499379441467, 649.7544377012231, 648.5079710178828, 647.4975712647844]

    assert np.allclose(res1, res1_expected)
    assert np.allclose(res2, res2_expected)
    assert np.allclose(res3, res3_expected)

    #import matplotlib.pyplot as plt
    #plt.plot(AR_values, np.asarray(res1)/1000, label='m=1')
    #plt.plot(AR_values, np.asarray(res2)/1000, label='m=2')
    #plt.plot(AR_values, np.asarray(res3)/1000, label='m=3')
    #plt.plot(AR_values, np.asarray(res_alu1)/1000, label='m=1')
    #plt.plot(AR_values, np.asarray(res_alu2)/1000, label='m=2')
    #plt.plot(AR_values, np.asarray(res_alu3)/1000, label='m=3')
    #plt.ylim(-0.1, 0.9)
    #plt.legend()
    #plt.xlabel('Length/Width')
    #plt.ylabel('Buckling load ${N_{xx}}_{crit}$ [N/mm]')
    #plt.title("Reproducing Kassapoglou's Figs. 6.4 and 6.5")
    #plt.show()

    AR_values = np.linspace(0.6, 5, 200)
    res = [calc_Nxx_crit(b*AR, b, None, 1, D11, D12, D22, D66) for AR in
           AR_values]
    #import matplotlib.pyplot as plt
    #plt.plot(AR_values, np.asarray(res))
    #plt.show()

    assert np.allclose(calc_Nxx_crit(b*AR, b, 1, 1, D11, D12, D22, D66),
                       calc_Nxx_crit(b*AR, b, 1, None, D11, D12, D22, D66))

    try:
        calc_Nxx_crit(b*AR, b, None, None, D11, D12, D22, D66)
    except NotImplementedError:
        pass


def test_calc_Nxy_crit():
    """Verificatoin based on Kassapoglou's Fig. 6.11

    """
    laminaprop = (68.9e9, 68.0e9, 0.05, 4.83e9, 4.83e9, 4.83e9)
    lam = laminated_plate(stack=[0, 90]*8, plyt=0.1905e-3/2, laminaprop=laminaprop)
    a_values = np.linspace(0.1, 0.5)
    res = []
    for a in a_values:
        res.append(calc_Nxy_crit(a, lam.D11, lam.D12, lam.D16, lam.D22, lam.D66))
    res_expected = [79688.55992406543, 68114.0022704454, 58889.57598574364, 51419.57333450177, 45285.73547400736, 40187.4044061502, 35903.96554281875, 32270.57385354717, 29162.053403091162, 26481.969879263816, 24155.06026735022, 22121.89066686104, 20335.02310316517, 18756.22315240477, 17354.39749457425, 16104.051205932257, 14984.120320908534, 13977.078850002274, 13068.248915899261, 12245.262872171586, 11497.640308736318, 10816.452732075362, 10194.055750316005, 9623.87366720392, 9100.225083361767, 8618.180819678446, 8173.447493599944, 7762.271588205649, 7381.359992966362, 7027.813861439158, 6699.073294971503, 6392.870873657024, 6107.192453563187, 5840.243960125795, 5590.423152014058, 5356.295523016742, 5136.573663123334, 4930.099522731353, 4735.829122489075, 4552.8193308193, 4380.216395633825, 4217.245969223063, 4063.204408198967, 3917.45116557157, 3779.4021210406136, 3648.5237195644836, 3524.327808168894, 3406.3670775281935, 3294.2310286958063, 3187.5423969626177]
    assert np.allclose(res, res_expected)

    try:
        calc_Nxy_crit(a, -lam.D11, lam.D12, lam.D16, lam.D22, lam.D66)
    except RuntimeError:
        pass

    #import matplotlib.pyplot as plt
    #plt.plot(a_values*1000, np.asarray(res)/1000)
    #plt.xlabel('Plate dimension a [mm]')
    #plt.ylabel('Buckling load [N/mm]')
    #plt.title("Reproducing Kassapoglou's Fig. 6.11")


def test_calc_Nxx_crit_combined_shear_full():
    D11, D12, D22, D66 = 0.66, 0.47, 0.66, 0.49
    b = 0.508/2
    a = 2*b
    Nxx_crit = calc_Nxx_crit(a, b, 1, 1, D11, D12, D22, D66)
    Nxy_crit = calc_Nxy_crit(a, D11, D12, 0, D22, D66)
    x_values = []
    y_values = []
    for Nxy in np.linspace(0.01*Nxy_crit, 0.99*Nxy_crit):
        Nxx = calc_Nxx_crit_combined_shear_full(Nxy, a, b, D11, D12, 0, D22, 0, D66)
        assert np.isclose(Nxx, calc_Nxx_crit_combined_shear(Nxy/Nxx, a, b, D11, D12, D22, D66))
        x_values.append(Nxx/Nxx_crit)
        y_values.append(Nxy/Nxy_crit)

    #import matplotlib.pyplot as plt
    #plt.plot(x_values, y_values)
    #plt.ylim(0., 1.1)
    #plt.xlabel('Length/Width')
    #plt.ylabel('Buckling load ${N_{xx}}_{crit}$ [N/mm]')
    #plt.title("Reproducing Kassapoglou's Figs. 6.4 and 6.5")
    #plt.show()


def test_calc_beff():
    """Verificatoin based on Kassapoglou's Fig. 7.10

    """
    laminaprop = (68.9e9, 68.0e9, 0.05, 4.83e9, 4.83e9, 4.83e9)
    plyt = 0.1905e-3/2/16
    lam = laminated_plate(stack=[0]*8, plyt=plyt, laminaprop=laminaprop)

    Px = np.linspace(10, 1, 40)
    Pcr = 1

    a = b = 1

    A11 = lam.A11
    A22 = A11*10.
    A12 = A11*0.7
    beffs1 = calc_beff(b, Px, Pcr, A11, A12, A22)

    A11 = lam.A11
    A22 = A11*1.
    A12 = A11*0.3
    beffs2 = calc_beff(b, Px, Pcr, A11, A12, A22)

    A11 = lam.A11
    A22 = A11*0.1
    A12 = A11*0.05
    beffs3 = calc_beff(b, Px, Pcr, A11, A12, A22)

    beffs1_expected = [0.45507927187316505, 0.4551866068538131, 0.45529918984223244, 0.4554174153756495, 0.45554171855541725, 0.4556725803977709, 0.4558105340545082, 0.4559561720731905, 0.4561101549053356, 0.45627322092026296, 0.45644619824489074, 0.45663001883010385, 0.45682573524800907, 0.45703454085930917, 0.457257794166946, 0.45749704840613936, 0.45775408773319887, 0.45803097179621455, 0.45833009104349853, 0.4586542359140656, 0.45900668415263557, 0.45939131204105105, 0.4598127375544637, 0.460276506669263, 0.4607893388006151, 0.4613594544864163, 0.46199701937406856, 0.4627147556963568, 0.4635287998959824, 0.46445993031358884, 0.4655353656705211, 0.4667914699658148, 0.4682779456193354, 0.4700645666210135, 0.47225244831338414, 0.4749938710468252, 0.47852932921695196, 0.48326222513948147, 0.48992493085736866, 0.5]
    beffs2_expected = [0.31545741324921134, 0.31576330183988066, 0.31608462911037466, 0.31642259414225943, 0.3167785234899329, 0.31715388858246, 0.31755032605613837, 0.31796966161026835, 0.31841393811955543, 0.3188854489164087, 0.31938677738741617, 0.31992084432717677, 0.320490964882373, 0.3211009174311927, 0.3217550274223035, 0.3224582701062215, 0.3232163973196689, 0.3240360951599672, 0.32492518170158186, 0.32589285714285715, 0.3269500233535731, 0.3281096963761019, 0.3293875450334534, 0.33080260303687636, 0.332378223495702, 0.33414337788578374, 0.3361344537815126, 0.3383977900552486, 0.34099332839140106, 0.34400000000000003, 0.3475238922675934, 0.35171102661596954, 0.3567681007345226, 0.3629976580796253, 0.37086092715231794, 0.38109756097560976, 0.3949730700179534, 0.4148471615720524, 0.445682451253482, 0.5]
    beffs3_expected = [0.2037617554858934, 0.20407911001236095, 0.20441288359117424, 0.20476438427492838, 0.20513506285102973, 0.20552653285675004, 0.20594059405940593, 0.20637926012234198, 0.20684479135394776, 0.20733973366367295, 0.20786696514230893, 0.20842975206611566, 0.20903181662675333, 0.20967741935483866, 0.2103714600956234, 0.2111196025983951, 0.21192842942345927, 0.21280563613758807, 0.21376027693639116, 0.2148030783159801, 0.21594684385382062, 0.21720698254364088, 0.2186022070415134, 0.22015546918378676, 0.22189523248969983, 0.22385723231058233, 0.22608695652173913, 0.22864321608040203, 0.23160340821068942, 0.23507148864592092, 0.23919043238270468, 0.24416243654822334, 0.25028312570781425, 0.25800256081946227, 0.2680412371134021, 0.2816291161178509, 0.30105263157894746, 0.3310991957104557, 0.3837638376383766, 0.5]

    assert np.allclose(beffs1, beffs1_expected)
    assert np.allclose(beffs2, beffs2_expected)
    assert np.allclose(beffs3, beffs3_expected)

    #import matplotlib.pyplot as plt
    #plt.plot(Pcr/Px, beffs1/b)
    #plt.plot(Pcr/Px, beffs2/b)
    #plt.plot(Pcr/Px, beffs3/b)
    #plt.show()


if __name__ == '__main__':
    test_calc_Nxx_crit()
    test_calc_Nxy_crit()
    test_calc_Nxx_crit_combined_shear_full()
    test_calc_beff()

    if False:
        laminaprop = (68.9e9, 68.0e9, 0.05, 4.83e9, 4.83e9, 4.83e9)
        plyt = 0.1905e-3/2/16
        lam = laminated_plate(stack=[0]*8, plyt=plyt, laminaprop=laminaprop)

        Px = np.linspace(10, 1, 40)
        Pcr = 1

        b = 1

        A11 = lam.A11
        A22 = A11*10.
        A12 = A11*0.7
        beffs1 = calc_beff(b, Px, Pcr, A11, A12, A22)

        import matplotlib.pyplot as plt
        plt.plot(Pcr/Px, beffs1/a)
        plt.show()
