# Agent

## Properties

| Name | Type | Required | Description |
|---|---|---|---|
| name | string | No | The name of the agent. Defaults to a name generated by CDK |
| instruction | string | Yes | The instruction used by the agent that determines how it will perform its task. Must have a minimum of 40 characters |
| foundationModel | IInvokable | Yes | The foundation model used for orchestration by the agent |
| existingRole | iam.IRole | No | The existing IAM Role for the agent to use. Must have a trust policy allowing Bedrock service to assume the role. Defaults to a new created role |
| shouldPrepareAgent | boolean | No | Specifies whether to automatically update the `DRAFT` version of the agent after making changes. Defaults to false |
| idleSessionTTL | Duration | No | How long sessions should be kept open for the agent. Session expires if no conversation occurs during this time. Defaults to 1 hour |
| kmsKey | kms.IKey | No | The KMS key of the agent if custom encryption is configured. Defaults to AWS managed key |
| description | string | No | A description of the agent. Defaults to no description |
| knowledgeBases | IKnowledgeBase[] | No | The KnowledgeBases associated with the agent |
| actionGroups | AgentActionGroup[] | No | The Action Groups associated with the agent |
| guardrail | IGuardrail | No | The guardrail that will be associated with the agent |
| promptOverrideConfiguration | PromptOverrideConfiguration | No | Overrides some prompt templates in different parts of an agent sequence configuration |
| userInputEnabled | boolean | No | Select whether the agent can prompt additional information from the user when it lacks enough information. Defaults to false |
| codeInterpreterEnabled | boolean | No | Select whether the agent can generate, run, and troubleshoot code when trying to complete a task. Defaults to false |
| forceDelete | boolean | No | Whether to delete the resource even if it's in use. Defaults to true |

## Create an Agent

The following example creates an Agent with a simple instruction and default prompts that consults a Knowledge Base.

### Example

#### TypeScript

```ts
const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
});

agent.addKnowledgeBase(kb);
```

#### Python

```python
agent = bedrock.Agent(
    self,
    "Agent",
    foundation_model=bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
    instruction="You are a helpful and friendly agent that answers questions about insurance claims.",
)
  agent.add_knowledge_base(kb)
```

## Enable Cross Region Inference for Supported Models

You can also use system defined inference profiles to enable cross region inference requests for supported models. For instance:

### Example

#### TypeScript

```ts
const cris = bedrock.CrossRegionInferenceProfile.fromConfig({
  geoRegion: bedrock.CrossRegionInferenceProfileRegion.US,
  model: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_3_5_SONNET_V1_0,
});

const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: cris,
  instruction: 'You are a helpful and friendly agent that answers questions about agriculture.',
});
```

#### Python

```python
cris = bedrock.CrossRegionInferenceProfile.from_config(
  geo_region= bedrock.CrossRegionInferenceProfileRegion.US,
  model= bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_3_5_SONNET_V1_0
)

agent = bedrock.Agent(
    self,
    "Agent",
    foundation_model=cris,
    instruction="You are a helpful and friendly agent that answers questions about agriculture.",
)
```

For more information on cross region inference, please refer to [System defined inference profiles](#system-defined-inference-profiles)

## Prepare the Agent

The `Agent` constructs take an optional parameter `shouldPrepareAgent` to indicate that the Agent should be prepared after any updates to an agent, Knowledge Base association, or action group. This may increase the time to create and update those resources. By default, this value is false .

Creating an agent alias will not prepare the agent, so if you create an alias using the `AgentAlias` resource then you should set `shouldPrepareAgent` to **_true_**.

## Memory Configuration

Agents can maintain context across multiple sessions and recall past interactions using memory. This feature is useful for creating a more coherent conversational experience.

### Memory Options

You can configure memory for an agent using the `memory` property in the `AgentProps` interface. The memory configuration allows you to specify the type of memory and its properties.

### Example

#### TypeScript

```typescript
import { Agent, Memory, SessionSummaryMemoryProps } from 'src/cdk-lib/bedrock/agents';

const agent = new Agent(this, 'MyAgent', {
  name: 'MyAgent',
  instruction: 'Your instruction here',
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
  memory: Memory.sessionSummary({
        maxRecentSessions: 10, // Keep the last 20 session summaries
        memoryDurationDays: 20, // Retain summaries for 30 days
      }),
});
```

#### Python

```py
from src.cdk_lib.bedrock.agents import Agent, Memory, BedrockFoundationModel

agent = Agent(self, 'MyAgent',
    name='MyAgent',
    instruction='Your instruction here',
    foundation_model=BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
    memory=Memory.session_summary(
        max_recent_sessions=10,  # Keep the last 10 session summaries
        memory_duration_days=20,  # Retain summaries for 20 days
    ),
)
```

### Memory Properties

- **memoryDurationDays**: Duration in days for which session summaries are retained (1-365). Default is 30 days.
- **maxRecentSessions**: Maximum number of recent session summaries to include (minimum 1). Default is 20.

### Memory Types

Currently, the following memory type is supported:

- **SESSION_SUMMARY**: Uses memory summarization to enhance accuracy by summarizing sessions.

For more information on memory configuration, refer to the [AWS Bedrock documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/agents-memory.html).
