"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMetadata = void 0;
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const annotations_1 = require("../annotations");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const runtime_info_1 = require("./runtime-info");
const FILE_PATH = 'tree.json';
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 */
class TreeMetadata extends construct_compat_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    /**
     * Create tree.json
     * @internal
     */
    _synthesizeTree(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    annotations_1.Annotations.of(this).addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct),
                constructInfo: runtime_info_1.constructInfoFromConstruct(construct),
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cloud_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH,
            },
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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